local tribes = {
  [ "death knight"  ] = {
    [ "blood elf"   ] = { weight = 1, faction =    "horde" },
    [ "draenei"     ] = { weight = 1, faction = "alliance" },
    [ "dwarf"       ] = { weight = 1, faction = "alliance" },
    [ "gnome"       ] = { weight = 1, faction = "alliance" },
    [ "goblin"      ] = { weight = 1, faction =    "horde" },
    [ "half elf"    ] = { weight = 1, faction =  "neutral" },
    [ "high elf"    ] = { weight = 1, faction = "alliance" },
    [ "human"       ] = { weight = 1, faction = "alliance" },
    [ "night elf"   ] = { weight = 1, faction = "alliance" },
    [ "orc"         ] = { weight = 1, faction =    "horde" },
    [ "tauren"      ] = { weight = 1, faction =    "horde" },
    [ "troll"       ] = { weight = 1, faction =    "horde" },
    [ "worgen"      ] = { weight = 1, faction = "alliance" } },
  [ "dwarf"         ] = {
  [ "dark iron"   ] = { weight = 1, faction = "alliance" },
  [ "wildhammer"  ] = { weight = 1, faction = "alliance" },
  [ "bronzebeard" ] = { weight = 1, faction = "alliance" },
  [ "frostborn"   ] = { weight = 1, faction = "alliance" },
  [ "stonefist"   ] = { weight = 1, faction = "alliance" },
  [ "stormpike"   ] = { weight = 1, faction = "alliance" } },
  [ "night elf"     ] = {
  [ "kaldorei"    ] = { weight = 8, faction = "alliance" },
  [ "highborne"   ] = { weight = 1, faction = "alliance" } },
  [ "orc"           ] = {
  [ "fel"         ] = { weight = 1, faction =    "horde" },
  [ "brown"       ] = { weight = 1, faction =    "horde" } },
  [ "pandaren"      ] = {
  [ "huojin"      ] = { weight = 1, faction =    "horde" },
  [ "tushui"      ] = { weight = 1, faction = "alliance" },
  [ "independant" ] = { weight = 1, faction =  "neutral" } },
  [ "tauren"        ] = {
  [ "bloodhoof"   ] = { weight = 1, faction =    "horde" },
  [ "dawnstrider" ] = { weight = 1, faction =    "horde" },
  [ "grimtotem"   ] = { weight = 1, faction =    "horde" },
  [ "mistrunner"  ] = { weight = 1, faction =    "horde" },
  [ "ragetotem"   ] = { weight = 1, faction =    "horde" },
  [ "runetotem"   ] = { weight = 1, faction =    "horde" },
  [ "skychaser"   ] = { weight = 1, faction =    "horde" },
  [ "stonehoof"   ] = { weight = 1, faction =    "horde" },
  [ "thunderhorn" ] = { weight = 1, faction =    "horde" },
  [ "wildmane"    ] = { weight = 1, faction =    "horde" },
  [ "winterhoof"  ] = { weight = 1, faction =    "horde" },
  [ "clawhoof"    ] = { weight = 1, faction =    "horde" } },
  [ "troll"         ] = {
  [ "island"      ] = { weight = 1, faction =    "horde" },
  [ "forest"      ] = { weight = 1, faction =    "horde" },
  [ "jungle"      ] = { weight = 1, faction =    "horde" },
  [ "ice"         ] = { weight = 1, faction =    "horde" },
  [ "desert"      ] = { weight = 1, faction =    "horde" },
  [ ""            ] = { weight = 1, faction =    "horde" } },
  [ "undead"        ] = {
  [ "blood elf"   ] = { weight = 1, faction =    "horde" },
  [ "high elf"    ] = { weight = 2, faction =    "horde" },
  [ "human"       ] = { weight = 7, faction =    "horde" } },
}

zonesToTribes = {
  ["Thunder Totem"] = {  
    [   "tauren"] = mergeTables2({ ["highmountain"] = { weight = 30, faction =  "neutral" } }, tribes[   "tauren"]) },
  [ "Highmountain"] = {
    [   "tauren"] = mergeTables2({ ["highmountain"] = { weight = 30, faction =  "neutral" } }, tribes[   "tauren"]) },
  [      "Suramar"] = {
    ["night elf"] = mergeTables2({ [ "nightfallen"] = { weight =  2, faction =  "neutral" }, 
                                   [  "nightborne"] = { weight =  2, faction =  "neutral" } }, tribes["night elf"]) },
  [   "Val'sharah"] = {
    ["night elf"] = mergeTables2({ [            ""] = { weight = 20, faction =  "neutral" } }, tribes["night elf"]) },
  [       "Azsuna"] = {
    ["night elf"] = mergeTables2({ [    "farondis"] = { weight =  4, faction =  "neutral" } }, tribes["night elf"]) },
}

subzonesToTribes = { }

-- Name displayed for the Tribe trait.
local context	= { 
  ["blood elf"    ] = "",
  ["death knight" ] = "Died as",
  ["draenei"      ] = "",
  ["dwarf"        ] = "Clan",
  ["gnome"        ] = "",
  ["goblin"       ] = "",
  ["half elf"     ] = "",
  ["half orc"     ] = "",
  ["high elf"     ] = "",
  ["human"        ] = "",
  ["night elf"    ] = "Type",
  ["orc"          ] = "Clan",
  ["pandaren"     ] = "Faction",
  ["tauren"       ] = "Clan",
  ["troll"        ] = "Subrace",
  ["undead"       ] = "Died as",
  ["worgen"       ] = ""
}

---------------------------------------------------------------------------------------------------------------------------------------
-- | Functions for Generation and Retrieval | -----------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- Main generator should be CharacterTemplate -> { Trait }
-- This will later change to be CharacterTemplate -> { Trait, weight } or something similar.

-- Part 1: Get potential candidates
-- Part 2: Weed out unfeasable candidates, like those not within context or faction
-- Part 3: Select a random item

---------------------------------------------------------------------------------------------------------------------------------------
---- | Part 1 Functions: Generation | -------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- TribeDetails a => CharacterTemplate -> ZoneString -> SubZoneString -> { Tribe, a }

local function checkAllLocations(characterTemplate, zone, subZone)
  local race = characterTemplate.race
  return 
    -- Test whether the SubZone has values.
    ((subzonesToTribes[zone] or { })[subZone] or { })[race]
    -- Test whether the Zone has values.
    or (zonesToTribes[zone] or { })[race]
    -- Fallback to default.
    or tribes[race]
    -- Fallback in case some extreme breaking happened.
    or { [""] = { weight=1,faction="neutral" } }
end

local function checkZonesOnly(characterTemplate, zone)
  local race = characterTemplate.race
  return
    -- Test whether the Zone has values.
    (zonesToTribes[zone] or { })[race]
    -- Fallback to default.
    or tribes[race]
    -- Fallback in case some extreme breaking happened.
    or { [""] = { weight=1,faction="neutral" } }
end

local function dontCheckLocations(characterTemplate)
  local race = characterTemplate.race
  return
    -- Fallback to default.
    tribes[race]
    -- Fallback in case some extreme breaking happened.
    or { [""] = { weight=1,faction="neutral" } }
end

---------------------------------------------------------------------------------------------------------------------------------------
---- | Part 2 Functions: Filtering | --------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- These functions should be of type:
-- characterTemplate -> tribe -> zone -> subzone -> bool

local function isTribeOfFaction(characterTemplate, tribe)
  local race = characterTemplate.race
  if not tribes[race]        then return true  end -- This case holds races that don't have tribes.
  if not tribes[race][tribe] then return true  end -- If this is true, all neutral night elves would show up in horde, AS LONG AS nightelf somehow gets through the race check.
  -- Highmountain don't pass the above check for Horde. Either need a seperate table for just faction checking, or go crazy on the rules. For now... the latter and regret later.
  return tribes[race][tribe].faction == string.lower(UnitFactionGroup("player")) or tribes[race][tribe].faction == "neutral"
end

local function isTribeWithinContext(characterTemplate, tribe, zone, subZone)
  local race = characterTemplate.race
  if     zone == "Suramar"      then
    return isTribeOfFaction(characterTemplate, tribe) or (race == "night elf" and (tribe == "nightfallen" or tribe == "nightborne"))
  elseif zone == "Dalaran"      then
    return true
  elseif zone == "Broken Shore" then
    return true
  elseif zone == "Trueshot Lodge" then
    return true
  elseif zone == "Highmountain" then
    return isTribeOfFaction(characterTemplate, tribe) or (race == "tauren"    and tribe == "highmountain")
  elseif zone == "Val'sharah"   then
    return isTribeOfFaction(characterTemplate, tribe) or (race == "night elf" and tribe == "")
  elseif zone == "Azsuna"       then
    return isTribeOfFaction(characterTemplate, tribe) or (race == "night elf" and tribe == "farondis")
  else
    return isTribeOfFaction(characterTemplate, tribe)
  end
end

---------------------------------------------------------------------------------------------------------------------------------------
---- | Part 3 Functions: Selection | --------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- These functions should be of type:
-- { tribe, stuff } -> tribe
-- This might change in the future.

local function selectUsingWeightedRandom(selectionList)
  return weightedRandomFrom(keys(selectionList), function(tribe) return selectionList[tribe].weight end)
end

---------------------------------------------------------------------------------------------------------------------------------------
--| Combination Function | ------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

function createTribeGenerator(generatorF, filterF, selectorF)
  return function(characterTemplate, zone, subZone)
    return selectorF(
      filter2(
        function(tribe) return filterF(characterTemplate, tribe, zone, subZone) end, 
        generatorF(characterTemplate, zone, subZone)
      )
    )
  end
end

local contextAndSubZoneBasedTribeGenerator    = createTribeGenerator(   checkAllLocations,   isTribeWithinContext, selectUsingWeightedRandom)
local contextAndZoneBasedTribeGenerator       = createTribeGenerator(   checkZonesOnly,      isTribeWithinContext, selectUsingWeightedRandom)
local contextAndNoZoneBasedTribeGenerator     = createTribeGenerator(   dontCheckLocations,  isTribeWithinContext, selectUsingWeightedRandom)
local factionAndSubZoneBasedTribeGenerator    = createTribeGenerator(   checkAllLocations,   isTribeOfFaction,     selectUsingWeightedRandom)
local factionAndZoneBasedTribeGenerator       = createTribeGenerator(   checkZonesOnly,      isTribeOfFaction,     selectUsingWeightedRandom)
local factionAndNoZoneBasedTribeGenerator     = createTribeGenerator(   dontCheckLocations,  isTribeOfFaction,     selectUsingWeightedRandom)
local noFactionAndSubZoneBasedTribeGenerator  = createTribeGenerator(   checkAllLocations,   const(true),          selectUsingWeightedRandom)
local noFactionAndZoneBasedTribeGenerator     = createTribeGenerator(   checkZonesOnly,      const(true),          selectUsingWeightedRandom)
local noFactionAndNoZoneBasedTribeGenerator   = createTribeGenerator(   dontCheckLocations,  const(true),          selectUsingWeightedRandom)

function remakeTribeGenerator()
  generateRandomTribe = function() return "Error: generateRandomTribe is not set" end
  if not NPCG_Options.factionSettings.matching then
    -- faction matching is off
    if not NPCG_Options.locationSettings.zones then
      -- location matching is also off
      generateRandomTribe = noFactionAndNoZoneBasedTribeGenerator
    elseif NPCG_Options.locationSettings.subZones then
      generateRandomTribe = noFactionAndSubZoneBasedTribeGenerator
    else
      generateRandomTribe = noFactionAndZoneBasedTribeGenerator
    end
  elseif NPCG_Options.factionSettings.context then
    -- faction is on, but context is also on
    if not NPCG_Options.locationSettings.zones then
      -- location matching is off
      generateRandomTribe = contextAndNoZoneBasedTribeGenerator
    elseif NPCG_Options.locationSettings.subZones then
      generateRandomTribe = contextAndSubZoneBasedTribeGenerator
    else
      generateRandomTribe = contextAndZoneBasedTribeGenerator
    end
  else
    -- faction is on, context is off
    if not NPCG_Options.locationSettings.zones then
      -- location matching is off
      generateRandomTribe = factionAndNoZoneBasedTribeGenerator
    elseif NPCG_Options.locationSettings.subZones then
      generateRandomTribe = factionAndSubZoneBasedTribeGenerator
    else
      generateRandomTribe = factionAndZoneBasedTribeGenerator
    end
  end
end

--[[ LEGACY ]]--
-- generateRandomTribe :: Race a => a -> Tribe
--[[
function generateRandomTribe(characterTemplate)
  local selectionList = 
    -- Zone-based selection.
    (zonesToTribes[GetZoneText()] or tribes)[characterTemplate.race]
    -- Fallback to default selection.
    or tribes[characterTemplate.race]
    -- Fallback to empty.
    or { [""] = { weight=1,faction="neutral" } }
  
  if     NPCG_Options.factionSettings.matching and not NPCG_Options.factionSettings.context then
    selectionList = filter2(function(tribe) return isTribeOfFaction(    characterTemplate.race, tribe) end, selectionList)
  elseif NPCG_Options.factionSettings.context then
    selectionList = filter2(function(tribe) return isTribeWithinContext(characterTemplate.race, tribe) end, selectionList)
  end

  return weightedRandomFrom(keys(selectionList), function(x) return selectionList[x].weight end)
  --if not tribes[characterTemplate.race] then return "" end
  --return randomFrom((zonesToTribes[GetZoneText()] or tribes)[characterTemplate:getRace()] or tribes[characterTemplate:getRace()]) or "Error: No options found!"
end
]]

-- getTribeContext :: Race -> string
function getTribeContext(race)
	return context[race] or ""
end