---------------------------------------------------------------------------------------------------------------------------------------
-- | Data Model | ---------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- default :: { TraitDepencency } -> { TraitDepencency }
-- Provides defaults for trait dependencies, so others don't have to fill them in.
local function default(replacements)
  local replacements = replacements or { }
  return {
    classTypes  = replacements["classTypes"] or {       },
    races       = replacements[     "races"] or { "any" },
    faiths      = replacements[    "faiths"] or { "any" },
    genders     = replacements[   "genders"] or { "any" },
    alignments  = replacements["alignments"] or { "any" },
    ages        = replacements[      "ages"] or olderThan("child"),
    weight      = 1
  }
end

-- The classes and their dependencies and classTypes regardless of locations or other context.
local classes = {
-- a
[ "adventurer"            ] = default({
  classTypes = {
    "explorer",
    "mercenary", },
  ages = olderThan( "teen" ) }),
[ "alchemist"             ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "young adult" ) }),
[ "animalhandler"         ] = default({
  classTypes = {
    "animal",
    "hunter", },
  ages = olderThan( "child" ) }),
[ "antiquities dealer"    ] = default({
  classTypes = {
    "trader", },
  ages = olderThan( "adult" ) }),
[ "archer"                ] = default({
  classTypes = {
    "military",
    "hunter",
    "mercenary",
    },
  ages = agesBetween( "young adult", "mature adult" ) }),
[ "archeologist"          ] = default({
  classTypes = {
    "scholar",
    "explorer", },
  ages = olderThan( "young adult" ) }),
[ "\"archmage\""          ] = default({
  classTypes = {
    "children" },
  races = {
    "blood elf",
    "gnome",
    "half elf",
    "high elf",
    "human",
    "night elf" },
  ages = agesBetween( "child", "teen" ) }),
[ "armorer"               ] = default({
  classTypes = {
    "crafter",
    "military", },
  ages = olderThan( "young adult" ) }),
[ "assassin"              ] = default({
  classTypes = {
    "rogue",
    "covert", },
  races = allRacesExcept(
  "draenei" ),
  ages = olderThan( "teen" ) }),
[ "astrologer"            ] = default({
  classTypes = {
    "scholar" },
  races = {
    "blood elf",
    "gnome",
    "half elf",
    "high elf",
    "human",
    "night elf" },
  ages = olderThan( "young adult" ) }),
[ "astronomer"            ] = default({
  classTypes = {
    "scholar" },
  races = {
    "blood elf",
    "gnome",
    "half elf",
    "high elf",
    "human",
    "night elf" },
  ages = olderThan( "young adult" ) }),
[ "author"                ] = default({
  classTypes = {
    "scholar",
    "trader", },
  ages = olderThan( "young adult" ) }),
-- b
[ "baker"                 ] = default({
  classTypes = {
    "food",
    "trader", },
  ages = olderThan( "teen" ) }),
[ "bard"                  ] = default({
  classTypes = {
    "entertainer",
    "bar", },
  ages = olderThan( "teen" ) }),
[ "bartender"             ] = default({
  classTypes = {
    "bar", },   
  ages = agesBetween( "young adult", "mature adult" ) }),
[ "beggar"                ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = olderThan( "child" ) }),
[ "blacksmith"            ] = default({
  classTypes = {
    "crafter" },
  ages = agesBetween( "young adult", "adult" ) }),
[ "brat"                  ] = default({
  classTypes = {
    "children" },  
  ages = agesBetween( "child", "child" ) }),
[ "brawler"               ] = default({
  classTypes = {
    "bar",
    "mercenary", },
  races = allRacesExcept(
    "pandaren" ), 
  ages = olderThan( "teen" ) }),
[ "blood mage"            ] = default({
  classTypes = {
    "mage",
    "scholar", },
  races = {
    "blood elf" },  
  ages = olderThan( "young adult" ) }),
[ "bodyguard"             ] = default({
  classTypes = {
    "mercenary", },   
  ages = olderThan( "young adult" ) }),
[ "bookbinder"            ] = default({
  classTypes = {
    "crafter",
    "scholar" },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "undead",
    "worgen" },  
  ages = olderThan( "teen" ) }),
[ "bounty hunter"         ] = default({
  classTypes = {
    "mercenary",
    "rogue", },
  ages = olderThan( "young adult" ) }),
[ "bowyer"                ] = default({
  classTypes = {
    "crafter",
    "hunter" },
  ages = agesBetween( "young adult", "mature adult" ) }),
[ "brewer"                ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "teen" ) }),
[ "brothelkeeper"         ] = default({
  classTypes = {
    "illegal",
    "rogue" },
  races = {
    "blood elf",
    "draenei",
    "dwarf",
    "gnome",
    "goblin",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "orc",
    "pandaren",
    "troll",
    "undead",
    "worgen" },
  ages = olderThan("young adult") }),
[ "brothelworker"         ] = default({
  classTypes = {
    "illegal",
    "rogue" },
  races = {
    "blood elf",
    "draenei",
    "dwarf",
    "gnome",
    "goblin",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "orc",
    "pandaren",
    "troll",
    "worgen" },
  ages = olderThan( "teen" ) }),
[ "burglar"               ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = agesBetween( "teen", "adult" ) }),
[ "butcher"               ] = default({
  classTypes = {
    "food",
    "trader", },
  ages = olderThan( "teen" ) }),
-- c
[ "candlemaker"           ] = default({
  classTypes = {
    "crafter", },
  races = {
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "pandaren",
    "undead",
    "worgen" },
  ages = olderThan( "child" ) }),
[ "carpenter"             ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "teen" ) }),
[ "carpenter apprentice"  ] = default({
  classTypes = {
    "crafter",
    "apprentice" },
  ages = agesBetween( "child", "teen" ) }),
[ "clergy"                ] = default({
  classTypes = {
    "priest",
    "light", },
  faiths = faithsByTag( "light" ),
  ages = olderThan( "teen" ) }),
[ "clothier"              ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "child" ) }),
[ "cobbler"               ] = default({
  classTypes = {
    "crafter", },
  races = allRacesExcept(
    "draenei",
    "pandaren",
    "tauren" ),  
  ages = olderThan( "teen" ) }),
[ "cobbler apprentice"    ] = default({
  classTypes = {
    "crafter",
    "apprentice" },
  ages = agesBetween( "child","teen" ) }),
[ "commoner"              ] = default({
  classTypes = {
    "citizen", },
  ages = olderThan( "child" ) }),
[ "cook"                  ] = default({
  classTypes = {
    "food",
    "bar", },
  ages = agesBetween( "young adult", "mature adult" ) }),
[ "craftsman"             ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "teen" ) }),
[ "cultist"               ] = default({
  classTypes = {
    "priest",
    "dark" },
  faiths = faithsByTag( "cult" ), 
  ages = olderThan( "child" ) }),
-- d
[ "dark ranger"           ] = default({
  classTypes = {
    "hunter" },
  races = {
    "undead" },
  ages = olderThan( "young adult" ) }),
[ "demon hunter"          ] = default({
  classTypes = {
    "demon hunter" },
  races = {
    "blood elf", 
    "night elf" },
  faiths = {
    "the illidari" }, 
  ages = olderThan( "young adult" ) }),
[ "diplomat"              ] = default({
  classTypes = {
    "town support" },
  ages = olderThan( "young adult" ) }),
[ "doctor"                ] = default({
  classTypes = {
    "medical" },
  ages = olderThan( "adult" ) }),
[ "doomsayer"             ] = default({
  classTypes = {
    "town support",
    "dark" },
  faiths = faithsByTag( "cult" ),
  ages = olderThan( "child" ) }),
[ "druid"                 ] = default({
  classTypes = {
    "druid" },
  ages = agesBetween( "young adult", "venerable" ) }),
[ "drummer"               ] = default({
  classTypes = {
    "entertainer" },
  ages = olderThan( "child" ) }),
[ "drunk"                 ] = default({
  classTypes = {
    "illegal",
    "town support", },
  ages = olderThan( "teen" ) }),
-- e
[ "enchanter"             ] = default({
  classTypes = {
    "mage",
    "crafter" },  
  ages = olderThan( "teen" ) }),
[ "engineer"              ] = default({
  classTypes = {
    "crafter",
    "inventor", },
  ages = olderThan( "teen" ) }),
[ "entrepreneur"          ] = default({
  classTypes = {
    "trader" },
  ages = olderThan( "teen" ) }),
[ "explorer"              ] = default({
  classTypes = {
    "explorer" },
  ages = olderThan( "teen" ) }),
-- f
[ "farstrider"            ] = default({
  classTypes = {
    "military", "hunter" },
  races = {
    "blood elf" },
  ages = agesBetween( "adult", "mature adult" ) }),
[ "farmer"                ] = default({
  classTypes = {
    "gatherer" },
  ages = agesBetween( "young adult", "elder" ) }),
[ "fisherman"             ] = default({
  classTypes = {
    "gatherer",
    "harbor", },
  ages = agesBetween( "young adult", "adult" ) }),
[ "fishmonger"            ] = default({
  classTypes = {
    "trader" },
  ages = olderThan( "teen" ) }),
[ "fruit vendor"          ] = default({
  classTypes = {
    "trader" },   
  ages = olderThan( "child" ) }),
-- g
[ "gang member"           ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = olderThan( "teen" ) }),
[ "gardener"              ] = default({
  classTypes = {
    "town support" },
  races = {
    "blood elf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "pandaren",
    "worgen" },
  ages = olderThan( "child") }),
[ "guard"                 ] = default({
  classTypes = {
    "town support" },
  ages = olderThan( "young adult" ) }),
[ "grunt"                 ] = default({
  classTypes = {
    "town support",
    "mercenary", },
  races = {
    "orc" },  
  ages = agesBetween( "young adult", "adult" ) }),
-- h
[ "hairdresser"           ] = default({
  classTypes = {
    "town support" },   
  ages = olderThan( "teen" ) }),
[ "hay merchant"          ] = default({
  classTypes = {
    "trader" },   
  ages = olderThan( "young adult" ) }),
[ "headhunter"            ] = default({
  classTypes = {
    "mercenary", "hunter" },
  races = {
    "troll" },
  ages = olderThan( "teen" ) }),
[ "herald"                ] = default({
  classTypes = {
    "citizen" },
  races = {
    "blood elf",
    "draenei",
    "dwarf",
    "gnome",
    "goblin",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "orc",
    "pandaren",
    "worgen" },
  ages = olderThan( "teen" ) }),
[ "herbalist"             ] = default({
  classTypes = {
    "gatherer" },   
  ages = olderThan( "teen" ) }),
[ "homeless person"       ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages =  agesBetween( "child", "child" ) }),
[ "horse trainer"         ] = default({
  classTypes = {
    "animal" },
  races = {
  "blood elf",
  "dwarf",
  "half elf",
  "half orc",
  "high elf",
  "human",
  "undead",
  "worgen" },
  ages = olderThan( "teen" ) }),
[ "hunter"                ] = default({
  classTypes = {
    "hunter" },
  ages = agesBetween( "young adult", "mature adult" ) }),
-- i
[ "illuminator"           ] = default({
  classTypes = {
    "town support", },
  races = {
    "blood elf",
    "dwarf",
    "goblin",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "orc",
    "pandaren",
    "tauren",
    "worgen",
    "undead" },
  ages = olderThan( "teen" ) }),
[ "innkeeper"             ] = default({
  classTypes = {
    "bar",
    "citizen", },
  ages = olderThan( "adult" ) }),
[ "inventor"              ] = default({
  classTypes = {
    "inventor",
    "trader", },
  ages = olderThan( "young adult" ) }),
-- j
[ "jailer"                ] = default({
  classTypes = {
    "town support" },
  ages = olderThan( "young adult" ) }),
[ "jester"                ] = default({
  classTypes = {
    "entertainer" },
  ages = olderThan( "child" ) }),
[ "jewelcrafter"          ] = default({
  classTypes = {
    "crafter",
    "trader" },
  ages = olderThan( "young adult" ) }),
[ "juggler"               ] = default({
  classTypes = {
    "entertainer" },   
  ages = agesBetween("child", "elder") }),
-- k
[ "kitchen help"          ] = default({
  classTypes = {
    "bar",
    "food", },
  ages = olderThan( "child" ) }),
[ "knight"                ] = default({
  classTypes = {
    "paladin",
    "military" },
  races = {
    "human" },  
  ages = olderThan( "adult" ) }),
-- l
[ "laborer"               ] = default({
  classTypes = {
    "citizen" },
  ages = agesBetween( "teen", "mature adult" ) }),
[ "leatherworker"         ] = default({
  classTypes = {
    "crafter", "hunter" },
  ages = olderThan( "young adult" ) }),
[ "loan provider"         ] = default({
  classTypes = {
    "town support" },
  ages = olderThan( "young adult" ) }),
[ "lookout"               ] = default({
  classTypes = {
    "rogue",
    "mercenary" },
  ages = agesBetween( "child", "young adult" ) }),
[ "lorewalker"            ] = default({
  classTypes = {
    "scholar",
    "explorer", },
  races = {
    "pandaren" },
  ages = olderThan( "young adult" ) }),
[ "lumberjack"            ] = default({
  classTypes = {
    "gatherer" },   
  ages = agesBetween( "teen", "mature adult" ) }),
-- m
[ "mage"                  ] = default({
  classTypes = {
    "mage",
    "scholar", },
  ages = agesBetween( "young adult", "venerable" ) }),
[ "mathmatician"          ] = default({
  classTypes = {
    "scholar" },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "goblin",
    "half elf",
    "high elf",
    "human",
    "undead",
    "worgen" },
  ages = olderThan( "young adult" ) }),
[ "medic"                 ] = default({
  classTypes = {
    "medical" },
  ages = agesBetween( "young adult", "elder" ) }),
[ "mercenary"             ] = default({
  classTypes = {
    "mercenary",
    "warrior", },
  ages = agesBetween( "teen", "mature adult" ) }),
[ "merchant"              ] = default({
  classTypes = {
    "trader" },
  ages = agesBetween( "teen", "elder" ) }),
[ "messenger"             ] = default({
  classTypes = {
    "citizen" },
  ages = olderThan( "child" ) }),
[ "military advisor"      ] = default({
  classTypes = {
    "military", },
  ages = olderThan( "adult" ) }),
[ "mime"                  ] = default({
  classTypes = {
    "entertainer" },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "worgen" },
  ages = olderThan( "child" ) }),
[ "miner"                 ] = default({
  classTypes = {
    "gatherer" },
  ages = agesBetween( "teen", "mature adult" ) }),
[ "monk"                  ] = default({
  classTypes = {
    "monk" },
  ages = olderThan( "teen" ) }),
[ "musician"              ] = default({
  classTypes = {
    "entertainer" },
  ages = olderThan( "teen" ) }),
-- n
[ "nanny"                 ] = default({
  classTypes = {
    "children",
    "citizen", },
  ages = olderThan( "young adult" ) }),
[ "navigator"             ] = default({
  classTypes = {
    "harbor", },
  ages = olderThan( "teen" ) }),
[ "necromancer"           ] = default({
  classTypes = {
    "cult", },
  faiths = {
    "cult of the damned" },
  ages = olderThan( "adult" ) }),
[ "nurse"                 ] = default({
  classTypes = {
    "medical", },
  ages = olderThan( "teen" ) }),
-- o
[ "old clothes dealer"    ] = default({
  classTypes = {
    "trader", },
  ages = olderThan( "teen" ) }),
[ "outlaw"                ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = olderThan( "child" ) }),
-- p
[ "page"                  ] = default({
  classTypes = {
    "paladin", 
    "apprentice", },
  races = {
    "blood elf",
    "dwarf",
    "draenei",
    "goblin",
    "half elf",
    "high elf",
    "human",
    "night elf",
    "tauren" },
  ages = agesBetween( "child", "child" ) }),
[ "painter"               ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "teen" ) }),
[ "paladin"               ] = default({
  classTypes = {
    "paladin", },
  races = {
    "blood elf",
    "draenei",
    "dwarf",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "tauren" },
  faiths = faithsByTag( "light" ),
  ages = olderThan( "young adult" ) }),
[ "peasant"               ] = default({
  classTypes = {
    "citizen", },
  ages = olderThan( "child" ) }),
[ "pharmacist"            ] = default({
  classTypes = {
    "trader", },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "worgen" },
  ages = olderThan( "young adult" ) }),
[ "philosopher"           ] = default({
  classTypes = {
    "scholar", },
  ages = olderThan( "adult" ) }),
[ "physician"             ] = default({
  classTypes = {
    "medical", },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "goblin",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "worgen" },
  ages = olderThan( "young adult" ) }),
[ "pickpocket"            ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = agesBetween( "child", "teen" ) }),
[ "pie seller"            ] = default({
  classTypes = {
    "trader", },   
  ages = olderThan( "teen" ) }),
[ "piper"                 ] = default({
  classTypes = {
    "crafter", },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "pandaren",
    "tauren",
    "troll",
    "worgen" },
  ages = olderThan( "teen" ) }),
[ "piper apprentice"      ] = default({
  classTypes = {
    "crafter",
    "apprentice" },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "pandaren",
    "tauren",
    "troll",
    "worgen" },
  ages = agesBetween( "child", "teen" ) }),
[ "pirate"                ] = default({
  classTypes = {
    "harbor",
    "rogue",
    "illegal", },
  ages = olderThan( "teen" ) }),
[ "potter"                ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "teen" ) }),
[ "potter apprentice"     ] = default({
  classTypes = {
    "crafter", 
    "apprentice", },
  ages = agesBetween( "child", "teen" ) }),
[ "preacher"              ] = default({
  classTypes = {
    "priest", 
    "light", },
  ages = olderThan( "teen" ) }),
[ "priest"                ] = default({
  classTypes = {
    "priest",
    "light", },
  ages = agesBetween( "young adult", "venerable" ) }),
-- q
-- r
[ "ranger"                ] = default({
  classTypes = {
    "hunter", },
  races = {
    "blood elf",
    "human",
    "high elf",
    "undead" },
  ages = olderThan( "teen" ) }),
[ "rascal"                ] = default({
  classTypes = {
    "children", },
  ages = agesBetween( "child", "child" ) }),
[ "rebel"                 ] = default({
  classTypes = {
    "mercenary", },   
  ages = olderThan( "child" ) }),
-- s
[ "sailor"                ] = default({
  classTypes = {
    "harbor", },
  ages = olderThan( "teen" ) }),
[ "scholar"               ] = default({
  classTypes = {
    "scholar", },
  ages = olderThan( "teen" ) }),
[ "scout"                 ] = default({
  classTypes = {
    "hunter",
    "mercenary",
    "military", },   
  ages = agesBetween( "young adult", "mature adult" ) }),
[ "scribe"                ] = default({
  classTypes = {
    "scholar", },
  races = racesByType( "civilized" ),  
  ages = olderThan( "young adult" ) }),
[ "scullion"              ] = default({
  classTypes = {
    "crafter", },
  races = {
    "blood elf",
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "pandaren",
    "undead",
    "worgen" },
  ages = olderThan( "young adult" ) }),
[ "servant"               ] = default({
  classTypes = {
    "citizen", },
  ages = olderThan( "child" ) }),
[ "sewer"                 ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "child" ) }),
[ "shadow hunter"         ] = default({
  classTypes = {
    "dark" },
  races = {
    "troll" },
  ages = olderThan( "adult" ) }),
[ "shaman"                ] = default({
  classTypes = {
    "shaman", },
  ages = agesBetween( "young adult", "venerable" ) }),
[ "shepherd"              ] = default({
  classTypes = {
    "farmer" },
  ages = olderThan( "child" ) }),
[ "shopkeeper"            ] = default({
  classTypes = {
    "trader", },
  ages = agesBetween( "young adult" , "mature adult" ) }),
[ "skinner"               ] = default({
  classTypes = {
    "gatherer",
    "hunter", },
  ages = olderThan( "teen" ) }),
[ "skinner apprentice"    ] = default({
  classTypes = {
    "apprentice",
    "hunter",
    "trader", },
  ages = agesBetween( "child", "teen" ) }),
[ "soldier"               ] = default({
  classTypes = {
    "military",
    "hunter",
    "paladin", },
  ages = agesBetween( "teen", "mature adult" ) }),
[ "sorcerer"              ] = default({
  classTypes = {
    "mage",
    "warlock", },   
  ages = olderThan("young adult") }),
[ "spellbreaker"          ] = default({
  classTypes = {
    "town support",
    "military", },
  races = {
    "blood elf" },
  ages = olderThan( "young adult" ) }),
[ "spy"                   ] = default({
  classTypes = {
    "rogue",
    "covert", },
  races = allRacesExcept(
    "draenei",
    "tauren" ),
  ages = olderThan("child") }),
[ "squire"                ] = default({
  classTypes = {
    "apprentice", 
    "paladin", },
  races = {
    "blood elf",
    "draenei",
    "goblin",
    "half elf",
    "high elf",
    "human",
    "night elf",
    "tauren" },
  ages = agesBetween( "child", "young adult" ) }),
[ "stablekeeper"          ] = default({
  classTypes = {
    "animal",
    "hunter", },
  ages = olderThan( "teen" ) }),
[ "stablehand"            ] = default({
  classTypes = {
    "animal",
    "hunter" },
  ages = agesBetween( "child", "young adult" ) }),
[ "student"               ] = default({
  classTypes = {
    "children",
    "scholar",
    "priest",
    "mage" },
  ages = agesBetween( "child", "teen" ) }),
[ "surgeon"               ] = default({
  classTypes = {
    "medical" },
  races = racesByType( "civilized" ), 
  ages = olderThan( "young adult" ) }),
[ "swindler"              ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = olderThan( "child" ) }),
-- t
[ "tailor"                ] = default({
  classTypes = {
    "crafter",
    "trader", },
  ages = olderThan( "young adult" ) }),
[ "teacher"               ] = default({
  classTypes = {
    "children",
    "scholar", },
  ages = olderThan( "young adult" ) }),
[ "thief"                 ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = agesBetween( "child", "mature adult" ) }),
[ "thug"                  ] = default({
  classTypes = {
    "rogue",
    "illegal", },
  ages = olderThan( "teen" ) }),
[ "tinker"                ] = default({
  classTypes = {
    "crafter",
    "inventor", },
  races = {
    "gnome",
    "goblin" },
  ages = olderThan( "young adult" ) }),
[ "trapper"               ] = default({
  classTypes = {
    "gatherer",
    "hunter", },
  ages = olderThan( "teen" ) }),
-- u 
-- v
[ "vendor"                ] = default({
  classTypes = {
    "trader", },
  ages = olderThan( "teen" ) }),
-- w
[ "waiter"                ] = default({
  classTypes = {
    "bar", },
  ages = agesBetween( "child", "mature adult" ) }),
[ "warlock"               ] = default({
  classTypes = {
    "warlock",
    "dark" },   
  ages = agesBetween( "young adult",  "venerable" ) }),
[ "watchman"              ] = default({
  classTypes = {
    "town support",
    "hunter", },
  ages = olderThan( "teen" ) }),
[ "water vendor"          ] = default({
  classTypes = {
    "trader", },
  ages = olderThan( "teen" ) }),
[ "wine vendor"           ] = default({
  classTypes = {
    "trader", },
  races = {
    "blood elf",
    "draenei",
    "dwarf",
    "gnome",
    "goblin",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "night elf",
    "pandaren",
    "undead" },
  ages = olderThan( "teen" ) }),
[ "witch doctor"          ] = default({
  classTypes = {
    "nature" },
  races = {
    "troll" },
  ages = olderThan( "adult" ) }),
[ "woodcarver"            ] = default({
  classTypes = {
    "crafter", },
  ages = olderThan( "child" ) }),
[ "woodworker"            ] = default({
  classTypes = {
    "crafter", },
  ages = agesBetween( "young adult", "mature adult" ) }),
[ "woolseller"            ] = default({
  classTypes = {
    "trader", },
  races = {
    "dwarf",
    "gnome",
    "half elf",
    "half elf",
    "high elf",
    "human",
    "undead",
    "worgen" },
  ages = olderThan( "teen" ) }),
[ "wool-stapler"          ] = default({
  classTypes = {
    "crafter", },
  races = {
    "dwarf",
    "gnome",
    "half elf",
    "half orc",
    "high elf",
    "human",
    "undead",
    "worgen" },
  ages = olderThan( "teen" ) }),
-- x
-- y
-- z
}

---------------------------------------------------------------------------------------------------------------------------------------
-- | Queries | ------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- classesByType :: ClassType -> { Class }
function classesByType(classType)
  return filterKeysByValue(
    function(x) return
      contains(classType, x.classTypes)
    end,
    eitherOr(classType == "any", { }, classes)
  )
end

-- classesByType2 :: ClassType -> { Class, ClassConstraints }
function classesByType2(classType)
  -- this should be horrible?
  local classesWithClassType = classesByType(classType)
  local output = { }
  for _,v in ipairs(classesWithClassType) do -- Or is it not..? I can't wrap my head around it >.<
    output[v] = classes[v]
  end
  return output
end

-- This takes an Any approach
function classesByAnyOfTypes(...)
  return mergeTablesWithoutDuplicates(unpack(map(classesByType, {...})))
end

function classesByAnyOfTypes2(...)
  return mergeTablesWithoutDuplicates2(unpack(map(classesByType2, {...})))
end

-- allClassesExcept :: Class -> ... -> { Class }
function allClassesExcept(...)
  local exceptions = {...}
  return without(keys(classes),exceptions)
end

-- randomClass :: () -> Class
-- Generates a true random class without constraints.
-- This is a fallback that should not be used all too lightly.
function randomClass()
  return randomFrom(keys(classes))
end

---------------------------------------------------------------------------------------------------------------------------------------
-- | Location Based Generation | ------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------
local zonesToClasses = {
  ["Dalaran"              ] = nil,
  -- Order Halls
  ["Dreadscar Rift"       ] = classesByType2("warlock"),
  ["Hall of the Guardian" ] = classesByType2("mage"),
  ["Netherlight Temple"   ] = classesByType2("priest"),
  ["Skyhold"              ] = classesByType2("warrior"),
  ["The Dreamgrove"       ] = classesByType2("druid"),
  ["Trueshot Lodge"       ] = classesByType2("hunter"),
  
  -- Starter Zones
  ["Eversong Woods"       ] = nil,
}

local subzonesToClasses = {
  [ "Silvermoon City" ] = {
    ["Murder Row"         ] = classesByType2("rogue")
  },
  [ "Dalaran"         ] = {
    -- Rogue Order Hall
    ["Chamber of Shadows" ] = classesByType2("rogue"),
    ["Cloaked Vestibule"  ] = classesByType2("rogue"),
    ["Hidden Corridor"    ] = classesByType2("rogue"),
    ["The Hall of Shadows"] = classesByType2("rogue"),
    ["Uncrowned Vault"    ] = classesByType2("rogue"),
  },
}

---------------------------------------------------------------------------------------------------------------------------------------
-- | Functions for Generation and Retrieval | -----------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- Main generator should be CharacterTemplate -> { Trait }
-- This will later change to be CharacterTemplate -> { Trait, weight } or something similar.

-- Part 1: Get potential candidates
-- Part 2: Weed out unfeasable candidates, like those not within context or faction
-- Part 3: Select a random item

---------------------------------------------------------------------------------------------------------------------------------------
---- | Part 1 Functions: Generation | -------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

local function checkAllLocations(characterTemplate, zone, subZone)
  return 
    -- Test whether the SubZone has values.
    ((subzonesToClasses[zone] or { })[subZone])
    -- Test whether the Zone has values.
    or zonesToClasses[zone]
    -- Fallback to default.
    or classes
    -- Fallback in case some extreme breaking happened.
    or { }
end

local function checkZonesOnly(characterTemplate, zone)
  return
    -- Test whether the Zone has values.
    zonesToClasses[zone]
    -- Fallback to default.
    or classes
    -- Fallback in case some extreme breaking happened.
    or { }
end

local function dontCheckLocations(characterTemplate)
  return
    -- Fallback to default.
    classes
    -- Fallback in case some extreme breaking happened.
    or { }
end

---------------------------------------------------------------------------------------------------------------------------------------
---- | Part 2 Functions: Filtering | --------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- These functions should be of type:
-- characterTemplate -> class -> zone -> subzone -> bool

local function meetsTraitConstraints(characterTemplate, option)
  return  containsAny(classes[option].races,   { "any", characterTemplate.race   })
    and   containsAny(classes[option].ages,    { "any", characterTemplate.age    })
    and   containsAny(classes[option].faiths,  { "any", characterTemplate.faith  })
end

local function isWowClass(characterTemplate, option)
  return any(
    composeEqualsFunction(option),
    { "druid", "hunter", "mage", "monk", "paladin", "priest", "rogue", "shaman", "warlock", "warrior" })
end

---------------------------------------------------------------------------------------------------------------------------------------
---- | Part 3 Functions: Selection | --------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

-- These functions should be of type:
-- Arbitrary a => { class, a } -> class
-- This might change in the future.

local function randomKeyFrom(options)
  return randomFrom(keys(options))
end

local function selectUsingWeightedRandom(selectionList)
  return weightedRandomFrom(keys(selectionList), function(item) return selectionList[item].weight end)
end

---------------------------------------------------------------------------------------------------------------------------------------
--| Combination Function | ------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------------------

function createClassGenerator(generatorF, filterF, selectorF)
  return function(characterTemplate, zone, subZone)
    return selectorF(
      filter2(
        function(class) return filterF(characterTemplate, class, zone, subZone) end, 
        generatorF(characterTemplate, zone, subZone)
      )
    )
  end
end

local traitAndSubZoneBasedClassGenerator          = createClassGenerator(checkAllLocations,   meetsTraitConstraints,  selectUsingWeightedRandom)
local traitAndZoneBasedClassGenerator             = createClassGenerator(checkZonesOnly,      meetsTraitConstraints,  selectUsingWeightedRandom)
local traitAndNoLocationBasedClassGenerator       = createClassGenerator(dontCheckLocations,  meetsTraitConstraints,  selectUsingWeightedRandom)
local wowOnlyAndSubZoneBasedClassGenerator        = createClassGenerator(checkAllLocations,   isWowClass,             selectUsingWeightedRandom)
local wowOnlyAndZoneBasedClassGenerator           = createClassGenerator(checkZonesOnly,      isWowClass,             selectUsingWeightedRandom)
local wowOnlyAndNoLocationBasedClassGenerator     = createClassGenerator(dontCheckLocations,  isWowClass,             selectUsingWeightedRandom)
local noContextAndSubZoneBasedClassGenerator      = createClassGenerator(checkAllLocations,   const(true),            selectUsingWeightedRandom)
local noContextAndZoneBasedClassGenerator         = createClassGenerator(checkZonesOnly,      const(true),            selectUsingWeightedRandom)
local noContextAndNoLocationBasedClassGenerator   = createClassGenerator(dontCheckLocations,  const(true),            selectUsingWeightedRandom)

function remakeClassGenerator()
  generateRandomClass = function() return "Error: generateRandomClass is not set" end
  
  if not NPCG_Options.locationSettings.zones then
    generateRandomClass = function(characterTemplate)
      return traitAndNoLocationBasedClassGenerator(characterTemplate)
      or noContextAndNoLocationBasedClassGenerator(characterTemplate) end
  elseif NPCG_Options.locationSettings.subZones then
    generateRandomClass = function(characterTemplate, zone, subZone)
      return traitAndSubZoneBasedClassGenerator(characterTemplate, zone, subZone)
      or noContextAndSubZoneBasedClassGenerator(characterTemplate, zone, subZone)
    end
  else
    generateRandomClass = function(characterTemplate, zone)
      return traitAndZoneBasedClassGenerator(characterTemplate, zone)
      or noContextAndZoneBasedClassGenerator(characterTemplate, zone)
    end
  end
end

--[[ LEGACY: generateRandomClass
-- generateRandomClass :: (Race a, Age a) => a -> Class
function generateRandomClass(characterTemplate)
  local selectionList = 
    -- Subzone-based selection.
    ((subzonesToClasses[GetZoneText()] or { })[GetSubZoneText()])
    -- Zone-based selection.
    or zonesToClasses[GetZoneText()]
    -- Default
    or classes
  
  return randomFrom(
    filterKeysByValue(
      function(val) return
        (
          contains(characterTemplate:getRace(),   val.races) or 
          contains("any", val.races)
        ) and (
          contains(characterTemplate:getAge(),     val.ages) or
          contains("any", val.ages)
        ) and (
          contains(characterTemplate:getFaith(), val.faiths) or
          contains("any", val.faiths)
        )
      end,
      selectionList
    )
  ) or randomClass()
end
]]--

function testClassesByType2()
  print "Testing classesByType2(\"rogue\")"
  local results = classesByType2("rogue")
  print "Keys"
  logTable(keys(results))
  print "Values"
  map(logTableOfPropertyLists, results)
  print "Ended testing"
end

-- End of File.