function searchF(f, s, xs)
  return filter(
    function(x) return string.match(f(x), s) end,
    xs)
end

-- ucFirst :: string -> string
-- Capitalizes the first letter of a string.
function ucFirst(s)
  return s:gsub("^%l", string.upper)
end

-- lcFirst :: string -> string
-- Lower-cases the first letter of a string.
function lcFirst(s)
  return s:gsub("^%u", string.lower)
end

-- upperCaseWordsHelper :: string -> string -> string
-- Capitalizes x, and concatenates with xs.
local function upperCaseWordsHelper(x, xs)
  return x:upper() .. xs:lower()
end

-- ucWords :: string -> string
-- Capitalizes every word in a string.
function ucWords(s)
  return s:gsub("(%a)([%w_']*)", upperCaseWordsHelper)
end

-- underscoreToSpace :: string -> string
-- Replaces every occurance of "_" by a single whitespace.
function underscoreToSpace(s)
  return s:gsub("_", " ")
end

-- doubleUnderscoreToApostrophe :: string -> string
-- Replaces every occurance of "__", two underscores, by "'", an apostrophe.
function doubleUnderscoreToApostrophe(s)
	return s:gsub("__", "'")
end

-- fixGender :: Gender -> string -> string
-- Replaces occurances of xir-, xim- and xhe-macros by their gender-specific counterparts.
function fixGender(gender, s)
  return xir(gender, xim(gender, (xhe(gender, s))))
end

function inverseFixGender(s)
  return inverseXir(inverseXim(inverseXhe(s)))
end

-- xir :: Gender -> string -> string
-- Replaces occurances of xir-macros by their gender-specific counterpart.
function xir(gender, s)
  if     gender ==        "male" then return s:gsub("%%xir",        "his") 
  elseif gender ==      "female" then return s:gsub("%%xir",        "her")
  elseif gender == "androgynous" then return s:gsub("%%xir", "his or her")
  else                                return s
	end
end

function inverseXir(s)
  local output = s
  output = output:gsub(       " his ", " %%xir ")
  output = output:gsub(       " her ", " %%xir ")
  output = output:gsub(" his or her ", " %%xir ")
  return output
end

-- xim :: Gender -> string -> string
-- Replaces occurance of xim-macros by their gender-specific counterpart.
function xim(gender, s)
  if     gender ==        "male" then return s:gsub("%%xim",        "him") 
  elseif gender ==      "female" then return s:gsub("%%xim",        "her")
  elseif gender == "androgynous" then return s:gsub("%%xim", "him or her")
  else                                return s
	end
end

function inverseXim(s)
  local output = s
  output = output:gsub(       " him ", " %%xim ")
  output = output:gsub(       " her ", " %%xim ")
  output = output:gsub(" him or her ", " %%xim ")
  return output
end

-- xhe :: Gender -> string -> string
-- Replaces occurance of xhe-macros by their gender-specific counterpart.
function xhe(gender, s)
  if     gender ==        "male" then return s:gsub("%%xhe",        "he") 
  elseif gender ==      "female" then return s:gsub("%%xhe",       "she")
  elseif gender == "androgynous" then return s:gsub("%%xhe", "he or she")
  else                                return s
	end
end

function inverseXhe(s)
  local output = s
  output = output:gsub(        " he ", " %%xhe ")
  output = output:gsub(       " she ", " %%xhe ")
  output = output:gsub( " he or she ", " %%xhe ")
  return output
end

-- aOrAn :: string -> string
-- Prefixes a word with either "a" or "an" based on the word.
function aOrAn(word)
  if string.find(string.sub(word, 1,1), "[AEIOUaeiou]") then return "an"
  else return "a" end
end

-- combine :: string -> string
-- Concatenates two strings even if either is nil.
function combine(x,y)
  return (x or "") .. (y or "")
end

-- combineWithSpace :: string -> string
-- Concatenates two strings with a whitespace in between. If either is nil or empty, that one will be ignores.
function combineWithSpace(x,y)
  if      x == nil or x == "" then  return y
  elseif  y == nil or y == "" then  return x
  else                              return x .. " " .. y end
end

function nilOrWhitespace(s)
  return s == nil or s == ""
end

function nilOrWhitespaceOr(s, t)
  return nilOrWhitespace(s) or (s == t)
end

--[[ KeyValuePair
function KeyValuePair(k, v)
	return { key = k, value = v }
end
]]--

-- getIncremental :: int -> (() -> int)
function getIncremental(start)
  local i = start
  return function()
    i = i + 1
    return i - 1
  end
end

-- log :: a -> a
-- Prints a value to the chat and returns the value.
function log(x)
  print(x)
  return x
end

-- logTable :: { k -> a } -> { k -> a }
-- Prints all keys and values of a table to the chat and returns the table.
function logTable(t)
  for k,v in pairs(t) do
    print("(" .. k .. ", " .. v .. ")")
  end
  return t
end

function logTableOfPropertyLists(t)
  for k,vs in pairs(t) do
    for vk, v in pairs(vs) do
      print(k .. " = (" .. vk .. ", " .. v .. ")")
    end
  end
  return t
end

function logArray(t)
  for i,v in ipairs(t) do
    print("(" .. i .. ": " .. v .. ")")
  end
  return t
end

function logTableMapped(f, t)
  return logTable(map(f, t))
end

--[[ liftToNilMonad
function liftToNilMonad(f)
  return 
    function(x)
      if x == nil then
        return nil
      else
        return f(x)
      end
    end  
end
]]--

--[[ errorMonadic
function errorMonadic(x, funcs)
  local y = x
  return foldr(
    function(k, f)
      if k == nil then
        return nil
      else
       return f(k) -- <-- function on k, k should be a table.
      end
    end,
    y,
    funcs
  )
end
]]--

-- eitherOr :: a -> a -> a
-- Returns either the first or the second argument randomly.
function eitherOrRandom(x, y)
  return eitherOr(random(1,2) == 1, x, y)
end

-- eitherOrP :: bool -> a -> a -> a
-- Returns either the second argument if p is true, or the third argument otherwise.
function eitherOr(p, x, y)
  if p then
    return x
  else
    return y
  end
end

-- switch :: bool -> a -> ... -> a
-- Takes a number of booleans and cases, and returns the first case where the boolean before it is true. 
function switch(...)
  local xs = {...}
  
  local first = nil
  local setFirst = true
  
  for i,x in ipairs(xs) do
    if     setFirst then 
      first    = x      -- Found the predicate.
      setFirst = false  -- Swap case to read as value.
    elseif    first then 
      return x        -- Found predicate is true! Return value x.
    else
      setFirst = true -- Predicate is false, capture the next predicate.
    end
  end
end

-- randomFrom :: { a } -> a
-- Returns a random element from a list.
-- Implicitly, if list is empty, nil will be returned.
function randomFrom(list)
	local c = count(list)
	for _,v in pairs(list) do
		rand = random(1, c)
		if rand == 1 then
			return v 
		else
			c = c - 1
		end
	end
end

-- weightedRandomFrom :: { a } -> (a -> int) -> a
-- Returns a random element from a list using a weight function.
-- Implicitly, if the list is empty, nil will be returned.
function weightedRandomFrom(list, weightFunction)
  local sumOfWeights = sumBy(list, weightFunction)

  -- If we consider items to be duplicated by their weight in a fictive list we can take the item at the random index.
  local randomIndex = random(1, sumOfWeights)
  
  local foundKey = findKey(
    function(x)
      if    x >= randomIndex  then  return true
      else                    randomIndex = randomIndex - x
                              return false end
    end,
    map(weightFunction, list)
  )
    
  return list[foundKey]
end

-- uniqueRandomsFrom :: int -> { a } -> { a }
-- Returns a number between 1 and maxReturns of random elements from a list.
function uniqueRandomsFrom(maxReturns, list)
  if maxReturns <= 0 then return { } end
  
  local output = { }
	local c = count(list)
	for _,v in pairs(list) do
		rand = random(1, c)
		if rand == 1 then
			table.insert(output, v) -- This line is changed to hopefully fix the bug >.>
      maxReturns = maxReturns - 1
      if maxReturns <= 0 then
        return output
      end
		else  -- Else clause jumbles up the c variable, this guarantees that there could be less returns than maxReturns.
			c = c - 1
		end
	end
  return output
end

-- uniqueRandomsFromCartesian :: { a } -> { b } -> a,b
-- Returns a random element of the first list, and one of the second list, ensuring they are not the same.
-- (x, y) | x <- xs and y <- ys and x ~= y 
function uniqueRandomsFromCartesian(xs, ys)
  local x = randomFrom(xs)
  local y = randomFrom(ys)
  while x == y and y ~= nil do -- This can loop infinitely.
    y = randomFrom(ys)
  end
  return x, y
end

-- End of File.