--[[ getOrDefault
function getOrDefault(list, key, default)
  return (list or { })[key] or default
end
]]

--[[ getOrNil
function getOrNil(list, key)
  return (list or { })[key]
end
]]

-------------------------------------------------------------------------------------
-- | Prelude | ----------------------------------------------------------------------
-------------------------------------------------------------------------------------

function id(x)
  return x
end

-- replicate :: int -> a -> { a }
-- Returns a list of length n where every element is x.
function replicate(n, x)
	local output = { }
	for i=1, n do table.insert(output, x) end
	return output
end

-- iterate :: (a -> ()) -> int -> int -> ()
-- Performs a certain action incrementally, starting from x, then x+1 until (inclusive) n.
function iterate(action, x, n)
  for i=x, n do
    action(i)
  end
end

-- forEach :: (a -> ()) -> { a } -> ()
-- Performs the same action on every element in a list.
function forEach(a, xs)
	for _,x in ipairs(xs) do
		a(x)
	end
end

--[[ forEachKey
function forEachKey(a, xs)
	for k,_ in pairs(xs) do
		a(k)
	end
end
]]

--[[ forEachKeyValuePair
function forEachKeyValuePair(a, xs)
	for k,v in pairs(xs) do
		a(k,v)
	end
end
]]

-- keys :: { k -> _ } -> { k }
-- Returns a list of keys from a table.
function keys(xs)
  local output = { }
  for k,_ in pairs(xs) do
    table.insert(output, k)
  end
  return output
end

--[[ values
function values(xs)
  local output = { }
  for _,v in pairs(xs) do
    table.insert(output, v)
  end
  return output
end
]]

--[[ keyValuePairs
function keyValuePairs(xs)
  local output = { }
  for k,v in pairs(xs) do
    table.insert(output, KeyValuePair(k,v))
  end
  return output
end
]]

-- map :: (a -> b) -> { k -> a } -> { k -> b }
-- Applies a function to every item in a table and returns the result.
-- Ordering is preserved.
function map(f, xs)
	local output = { }
	for k,v in pairs(xs) do
		output[k] = f(v)
	end
	return output
end

--[[ mapKeys
function mapKeys(f, xs)
  local output = { }
  for k,_ in pairs(xs) do
    table.insert(output, f(k))
  end
  return output
end
]]

--[[ mapKeyValuePairs
function mapKeyValuePairs(f, xs)
  local output = { }
  for k,v in pairs(xs) do
    table.insert(output, f(KeyValuePair(k,v)))
  end
  return output
end
]]

-- filter :: (a -> bool) -> { a } -> { b }
-- Returns a list of all elements in a list that satisfy a predicate.
function filter(p, xs)
	local output = { }
	for _,v in pairs(xs) do
		if p(v) then 
			table.insert(output, v)
		end
	end
	return output
end

function filter2(p, xs)
  local output = { }
	for k,v in pairs(xs) do
		if p(k) then 
			output[k] = v
		end
	end
	return output
end

-- filterKeysByValue :: (a -> bool) -> { k -> a } -> { k }
-- Returns a list of all keys in a list where the value satisfies a certain predicate.
function filterKeysByValue(p, xs)
  local output = { }
  for k,v in pairs(xs) do
    if p(v) then 
      table.insert(output, k)
    end
  end
  return output
end

-- foldr :: (a -> b -> b) -> b -> { a } -> b
-- Applies an accumulator function on every element of a given list.
function foldr(op, val, xs)
	for _,v in ipairs(xs) do
		val = op(val, v)
	end
	return val
end

function foldr2(op, val, xs)
	for _,v in pairs(xs) do
		val = op(val, v)
	end
	return val
end

--[[ foldrByKey
function foldrByKey(op, val, xs)
  for k,_ in pairs(xs) do
    val = op(val, k)
  end
  return val
end
]]

--[[ foldrByKeyValuePair
function foldrByKeyValuePair(op, val, xs)
  for k,v in pairs(xs) do
    val = op(val, KeyValuePair(k,v))
  end
  return val
end
]]

-- any :: (a -> bool) -> { a } -> bool
-- Returns whether any of the elements in a list satisfies a predicate.
function any(p, xs)
	for _,v in pairs(xs) do
		if p(v) then
			return true
		end
	end
	return false
end


function all(p, xs)
	return not any(
		function(x) return not p(x) end, 
		xs)
end


--[[ none
function none(p, xs)
	return not any(p, xs)
end
]]

-- contains :: a -> { a } -> bool
-- Returns whether a value is contained in a list.
function contains(val, xs)
	return any(
		function(x) return x == val end, 
		xs)
end

function containsAny(xs, values)
  return any(
    function(x) return contains(x, xs) end,
    values)
end

function containsAll(xs, values)
  return all(
    function(x) return contains(x, xs) end,
    values)
end

--[[
function containsKey(key, xs)
	for k,_ in pairs(xs) do
		if k == key then return true end
	end
	return false
end
]]

-- const :: a -> (b -> a)
-- Returns a function that always returns x.
function const(x)
  return function(...) return x end
end

-- invokeAll :: { (() -> ()) } -> ()
-- Invokes all actions in a list.
function invokeAll(xs)
  forEach(function(x) x() end, xs)
end

-- count :: { a } -> int
-- Counts the number of elements in a list.
function count(list)
	return foldr(
    function(accum, x) return accum + 1 end, 0, 
    list
  )
end

-- sum :: Num a => { a } -> a
-- Computes the sum of the items in the list.
-- Items must be numbers.
function sum(list)
  return foldr(
    function(accum, x) return accum + x end,
    0,
    list
  )
end

-- sumBy :: Num b => { a } -> (a -> b) -> b
-- Computes the sum of the items in the list, using a function to convert the items into numbers.
function sumBy(list, by)
  return foldr2(
    function(accum, x) return accum + by(x) end,
    0,
    list
  )
end

-- mergeTables :: { a } -> { a } -> { a }
--[[ Merges an arbitrary amount of arrays into one.
-- Ensures:
-- 1) ({1},{2}) = {1,2}
-- 2) ({ },{2}) = {  2}
-- 3) ({1},{ }) = {  1}
-- 4) (nil,{2}) = {  2}
-- 5) ({1},nil) = {  1} ]]
function mergeTables(...)
	local output = { }
	for _,t in pairs({...}) do  
		for _,v in pairs(t or { }) do -- <-- only takes values, indicating it's a list, but also uses pairs, which indicates it should be a table..
			table.insert(output, v)
		end
	end
	return output
end

function mergeTables2(...)
  local output = { }
  for _,t in ipairs({...}) do
    for k,v in pairs(t or { }) do
      output[k] = v
    end
  end
  return output
end

function mergeTablesWithoutDuplicates(...)
  local output = { }
	for _,t in pairs({...}) do  -- <-- only takes values, indicating it's a list, but also uses pairs, which indicates it should be a table..
		for _,v in pairs(t or { }) do
			if not contains(v, output) then
        table.insert(output, v)
      end
		end
	end
	return output
end

function mergeTablesWithoutDuplicates2(...)
  local output = { }
	for _,t in pairs({...}) do  -- <-- only takes values, indicating it's a list, but also uses pairs, which indicates it should be a table..
		for k,v in pairs(t or { }) do
			--if not contains(k, output) then
        output[k] = v
      --end
		end
	end
	return output
end

-- intersect :: { a } -> { a } -> { a } 
-- Returns the intersection between two lists, aka the elements that are both in xs and ys.
-- intersect( {1},   {1,1} ) returns {1}
-- intersect( {1,1}, {1}   ) returns {1,1}
function intersect(xs, ys)
  -- Rough way to calculate an intersection between two lists.
  local set = { }
  local output = { }
  for _,y in ipairs(ys) do
    set[y] = true
  end
  for _,x in ipairs(xs) do
    if set[x] then
      table.insert(output, x)
    end
  end
  return output
end

-- without :: { a } -> { a } -> { a }
-- Removes all occurances of certain values from a list.
function without(list, values)
  return filter(
    function(x) return not contains(x, values) end,
    list)
end

--[[function find(p, xs)
  forEach(
    function(x)
      if p(x)then
        return x
      end
    end,
    xs)
  return nil
end]]

function findKey(p, xs)
  for k,v in pairs(xs) do
    if p(v) then return k end
  end
  return nil
end

-- skipWhile :: (a -> bool) -> { a } -> { a }
-- Returns a list where all items until the predicate is satisfied are ignored. The element where the predicate is true is included.
function skipWhile(p, xs)
  local output = { }
  local found = false
  for _,v in ipairs(xs) do
    if p(v) then
      found = true
    end
    if found then 
      table.insert(output, v)
    end
  end
  return output
end

-- takeWhile :: (a -> bool) -> { a } -> { a }
-- Returns a list where all items after the predicate is satisfied are ignored. The element where the predicate is true is not included.
function takeWhile(p, xs)
  local output = { }
  local found = false
  for _,v in ipairs(xs) do
    if   p(v) then return output
    else           table.insert(output, v) 
    end
  end
  return output
end

--[[ flip
function flip(f)
  return function(x,y) return f(y,x) end
end
]]

function equals(x, y)
  return x == y
end

function composeEqualsFunction(y)
  return function(x) return x == y end
end

-- End of File.