--[[

Zum: A World of Warcraft (WoW) AddOn that utilizes the WoW Widget API
to make simple and lightweight enhancements to the WoW chat Graphical User Interface (GUI).

Author: Ümit Eronat
Date: 2018-05-12 (SI)
Version: 1.0.0

These enhancements include:
- Categorization of chats based on player names,
- Providing a modern "Messenger-like" chat GUI in chat (especially in
whisper/Battle.net whisper) frames,
- Clearing chat messages in a chat frame of this addon,
- Copying texts from chat message frames to the WoW editbox so that 
they can be copied/cut and pasted elsewhere,
- Utilizing the WoW chat customization options such as adding timestamps,
changing font size and color and changing frame sizes,
- Having a Look and Feel (LaF) that is as close to WoW's default LaF as possible,
- Still retaining the original chat functionality of WoW and being able to switch
between this addon's functionalities and WoW's in a very simple way.

Copyright (C) 2018  Ümit Eronat

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

--]]
_G["NAME_STRING_WIDTH"] = 100

local curTimestampFormat = "none"

local function extendChatFrameMinResize(id)
    cf = _G["ChatFrame" .. id]
    local minResW, minResH = 210, 150
    cf:SetMinResize(minResW, minResH)
    local w, h = cf:GetWidth(), cf:GetHeight()

    if (w < minResW + 1) then -- The "+ 1" is to account for the floating point error
        cf:SetWidth(minResW)
    end

    if (h < minResH + 1) then -- The "+ 1" is to account for the floating point error
        cf:SetHeight(minResH)
    end
end

local function getMsgFrameMinHeight(msgFrame, width)
    local _1, fontHeight, _2 = msgFrame.msgString:GetFont()
    local offset = 14
    if (msgFrame.timeString and msgFrame.timeString:IsShown()) then
        offset = offset + fontHeight * 1.25
    end
    return math.min(
        150,
        offset +
            math.ceil((msgFrame.msgString:GetStringWidth()) / (math.max(10, width - 50))) *
                (fontHeight + msgFrame.msgString:GetSpacing() + 2)
    )
end

local function msgFrame_OnSizeChanged(self, width, height)
    local minHeight = getMsgFrameMinHeight(self, width)
    self:SetHeight(minHeight)

    local msgFramesContent = self:GetParent()
    local msgFrames = msgFramesContent.msgFrames

    if (self == msgFrames[1]) then
        msgFramesContent.curHeight = 0
    end

    msgFramesContent.curHeight = msgFramesContent.curHeight + minHeight

    if (self == msgFrames[#msgFrames]) then
        local needsScroll = false
        if (msgFramesContent:GetHeight() < msgFramesContent.curHeight) then
            needsScroll = true
        end

        msgFramesContent:SetHeight(msgFramesContent.curHeight)
        if (needsScroll) then
            local scrollFrame = msgFramesContent:GetParent()
            scrollFrame:SetVerticalScroll(scrollFrame:GetVerticalScrollRange())
        end
    end
end

local function getPlayerToken(frameId, playerName, type)
    return frameId .. "_" .. playerName .. "_" .. type
end

local function switchChatFrameView(cf, viewState, ...)
    local cflvf = _G[cf:GetName() .. "ListViewFrame"]

    if (viewState == "listView") then
        local playerTokenFrame = ...

        CloseDropDownMenus(1)
        cf.viewState = viewState
        cf.FontStringContainer:Hide()
        cf:SetScrollOffset(0)
        cf:SetScrollOffset(0)
        cflvf:Show()
        if (playerTokenFrame) then
            playerTokenFrame:Hide()
            UIFrameFlashStop(playerTokenFrame.playerListFrame.glowFrame)
            playerTokenFrame.playerListFrame.isBeingViewed = false
        end
    elseif (viewState == "playerChatView") then
        local playerTokenFrame = ...

        cf.viewState = viewState
        cf.curPlayerTokenFrame = playerTokenFrame
        cf:SetScrollOffset(0)
        cflvf:Hide()
        if (playerTokenFrame) then
            UIFrameFlashStop(playerTokenFrame.playerListFrame.glowFrame)
            playerTokenFrame:Show()
            playerTokenFrame.scrollFrame:SetVerticalScroll(playerTokenFrame.scrollFrame:GetVerticalScrollRange())
        end
    else -- "normalView"
        cf.viewState = "normalView"
        cf.FontStringContainer:Show()
        cf:SetScrollOffset(0)
        cflvf:Hide()
        FCF_StopAlertFlash(cf)

        if (cf.curPlayerTokenFrame) then
            cf.curPlayerTokenFrame.playerListFrame.isBeingViewed = false
            cf.curPlayerTokenFrame:Hide()
        end
    end
end

local function switchToListView(switchBtn)
    switchBtn.icon:SetTexture("Interface/AddOns/Zum/images/unchecked")

    local cf = _G["ChatFrame" .. switchBtn.id]
    switchChatFrameView(cf, "listView")
end

local function switchToNormalView(switchBtn)
    switchBtn.icon:SetTexture("Interface/AddOns/Zum/images/checked")

    local cf = _G["ChatFrame" .. switchBtn.id]
    switchChatFrameView(cf, "normalView")
end

local function MsgFrameDropDownMenu_CopyMsg(self)
    DEFAULT_CHAT_FRAME.editBox:Show()
    DEFAULT_CHAT_FRAME.editBox:SetText(self.value)
    DEFAULT_CHAT_FRAME.editBox:HighlightText()
    DEFAULT_CHAT_FRAME.editBox:SetFocus()
end

local function MsgFrameDropDownMenu_Initialize(dropDown)
    local info = nil

    info = UIDropDownMenu_CreateInfo()
    info.text = "Message"
    info.notClickable = 1
    info.isTitle = 1
    info.notCheckable = 1
    UIDropDownMenu_AddButton(info)

    info = UIDropDownMenu_CreateInfo()
    info.text = "Copy to Editbox"
    info.func = MsgFrameDropDownMenu_CopyMsg
    if dropDown.msg then
        info.value = dropDown.msg.text
    end
    info.notCheckable = 1
    UIDropDownMenu_AddButton(info)
end

local function MsgFrameDropDownMenu_OnLoad(self)
    UIDropDownMenu_Initialize(self, MsgFrameDropDownMenu_Initialize, "MENU")
    UIDropDownMenu_SetButtonWidth(self, 50)
    UIDropDownMenu_SetWidth(self, 50)
end

local function registerChatEvents(self)
    self:RegisterEvent("UPDATE_CHAT_COLOR")
    self:RegisterEvent("CHAT_MSG_SAY")
    self:RegisterEvent("CHAT_MSG_YELL")
    self:RegisterEvent("CHAT_MSG_WHISPER")
    self:RegisterEvent("CHAT_MSG_WHISPER_INFORM")
    self:RegisterEvent("CHAT_MSG_PARTY")
    self:RegisterEvent("CHAT_MSG_PARTY_LEADER")
    self:RegisterEvent("CHAT_MSG_RAID")
    self:RegisterEvent("CHAT_MSG_RAID_LEADER")
    self:RegisterEvent("CHAT_MSG_RAID_WARNING")
    self:RegisterEvent("CHAT_MSG_BATTLEGROUND")
    self:RegisterEvent("CHAT_MSG_BATTLEGROUND_LEADER")
    self:RegisterEvent("CHAT_MSG_GUILD")
    self:RegisterEvent("CHAT_MSG_OFFICER")
    self:RegisterEvent("CHAT_MSG_AFK")
    self:RegisterEvent("CHAT_MSG_DND")
    self:RegisterEvent("CHAT_MSG_IGNORED")
    self:RegisterEvent("CHAT_MSG_BN_WHISPER")
    self:RegisterEvent("CHAT_MSG_BN_WHISPER_INFORM")

    hooksecurefunc("SetChatWindowSize", handleSetChatWindowFontSize)
    hooksecurefunc("SetCVar", handleSetCVar)
    hooksecurefunc("FCF_UnDockFrame", handleFCF_UnDockFrame)
end

local function scrollChatFrame(cf, direction)
    if (cf.viewState == "normalView") then
        if (direction == "up") then
            cf:OrigScrollUp()
        elseif (direction == "down") then
            cf:OrigScrollDown()
        else -- "bottom"
            cf:OrigScrollToBottom()
        end
    elseif (cf.viewState == "listView") then
        local cflvf = _G[cf:GetName() .. "ListViewFrame"]

        if (direction == "up") then
            cflvf:SetVerticalScroll(math.max(0, cflvf:GetVerticalScroll() - 50))
        elseif (direction == "down") then
            cflvf:SetVerticalScroll(math.min(cflvf:GetVerticalScrollRange(), cflvf:GetVerticalScroll() + 50))
        else -- "bottom"
            cflvf:SetVerticalScroll(cflvf:GetVerticalScrollRange())
        end
    else -- "playerChatView"
        local ptfsf = cf.curPlayerTokenFrame.scrollFrame

        if (direction == "up") then
            ptfsf:SetVerticalScroll(math.max(0, ptfsf:GetVerticalScroll() - 10))
        elseif (direction == "down") then
            ptfsf:SetVerticalScroll(math.min(ptfsf:GetVerticalScrollRange(), ptfsf:GetVerticalScroll() + 10))
        else -- "bottom"
            ptfsf:SetVerticalScroll(ptfsf:GetVerticalScrollRange())
        end
    end
end

local function initListViewFrame(id)
    local cf = _G["ChatFrame" .. id]

    cf.OrigScrollUp = cf.ScrollUp
    cf.ScrollUp = function()
        scrollChatFrame(cf, "up")
    end

    cf.OrigScrollDown = cf.ScrollDown
    cf.ScrollDown = function()
        scrollChatFrame(cf, "down")
    end

    cf.OrigScrollToBottom = cf.ScrollToBottom
    cf.ScrollToBottom = function()
        scrollChatFrame(cf, "bottom")
    end

    local name = "ChatFrame" .. id .. "ListViewFrame"
    local cflvf = _G[name]
    if not cflvf then
        cflvf = CreateFrame("ScrollFrame", name, cf, "UIPanelScrollFrameTemplate")
        cflvf.id = id
        cflvf:SetFrameStrata("BACKGROUND")
        cflvf:SetSize(cf:GetWidth(), cf:GetHeight())
        cflvf:SetPoint("TOPLEFT", cf, 0, 0)
        cflvf:SetPoint("RIGHT", cf, 0, 0)
        cflvf:SetPoint("BOTTOM", cf, 0, 0)
        cflvf:SetBackdrop(
            {
                bgFile = "Interface/Tooltips/UI-Tooltip-Background",
                edgeSize = 16
            }
        )
        cflvf:SetBackdropColor(0, 0, 0, 0)

        local sb = _G[name .. "ScrollBar"]
        local sbub = _G[name .. "ScrollBarScrollUpButton"]
        local sbdb = _G[name .. "ScrollBarScrollDownButton"]
        sb:Hide()
        sb.Show = function()
        end
        sbub:Hide()
        sbub.Show = function()
        end
        sbdb:Hide()
        sbdb.Show = function()
        end
    end

    name = "ChatFrame" .. id .. "ListViewFrameContent"
    cflvfc = _G[name]
    if not cflvfc then
        cflvfc = CreateFrame("Frame", name, cflvf)
        cflvf:SetScrollChild(cflvfc)
        cflvfc.id = id
        cflvfc.playerListFrames = {}
        cflvfc.curTopListFrame = nil
        cflvfc:SetFrameStrata("BACKGROUND")
        cflvfc:SetSize(cflvf:GetWidth(), 0)
    end
end

local function initSwitchButton(id)
    local cfbf = _G["ChatFrame" .. id .. "ButtonFrame"]
    local name = "ChatFrame" .. id .. "SwitchButton"
    local switchBtn = _G[name]
    if not switchBtn then
        switchBtn = CreateFrame("CheckButton", name, cfbf, "UIPanelSquareButton")
        switchBtn.id = id
        switchBtn.icon:SetTexCoord(0, 1, 0, 1)
        switchBtn:SetChecked(true)
        switchToNormalView(switchBtn)
        switchBtn:SetFrameStrata("LOW")
        switchBtn:SetSize(28, 26)
        switchBtn:SetPoint("TOPLEFT", cfbf, 0, 0)
        switchBtn:SetScript("OnClick", SwitchButton_OnClick)
    end
end

local function initMsgFrameOptionsMenu(frameId)
    local cf = _G["ChatFrame" .. frameId]
    if not cf.msgFrameDropDownMenu then
        FloatingChatFrame_Update(frameId)
        name = cf:GetName() .. "ListViewPlayerMessageMenu"
        cf.MsgFrameDropDownMenu = CreateFrame("Frame", name, cf, "UIDropDownMenuTemplate")
        MsgFrameDropDownMenu_OnLoad(cf.MsgFrameDropDownMenu)
        cf.MsgFrameDropDownMenu:SetScript(
            "OnShow",
            function(self)
                MsgFrameDropDownMenu_OnLoad(self)
            end
        )
    end
end

function Zum_OnEvent(self, event, ...)
    if (eventCallbacks[event]) then
        eventCallbacks[event](self, event, ...)
    else
        -- Default, for registered but unhandled events
    end
end

function registerEvents(self)
    self:RegisterEvent("PLAYER_LOGIN")
    registerChatEvents(self)
end

function initChatFrames()
    StaticPopupDialogs["CLEAR_CHAT"] = {
        text = "Are you sure you want to clear this chat?",
        button1 = "Yes",
        button2 = "No",
        OnAccept = function(self, playerTokenFrame)
            handleClearChat(playerTokenFrame)
        end,
        timeout = 0,
        whileDead = true,
        hideOnEscape = true,
        preferredIndex = 3 -- avoid some UI taint
    }

    curTimestampFormat = GetCVar("showTimestamps")

    for id = 1, 10 do
        if (id ~= 2) then
            initListViewFrame(id)
            initSwitchButton(id)
            initMsgFrameOptionsMenu(id)
        end
        extendChatFrameMinResize(id)
    end
end

function SwitchButton_OnClick(self, button, down)
    PlaySound(SOUNDKIT.U_CHAT_SCROLL_BUTTON)
    if (self:GetChecked()) then
        switchToNormalView(self)
    else
        switchToListView(self)
    end
end

function handleFCF_UnDockFrame(cf)
    if (cf:IsShown()) then
        extendChatFrameMinResize(cf:GetID())
    end
end

function handleSetCVar(cvar, value, scriptCVar)
    if (cvar == "showTimestamps") then
        handleShowTimestampsChange(value)
    end
end

function handleShowTimestampsChange(format)
    curTimestampFormat = format

    for id = 1, 10 do
        if (id ~= 2) then
            local cf = _G["ChatFrame" .. id]
            local cflvfc = _G["ChatFrame" .. id .. "ListViewFrameContent"]
            if cflvfc then
                local fontFile, fontHeight, _unused = cf:GetFont()
                for i, playerListFrame in pairs(cflvfc.playerListFrames) do
                    local playerTokenFrame = playerListFrame.playerTokenFrame

                    for i, msgFrame in pairs(playerTokenFrame.scrollFrame.contentFrame.msgFrames) do
                        local chatTypeInfo = ChatTypeInfo[msgFrame.msg.chatType]
                        if (curTimestampFormat ~= "none") then
                            if not msgFrame.timeString then
                                msgFrame.timeString = msgFrame:CreateFontString(nil, "OVERLAY")
                                msgFrame.timeString:SetFont(fontFile, fontHeight - 2)
                                msgFrame.timeString:SetJustifyV("TOP")
                                msgFrame.timeString:SetWidth(playerTokenFrame:GetWidth() - _G["NAME_STRING_WIDTH"])
                                msgFrame.timeString:SetTextColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
                            end

                            if (msgFrame.isSentByUs) then
                                msgFrame.timeString:SetJustifyH("RIGHT")
                                msgFrame.timeString:SetPoint("TOPLEFT", 10, -6)
                                msgFrame.timeString:SetPoint("RIGHT", -15, 0)
                            else
                                msgFrame.timeString:SetJustifyH("LEFT")
                                msgFrame.timeString:SetPoint("TOPLEFT", 18, -6)
                                msgFrame.timeString:SetPoint("RIGHT", -10, 0)
                            end
                            msgFrame.timeString:SetText(date(curTimestampFormat, msgFrame.time))
                            msgFrame.timeString:Show()
                        else
                            if msgFrame.timeString then
                                msgFrame.timeString:Hide()
                            end
                        end

                        if (msgFrame.isSentByUs) then
                            if (msgFrame.timeString and msgFrame.timeString:IsShown()) then
                                msgFrame.msgString:SetPoint("TOP", msgFrame.timeString, 0, -fontHeight - 3)
                            else
                                msgFrame.msgString:SetPoint("TOP", 0, -7)
                            end
                        else
                            if (msgFrame.timeString and msgFrame.timeString:IsShown()) then
                                msgFrame.msgString:SetPoint("TOP", msgFrame.timeString, 0, -fontHeight - 3)
                            else
                                msgFrame.msgString:SetPoint("TOP", 0, -7)
                            end
                        end
                        msgFrame:SetHeight(getMsgFrameMinHeight(msgFrame, msgFrame:GetWidth()))
                    end
                end
            end
        end
    end
end

function handleSetChatWindowFontSize(frameId, fontSize)
    local cflvfc = _G["ChatFrame" .. frameId .. "ListViewFrameContent"]
    if cflvfc then
        for i, playerListFrame in pairs(cflvfc.playerListFrames) do
            local fontFile, _unused, _unused2 = playerListFrame.playerNameString:GetFont()
            local playerTokenFrame = playerListFrame.playerTokenFrame

            playerListFrame.playerNameString:SetFont(fontFile, fontSize)
            playerListFrame.lastMessagesFrame:SetFont(fontFile, fontSize)
            playerTokenFrame.playerNameString:SetFont(fontFile, fontSize)

            for i, msgFrame in pairs(playerTokenFrame.scrollFrame.contentFrame.msgFrames) do
                if msgFrame.timeString then
                    msgFrame.timeString:SetFont(fontFile, fontSize - 2)

                    if (msgFrame.isSentByUs) then
                        if (msgFrame.timeString and msgFrame.timeString:IsShown()) then
                            msgFrame.msgString:SetPoint("TOP", msgFrame.timeString, 0, -fontSize - 2)
                        else
                            msgFrame.msgString:SetPoint("TOP", 0, -6)
                        end
                    else
                        if (msgFrame.timeString and msgFrame.timeString:IsShown()) then
                            msgFrame.msgString:SetPoint("TOP", msgFrame.timeString, 0, -fontSize - 2)
                        else
                            msgFrame.msgString:SetPoint("TOP", 0, -6)
                        end
                    end
                end

                msgFrame.msgString:SetFont(fontFile, fontSize)
                msgFrame:SetWidth(getMsgFrameMinHeight(msgFrame, msgFrame:GetWidth()))
            end
        end
    end
end

function handleUpdateChatColor(self, event, ...)
    local chatType, r, g, b = ...

    for id = 1, 10 do
        if (id ~= 2) then
            local cflvfc = _G["ChatFrame" .. id .. "ListViewFrameContent"]

            if cflvfc then
                for i, playerListFrame in pairs(cflvfc.playerListFrames) do
                    local playerTokenFrame = playerListFrame.playerTokenFrame

                    if
                        (chatType == "AFK" or chatType == "DND" or chatType == "IGNORED" or
                            playerTokenFrame.chatType == chatType)
                     then
                        if (playerTokenFrame.chatType == chatType) then
                            playerListFrame:SetBackdropBorderColor(r, g, b)
                            if (chatType == "BN_WHISPER") then
                                playerListFrame.playerNameString:SetTextColor(r, g, b)
                                playerTokenFrame.playerNameString:SetTextColor(r, g, b)
                            end
                            playerListFrame.lastMessagesFrame:SetTextColor(r, g, b)
                        end

                        for i, msgFrame in pairs(playerTokenFrame.scrollFrame.contentFrame.msgFrames) do
                            if (msgFrame.msg.chatType == chatType) then
                                msgFrame:SetBackdropBorderColor(r, g, b)
                                msgFrame.msgString:SetTextColor(r, g, b)
                                if (msgFrame.timeString) then
                                    msgFrame.timeString:SetTextColor(r, g, b)
                                end
                            end
                        end
                    end
                end
            end
        end
    end
end

function handleClearChat(playerTokenFrame)
    if playerTokenFrame then
        for i, msgFrame in pairs(playerTokenFrame.scrollFrame.contentFrame.msgFrames) do
            msgFrame:Hide()
        end

        playerTokenFrame.scrollFrame.contentFrame.msgFrames = {}
        playerTokenFrame.scrollFrame.contentFrame.curFrameIndex = 1
        playerTokenFrame.scrollFrame.contentFrame.curHeight = 0
        playerTokenFrame.scrollFrame.contentFrame:SetHeight(0)
        local playerListFrame = playerTokenFrame.playerListFrame
        playerListFrame.lastMessagesFrame:Clear()
    end
end

function handleClosePlayerListFrame(playerListFrame)
    local cflvfc = playerListFrame:GetParent()

    if (playerListFrame ~= cflvfc.curTopListFrame) then
        if (playerListFrame.downerListFrame) then
            playerListFrame.downerListFrame:SetPoint("TOPLEFT", playerListFrame.upperListFrame, "BOTTOMLEFT", 0, 0)
            playerListFrame.upperListFrame.downerListFrame = playerListFrame.downerListFrame
            playerListFrame.downerListFrame.upperListFrame = playerListFrame.upperListFrame
        else
            playerListFrame.upperListFrame.downerListFrame = nil
        end
    else
        if (playerListFrame.downerListFrame) then
            playerListFrame.downerListFrame:SetPoint("TOPLEFT", cflvfc, 0, 0)
            playerListFrame.downerListFrame.upperListFrame = nil
            cflvfc.curTopListFrame = playerListFrame.downerListFrame
        else
            cflvfc.curTopListFrame = nil
        end
    end

    playerListFrame.upperListFrame = nil
    playerListFrame.downerListFrame = nil
    playerListFrame:ClearAllPoints()
    playerListFrame:Hide()
end

function handleChatMessages(self, event, ...)
    -- Check each message here
    if (strsub(event, 1, 8) == "CHAT_MSG") then
        local msg = {}

        local arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12, arg13, arg14, arg15 = ...
        local chatType = strsub(event, 10)

        if (chatType ~= "BN_WHISPER") then
            msg.playerClassId = select(2, GetPlayerInfoByGUID(arg12))
        else
            msg.presenceId = arg13
            msg.playerClassId = "none"
        end

        if (chatType == "AFK") then
            arg1 = "[Player is away: " .. arg1 .. "]"
            msg.isWhisperWarning = true
        elseif (chatType == "DND") then
            arg1 = "[Player doesn't wish to be disturbed: " .. arg1 .. "]"
            msg.isWhisperWarning = true
        elseif (chatType == "IGNORED") then
            arg1 = "[Player is ignoring you.]"
            msg.isWhisperWarning = true
        else
            msg.lineId = arg11
            msg.isWhisperWarning = false
        end

        msg.text = arg1
        msg.playerName = arg2
        msg.chatType = chatType
        msg.isSentByUs = false

        for id = 1, 10 do
            if (id ~= 2) then
                local curFrameGroups = {GetChatWindowMessages(id)}
                local cf = _G["ChatFrame" .. id]

                if (chatType == "AFK" or chatType == "DND" or chatType == "IGNORED") then
                    addMessageToChatFrame(cf, msg)
                else
                    for i, v in pairs(curFrameGroups) do
                        if (v == chatType) then
                            addMessageToChatFrame(cf, msg)
                        end
                    end
                end
            end
        end
    end
end

function handleWhisperInform(self, event, ...)
    -- Check whisper information here
    if (strsub(event, 1, 8) == "CHAT_MSG") then
        local msg = {}

        local arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12, arg13, arg14, arg15 = ...
        local chatType = strsub(event, 10)
        chatType = strsub(chatType, 1, string.find(chatType, "INFORM") - 2)

        if (chatType ~= "BN_WHISPER") then
            msg.playerClassId = select(2, GetPlayerInfoByGUID(arg12))
        else
            msg.presenceId = arg13
            msg.playerClassId = "none"
        end

        msg.text = arg1
        msg.playerName = arg2
        msg.chatType = chatType
        msg.isSentByUs = true
        msg.isWhisperWarning = false

        for id = 1, 10 do
            if (id ~= 2) then
                local curFrameGroups = {GetChatWindowMessages(id)}
                local cf = _G["ChatFrame" .. id]

                for i, v in pairs(curFrameGroups) do
                    if (v == chatType) then
                        addMessageToChatFrame(cf, msg)
                    end
                end
            end
        end
    end
end

function addMessageToChatFrame(cf, msg)
    local cflvf = _G[cf:GetName() .. "ListViewFrame"]

    local tokenChatType = msg.chatType
    if (msg.chatType == "AFK" or msg.chatType == "DND" or msg.chatType == "IGNORED") then
        tokenChatType = "WHISPER"
    end

    local chatTypeInfo = ChatTypeInfo[msg.chatType]
    local tokenChatTypeInfo = ChatTypeInfo[tokenChatType]
    local playerToken = getPlayerToken(cf:GetID(), msg.playerName, tokenChatType)

    local playerInfo = {}
    playerInfo.playerToken = playerToken
    playerInfo.playerName = msg.playerName
    playerInfo.playerClassId = msg.playerClassId
    playerInfo.presenceId = msg.presenceId

    local name = "ChatFrame" .. playerToken .. "PlayerTokenFrame"
    local playerTokenFrame = _G[name]
    if not playerTokenFrame then
        -- If this message is a whisper warning ("Player is afk, ignoring you, or busy") then don't create a new frame
        -- Otherwise, every chat frame will add this warning message
        if msg.isWhisperWarning then
            return nil
        end

        playerTokenFrame = createPlayerTokenFrame(name, cf, tokenChatType, playerInfo)
    end

    local fontFile, fontHeight, _unused = cf:GetFont()
    msg.fontFile = fontFile
    msg.fontHeight = fontHeight

    addMessageToPlayerTokenFrame(playerTokenFrame, msg)

    name = "ChatFrame" .. playerToken .. "PlayerListFrame"
    local playerListFrame = _G[name]
    if (not playerListFrame) then
        playerListFrame = createPlayerListFrame(name, cf, playerTokenFrame, playerInfo)
    elseif (not playerListFrame:IsShown()) then
        local cflvfc = _G[cflvf:GetName() .. "Content"]
        if (cflvfc.curTopListFrame) then
            local prevTopListFrame = cflvfc.curTopListFrame
            prevTopListFrame:SetPoint("TOPLEFT", playerListFrame, "BOTTOMLEFT", 0, 0)
            prevTopListFrame.upperListFrame = playerListFrame
            playerListFrame.downerListFrame = prevTopListFrame
        end
        playerListFrame:SetPoint("CENTER", 0, 0)
        playerListFrame:SetPoint("TOPLEFT", cflvfc, 0, 0)
        playerListFrame:SetPoint("RIGHT", cflvf, 0, 0)
        playerListFrame:Show()
        cflvfc.curTopListFrame = playerListFrame
    end

    if (not msg.isSentByUs) then
        playerListFrame.lastMessagesFrame:AddMessage(msg.text)
        playerListFrame.lastMessagesFrame:ScrollToBottom()

        if (tokenChatType == "WHISPER" or tokenChatType == "BN_WHISPER") then
            if (not playerListFrame.isBeingViewed) then
                UIFrameFlash(playerListFrame.glowFrame, 1.0, 1.0, -1, false, 0, 0)
                PlaySound(SOUNDKIT.TELL_MESSAGE)
            end
        end
    end
end

function addMessageToPlayerTokenFrame(playerTokenFrame, msg)
    local chatTypeInfo = ChatTypeInfo[msg.chatType]

    name =
        "ChatFrame" ..
        playerTokenFrame.playerToken .. "PlayerFrameMsgFrame" .. playerTokenFrame.scrollFrame.contentFrame.curFrameIndex

    local newMsgFrame = CreateFrame("Button", name, playerTokenFrame.scrollFrame.contentFrame)
    newMsgFrame.msg = msg
    newMsgFrame.time = time()
    newMsgFrame.isSentByUs = msg.isSentByUs
    newMsgFrame:SetFrameStrata("LOW")
    newMsgFrame:SetHyperlinksEnabled(true)
    newMsgFrame:SetWidth(playerTokenFrame:GetWidth() - _G["NAME_STRING_WIDTH"])
    if (playerTokenFrame.scrollFrame.contentFrame.msgFrames[#playerTokenFrame.scrollFrame.contentFrame.msgFrames]) then
        newMsgFrame:SetPoint(
            "TOP",
            playerTokenFrame.scrollFrame.contentFrame.msgFrames[#playerTokenFrame.scrollFrame.contentFrame.msgFrames],
            "BOTTOM",
            0,
            0
        )
    else
        newMsgFrame:SetPoint("TOP", playerTokenFrame.scrollFrame.contentFrame, 0, 0)
    end

    if (msg.isSentByUs) then
        newMsgFrame:SetBackdrop(
            {
                bgFile = "Interface/Tooltips/UI-Tooltip-Background",
                edgeFile = "Interface/Addons/Zum/images/UI-RightBubble-Border",
                edgeSize = 14,
                insets = {left = 4, right = 12, top = 4, bottom = 4}
            }
        )
        newMsgFrame:SetPoint("LEFT", playerTokenFrame.scrollFrame, 60, 0)
        newMsgFrame:SetPoint("RIGHT", playerTokenFrame.scrollFrame, 0, 0)
    else
        newMsgFrame:SetBackdrop(
            {
                bgFile = "Interface/Tooltips/UI-Tooltip-Background",
                edgeFile = "Interface/Addons/Zum/images/UI-LeftBubble-Border",
                edgeSize = 14,
                insets = {left = 12, right = 4, top = 4, bottom = 4}
            }
        )
        newMsgFrame:SetPoint("LEFT", playerTokenFrame.scrollFrame, 0, 0)
        newMsgFrame:SetPoint("RIGHT", playerTokenFrame.scrollFrame, -60, 0)
    end

    newMsgFrame:SetBackdropBorderColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    newMsgFrame:SetBackdropColor(0, 0, 0, 0.5)
    newMsgFrame:SetScript("OnSizeChanged", msgFrame_OnSizeChanged)
    newMsgFrame:SetScript(
        "OnHyperlinkClick",
        function(...)
            ChatFrame_OnHyperlinkShow(...)
        end
    )

    newMsgFrame:SetScript(
        "OnEnter",
        function(self, ...)
            self:SetBackdropColor(0, 0, 0, 0.8)
        end
    )

    newMsgFrame:SetScript(
        "OnLeave",
        function(self, ...)
            self:SetBackdropColor(0, 0, 0, 0.5)
        end
    )

    newMsgFrame:RegisterForClicks("LeftButtonDown", "RightButtonUp")
    newMsgFrame:SetScript(
        "OnClick",
        function(self, button, down)
            if (button == "RightButton") then
                local cf = playerTokenFrame:GetParent()
                cf.MsgFrameDropDownMenu.msg = self.msg
                ToggleDropDownMenu(1, nil, cf.MsgFrameDropDownMenu, "cursor", 0, 0)
            elseif (button == "LeftButton") then
                CloseDropDownMenus(1)
            end
        end
    )

    if (curTimestampFormat ~= "none") then
        if not newMsgFrame.timeString then
            newMsgFrame.timeString = newMsgFrame:CreateFontString(nil, "OVERLAY")
            newMsgFrame.timeString:SetFont(msg.fontFile, msg.fontHeight - 2)
            newMsgFrame.timeString:SetJustifyV("TOP")
            newMsgFrame.timeString:SetWidth(playerTokenFrame:GetWidth() - _G["NAME_STRING_WIDTH"])
            newMsgFrame.timeString:SetTextColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
            if (msg.isSentByUs) then
                newMsgFrame.timeString:SetJustifyH("RIGHT")
                newMsgFrame.timeString:SetPoint("TOPLEFT", 10, -6)
                newMsgFrame.timeString:SetPoint("RIGHT", -15, 0)
            else
                newMsgFrame.timeString:SetJustifyH("LEFT")
                newMsgFrame.timeString:SetPoint("TOPLEFT", 18, -6)
                newMsgFrame.timeString:SetPoint("RIGHT", -10, 0)
            end

            newMsgFrame.timeString:SetText(date(curTimestampFormat, newMsgFrame.time))
        end
    end

    newMsgFrame.msgString = newMsgFrame:CreateFontString(nil, "OVERLAY")
    newMsgFrame.msgString:SetFont(msg.fontFile, msg.fontHeight)
    if (msg.isSentByUs) then
        newMsgFrame.msgString:SetJustifyH("RIGHT")
        newMsgFrame.msgString:SetPoint("LEFT", 10, 0)
        newMsgFrame.msgString:SetPoint("BOTTOMRIGHT", -18, 6)
        if (newMsgFrame.timeString) then
            newMsgFrame.msgString:SetPoint("TOP", newMsgFrame.timeString, 0, -msg.fontHeight - 3)
        else
            newMsgFrame.msgString:SetPoint("TOP", 0, -7)
        end
    else
        newMsgFrame.msgString:SetJustifyH("LEFT")
        newMsgFrame.msgString:SetPoint("LEFT", 18, 0)
        newMsgFrame.msgString:SetPoint("BOTTOMRIGHT", -10, 6)
        if (newMsgFrame.timeString) then
            newMsgFrame.msgString:SetPoint("TOP", newMsgFrame.timeString, 0, -msg.fontHeight - 3)
        else
            newMsgFrame.msgString:SetPoint("TOP", 0, -7)
        end
    end
    newMsgFrame.msgString:SetShadowColor(0, 0, 0, 1)
    newMsgFrame.msgString:SetShadowOffset(1, -1)
    newMsgFrame.msgString:SetJustifyV("TOP")
    newMsgFrame.msgString:SetWordWrap(true)
    newMsgFrame.msgString:SetNonSpaceWrap(true)
    newMsgFrame.msgString:SetWidth(playerTokenFrame:GetWidth() - _G["NAME_STRING_WIDTH"])
    newMsgFrame.msgString:SetText(msg.text)
    newMsgFrame.msgString:SetTextColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    newMsgFrame:SetHeight(getMsgFrameMinHeight(newMsgFrame, newMsgFrame:GetWidth()))

    if (not msg.isSentByUs) then
        newMsgFrame.glowFrame = CreateFrame("Frame", nil, newMsgFrame)
        newMsgFrame.glowFrame:SetPoint("TOPLEFT", 0, 0)
        newMsgFrame.glowFrame:SetPoint("BOTTOMRIGHT", 0, 0)
        newMsgFrame.glowFrame:SetFrameStrata("MEDIUM")
        newMsgFrame.glowFrame:SetBackdrop(
            {
                edgeFile = "Interface/AddOns/Zum/images/UI-LeftBubble-GlowBorder",
                edgeSize = 14,
                alphaMode = "ADD",
                insets = {left = 12, right = 4, top = 4, bottom = 4}
            }
        )
        newMsgFrame.glowFrame:SetBackdropBorderColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
        newMsgFrame.glowFrame:Hide()

        UIFrameFlash(newMsgFrame.glowFrame, 1.0, 1.0, 4, false, 0, 0)
    end

    table.insert(playerTokenFrame.scrollFrame.contentFrame.msgFrames, newMsgFrame)
    playerTokenFrame.scrollFrame.contentFrame:SetWidth(playerTokenFrame.scrollFrame:GetWidth())
    playerTokenFrame.scrollFrame.contentFrame:SetHeight(
        playerTokenFrame.scrollFrame.contentFrame:GetHeight() + newMsgFrame:GetHeight()
    )

    playerTokenFrame.scrollFrame.contentFrame.curFrameIndex = #playerTokenFrame.scrollFrame.contentFrame.msgFrames + 1
    playerTokenFrame.scrollFrame:UpdateScrollChildRect()
    playerTokenFrame.scrollFrame:SetVerticalScroll(playerTokenFrame.scrollFrame:GetVerticalScrollRange())
end

function createPlayerTokenFrame(name, cf, chatType, playerInfo)
    local cflvf = _G[cf:GetName() .. "ListViewFrame"]
    local chatTypeInfo = ChatTypeInfo[chatType]

    playerTokenFrame = CreateFrame("Button", name, cf)
    playerTokenFrame.chatType = chatType
    playerTokenFrame.playerToken = playerInfo.playerToken
    playerTokenFrame.playerName = playerInfo.playerName
    playerTokenFrame.playerClassId = playerInfo.playerClassId
    if (playerInfo.presenceId) then
        playerTokenFrame.presenceId = playerInfo.presenceId
    end
    playerTokenFrame:SetFrameStrata("BACKGROUND")
    playerTokenFrame:SetSize(cf:GetWidth(), cf:GetHeight())
    playerTokenFrame:SetPoint("TOPLEFT", cf, 0, 0)
    playerTokenFrame:SetPoint("BOTTOMRIGHT", cf, 0, 5)
    playerTokenFrame:SetBackdrop(
        {
            bgFile = "Interface/Tooltips/UI-Tooltip-Background"
        }
    )
    playerTokenFrame:SetBackdropBorderColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    playerTokenFrame:SetBackdropColor(0, 0, 0, 0)

    playerTokenFrame:RegisterForClicks("AnyUp", "AnyDown")
    playerTokenFrame:SetScript(
        "OnClick",
        function(self, button, down)
            if (button == "RightButton") then
                switchChatFrameView(cf, "listView", self)
            end
        end
    )

    playerTokenFrame:SetPropagateKeyboardInput(true)
    playerTokenFrame:SetScript(
        "OnKeyDown",
        function(self, keyDown)
            if (keyDown == "PAGEUP") then
                self.scrollFrame:SetVerticalScroll(math.max(0, self.scrollFrame:GetVerticalScroll() - 50))
            elseif (keyDown == "PAGEDOWN") then
                self.scrollFrame:SetVerticalScroll(
                    math.min(self.scrollFrame:GetVerticalScroll() + 50, self.scrollFrame:GetVerticalScrollRange())
                )
            end
        end
    )

    playerTokenFrame:SetScript(
        "OnMouseWheel",
        function(self, delta)
            local ptfsf = self.scrollFrame
            if (delta > 0) then
                ptfsf:SetVerticalScroll(math.max(0, ptfsf:GetVerticalScroll() - 20))
            elseif (delta < 0) then
                ptfsf:SetVerticalScroll(math.min(ptfsf:GetVerticalScrollRange(), ptfsf:GetVerticalScroll() + 20))
            end
        end
    )

    name = playerTokenFrame:GetName() .. "NameStringFrame"
    playerTokenFrame.playerNameStringFrame = CreateFrame("Frame", name, playerTokenFrame)
    playerTokenFrame.playerNameStringFrame:SetPoint("TOPLEFT", playerTokenFrame, 0, 0)
    playerTokenFrame.playerNameStringFrame:SetWidth(_G["NAME_STRING_WIDTH"])
    playerTokenFrame.playerNameStringFrame:SetHeight(50)
    playerTokenFrame.playerNameStringFrame:SetBackdrop(
        {bgFile = "Interface/AddOns/Zum/images/UI-PlayerName-Background2"}
    )
    playerTokenFrame.playerNameStringFrame:SetBackdropColor(0, 0, 0, 0.6)
    playerTokenFrame.playerNameStringFrame:SetScript(
        "OnMouseDown",
        function(self, button)
            local ptf = self:GetParent()

            if (button == "LeftButton") then
                ChatFrame_OpenChat("/w " .. ptf.playerName .. " ", cf)
            elseif (button == "RightButton") then
                if (chatType == "BN_WHISPER") then
                    FriendsFrame_ShowBNDropdown(ptf.playerName, true, 0, ptf.chatType, cf, nil, ptf.presenceId)
                else
                    FriendsFrame_ShowDropdown(ptf.playerName, true, 0, ptf.chatType, cf, nil, false)
                end
            end
        end
    )

    local fontFile, fontHeight, _unused = cf:GetFont()

    playerTokenFrame.playerNameString = playerTokenFrame.playerNameStringFrame:CreateFontString(nil, "OVERLAY")
    playerTokenFrame.playerNameString:SetFont(fontFile, fontHeight)
    playerTokenFrame.playerNameString:SetPoint("TOPLEFT", playerTokenFrame.playerNameStringFrame, 10, 0)
    playerTokenFrame.playerNameString:SetPoint("BOTTOMRIGHT", playerTokenFrame.playerNameStringFrame, 0, 0)
    if (playerTokenFrame.playerClassId == "none") then
        playerTokenFrame.playerNameString:SetTextColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    else
        local r, g, b =
            RAID_CLASS_COLORS[playerTokenFrame.playerClassId].r,
            RAID_CLASS_COLORS[playerTokenFrame.playerClassId].g,
            RAID_CLASS_COLORS[playerTokenFrame.playerClassId].b
        playerTokenFrame.playerNameString:SetTextColor(r, g, b)
    end
    playerTokenFrame.playerNameString:SetShadowColor(0, 0, 0, 1)
    playerTokenFrame.playerNameString:SetShadowOffset(1, -1)
    playerTokenFrame.playerNameString:SetWidth(playerTokenFrame.playerNameStringFrame:GetWidth())
    playerTokenFrame.playerNameString:SetHeight(playerTokenFrame.playerNameStringFrame:GetHeight())
    playerTokenFrame.playerNameString:SetJustifyH("LEFT")
    playerTokenFrame.playerNameString:SetJustifyV("MIDDLE")

    playerTokenFrame.playerNameString:SetText(playerTokenFrame.playerName)

    name = playerTokenFrame:GetName() .. "MsgFramesBase"
    playerTokenFrame.scrollFrame = CreateFrame("ScrollFrame", name, playerTokenFrame, "UIPanelScrollFrameTemplate")
    playerTokenFrame.scrollFrame:SetFrameStrata("BACKGROUND")
    playerTokenFrame.scrollFrame:SetPoint("TOP", playerTokenFrame, 0, 0)
    playerTokenFrame.scrollFrame:SetPoint("LEFT", playerTokenFrame.playerNameStringFrame, "RIGHT", 0, 0)
    playerTokenFrame.scrollFrame:SetPoint("RIGHT", playerTokenFrame, 0, 0)
    playerTokenFrame.scrollFrame:SetPoint("BOTTOM", playerTokenFrame, 0, 5)
    playerTokenFrame.scrollFrame:SetBackdrop(
        {
            bgFile = "Interface/Tooltips/UI-Tooltip-Background",
            edgeSize = 16
        }
    )
    playerTokenFrame.scrollFrame:SetBackdropColor(0, 0, 0, 0)
    playerTokenFrame.scrollFrame.OrigSetVerticalScroll = playerTokenFrame.scrollFrame.SetVerticalScroll
    playerTokenFrame.scrollFrame.SetVerticalScroll = function(self, scroll)
        local cf = self:GetParent():GetParent()
        if (cf.viewState == "playerChatView") then
            -- This has got to be one of the ugliest workarounds ever...
            if (self:GetVerticalScrollRange() - 25 > scroll) then
                cf:SetScrollOffset(1)
            else
                cf:SetScrollOffset(0)
            end
        end

        return self:OrigSetVerticalScroll(scroll)
    end

    playerTokenFrame.scrollFrame:SetScript(
        "OnMouseWheel",
        function(self, delta)
            local sb = _G[self:GetName() .. "ScrollBar"]
            if (delta > 0) then
                self:SetVerticalScroll(math.max(0, self:GetVerticalScroll() - 20))
            elseif (delta < 0) then
                self:SetVerticalScroll(math.min(self:GetVerticalScrollRange(), self:GetVerticalScroll() + 20))
            end
        end
    )

    playerTokenFrame.scrollFrame:SetScript(
        "OnSizeChanged",
        function(self, width, height)
        end
    )

    local sb = _G[name .. "ScrollBar"]
    local sbub = _G[name .. "ScrollBarScrollUpButton"]
    local sbdb = _G[name .. "ScrollBarScrollDownButton"]
    sb:Hide()
    sb.Show = function()
    end
    sbub:Hide()
    sbub.Show = function()
    end
    sbdb:Hide()
    sbdb.Show = function()
    end

    name = playerTokenFrame:GetName() .. "MsgFrames"
    playerTokenFrame.scrollFrame.contentFrame = CreateFrame("Frame", name, playerTokenFrame.scrollFrame)
    playerTokenFrame.scrollFrame.contentFrame.msgFrames = {}
    playerTokenFrame.scrollFrame.contentFrame.curFrameIndex = 1
    playerTokenFrame.scrollFrame.contentFrame.curHeight = 0
    playerTokenFrame.scrollFrame:SetScrollChild(playerTokenFrame.scrollFrame.contentFrame)
    playerTokenFrame.scrollFrame.contentFrame:SetFrameStrata("BACKGROUND")
    playerTokenFrame.scrollFrame.contentFrame:SetSize(playerTokenFrame.scrollFrame:GetWidth(), 0)
    playerTokenFrame.scrollFrame.contentFrame:SetPoint("TOPLEFT", playerTokenFrame.scrollFrame, 0, 0)
    playerTokenFrame.scrollFrame.contentFrame:SetScript(
        "OnUpdate",
        function(self, elapsedSecs)
            self:SetWidth(self:GetParent():GetWidth())
        end
    )

    name = playerTokenFrame:GetName() .. "ClearChatButton"
    playerTokenFrame.clearChatButton = CreateFrame("Button", name, playerTokenFrame, "UIPanelButtonGrayTemplate")
    playerTokenFrame.clearChatButton:SetFrameStrata("BACKGROUND")
    playerTokenFrame.clearChatButton:SetSize(playerTokenFrame.playerNameStringFrame:GetWidth() - 10, 22)
    playerTokenFrame.clearChatButton:SetPoint("TOPLEFT", playerTokenFrame.playerNameStringFrame, "BOTTOMLEFT", 5, -5)
    playerTokenFrame.clearChatButton:SetText("Clear Chat")
    playerTokenFrame.clearChatButton:SetNormalTexture("Interface/AddOns/Zum/images/UI-ChatButton-Normal")
    playerTokenFrame.clearChatButton:SetPushedTexture("Interface/AddOns/Zum/images/UI-ChatButton-Down")
    playerTokenFrame.clearChatButton:SetHighlightTexture("Interface/AddOns/Zum/images/UI-ChatButton-Highlight")
    playerTokenFrame.clearChatButton:SetNormalFontObject("GameFontNormalSmall")
    playerTokenFrame.clearChatButton:SetHighlightFontObject("GameFontHighlightSmall")
    playerTokenFrame.clearChatButton:SetScript(
        "OnClick",
        function(self, button, down)
            local ccDlg = StaticPopup_Show("CLEAR_CHAT")
            ccDlg.data = self:GetParent()
        end
    )

    name = playerTokenFrame:GetName() .. "BackButton"
    playerTokenFrame.backButton = CreateFrame("Button", name, playerTokenFrame, "UIPanelButtonGrayTemplate")
    playerTokenFrame.backButton:SetFrameStrata("BACKGROUND")
    playerTokenFrame.backButton:SetSize(playerTokenFrame.playerNameStringFrame:GetWidth() - 10, 22)
    playerTokenFrame.backButton:SetPoint("TOPLEFT", playerTokenFrame.clearChatButton, "BOTTOMLEFT", 0, 0)
    playerTokenFrame.backButton:SetText("Back")
    playerTokenFrame.backButton:SetNormalTexture("Interface/AddOns/Zum/images/UI-ChatButton-Normal")
    playerTokenFrame.backButton:SetPushedTexture("Interface/AddOns/Zum/images/UI-ChatButton-Down")
    playerTokenFrame.backButton:SetHighlightTexture("Interface/AddOns/Zum/images/UI-ChatButton-Highlight")
    playerTokenFrame.backButton:SetNormalFontObject("GameFontNormalSmall")
    playerTokenFrame.backButton:SetHighlightFontObject("GameFontHighlightSmall")
    playerTokenFrame.backButton:SetScript(
        "OnClick",
        function(self, button, down)
            switchChatFrameView(cf, "listView", self:GetParent())
        end
    )

    return playerTokenFrame
end

function createPlayerListFrame(name, cf, playerTokenFrame, playerInfo)
    local cflvf = _G[cf:GetName() .. "ListViewFrame"]
    local cflvfc = _G[cf:GetName() .. "ListViewFrameContent"]
    local chatTypeInfo = ChatTypeInfo[playerTokenFrame.chatType]

    playerListFrame = CreateFrame("Button", name, cflvfc)
    playerListFrame.playerTokenFrame = playerTokenFrame
    playerTokenFrame.playerListFrame = playerListFrame
    playerTokenFrame:Hide()
    playerListFrame:SetFrameStrata("BACKGROUND")
    playerListFrame:SetSize(cflvfc:GetWidth(), 32)
    playerListFrame:SetPoint("CENTER", 0, 0)
    playerListFrame:SetPoint("RIGHT", cflvf, 0, 0)
    playerListFrame:SetPoint("TOPLEFT", cflvfc, 0, 0)
    if (cflvfc.curTopListFrame) then
        local prevTopListFrame = cflvfc.curTopListFrame
        prevTopListFrame:SetPoint("TOPLEFT", playerListFrame, "BOTTOMLEFT", 0, 0)
        prevTopListFrame.upperListFrame = playerListFrame
        playerListFrame.downerListFrame = prevTopListFrame
    end
    cflvfc.curTopListFrame = playerListFrame

    playerListFrame:SetBackdrop(
        {
            bgFile = "Interface/Tooltips/UI-Tooltip-Background",
            edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
            edgeSize = 12,
            insets = {left = 2, right = 2, top = 2, bottom = 2}
        }
    )
    playerListFrame:SetBackdropBorderColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    playerListFrame:SetBackdropColor(0, 0, 0, 0.3)

    playerListFrame:SetScript(
        "OnEnter",
        function(self, ...)
            self:SetBackdropColor(0, 0, 0, 0.6)
        end
    )

    playerListFrame:SetScript(
        "OnLeave",
        function(self, ...)
            self:SetBackdropBorderColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
            self:SetBackdropColor(0, 0, 0, 0.3)
        end
    )

    playerListFrame:SetScript(
        "OnMouseDown",
        function(self, ...)
            self:SetBackdropColor(0, 0, 0, 0.9)
        end
    )

    playerListFrame:SetScript(
        "OnMouseUp",
        function(self, ...)
            self:SetBackdropColor(0, 0, 0, 0.6)
        end
    )

    playerListFrame:SetScript(
        "OnClick",
        function(self, button, down)
            if (button == "LeftButton") then
                self.isBeingViewed = true
                switchChatFrameView(cf, "playerChatView", self.playerTokenFrame)
            end
        end
    )

    table.insert(cflvfc.playerListFrames, playerListFrame)
    cflvfc:SetHeight(cflvfc:GetHeight() + playerListFrame:GetHeight())
    cflvf:UpdateScrollChildRect()

    playerListFrame.glowFrame = CreateFrame("Frame", nil, playerListFrame)
    playerListFrame.glowFrame:SetPoint("TOPLEFT", 0, 0)
    playerListFrame.glowFrame:SetPoint("BOTTOMRIGHT", 0, 0)
    playerListFrame.glowFrame:SetBackdrop(
        {
            edgeFile = "Interface/AddOns/Zum/images/UI-PlayerListFrame-GlowBorder",
            edgeSize = 12,
            alphaMode = "ADD",
            insets = {left = 2, right = 2, top = 2, bottom = 2}
        }
    )
    playerListFrame.glowFrame:SetBackdropBorderColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    playerListFrame.glowFrame:Hide()

    local fontFile, fontHeight, _unused = cf:GetFont()

    playerListFrame.playerNameString = playerListFrame:CreateFontString(nil, "OVERLAY")
    playerListFrame.playerNameString:SetFont(fontFile, fontHeight)
    playerListFrame.playerNameString:SetPoint("TOP", playerListFrame, 0, 0)
    playerListFrame.playerNameString:SetPoint("BOTTOM", playerListFrame, 0, 0)
    playerListFrame.playerNameString:SetPoint("LEFT", playerListFrame, 15, 0)
    if (playerInfo.playerClassId == "none") then
        playerListFrame.playerNameString:SetTextColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    else
        local r, g, b =
            RAID_CLASS_COLORS[playerInfo.playerClassId].r,
            RAID_CLASS_COLORS[playerInfo.playerClassId].g,
            RAID_CLASS_COLORS[playerInfo.playerClassId].b
        playerListFrame.playerNameString:SetTextColor(r, g, b)
    end
    playerListFrame.playerNameString:SetShadowColor(0, 0, 0, 1)
    playerListFrame.playerNameString:SetShadowOffset(1, -1)
    playerListFrame.playerNameString:SetWidth(_G["NAME_STRING_WIDTH"] + 30)
    playerListFrame.playerNameString:SetHeight(playerListFrame:GetHeight())
    playerListFrame.playerNameString:SetJustifyH("LEFT")
    playerListFrame.playerNameString:SetJustifyV("MIDDLE")

    playerListFrame.playerNameString:SetText(playerInfo.playerName)

    name = playerListFrame:GetName() .. "CloseButton"
    playerListFrame.closeButton = CreateFrame("Button", name, playerListFrame)
    playerListFrame.closeButton:SetPoint("TOPRIGHT", playerListFrame, -6, -6)
    playerListFrame.closeButton:SetPoint("BOTTOMRIGHT", playerListFrame, -6, 6)
    playerListFrame.closeButton:SetWidth(20)
    playerListFrame.closeButton:SetNormalTexture("Interface/AddOns/Zum/images/UI-PlayerListFrame-CloseButtonNormal")
    playerListFrame.closeButton:SetHighlightTexture(
        "Interface/AddOns/Zum/images/UI-PlayerListFrame-CloseButtonHighlight"
    )
    playerListFrame.closeButton:SetPushedTexture("Interface/AddOns/Zum/images/UI-PlayerListFrame-CloseButtonPushed")
    playerListFrame.closeButton:RegisterForClicks("LeftButtonDown", "LeftButtonUp")
    playerListFrame.closeButton:SetScript(
        "OnClick",
        function(self, button, down)
            if (not down) then
                handleClosePlayerListFrame(self:GetParent())
            end
        end
    )

    name = playerListFrame:GetName() .. "LastMessagesFrame"
    playerListFrame.lastMessagesFrame = CreateFrame("ScrollingMessageFrame", name, playerListFrame)
    playerListFrame.lastMessagesFrame:SetFont(fontFile, fontHeight)
    playerListFrame.lastMessagesFrame:SetPoint("TOPLEFT", playerListFrame, _G["NAME_STRING_WIDTH"] + 55, -5)
    playerListFrame.lastMessagesFrame:SetPoint("BOTTOM", playerListFrame, 0, 10)
    playerListFrame.lastMessagesFrame:SetPoint("RIGHT", playerListFrame.closeButton, "LEFT", 0, 0)
    playerListFrame.lastMessagesFrame:SetTextColor(chatTypeInfo.r, chatTypeInfo.g, chatTypeInfo.b)
    playerListFrame.lastMessagesFrame:SetShadowColor(0, 0, 0, 1)
    playerListFrame.lastMessagesFrame:SetShadowOffset(1, -1)
    playerListFrame.lastMessagesFrame:SetHeight(playerListFrame:GetHeight() - 10)
    playerListFrame.lastMessagesFrame:SetJustifyH("LEFT")
    playerListFrame.lastMessagesFrame:SetInsertMode("bottom")
    playerListFrame.lastMessagesFrame:SetMaxLines(10)
    playerListFrame.lastMessagesFrame:SetFading(false)
    playerListFrame.lastMessagesFrame:SetIndentedWordWrap(false)

    return playerListFrame
end

do
    eventCallbacks = {
        ["PLAYER_LOGIN"] = function()
            initChatFrames()
        end,
        ["CHAT_MSG_SAY"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_YELL"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_WHISPER"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_PARTY"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_PARTY_LEADER"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_RAID"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_RAID_LEADER"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_RAID_WARNING"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_BATTLEGROUND"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_BATTLEGROUND_LEADER"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_GUILD"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_OFFICER"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_AFK"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_DND"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_IGNORED"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_BN_WHISPER"] = function(self, event, ...)
            handleChatMessages(self, event, ...)
        end,
        ["CHAT_MSG_WHISPER_INFORM"] = function(self, event, ...)
            handleWhisperInform(self, event, ...)
        end,
        ["CHAT_MSG_BN_WHISPER_INFORM"] = function(self, event, ...)
            handleWhisperInform(self, event, ...)
        end,
        ["UPDATE_CHAT_COLOR"] = function(self, event, ...)
            local chatType = ...
            if (chatColorEventCallbacks[chatType]) then
                chatColorEventCallbacks[chatType](self, event, ...)
            end
        end
    }
end

do
    chatColorEventCallbacks = {
        ["SAY"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["YELL"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["WHISPER"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["PARTY"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["PARTY_LEADER"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["RAID"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["RAID_LEADER"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["RAID_WARNING"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["BATTLEGROUND"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["BATTLEGROUND_LEADER"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["GUILD"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["OFFICER"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["AFK"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["DNG"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["IGNORED"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end,
        ["BN_WHISPER"] = function(self, event, ...)
            handleUpdateChatColor(self, event, ...)
        end
    }
end
