-- Set up variables used globally within the addon
local CooldownFrameContainer
local TBCooldownFrame, TBCooldown, TBCooldownTextFrame, TBCooldownFrameFS
local SIWCooldownFrame, SIWCooldown, SIWCooldownTextFrame, SIWCooldownFrameFS
local BCooldownFrame, BCooldown, BCooldownTextFrame, BCooldownFrameFS
local JRCooldownFrame, JRCooldown, JRCooldownTextFrame, JRCooldownFrameFS
local BTCooldownFrame, BTCooldown, BTCooldownTextFrame, BTCooldownFrameFS
local GMCooldownFrame, GMCooldown, GMCooldownTextFrame, GMCooldownFrameFS
local BuffMessage
local Buffs
local BuffsOrdered
local CooldownText
local EventFrame
local Initialised = false

-- Cooldown frame container. This one holds the graphical buff represantations. We cannot initialise the text/icon frames directly after PLAYER_LOGIN because their saved positioning will then get lost.
CooldownFrameContainer = CreateFrame("Frame", "TTB_CooldownFrameContainer", UIParent)
CooldownFrameContainer:SetSize(310, 60)
CooldownFrameContainer:SetPoint("CENTER")
CooldownFrameContainer:SetBackdrop({
	bgFile = [[Interface\DialogFrame\UI-DialogBox-Background]],
	tile = true, tileSize = 16, edgeSize = 16,
	insets = { left = 3, right = 3, top = 5, bottom = 3 }
})
CooldownFrameContainer:Hide()

-- Prepare movable
CooldownFrameContainer:SetMovable(true)
CooldownFrameContainer:RegisterForDrag("LeftButton")
CooldownFrameContainer:SetScript("OnDragStart", CooldownFrameContainer.StartMoving)

-- Let the container "snap" to the edges of the screen when it overlaps them
CooldownFrameContainer:SetScript("OnDragStop", function(self, event, ...)
	CooldownFrameContainer:StopMovingOrSizing()
	local x = CooldownFrameContainer:GetLeft()
	local y = CooldownFrameContainer:GetBottom()
	if(x < 0) then
		x = 0
	elseif(x + CooldownFrameContainer:GetWidth() > GetScreenWidth()) then
		x = GetScreenWidth() - CooldownFrameContainer:GetWidth()
	end
	if(y < 0) then
		y = 0
	elseif(y + CooldownFrameContainer:GetHeight() > GetScreenHeight()) then
		y = GetScreenHeight() - CooldownFrameContainer:GetHeight()
	end
	CooldownFrameContainer:ClearAllPoints()
	CooldownFrameContainer:SetPoint("BOTTOMLEFT", x, y)
end)

-- MessageFrame to display textual output of buffs
BuffMessage = CreateFrame("MessageFrame", "TTB_BuffMessage", TestFrame)
BuffMessage:SetSize(70, 100)
BuffMessage:SetPoint("CENTER")
BuffMessage:SetBackdrop({
	bgFile = [[Interface\DialogFrame\UI-DialogBox-Background]],
	tile = true, tileSize = 16, edgeSize = 16,
	insets = { left = 3, right = 3, top = 5, bottom = 3 }
})

-- Prepare movable
BuffMessage:SetMovable(true)
BuffMessage:RegisterForDrag("LeftButton")
BuffMessage:SetScript("OnDragStart", BuffMessage.StartMoving)

-- Let the container "snap" to the edges of the screen when it overlaps them. We need to multiply screen values by UIParent:GetEffectiveScale() here, because this MessageFrame is affected by UI scaling.
BuffMessage:SetScript("OnDragStop", function(self, event, ...)
	BuffMessage:StopMovingOrSizing()
	local x = BuffMessage:GetLeft()
	local y = BuffMessage:GetBottom()
	if(x < 0) then
		x = 0
	elseif(x + BuffMessage:GetWidth() > GetScreenWidth() * UIParent:GetEffectiveScale()) then
		x = GetScreenWidth() * UIParent:GetEffectiveScale() - BuffMessage:GetWidth()
	end
	if(y < 0) then
		y = 0
	elseif(y + BuffMessage:GetHeight() > GetScreenHeight() * UIParent:GetEffectiveScale()) then
		y = GetScreenHeight() * UIParent:GetEffectiveScale() - BuffMessage:GetHeight()
	end
	BuffMessage:ClearAllPoints()
	BuffMessage:SetPoint("BOTTOMLEFT", x, y)
end)

BuffMessage:SetFading(false)
BuffMessage:SetFontObject("GameFontNormalSmall")
BuffMessage:SetJustifyH("LEFT")
BuffMessage:SetJustifyV("CENTER")
BuffMessage:SetSpacing("2")
BuffMessage:Hide()

--[[ This function creates the icons which are in the CooldownFrameContainer.
		@param frameName			Name of the frame (rather cosmetic)
		@param posLeft				Position from the left of the icon box
		@param texture				Link to the texture file to use for the icon

		@return WrapperFrame		Wrapper frame for the cooldown animation
		@return CooldownFrame		Cooldown animation frame to be placed on the WrapperFrame
		@return WrapperTextFrame	Wrapper frame to hold the remaining duration of the buff
		@return WrapperTextFrameFS	Font string to be displayed on the WrapperTextFrame
]]--
function CreateBuffFrame(frameName, posLeft, textureFile)
	local WrapperFrame = CreateFrame("Frame", frameName, CooldownFrameContainer)
	WrapperFrame:SetSize(40, 40)
	WrapperFrame:SetPoint("LEFT", posLeft, 0)
	WrapperFrame:SetAlpha(.25)
	local texture = WrapperFrame:CreateTexture()
	texture:SetAllPoints()
	texture:SetTexture(textureFile)
	local CooldownFrame = CreateFrame("Cooldown", frameName .. "Cooldown", WrapperFrame, "CooldownFrameTemplate")
	CooldownFrame:SetReverse(true)
	CooldownFrame:SetSize(40, 40)
	CooldownFrame:SetHideCountdownNumbers(true)
	local WrapperTextFrame = CreateFrame("Frame", frameName .. "Text", CooldownFrameContainer)
	WrapperTextFrame:SetSize(40, 40)
	WrapperTextFrame:SetPoint("LEFT", posLeft, 0)
	WrapperTextFrame:SetFrameStrata("TOOLTIP")
	local WrapperTextFrameFS = WrapperTextFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
	WrapperTextFrameFS:SetFont("Fonts\\FRIZQT__.TTF", 18, "OUTLINE")
	WrapperTextFrameFS:SetPoint("CENTER")
	WrapperTextFrameFS:SetText("")
	WrapperTextFrameFS:SetTextColor(1, 1, 1, 1)

	return WrapperFrame, CooldownFrame, WrapperTextFrame, WrapperTextFrameFS
end

-- This function updates the remaining buff duration in icon mode
local function onUpdate(self, elapsed)
	if(TTBDB.iconmode) then
		for key, value in ipairs(BuffsOrdered) do
			if(Buffs[value]["active"] == true) then
				for i = 1, 40 do
					local name, rank, icon, count, dispelType, duration, expires, caster, isStealable, nameplateShowPersonal, spellID, canApplyAura, isBossDebuff, _, nameplateShowAll, timeMod, value1, value2, value3 = UnitBuff("player" , i)
					spellID = tostring(spellID)
					if(spellID == value) then
						Buffs[value]["fontstring"]:SetText(math.ceil(expires - GetTime()))
						Buffs[value]["fontstring"]:SetAlpha(1)
					end
				end
			else
				Buffs[value]["fontstring"]:SetText("")
			end
		end
	end
end

-- Main Event listener
EventFrame = CreateFrame("Frame")
EventFrame:RegisterEvent("ADDON_LOADED")
EventFrame:SetScript("OnEvent", function(self, event, ...)
	if(event == "ADDON_LOADED" and ... == "TrollTheBones") then
		-- Only load addon if current class is rogue and spec is outlaw
		classDisplayName, class, classID = UnitClass("player")
		if(classID == 4) then
			-- Set event handler for talent switch
			EventFrame:RegisterEvent("ACTIVE_TALENT_GROUP_CHANGED")
			-- Initialize saved variables if called for the first time
			SetStandardValues()
			if(IsLoggedIn()) then
				if(GetSpecialization() == 2) then
					initAddon()
				end
			else
				EventFrame:RegisterEvent('PLAYER_LOGIN')
			end
		end
	end

	if(event == "PLAYER_LOGIN") then
		if(GetSpecialization() == 2) then
			initAddon()
		end
	end

	-- If we respec, check if spec is outlaw and enable/disable the addon
	if(event == "ACTIVE_TALENT_GROUP_CHANGED") then
		if(GetSpecialization() == 2) then
			if(Initialised == false) then
				initAddon()
			end
			EventFrame:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
			EventFrame:RegisterEvent("ONUPDATE")
			EventFrame:RegisterEvent("UNIT_AURA")
			EventFrame:RegisterEvent("PLAYER_REGEN_ENABLED")
			EventFrame:RegisterEvent("PLAYER_REGEN_DISABLED")
		else
			EventFrame:UnregisterEvent("UNIT_SPELLCAST_SUCCEEDED")
			EventFrame:UnregisterEvent("ONUPDATE")
			EventFrame:UnregisterEvent("UNIT_AURA")
			EventFrame:UnregisterEvent("PLAYER_REGEN_ENABLED")
			EventFrame:UnregisterEvent("PLAYER_REGEN_DISABLED")
			if(CooldownFrameContainer ~= nil) then
				CooldownFrameContainer:Hide()
			end
			if(BuffMessage ~= nil) then
				BuffMessage:Hide()
			end
		end
	end

	-- Clear the MessageFrame/set every cooldown icon to inactive if no buff is up
	local BuffsActive = true
	if(event == "UNIT_AURA") then
		local hasBuff = false
		for i = 1, 40 do
			local name, rank, icon, count, dispelType, duration, expires, caster, isStealable, nameplateShowPersonal, spellID, canApplyAura, isBossDebuff, _, nameplateShowAll, timeMod, value1, value2, value3 = UnitBuff("player" , i)
			spellID = tostring(spellID)
			if(Buffs[spellID] ~= nil) then
				hasBuff = true
				BuffsActive = true
			end
		end
		if(hasBuff == false) then
			BuffMessage:Clear()
			BuffsActive = false
			for key, value in ipairs(BuffsOrdered) do
				Buffs[value]["frame"]:SetAlpha(.25)
				Buffs[value]["fontstring"]:SetText("")
			end
			-- Hide the output frames if the player left combat and did not explicitly show the frames via /ttb showicons/showtextbox
			if(UnitAffectingCombat("player") == false and not TTBDB.showicons) then
				if(CooldownFrameContainer:IsShown()) then
					CooldownFrameContainer:Hide()
				end
			end
			if(UnitAffectingCombat("player") == false and not TTBDB.showtextbox) then
				if(BuffMessage:IsShown()) then
					BuffMessage:Hide()
				end
			end
		end
	end

	-- Show textbox/iconbox if entering combat
	if(event == "PLAYER_REGEN_DISABLED" and TTBDB.textboxmode == true and BuffMessage:IsShown() == false) then
		UIFrameFadeIn(BuffMessage, 1, 0, 1)
	end
	if(event == "PLAYER_REGEN_DISABLED" and TTBDB.iconmode == true and CooldownFrameContainer:IsShown() == false) then
		UIFrameFadeIn(CooldownFrameContainer, 1, 0, 1)
	end
	
	if(event == "UNIT_SPELLCAST_SUCCEEDED") then
		local unit = ...
		local unitID, spell, rank, lineID, spellID = select(1, ...)
		if(unit ~= "player" or spellID ~= 193316) then -- Check if player rolled the bones: return if not
			return
		end
		-- Show textbox/iconbox if entering combat
		if(TTBDB.textboxmode == true and BuffMessage:IsShown() == false) then
			UIFrameFadeIn(BuffMessage, 1, 0, 1)
		end
		if(TTBDB.iconmode == true and CooldownFrameContainer:IsShown() == false) then
			UIFrameFadeIn(CooldownFrameContainer, 1, 0, 1)
		end
		local amount = 0 -- Number of buffs active
		TTBDB.rolls = TTBDB.rolls + 1
		TTBDB.lastfive = TTBDB.lastfive + 1
		-- Set all to inactive
		for buff in pairs(Buffs) do
			Buffs[buff]["active"] = false
			Buffs[buff]["red"] = 1
			Buffs[buff]["green"] = 1
			Buffs[buff]["blue"] = 1
		end
		-- Check if buffs are active and update buff table. Walk through all player buffs, since ID detection is necessary on the German client. Jolly Roger and Loaded Dice (Golden Trait) are named the same there.
		for i = 1, 40 do
			local name, rank, icon, count, dispelType, duration, expires, caster, isStealable, nameplateShowPersonal, spellID, canApplyAura, isBossDebuff, _, nameplateShowAll, timeMod, value1, value2, value3 = UnitBuff("player" , i)
			spellID = tostring(spellID)
			if(Buffs[spellID] ~= nil) then
				Buffs[spellID]["active"] = true
				Buffs[spellID]["red"] = 1.0
				Buffs[spellID]["green"] = 0
				Buffs[spellID]["blue"] = 0
				amount = amount + 1
			end
		end
		-- Update the textbox/icon box if mode is set to true
		if(TTBDB.textboxmode == true) then
			DisplayText()
		end
		if(TTBDB.iconmode == true) then
			DisplayCooldownIcons()
		end
		if(amount > 1) then -- Actions for 1+ buffs
			if(TTBDB.rolls >= TTBDB.threshold) then -- Message after ending the bad luck streak
				if(TTBDB.outputchannel ~= "none") then
					SendChatMessage(string.gsub(TTBDB.finallyMessage, '{n}', TTBDB.rolls), TTBDB.outputchannel)
				end
				-- Update record
				if(TTBDB.rolls > TTBDB.record) then
					TTBDB.record = TTBDB.rolls
				end
			end
			TTBDB.rolls = 0
		end
		if(amount == 5) then -- Actions for five buffs
			-- Update lastfiverecord
			if(TTBDB.lastfive > TTBDB.lastfiverecord) then
				TTBDB.lastfiverecord = TTBDB.lastfive
			end
			if(TTBDB.outputchannel ~= "none") then
				SendChatMessage(string.gsub(TTBDB.hamMessage, '{n}', TTBDB.lastfive), TTBDB.outputchannel)
			end
			TTBDB.rolls = 0
			TTBDB.lastfive = 0
			-- Play Leeroy
			if(TTBDB.leeroy == true) then
				PlaySoundFile("Interface\\AddOns\\TrollTheBones\\sounds\\leeroy-jenkins.ogg", "Master")
			end
		end
		-- Play sounds
		if(TTBDB.playsound == true) then
			if(amount == 1) then
				PlaySoundFile("Interface\\AddOns\\TrollTheBones\\sounds\\1-one.ogg", "Master")
			elseif(amount == 2) then
				PlaySoundFile("Interface\\AddOns\\TrollTheBones\\sounds\\2-two.ogg", "Master")
			elseif(amount == 5) then
				if(TTBDB.leeroy == false) then -- Only play if leeroy mode is set to false
					PlaySoundFile("Interface\\AddOns\\TrollTheBones\\sounds\\5-five.ogg", "Master")
				end
			end
		end
	end
end)

--[[ This function initialises the whole addon if the Outlaw spec was detected. It sets all EventHandlers and variables and creates all frames. ]]--
function initAddon()
	Initialised = true

	if(TTBDB.lockicons == true) then
		CooldownFrameContainer:EnableMouse(false)
	else
		CooldownFrameContainer:EnableMouse(true)
	end

	-- True Bearing cooldown frame
	TBCooldownFrame, TBCooldown, TBCooldownTextFrame, TBCooldownFrameFS = CreateBuffFrame("TTB_TBCooldownFrame", 10, "Interface\\Icons\\Ability_rogue_rollthebones04")
	-- Shark Infested Waters cooldown frame
	SIWCooldownFrame, SIWCooldown, SIWCooldownTextFrame, SIWCooldownFrameFS = CreateBuffFrame("TTB_SIWCooldownFrame", 60, "Interface\\Icons\\Ability_rogue_rollthebones03")
	-- Broadsides cooldown frame
	BCooldownFrame, BCooldown, BCooldownTextFrame, BCooldownFrameFS = CreateBuffFrame("TTB_BCooldownFrame", 110, "Interface\\Icons\\Ability_rogue_rollthebones07")
	-- Jolly Roger cooldown frame
	JRCooldownFrame, JRCooldown, JRCooldownTextFrame, JRCooldownFrameFS = CreateBuffFrame("TTB_JRCooldownFrame", 160, "Interface\\Icons\\Ability_rogue_rollthebones01")
	-- Buried Treasure frame
	BTCooldownFrame, BTCooldown, BTCooldownTextFrame, BTCooldownFrameFS = CreateBuffFrame("TTB_BTCooldownFrame", 210, "Interface\\Icons\\Ability_rogue_rollthebones05")
	-- Grand Melee frame
	GMCooldownFrame, GMCooldown, GMCooldownTextFrame, GMCooldownFrameFS = CreateBuffFrame("TTB_GMCooldownFrame", 260, "Interface\\Icons\\Ability_rogue_rollthebones02")
	
	if(TTBDB.locktextbox == true) then
		BuffMessage:EnableMouse(false)
	else
		BuffMessage:EnableMouse(true)
	end

	-- RTB spell IDs
	local SharkInfestedWaters = "193357"
	local TrueBearing = "193359"
	local JollyRoger = "199603"
	local Broadsides = "193356"
	local BuriedTreasure = "199600"
	local GrandMelee = "193358"

	-- For retrieving the buffs in order (roughly best to worst)
	BuffsOrdered = {
		TrueBearing,
		SharkInfestedWaters,
		Broadsides,
		JollyRoger,
		BuriedTreasure,
		GrandMelee,
	}

	-- This table contains the RTB Buffs and some information attached to them, such as the corresponding frames in the cooldown frame container
	Buffs = {
		[TrueBearing] = {
			["active"] = false,
			["abbr"] = "CD Reduce",
			["red"] = 1,
			["green"] = 1,
			["blue"] = 1,
			["frame"] = TBCooldownFrame,
			["cdframe"] = TBCooldown,
			["fontstring"] = TBCooldownFrameFS,
		},
		[SharkInfestedWaters] = {
			["active"] = false,
			["abbr"] = "Crit",
			["red"] = 1,
			["green"] = 1,
			["blue"] = 1,
			["frame"] = SIWCooldownFrame,
			["cdframe"] = SIWCooldown,
			["fontstring"] = SIWCooldownFrameFS,
		},
		[Broadsides] = {
			["active"] = false,
			["abbr"] = "+CP",
			["red"] = 1,
			["green"] = 1,
			["blue"] = 1,
			["frame"] = BCooldownFrame,
			["cdframe"] = BCooldown,
			["fontstring"] = BCooldownFrameFS,
		},
		[JollyRoger] = {
			["active"] = false,
			["abbr"] = "Saber%",
			["red"] = 1,
			["green"] = 1,
			["blue"] = 1,
			["frame"] = JRCooldownFrame,
			["cdframe"] = JRCooldown,
			["fontstring"] = JRCooldownFrameFS,
		},
		[BuriedTreasure] = {
			["active"] = false,
			["abbr"] = "Energy Reg",
			["red"] = 1,
			["green"] = 1,
			["blue"] = 1,
			["frame"] = BTCooldownFrame,
			["cdframe"] = BTCooldown,
			["fontstring"] = BTCooldownFrameFS,
		},
		 -- Handgemenge/Grand Melee
		[GrandMelee] = {
			["active"] = false,
			["abbr"] = "AS + Leech",
			["red"] = 1,
			["green"] = 1,
			["blue"] = 1,
			["frame"] = GMCooldownFrame,
			["cdframe"] = GMCooldown,
			["fontstring"] = GMCooldownFrameFS,
		},
	}

	CooldownText = CreateFrame("frame")
	CooldownText:SetScript("OnUpdate", onUpdate)

	EventFrame:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
	EventFrame:RegisterEvent("ONUPDATE")
	EventFrame:RegisterEvent("UNIT_AURA")
	EventFrame:RegisterEvent("PLAYER_REGEN_ENABLED")
	EventFrame:RegisterEvent("PLAYER_REGEN_DISABLED")
end

-- Textual output of buffs
function DisplayText()
	BuffMessage:Clear()
	for key, value in ipairs(BuffsOrdered) do
		if(Buffs["" .. value]["active"] == true) then
			BuffMessage:AddMessage("  " .. Buffs["" .. value]["abbr"], 1.0, 0, 0, 1)
		else
			BuffMessage:AddMessage("  " .. Buffs["" .. value]["abbr"], 1, 1, 1, 1)
		end
	end
end

function DisplayCooldownIcons()
-- Graphical representation of buffs
	for key, value in ipairs(BuffsOrdered) do
		if(Buffs[value]["active"] == true) then
			Buffs[value]["frame"]:SetAlpha(1)
			for i = 1, 40 do
				local name, rank, icon, count, dispelType, duration, expires, caster, isStealable, nameplateShowPersonal, spellID, canApplyAura, isBossDebuff, _, nameplateShowAll, timeMod, value1, value2, value3 = UnitBuff("player" , i)
				spellID = tostring(spellID)
				if(spellID == value) then
					Buffs[value]["cdframe"]:SetCooldown(GetTime(), math.ceil(duration))
				end
			end
		else
			Buffs[value]["frame"]:SetAlpha(.25)
			Buffs[value]["cdframe"]:SetCooldown(0, 0)
		end
	end
end

-- Addon control panel
ACPanel = CreateFrame("Frame", "TTB_ACPanel", UIParent)
ACPanel.name = "Troll The Bones"
InterfaceOptions_AddCategory(ACPanel)

-- Check Button creator
local uniquealyzer = 1
function createCheckbutton(parent, x_loc, y_loc, displayname)
	uniquealyzer = uniquealyzer + 1
	
	local checkbutton = CreateFrame("CheckButton", "TTB_CheckButton0" .. uniquealyzer, parent, "ChatConfigCheckButtonTemplate")
	checkbutton:SetPoint("TOPLEFT", x_loc, y_loc)
	getglobal(checkbutton:GetName() .. 'Text'):SetText(displayname)

	return checkbutton
end

local LeeroyCheckbox = createCheckbutton(ACPanel, 10, -10, "Leeroy mode")
LeeroyCheckbox.tooltip = "Turn it on and scream along!"

local PlaySoundCheckbox = createCheckbutton(ACPanel, 10, -40, "Play sound")
PlaySoundCheckbox.tooltip = "Check to play a short sound (\"One\" to \"Five\") after gaining bones buffs."

local TextboxModeCheckbox = createCheckbutton(ACPanel, 10, -120, "Textbox mode")
TextboxModeCheckbox.tooltip = "When in combat, this option displays a little textbox showing the six buffs line by line, roughly ordered from best to worst."

local IconModeCheckbox = createCheckbutton(ACPanel, 10, -150, "Icon mode")
IconModeCheckbox.tooltip = "When in combat, this will display a bar with the icons for your six buffs showing which are active. They are roughly ordered from best to worst (left to right). "

-- Threshold Slider
local name = "Roll Threshold"
local template = "OptionsSliderTemplate"
local ACPThresholdSlider = CreateFrame("Slider", name, ACPanel, template)
ACPThresholdSlider:SetPoint("TOPLEFT", 10, -90)
ACPThresholdSlider.textLow = _G[name .. "Low"]
ACPThresholdSlider.textHigh = _G[name .. "High"]
ACPThresholdSlider.text = _G[name.."Text"]
ACPThresholdSlider:SetMinMaxValues(1, 25)
ACPThresholdSlider.minValue, ACPThresholdSlider.maxValue = ACPThresholdSlider:GetMinMaxValues() 
ACPThresholdSlider.textLow:SetText(ACPThresholdSlider.minValue)
ACPThresholdSlider.textHigh:SetText(ACPThresholdSlider.maxValue)
ACPThresholdSlider.text:SetText(name)
ACPThresholdSlider:EnableMouseWheel(true)
ACPThresholdSlider.tooltipText = "Sets the number of rolls you have to experience painfully before the addon starts yelling."
ACPThresholdSlider:SetValueStep(1)
ACPThresholdSlider:SetScript("OnMouseWheel", function(self, delta)
	if delta > 0 then
		self:SetValue(self:GetValue() + self:GetValueStep())
	else
		self:SetValue(self:GetValue() - self:GetValueStep())
	end
end)
ACPThresholdSlider:SetScript("OnValueChanged", function(self,event,arg1)
	ACPThresholdSlider.text:SetText("Roll Threshold (" .. math.floor(event) .. ")")
end)

-- Ham EditBox
local ACPHamInputBox = CreateFrame("EditBox", nil, ACPanel, "InputBoxTemplate")
ACPHamInputBox:SetHeight(35)
ACPHamInputBox:SetWidth(250)
ACPHamInputBox:SetPoint("TOPLEFT", 10, -200)
ACPHamInputBox:SetFontObject("GameFontHighlight")
ACPHamInputBox:SetTextInsets(0, 0, 0, 1)
ACPHamInputBox:SetAutoFocus(false)

local ACPHamInputBoxLabel = CreateFrame("Frame", "ACPHamInputBoxLabel", ACPanel)
ACPHamInputBoxLabel:SetWidth(95)
ACPHamInputBoxLabel:SetHeight(20)
ACPHamInputBoxLabel:SetPoint('TOPLEFT', 10, -180)

local ACPHamInputBoxLabelText = ACPHamInputBoxLabel:CreateFontString()
ACPHamInputBoxLabelText:SetPoint("LEFT")
ACPHamInputBoxLabelText:SetSize(95, 20)
ACPHamInputBoxLabelText:SetFont("Fonts\\FRIZQT__.TTF", 12, "OUTLINE")
ACPHamInputBoxLabelText:SetText("Ham message")

-- Finally EditBox
local ACPFinallyInputBox = CreateFrame("EditBox", nil, ACPanel, "InputBoxTemplate")
ACPFinallyInputBox:SetHeight(35)
ACPFinallyInputBox:SetWidth(250)
ACPFinallyInputBox:SetPoint("TOPLEFT", 10, -260)
ACPFinallyInputBox:SetFontObject("GameFontHighlight")
ACPFinallyInputBox:SetTextInsets(0, 0, 0, 1)
ACPFinallyInputBox:SetAutoFocus(false)

local ACPFinallyInputBoxLabel = CreateFrame("Frame", "ACPFinallyInputBoxLabel", ACPanel)
ACPFinallyInputBoxLabel:SetWidth(105)
ACPFinallyInputBoxLabel:SetHeight(20)
ACPFinallyInputBoxLabel:SetPoint("TOPLEFT", 10, -240)

local ACPFinallyInputBoxLabelText = ACPFinallyInputBoxLabel:CreateFontString()
ACPFinallyInputBoxLabelText:SetPoint("LEFT")
ACPFinallyInputBoxLabelText:SetSize(105, 20)
ACPFinallyInputBoxLabelText:SetFont("Fonts\\FRIZQT__.TTF", 12, "OUTLINE")
ACPFinallyInputBoxLabelText:SetText("Finally message")

-- Output channel dropdown menu
local OutputChannelDropDownLabel = CreateFrame("Frame", "OutputChannelDropDownLabel", ACPanel)
OutputChannelDropDownLabel:SetWidth(105)
OutputChannelDropDownLabel:SetHeight(20)
OutputChannelDropDownLabel:SetPoint("TOPLEFT", 10, -300)

local OutputChannelDropDownLabelText = OutputChannelDropDownLabel:CreateFontString()
OutputChannelDropDownLabelText:SetPoint("LEFT")
OutputChannelDropDownLabelText:SetSize(105, 20)
OutputChannelDropDownLabelText:SetFont("Fonts\\FRIZQT__.TTF", 12, "OUTLINE")
OutputChannelDropDownLabelText:SetText("Output channel")

local OutputChannelDropDown = CreateFrame("Button", "OutputChannelDropDown", ACPanel, "UIDropDownMenuTemplate")
OutputChannelDropDown:SetPoint("TOPLEFT", 0, -320)
OutputChannelDropDown:Show()
 
local outputChannels = {
   "none",
   "say",
   "yell",
   "guild",
   "group",
   "raid",
}
 
local function OnClick(self)
   UIDropDownMenu_SetSelectedID(OutputChannelDropDown, self:GetID())
   UIDropDownMenu_SetSelectedValue(OutputChannelDropDown, self.value)
end
 
local function initialize(self, level)
	local info = UIDropDownMenu_CreateInfo()
	for k,v in pairs(outputChannels) do
		info = UIDropDownMenu_CreateInfo()
		info.text = v
		info.value = v
		info.func = OnClick
		UIDropDownMenu_AddButton(info, level)
	end
end

UIDropDownMenu_Initialize(OutputChannelDropDown, initialize)
UIDropDownMenu_SetWidth(OutputChannelDropDown, 120);
UIDropDownMenu_SetButtonWidth(OutputChannelDropDown, 120)
UIDropDownMenu_JustifyText(OutputChannelDropDown, "LEFT")

-- Save changes after click on 'Okay'
ACPanel.okay = function(self)
	TTBDB.leeroy = LeeroyCheckbox:GetChecked()
	TTBDB.playsound = PlaySoundCheckbox:GetChecked()
	TTBDB.textboxmode = TextboxModeCheckbox:GetChecked()
	if(TextboxModeCheckbox:GetChecked() == false) then
		TTBDB.showtextbox = false
		
		if(BuffMessage:IsShown() == true) then
			UIFrameFadeOut(BuffMessage, 1, 1, 0)
		end
	end
	TTBDB.iconmode = IconModeCheckbox:GetChecked()
	if(IconModeCheckbox:GetChecked() == false) then
		TTBDB.showicons = false
		
		if(CooldownFrameContainer:IsShown() == true) then
			UIFrameFadeOut(CooldownFrameContainer, 1, 1, 0)
		end
	end
	TTBDB.threshold = math.floor(ACPThresholdSlider:GetValue())
	TTBDB.hamMessage = ACPHamInputBox:GetText()
	TTBDB.finallyMessage = ACPFinallyInputBox:GetText()
	TTBDB.outputchannel = UIDropDownMenu_GetSelectedValue(OutputChannelDropDown)
end

ACPanel:SetScript("OnShow", function()
	LeeroyCheckbox:SetChecked(TTBDB.leeroy)
	PlaySoundCheckbox:SetChecked(TTBDB.playsound)
	TextboxModeCheckbox:SetChecked(TTBDB.textboxmode)
	IconModeCheckbox:SetChecked(TTBDB.iconmode)
	ACPThresholdSlider.text:SetText("Roll Threshold (" .. TTBDB.threshold .. ")")
	ACPThresholdSlider:SetValue(TTBDB.threshold)
	ACPHamInputBox:SetText(TTBDB.hamMessage)
	ACPFinallyInputBox:SetText(TTBDB.finallyMessage)
	ACPHamInputBox:SetCursorPosition(0)
	ACPFinallyInputBox:SetCursorPosition(0)
	UIDropDownMenu_SetSelectedValue(OutputChannelDropDown, TTBDB.outputchannel, TTBDB.outputchannel)
	UIDropDownMenu_SetText(OutputChannelDropDown, TTBDB.outputchannel)
end)

-- Set standard saved variables
function SetStandardValues()
	if(TTBDB == nil) then
		TTBDB = {}
	end
	if(not TTBDB.rolls) then
		TTBDB.rolls = 0
	end
	if(not TTBDB.hamMessage) then
		TTBDB.hamMessage = "I rolled 'em all. Pulling Azeroth NOW!"
	end
	if(not TTBDB.finallyMessage) then
		TTBDB.finallyMessage = "I was just trolled by the bones {n} times."
	end
	if(not TTBDB.threshold) then
		TTBDB.threshold = 5
	end
	if(not TTBDB.playsound) then
		TTBDB.playsound = false
	end
	if(not TTBDB.leeroy) then
		TTBDB.leeroy = false
	end
	if(not TTBDB.record) then
		TTBDB.record = 0
	end
	if(not TTBDB.lastfive) then
		TTBDB.lastfive = 0
	end
	if(not TTBDB.lastfiverecord) then
		TTBDB.lastfiverecord = 0
	end
	if(not TTBDB.textboxmode) then
		TTBDB.textboxmode = false
	end
	if(not TTBDB.showtextbox) then
		TTBDB.showtextbox = false
	end
	if(not TTBDB.locktextbox) then
		TTBDB.locktextbox = false
	end
	if(not TTBDB.iconmode) then
		TTBDB.iconmode = false
	end
	if(not TTBDB.showicons) then
		TTBDB.showicons = false
	end
	if(not TTBDB.lockicons) then
		TTBDB.lockicons = false
	end
	if(not TTBDB.outputchannel) then
		TTBDB.outputchannel = "yell"
	end

	LeeroyCheckbox:SetChecked(TTBDB.leeroy)
	PlaySoundCheckbox:SetChecked(TTBDB.playsound)
	TextboxModeCheckbox:SetChecked(TTBDB.textboxmode)
	IconModeCheckbox:SetChecked(TTBDB.iconmode)
	ACPThresholdSlider.text:SetText("Roll Threshold (" .. TTBDB.threshold .. ")")
	ACPThresholdSlider:SetValue(TTBDB.threshold)
	ACPHamInputBox:SetText(TTBDB.hamMessage)
	ACPFinallyInputBox:SetText(TTBDB.finallyMessage)
	ACPHamInputBox:SetCursorPosition(0)
	ACPFinallyInputBox:SetCursorPosition(0)
	UIDropDownMenu_SetSelectedValue(OutputChannelDropDown, TTBDB.outputchannel, TTBDB.outputchannel)
	UIDropDownMenu_SetText(OutputChannelDropDown, TTBDB.outputchannel)
end

-- Slash commands
SLASH_TTB1 = '/ttb'
function SlashCmdList.TTB(msg, editbox)
	local command, rest = msg:match("^(%S*)%s*(.-)$");
	if(command == "" and rest == "") then
		InterfaceOptionsFrame_OpenToCategory(ACPanel.name);
		InterfaceOptionsFrame_OpenToCategory(ACPanel.name);
	elseif(command == "ham" and rest ~= "") then
		print("Ham message set to: " .. rest)
		TTBDB.hamMessage = rest
		ACPHamInputBox:SetText(TTBDB.hamMessage)
	elseif(command == "ham" and rest == "") then
		print("Ham message currently is: " .. TTBDB.hamMessage)
	elseif(command == "finally" and rest ~= "") then 
		print("Finally message set to: " .. rest)
		TTBDB.finallyMessage = rest
	elseif(command == "finally" and rest == "") then
		print("Finally message currently is: " .. TTBDB.finallyMessage)
	elseif(command == "threshold" and rest ~= "" and tonumber(rest) ~= nil) then
		if(tonumber(rest) > 0 and tonumber(rest) <= 25) then
			print("Threshold set to: " .. rest);
			TTBDB.threshold = tonumber(rest)
			ACPThresholdSlider:SetValue(TTBDB.threshold)
		else
			print("Please specify a number between 1 and 25.")
		end
	elseif(command == "threshold" and rest == "") then
		print("Threshold currently is: " .. TTBDB.threshold)
	elseif(command == "playsound" and rest == "") then
		print("Sound output currently is set to: " .. tostring(TTBDB.playsound))
	elseif(command == "record" and rest == "") then
		print("Your current record of rerolling is " .. TTBDB.record .. ". Your longest dry spell was " .. TTBDB.lastfiverecord .. " rolls.")
	elseif(command == "lastfive" and rest == "") then
		print("The last time you've rolled 'em all was " .. (TTBDB.lastfive + 1) .. " rolls ago.")
	elseif(command == "playsound" and (rest == "false" or rest == "true")) then
		if(rest == "true") then
			TTBDB.playsound = true
			PlaySoundCheckbox:SetChecked(true)
		elseif(rest == "false") then
			TTBDB.playsound = false
			PlaySoundCheckbox:SetChecked(false)
		end
		print("Sound output set to: " .. tostring(TTBDB.playsound))
	elseif(command == "leeroy" and rest == "") then
		print("Leeroy mode currently is set to: " .. tostring(TTBDB.leeroy))
	elseif(command == "leeroy" and (rest == "true" or rest == "false")) then
		if(rest == "true") then
			TTBDB.leeroy = true
			LeeroyCheckbox:SetChecked(true)
		elseif(rest == "false") then
			TTBDB.leeroy = false
			LeeroyCheckbox:SetChecked(false)
		end
		print("Leeroy mode set to: " .. tostring(TTBDB.leeroy))
	elseif(command == "textboxmode" and (rest == "true" or rest == "false")) then
		if(rest == "true") then
			TTBDB.textboxmode = true
			TextboxModeCheckbox:SetChecked(true)
		elseif(rest == "false") then
			TTBDB.textboxmode = false
			TTBDB.showtextbox = false
			TextboxModeCheckbox:SetChecked(false)
			UIFrameFadeOut(BuffMessage, 1, 1, 0)
		end
		print("Textbox mode set to: " .. tostring(TTBDB.textboxmode))
	elseif(command == "showtextbox") then
		TTBDB.showtextbox = true
		UIFrameFadeIn(BuffMessage, 1, 0, 1)
	elseif(command == "hidetextbox") then
		TTBDB.showtextbox = false
		if(BuffMessage:IsShown() == true) then
			UIFrameFadeOut(BuffMessage, 1, 1, 0)
		end
	elseif(command == "locktextbox") then
		TTBDB.locktextbox = true
		BuffMessage:EnableMouse(false)
	elseif(command == "unlocktextbox") then
		TTBDB.locktextbox = false
		BuffMessage:EnableMouse(true)
	elseif(command == "iconmode" and (rest == "true" or rest == "false")) then
		if(rest == "true") then
			TTBDB.iconmode = true
		elseif(rest == "false") then
			TTBDB.iconmode = false
			TTBDB.showicons = false
			CooldownFrameContainer:Hide()
		end
		print("Icon mode mode set to: " .. tostring(TTBDB.iconmode))
	elseif(command == "showicons") then
		TTBDB.showicons = true
		UIFrameFadeIn(CooldownFrameContainer, 1, 0, 1)
	elseif(command == "hideicons") then
		TTBDB.showicons = false
		if(CooldownFrameContainer:IsShown()) then
			UIFrameFadeOut(CooldownFrameContainer, 1, 1, 0)
		end
	elseif(command == "lockicons") then
		TTBDB.lockicons = true
		CooldownFrameContainer:EnableMouse(false)
	elseif(command == "unlockicons") then
		TTBDB.lockicons = false
		CooldownFrameContainer:EnableMouse(true)
	elseif(command == "resetpos") then
		CooldownFrameContainer:ClearAllPoints()
		CooldownFrameContainer:SetPoint("CENTER")
		BuffMessage:ClearAllPoints()
		BuffMessage:SetPoint("CENTER")
	elseif(command == "center") then
		CooldownFrameContainer:ClearAllPoints()
		CooldownFrameContainer:SetPoint("BOTTOMLEFT", "UIParent", "BOTTOMLEFT", GetScreenWidth() / 2 - CooldownFrameContainer:GetWidth() / 2, CooldownFrameContainer:GetBottom())
		BuffMessage:ClearAllPoints()
		BuffMessage:SetPoint("BOTTOMLEFT", "UIParent", "BOTTOMLEFT", GetScreenWidth() / 2 * UIParent:GetEffectiveScale() - BuffMessage:GetWidth() / 2 * UIParent:GetEffectiveScale(), BuffMessage:GetBottom())
	elseif(command == "output" and (rest == "")) then
		print("Your current output channel is: " .. TTBDB.outputchannel)
	elseif(command == "output" and (rest == "none" or rest == "say" or rest == "yell"  or rest == "guild" or rest == "group" or rest == "raid")) then
		TTBDB.outputchannel = rest
		UIDropDownMenu_SetSelectedValue(OutputChannelDropDown, rest, rest)
		UIDropDownMenu_SetText(OutputChannelDropDown, rest)
	else
		print("This is not a command of Troll The Bones. Please consider reading the documentation at https://mods.curse.com/addons/wow/troll-the-bones.")
	end
end