-- Main addon frame and event registration
local TrokkoMobCounter = CreateFrame("Frame")
TrokkoMobCounter:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED") -- Track combat events
TrokkoMobCounter:RegisterEvent("ADDON_LOADED") -- Handle addon initialization

-- Character and realm information for data storage
local realmName = GetRealmName()
local characterName = UnitName("player")
local characterKey = realmName .. "-" .. characterName

-- Data storage tables
local unitKills = {} -- Stores total kills for each mob
local unitKillsSession = {} -- Stores kills in current session for each mob
local sortColumn = "Name" -- Current column being sorted
local sortAscending = true -- Sort direction
TrokkoMobCounterDB = TrokkoMobCounterDB or {} -- Global database for persistent storage

-- Event handler for combat and addon events
function TrokkoMobCounter:OnEvent(event, ...)
    if event == "COMBAT_LOG_EVENT_UNFILTERED" then
        -- Extract combat log information
        local timestamp, subevent, _, sourceGUID, _, _, _, destGUID, destName, _, _, _ = CombatLogGetCurrentEventInfo()
        -- Check if a unit died and it's a creature (exclude players and pets)
        if subevent == "UNIT_DIED" and destName then
            local destType = select(1, strsplit("-", destGUID))
            if destType == "Creature" then
                -- Increment kill counters
                unitKills[destName] = (unitKills[destName] or 0) + 1
                unitKillsSession[destName] = (unitKillsSession[destName] or 0) + 1
            end
        end
    elseif event == "ADDON_LOADED" then
        local addonName = ...
        if addonName == "TrokkoMobCounter" then
            TrokkoMobCounter:OnInitialize()
            self:RegisterEvent("PLAYER_LOGOUT") -- Register logout event to save data
        end
    elseif event == "PLAYER_LOGOUT" then
        SaveData(unitKills, unitKillsSession) -- Save data before logging out
    end
end

-- Save data to persistent storage
function SaveData(mobData, sessionMobData)
    TrokkoMobCounterDB = TrokkoMobCounterDB or {}
    TrokkoMobCounterDB[characterKey] = TrokkoMobCounterDB[characterKey] or {}
    TrokkoMobCounterDB[characterKey].unitKills = mobData
    TrokkoMobCounterDB[characterKey].unitKillsSession = sessionMobData
    print("Kill counter saved:", mobData, sessionMobData)
end

-- Initialize addon and load saved data
function TrokkoMobCounter:OnInitialize()
    TrokkoMobCounterDB = TrokkoMobCounterDB or {}
    TrokkoMobCounterDB[characterKey] = TrokkoMobCounterDB[characterKey] or {}
    unitKills = TrokkoMobCounterDB[characterKey].unitKills or {}
    unitKillsSession = {} -- Reset session data
    print("Kill counter loaded:", unitKills)
end

-- Add kill information to unit tooltips
local function OnTooltipSetUnit(tooltip)
    local _, unit = tooltip:GetUnit()
    if unit and unitKills[UnitName(unit)] then
        local unitKillsCount = unitKills[UnitName(unit)]
        local unitKillsSessionCount = unitKillsSession[UnitName(unit)] or 0
        tooltip:AddLine("Kills Total: " .. unitKillsCount, 1, 1, 1)
        tooltip:AddLine("Kills this session: " .. unitKillsSessionCount, 1, 1, 1)
        tooltip:Show()
    end
end
GameTooltip:HookScript("OnTooltipSetUnit", OnTooltipSetUnit)

-- Create the main display frame
local function CreateDisplayFrame()
    -- Create a draggable frame with basic template
    local frame = CreateFrame("Frame", "TrokkoMobCounterFrame", UIParent, "BasicFrameTemplateWithInset")
    frame:SetSize(400, 400)
    frame:SetPoint("CENTER")
    frame:SetMovable(true)
    frame:EnableMouse(true)
    frame:RegisterForDrag("LeftButton")
    frame:SetScript("OnDragStart", frame.StartMoving)
    frame:SetScript("OnDragStop", frame.StopMovingOrSizing)
    
    -- Add title
    frame.title = frame:CreateFontString(nil, "OVERLAY")
    frame.title:SetFontObject("GameFontHighlightLarge")
    frame.title:SetPoint("CENTER", frame.TitleBg, "CENTER", 5, 0)
    frame.title:SetText("Mob Kill Counter")
    
    -- Create scrollable content area
    frame.scrollFrame = CreateFrame("ScrollFrame", nil, frame, "UIPanelScrollFrameTemplate")
    frame.scrollFrame:SetPoint("TOPLEFT", 10, -30)
    frame.scrollFrame:SetPoint("BOTTOMRIGHT", -30, 50)
    frame.content = CreateFrame("Frame", nil, frame.scrollFrame)
    frame.content:SetSize(360, 360)
    frame.scrollFrame:SetScrollChild(frame.content)
    
    -- Function to create column headers
    local function CreateHeader(text, xOffset)
        local header = CreateFrame("Button", nil, frame.content)
        header:SetSize(120, 20)
        header:SetPoint("TOPLEFT", frame.content, "TOPLEFT", xOffset, -10)
        header.text = header:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
        header.text:SetPoint("CENTER")
        header.text:SetText(text)
        header.text:SetTextColor(1, 1, 1)
        header:SetNormalFontObject("GameFontNormalLarge")
        
        -- Add underline to header
        header.underline = header:CreateTexture(nil, "BACKGROUND")
        header.underline:SetColorTexture(1, 1, 1, 0.5)
        header.underline:SetHeight(1)
        header.underline:SetPoint("BOTTOMLEFT", header, "BOTTOMLEFT", 0, -1)
        header.underline:SetPoint("BOTTOMRIGHT", header, "BOTTOMRIGHT", 0, -1)
        
        -- Add sort indicator
        header.sortIndicator = header:CreateFontString(nil, "OVERLAY", "GameFontNormal")
        header.sortIndicator:SetPoint("LEFT", header.text, "RIGHT", 5, 0)
        header.sortIndicator:Hide()
        
        return header
    end
    
    -- Create column headers
    frame.nameHeader = CreateHeader("Name", 30)
    frame.totalKillsHeader = CreateHeader("Total kills", 150)
    frame.sessionKillsHeader = CreateHeader("Kills this session", 270)
    frame.content.lines = {}
    
    -- Add clear all data button
    local clearButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    clearButton:SetSize(100, 24)
    clearButton:SetPoint("BOTTOM", frame, "BOTTOM", 0, 10)
    clearButton:SetText("Clear All Data")
    clearButton:SetScript("OnClick", function()
        StaticPopup_Show("CONFIRM_CLEAR_DATA")
    end)
    
    frame:Hide()
    return frame
end

-- Confirmation dialog for clearing all data
StaticPopupDialogs["CONFIRM_CLEAR_DATA"] = {
    text = "Are you sure you want to clear all data?",
    button1 = "Yes",
    button2 = "No",
    OnAccept = function()
        unitKills = {}
        unitKillsSession = {}
        SaveData(unitKills, unitKillsSession)
        print("All data cleared.")
        ShowKillList()
    end,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
    preferredIndex = 3,
}

-- Confirmation dialog for deleting individual entries
StaticPopupDialogs["CONFIRM_DELETE_ENTRY"] = {
    text = "Are you sure you want to delete the entry for %s?",
    button1 = "Yes",
    button2 = "No",
    OnAccept = function(self, data)
        unitKills[data] = nil
        unitKillsSession[data] = nil
        SaveData(unitKills, unitKillsSession)
        print("Entry for " .. data .. " deleted.")
        ShowKillList()
    end,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
    preferredIndex = 3,
}

local displayFrame = CreateDisplayFrame()

-- Sort data based on selected column
local function SortData(column)
    if sortColumn == column then
        sortAscending = not sortAscending
    else
        sortColumn = column
        sortAscending = true
    end
    
    -- Convert data to sortable table
    local sortedData = {}
    for name, count in pairs(unitKills) do
        table.insert(sortedData, { name = name, totalKills = count, sessionKills = unitKillsSession[name] or 0 })
    end
    
    -- Sort based on selected column and direction
    table.sort(sortedData, function(a, b)
        if sortColumn == "Name" then
            if sortAscending then
                return a.name < b.name
            else
                return a.name > b.name
            end
        elseif sortColumn == "Total kills" then
            if sortAscending then
                return a.totalKills < b.totalKills
            else
                return a.totalKills > b.totalKills
            end
        elseif sortColumn == "Kills this session" then
            if sortAscending then
                return a.sessionKills < b.sessionKills
            else
                return a.sessionKills > b.sessionKills
            end
        end
    end)
    return sortedData
end

-- Display the kill list in the frame
local function ShowKillList()
    local sortedData = SortData(sortColumn)
    local lines = displayFrame.content.lines
    
    -- Update sort indicators
    displayFrame.nameHeader.sortIndicator:Hide()
    displayFrame.totalKillsHeader.sortIndicator:Hide()
    displayFrame.sessionKillsHeader.sortIndicator:Hide()
    
    -- Show sort indicator for active column
    local activeHeader = nil
    if sortColumn == "Name" then
        activeHeader = displayFrame.nameHeader
    elseif sortColumn == "Total kills" then
        activeHeader = displayFrame.totalKillsHeader
    elseif sortColumn == "Kills this session" then
        activeHeader = displayFrame.sessionKillsHeader
    end
    
    if activeHeader then
        activeHeader.sortIndicator:Show()
        activeHeader.sortIndicator:SetText(sortAscending and "↑" or "↓")
    end
    
    -- Calculate maximum width needed for names
    local tempFontString = displayFrame.content:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    local maxNameWidth = 120
    for _, data in ipairs(sortedData) do
        tempFontString:SetText(data.name)
        local nameWidth = tempFontString:GetStringWidth() + 10
        maxNameWidth = math.max(maxNameWidth, nameWidth)
    end
    tempFontString:Hide()
    
    -- Position column headers
    displayFrame.totalKillsHeader:SetPoint("TOPLEFT", maxNameWidth, -10)
    displayFrame.sessionKillsHeader:SetPoint("TOPLEFT", maxNameWidth + 120, -10)
    
    -- Create or update lines for each entry
    local yOffset = -30
    for i, data in ipairs(sortedData) do
        local line = lines[i]
        if not line then
            line = CreateFrame("Frame", nil, displayFrame.content)
            line:SetSize(maxNameWidth + 240, 20)
            
            -- Create delete button
            line.deleteButton = CreateFrame("Button", nil, line)
            line.deleteButton:SetSize(16, 16)
            line.deleteButton:SetPoint("LEFT", 0, 0)
            
            local deleteTexture = line.deleteButton:CreateTexture(nil, "OVERLAY")
            deleteTexture:SetTexture("Interface\\Buttons\\UI-GroupLoot-Pass-Up")
            deleteTexture:SetAllPoints()
            line.deleteButton:SetNormalTexture(deleteTexture)
            
            local highlightTexture = line.deleteButton:CreateTexture(nil, "HIGHLIGHT")
            highlightTexture:SetTexture("Interface\\Buttons\\UI-GroupLoot-Pass-Down")
            highlightTexture:SetAllPoints()
            line.deleteButton:SetHighlightTexture(highlightTexture)
            
            line.deleteButton:SetScript("OnClick", function()
                StaticPopup_Show("CONFIRM_DELETE_ENTRY", data.name)
            end)
            
            -- Create text fields for each column
            line.name = line:CreateFontString(nil, "OVERLAY", "GameFontNormal")
            line.name:SetPoint("LEFT", line.deleteButton, "RIGHT", 5, 0)
            line.name:SetWidth(maxNameWidth)
            line.name:SetJustifyH("LEFT")
            line.totalKills = line:CreateFontString(nil, "OVERLAY", "GameFontNormal")
            line.totalKills:SetPoint("LEFT", maxNameWidth, 0)
            line.sessionKills = line:CreateFontString(nil, "OVERLAY", "GameFontNormal")
            line.sessionKills:SetPoint("LEFT", maxNameWidth + 120, 0)
            lines[i] = line
        end
        
        -- Update line content
        line.name:SetText(data.name)
        line.totalKills:SetText(data.totalKills)
        line.sessionKills:SetText(data.sessionKills)
        line:SetPoint("TOPLEFT", 0, yOffset - (i - 1) * 20)
        line:Show()
    end
    
    displayFrame.content:SetSize(maxNameWidth + 240, 360)
    displayFrame:Show()
end

-- Set up column header click handlers
displayFrame.nameHeader:SetScript("OnClick", function()
    SortData("Name")
    ShowKillList()
end)
displayFrame.totalKillsHeader:SetScript("OnClick", function()
    SortData("Total kills")
    ShowKillList()
end)
displayFrame.sessionKillsHeader:SetScript("OnClick", function()
    SortData("Kills this session")
    ShowKillList()
end)

-- Close frame on ESC key
local function CloseDisplayFrame()
    displayFrame:Hide()
end
displayFrame:SetScript("OnKeyDown", function(self, key)
    if key == "ESCAPE" then
        CloseDisplayFrame()
    end
end)

-- Create and configure minimap button
local minimapButton = CreateFrame("Button", "TrokkoMobCounterMinimapButton", Minimap)
minimapButton:SetSize(32, 32)
minimapButton:SetFrameStrata("MEDIUM")
minimapButton:SetPoint("TOPLEFT", Minimap, "TOPLEFT", 52 - (80 * cos(45)), (80 * sin(45)) - 52)
minimapButton:SetNormalTexture("Interface\\AddOns\\TrokkoMobCounter\\Trokko_Logotype.tga")
minimapButton:SetHighlightTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight", "ADD")

-- Add mask texture for minimap button
local maskTexture = minimapButton:CreateTexture(nil, "OVERLAY")
maskTexture:SetTexture("Interface\\Minimap\\UI-Minimap-Background")
maskTexture:SetPoint("CENTER", minimapButton, "CENTER", 0, 0)
maskTexture:SetSize(30, 30)
minimapButton.mask = maskTexture

-- Make minimap button draggable
minimapButton:SetMovable(true)
minimapButton:EnableMouse(true)
minimapButton:RegisterForDrag("LeftButton")

-- Update minimap button position
local function UpdateMinimapButtonPosition()
    local x, y = GetCursorPosition()
    local minimapX, minimapY = Minimap:GetCenter()
    local scale = Minimap:GetEffectiveScale()
    x = (x / scale) - minimapX
    y = (y / scale) - minimapY
    local distance = sqrt(x * x + y * y)
    if distance < 80 then
        minimapButton:SetPoint("CENTER", Minimap, "CENTER", x, y)
    else
        local angle = atan2(y, x)
        x = cos(angle) * 80
        y = sin(angle) * 80
        minimapButton:SetPoint("CENTER", Minimap, "CENTER", x, y)
    end
end

-- Set up minimap button drag handlers
minimapButton:SetScript("OnDragStart", minimapButton.StartMoving)
minimapButton:SetScript("OnDragStop", function()
    minimapButton:StopMovingOrSizing()
    UpdateMinimapButtonPosition()
    local point, relativeTo, relativePoint, xOfs, yOfs = minimapButton:GetPoint()
    TrokkoMobCounterDB.minimapButtonPosition = {point, relativeTo, relativePoint, xOfs, yOfs}
end)

-- Set up minimap button tooltip
minimapButton:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
    GameTooltip:SetText("TrokkoMobCounter", 1, 1, 1)
    GameTooltip:AddLine("Left-click to open the kill list.", 1, 1, 1)
    GameTooltip:AddLine("Right-click and drag to move.", 1, 1, 1)
    GameTooltip:Show()
end)
minimapButton:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
end)

-- Set up minimap button click handlers
minimapButton:SetScript("OnMouseDown", function(self, button)
    if button == "LeftButton" then
        if displayFrame:IsShown() then
            displayFrame:Hide()
        else
            ShowKillList()
        end
    elseif button == "RightButton" then
        minimapButton:StartMoving()
    end
end)
minimapButton:SetScript("OnMouseUp", function(self, button)
    if button == "RightButton" then
        minimapButton:StopMovingOrSizing()
        UpdateMinimapButtonPosition()
    end
end)

-- Restore minimap button position if saved
if TrokkoMobCounterDB.minimapButtonPosition then
    local point, relativeTo, relativePoint, xOfs, yOfs = unpack(TrokkoMobCounterDB.minimapButtonPosition)
    minimapButton:ClearAllPoints()
    minimapButton:SetPoint(point, relativeTo, relativePoint, xOfs, yOfs)
else
    UpdateMinimapButtonPosition()
end

-- Set up slash commands
SLASH_TMOB1 = "/tmob"
SlashCmdList["TMOB"] = function(msg)
    local command = msg:trim()
    if command == "list" then
        ShowKillList()
    else
        print("Usage:")
        print("/tmob list - Show the kill list.")
    end
end

-- Set the event handler
TrokkoMobCounter:SetScript("OnEvent", TrokkoMobCounter.OnEvent)