
local LDB = LibStub:GetLibrary( "LibDataBroker-1.1" )

local DEFAULT_SAVEDATA = {
	show_ilvl = true;
	show_pve_talents = true;
	show_pvp_talents = false;
	
	-- per char --
	selected_build = nil;
}

local function CreateSafe( source )
	return setmetatable( {}, {
		__index = function( table, key )
			local a = _G[source][key]
			if a == nil then
				return DEFAULT_SAVEDATA[key]
			end
			return a
		end;
		
		__newindex = function( table, key, value )
			_G[source][key] = value
		end;
	})
end

local Safe = CreateSafe( "TitanTalentsSaved" )
rawset( Safe, "char", CreateSafe( "TitanTalentsCharSaved" ))

local m_ldb
local m_frame = CreateFrame( "Frame" )
local m_config_menu = nil

local function DumpTalents_GetSet( rows, source )
	local results = {}
	for row = 1,rows do
		for col = 1,3 do
			local id, name, tex, sel = source( row, col, 1 )
			if sel then
				results[row] = {
					id   = id;
					name = name;
					tex  = tex;
					col  = col;
				}
				break
			end
		end
	end
	return results
end
	
local function DumpTalents()
	local talents = {}
	talents.pve = DumpTalents_GetSet( 7, GetTalentInfo )
	talents.pvp = DumpTalents_GetSet( 6, GetPvpTalentInfo )
	return talents
end

local function GetBuild()
	local talents = DumpTalents()
	local build = ""
	for i = 1,7 do
		local a = talents.pve[i]
		build = build .. (a and a.col or "0")
	end
	build = build .. "-"
	for i = 1,6 do
		local a = talents.pvp[i]
		build = build .. (a and a.col or "0")
	end
	
	return build
end

local function LearnBuild( build )
	if InCombatLockdown() then return false end
	
	local pve, pvp = build:match("([^-]*)%-(.*)")
	if not pve then
		return ""
	end
	local pve_talent_ids = {}
	local pvp_talent_ids = {}
	for i = 1,7 do
		local col = pve:sub(i,i)
		if col ~= "0" then
			local id = GetTalentInfo( i, col, 1 )
			table.insert( pve_talent_ids, id )
		end
	end
	for i = 1,6 do
		local col = pvp:sub(i,i)
		if col ~= "0" then
			local id = GetPvpTalentInfo( i, col, 1 )
			table.insert( pvp_talent_ids, id )
		end
	end
	
	LearnTalents( unpack( pve_talent_ids ))
	LearnPvpTalents( unpack( pvp_talent_ids ))
	
	return true
end

local function OnTooltipShow()
	GameTooltip:AddLine( "Talents", 1, 1, 1 )
	GameTooltip:AddLine( " " )
	GameTooltip:AddLine( "|cff00ff00Left-click|r to change builds.", 1, 1, 1 )
	GameTooltip:AddLine( "|cff00ff00Right-click|r for addon options.", 1, 1, 1 )
end

local function GetTalentTexture( row, col, pvp )
	if pvp then
		local _, _, tex = GetPvpTalentInfo( row, col, 1 )
		return tex
	else
		local _, _, tex = GetTalentInfo( row, col, 1 )
		return tex
	
	end
end

local function BuildTalentIconString( build )

	local pve, pvp = build:match("([^-]*)%-(.*)")
	if not pve then
		return ""
	end
	
	local output = {}
	if Safe.show_pve_talents then
		for row = 1,7 do
			local col = pve:sub(row,row)
			if col ~= "0" then
				table.insert( output, "|T" .. GetTalentTexture( row, col ) .. ":0:0:0:0:100:100:10:90:10:90:255:255:255|t" )
			end
		end
		if Safe.show_pvp_talents then
			table.insert( output, "" ) -- extra space
		end
	end
	if Safe.show_pvp_talents then
		for row = 1,6 do 
			local col = pvp:sub(row,row)
			if col ~= "0" then
				table.insert( output, "|T" .. GetTalentTexture( row, col, true ) .. ":0:0:0:0:100:100:10:90:10:90:255:255:255|t" )
			end
		end
	end
	return table.concat( output, " " )
end

local m_last_update

local function UpdateText()
	if m_last_update == GetTime() then return end
	m_last_update = GetTime()
	local text = {}
	if Safe.show_ilvl then
		
		local _, ilvl = GetAverageItemLevel()
		--table.insert( text, "|TInterface\\Icons\\Garrison_PurpleArmor:0:0:0:0:100:100:10:90:10:90:255:255:255|t" )
		table.insert( text, string.format( "%.1f", ilvl ))
	end
	
	local build = GetBuild()
	
	if Safe.show_pve_talents or Safe.show_pvp_talents then
		table.insert( text, BuildTalentIconString( build ))
	end
	
	text = table.concat( text, " " )
	m_ldb.text = text
end

-------------------------------------------------------------------------------
local function ConfigMenuSetup( self, level, menuList )
	if level == 1 then
		local info
		info = UIDropDownMenu_CreateInfo()
		info.text = "Configuration"
		info.isTitle = true
		info.notCheckable = true
		UIDropDownMenu_AddButton( info, level )
		
		info = UIDropDownMenu_CreateInfo()
		info.text          = "Show Item Level"
		info.notCheckable  = false
		info.isNotRadio    = true
		info.checked       = Safe.show_ilvl
		info.func = function()
			Safe.show_ilvl = not Safe.show_ilvl
			UpdateText()
		end
		info.tooltipTitle     = "Show Item Level"
		info.tooltipText      = "Click to toggle displaying your average item level."
		info.tooltipOnButton  = true
		info.keepShownOnClick = true
		UIDropDownMenu_AddButton( info, level )
		
		info = UIDropDownMenu_CreateInfo()
		info.text          = "Show Talents"
		info.notCheckable  = false
		info.isNotRadio    = true
		info.checked       = Safe.show_pve_talents
		info.func = function()
			Safe.show_pve_talents = not Safe.show_pve_talents
			UpdateText()
		end
		info.tooltipTitle     = "Show Talents"
		info.tooltipText      = "Click to toggle displaying your normal talents."
		info.tooltipOnButton  = true
		info.keepShownOnClick = true
		UIDropDownMenu_AddButton( info, level )
		
		info = UIDropDownMenu_CreateInfo()
		info.text          = "Show Honor Talents"
		info.notCheckable  = false
		info.isNotRadio    = true
		info.checked       = Safe.show_pvp_talents
		info.func = function()
			Safe.show_pvp_talents = not Safe.show_pvp_talents
			UpdateText()
		end
		info.tooltipTitle     = "Show Honor Talents"
		info.tooltipText      = "Click to toggle displaying your honor talents."
		info.tooltipOnButton  = true
		info.keepShownOnClick = true
		UIDropDownMenu_AddButton( info, level )
	end
end

local function DoSaveBuild( name, build )
	if name == "" then return end
	Safe.char.builds = Safe.char.builds or {}
	Safe.char.builds[name] = build
	Safe.char.selected_build = name
end

StaticPopupDialogs["TITANTALENTS_SAVEAS"] = {
	text         = "Enter name of build.";
	button1      = "Save";
	button2      = "Cancel";
	hasEditBox   = true;
	hideOnEscape = true;
	whileDead    = true;
	timeout      = 0;
	OnAccept = function( self, data )
		local name = self.editBox:GetText()
		DoSaveBuild( name, data.build )
	end;
	EditBoxOnEscapePressed = function(self)
		self:GetParent():Hide()
	end;
	EditBoxOnEnterPressed = function(self, data)
		local parent = self:GetParent()
		local name = self:GetText()
		DoSaveBuild( name, data.build )
		self:SetText("")
		parent:Hide()
	end;
}

local function SaveBuildAs( self, arg1, arg2, checked )
	StaticPopup_Show( "TITANTALENTS_SAVEAS", nil, nil, { build = arg1 } )
end

local function SaveBuild( self, arg1, arg2, checked )
	local selected_build = Safe.char.selected_build
	if not selected_build then return end
	Safe.char.builds[selected_build] = arg1
end

local function DeleteBuild( self, arg1, arg2, checked )
	local selected_build = Safe.char.selected_build
	if not selected_build then return end
	if not Safe.char.builds[selected_build] then
		Safe.char.selected_build = nil
		return
	end
	Safe.char.builds[selected_build] = nil
	Safe.char.selected_build = nil
end

local function LoadBuild( self, arg1, arg2, checked )
	if not Safe.char.builds[arg1] then return end
	
	-- select talents
	if LearnBuild( Safe.char.builds[arg1] ) then
		Safe.char.selected_build = arg1
	end
end

-------------------------------------------------------------------------------
local function BuildsMenuSetup( self, level, menuList )
	if level == 1 then
		local build = GetBuild()
		local build_list = Safe.char.builds
		local sorted_list = {}
		local selected_build = Safe.char.selected_build
		if selected_build and not build_list[selected_build] then 
			selected_build = nil 
		end
		local selected_build_changed = false
		for k, v in pairs( build_list ) do
			if k == selected_build then
				if v ~= build then
					selected_build_changed = true
				end
			end
			table.insert( sorted_list, { name = k, build = v })
		end
		table.sort( sorted_list, function( a, b ) return a.name < b.name end )
		
		local function ListBuild( selected, changes, name, build )
			local info = UIDropDownMenu_CreateInfo()
			info.text = BuildTalentIconString(build) .. "  " .. name
			
			info.notCheckable = false
			if selected then
				if changes then
					info.text = info.text .. " *"
				end
				info.checked = true
			else
				info.checked          = false
				info.tooltipTitle     = name
				
			end
			
			info.tooltipTitle     = name
			if changes then
				info.tooltipTitle = info.tooltipTitle .. " (Changed)"
			end
			info.tooltipText      = "Click to load this build."
			info.tooltipOnButton  = true
			info.arg1             = name
			info.func             = LoadBuild
			UIDropDownMenu_AddButton( info, level )
		end
		
		for _,v in ipairs( sorted_list ) do
			local selected = (v.name == selected_build)
			local changes = selected and selected_build_changed
			ListBuild( selected, changes, v.name, v.build )
		end
		
		if not selected_build and #sorted_list == 0 then
			local info = UIDropDownMenu_CreateInfo()
			info.isTitle      = true
			info.notCheckable = true
			info.text         = "(No saved builds.)"
			UIDropDownMenu_AddButton( info, level )
		end
		
		local info = UIDropDownMenu_CreateInfo()
		if C_Club then -- 7.x compat
			UIDropDownMenu_AddSeparator( level )
		else
			UIDropDownMenu_AddSeparator( info, level )
		end
		
		if selected_build and selected_build_changed then
			local info = UIDropDownMenu_CreateInfo()
			info.text             = "Save"
			info.notCheckable     = true
			info.tooltipTitle     = "Save"
			info.tooltipText      = "Save changes to this build."
			info.tooltipOnButton  = true
			info.arg1             = build
			info.func             = SaveBuild
			UIDropDownMenu_AddButton( info, level )
			
			local info = UIDropDownMenu_CreateInfo()
			info.text             = "Save As"
			info.notCheckable     = true
			info.tooltipTitle     = "Save As"
			info.tooltipText      = "Save this build as something new."
			info.tooltipOnButton  = true
			info.arg1             = build
			info.func             = SaveBuildAs
			UIDropDownMenu_AddButton( info, level )
		end
		
		if not selected_build then
			local info = UIDropDownMenu_CreateInfo()
			info.text             = "Save"
			info.notCheckable     = true
			info.tooltipTitle     = "Save"
			info.tooltipText      = "Save this build."
			info.tooltipOnButton  = true
			info.arg1             = build
			info.func             = SaveBuildAs
			UIDropDownMenu_AddButton( info, level )
		end
		
		if selected_build then
			local info            = UIDropDownMenu_CreateInfo()
			info.text             = "Delete"
			info.notCheckable     = true
			info.tooltipTitle     = "Delete"
			info.tooltipText      = "Delete this build."
			info.tooltipOnButton  = true
			info.func             = DeleteBuild
			UIDropDownMenu_AddButton( info, level )
		end
	end
	
end

local function SetupMenuFrame( name, init )
	if not _G[name] then
		local f = CreateFrame( "Button", name, UIParent, "UIDropDownMenuTemplate" )
		f.displayMode = "MENU"
		f.menu_init = init
	end
end

local function ToggleMenu( frame, menu )
	PlaySound(SOUNDKIT.IG_MAINMENU_OPTION_CHECKBOX_ON);
	if UIDROPDOWNMENU_OPEN_MENU == menu then
		-- the menu is already open at the same parent, so we close it.
		ToggleDropDownMenu( 1, nil, menu )
		return
	end
	
	UIDropDownMenu_Initialize( menu, menu.menu_init )
	UIDropDownMenu_JustifyText( menu, "LEFT" )
	ToggleDropDownMenu( 1, nil, menu, frame:GetName(), 0, 0 )
end

local function ToggleConfigMenu( frame )
	SetupMenuFrame( "TitanTalentsConfigMenu", ConfigMenuSetup )
	ToggleMenu( frame, TitanTalentsConfigMenu )
end

local function ToggleBuildsMenu( frame )
	SetupMenuFrame( "TitanTalentsBuildsMenu", BuildsMenuSetup )
	ToggleMenu( frame, TitanTalentsBuildsMenu )
end

local function OnClick( frame, button )
	GameTooltip:Hide()
	if button == "LeftButton" then
		-- show talents menu
		ToggleBuildsMenu( frame )
	elseif button == "RightButton" then
		-- show configuration menu
		ToggleConfigMenu( frame )
	end
end

-------------------------------------------------------------------------------
local function Start()
	m_frame:UnregisterEvent( "ADDON_LOADED" )
	TitanTalentsSaved = TitanTalentsSaved or {}
	TitanTalentsCharSaved = TitanTalentsCharSaved or {}
	
	Safe.char.builds = Safe.char.builds or {}
	
	m_ldb = LDB:NewDataObject( "TitanTalents", {
		type = "data source";
		text = "";
		icon = "Interface\\Icons\\Garrison_PurpleArmor";
		OnClick = OnClick;
		OnTooltipShow = OnTooltipShow;
	})
	UpdateText()
	
	m_frame:RegisterEvent( "BAG_UPDATE" )
	m_frame:RegisterEvent( "PLAYER_ENTERING_WORLD" )
	m_frame:RegisterEvent( "PLAYER_TALENT_UPDATE" )
	m_frame:RegisterEvent( "UNIT_INVENTORY_CHANGED" )
end

-------------------------------------------------------------------------------
local function OnEvent( self, event, ... )
	if event == "ADDON_LOADED" then
		local addon_name = ...
		if addon_name == "TitanTalents" then
			Start()
		end
	elseif event == "BAG_UPDATE" or event == "PLAYER_ENTERING_WORLD"
	       or event == "PLAYER_TALENT_UPDATE"
		   or event == "UNIT_INVENTORY_CHANGED" then
		
		UpdateText()
		C_Timer.After( 1.5, UpdateText )
	end
end

m_frame:SetScript( "OnEvent", OnEvent )

m_frame:RegisterEvent( "ADDON_LOADED" )
