local trace = print

TEATIMERS.MAXBARSPACING = 24;
TEATIMERS.MAXBARPADDING = 12;

local GetActiveTalentGroup = _G.GetActiveSpecGroup

local LSM = LibStub("LibSharedMedia-3.0", true);
local textureList = LSM:List("statusbar");
local fontList = LSM:List("font");
local TeaTimers_OldProfile = nil;
local TeaTimers_OldSettings = nil;

local L = LibStub("AceLocale-3.0"):GetLocale("TeaTimers")

TeaTimersOptions = {}
TeaTimersMenuBar = {}

function TeaTimers.FindProfileByName(profName)
    local key
    for k,t in pairs(TeaTimers_Profiles) do
        if t.name == profName then
            return k
        end
    end
end

function TeaTimers.SlashCommand(cmd)
    local args = {}
    for arg in cmd:gmatch("(%S+)") do
        table.insert(args, arg)
    end

    cmd = args[1]
    table.remove(args,1)
    
    if not cmd then
        TeaTimers.LockToggle();
    elseif ( cmd == TEATIMERS.CMD_RESET ) then
        TeaTimers.Reset();
    elseif ( cmd == TEATIMERS.CMD_SHOW ) then
        TeaTimers.Show(true);
    elseif ( cmd == TEATIMERS.CMD_HIDE ) then
        TeaTimers.Show(false);
    elseif ( cmd == TEATIMERS.CMD_PROFILE ) then
        if args[1] then
            local profileName = table.concat(args, " ")
            local key = TeaTimers.FindProfileByName( profileName )
            if key then
                TeaTimers.ChangeProfile(key)
                TeaTimersOptions.UIPanel_Profile_Update()
            else
                print("Could not find a profile named '",profileName,"'");
            end
        else
            local spec = GetActiveTalentGroup()
            local profile = TeaTimers.CharSettings.Specs[spec]
            print("Current profile is \""..profile.."\"") -- LOCME!
        end
    else
        print("Unknown command",cmd)
    end    
end

function TeaTimers.LockToggle(bLock)
    if nil == bLock then 
        if TeaTimers.CharSettings["Locked"] then
            bLock = false;
        else
            bLock = true;
        end
    end

    TeaTimers.Show(true);
    PlaySound(SOUNDKIT.U_CHAT_SCROLL_BUTTON);

    if TeaTimers.CharSettings["Locked"] ~= bLock then
        TeaTimers.CharSettings["Locked"] = bLock;
        TeaTimers.last_cast = {};
        TeaTimers.Update();
    end
end


-- -----------------------------
-- INTERFACE OPTIONS PANEL: MAIN
-- -----------------------------

function TeaTimersOptions.UIPanel_OnLoad(self)
    local panelName = self:GetName();
    local numberbarsLabel = _G[panelName.."NumberbarsLabel"];
    local fixedDurationLabel = _G[panelName.."FixedDurationLabel"];
    _G[panelName.."Version"]:SetText(TEATIMERS.VERSION);
    _G[panelName.."SubText1"]:SetText(TEATIMERS.UIPANEL_SUBTEXT1);
    numberbarsLabel:SetText(TEATIMERS.UIPANEL_NUMBERBARS);
    numberbarsLabel:SetWidth(50);
    fixedDurationLabel:SetText(TEATIMERS.UIPANEL_FIXEDDURATION);
    fixedDurationLabel:SetWidth(50);
end

function TeaTimersOptions.UIPanel_OnShow()
    TeaTimers_OldProfile =TeaTimers.ProfileSettings;
    TeaTimers_OldSettings = CopyTable(TeaTimers.ProfileSettings);
    TeaTimersOptions.UIPanel_Update();
end

function TeaTimersOptions.UIPanel_Update()
    local panelName = "InterfaceOptionsTeaTimersPanel";
    if not _G[panelName]:IsVisible() then return end

    local settings = TeaTimers.ProfileSettings;

    for groupID = 1, settings.nGroups do
        TeaTimersOptions.GroupEnableButton_Update(groupID);
        TeaTimersOptions.NumberbarsWidget_Update(groupID);
        _G[panelName.."Group"..groupID.."FixedDurationBox"]:SetText(settings.Groups[groupID]["FixedDuration"] or "");
    end
end

function TeaTimersOptions.GroupEnableButton_Update(groupID)
    local button = _G["InterfaceOptionsTeaTimersPanelGroup"..groupID.."EnableButton"];
    button:SetChecked(TeaTimers.ProfileSettings.Groups[groupID]["Enabled"]);
end

function TeaTimersOptions.GroupEnableButton_OnClick(self)
    local groupID = self:GetParent():GetID();
    if ( self:GetChecked() ) then
        if groupID > TeaTimers.ProfileSettings.nGroups then
            TeaTimers.ProfileSettings.nGroups = groupID
        end
        TeaTimers.ProfileSettings.Groups[groupID]["Enabled"] = true;
    else
        TeaTimers.ProfileSettings.Groups[groupID]["Enabled"] = false;
    end
    TeaTimers.Update();
end

function TeaTimersOptions.NumberbarsWidget_Update(groupID)
    local widgetName = "InterfaceOptionsTeaTimersPanelGroup"..groupID.."NumberbarsWidget";
    local text = _G[widgetName.."Text"];
    local leftButton = _G[widgetName.."LeftButton"];
    local rightButton = _G[widgetName.."RightButton"];
    local numberBars = TeaTimers.ProfileSettings.Groups[groupID]["NumberBars"];
    text:SetText(numberBars);
    leftButton:Enable();
    rightButton:Enable();
    if ( numberBars == 1 ) then
        leftButton:Disable();
    elseif ( numberBars == TEATIMERS.MAXBARS ) then
        rightButton:Disable();
    end
end

function TeaTimersOptions.NumberbarsButton_OnClick(self, increment)
    local groupID = self:GetParent():GetParent():GetID();
    local oldNumber = TeaTimers.ProfileSettings.Groups[groupID]["NumberBars"];
    if ( oldNumber == 1 ) and ( increment < 0 ) then 
        return;
    elseif ( oldNumber == TEATIMERS.MAXBARS ) and ( increment > 0 ) then
        return;
    end
    TeaTimers.ProfileSettings.Groups[groupID]["NumberBars"] = oldNumber + increment;
    TeaTimers.Group_Update(groupID);
    TeaTimersOptions.NumberbarsWidget_Update(groupID);
end

function TeaTimersOptions.FixedDurationEditBox_OnTextChanged(self)
    local enteredText = self:GetText();
    if enteredText == "" then
        TeaTimers.ProfileSettings.Groups[self:GetParent():GetID()]["FixedDuration"] = nil;
    else
        TeaTimers.ProfileSettings.Groups[self:GetParent():GetID()]["FixedDuration"] = enteredText;
    end
    TeaTimers.Update();
end

function TeaTimersOptions.Cancel()
    -- Can't copy the table here since ProfileSettings needs to point to the right place in
    -- TeaTimers_Globals.Profiles or in TeaTimers_CharSettings.Profiles
	-- FIXME: This is only restoring a small fraction of the total settings.
    TeaTimers.RestoreTableFromCopy(TeaTimers_OldProfile, TeaTimers_OldSettings);
    -- FIXME: Close context menu if it's open; it may be referring to bar that doesn't exist
    TeaTimers.Update();
end


-- -----------------------------------
-- INTERFACE OPTIONS PANEL: APPEARANCE
-- -----------------------------------
TeaTimersOptions.DefaultSelectedColor =   { 0.1, 0.6, 0.8, 1 }
TeaTimersOptions.DefaultNormalColor = { 0.7, 0.7, 0.7, 0 }

function TeaTimersOptions.UIPanel_Appearance_OnLoad(self)
    self.name = TEATIMERS.UIPANEL_APPEARANCE;
    self.parent = "TeaTimers"
    self.default = TeaTimers.ResetCharacter
    self.cancel = TeaTimersOptions.Cancel
    -- need different way to handle cancel?  users might open appearance panel without opening main panel
    InterfaceOptions_AddCategory(self)
    
    local panelName = self:GetName()
    _G[panelName.."Version"]:SetText(TEATIMERS.VERSION)
    _G[panelName.."SubText1"]:SetText(TEATIMERS.UIPANEL_APPEARANCE_SUBTEXT1)

    self.Textures.fnClick = TeaTimersOptions.OnClickTextureItem
    self.Textures.configure = function(i, btn, label) 
        btn.Bg:SetTexture(TeaTimers.LSM:Fetch("statusbar",label))
    end
    self.Textures.List.update = TeaTimersOptions.UpdateBarTextureDropDown
    self.Textures.normal_color =  { 0.7, 0.7, 0.7, 1 }

    self.Fonts.fnClick = TeaTimersOptions.OnClickFontItem
    self.Fonts.configure = function(i, btn, label) 
        local fontPath = TeaTimers.LSM:Fetch("font",label)
        btn.text:SetFont(fontPath, 12)
        btn.Bg:SetTexture(TeaTimers.LSM:Fetch("statusbar","Minimalist"))
    end
    self.Fonts.List.update = TeaTimersOptions.UpdateBarFontDropDown

    _G[panelName.."TexturesTitle"]:SetText("Texture:") -- LOCME
    _G[panelName.."FontsTitle"]:SetText("Font:") -- LOCME
end

function TeaTimersOptions.UIPanel_Appearance_OnShow(self)
    TeaTimersOptions.UIPanel_Appearance_Update();

    -- todo: Cache this? Update needs it to
    local idxCurrent = 1
    for i = 1, #textureList do
        if TeaTimers.ProfileSettings["BarTexture"] == textureList[i] then
            idxCurrent = i
            break;
        end
    end
    local idxScroll = idxCurrent - 3
    if idxScroll < 0 then
        idxScroll = 0
    end
    self.Textures.List.scrollBar:SetValue(idxScroll * self.Textures.List.buttonHeight+0.1)
    HybridScrollFrame_OnMouseWheel(self.Textures.List, 1, 0.1);

    for i = 1, #fontList do
        if TeaTimers.ProfileSettings["BarFont"] == fontList[i] then
            idxCurrent = i
            break;
        end
    end
    idxScroll = idxCurrent - 3
    if idxScroll < 0 then
        idxScroll = 0
    end
    self.Fonts.List.scrollBar:SetValue(idxScroll * self.Fonts.List.buttonHeight+0.1)
    HybridScrollFrame_OnMouseWheel(self.Fonts.List, 1, 0.1);
end

function TeaTimersOptions.UIPanel_Appearance_Update()
    local panelName = "InterfaceOptionsTeaTimersAppearancePanel";
    local panel = _G[panelName]
    if not panel or not panel:IsVisible() then return end
    
    local settings = TeaTimers.ProfileSettings;
    local barSpacingSlider = _G[panelName.."BarSpacingSlider"];
    local barPaddingSlider = _G[panelName.."BarPaddingSlider"];
    local fontSizeSlider = _G[panelName.."FontSizeSlider"];
    local fontOutlineSlider = _G[panelName.."FontOutlineSlider"];

    -- Mimic the behavior of the context menu, and force the alpha to one in the swatch
    local r,g,b = unpack(settings.BkgdColor);
    _G[panelName.."BackgroundColorButtonNormalTexture"]:SetVertexColor(r,g,b,1);

    barSpacingSlider:SetMinMaxValues(0, TEATIMERS.MAXBARSPACING);
    barSpacingSlider:SetValue(settings.BarSpacing);
    barSpacingSlider:SetValueStep(0.25);
    barPaddingSlider:SetMinMaxValues(0, TEATIMERS.MAXBARPADDING);
    barPaddingSlider:SetValue(settings.BarPadding);
    barPaddingSlider:SetValueStep(0.25);
    fontSizeSlider:SetMinMaxValues(5,20);
    fontSizeSlider:SetValue(settings.FontSize);
    fontSizeSlider:SetValueStep(0.5);
    fontOutlineSlider:SetMinMaxValues(0,2);
    fontOutlineSlider:SetValue(settings.FontOutline);
    fontOutlineSlider:SetValueStep(1);

    TeaTimersOptions.UpdateBarTextureDropDown(_G[panelName.."Textures"]);
    TeaTimersOptions.UpdateBarFontDropDown(_G[panelName.."Fonts"]);
end

-- -----------------------------------
-- INTERFACE OPTIONS PANEL: PROFILE
-- -----------------------------------

function TeaTimersOptions.UIPanel_Profile_OnLoad(self)
    self.name = TEATIMERS.UIPANEL_PROFILE;
    self.parent = "TeaTimers";
    self.default = TeaTimers.ResetCharacter;
    ---- self.cancel = TeaTimers.Cancel;
    ---- need different way to handle cancel?  users might open appearance panel without opening main panel
    InterfaceOptions_AddCategory(self);

    local panelName = self:GetName();
    _G[panelName.."Version"]:SetText(TEATIMERS.VERSION);
    _G[panelName.."SubText1"]:SetText(TEATIMERS.UIPANEL_PROFILES_SUBTEXT1);

    self.Profiles.configure = function(i, btn, label) 
        btn.Bg:SetTexture(TeaTimers.LSM:Fetch("statusbar","Minimalist"))
    end
    self.Profiles.List.update = TeaTimersOptions.UpdateProfileList
    self.Profiles.fnClick = function(self)
        local scrollPanel = self:GetParent():GetParent():GetParent()
        scrollPanel.curSel = self.text:GetText()
        TeaTimersOptions.UpdateProfileList()
    end
end

function TeaTimersOptions.UIPanel_Profile_OnShow(self)
    TeaTimersOptions.RebuildProfileList(self)
    TeaTimersOptions.UIPanel_Profile_Update();
end

function TeaTimersOptions.UIPanel_Profile_Update()
    local panelName = "InterfaceOptionsTeaTimersProfilePanel";
    local title
	-- FIXME: Use GetSpecializationInfoForClassID(UnitClass("player"), GetSpecialization()) instead of primary
    _G[panelName.."ProfilesTitle"]:SetText(TEATIMERS.UIPANEL_CURRENTPRIMARY)
    local self = _G[panelName]
    if not self:IsVisible() then return end
    TeaTimersOptions.UpdateProfileList()
end

function TeaTimersOptions.RebuildProfileList(profilePanel)
    local scrollPanel = profilePanel.Profiles
    local oldKey
    if ( scrollPanel.curSel and scrollPanel.profileMap ) then
        oldKey = scrollPanel.profileMap[scrollPanel.curSel].key
    end

    if not scrollPanel.profileNames then
        scrollPanel.profileNames = { }
    end
    scrollPanel.profileMap = { }

    local allNames = scrollPanel.profileNames
    local allRefs = scrollPanel.profileMap

    local n = 0
    local subList = TeaTimers_Profiles
    if subList then
        for profKey, rProfile in pairs(subList) do
            n = n + 1
            local profName
            if TeaTimers_Globals.Profiles[profKey] == rProfile then
                profName = 'Account: '..rProfile.name -- FIXME Localization
            else
                profName = 'Character: '..rProfile.name -- Fixme: Character-Server:
            end
            allNames[n] = profName
            allRefs[profName] = { ref = rProfile, global=true, key=profKey }
            if ( profKey == oldKey ) then
                scrollPanel.curSel = profName;
            end
        end
    end
    while n < #allNames do
        table.remove(allNames)
    end

    table.sort(allNames, function(lhs,rhs) return string.upper(lhs)<string.upper(rhs) end )
    TeaTimersOptions.UpdateProfileList()
end

function TeaTimersOptions.IsProfileNameAvailable(newName)
    if not newName or newName == "" then
        return false;
    end

    for k, profile in pairs(TeaTimers_Profiles) do
        if profile.name == newName then
            return false;
        end
    end
    return true;
end

function TeaTimersOptions.UpdateProfileList()
    local panel = _G["InterfaceOptionsTeaTimersProfilePanel"]
    local scrollPanel = panel.Profiles
    if scrollPanel.profileNames then
        local curProfile
        for n,r in pairs(scrollPanel.profileMap) do
            if r.ref == TeaTimers.ProfileSettings then
                curProfile = n
                break;
            end
        end

	if not scrollPanel.curSel or not scrollPanel.profileMap[scrollPanel.curSel] then
            scrollPanel.curSel = curProfile
        end
        local curSel = scrollPanel.curSel

        TeaTimersOptions.UpdateScrollPanel(scrollPanel, scrollPanel.profileNames, curSel, curProfile)

        local optionsPanel = scrollPanel:GetParent()
        if curSel == curProfile then
            optionsPanel.SwitchToBtn:Disable()
        else
            optionsPanel.SwitchToBtn:Enable()
        end

        if curSel == curProfile then
            optionsPanel.DeleteBtn:Disable()
        else
            optionsPanel.DeleteBtn:Enable()
        end

        local curEntry = optionsPanel.NewName:GetText()
        if TeaTimersOptions.IsProfileNameAvailable(curEntry) then
            optionsPanel.RenameBtn:Enable()
            optionsPanel.CopyBtn:Enable()
        else
            optionsPanel.RenameBtn:Disable()
            optionsPanel.CopyBtn:Disable()
        end

        local rSelectedProfile = scrollPanel.profileMap[curSel].ref;
        local rSelectedKey = scrollPanel.profileMap[curSel].key;
        if ( rSelectedProfile and rSelectedKey and TeaTimers_Globals.Profiles[rSelectedKey] == rSelectedProfile ) then
            optionsPanel.PrivateBtn:Show();
            optionsPanel.PublicBtn:Hide();
        else
            optionsPanel.PrivateBtn:Hide();
            optionsPanel.PublicBtn:Show();
        end
    end
end

function TeaTimersOptions.UIPanel_Profile_SwitchToSelected(panel)
    local scrollPanel = panel.Profiles
    local curSel = scrollPanel.curSel
    if curSel then
        TeaTimers.ChangeProfile( scrollPanel.profileMap[curSel].key )
        TeaTimersOptions.UpdateProfileList()
    end
end

StaticPopupDialogs["TEATIMERS.CONFIRMDLG"] = {
    button1 = YES,
    button2 = NO,
    timeout = 0,
    hideOnEscape = 1,
    OnShow = function(self)
        self.oldStrata = self:GetFrameStrata()
        self:SetFrameStrata("TOOLTIP")
    end,
    OnHide = function(self)
        if self.oldStrata then 
            self:SetFrameStrata(self.oldStrata) 
        end
    end
};
function TeaTimersOptions.UIPanel_Profile_DeleteSelected(panel)
    local scrollPanel = panel.Profiles
    local curSel = scrollPanel.curSel
    if curSel then
        local k = scrollPanel.profileMap[curSel].key
        local dlgInfo = StaticPopupDialogs["TEATIMERS.CONFIRMDLG"]
        dlgInfo.text = "Are you sure you want to delete the profile: ".. curSel .."?"
        dlgInfo.OnAccept = function(self, data)
            if TeaTimers_Profiles[k] == TeaTimers.ProfileSettings then
                print("Won't delete the active profile!")
            else
                TeaTimers_Profiles[k] = nil;
                if TeaTimers_Globals.Profiles[k] then
                    print("deleted account-wide profile", TeaTimers_Globals.Profiles[k].name) -- LOCME
                    TeaTimers_Globals.Profiles[k] = nil;
                elseif TeaTimers_CharSettings.Profiles[k] then
                    print("deleted character profile", TeaTimers_CharSettings.Profiles[k].name) -- LOCME
                    TeaTimers_CharSettings.Profiles[k] = nil;
                end
                TeaTimersOptions.RebuildProfileList(panel)
            end
        end
        StaticPopup_Show("TEATIMERS.CONFIRMDLG");
    end
end

function TeaTimersOptions.UIPanel_Profile_CopySelected(panel)
    local scrollPanel = panel.Profiles
    local curSel = scrollPanel.curSel
    local edit = panel.NewName
    local newName = edit:GetText()
    edit:ClearFocus()
    if scrollPanel.curSel and TeaTimersOptions.IsProfileNameAvailable(newName) then
        local keyNew = TeaTimers.CreateProfile(CopyTable(scrollPanel.profileMap[curSel].ref), nil, newName)
        TeaTimers.ChangeProfile(keyNew)
        TeaTimersOptions.RebuildProfileList(panel)
        edit:SetText("");
        print("Copied",curSel,"to",newName,"and made it the active profile")
    end
end


function TeaTimersOptions.UIPanel_Profile_RenameSelected(panel)
    local scrollPanel = panel.Profiles
    local edit = panel.NewName
    local newName = edit:GetText()
    edit:ClearFocus()
    if scrollPanel.curSel and TeaTimersOptions.IsProfileNameAvailable(newName) then
        local key = scrollPanel.profileMap[scrollPanel.curSel].key
        print("Renaming profile",TeaTimers_Profiles[key].name,"to",newName)
        TeaTimers_Profiles[key].name = newName;
        edit:SetText("");
        TeaTimersOptions.RebuildProfileList(panel)
    end
end

function TeaTimersOptions.UIPanel_Profile_PublicizeSelected(panel)
    local scrollPanel = panel.Profiles
    if scrollPanel.curSel then
        local ref = scrollPanel.profileMap[scrollPanel.curSel].ref
        local key = scrollPanel.profileMap[scrollPanel.curSel].key
        TeaTimers_Globals.Profiles[key] = ref
        TeaTimers_CharSettings.Profiles[key] = nil
        TeaTimersOptions.RebuildProfileList(panel)
    end
end

function TeaTimersOptions.UIPanel_Profile_PrivatizeSelected(panel)
    local scrollPanel = panel.Profiles
    if scrollPanel.curSel then
        local ref = scrollPanel.profileMap[scrollPanel.curSel].ref
        local key = scrollPanel.profileMap[scrollPanel.curSel].key
        TeaTimers_Globals.Profiles[key] = nil
        TeaTimers_CharSettings.Profiles[key] = ref
        TeaTimersOptions.RebuildProfileList(panel)
    end
end

-----

function TeaTimersOptions.OnClickTextureItem(self)
    TeaTimers.ProfileSettings["BarTexture"] = self.text:GetText()
    TeaTimers.Update()
    TeaTimersOptions.UIPanel_Appearance_Update()
end


function TeaTimersOptions.OnClickFontItem(self)
    TeaTimers.ProfileSettings["BarFont"] = self.text:GetText()
    TeaTimers.Update()
    TeaTimersOptions.UIPanel_Appearance_Update()
end



function TeaTimersOptions.ChooseColor(variable)
    info = UIDropDownMenu_CreateInfo();
    info.r, info.g, info.b, info.opacity = unpack(TeaTimers.ProfileSettings[variable]);
    info.opacity = 1 - info.opacity;
    info.hasOpacity = true;
    info.opacityFunc = TeaTimersOptions.SetOpacity;
    info.swatchFunc = TeaTimersOptions.SetColor;
    info.cancelFunc = TeaTimersOptions.CancelColor;
    info.extraInfo = variable;
    -- Not sure if I should leave this state around or not.  It seems like the
    -- correct strata to have it at anyway, so I'm going to leave it there for now
    ColorPickerFrame:SetFrameStrata("FULLSCREEN_DIALOG");
    OpenColorPicker(info);
end

function TeaTimersOptions.SetColor()
    local variable = ColorPickerFrame.extraInfo;
    local r,g,b = ColorPickerFrame:GetColorRGB();
    TeaTimers.ProfileSettings[variable][1] = r;
    TeaTimers.ProfileSettings[variable][2] = g;
    TeaTimers.ProfileSettings[variable][3] = b;
    TeaTimers.Update();
    TeaTimersOptions.UIPanel_Appearance_Update();
end

function TeaTimersOptions.SetOpacity()
    local variable = ColorPickerFrame.extraInfo;
    TeaTimers.ProfileSettings[variable][4] = 1 - OpacitySliderFrame:GetValue();
    TeaTimers.Update();
    TeaTimersOptions.UIPanel_Appearance_Update();
end

function TeaTimersOptions.CancelColor(previousValues)
    if ( previousValues ) then
        local variable = ColorPickerFrame.extraInfo;
        TeaTimers.ProfileSettings[variable] = {previousValues.r, previousValues.g, previousValues.b, previousValues.opacity};
        TeaTimers.Update();
        TeaTimersOptions.UIPanel_Appearance_Update();
    end
end

function TeaTimersOptions.UIPanel_Appearance_OnSizeChanged(self)
    -- Despite my best efforts, the scroll bars insist on being outside the width of their
    local mid = self:GetWidth()/2 --+ _G[self:GetName().."TexturesListScrollBar"]:GetWidth()
    local textures = self.Textures
    local leftTextures = textures:GetLeft()
    if mid and mid > 0 and textures and leftTextures then
        local ofs = leftTextures - self:GetLeft()
        textures:SetWidth(mid - ofs)
    end
end


function TeaTimersOptions.OnScrollFrameSized(self)
    local old_value = self.scrollBar:GetValue();
    local scrollFrame = self:GetParent();

    HybridScrollFrame_CreateButtons(self, "TeaTimersScrollItemTemplate")
    --scrollFrame.Update(scrollFrame)

    local max_value = self.range or self:GetHeight()
    self.scrollBar:SetValue(min(old_value, max_value));
    -- Work around a bug in HybridScrollFrame; it can't scroll by whole items (wow 4.1)
    --self.stepSize = self.buttons[1]:GetHeight()*.9
end


function TeaTimersOptions.UpdateScrollPanel(panel, list, selected, checked)
    local Value = _G[panel:GetName().."Value"]
    Value:SetText(checked)

    local PanelList = panel.List
    local buttons = PanelList.buttons
    HybridScrollFrame_Update(PanelList, #(list) * buttons[1]:GetHeight() , PanelList:GetHeight())

    local numButtons = #buttons;
    local scrollOffset = HybridScrollFrame_GetOffset(PanelList);
    local label;
    for i = 1, numButtons do
        local idx = i + scrollOffset
        label = list[idx]
        if ( label ) then
            buttons[i]:Show();
            buttons[i].text:SetText(label);

            if ( label == checked ) then
                buttons[i].Check:Show();
            else
                buttons[i].Check:Hide();
            end
            if ( label == selected ) then
                local color = panel.selected_color
                if not color then color = TeaTimersOptions.DefaultSelectedColor end
                buttons[i].Bg:SetVertexColor(unpack(color));
            else
                local color = panel.normal_color
                if not color then color = TeaTimersOptions.DefaultNormalColor end
                buttons[i].Bg:SetVertexColor(unpack(color));
            end

            panel.configure(i, buttons[i], label)
        else
            buttons[i]:Hide();
        end
    end
end

--function TeaTimersOptions.OnScrollFrameScrolled(self)
    --local scrollPanel = self:GetParent()
    --local fn = scrollPanel.Update
    --if fn then fn(scrollPanel) end
--end
--
function TeaTimersOptions.UpdateBarTextureDropDown()
    local scrollPanel = _G["InterfaceOptionsTeaTimersAppearancePanelTextures"]
    TeaTimersOptions.UpdateScrollPanel(scrollPanel, textureList, TeaTimers.ProfileSettings.BarTexture, TeaTimers.ProfileSettings.BarTexture)
end

function TeaTimersOptions.UpdateBarFontDropDown()
    local scrollPanel = _G["InterfaceOptionsTeaTimersAppearancePanelFonts"]
    TeaTimersOptions.UpdateScrollPanel(scrollPanel, fontList, nil, TeaTimers.ProfileSettings.BarFont)
end

-- --------
-- BAR GUI
-- --------

TeaTimersMenuBar.CurrentBar = { groupID = 1, barID = 1 };        -- a dirty hack, i know.

StaticPopupDialogs["TEATIMERS.CHOOSENAME_DIALOG"] = {
    text = TEATIMERS.CHOOSENAME_DIALOG,
    button1 = ACCEPT,
    button2 = CANCEL,
    hasEditBox = 1,
    editBoxWidth = 300,
    maxLetters = 0,
    OnAccept = function(self)
        local text = self.editBox:GetText();
        local variable = self.variable;
        if ( nil ~= variable ) then
            TeaTimersMenuBar.BarMenu_ChooseName(text, variable);
        end
    end,
    EditBoxOnEnterPressed = function(self)
        StaticPopupDialogs["TEATIMERS.CHOOSENAME_DIALOG"].OnAccept(self:GetParent())
        self:GetParent():Hide();
    end,
    EditBoxOnEscapePressed = function(self)
        self:GetParent():Hide();
    end,
    OnHide = function(self)
    -- Removed for wow 3.3.5, it seems like there is a focu stack
    -- now that obsoletes this anyway.  If not, there isn't a 
    -- single ChatFrameEditBox anymore, there's ChatFrame1EditBox etc.
        -- if ( ChatFrameEditBox:IsVisible() ) then
        --    ChatFrameEditBox:SetFocus();
        -- end
        self.editBox:SetText("");
    end,
    timeout = 0,
    whileDead = 1,
    hideOnEscape = 1,
};

TeaTimersMenuBar.BarMenu_MoreOptions = {
    { VariableName = "Enabled", MenuText = TEATIMERS.BARMENU_ENABLE },
    { VariableName = "AuraName", MenuText = TEATIMERS.BARMENU_CHOOSENAME, Type = "Dialog", DialogText = "CHOOSENAME_DIALOG" },
    { VariableName = "BuffOrDebuff", MenuText = TEATIMERS.BARMENU_BUFFORDEBUFF, Type = "Submenu" },
    { VariableName = "Options", MenuText = "Settings", Type = "Submenu" },
    {},
    { VariableName = "TimeFormat", MenuText = TEATIMERS.BARMENU_TIMEFORMAT, Type = "Submenu" },
    { VariableName = "Show", MenuText = TEATIMERS.BARMENU_SHOW, Type = "Submenu" },
    { VariableName = "VisualCastTime", MenuText = TEATIMERS.BARMENU_VISUALCASTTIME, Type = "Submenu" },
    { VariableName = "BlinkSettings", MenuText = "Blink Settings", Type = "Submenu" }, -- LOCME
    { VariableName = "BarColor", MenuText = TEATIMERS.BARMENU_BARCOLOR, Type = "Color" },
    {},
    { VariableName = "ImportExport", MenuText = TEATIMERS.BARMENU_IMPORTEXPORT, Type = "Dialog", DialogText = "IMPORTEXPORT_DIALOG" },
}

TeaTimersMenuBar.BarMenu_SubMenus = {
    -- the keys on this table need to match the settings variable names
    BuffOrDebuff = {
          { Setting = "HELPFUL", MenuText = TEATIMERS.BARMENU_HELPFUL },
          { Setting = "HARMFUL", MenuText = TEATIMERS.BARMENU_HARMFUL },
          { Setting = "TOTEM", MenuText = TEATIMERS.BARMENU_TOTEM },
          { Setting = "CASTCD", MenuText = TEATIMERS.BARMENU_CASTCD },
          { Setting = "BUFFCD", MenuText = TEATIMERS.BARMENU_BUFFCD },
-- Now that Victory Rush adds a buff when you can use it, this confusing option is being removed.
-- The code that drives it remains so that any existing users' bars won't break.
--          { Setting = "USABLE", MenuText = TEATIMERS.BARMENU_USABLE },
          { Setting = "EQUIPSLOT", MenuText = TEATIMERS.BARMENU_EQUIPSLOT },
          { Setting = "POWER", MenuText = TEATIMERS.BARMENU_POWER }
    },
    TimeFormat = {
          { Setting = "Fmt_SingleUnit", MenuText = TEATIMERS.FMT_SINGLEUNIT },
          { Setting = "Fmt_TwoUnits", MenuText = TEATIMERS.FMT_TWOUNITS },
          { Setting = "Fmt_Float", MenuText = TEATIMERS.FMT_FLOAT },
    },
    Unit = {
        { Setting = "player", MenuText = TEATIMERS.BARMENU_PLAYER },
        { Setting = "target", MenuText = TEATIMERS.BARMENU_TARGET },
        { Setting = "targettarget", MenuText = TEATIMERS.BARMENU_TARGETTARGET },
        { Setting = "focus", MenuText = TEATIMERS.BARMENU_FOCUS },
        { Setting = "pet", MenuText = TEATIMERS.BARMENU_PET },
        { Setting = "vehicle", MenuText = TEATIMERS.BARMENU_VEHICLE },
        { Setting = "lastraid", MenuText = TEATIMERS.BARMENU_LAST_RAID },
    },
    DebuffUnit = {
        { Setting = "player", MenuText = TEATIMERS.BARMENU_PLAYER },
        { Setting = "target", MenuText = TEATIMERS.BARMENU_TARGET },
        { Setting = "targettarget", MenuText = TEATIMERS.BARMENU_TARGETTARGET },
        { Setting = "focus", MenuText = TEATIMERS.BARMENU_FOCUS },
        { Setting = "pet", MenuText = TEATIMERS.BARMENU_PET },
        { Setting = "vehicle", MenuText = TEATIMERS.BARMENU_VEHICLE },
    },
    Opt_HELPFUL = {
      { VariableName = "Unit", MenuText = TEATIMERS.BARMENU_CHOOSEUNIT, Type = "Submenu" },
      { VariableName = "bDetectExtends", MenuText = "Track duration increases" }, -- LOCME
      { VariableName = "OnlyMine", MenuText = TEATIMERS.BARMENU_ONLYMINE },
      { VariableName = "show_all_stacks", MenuText = "Sum stacks from all casters" },
    },
    Opt_HARMFUL = {
      { VariableName = "DebuffUnit", MenuText = TEATIMERS.BARMENU_CHOOSEUNIT, Type = "Submenu" },
      { VariableName = "bDetectExtends", MenuText = "Track duration increases" }, -- LOCME
      { VariableName = "OnlyMine", MenuText = TEATIMERS.BARMENU_ONLYMINE },
      { VariableName = "show_all_stacks", MenuText = "Sum stacks from all casters" },
    },
    Opt_TOTEM = {},
    Opt_CASTCD = 
    {
        { VariableName = "append_cd", MenuText = "Append \"CD\"" }, -- LOCME
        { VariableName = "show_charges", MenuText = "Show first and last charge CD" }, -- LOCME
    },
    Opt_EQUIPSLOT = 
    {
        { VariableName = "append_cd", MenuText = "Append \"CD\"" }, -- LOCME
    },
    Opt_POWER = 
    {
      { VariableName = "Unit", MenuText = TEATIMERS.BARMENU_CHOOSEUNIT, Type = "Submenu" },
      { VariableName = "power_sole", MenuText = "Only Show When Primary" }, -- LOCME
    },
    Opt_BUFFCD = 
    {
        { VariableName = "buffcd_duration", MenuText = "Cooldown duration...", Type = "Dialog", DialogText = "BUFFCD_DURATION_DIALOG", Numeric=true },
        { VariableName = "buffcd_reset_spells", MenuText = "Reset on buff...", Type = "Dialog", DialogText = "BUFFCD_RESET_DIALOG" },
        { VariableName = "append_cd", MenuText = "Append \"CD\"" }, -- LOCME
    },
    Opt_USABLE =
    {
        { VariableName = "usable_duration", MenuText = "Usable duration...",  Type = "Dialog", DialogText = "USABLE_DURATION_DIALOG", Numeric=true },
        { VariableName = "append_usable", MenuText = "Append \"Usable\"" }, -- LOCME
    },
    EquipmentSlotList =
    {
        { Setting = "1", MenuText = TEATIMERS.ITEM_NAMES[1] },
        { Setting = "2", MenuText = TEATIMERS.ITEM_NAMES[2] },
        { Setting = "3", MenuText = TEATIMERS.ITEM_NAMES[3] },
        { Setting = "4", MenuText = TEATIMERS.ITEM_NAMES[4] },
        { Setting = "5", MenuText = TEATIMERS.ITEM_NAMES[5] },
        { Setting = "6", MenuText = TEATIMERS.ITEM_NAMES[6] },
        { Setting = "7", MenuText = TEATIMERS.ITEM_NAMES[7] },
        { Setting = "8", MenuText = TEATIMERS.ITEM_NAMES[8] },
        { Setting = "9", MenuText = TEATIMERS.ITEM_NAMES[9] },
        { Setting = "10", MenuText = TEATIMERS.ITEM_NAMES[10] },
        { Setting = "11", MenuText = TEATIMERS.ITEM_NAMES[11] },
        { Setting = "12", MenuText = TEATIMERS.ITEM_NAMES[12] },
        { Setting = "13", MenuText = TEATIMERS.ITEM_NAMES[13] },
        { Setting = "14", MenuText = TEATIMERS.ITEM_NAMES[14] },
        { Setting = "15", MenuText = TEATIMERS.ITEM_NAMES[15] },
        { Setting = "16", MenuText = TEATIMERS.ITEM_NAMES[16] },
        { Setting = "17", MenuText = TEATIMERS.ITEM_NAMES[17] },
        { Setting = "18", MenuText = TEATIMERS.ITEM_NAMES[18] },
        { Setting = "19", MenuText = TEATIMERS.ITEM_NAMES[19] },
    },
    PowerTypeList =
    {
    },
    VisualCastTime = {
        { VariableName = "vct_enabled", MenuText = TEATIMERS.BARMENU_VCT_ENABLE },
        { VariableName = "vct_color", MenuText = TEATIMERS.BARMENU_VCT_COLOR, Type = "Color" },
        { VariableName = "vct_spell", MenuText = TEATIMERS.BARMENU_VCT_SPELL, Type = "Dialog", DialogText = "CHOOSE_VCT_SPELL_DIALOG" },
        { VariableName = "vct_extra", MenuText = TEATIMERS.BARMENU_VCT_EXTRA, Type = "Dialog", DialogText = "CHOOSE_VCT_EXTRA_DIALOG", Numeric=true },
    },
    Show = {
        { VariableName = "show_icon",      MenuText = TEATIMERS.BARMENU_SHOW_ICON },
        { VariableName = "show_text",      MenuText = TEATIMERS.BARMENU_SHOW_TEXT },
        { VariableName = "show_count",     MenuText = TEATIMERS.BARMENU_SHOW_COUNT },
        { VariableName = "show_time",      MenuText = TEATIMERS.BARMENU_SHOW_TIME },
        { VariableName = "show_spark",     MenuText = TEATIMERS.BARMENU_SHOW_SPARK },
        { VariableName = "show_mypip",     MenuText = TEATIMERS.BARMENU_SHOW_MYPIP },
        { VariableName = "show_ttn1",      MenuText = TEATIMERS.BARMENU_SHOW_TTN1 },
        { VariableName = "show_ttn2",      MenuText = TEATIMERS.BARMENU_SHOW_TTN2 },
        { VariableName = "show_ttn3",      MenuText = TEATIMERS.BARMENU_SHOW_TTN3 },
        { VariableName = "show_text_user", MenuText = TEATIMERS.BARMENU_SHOW_TEXT_USER, Type = "Dialog", DialogText = "CHOOSE_OVERRIDE_TEXT", Checked = function(settings) return "" ~= settings.show_text_user end },
    },
    BlinkSettings = {
        { VariableName = "blink_enabled", MenuText = TEATIMERS.BARMENU_VCT_ENABLE },
        { VariableName = "blink_label", MenuText = "Bar text while blinking...", Type = "Dialog", DialogText="CHOOSE_BLINK_TITLE_DIALOG" }, 
        { VariableName = "MissingBlink", MenuText = "Bar color when blinking...", Type = "Color" }, -- LOCME
        { VariableName = "blink_ooc", MenuText = "Blink out of combat" }, -- LOCME
        { VariableName = "blink_boss", MenuText = "Blink only for bosses" }, -- LOCME
    },
};

TeaTimersMenuBar.VariableRedirects =
{
  DebuffUnit = "Unit",
  EquipmentSlotList = "AuraName",
  PowerTypeList = "AuraName",
}

function TeaTimersMenuBar.ShowMenu(bar)
    TeaTimersMenuBar.CurrentBar["barID"] = bar:GetID();
    TeaTimersMenuBar.CurrentBar["groupID"] = bar:GetParent():GetID();
    if not TeaTimersMenuBar.DropDown then
        TeaTimersMenuBar.DropDown = CreateFrame("Frame", "TeaTimersDropDown", nil, "TeaTimers_DropDownTemplate")
    end

    -- There's no OpenDropDownMenu that forces it to show in the new place,
    -- so we have to check if the first Toggle opened or closed it
    ToggleDropDownMenu(1, nil, TeaTimersMenuBar.DropDown, "cursor", 0, 0);
    if not DropDownList1:IsShown() then
        ToggleDropDownMenu(1, nil, TeaTimersMenuBar.DropDown, "cursor", 0, 0);
    end
end

function TeaTimersMenuBar.BarMenu_AddButton(barSettings, i_desc, i_parent)
    info = UIDropDownMenu_CreateInfo();
    local item_type = i_desc["Type"];
    info.text = i_desc["MenuText"];
    local varSettings
    if ( nil ~= i_desc["Setting"]) then
        item_type = "SetVar"
        local v = TeaTimersMenuBar.VariableRedirects[i_parent] or i_parent
        varSettings = barSettings[v]
    else
        info.value = i_desc["VariableName"];
        varSettings = barSettings[info.value];
    end
    
    if ( not varSettings and (item_type == "Check" or item_type == "Color") ) then
        print (string.format("NTK: Could not find %s in", info.value), barSettings); 
        return
    end
    
    info.hasArrow = false;
    local b = i_desc["Checked"]
    if b then
        if type(b) == "function" then
            info.checked = b(barSettings)
        else
            info.checked = b
        end
    end

    info.keepShownOnClick = true;
    info.notCheckable = false; -- indent everything
    info.hideUnCheck = true; -- but hide the empty checkbox/radio

    if ( not item_type and not text and not info.value ) then
        info.func = TeaTimersMenuBar.BarMenu_IgnoreToggle;
        info.disabled = true;
    elseif ( nil == item_type or item_type == "Check" ) then
        info.func = TeaTimersMenuBar.BarMenu_ToggleSetting;
        info.checked = (nil ~= varSettings and varSettings);
        info.hideUnCheck = nil;
        info.isNotRadio = true;
    elseif ( item_type == "SetVar" ) then
        info.func = TeaTimersMenuBar.BarMenu_ChooseSetting;
        info.value = i_desc["Setting"];
        info.checked = (varSettings == info.value);
        info.hideUnCheck = nil;
        info.keepShownOnClick = false;
    elseif ( item_type == "Submenu" ) then
        info.hasArrow = true;
        info.isNotRadio = true;
        info.func = TeaTimersMenuBar.BarMenu_IgnoreToggle;
    elseif ( item_type == "Dialog" ) then
        info.func = TeaTimersMenuBar.BarMenu_ShowNameDialog;
        info.keepShownOnClick = false;
        info.value = {variable = i_desc.VariableName, text = i_desc.DialogText, numeric = i_desc.Numeric };
    elseif ( item_type == "Color" ) then
        info.hasColorSwatch = 1;
        info.hasOpacity = true;
        info.r = varSettings.r;
        info.g = varSettings.g;
        info.b = varSettings.b;
        info.opacity = 1 - varSettings.a;
        info.swatchFunc = TeaTimersMenuBar.BarMenu_SetColor;
        info.opacityFunc = TeaTimersMenuBar.BarMenu_SetOpacity;
        info.cancelFunc = TeaTimersMenuBar.BarMenu_CancelColor;

        info.func = UIDropDownMenuButton_OpenColorPicker;
        info.keepShownOnClick = false;
    end
  
    UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
    
    -- Code to get the button copied from UIDropDownMenu_AddButton
    local level = UIDROPDOWNMENU_MENU_LEVEL;
    local listFrame = _G["DropDownList"..level];
    local index = listFrame and (listFrame.numButtons) or 1;
    local listFrameName = listFrame:GetName();
    local buttonName = listFrameName.."Button"..index;
    if ( item_type == "Color" ) then
        -- Sadly, extraInfo isn't a field propogated to the button
        local button = _G[buttonName];
        button.extraInfo = info.value;
    end
    if ( info.hideUnCheck ) then
        local checkBG = _G[buttonName.."UnCheck"];
        checkBG:Hide();
    end
end

function TeaTimersMenuBar.BarMenu_Initialize()
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local barSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID];

    if ( barSettings.MissingBlink.a == 0 ) then
        barSettings.blink_enabled = false;
    end
    TeaTimersMenuBar.BarMenu_SubMenus.Options = TeaTimersMenuBar.BarMenu_SubMenus["Opt_"..barSettings.BuffOrDebuff];
   
    if ( UIDROPDOWNMENU_MENU_LEVEL > 1 ) then
        if ( UIDROPDOWNMENU_MENU_VALUE == "VisualCastTime" ) then
            -- Create a summary title for the visual cast time submenu
            local title = "";
            if ( barSettings.vct_spell and "" ~= barSettings.vct_spell ) then
                title = title .. barSettings.vct_spell;
            end
            local fExtra = tonumber(barSettings.vct_extra);
            if ( fExtra and fExtra > 0 ) then
                if ("" ~= title) then
                    title = title .. " + ";
                end
                title = title .. string.format("%0.1fs", fExtra);
            end
            if ( "" ~= title ) then
                local info = UIDropDownMenu_CreateInfo();
                info.text = title;
                info.isTitle = true;
                info.notCheckable = true; -- unindent
                UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
            end
        end
        
        local subMenus = TeaTimersMenuBar.BarMenu_SubMenus;
        for index, value in ipairs(subMenus[UIDROPDOWNMENU_MENU_VALUE]) do
            TeaTimersMenuBar.BarMenu_AddButton(barSettings, value, UIDROPDOWNMENU_MENU_VALUE);
        end

        if ( false == barSettings.OnlyMine and UIDROPDOWNMENU_MENU_LEVEL == 2 ) then
            TeaTimersMenuBar.BarMenu_UncheckAndDisable(2, "bDetectExtends", false);
        end
        return;
    end
    
    -- show name
    if ( barSettings.AuraName ) and ( barSettings.AuraName ~= "" ) then
        local info = UIDropDownMenu_CreateInfo();
        info.text = TeaTimers.PrettyName(barSettings);
        info.isTitle = true;
        info.notCheckable = true; --unindent
        UIDropDownMenu_AddButton(info);
    end

    local moreOptions = TeaTimersMenuBar.BarMenu_MoreOptions;
    for index, value in ipairs(moreOptions) do
        TeaTimersMenuBar.BarMenu_AddButton(barSettings, moreOptions[index]);
    end

    TeaTimersMenuBar.BarMenu_UpdateSettings(barSettings);
end

function TeaTimersMenuBar.BarMenu_IgnoreToggle(self, a1, a2, checked)
    local button = TeaTimersMenuBar.BarMenu_GetItem(TeaTimersMenuBar.BarMenu_GetItemLevel(self), self.value);
    if ( button ) then
        local checkName = button:GetName() .. "Check";
        _G[checkName]:Hide();
        button.checked = false;
    end
end

function TeaTimersMenuBar.BarMenu_ToggleSetting(self, a1, a2, checked)
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local barSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID];
    barSettings[self.value] = self.checked;
    local level = TeaTimersMenuBar.BarMenu_GetItemLevel(self);
    
    if ( self.value == "OnlyMine" ) then 
        if ( false == self.checked ) then
            TeaTimersMenuBar.BarMenu_UncheckAndDisable(level, "bDetectExtends", false);
        else
            TeaTimersMenuBar.BarMenu_EnableItem(level, "bDetectExtends");
            TeaTimersMenuBar.BarMenu_CheckItem(level, "show_all_stacks", false);
        end
    elseif ( self.value == "blink_enabled" ) then
        if ( true == self.checked and barSettings.MissingBlink.a == 0 ) then
            barSettings.MissingBlink.a = 0.5
        end
    elseif ( self.value == "show_all_stacks" ) then
        if ( true == self.checked ) then
            TeaTimersMenuBar.BarMenu_CheckItem(level, "OnlyMine", false);
        end
    end
    TeaTimers.Bar_Update(groupID, barID);
end

function TeaTimersMenuBar.BarMenu_GetItemLevel(i_button)
    local path = i_button:GetName();
    local levelStr = path:match("%d+");
    return tonumber(levelStr);
end

function TeaTimersMenuBar.BarMenu_GetItem(i_level, i_valueName)
    local listFrame = _G["DropDownList"..i_level];
    local listFrameName = listFrame:GetName();
    local n = listFrame.numButtons;
    for index=1,n do
        local button = _G[listFrameName.."Button"..index];
        local txt;
        if ( type(button.value) == "table" ) then
            txt = button.value.variable;
        else
            txt = button.value;
        end
        if ( txt == i_valueName ) then
            return button;
        end
    end
    return nil;
end

function TeaTimersMenuBar.BarMenu_CheckItem(i_level, i_valueName, i_bCheck)
    local button = TeaTimersMenuBar.BarMenu_GetItem(i_level, i_valueName);
    if ( button ) then
        local checkName = button:GetName() .. "Check";
        local check = _G[checkName];
        if ( i_bCheck ) then
            check:Show();
            button.checked = true;
        else
            check:Hide();
            button.checked = false;
        end
        TeaTimersMenuBar.BarMenu_ToggleSetting(button);
    end
end

function TeaTimersMenuBar.BarMenu_EnableItem(i_level, i_valueName)
    local button = TeaTimersMenuBar.BarMenu_GetItem(i_level, i_valueName)
    if ( button ) then
        button:Enable();
    end
end

function TeaTimersMenuBar.BarMenu_UncheckAndDisable(i_level, i_valueName)
    local button = TeaTimersMenuBar.BarMenu_GetItem(i_level, i_valueName);
    if ( button ) then
        TeaTimersMenuBar.BarMenu_CheckItem(i_level, i_valueName, false);
        button:Disable();
    end
end

function TeaTimersMenuBar.BarMenu_UpdateSettings(barSettings)
    local type = barSettings.BuffOrDebuff;
    
    -- Set up the options submenu to the corrent name and contents
    local Opt = TeaTimersMenuBar.BarMenu_SubMenus["Opt_"..type];
    if ( not Opt ) then Opt = {} end
    TeaTimersMenuBar.BarMenu_SubMenus.Options = Opt;
    local button = TeaTimersMenuBar.BarMenu_GetItem(1, "Options");
    if button then
        local arrow = _G[button:GetName().."ExpandArrow"]
        local lbl = ""
        if #Opt == 0 then
            lbl = lbl .. "No "
            button:Disable();
            arrow:Hide();
        else
            button:Enable();
            arrow:Show();
        end
        -- LOCME
        lbl = lbl .. TEATIMERS["BARMENU_"..type].. " Settings";
        button:SetText(lbl);
    end

    -- Set up the aura name menu option to behave the right way
    if ( type == "EQUIPSLOT" ) then
        button = TeaTimersMenuBar.BarMenu_GetItem(1, "AuraName");
        if ( button ) then
            button.oldvalue = button.value
        else
            button = TeaTimersMenuBar.BarMenu_GetItem(1, "PowerTypeList")
        end
        if ( button ) then
            local arrow = _G[button:GetName().."ExpandArrow"]
            arrow:Show();
            button.hasArrow = true
            button.value = "EquipmentSlotList"
            button:SetText(TEATIMERS.BARMENU_CHOOSESLOT)
            -- TODO: really should disable the button press verb somehow
        end
    elseif ( type == "POWER" ) then
        button = TeaTimersMenuBar.BarMenu_GetItem(1, "AuraName");
        if ( button ) then
          button.oldvalue = button.value
        else
            button = TeaTimersMenuBar.BarMenu_GetItem(1, "EquipmentSlotList")
        end
        if ( button ) then
            local arrow = _G[button:GetName().."ExpandArrow"]
            arrow:Show();
            button.hasArrow = true
            button.value = "PowerTypeList"
            button:SetText(TEATIMERS.BARMENU_CHOOSEPOWER)
            -- TODO: really should disable the button press verb somehow
        end
    else
        button = TeaTimersMenuBar.BarMenu_GetItem(1, "EquipmentSlotList");
        if not button then button = TeaTimersMenuBar.BarMenu_GetItem(1, "PowerTypeList") end
        if ( button ) then
            local arrow = _G[button:GetName().."ExpandArrow"]
            arrow:Hide();
            button.hasArrow = false
            if button.oldvalue then button.value = button.oldvalue end
            button:SetText(TEATIMERS.BARMENU_CHOOSENAME)
        end
    end
end

function TeaTimersMenuBar.BarMenu_ChooseSetting(self, a1, a2, checked)
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local barSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID]
    local v = TeaTimersMenuBar.VariableRedirects[UIDROPDOWNMENU_MENU_VALUE] or UIDROPDOWNMENU_MENU_VALUE
    barSettings[v] = self.value;
    TeaTimers.Bar_Update(groupID, barID);
    
    if ( v == "BuffOrDebuff" ) then
        TeaTimersMenuBar.BarMenu_UpdateSettings(barSettings)
    end
end

-- TODO: There has to be a better way to do this, this has pretty bad user feel
function TeaTimersMenuBar.EditBox_Numeric_OnTextChanged(self, isUserInput)
    if ( isUserInput ) then
        local txt = self:GetText();
        local culled = txt:gsub("[^0-9.]",""); -- Remove non-digits
        local iPeriod = culled:find("[.]");
        if ( nil ~= iPeriod ) then
            local before = culled:sub(1, iPeriod);
            local after = string.gsub( culled:sub(iPeriod+1), "[.]", "" );
            culled = before .. after;
        end
        if ( txt ~= culled ) then
            self:SetText(culled);
        end
    end
    
    if ( TeaTimersMenuBar.EditBox_Original_OnTextChanged ) then
        TeaTimersMenuBar.EditBox_Original_OnTextChanged(self, isUserInput);
    end
end

TeaTimersImportExport = {}
function TeaTimersImportExport.CombineKeyValue(key,value)
    local vClean = value
    if type(vClean) == "string" and value:byte(1) ~= 123 then
        if (tostring(tonumber(vClean)) == vClean) or vClean == "true" or vClean == "false" then
            vClean = '"' .. vClean .. '"'
        elseif (vClean:find(",") or vClean:find("}") or vClean:byte(1) == 34) then
            vClean = '"' .. tostring(value):gsub('"', '\\"') .. '"'
        end
    end

    if key then
        -- not possible for key to contain = right now, so we don't have to sanitize it
        return key .. "=" .. tostring(vClean)
    else
        return vClean
    end
end

function TeaTimersImportExport.TableToString(v)
    local i = 1
    local ret= "{"
    for index, value in pairs(v) do
        if i ~= 1 then
            ret = ret .. ","
        end
        local k
        if index ~= i then
            k = TEATIMERS.SHORTENINGS[index] or index
        end
        if  type(value) == "table" then
            value = TeaTimersImportExport.TableToString(value)
        end
        ret = ret .. TeaTimersImportExport.CombineKeyValue(k, value)
        i = i+1;
    end
    ret = ret .. "}"
    return ret
end

function TeaTimersImportExport.ExportBarSettingsToString(barSettings)
    local pruned = CopyTable(barSettings)
    TeaTimers.RemoveDefaultValues(pruned, TEATIMERS.BAR_DEFAULTS)
    return 'bv1:' .. TeaTimersImportExport.TableToString(pruned);
end

--[[ Test Cases
/script MemberDump( TeaTimersImportExport.StringToTable( '{a,b,c}' ) )
    members
      1 a
      2 b
      3 c

/script MemberDump( TeaTimersImportExport.StringToTable( '{Aura=Frost Fever,Unit=target,Clr={g=0.4471,r=0.2784},Typ=HARMFUL}' ) )
    members
      BuffOrDebuff HARMFUL
      BarColor table: 216B04C0
      |  g 0.4471
      |  r 0.2784
      AuraName Frost Fever
      Unit target

/script MemberDump( TeaTimersImportExport.StringToTable( '{"a","b","c"}' ) )
    members
      1 a
      2 b
      3 c

/script MemberDump( TeaTimersImportExport.StringToTable( '{"a,b","b=c","{c={d}}"}' ) )
    members
      1 a,b
      2 b=c
      3 {c={d}}

/script local t = {'\\",\'','}'} local p = TeaTimersImportExport.TableToString(t) print (p) MemberDump( TeaTimersImportExport.StringToTable( p ) )
    {"\\",'","}"}
    members
      1 \",'
      2 }

/script local p = TeaTimersImportExport.TableToString( {} ) print (p) MemberDump( TeaTimersImportExport.StringToTable( p ) )
    {}
    members

    I don't think this can come up, but might as well be robust
/script local p = TeaTimersImportExport.TableToString( {{{}}} ) print (p) MemberDump( TeaTimersImportExport.StringToTable( p ) )
    {{{}}}
    members
      1 table: 216A2428
      |  1 table: 216A0510

    I don't think this can come up, but might as well be robust
/script local p = TeaTimersImportExport.TableToString( {{{"a"}}} ) print (p) MemberDump( TeaTimersImportExport.StringToTable( p ) )
    {{{a}}}
    members
      1 table: 27D68048
      |  1 table: 27D68098
      |  |  1 a

    User Error                                   1234567890123456789012
/script MemberDump( TeaTimersImportExport.StringToTable( '{"a,b","b=c","{c={d}}",{' ) )
    Unexpected end of string
    nil

    User Error                                   1234567890123456789012
/script MemberDump( TeaTimersImportExport.StringToTable( '{"a,b","b=c""{c={d}}"' ) )
    Illegal quote at 12
    nil
]]--
function TeaTimersImportExport.StringToTable(text, ofs)
    local cur = ofs or 1

    if text:byte(cur+1) == 125 then
        return {},cur+1
    end

    local i = 0
    local ret = {}
    while text:byte(cur) ~= 125 do
        if not text:byte(cur) then
            print("Unexpected end of string")
            return nil,nil
        end
        i = i + 1
        cur = cur + 1 -- advance past the { or ,
        local hasKey, eq, delim
        -- If it's not a quote or a {, it should be a key+equals or value+delimeter
        if text:byte(cur) ~= 34 and text:byte(cur) ~= 123 then 
            eq = text:find("=", cur)
            local comma = text:find(",", cur) 
            delim = text:find("}", cur) or comma
            if comma and delim > comma then
                delim = comma 
            end

            if not delim then 
                print("Unexpected end of string")
                return nil, nil
            end
            hasKey = (eq and eq < delim)
        end

        local k,v
        if not hasKey then
            k = i
        else
            k = text:sub(cur,eq-1)
            k = TEATIMERS.LENGTHENINGS[k] or k
            if not k or k == "" then
                print("Error parsing key at", cur)
                return nil,nil
            end
            cur = eq+1
        end

        if not text:byte(cur) then 
            print("Unexpected end of string")
            return nil,nil
        elseif text:byte(cur) == 123 then -- '{'
            v, cur = TeaTimersImportExport.StringToTable(text, cur)
            if not v then return nil,nil end
            cur = cur+1
        else
            if text:byte(cur) == 34 then -- '"'
                -- find the closing quote
                local endq = cur
                delim=nil
                while not delim do
                    endq = text:find('"', endq+1)
                    if not endq then
                        print("Could not find closing quote begun at", cur)
                        return nil, nil
                    end
                    if text:byte(endq-1) ~= 92 then -- \
                        delim = endq+1
                        if text:byte(delim) ~= 125 and text:byte(delim) ~= 44 then
                            print("Illegal quote at", endq)
                            return nil, nil
                        end
                    end
                end
                v = text:sub(cur+1,delim-2)
                v = gsub(v, '\\"', '"')
            else
                v = text:sub(cur,delim-1)
                local n = tonumber(v)
                if tostring(n) == v  then
                    v = n
                elseif v == "true" then
                    v = true
                elseif v == "false" then
                    v = false
                end
            end
            if v==nil or v == "" then
                print("Error parsing value at",cur)
            end
            cur = delim
        end

        ret[k] = v
    end

    return ret,cur
end

function TeaTimersImportExport.ImportBarSettingsFromString(text, bars, barID)
    local pruned
    if text and text ~= "" then
        local ver, packed = text:match("bv(%d+):(.*)")
        if not ver then
            print("Could not find bar settings header")
        elseif not packed then
            print("Could not find bar settings")
        end
        pruned = TeaTimersImportExport.StringToTable(packed)
    else
        pruned = {}
    end

    if pruned then
        TeaTimers.AddDefaultsToTable(pruned, TEATIMERS.BAR_DEFAULTS)
        bars[barID] = pruned
    end
end

function TeaTimersMenuBar.BarMenu_ShowNameDialog(self, a1, a2, checked)
    if not self.value.text or not TEATIMERS[self.value.text] then return end

    StaticPopupDialogs["TEATIMERS.CHOOSENAME_DIALOG"].text = TEATIMERS[self.value.text];
    local dialog = StaticPopup_Show("TEATIMERS.CHOOSENAME_DIALOG");
    dialog.variable = self.value.variable;

    local edit = _G[dialog:GetName().."EditBox"];
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local barSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID];

    local numeric = self.value.numeric or false;
    -- TODO: There has to be a better way to do this, this has pretty bad user  feel
    if ( nil == TeaTimersMenuBar.EditBox_Original_OnTextChanged ) then
        TeaTimersMenuBar.EditBox_Original_OnTextChanged = edit:GetScript("OnTextChanged");
    end
    if ( numeric ) then
        edit:SetScript("OnTextChanged", TeaTimersMenuBar.EditBox_Numeric_OnTextChanged);
    else
        edit:SetScript("OnTextChanged", TeaTimersMenuBar.EditBox_Original_OnTextChanged);
    end
    
    edit:SetFocus();
    if ( dialog.variable ~= "ImportExport" ) then
        edit:SetText( barSettings[dialog.variable] );
    else
        edit:SetText( TeaTimersImportExport.ExportBarSettingsToString(barSettings) );
        edit:HighlightText();
    end
end

function TeaTimersMenuBar.BarMenu_ChooseName(text, variable)
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local barSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID];
    if ( variable ~= "ImportExport" ) then
        barSettings[variable] = text;
    else
        TeaTimersImportExport.ImportBarSettingsFromString(text, TeaTimers.ProfileSettings.Groups[groupID]["Bars"], barID);
    end

    TeaTimers.Bar_Update(groupID, barID);
end

function MemberDump(v, bIndex, filter, indent, recurse)
    if v == nil then 
        print("nil")
        return
    elseif type(v) == "table" then
		if not indent then 
			indent = " " 
			print("members")
		end
		for index, value in pairs(v) do
			if (not filter) or (type(index) == "string" and index:find(filter)) then
				print(indent, index, value);
				if (recurse and type(value) == "table") then 
				    MemberDump(value, nil, nil, indent.." | ",true) 
				end
			end
		end
    else
        if not indent then indent = "" end
        print(indent,v)
    end
	
	if type(v) == "table" or not recurse then
		local mt = getmetatable(v)
		if ( mt ) then
			print("metatable")
			for index, value in pairs(mt) do
				if (not filter) or (type(index) == "string" and index:find(filter)) then
					print(indent, index, value);
				end
			end
			if ( mt.__index and bIndex) then
				print("__index")
				for index, value in pairs(mt.__index) do
					if (not filter) or (type(index) == "string" and index:find(filter)) then
						print(indent, index, value);
					end
				end
			end
		end
    end
end

function TeaTimersMenuBar.BarMenu_SetColor()
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local varSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID][ColorPickerFrame.extraInfo];

    varSettings.r,varSettings.g,varSettings.b = ColorPickerFrame:GetColorRGB();
    TeaTimers.Bar_Update(groupID, barID);
end

function TeaTimersMenuBar.BarMenu_SetOpacity()
    local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
    local barID = TeaTimersMenuBar.CurrentBar["barID"];
    local varSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID][ColorPickerFrame.extraInfo];

    varSettings.a = 1 - OpacitySliderFrame:GetValue();
    TeaTimers.Bar_Update(groupID, barID);
end

function TeaTimersMenuBar.BarMenu_CancelColor(previousValues)
    if ( previousValues.r ) then
        local groupID = TeaTimersMenuBar.CurrentBar["groupID"];
        local barID = TeaTimersMenuBar.CurrentBar["barID"];
        local varSettings = TeaTimers.ProfileSettings.Groups[groupID]["Bars"][barID][ColorPickerFrame.extraInfo];

        varSettings.r = previousValues.r;
        varSettings.g = previousValues.g;
        varSettings.b = previousValues.b;
        varSettings.a = 1 - previousValues.opacity;
        TeaTimers.Bar_Update(groupID, barID);
    end
end


-- -------------
-- RESIZE BUTTON
-- -------------

function TeaTimers.Resizebutton_OnEnter(self)
    local tooltip = _G["GameTooltip"];
    GameTooltip_SetDefaultAnchor(tooltip, self);
    tooltip:AddLine(TEATIMERS.RESIZE_TOOLTIP, NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b, 1);
    tooltip:Show();
end

function TeaTimers.StartSizing(self, button)
    local group = self:GetParent();
    local groupID = self:GetParent():GetID();
    group.oldScale = group:GetScale();
    group.oldX = group:GetLeft();
    group.oldY = group:GetTop();
    --    group:ClearAllPoints();
    --    group:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", group.oldX, group.oldY);
    self.oldCursorX, self.oldCursorY = GetCursorPosition(UIParent);
    self.oldWidth = _G[group:GetName().."Bar1"]:GetWidth();
    self:SetScript("OnUpdate", TeaTimers.Sizing_OnUpdate);
end

function TeaTimers.Sizing_OnUpdate(self)
    local uiScale = UIParent:GetScale();
    local cursorX, cursorY = GetCursorPosition(UIParent);
    local group = self:GetParent();
    local groupID = self:GetParent():GetID();

    -- calculate & set new scale
    local newYScale = group.oldScale * (cursorY/uiScale - group.oldY*group.oldScale) / (self.oldCursorY/uiScale - group.oldY*group.oldScale) ;
    local newScale = max(0.25, newYScale);
    
    -- clamp the scale so the group is a whole number of pixels tall
    local bar1 = _G[group:GetName().."Bar1"]
    local barHeight = bar1:GetHeight()
    local newHeight = newScale * barHeight
    newHeight = math.floor(newHeight + 0.0002)
    newScale = newHeight / barHeight
    group:SetScale(newScale);

    -- set new frame coords to keep same on-screen position
    local newX = group.oldX * group.oldScale / newScale;
    local newY = group.oldY * group.oldScale / newScale;
    group:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", newX, newY);

    -- calculate & set new bar width
    local newWidth = max(50, ((cursorX - self.oldCursorX)/uiScale + self.oldWidth * group.oldScale)/newScale);
    TeaTimers.SetWidth(groupID, newWidth);
    
end

function TeaTimers.SetWidth(groupID, width)
    for barID = 1, TeaTimers.ProfileSettings.Groups[groupID]["NumberBars"] do
        local bar = _G["TeaTimers_Group"..groupID.."Bar"..barID];
        local background = _G[bar:GetName().."Background"];
        local text = _G[bar:GetName().."Text"];
        bar:SetWidth(width);
        text:SetWidth(width-60);
        TeaTimers.SizeBackground(bar, bar.settings.show_icon);
    end
    TeaTimers.ProfileSettings.Groups[groupID]["Width"] = width;        -- move this to StopSizing?
end

function TeaTimers.StopSizing(self, button)
    self:SetScript("OnUpdate", nil)
    local groupID = self:GetParent():GetID();
    TeaTimers.ProfileSettings.Groups[groupID]["Scale"] = self:GetParent():GetScale();
    TeaTimers.SavePosition(self:GetParent(), groupID);
end

function TeaTimers.SavePosition(group, groupID)
    groupID = groupID or group:GetID();
    local point, _, relativePoint, xOfs, yOfs = group:GetPoint();
    TeaTimers.ProfileSettings.Groups[groupID]["Position"] = {point, relativePoint, xOfs, yOfs};
end
