--[[---------------------------------------
	VARIABLE CACHING
-----------------------------------------]]

-- Lua Functions
local pairs, print, select, tonumber, tostring, type, unpack = pairs, print, select, tonumber, tostring, type, unpack
local find, format, lower = string.find, string.format, string.lower
-- WoW API Functions / Variables
local GetBindingKey = GetBindingKey
local GetCVar,SetCVar = GetCVar,SetCVar
local IsShiftKeyDown = IsShiftKeyDown
local PlaySoundFile = PlaySoundFile

local SCA, Ace3, LSM = unpack(select(2,...))

local function GetAudioBinding(command)
	local binding1,binding2 = GetBindingKey(command)
	local bindingMsg = ''
	
	if (binding1 and binding2) then
		bindingMsg = binding1.." or "..binding2
	elseif (binding1) then
		bindingMsg = binding1
	else
		bindingMsg = binding2
	end
	
	return bindingMsg
end

function Ace3:Spacer(order,width,db,expectedSource)
	local spacing = {
		order = order,
		type = "description",
		name = "",
		hidden = function()
			if (db) then
				if (self:GetSelected(db) ~= "empty" and db[self:GetSelected(db)].options.source == expectedSource) then
					return true
				else
					return false
				end
			else
				return false
			end
		end,
		width = width,
	}
	
	return spacing
end

--[[-------------------------------------------
	
	OPTION TOGGLE TOOLS
	
---------------------------------------------]]
function Ace3:TutorialToggle(db,order,width)
	local tutorial = {
		order = order,
		type = "toggle",
		--name = "|cFF94FF5FTutorial|r",
		name = "Tutorial",
		desc = "Displays tutorial panels to help you assign sounds to specific spells.\n\n|cFF00FF00Right-clicking a panel will hide it|r",
		get = function()
			return db.tutorial.enabled
		end,
		disabled = true,
		set = function(self,value)
			db.tutorial.enabled = value
			Ace3:UpdateTutorial(db)
			
			if (not value) then
				for k,_ in pairs(db.tutorial.flags) do
					db.tutorial.flags[k] = false
				end
			end
		end,
		width = width,
	}
	
	return tutorial
end

function Ace3:EventToggle(db,event,order,errorName,width,title,desc,tab,isSpecific,noTutorial)
	local event = {
		order = order,
		type = "toggle",
		name = title,
		desc = desc,
		disabled = function()
			if (isSpecific) then
				if (db.validSpell) then
					if (self:CheckSpell(event,db.validSpell)) then
						return false
					else
						return true
					end
				else
					return true
				end
			else
				if (not db.playSong) then
					return false
				else
					return true
				end
			end
		end,
		get = function() 
			_G["SCA_EVENT_TEST"] = db
			return db.events[event]
		end,
		set = function(this,value)
			db.events[event] = value

			if (SCA.temp.errors[errorName]) then
				SCA.temp.errors[errorName] = nil
			end

			if (isSpecific) then
				if (value and self:CheckSpell(event,db.validSpell)) then
					db.numEvents = db.numEvents + 1
				else
					db.numEvents = db.numEvents - 1
				end
			else
				if (value) then
					db.numEvents = db.numEvents + 1
				else
					db.numEvents = db.numEvents - 1
				end
			end
			
			if (db.isSeparate and db.numEvents <= 1) then
				db.isSeparate = false
			end

			if (tab) then
				self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
			else
				self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
			end
			
			if (not noTutorial) then
				db.tutorial.flags.spellEvents = true
				self:UpdateTutorial(db,'toggle')
			end
			
			if (self:GetSelected(db) == "empty") then
				self:NavigateInterfaceOptions(SCA.temp.selectedMenuItem,true)
			end
		end,
		width = width,
	}
	
	return event
end

--[[-------------------------------------------
	
	REPORTING TOOLS
	
---------------------------------------------]]
function Ace3:DisabledError(db,order,event)
	local disabledError = {
		order = order,
		type = "description",
		hidden = function()
			if (not event) then
				if (db.enabled) then
					return true
				else
					return false
				end
			else
				if (db.events[Ace3:GetSelected(db)]) then
					return true
				else
					return false
				end
			end
		end,
		name = "|cFFFF8585CURRENTLY DISABLED IN QUICK ACCESS|r",
		width = "full",
	}
	
	return disabledError
end

function Ace3:SoundList(db,order,isSpecific)
	local soundList = {
		name = "Sound List",
		order = order,
		type = "group",
		dialogInline = true,
		hidden = function()
			if (isSpecific) then
				if (SCA.temp.addSound and db.isSeparate) then
					return false
				else
					return true
				end
			else
				if (db.isSeparate) then
					return false
				else
					return true
				end
			end
		end,
		args = {
			messages = {
				order = 1,
				type = "description",
				name = " ",
			},
		},
	}
	
	return soundList
end

function Ace3:Errors(errorName,order,isSpecific)
	local errors = {
		name = "Errors",
		order = order,
		type = "group",
		dialogInline = true,
		hidden = function()
			if (isSpecific) then
				if (SCA.temp.addSound and SCA.temp.errors[errorName]) then
					return false
				else
					return true
				end
			else
				if (SCA.temp.errors[errorName]) then
					return false
				else
					return true
				end
			end
		end,
		args = {
			messages = {
				order = 1,
				type = "description",
				name = " ",
			},
		},
	}
	
	return errors
end

--[[-------------------------------------------
	
	SOUND CONFIG TOOLS
	
---------------------------------------------]]
function Ace3:Source(db,order,errorName,values,isSpecific,noTutorial)
	local source = {
		order = order,
		type = "select",
		name = "Audio Source",
		desc = "Choose which sound pack you want to play for crit sounds",
		disabled = function()
			if (db.playSong) then
				return true
			else
				return false
			end
		end,
		get = function()
			local val

			if (self:GetSelected(db) == "empty") then
				val = "packs"
			else
				val = db[db.keys[db.selected]].options.source
			end
			
			return val
		end,
		set = function(_,value)
			if (SCA.temp.errors[errorName]) then
				SCA.temp.errors[errorName] = nil
			end
			
			db[self:GetSelected(db)].options.source = value
			
			if (not noTutorial) then
				db.tutorial.flags.source = true
				self:UpdateTutorial(db)
			end
		end,
		values = values,
	}
	
	return source
end

function Ace3:Channel(db,order,values,isSpecific)
	local channel = {
		order = order,
		type = "select",
		name = "Sound Channel",
		desc = "Choose which channel for the audio to play",
		disabled = function()
			if (db.playSong) then
				return true
			else
				return false
			end
		end,
		get = function()
			local val

			if (self:GetSelected(db) == "empty") then
				val = "Master"
			else
				val = db[self:GetSelected(db)].options.channel
			end
			
			return val
		end,
		set = function(_,value)
			db[self:GetSelected(db)].options.channel = value
		end,
		values = values,
	}
	
	return channel
end

function Ace3:Volume(db,order)
	local volume = {
		order = order,
		type = "range",
		name = "Volume",
		desc = "Adjust the volume",
		min = 0,
		max = 1,
		step = 0.05,
		bigStep = 0.05,
		get = function()
			local val

			if (self:GetSelected(db) == "all") then
				val = "Master"
			else
				val = db[self:GetSelected(db)].options.channel
			end
			
			return tonumber(format("%.2f",GetCVar("Sound_"..val.."Volume")))
		end,
		set = function(_,value)
			SetCVar("Sound_"..db[self:GetSelected(db)].options.channel.."Volume",value)
		end,
	}
	
	return volume
end

function Ace3:PackSelect(db,order,tab,isSpecific)
	local packSelect = {
		order = order,
		type = "select",
		name = "Sound Pack",
		desc = "Choose which sound pack you want to play for crit sounds",
		descStyle = "inline",
		hidden = function()
			if (self:GetSelected(db) ~= "empty") then
				if (db[self:GetSelected(db)].options.source ~= "packs") then
					return true
				else
					return false
				end
			else
				return false
			end
		end,
		disabled = function()
			if (#SCA.library == 0) then
				return true
			else
				if (isSpecific) then
					if (not db.validSpell or db.numEvents == 0) then
						return true
					else
						return false
					end
				elseif (db.numEvents == 0) then
					return true
				else
					return false
				end
			end
		end,
		get = function()
			if (db[self:GetSelected(db)].options.source == "packs" and #SCA.library == 0) then
				db[self:GetSelected(db)].packs.value = "empty"
			elseif (#SCA.library > 0 and db[self:GetSelected(db)].packs.path ~= '') then
				for i=1,#SCA.library do
					if (db[self:GetSelected(db)].packs.path == SCA.library[i].path) then
						db[self:GetSelected(db)].packs.value = SCA.library[i].value
					end
				end
			end
			
			return db[self:GetSelected(db)].packs.value
		end,
		set = function(this,value)
			for i=1,#SCA.library do
				if (value == SCA.library[i].value) then
					local library = SCA.library[i]

					if (value ~= "empty") then
						db[self:GetSelected(db)].validAudio = true
						db[self:GetSelected(db)].packs.value = library.value
						db[self:GetSelected(db)].packs.text = library.text
						db[self:GetSelected(db)].packs.path = library.path
						db[self:GetSelected(db)].packs.max = library.max
					else
						db[self:GetSelected(db)].validAudio = false
						db[self:GetSelected(db)].packs.value = value
						db[self:GetSelected(db)].packs.text = ''
						db[self:GetSelected(db)].packs.path = ''
						db[self:GetSelected(db)].packs.max = 1
					end
						
					break
				end
			end

			if (tab) then
				self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
			else
				self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
			end
		end,
		values = function()
			local packs = {}

			if (#SCA.library > 0) then
				for i=1,#SCA.library do
					packs[SCA.library[i].value] = SCA.library[i].text
				end
				
				packs["empty"] = "Select a pack"
			else
				packs["empty"] = "No packs installed"
			end
			
			return packs
		end,
	}
	
	return packSelect
end

function Ace3:SharedMedia(db,order,tab,isSpecific)
	local sharedMedia = {
		order = order,
		type = "select",
		dialogControl = "LSM30_Sound",
		name = "Select Audio",
		desc = "Select an audio file from the SharedMedia library",
		hidden = function()
			if (db[self:GetSelected(db)].options.source ~= "shared") then
				return true
			else
				return false
			end
		end,
		disabled = function()
			if (isSpecific) then
				if (not db.validSpell or db.numEvents == 0) then
					return true
				else
					return false
				end
			elseif (db.numEvents == 0) then
				return true
			else
				return false
			end
		end,
		get = function()
			if (#db[self:GetSelected(db)].shared.value > 0 and db[self:GetSelected(db)].shared.value ~= "None") then
				db[self:GetSelected(db)].validAudio = true
			else
				db[self:GetSelected(db)].validAudio = false
			end

			return db[self:GetSelected(db)].shared.value
		end,
		set = function(this,value)
			if (value ~= "None") then
				db[self:GetSelected(db)].validAudio = true
			else
				db[self:GetSelected(db)].validAudio = false
			end
			
			db[self:GetSelected(db)].shared.value = value
				
			if (tab) then
				self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
			else
				self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
			end
		end,
		values = LSM:HashTable("sound"),
	}
	
	return sharedMedia
end

function Ace3:AudioSource(db,order,values,tab,isSpecific,noTutorial)
	local audioSource = {
		order = order,
		type = "select",
		name = "Audio Type",
		desc = "Choose whether your custom audio is a file or sound pack",
		hidden = function()
			if (db[self:GetSelected(db)].options.source ~= "custom") then
				return true
			else
				return false
			end
		end,
		disabled = function()
			if (isSpecific) then
				if (not db.validSpell or db.numEvents == 0) then
					return true
				else
					return false
				end
			elseif (db.numEvents == 0) then
				return true
			else
				return false
			end
		end,
		get = function(this)
			if (db[self:GetSelected(db)].custom.customType == "file") then
				if (tab) then
					this.options.args[tab].args.SoundConfig.args.audioLocation.width = "full"
				else
					this.options.args.SoundConfig.args.audioLocation.width = "full"
				end
			else
				if (tab) then
					this.options.args[tab].args.SoundConfig.args.audioLocation.width = "double"
				else
					this.options.args.SoundConfig.args.audioLocation.width = "double"
				end
			end
			
			return db[self:GetSelected(db)].custom.customType
		end,
		set = function(this,value)
			db[self:GetSelected(db)].custom.customType = value
			
			if (not noTutorial) then
				db.tutorial.flags.customType = true
				self:UpdateTutorial(db)
			end
				
			if (value == "file") then
				if (tab) then
					this.options.args[tab].args.SoundConfig.args.audioLocation.width = "full"
				else
					this.options.args.SoundConfig.args.audioLocation.width = "full"
				end
			else
				if (tab) then
					this.options.args[tab].args.SoundConfig.args.audioLocation.width = "double"
				else
					this.options.args.SoundConfig.args.audioLocation.width = "double"
				end
			end
		end,
		values = values,
	}
	
	return audioSource
end

function Ace3:ToggleSeparate(db,order,tab,isSpecific,noTutorial,bypass)
	local toggleSeparate = {
		order = order,
		type = "toggle",
		name = "Separate Sound Events",
		disabled = function()
			if (db.numEvents > 1 and not db.playSong) then
				return false
			else
				return true
			end
		end,
		get = function() 
			return db.isSeparate
		end,
		set = function(this,value)
			db.isSeparate = value
			
			if (not value) then
				if (db.keys) then
					db.selected = 0
				else
					db.selected = 'all'
				end
			else
				if (db.selected == 0) then
					db.selected = self:SetSelected(db,nil,true)
				end
			end

			if (tab) then
				self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
			else
				self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
			end
			
			if (not noTutorial) then
				db.tutorial.flags.isSeparate = true
				self:UpdateTutorial(db)
			end
		end,
	}
	
	return toggleSeparate
end

function Ace3:IsSCARandom(db,order,isSpecific)
	local isSCARandom = {
		order = order,
		type = "toggle",
		name = "Random Order",
		desc = 'Toggle whether files will be played in random or sequential order.',
		disabled = function()
			if (#SCA.library == 0) then
				return true
			else
				if (isSpecific) then
					if (not db.validSpell or db.numEvents == 0) then
						return true
					else
						return false
					end
				elseif (db.numEvents == 0) then
					return true
				else
					return false
				end
			end
		end,
		hidden = function()
			if (db[self:GetSelected(db)].options.source ~= "packs") then
				return true
			else
				return false
			end
		end,
		get = function()
			return db[self:GetSelected(db)].packs.isRandom.enabled
		end,
		set = function(_,value)
			db[self:GetSelected(db)].packs.isRandom.enabled = value
		end,
		width = "double",
	}
	
	return isSCARandom
end

function Ace3:AudioLocation(db,order,isSpecific,noTutorial)
	local audioLocation = {
		type = "input",
		order = order,				
		name = "Audio Path",
		desc = "Audio file/path location.\n\nFile extensions not necessary.\n\nOnly |cFF00FF00MP3|r and |cFF00FF00OGG|r files are supported!",
		
		hidden = function()
			if (db[self:GetSelected(db)].options.source ~= "custom") then
				return true
			else
				return false
			end
		end,
		disabled = function()
			if (isSpecific) then
				if (not db.validSpell or db.numEvents == 0) then
					return true
				else
					return false
				end
			elseif (db.numEvents == 0) then
				return true
			else
				return false
			end
		end,
		get = function()
			local path
			
			if (db.playSong) then
				path = 'music'
			else
				path = 'path'
			end
			
			return self:VerifyCustomPath(db[self:GetSelected(db)].custom[path],db[self:GetSelected(db)])
		end,
		set = function(_,value)
			local path
			
			if (db.playSong) then
				path = 'music'
			else
				path = 'path'
			end
			
			db[self:GetSelected(db)].custom[path] = self:VerifyCustomPath(value,db[self:GetSelected(db)])
			
			if (not noTutorial) then
				db.tutorial.flags.audioLocationFile = true
				db.tutorial.flags.audioLocationPack = true
				self:UpdateTutorial(db)
			end
		end,
	}
	
	return audioLocation
end

function Ace3:NumFiles(db,order,errorName,tab,isSpecific,noTutorial)
	local numFiles = {
		type = "input",
		order = order,
		name = "Number of files",
		desc = "Indicate the amount of files in the sound pack",
		hidden = function()
			if (db[self:GetSelected(db)].custom.customType == "file" or db[self:GetSelected(db)].options.source ~= "custom") then
				return true
			else
				return false
			end
		end,
		disabled = function()
			if (isSpecific) then
				if (not db.validSpell or db.numEvents == 0) then
					return true
				else
					return false
				end
			elseif (db.numEvents == 0) then
				return true
			else
				return false
			end
		end,
		get = function()
			return tostring(db[self:GetSelected(db)].custom.max)
		end,
		set = function(this,value)
			local errors
			
			if (tab) then
				errors = this.options.args[tab].args.errors.args.messages
			else
				errors = this.options.args.errors.args.messages
			end
			
			if (type(tonumber(value)) == "number") then
				db[self:GetSelected(db)].custom.max = tonumber(value)
				
				errors.name = " "
				SCA.temp.errors[errorName] = nil
			else
				errors.name = "Number of Files: |cFFFF5F99Numeric values only.|r"
				
				SCA.temp.errors[errorName] = true
			end
			
			if (not noTutorial) then
				db.tutorial.flags.numFiles = true
				self:UpdateTutorial(db)
			end
		end,
	}
	
	return numFiles
end

function Ace3:TestSound(db,order,errorName,tab,isSpecific,noTutorial)
	local test = {
		order = order,
		type = "execute",
		name = function()
			if (not db.playSong) then
				return "Test Sound"
			else
				if (SCA.temp.music.isSongPlaying) then
					return "Stop Music"
				else
					return "Test Music"
				end
			end
		end,
		hidden = function()
			if (db[self:GetSelected(db)].options.source ~= "custom") then
				return true
			else
				return false
			end
		end,
		disabled = function()
			if (isSpecific) then
				if (not db.validSpell or db.numEvents == 0) then
					return true
				else
					return false
				end
			elseif (db.numEvents == 0) then
				return true
			else
				return false
			end
		end,
		func = function(this)
			local errors,msg
			
			if (tab) then
				errors = this.options.args[tab].args.errors.args.messages
			else
				errors = this.options.args.errors.args.messages
			end
			
			if (not db.playSong) then
				if (db[self:GetSelected(db)].custom.customType == "pack") then
					if (not PlaySoundFile(db[self:GetSelected(db)].custom.path..tostring(self:GetSoundFileNumber(db[self:GetSelected(db)],"custom"))..".mp3",db[self:GetSelected(db)].options.channel) and not PlaySoundFile(db[self:GetSelected(db)].custom.path..tostring(self:GetSoundFileNumber(db[self:GetSelected(db)],"custom"))..".ogg",db[self:GetSelected(db)].options.channel)) then
						msg = "Custom Audio: |cFFFF5F99"
						
						if (GetCVar("Sound_EnableAllSound") == "0") then
							msg = msg.."All sound is muted! Enable all sound in the system options|r"
						elseif (db[self:GetSelected(db)].options.channel == "SFX" and GetCVar("Sound_EnableSFX") == "0") then
							local bindingMsg = GetAudioBinding("TOGGLESOUND")
							
							msg = msg.."Sound Effects is muted! Unmute Sound Effects ("..bindingMsg..") or choose a different audio channel|r"
						elseif (db[self:GetSelected(db)].options.channel == "Music" and GetCVar("Sound_EnableMusic") == "0") then
							local bindingMsg = GetAudioBinding("TOGGLEMUSIC")
							
							msg = msg.."Music is muted! Unmute Music ("..bindingMsg..") or choose a different audio channel|r"
						else
							msg = msg.."Audio file not found!.|r"
						end
						
						errors.name = msg
						
						SCA.temp.errors[errorName] = true
						
						if (not db[self:GetSelected(db)].custom.isRandom.enabled) then
							db[self:GetSelected(db)].custom.isRandom.currentFile = db[self:GetSelected(db)].custom.isRandom.currentFile - 2
						end
						
						db[self:GetSelected(db)].validAudio = false
					else
						errors.name = " "
						SCA.temp.errors[errorName] = nil
						
						if (not db[self:GetSelected(db)].custom.isRandom.enabled) then
							db[self:GetSelected(db)].custom.isRandom.currentFile = db[self:GetSelected(db)].custom.isRandom.currentFile - 1
						end

						db[self:GetSelected(db)].validAudio = true
					end
				else
					if (not PlaySoundFile(db[self:GetSelected(db)].custom.path..".mp3",db[self:GetSelected(db)].options.channel) and not PlaySoundFile(db[self:GetSelected(db)].custom.path..".ogg",db[self:GetSelected(db)].options.channel)) then
						msg = "Custom Audio: |cFFFF5F99"
						
						if (GetCVar("Sound_EnableAllSound") == "0") then
							msg = msg.."All sound is muted! Enable all sound in the system options|r"
						elseif (db[self:GetSelected(db)].options.channel == "SFX" and GetCVar("Sound_EnableSFX") == "0") then
							local bindingMsg = GetAudioBinding("TOGGLESOUND")
							
							msg = msg.."Sound Effects is muted! Unmute Sound Effects ("..bindingMsg..") or choose a different audio channel|r"
						elseif (db[self:GetSelected(db)].options.channel == "Music" and GetCVar("Sound_EnableMusic") == "0") then
							local bindingMsg = GetAudioBinding("TOGGLEMUSIC")
							
							msg = msg.."Music is muted! Unmute Music ("..bindingMsg..") or choose a different audio channel|r"
						else
							msg = msg.."Audio file not found!.|r"
						end
						
						errors.name = msg
						
						SCA.temp.errors[errorName] = true
						
						db[self:GetSelected(db)].validAudio = false
					else
						errors.name = " "
						SCA.temp.errors[errorName] = nil

						db[self:GetSelected(db)].validAudio = true
					end
				end
				
				if (tab) then
					self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
				else
					self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
				end
			else
				self:MusicPlayer(db[self:GetSelected(db)])
			end
			
			if (not noTutorial) then
				self:UpdateTutorial(db)
			end
		end,
	}
	
	return test
end

function Ace3:IsCustomRandom(db,order)
	local isCustomRandom = {
		order = order,
		type = "toggle",
		name = "Random Order",
		desc = 'Toggle whether files will be played in random or sequential order.',
		hidden = function()
			if (db[self:GetSelected(db)].custom.customType == "file" or db[self:GetSelected(db)].options.source ~= "custom") then
				return true
			else
				return false
			end
		end,
		get = function()
			return db[self:GetSelected(db)].custom.isRandom.enabled
		end,
		set = function(_,value)
			db[self:GetSelected(db)].custom.isRandom.enabled = value
		end,
		width = "double",
	}
	
	return isCustomRandom
end

function Ace3:SoftReset(db,order,tab,isSpecific)
	local reset = {
		order = order,
		type = "execute",
		name = "Reset",
		disabled = function()
			if (db[self:GetSelected(db)].validAudio) then
				return false
			else
				return true
			end
		end,
		desc = function()
			return "Resets the sound for: |cFF00FF00"..db[self:GetSelected(db)].label.."|r"
		end,
		func = function(this)
			self:ResetDB(db)

			if (tab) then
				self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
			else
				self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
			end
		end,
		width = "half",
	}
	
	return reset
end

function Ace3:HardReset(db,order,tab,isSpecific)
	local reset = {
		order = order,
		type = "execute",
		name = function()
			if (IsShiftKeyDown()) then
				return "Reset Everything"
			else
				return "Reset All Sounds"
			end
		end,
		hidden = function()
			if (isSpecific) then
				if (SCA.temp.addSound) then
					return false
				else
					return true
				end
			else
				return false
			end
		end,
		desc = function()
			if (IsShiftKeyDown()) then
				return "Resets all settings and sound configurations"
			else
				return "Resets all sound configurations.\n\n|cFF00FF00Holding SHIFT while clicking will reset EVERYTHING in this tab."
			end
		end,
		disabled = function()
			if (IsShiftKeyDown()) then
				return false
			else
				for k,v in pairs(db.events) do
					if (db[k].validAudio) then
						return false
					end
				end
				
				return true
			end
		end,
		func = function(this)
			self:ResetDB(db,IsShiftKeyDown())
			
			if (tab) then
				self:UpdateSeparateSoundMessage(db,this.options.args[tab],isSpecific)
			else
				self:UpdateSeparateSoundMessage(db,this.options,isSpecific)
			end
		end,
	}
	
	return reset
end

--[[-------------------------------------------
	
	DAMAGE & HEALING SETUP TOOLS
	
---------------------------------------------]]
function Ace3:Threshold(db,event,title,errorName,tab)
	local threshold = {
		order = 4,
		type = "input",
		name = title.." Threshold",
		desc = "Restrict the audio queues to a specific "..lower(title).." threshold\n\n|cFFFF5F99Digit group is automatically applied; do not enter commas.|r",
		get = function()
			return BreakUpLargeNumbers(tonumber(db[event].options.threshold))
		end,
		set = function(this,value)
			local errors
			
			if (tab) then
				errors = this.options.args[tab].args.errors.args.messages
			else
				errors = this.options.args.errors.args.messages
			end
			
			if (type(tonumber(value)) == "number") then
				db[event].options.threshold = tonumber(value)
				errors.name = " "
				SCA.temp.errors[errorName] = nil
			else
				local isComma = find(value,",")
				
				if (isComma) then
					errors.name = title.." Threshold: |cFFFF5F99Do not enter commas; they are added automatically.|r"
				else
					errors.name = title.." Threshold: |cFFFF5F99Numeric values only.|r"
				end
				
				SCA.temp.errors[errorName] = true
			end
		end,
	}
	
	return threshold
end

function Ace3:CritSelect(db,event,triggerList)
	local critSelect = {
		order = 5,
		type = "select",
		name = "Crit / Non-Crit",
		desc = "Choose which sound pack you want to play for crit sounds",
		descStyle = "inline",
		get = function()
			return db[event].options.trigger
		end,
		set = function(_,value)
			db[event].options.trigger = value
		end,
		values = triggerList,
	}
	
	return critSelect
end