-- Lua Functions
local pairs, print, select, type, unpack = pairs, print, select, type, unpack
local lower, sub = string.lower, string.sub
local tinsert, wipe  = table.insert, table.wipe

local SCA, Ace3, LSM = unpack(select(2,...))

--[[---------------------------------------
	CONSTANTS
-----------------------------------------]]
local SOUND_SOURCE_OPTIONS = {
	["packs"] = "SCA Audio Packs",
	["shared"] = "SharedMedia",
	["custom"] = "Custom Sounds",
}

local SOUND_CHANNEL_OPTIONS = {
	["Master"] = MASTER,
	["SFX"] = SOUND,
	["Music"] = MUSIC_VOLUME,
	["Ambience"] = AMBIENCE_VOLUME,
	["Dialog"] = DIALOG_VOLUME,
}

local SOUND_CUSTOM_TYPES = {
	["file"] = "Single Audio File",
	["pack"] = "Custom Audio Pack",
}

local SOUND_CRIT_TRIGGERS = {
	['crit'] = "Crits Only",
	['nocrit'] = "Non-crits Only",
	['always'] = "Always On",
}

local function GetChatTabInfo(self,db,isCustom)
	local tabInfo = {
		type = "group",
		order = db.order,
		name = db.info.label,
		desc = '',
		args = {
			spacer = {
				order = 0,
				hidden = function()
					if (isCustom and db.isNewChannelAdded) then
						return false
					else
						return true
					end	
				end,
				type = "description",
				name = '|cFFFF2010A new custom channel has been added, you must reload the UI for it to appear here.|r',
			},
			disabledError = self:DisabledError(db,0),
			options = {
				name = "Toggle Options",
				order = 1,
				type = "group",
				guiInline = true,
				args = self:BuildToggleCheckboxes(db,nil,true),
			},
			soundList = self:SoundList(db,2),
			errors = self:Errors(db.errorName,3),
			SoundConfig = {
				name = "Audio Config",
				order = 4,
				type = "group",
				guiInline = true,
				disabled = function()
					if (db.numEvents == 0) then
						return true
					else
						return false
					end
				end,
				args = {
					source = self:Source(db,1,db.errorName,SOUND_SOURCE_OPTIONS,nil,true),
					channel = self:Channel(db,2,SOUND_CHANNEL_OPTIONS),
					volume = self:Volume(db,3),
					--[[-----------------------------
						SCA Sound Packs
					-------------------------------]]
					packSelect = self:PackSelect(db,4,db.info.tabName),
					isSCARandom = self:IsSCARandom(db,7),
					--[[-----------------------------
						SharedMedia
					-------------------------------]]
					sharedMedia = self:SharedMedia(db,4,db.info.tabName),
					--[[-----------------------------
						Custom Audio
					-------------------------------]]
					audioSource = self:AudioSource(db,4,SOUND_CUSTOM_TYPES,db.info.tabName,nil,true),
					event = {
						order = 5,
						type = "select",
						name = db.info.selectTitle,
						desc = "Choose which chat channel you want to edit",
						disabled = function()
							if (db.isSeparate) then
								return false
							else
								return true
							end
						end,
						get = function()
							return db.selected
						end,
						set = function(_,value)
							db.selected = value
						end,
						values = function()
							local args,sortTable = {},{}

							sortTable = self:GetRefreshedTable(db)
							
							for i=1,#sortTable do
								if (sortTable[i] ~= "empty") then
									args[i] = db[sortTable[i]].label
								end
							end

							if (db.numEvents == 0) then
								args[0] = "No chat channels"
							elseif (self:GetSelected(db) == "all") then
								args[0] = 'Select a chat channel'
							end
								
							wipe(sortTable)

							return args
						end,
					},
					toggleSeparate = self:ToggleSeparate(db,6,db.info.tabName,nil,true),
					audioLocation = self:AudioLocation(db,7,nil,true),
					numFiles = self:NumFiles(db,8,db.errorName,db.info.tabName,nil,true),
					test = self:TestSound(db,9,db.errorName,db.info.tabName,nil,true),
					isCustomRandom = self:IsCustomRandom(db,10),
					reset = self:SoftReset(db,11,db.info.tabName),
				},
			},
			ChatConfig = {
				name = "Chat Config",
				order = 5,
				type = "group",
				guiInline = true,
				args = {
					toggle = {
						order = 1,
						type = "toggle",
						name = "Timeout",
						desc = "Toggles a timeout that will delay sounds from playing based on the provided delay amount",
						get = function()
							return db[self:GetSelected(db)].options.timeout.enabled
						end,
						set = function(_,value)
							db[self:GetSelected(db)].options.timeout.enabled = value
						end,
					},
					delay = {
						order = 2,
						type = "input",
						name = "Delay",
						desc = "Set a delay value, in seconds.",
						disabled = function()
							if (db[self:GetSelected(db)].options.timeout.enabled) then
								return false
							else
								return true
							end
						end,
						get = function()
							return db[self:GetSelected(db)].options.timeout.delay
						end,
						set = function(_,value)
							db[self:GetSelected(db)].options.timeout.delay = value
						end,
						width = "half",
					},
					ignoreMe = {
						order = 3,
						type = "toggle",
						name = "Ignore Me",
						desc = "Do not play a sound if that chat message was sent by you",
						get = function()
							return db[self:GetSelected(db)].options.ignoreMe
						end,
						set = function(_,value)
							db[self:GetSelected(db)].options.ignoreMe = value
						end,
						width = "half",
					},
				},
			},
			reset = self:HardReset(db,6,db.info.tabName),
		},
	}
	
	return tabInfo
end

function Ace3:ChatTabBuilder()
	local tabs = {}
	--local db = self.db.profile
	--local db = SCA.profile

	local chat = SCA.profile.audio.chat

	tabs["disabledError"] = self:DisabledError(chat,0)
	for k,_ in pairs(chat) do
		if (type(chat[k]) == "table") then
			tabs[chat[k].info.tabName] = GetChatTabInfo(self,chat[k])
		end
	end
	
	chat = SCA.char.chat
	
	--for k,_ in pairs(chat.events) do
		--if (type(chat[k]) == "table") then
			tabs[chat.info.tabName] = GetChatTabInfo(self,chat,true)
		--end
	--end

	return tabs
end

function Ace3:CombatTabBuilder()
	local tabs = {}
	--local db = self.db.profile
	local db = SCA.profile

	local combat = db.audio.combat

	tabs["disabledError"] = self:DisabledError(combat,0)
	for k,_ in pairs(combat) do
		if (type(combat[k]) == "table") then
			tabs[combat[k].info.tabName] = {
				type = "group",
				order = combat[k].order,
				name = combat[k].info.label,
				desc = '',
				args = {
					disabledError = self:DisabledError(combat[k],0),
					options = {
						name = "Toggle Options",
						order = 1,
						type = "group",
						guiInline = true,
						args = self:BuildToggleCheckboxes(combat[k],nil,true),
					},
					soundList = self:SoundList(combat[k],2),
					errors = self:Errors(combat[k].info.errorName,3),
					SoundConfig = {
						name = "Audio Config",
						order = 4,
						type = "group",
						guiInline = true,
						disabled = function()
							if (combat[k].numEvents == 0) then
								return true
							else
								return false
							end
						end,
						args = {
							source = self:Source(combat[k],1,combat[k].info.errorName,SOUND_SOURCE_OPTIONS,nil,true),
							channel = self:Channel(combat[k],2,SOUND_CHANNEL_OPTIONS),
							volume = self:Volume(combat[k],3),
							--[[-----------------------------
								SCA Sound Packs
							-------------------------------]]
							packSelect = self:PackSelect(combat[k],4,combat[k].info.tabName),
							isSCARandom = self:IsSCARandom(combat[k],7),
							--[[-----------------------------
								SharedMedia
							-------------------------------]]
							sharedMedia = self:SharedMedia(combat[k],4,combat[k].info.tabName),
							--[[-----------------------------
								Custom Audio
							-------------------------------]]
							audioSource = self:AudioSource(combat[k],4,SOUND_CUSTOM_TYPES,combat[k].info.tabName,nil,true),
							events = {
								order = 5,
								type = "select",
								name = combat[k].info.selectTitle,
								desc = "Choose which event you want to edit",
								disabled = function()
									if (combat[k].isSeparate and not combat[k].playSong) then
										return false
									else
										return true
									end
								end,
								get = function()
									return combat[k].selected
								end,
								set = function(_,value)
									combat[k].selected = value
								end,
								values = function()
									local args,sortTable = {},{}

									sortTable = self:GetRefreshedTable(combat[k])
									
									for i=1,#sortTable do
										if (sortTable[i] ~= "empty") then
											args[i] = combat[k][sortTable[i]].label
										end
									end
									
									if (combat[k].numEvents == 0) then
										args[0] = "No combat events"
									elseif (self:GetSelected(combat[k]) == "all") then
										if ((k == "status" and not combat[k].options.playSong) or k ~= "status") then
											args[0] = 'Select a combat event'
										end
									end
										
									wipe(sortTable)

									return args
								end,
							},
							toggleSeparate = self:ToggleSeparate(combat[k],6,combat[k].info.tabName,nil,true),
							audioLocation = self:AudioLocation(combat[k],7,nil,true),
							numFiles = self:NumFiles(combat[k],8,combat[k].info.errorName,combat[k].info.tabName,nil,true),
							test = self:TestSound(combat[k],9,combat[k].info.errorName,combat[k].info.tabName,nil,true,true),
							isCustomRandom = self:IsCustomRandom(combat[k],10),
							reset = self:SoftReset(combat[k],11,combat[k].info.tabName),
						},
					},
					CombatStartConfig = {
						name = "Combat Start Config",
						order = 5,
						type = "group",
						guiInline = true,
						hidden = function()
							if (combat[k].events.onStart) then
								return false
							else
								return true
							end
						end,
						args = {
							zoneSelect = {
								order = 1,
								type = "select",
								name = "Zone Combat Trigger",
								desc = "Select when you want combat sounds/songs to play.",
								get = function()
									return combat.status.options.zoneTrigger
								end,
								set = function(self,value)
									combat.status.options.zoneTrigger = value
								end,
								values = {
									[0] = "Always",
									[1] = "Never",
									[2] = "Scalable Content",
								},
							},
							dungeonSelect = {
								order = 2,
								type = "select",
								name = "Dungeon Combat Trigger",
								desc = "Select when you want combat sounds/songs to play.",
								get = function()
									return combat.status.options.dungeonTrigger
								end,
								set = function(self,value)
									combat.status.options.dungeonTrigger = value
								end,
								values = {
									[0] = "Always",
									[1] = "Never",
									[2] = "Scalable Content",
								},
							},
							raidSelect = {
								order = 3,
								type = "select",
								name = "Raid Combat Trigger",
								desc = "Select when you want combat sounds/songs to play.",
								get = function()
									return combat.status.options.raidTrigger
								end,
								set = function(self,value)
									combat.status.options.raidTrigger = value
								end,
								values = {
									[0] = "Always",
									[1] = "Never",
									[2] = "Scalable Content",
								},
							},
							playSong = {
								order = 4,
								type = "toggle",
								name = "Combat Song",
								desc = "Toggle whether a song, or a quick sound, will be played when entering combat.\n\nWhen enabled, any sound chosen will loop until combat ends. Playlists will be added in a future update.",
								get = function()
									return combat.status.playSong
								end,
								set = function(self,value)
									combat.status.playSong = value
									
									if (value) then
										-- Backup current "All" data
										combat.status.holder.selected = combat.status.selected
										combat.status.holder.isSeparate = combat.status.isSeparate
										combat.status.holder.selected = combat.status.selected
										combat.status.holder.onStop = combat.status.events.onStop
										combat.status.holder.source = combat.status.all.options.source
										combat.status.holder.channel = combat.status.all.options.channel
										
										combat.status.holder.path = combat.status.all.custom.path
										combat.status.holder.max = combat.status.all.custom.max
										combat.status.holder.customType = combat.status.all.custom.customType
										
										combat.status.holder.isRandom = {}
										combat.status.holder.isRandom.enabled = combat.status.all.custom.isRandom.enabled
										combat.status.holder.isRandom.currentFile = combat.status.all.custom.isRandom.currentFile

										-- Force values that we don't want changed
										combat.status.events.onStop = false
										combat.status.selected = 0
										combat.status.all.options.source = "custom"
										combat.status.all.options.channel = "Music"
									else
										-- Restore current "All" data
										combat.status.isSeparate = combat.status.holder.isSeparate
										combat.status.selected = combat.status.holder.selected
										combat.status.events.onStop = combat.status.holder.onStop
										combat.status.all.options.source = combat.status.holder.source
										combat.status.all.options.channel = combat.status.holder.channel
										
										combat.status.all.custom.path = combat.status.holder.path
										combat.status.all.custom.max = combat.status.holder.max
										combat.status.all.custom.customType = combat.status.holder.customType

										combat.status.all.custom.isRandom.enabled = combat.status.holder.isRandom.enabled
										combat.status.all.custom.isRandom.currentFile = combat.status.holder.isRandom.currentFile

										-- Reset "Holder" table
										combat.status.holder = nil
										combat.status.holder = {}
									end
								end,
							},
						},
					},
					reset = self:HardReset(combat[k],6,combat[k].info.tabName),
				},
			}
		end
	end
	
	return tabs
end

function Ace3:LootTabBuilder()
	local tabs = {}
	--local db = self.db.profile
	local db = SCA.profile

	local looting = db.audio.looting

	tabs["disabledError"] = self:DisabledError(looting,0)
	for k,_ in pairs(looting) do
		if (type(looting[k]) == "table") then
			tabs[looting[k].info.tabName] = {
				type = "group",
				order = looting[k].order,
				name = looting[k].info.label,
				desc = '',
				args = {
					disabledError = self:DisabledError(looting[k],0),
					options = {
						name = "Toggle Options",
						order = 1,
						type = "group",
						guiInline = true,
						args = self:BuildToggleCheckboxes(looting[k],nil,true),
					},
					soundList = self:SoundList(looting[k],2),
					errors = self:Errors(looting[k].info.errorName,3),
					SoundConfig = {
						name = "Audio Config",
						order = 4,
						type = "group",
						guiInline = true,
						disabled = function()
							if (looting[k].numEvents == 0) then
								return true
							else
								return false
							end
						end,
						args = {
							source = self:Source(looting[k],1,looting[k].info.errorName,SOUND_SOURCE_OPTIONS,nil,true),
							channel = self:Channel(looting[k],2,SOUND_CHANNEL_OPTIONS),
							volume = self:Volume(looting[k],3),
							--[[-----------------------------
								SCA Sound Packs
							-------------------------------]]
							packSelect = self:PackSelect(looting[k],4,looting[k].info.tabName),
							isSCARandom = self:IsSCARandom(looting[k],7),
							--[[-----------------------------
								SharedMedia
							-------------------------------]]
							sharedMedia = self:SharedMedia(looting[k],4,looting[k].info.tabName),
							--[[-----------------------------
								Custom Audio
							-------------------------------]]
							audioSource = self:AudioSource(looting[k],4,SOUND_CUSTOM_TYPES,looting[k].info.tabName,nil,true),
							quality = {
								order = 5,
								type = "select",
								name = looting[k].info.selectTitle,
								desc = '',
								disabled = function()
									if (looting[k].isSeparate and looting[k].numEvents > 1) then
										return false
									else
										return true
									end
								end,
								get = function()
									return looting[k].selected
								end,
								set = function(_,value)
									looting[k].selected = value
								end,
								values = function()
									local args,sortTable = {},{}
									
									sortTable = self:GetRefreshedTable(looting[k])
									
									for i=1,#sortTable do
										if (sortTable[i] ~= "empty") then
											args[i] = looting[k][sortTable[i]].label
										end
									end

									if (looting[k].numEvents == 0) then
										args[0] = "No loot qualities/types"
									elseif (self:GetSelected(looting[k]) == "all") then
										args[0] = 'Select a '..lower(looting[k].info.selectTitle)
									end
									
									wipe(sortTable)
									
									return args
								end,
							},
							toggleSeparate = self:ToggleSeparate(looting[k],6,looting[k].info.tabName,nil,true),
							audioLocation = self:AudioLocation(looting[k],7,nil,true),
							numFiles = self:NumFiles(looting[k],8,looting[k].info.errorName,looting[k].info.tabName,nil,true),
							test = self:TestSound(looting[k],9,looting[k].info.errorName,looting[k].info.tabName,nil,true),
							isCustomRandom = self:IsCustomRandom(looting[k],10),
							reset = self:SoftReset(looting[k],11,looting[k].info.tabName),
						},
					},
					reset = self:HardReset(looting[k],5,looting[k].info.tabName),
				},
			}
		end
	end
	
	return tabs
end

function Ace3:MoveTabBuilder()
	local tabs = {}
	--local db = self.db.profile
	local db = SCA.profile

	local move = db.audio.move

	tabs["disabledError"] = self:DisabledError(move,0)
	for k,_ in pairs(move) do
		if (type(move[k]) == "table") then
			tabs[move[k].info.tabName] = {
				type = "group",
				order = move[k].order,
				name = move[k].info.label,
				desc = function()
					if (k == "forward") then
						return "|cFFFF0000Does not support moving forward with the mouse|r"
					end
				end,
				args = {
					disabledError = self:DisabledError(move[k],0),
					options = {
						name = "General Options",
						order = 1,
						type = "group",
						guiInline = true,
						args = self:BuildToggleCheckboxes(move[k],nil,true),
					},
					triggers = {
						name = "Sound Triggers",
						order = 2,
						hidden = function()
							if (k == "pitchup" or k == "pitchdown" or k == "autorun") then
								return true
							else
								return false
							end
						end,
						type = "group",
						guiInline = true,
						args = {
							running = {
								order = 1,
								type = "toggle",
								name = "On Foot",
								desc = "Toggle whether or not a sound will play while on foot",
								get = function()
									return move[k].triggers.running
								end,
								set = function(_,value)
									move[k].triggers.running = value
								end,
								width = "half",
							},
							mounted = {
								order = 2,
								type = "toggle",
								name = "Mounted",
								desc = "Toggle whether or not a sound will play while mounted",
								get = function()
									return move[k].triggers.mounted
								end,
								set = function(_,value)
									move[k].triggers.mounted = value
								end,
								width = "half",
							},
							flying = {
								order = 3,
								type = "toggle",
								name = "Flying",
								disabled = function()
									if (move[k].triggers.mounted) then
										return false
									else
										return true
									end
								end,
								desc = "Toggle whether or not a sound will play while flying",
								get = function()
									return move[k].triggers.flying
								end,
								set = function(_,value)
									move[k].triggers.flying = value
								end,
								width = "half",
							},
							swimming = {
								order = 4,
								type = "toggle",
								name = "Swimming",
								desc = "Toggle whether or not a sound will play while swimming",
								get = function()
									return move[k].triggers.swimming
								end,
								set = function(_,value)
									move[k].triggers.swimming = value
								end,
								width = "normal",
							},
							
						},
					},
					soundList = self:SoundList(move[k],3),
					errors = self:Errors(move[k].info.errorName,4),
					SoundConfig = {
						name = "Audio Config",
						order = 5,
						type = "group",
						guiInline = true,
						disabled = function()
							if (move[k].numEvents == 0) then
								return true
							else
								return false
							end
						end,
						args = {
							source = self:Source(move[k],1,move[k].info.errorName,SOUND_SOURCE_OPTIONS,nil,true),
							channel = self:Channel(move[k],2,SOUND_CHANNEL_OPTIONS),
							volume = self:Volume(move[k],3),
							--[[-----------------------------
								--SCA Sound Packs
							-------------------------------]]
							packSelect = self:PackSelect(move[k],4,move[k].info.tabName),
							isSCARandom = self:IsSCARandom(move[k],7),
							--[[-----------------------------
								--SharedMedia
							-------------------------------]]
							sharedMedia = self:SharedMedia(move[k],4,move[k].info.tabName),
							--[[-----------------------------
								--Custom Audio
							-------------------------------]]
							audioSource = self:AudioSource(move[k],4,SOUND_CUSTOM_TYPES,move[k].info.tabName,nil,true),
							actions = {
								order = 5,
								type = "select",
								name = move[k].info.selectTitle,
								desc = "Choose whether your custom audio is a file or sound pack",
								disabled = function()
									if (move[k].isSeparate and move[k].numEvents > 1) then
										return false
									else
										return true
									end
								end,
								get = function()
									return move[k].selected
								end,
								set = function(_,value)
									move[k].selected = value
								end,
								values = function()
									local args,sortTable = {},{}
									
									sortTable = self:GetRefreshedTable(move[k])
									
									for i=1,#sortTable do
										if (sortTable[i] ~= "empty") then
											args[i] = move[k][sortTable[i]].label
										end
									end
									
									if (move[k].numEvents == 0) then
										args[0] = "No actions"
									elseif (self:GetSelected(move[k]) == "all") then
										args[0] = "Select a move action"
									end
									
									return args
								end
							},
							toggleSeparate = self:ToggleSeparate(move[k],6,move[k].info.tabName,nil,true),
							audioLocation = self:AudioLocation(move[k],7,nil,true),
							numFiles = self:NumFiles(move[k],8,move[k].info.errorName,move[k].info.tabName,nil,true),
							test = self:TestSound(move[k],9,move[k].info.errorName,move[k].info.tabName,nil,true),
							isCustomRandom = self:IsCustomRandom(move[k],10),
							reset = self:SoftReset(move[k],11,move[k].info.tabName),
						},
					},
					reset = self:HardReset(move[k],6,move[k].info.tabName),
				},
			}
		end
	end

	return tabs
end

function Ace3:SpellTabBuilder(db,title,event)
	local tab = {
		name = title,
		order = 1,
		type = "group",
		args = {
			disabledError = self:DisabledError(db,0),
			SpellConfig = {
				name = title.." Events",
				order = 1,
				type = "group",
				guiInline = true,
				args = self:BuildToggleCheckboxes(db),
			},
			soundList = self:SoundList(db,2),
			errors = self:Errors(db.info.errorName,3),
			SoundConfig = {
				name = "Audio Config",
				order = 5,
				type = "group",
				disabled = function()
					if (db.numEvents == 0) then
						return true
					else
						return false
					end
				end,
				guiInline = true,
				args = {
					source = self:Source(db,1,db.info.errorName,SOUND_SOURCE_OPTIONS),
					channel = self:Channel(db,2,SOUND_CHANNEL_OPTIONS),
					volume = self:Volume(db,3),
					--[[-----------------------------
						SCA Sound Packs
					-------------------------------]]
					packSelect = self:PackSelect(db,4,db.info.tabName),
					isSCARandom = self:IsSCARandom(db,7),
					--[[-----------------------------
						SharedMedia
					-------------------------------]]
					sharedMedia = self:SharedMedia(db,4,db.info.tabName),
					--[[-----------------------------
						Custom Audio
					-------------------------------]]
					audioSource = self:AudioSource(db,4,SOUND_CUSTOM_TYPES,db.info.tabName),
					soundEvents = {
						order = 5,
						type = "select",
						name = "Spell Events",
						desc = "Choose whether your custom audio is a file or sound pack",
						disabled = function()
							if (db.isSeparate and db.numEvents > 1) then
								return false
							else
								return true
							end
						end,
						get = function()							
							return db.selected
						end,
						set = function(_,value)
							db.selected = value
							
							db.tutorial.flags.spellEvent = true
							self:UpdateTutorial(db)
						end,
						values = function()
							local args,sortTable = {},{}
							
							sortTable = self:GetRefreshedTable(db)
									
							for i=1,#sortTable do
								if (sortTable[i] ~= "empty") then
									args[i] = db[sortTable[i]].label
								end
							end
									
							if (db.numEvents == 0) then
								args[0] = "No spell events"
							elseif (self:GetSelected(db) == "all") then
								args[0] = "Select a spell event"
							end
							
							return args
						end
					},
					toggleSeparate = self:ToggleSeparate(db,6,db.info.tabName),
					audioLocation = self:AudioLocation(db,7),
					numFiles = self:NumFiles(db,8,db.info.errorName,db.info.tabName),
					test = self:TestSound(db,9,db.info.errorName,db.info.tabName),
					isCustomRandom = self:IsCustomRandom(db,10),
					reset = self:SoftReset(db,11,db.info.tabName),
				},
			},
			[title.."TriggerHeader"] = {
				order = 6,
				type = 'group',
				name = title..' Triggers',
				guiInline = true,
				hidden = function()
					if (db.events[event]) then
						return false
					else
						return true
					end
				end,
				args = {
					spells = {
						order = 1,
						type = "toggle",
						name = "Spells",
						desc = '',
						get = function()
							return db[self:GetEvent(db,event)].options.spells
						end,
						set = function(_,value)
							db[self:GetEvent(db,event)].options.spells = value
						end,
					},
					[lower(sub(title,0,1)).."ots"] = {
						order = 2,
						type = "toggle",
						name = sub(title,0,1).."oTs",
						desc = '',
						get = function()
							return db[self:GetEvent(db,event)].options[lower(sub(title,0,1)).."ots"]
						end,
						set = function(_,value)
							db[self:GetEvent(db,event)].options[lower(sub(title,0,1)).."ots"] = value
						end,
					},
					pets = {
						order = 3,
						type = "toggle",
						name = "Pets",
						desc = "Damage that your pet does",
						hidden = function()
							if (event == "onDamage") then
								return false
							else
								return true
							end
						end,
						get = function()
							return db[self:GetEvent(db,event)].options.pet
						end,
						set = function(_,value)
							db[self:GetEvent(db,event)].options.pet = value
						end,
					},
					spacing = {
						order = 4,
						type = "description",
						name = " ",
						hidden = function()
							if (event == "onHeal") then
								return false
							else
								return true
							end
						end,
						width = "normal",
					},
					threshold = self:Threshold(db,self:GetEvent(db,event),title,db.info.errorName,db.info.tabName),
					critSelect = self:CritSelect(db,self:GetEvent(db,event),SOUND_CRIT_TRIGGERS),
					swing = {
						order = 7,
						type = "toggle",
						name = "Auto-Attacks",
						hidden = function()
							if (event == "onDamage") then
								return false
							else
								return true
							end
						end,
						desc = '',
						get = function()
							return db[self:GetEvent(db,event)].options.swing
						end,
						set = function(_,value)
							db[self:GetEvent(db,event)].options.swing = value
						end,
					},
				},
			},
			reset = self:HardReset(db,7,db.info.tabName),
		},
	}
	
	return tab
end