-- Lua Functions
local pairs, print, select, tonumber, type, unpack = pairs, print, select, tonumber, type, unpack
local find, gsub, lower, sub = string.find, string.gsub, string.lower, string.sub
local foreach, tinsert, tsort, wipe  = table.foreach, table.insert, table.sort, table.wipe
-- WoW API Functions / Variables
local CreateFont, CreateFrame = CreateFont, CreateFrame
local GetSpecialization, GetSpecializationInfo = GetSpecialization, GetSpecializationInfo
local InterfaceOptionsFrame = InterfaceOptionsFrame
local IsControlKeyDown, IsShiftKeyDown = IsControlKeyDown, IsShiftKeyDown
local IsAddOnLoaded, LoadAddOn = IsAddOnLoaded, LoadAddOn
local NPE_TutorialPointerFrame = NPE_TutorialPointerFrame
local UnitClass = UnitClass
local backupString,backupCtr = '',1

local SCA, Ace3, LSM = unpack(select(2,...))

--[[---------------------------------------
	CONSTANTS
-----------------------------------------]]
local SOUND_CUSTOM_TYPES = {
	["file"] = "Single Audio File",
	["pack"] = "Custom Audio Pack",
}

local SOUND_CRIT_TRIGGERS = {
	['crit'] = "Crits Only",
	['nocrit'] = "Non-crits Only",
	['always'] = "Always On",
}

local SPELL_EVENTS = {
	['onCast'] = "On Cast Start",
	['onSuccess'] = "On Cast Success",
	['onFail'] = "On Cast Fail",
	['onDamage'] = "On Damage Dealt",
	['onHeal'] = "On Heal Dealt",
}

local function InsertTab(tabNum)
	local tab = ''
	
	for i=1,tabNum do
		tab = tab.."    "
	end
	
	return tab
end

local function FormatString(key,val)
	local keyString,valString = '',''
	
	if (type(key) == "number") then
		keyString = [=[[%d]]=]
	else
		keyString = [[%s]]
	end
	
	if (type(val) == "number") then
		valString = [[%d]]
	elseif (type(val) == "string") then
		valString = [["%s"]]
	else
		valString = [[%s]]
	end
	
	if (type(val) == "boolean") then
		val = tostring(val)
	else
		val = gsub(val,"\\","\\\\")
		val = gsub(val,"\n","\\n")
	end
	
	return string.format(keyString..[[ = ]]..valString..[[,]],key,gsub(val,"\|","~"))
end

local function ParseBackupTable(db)
	for k,v in pairs(db) do
		if (type(v) == "table") then
			if (type(k) == "number") then
				backupString = backupString..string.format([=[[%d] = {]=],k)
			else
				backupString = backupString..string.format([[%s = {]],k)
			end
			backupCtr = backupCtr + 1
			ParseBackupTable(db[k])
		else
			backupString = backupString..FormatString(k,v)
		end
	end
	
	backupCtr = backupCtr - 1
	backupString = backupString.."},"

	return
end

function Ace3:BackupRestore(data,isRestore)
	--local db = self.db.profile.audio
	local db = SCA.profile.audio
	
	if (not isRestore) then
		if (data == "all") then
			db = db
		else
			db = db[data]
		end
		
		backupString = '{backupType="'..data..'",'
		
		for k,v in pairs(db) do
			if (type(v) == "table") then
				if (type(k) == "number") then
					backupString = backupString..string.format([=[[%d] = {]=],k)
				else
					backupString = backupString..string.format([[%s = {]],k)
				end
				backupCtr = backupCtr + 1
				ParseBackupTable(db[k])
			else
				backupString = backupString..FormatString(k,v)
			end
		end
		
		backupString = backupString.."}"
		return backupString
	else
		local args = {}
		local tableCtr = 0
		local backupType = ''
		local backupTable
		if (pcall(function() assert(loadstring("return "..gsub(data,"~","\|"))) end)) then
			backupTable = assert(loadstring("return "..gsub(data,"~","\|")))
		else
			return "      Invalid Backup String"
		end

		backupTable = backupTable()
		for k,v in pairs(backupTable) do
			if (k == "backupType") then
				backupType = v
				backupTable.backupType = nil
			end
		end
		
		if (backupType ~= "all") then
			--self.db.profile.audio[backupType] = backupTable
			SCA.profile.audio[backupType] = backupTable
		else
			--self.db.profile.audio = backupTable
			SCA.profile.audio = backupTable
		end
		
		ReloadUI()
	end
end

function Ace3:BuildToggleCheckboxes(db,isSpecific,noTutorial)
	local args = {}
	
	for k,_ in pairs(db.events) do
		args[k] = self:EventToggle(db,k,db[k].order,db.errorName,db.info.width,db[k].label,"Toggle whether or not a sound will play when "..db[k].desc,db.info.tabName,isSpecific,noTutorial)
	end
	
	if (not noTutorial) then
		args["spacing1"] = self:Spacer(9,"half")
		args["tutorial"] = self:TutorialToggle(db,10,"half")
	end
	
	return args
end

function Ace3:CheckIgnoreSpell(newIgnore,errors,isValidSpell,spellID,name,errorMsg)
	--local spells = self.db.profile.audio.spells
	local spells = SCA.profile.audio.spells
	
	if (isValidSpell and not spells.ignored[spellID]) then
		errors.name = " "
		
		newIgnore.validSpell = name
		SCA.temp.errors.ignored = nil
		
		
		if (spells.specific[spellID]) then
			errors.name = name.." |cFFFF9B00has a sound assigned to it. Ignoring it will remove its sound|r"
			SCA.temp.errors.ignored = true
		end
	else
		if (spells.ignored[spellID]) then
			errors.name = "|cFFFF5F99Spell is already ignored|r"
		else
			errors.name = errorMsg
		end
		
		newIgnore.validSpell = ''
		SCA.temp.errors.ignored = true
	end
end

function Ace3:CheckSpecificSpell(newSound,errors,isValidSpell,spellID,name,errorMsg)
	--local spells = self.db.profile.audio.spells
	local spells = SCA.profile.audio.spells
	
	if (isValidSpell) then
		newSound.numEvents = 0
		newSound.validSpell = name
		
		if (spells.specific[spellID]) then
			errors.name = name.." |cFFFF9B00's sound will be updated!|r"
			SCA.temp.errors.newSound = true
		elseif (spells.ignored[spellID]) then
			errors.name = name.." |cFFFF9B00is being ignored and will be removed from the ignored list|r"
			SCA.temp.errors.newSound = true
		else
			errors.name = " "
			SCA.temp.errors.newSound = nil
		end
		
		for k,_ in pairs(newSound.events) do
			if (self:CheckSpell(k,name)) then
				newSound.numEvents = newSound.numEvents + 1
				newSound.events[k] = true
			else
				newSound.events[k] = false
			end
		end
	else
		errors.name = errorMsg
		SCA.temp.errors.newSound = true
		newSound.validSpell = false
	end
end

local isRecording = false
local isDamage = false
function Ace3:CopyTable(currentTable, defaultTable)
	if (type(currentTable) ~= "table") then 
		currentTable = {} 
	end

	if (type(defaultTable) == 'table') then
		for k, v in pairs(defaultTable) do
			if type(v) == "table" then
				if (k == "damage") then
					isDamage = true
				end
				
				if (k == "events" and isDamage) then
					isRecording = true
				end
				
				if (isRecording) then
					--SCA.DataFrame.text:SetText(tostring(SCA.DataFrame.text:GetText())..tostring(k).."\n")
				end
				v = self:CopyTable(currentTable[k], v)
			else
				if (isRecording) then
					--SCA.DataFrame.text:SetText(tostring(SCA.DataFrame.text:GetText()).."---"..tostring(v).."\n")
				end
				currentTable[k] = v
			end
		end
	end

	return currentTable
end

function Ace3:CreateSpellFrame(options)
	local sortTable = {}
	local class,EnglishClass = UnitClass("player")
	local _,specName = GetSpecializationInfo(GetSpecialization())
	local spec = GetSpecialization()
	local spells = SCA.data.spells
	local SpellFrame = CreateFrame("frame")
	
	SpellFrame:SetWidth(200)
	SpellFrame:SetPoint("TOPLEFT",InterfaceOptionsFrame,"TOPLEFT",355,-25)
	SpellFrame:SetFrameStrata("TOOLTIP")
	SpellFrame:SetBackdrop(SCA.BackdropSB)
	SpellFrame:SetBackdropColor(0,0,0,1)
	SpellFrame:SetBackdropBorderColor(1,1,1,1)
	SpellFrame:Hide()
	
	SpellFrame.text = SpellFrame:CreateFontString(nil,'HIGH', 'GameFontHighlightLarge')
	SpellFrame.text:SetWidth(SpellFrame:GetWidth())
	SpellFrame.text:SetHeight(20)
	SpellFrame.text:SetPoint("TOPLEFT",SpellFrame,"TOPLEFT",0,0)
	SpellFrame.text:SetFont("Fonts\\FRIZQT__.TTF", 8)
	SpellFrame.text:SetTextColor(0,1,0,1)
	SpellFrame.text:SetText(specName.." "..class.." Spells")
	
	SpellFrame.dFont = CreateFont("SCA Disabled Font")
	SpellFrame.dFont:SetJustifyH("LEFT")
	SpellFrame.dFont:SetTextColor(0.3,0.3,0.3,1)
	SpellFrame.dFont:SetFont("Fonts\\FRIZQT__.TTF", 8)
	
	SpellFrame.eFont = CreateFont("SCA Error Font")
	SpellFrame.eFont:SetJustifyH("LEFT")
	SpellFrame.eFont:SetTextColor(1,0,0,1)
	SpellFrame.eFont:SetFont("Fonts\\FRIZQT__.TTF", 8)
	
	SpellFrame.cFont = CreateFont("SCA Caution Font")
	SpellFrame.cFont:SetJustifyH("LEFT")
	SpellFrame.cFont:SetTextColor(1,0.6,0,1)
	SpellFrame.cFont:SetFont("Fonts\\FRIZQT__.TTF", 8)
	
	SpellFrame.CFont = CreateFont("SCA Caution Highlight Font")
	SpellFrame.CFont:SetJustifyH("LEFT")
	SpellFrame.CFont:SetTextColor(1,0.6,0,1)
	SpellFrame.CFont:SetFont("Fonts\\FRIZQT__.TTF", 9)
	
	SpellFrame.nFont = CreateFont("SCA Normal Font")
	SpellFrame.nFont:SetJustifyH("LEFT")
	SpellFrame.nFont:SetTextColor(1,1,1,1)
	SpellFrame.nFont:SetFont("Fonts\\FRIZQT__.TTF", 8)
	
	SpellFrame.hFont = CreateFont("SCA Highlight Font")
	SpellFrame.hFont:SetJustifyH("LEFT")
	SpellFrame.hFont:SetTextColor(1,1,0.25,1)
	SpellFrame.hFont:SetFont("Fonts\\FRIZQT__.TTF", 9)
	
	SpellFrame.sFont = CreateFont("SCA Selected Font")
	SpellFrame.sFont:SetJustifyH("LEFT")
	SpellFrame.sFont:SetTextColor(0.1,1,0.25,1)
	SpellFrame.sFont:SetFont("Fonts\\FRIZQT__.TTF", 9)
	
	for k,v in pairs(spells.spellNames[EnglishClass][spec]) do
		tinsert(sortTable,k.." ("..v..")")
	end
	
	SpellFrame:SetHeight((#sortTable * 10) + 45)
	
	tsort(sortTable)
	
	for i=1,#sortTable do	
		local spellNameLoc = find(sortTable[i],"%(")
		local spellName = sub(sortTable[i],0,(spellNameLoc-2))
		
		SpellFrame[spellName] = CreateFrame("Button",nil,SpellFrame)
		SpellFrame[spellName]:SetWidth(SpellFrame:GetWidth())
		SpellFrame[spellName]:SetHeight(20)
		if (i == 1) then
			SpellFrame[spellName]:SetPoint("TOPLEFT",SpellFrame,"TOPLEFT",0,-20)
		else
			SpellFrame[spellName]:SetPoint("TOPLEFT",SpellFrame,"TOPLEFT",0,(-10*i)-10)
		end
		
		if (spells.pvp[EnglishClass][spec][spells.spellNames[EnglishClass][spec][spellName]]) then
			SpellFrame[spellName]:SetText(sortTable[i].." |cFFFF0000(PvP)|r")
		elseif (spells.artifact[EnglishClass][spec][spells.spellNames[EnglishClass][spec][spellName]]) then
			SpellFrame[spellName]:SetText(sortTable[i].." |cFF00F3FF(Artifact)|r")
		else
			SpellFrame[spellName]:SetText(sortTable[i])
		end
		
		SpellFrame[spellName]:SetNormalFontObject(SpellFrame.nFont)
		SpellFrame[spellName]:SetHighlightFontObject(SpellFrame.hFont)
		SpellFrame[spellName]:Show()

		SpellFrame[spellName]:SetScript("OnClick",function(this,button)
			local spellNameLoc = find(this:GetText(),"%(")
			local spellName = sub(this:GetText(),0,(spellNameLoc-2))
			local newItem
			
			if (SCA.temp.addSound) then
				local errors = options.args.SpecificTab.args.errors.args.messages
				newItem = SCA.temp.newSound
				newItem.validSpell = spellName
				
				self:CheckSpecificSpell(newItem,errors,self:CheckSpell('onSuccess',spellName))
				self:UpdateTutorial(newItem)
			else
				local errors = options.args.ignoreTab.args.errors.args.messages
				newItem = SCA.temp.newIgnore
				newItem.validSpell = spellName

				self:CheckIgnoreSpell(newItem,errors,self:CheckSpell('onSuccess',spellName))
			end
			
			self:NavigateInterfaceOptions(SCA.temp.selectedMenuItem,true)

			SpellFrame:Hide()
		end)
	end

	SpellFrame.close = CreateFrame("Button",nil,SpellFrame)
	SpellFrame.close:SetWidth(SpellFrame:GetWidth())
	SpellFrame.close:SetHeight(20)
	SpellFrame.close:SetPoint("TOPLEFT",SpellFrame,"TOPLEFT",0,floor((SpellFrame:GetHeight() - 20) * -1))
	SpellFrame.close:SetText("Close")
	SpellFrame.close:SetNormalFontObject(SpellFrame.nFont)
	SpellFrame.close:SetHighlightFontObject(SpellFrame.hFont)
	SpellFrame.close:Show()

	SpellFrame.close:SetScript("OnClick",function(self,button)
		SpellFrame:Hide()
	end)
		
	SCA.SpellFrame = SpellFrame
end

-- Destroys a Tutorial Panel when it is no longer needed or right-clicked by the user.
function Ace3:DestroyTutorialPanel(db,panel)
	local tutorial = db.tutorial

	NPE_TutorialPointerFrame:Hide(tutorial.ids[panel])
	tutorial.ids[panel] = nil
	tutorial.frameCount = tutorial.frameCount - 1
end

function Ace3:GetEvent(db,event)
	if (db.isSeparate) then
		return event
	else
		if (type(db.events[event]) == "boolean" or db.playSong) then
			return 'all'
		else
			return nil
		end
	end
end

function Ace3:GetRefreshedTable(db,isSpecific,isRoot)
	local tempTable,newTable = {},{}
	local confirmSpell = true
	local tableCtr,tableSize = 1,0
	
	-- Input table key names into a temporary table that are valid.
	if (isRoot) then
		for k,v in pairs(db) do
			if (db[k].order) then
				tinsert(tempTable,db[k].order,k)
				tableSize = tableSize + 1
			end
		end
	else
		for k,v in pairs(db.events) do
			if (isSpecific) then
				confirmSpell = self:CheckSpell(k,db.validSpell)
			end
			if (v and confirmSpell) then
				tinsert(tempTable,db[k].order,k)
			else
				tinsert(tempTable,db[k].order,"empty")
			end
			tableSize = tableSize + 1
		end
	end
	
	

	-- For some reason, some table slots are skipped; they need to be fixed.
	foreach(tempTable,function(k,v)
		if (k ~= tableCtr) then
			tempTable[tableCtr] = v
		end
		tableCtr = tableCtr + 1
	end)
	
	-- The table may now have extra slots; they need to be trimmed
	for i=1,#tempTable do
		if (i > tableSize) then
			tempTable[i] = nil
		end
	end
	
	-- Build the new table of values
	for i=1,#tempTable do
		if (tempTable[i] ~= "empty") then
			tinsert(newTable,tempTable[i])
		end
	end
	
	-- We no longer need the tempTable, so let's clear it
	wipe(tempTable)
	
	--if (not isRoot) then
		db.keys = nil
		db.keys = {}
		
		-- Rebuild key table
		for i=0,#newTable do
			if (i == 0) then
				db.keys[i] = "all"
			else
				db.keys[i] = newTable[i]
			end
		end
	
	-- Check to see if the selected event still exists in the keys table. If not, we need to select a new event.
	if (not db.isSeparate and db.selected ~= 0) then
		db.selected = 0
	elseif (not db.keys[db.selected]) then
		for i=1,(#db.keys - 1) do
			if (db.keys[i]) then
				db.selected = i
			end
		end
	end
	--end
	
	return newTable
end

function Ace3:GetSoundInfo(db,event,source,isSeparate)
	local msg = ''
	
	if (isSeparate) then
		if (type(isSeparate) == "string") then
			msg = msg..isSeparate..": "
		elseif (source == "packs" or source == "shared" or source == "custom") then
			msg = msg..gsub(SPELL_EVENTS[event],"On ","")..": "
		end
	end
		
	if (source == "packs") then
		msg = msg.."|cffffd200SCA \"|r|cFF94FF5F"..db[event].packs.text.."|r|cffffd200\" Pack|r"
		
		if (event == "onDamage") then
			msg = msg.." |cFFFF0000|| |cFF73FFF0"..SOUND_CRIT_TRIGGERS[db[event].options.trigger].."|r"
		end
		
		if (event == "onHeal") then
			msg = msg.." |cFFFF0000|| |cFF73FFF0"..SOUND_CRIT_TRIGGERS[db[event].options.trigger].."|r"
		end
		
		if (db[event].packs.isRandom.enabled) then
			msg = msg.." |cFFFF0000|| |cFFEAFF73Random|r"
		else
			msg = msg.." |cFFFF0000|| |cFFEAFF73Sequential|r"
		end
	elseif (source == "shared") then
		msg = msg.."|cffffd200SharedMedia|r |cFFFF0000|| |cFF94FF5F"..db[event].shared.value.."|r"

		if (event == "onDamage") then
			msg = msg.." |cFFFF0000|| |cFF73FFF0"..SOUND_CRIT_TRIGGERS[db[event].options.trigger].."|r"
		end
		
		if (event == "onHeal") then
			msg = msg.." |cFFFF0000|| |cFF73FFF0"..SOUND_CRIT_TRIGGERS[db[event].options.trigger].."|r"
		end
	elseif (source == "custom") then
		msg = msg.."|cffffd200"..SOUND_CUSTOM_TYPES[db[event].custom.customType].."|r |cFFFF0000|| |cFF94FF5F"..db[event].custom.path.."|r"
		
		if (event == "onDamage") then
			msg = msg.." |cFFFF0000|| |cFF73FFF0"..SOUND_CRIT_TRIGGERS[db[event].options.trigger].."|r"
		end
		
		if (event == "onHeal") then
			msg = msg.." |cFFFF0000|| |cFF73FFF0"..SOUND_CRIT_TRIGGERS[db[event].options.trigger].."|r"
		end
		
		if (db[event].custom.customType == "pack") then
			if (tonumber(db[event].custom.max) > 1) then
				msg = msg.." |cFFFF0000|||cFFEC73FF"..db[event].custom.max.." files|r"
			end
		
			if (db[event].custom.isRandom.enabled) then
				msg = msg.." |cFFFF0000|| |cFFEAFF73Random|r"
			else
				msg = msg.." |cFFFF0000|| |cFFEAFF73Sequential|r"
			end
		end
	end

	return msg
end

function Ace3:GroupBuilder(db,width,toggleOnly,bypass)
	local args = {}
	local order = 0

	local root
	if (bypass) then
		root = db[bypass]
	else
		root = db
	end
	
	args["allToggle"] = {
		order = 1,
		type = "toggle",
		name = "Enabled",
		desc = '',
		get = function()
			return root.enabled
		end,
		set = function(self,value)
			root.enabled = value
		end,
	}
	args["spacing"] = {
		order = 2,
		type = "description",
		name = " ",
		width = "double",
	}
	order = order + 2
	
	if (not toggleOnly) then
		local events,root
		if (bypass) then
			events = db[bypass].events
			root = db[bypass]
		else
			events = db.events
			root = db
		end
		
		for k,v in pairs(events) do
			args[k.."Toggle"] = {
				order = root[k].order + order,
				type = "toggle",
				name = root[k].label,
				desc = '',
				disabled = function()
					if (not bypass) then
						if (db.enabled) then
							return false
						else
							return true
						end
					else
						if (not db.enabled) then
							return true
						else
							return not root.enabled
						end
					end
				end,
				get = function()
					return events[k]
				end,
				set = function(self,value)
					events[k] = value
					
					if (value) then
						root.numEvents = root.numEvents + 1
					else
						root.numEvents = root.numEvents - 1
					end
				end,
				width = width,
			}
		end
	end
	
	return args
end

--[[-----------------------------------------------------------------
	This function runs when the Interface Options Frame is opened.
	It prepares the config tools for all options.
-----------------------------------------------------------------]]--
function Ace3:InitOptions(db,options,noTutorial)
	db.numEvents = 0
	
	if (not noTutorial) then
		db.tutorial.enabled = false
		db.tutorial.ids = nil
		db.tutorial.ids = {}
	end

	if (db.isSeparate) then
		self:UpdateSeparateSoundMessage(db,options)
	end

	for k,v in pairs(db.events) do
		if (v) then
			db.numEvents = db.numEvents + 1
		end
	end
end



-- Refreshes the Spell Assignment list
function Ace3:ReloadSpellAssignList(options,db)
	local args = {}
	local _,className = UnitClass("player")
	local spec = GetSpecialization()
	local spells = SCA.data.spells
	local sortTable = {}
	
	-- Insert spell names from "Specific" table into local table to be sorted
	for k,v in pairs(db.specific) do
		if (spells.overload[className]) then
			if (spells.overload[className][spec][k]) then
				tinsert(sortTable,spells.overload[className][spec][k])
			else
				tinsert(sortTable,spells.onSuccess[className][spec][k])
			end
		else
			tinsert(sortTable,spells.onSuccess[className][spec][k])
		end
	end
	
	tsort(sortTable)
	
	for i=1,#sortTable do
		local name = sortTable[i]

		if (spells.spellNames[className][spec][sortTable[i]] or spells.overload[className][spec][sortTable[i]]) then
			local specificTbl = db.specific[spells.spellNames[className][spec][sortTable[i]]]
			if (i == 1) then
				args["help"] = {
					order = 1,
					type = "description",
					name = "EDIT: Shift+Click. DELETE: Ctrl+Click",
					desc = "HELP TIPS",
				}
			end
			args["spell"..i] = {
				order = i+1,
				type = "toggle",
				desc = function()
					local msg = ''
					local ctr = 0
					if (not specificTbl.isSeparate) then
						return self:GetSoundInfo(specificTbl,'all',specificTbl.all.options.source)
					else
						for k,v in pairs(SPELL_EVENTS) do
							if (specificTbl[k]) then
								--if (specificTbl.isSeparate) then
									if (ctr > 0) then
										msg = msg.."\n"
									end
									
									for j,_ in pairs(specificTbl[k]) do
										msg = msg..self:GetSoundInfo(specificTbl,k,j,specificTbl.isSeparate)
										
										ctr = ctr + 1
									end
								--[[else
									

									break
								end]]
							end
						end
						
						return msg
					end
				end,
				descStyle = "inline",
				name = "|cFFEBC000"..name.."|r",
				get = function() 
					return true
				end,
				set = function(this,value)
					if (IsShiftKeyDown()) then
						local ctr = 0
						local newSound = SCA.temp.newSound
						
						SCA.temp.addSound = true
						SCA.temp.editSound = true
						
						newSound.validSpell = name
						
						newSound.isSeparate = specificTbl.isSeparate
						
						for k,_ in pairs(SPELL_EVENTS) do
							if (specificTbl[k]) then
								newSound[k].validAudio = true
								newSound.events[k] = true
								
								newSound[k].options.source = specificTbl[k].options.source
								newSound[k].options.channel = specificTbl[k].options.channel
								
								if (k == "onDamage" or k == "onHeal") then
									newSound[k].options.trigger = specificTbl[k].options.trigger
									newSound[k].options.threshold = specificTbl[k].options.threshold
								end
								
								if (specificTbl[k].packs) then
									newSound[k].packs.value = specificTbl[k].packs.value
									newSound[k].packs.text = specificTbl[k].packs.text
									newSound[k].packs.path = specificTbl[k].packs.path
									newSound[k].packs.max = specificTbl[k].packs.max
									newSound[k].packs.isRandom.enabled = specificTbl[k].packs.isRandom.enabled
									newSound[k].packs.isRandom.currentFile = specificTbl[k].packs.isRandom.currentFile
								end
								
								if (specificTbl[k].shared) then
									newSound[k].shared.value = specificTbl[k].shared.value
								end
								
								if (specificTbl[k].custom) then
									newSound[k].custom.customType = specificTbl[k].custom.customType
									newSound[k].custom.path = specificTbl[k].custom.path
									newSound[k].custom.max = specificTbl[k].custom.max
									newSound[k].custom.isRandom.enabled = specificTbl[k].custom.isRandom.enabled
									newSound[k].custom.isRandom.currentFile = specificTbl[k].custom.isRandom.currentFile
								end
								ctr = ctr + 1
							end
						end
						
						if (newSound.isSeparate) then
							self:UpdateSeparateSoundMessage(newSound,this.options.args.SpecificTab,true)
						end
						
						newSound.numEvents = ctr
					elseif (IsControlKeyDown()) then
						db.specific[spells.spellNames[className][spec][sortTable[i]]] = nil
						self:ReloadSpellAssignList(options,db)
					end
				end,
				width = "full",
			}
		end
	end

	options.args.SpecificTab.args.assignedSpells.args = args
end

-- Refreshes the Spell Ignore List
function Ace3:ReloadSpellIgnoreList(options,tab)
	--local db = self.db.profile
	local db = SCA.profile
	local args = {}
	local sortTable = {}

	for k,_ in pairs(db.audio.spells.ignored) do
		local _,_,name =  self:CheckSpell('onSuccess',k)
		
		tinsert(sortTable,name)
	end
	
	tsort(sortTable)
	
	for i=1,#sortTable do
		local _,spellID,name = self:CheckSpell('onSuccess',sortTable[i])
		
		args["spell"..i] = {
			order = i,
			type = "toggle",
			desc = "Spell ID: "..spellID,
			descStyle = "inline",
			name = "|cFFEBC000"..name.."|r",
			get = function() 
				return true
			end,
			set = function(self,value)
				db.audio.spells.ignored[spellID] = nil
				self:ReloadSpellIgnoreList(options)
			end,
		}
	end

	options.args.ignoreTab.args.ignoreList.args = args
end

-- Relocates a Tutorial panel when an error or a sound list is displayed or hidden
function Ace3:RelocateTutorialPanel(db,panel,ignoreSeparate)
	local tutorial = db.tutorial
	local frame = NPE_TutorialPointerFrame.InUseFrames[tutorial.ids[panel]]
	
	if (frame) then
		local point,parent,relativePoint = frame:GetPoint()
		local newY = 0
		local oldX = tutorial.ofs[panel].x
		local oldY = tutorial.ofs[panel].y

		frame:ClearAllPoints()
		
		if (db.isSeparate and SCA.temp.errors[db.dbName] and not ignoreSeparate) then
			if (panel == 'customType') then
			end
			newY = oldY - 122
		elseif (db.isSeparate and not ignoreSeparate) then
			if (panel == 'customType') then
			end
			newY = oldY - 66
		elseif (SCA.temp.errors[db.dbName]) then
			if (panel == 'customType') then
			end
			newY = oldY - 56
		else
			if (panel == 'customType') then
			end
			newY = oldY
		end
		
		frame:SetPoint(point,parent,relativePoint,oldX,newY)
	end
end

function Ace3:ResetDB(db,isResetAll,isNewSound)
	if (isNewSound) then
		SCA.temp.addSound = false
		SCA.temp.editSound = false
	end
	
	if (isResetAll) then
		if (db.validSpell) then
			db.validSpell = false
		end
		
		if (db.numEvents) then
			db.numEvents = 0
		end
		
		if (db.isSeparate) then
			db.isSeparate = false
		end
		
		if (db.selected) then
			db.selected = 0
		end

		if (db.tutorial) then
			db.tutorial.enabled = false
			db.tutorial.flags.isSeparate = false
			db.tutorial.flags.spellEvent = false
			db.tutorial.flags.spellEvents = false
			db.tutorial.flags.source = false
			db.tutorial.flags.customType = false
			db.tutorial.flags.numFiles = false
			db.tutorial.flags.audioLocationPack = false
			db.tutorial.flags.audioLocationFile = false
			
			db.tutorial.ids = nil
			db.tutorial.ids = {}
		end
	end
	
	if (not db.isSeparate) then
		if (isResetAll) then
			for k,_ in pairs(db.events) do
				db.events[k] = false
			end
		
			if (db.all.options.trigger) then
				db.all.options.trigger = "crit"
			end
			
			if (db.all.options.swing ~= nil and not db.all.options.swing) then
				db.all.options.swing = true
			end
			
			if (db.all.options.pet ~= nil and not db.all.options.pet) then
				db.all.options.pet = true
			end
			
			if (db.all.options.dots ~= nil and not db.all.options.dots) then
				db.all.options.dots = true
			end
			
			if (db.all.options.hots ~= nil and not db.all.options.hots) then
				db.all.options.hots = true
			end
			
			if (db.all.options.spells ~= nil and not db.all.options.spells) then
				db.all.options.spells = true
			end
			
			if (db.all.options.threshold) then
				db.all.options.threshold = 0
			end
		end
		
		if (db.all.validAudio) then
			db.all.validAudio = false
		end
		
		db.all.options.source = "packs"
		db.all.options.channel = "Master"
		
		db.all.packs.value = "empty"
		db.all.packs.text = ''
		db.all.packs.path = ''
		db.all.packs.max = 1
		db.all.packs.isRandom.enabled = true
		db.all.packs.isRandom.currentFile = 1
		
		db.all.shared.value = "None"
		
		db.all.custom.path = "Interface\\AUDIO\\"
		db.all.custom.max = 1
		db.all.custom.customType = "file"
		db.all.custom.isRandom.enabled = true
		db.all.custom.isRandom.currentFile = 1
	else
		if (not isResetAll) then
			local k = self:GetSelected(db)
			
			if (db[k].validAudio) then
				db[k].validAudio = false
			end
			
			db[k].options.source = "packs"
			db[k].options.channel = "Master"
			
			db[k].packs.value = "empty"
			db[k].packs.text = ''
			db[k].packs.path = ''
			db[k].packs.max = 1
			db[k].packs.isRandom.enabled = true
			db[k].packs.isRandom.currentFile = 1
			
			db[k].shared.value = "None"
			
			db[k].custom.path = "Interface\\AUDIO\\"
			db[k].custom.max = 1
			db[k].custom.customType = "file"
			db[k].custom.isRandom.enabled = true
			db[k].custom.isRandom.currentFile = 1
		else
			for k,_ in pairs(db.events) do
				db.events[k] = false
				
				if (k == "onDamage" or k == "onHeal") then
					db[k].options.trigger = "crit"
					db[k].options.threshold = 0
				end
				
				if (db[k].validAudio) then
					db[k].validAudio = false
				end
				
				db[k].options.source = "packs"
				db[k].options.channel = "Master"
				
				db[k].packs.value = "empty"
				db[k].packs.text = ''
				db[k].packs.path = ''
				db[k].packs.max = 1
				db[k].packs.isRandom.enabled = true
				db[k].packs.isRandom.currentFile = 1
				
				db[k].shared.value = "None"
				
				db[k].custom.path = "Interface\\AUDIO\\"
				db[k].custom.max = 1
				db[k].custom.customType = "file"
				db[k].custom.isRandom.enabled = true
				db[k].custom.isRandom.currentFile = 1
			end
		end
	end
	
	GameTooltip:Hide()
end

function Ace3:SubGroupBuilder(db)
	local args = {}
	
	args["GlobalToggle"] = {
		order = 1,
		type = "toggle",
		name = "Enabled",
		desc = '',
		get = function()
			return db.enabled
		end,
		set = function(self,value)
			db.enabled = value
		end,
	}
	
	local ctr = 1
	for k,_ in pairs(db) do
		if (type(db[k]) == "table") then
			args[k] = {
				name = db[k].info.label,
				order = db[k].order,
				type = "group",
				disabled = function()
					if (db.enabled) then
						return false
					else
						return true
					end
				end,
				args = self:GroupBuilder(db,nil,nil,k),
			}
			ctr = ctr + 1
		end
	end
	
	return args
end

function Ace3:UpdateSeparateSoundMessage(soundTbl,options,isSpecific)
	local messages = options.args.soundList.args.messages
	
	if (soundTbl.isSeparate) then
		local newMsg = ''
		local msgCache = messages.name
		
		if (isSpecific) then
			local ctr = 1
			local tempTable = self:GetRefreshedTable(soundTbl,true)
			
			for i=1,#tempTable do
				local k = tempTable[i]

				if (ctr > 1) then
					newMsg = newMsg.."\n"
				end

				newMsg = newMsg..gsub(SPELL_EVENTS[k],"On ","")..": "
				if (soundTbl[k].validAudio) then
					newMsg = newMsg..self:GetSoundInfo(soundTbl,k,soundTbl[k].options.source)
				else
					newMsg = newMsg..""
				end
				
				ctr = ctr + 1
			end
			
			wipe(tempTable)
		else
			local tempTable = self:GetRefreshedTable(soundTbl)
			
			local ctr = 1
			for i=1,#tempTable do
				local k = tempTable[i]
				
				if (ctr > 1) then
					newMsg = newMsg.."\n"
				end

				newMsg = newMsg..gsub(soundTbl[k].label,"On ","")..": "
				if (soundTbl[k].validAudio) then
					newMsg = newMsg..self:GetSoundInfo(soundTbl,k,soundTbl[k].options.source)
				else
					newMsg = newMsg..""
				end
				
				ctr = ctr + 1
			end
			
			wipe(tempTable)
		end
		
		messages.name = newMsg
	else
		messages.name = ' '
	end
end

function Ace3:UpdateTutorial(db,trigger)
	local tutorial = db.tutorial
	
	if (tutorial.enabled) then
		if (not IsAddOnLoaded("Blizzard_Tutorial")) then
			LoadAddOn("Blizzard_Tutorial")
		end
		
		if (db.validSpell == false) then
			tutorial.ids.spellName = NPE_TutorialPointerFrame:Show("In order to access the sound controls, you first must input a valid spell name or ID.\n\n|cFF00FF00Read the textbox's tooltip for more details.|r","DOWN",InterfaceOptionsFrame,331,-150,"TOPLEFT")
			tutorial.frameCount = tutorial.frameCount + 1
		else
			if (db.validSpell and db.selected == "empty" and (trigger ~= "toggle" or trigger == "toggle")) then
				db.selected = "onSuccess"
			end
			
			if (tutorial.ids.spellName) then
				self:DestroyTutorialPanel(db,'spellName')
			end

			if (not tutorial.ids.spellEvents and not tutorial.flags.spellEvents) then
				tutorial.ids.spellEvents = NPE_TutorialPointerFrame:Show("Toggling these \"|cFFFFFFFFSpell Events|r\" will set which events will trigger sounds.","DOWN",InterfaceOptionsFrame,tutorial.ofs.spellEvents.x,tutorial.ofs.spellEvents.y,"TOPLEFT")
				tutorial.frameCount = tutorial.frameCount + 1
			else
				if (tutorial.flags.spellEvents and tutorial.ids.spellEvents) then
					self:DestroyTutorialPanel(db,'spellEvents')
				end
			end
			
			if (not tutorial.ids.source and not tutorial.flags.source and db[db.selected].options.source == "packs") then
				tutorial.ids.source = NPE_TutorialPointerFrame:Show("Using this dropdown allows you to select the source of sounds you want to choose from.","RIGHT",InterfaceOptionsFrame,tutorial.ofs.source.x,tutorial.ofs.source.y,"TOPLEFT")
				tutorial.frameCount = tutorial.frameCount + 1
				
				if (db.isSeparate or SCA.temp.errors[db.dbName]) then
					self:RelocateTutorialPanel(db,'source')
				end
			else
				if (tutorial.flags.source) then
					self:DestroyTutorialPanel(db,'source')
				end
				
				self:RelocateTutorialPanel(db,'source')
			end
			
			if (not tutorial.ids.isSeparate and not db.isSeparate and not tutorial.flags.isSeparate) then 
				tutorial.ids.isSeparate = NPE_TutorialPointerFrame:Show("Toggling this will allow the ability to separate the spell events; allowing you to add a specific sound for each event.\n\n|cFFFF5F99Having more than one spell event selected is required.|r","LEFT",InterfaceOptionsFrame,tutorial.ofs.isSeparate.x,tutorial.ofs.isSeparate.y,"TOPLEFT")
				tutorial.frameCount = tutorial.frameCount + 1
			else
				if (db.isSeparate or tutorial.flags.isSeparate) then
					self:DestroyTutorialPanel(db,'isSeparate')
				end
				
				self:RelocateTutorialPanel(db,'isSeparate')
			end
			
			if (db.isSeparate) then
				if (not tutorial.ids.spellEvent and not tutorial.flags.spellEvent) then
					tutorial.ids.spellEvent = NPE_TutorialPointerFrame:Show("Use this dropdown to select the specific spell events to assign sounds.","DOWN",InterfaceOptionsFrame,tutorial.ofs.spellEvent.x,tutorial.ofs.spellEvent.y,"TOPLEFT")
					tutorial.frameCount = tutorial.frameCount + 1
				else
					if (tutorial.flags.spellEvent) then
						self:DestroyTutorialPanel(db,'spellEvent')
					end
					
					self:RelocateTutorialPanel(db,'spellEvent',true)
				end
			else
				if (tutorial.ids.spellEvent) then
					self:DestroyTutorialPanel(db,'spellEvent')
				end
				
				self:RelocateTutorialPanel(db,'spellEvent',true)
			end
			
			if (db[db.selected].options.source == "custom") then
				if (not tutorial.ids.customType and not tutorial.flags.customType) then
					tutorial.ids.customType = NPE_TutorialPointerFrame:Show("This dropdown sets up the ability to add a single sound file or a custom sound pack","RIGHT",InterfaceOptionsFrame,tutorial.ofs.customType.x,tutorial.ofs.customType.y,"TOPLEFT")
					tutorial.frameCount = tutorial.frameCount + 1
				
					if (db.isSeparate or SCA.temp.errors[db.dbName]) then
						self:RelocateTutorialPanel(db,'customType')
					end
				else
					if (tutorial.flags.customType and tutorial.ids.customType) then
						self:DestroyTutorialPanel(db,'customType')
					end
					
					self:RelocateTutorialPanel(db,'customType')
				end
				
				if (not tutorial.ids.audioLocationFile) then
					if (db[db.selected].custom.customType == "file" and not tutorial.flags.audioLocationFile) then
						tutorial.ids.audioLocationFile = NPE_TutorialPointerFrame:Show("Set the location of the audio file. The file |cFF00FF00MUST|r be located somewhere within the \"|cFFFFFFFFInterface|r\" folder.\r\r|cFF00FF00Recommended:\nInterface\\audio\\<file>","UP",InterfaceOptionsFrame,tutorial.ofs.audioLocationFile.x,tutorial.ofs.audioLocationFile.y,"TOPLEFT")
						tutorial.frameCount = tutorial.frameCount + 1
				
						if (db.isSeparate or SCA.temp.errors[db.dbName]) then
							self:RelocateTutorialPanel(db,'audioLocationFile')
						end
					end
				else
					if (tutorial.flags.audioLocationFile or db[db.selected].custom.customType == "pack") then
						self:DestroyTutorialPanel(db,'audioLocationFile')
					end
					
					self:RelocateTutorialPanel(db,'audioLocationFile')
				end
				
				if (not tutorial.ids.audioLocationPack) then
					if (db[db.selected].custom.customType == "pack" and not tutorial.flags.audioLocationPack) then
						tutorial.ids.audioLocationPack = NPE_TutorialPointerFrame:Show("Set the location of the audio pack. The pack |cFF00FF00MUST|r be located somewhere within the \"|cFFFFFFFFInterface|r\" folder.\r\r|cFF00FF00Recommended:\nInterface\\audio\\<pack>\\","UP",InterfaceOptionsFrame,tutorial.ofs.audioLocationPack.x,tutorial.ofs.audioLocationPack.y,"TOPLEFT")
						tutorial.frameCount = tutorial.frameCount + 1
				
						if (db.isSeparate or SCA.temp.errors[db.dbName]) then
							self:RelocateTutorialPanel(db,'audioLocationPack')
						end
					end
				else
					if (tutorial.flags.audioLocationPack or db[db.selected].custom.customType == "file") then
						self:DestroyTutorialPanel(db,'audioLocationPack')
					end
					
					self:RelocateTutorialPanel(db,'audioLocationPack')
				end
				
				if (not tutorial.ids.numFiles) then
					if (db[db.selected].custom.customType == "pack" and not tutorial.flags.numFiles) then
						tutorial.ids.numFiles = NPE_TutorialPointerFrame:Show("In order for the pack to be registered to work correctly, you must enter the number of files within the custom pack folder.","UP",InterfaceOptionsFrame,tutorial.ofs.numFiles.x,tutorial.ofs.numFiles.y,"TOPLEFT")
						tutorial.frameCount = tutorial.frameCount + 1
				
						if (db.isSeparate or SCA.temp.errors[db.dbName]) then
							self:RelocateTutorialPanel(db,'numFiles')
						end
					end
				else
					if (tutorial.flags.numFiles) then
						self:DestroyTutorialPanel(db,'numFiles')
					end
					
					self:RelocateTutorialPanel(db,'numFiles')
				end
			else
				if (tutorial.ids.customType) then
					self:DestroyTutorialPanel(db,'customType')
				end
				
				if (tutorial.ids.audioLocationFile) then
					self:DestroyTutorialPanel(db,'audioLocationFile')
				end
				
				if (tutorial.ids.audioLocationPack) then
					self:DestroyTutorialPanel(db,'audioLocationPack')
				end
				
				if (tutorial.ids.numFiles) then
					self:DestroyTutorialPanel(db,'numFiles')
				end
			end
			
			for k,v in pairs(tutorial.ids) do
				if (tutorial.ids[k]) then
					NPE_TutorialPointerFrame.InUseFrames[tutorial.ids[k]].Content:SetScript("OnMouseDown",function(self,button)
						if (button == "RightButton") then
							self:DestroyTutorialPanel(db,k)
						end
						
						if (tutorial.frameCount == 0) then
							tutorial.enabled = false
						end
					end)
				end
			end
		end
	else
		for k,_ in pairs(tutorial.ids) do
			self:DestroyTutorialPanel(db,k)
		end
	end
end

-- Either removes the file extension, if input by the user,
-- and adds a backslash, if missing, to the path if a pack type is selected.
function Ace3:VerifyCustomPath(value,soundInfo)
	if (find(value,".mp3")) then
		value = gsub(value,".mp3",'')
	elseif (find(value,".ogg")) then
		value = gsub(value,".ogg",'')
	end
	
	if (value ~= "Interface\\AUDIO\\") then
		if (sub(value,#value) ~= [[\]] and soundInfo.custom.customType == "pack" and #value > 0) then
			value = value..[[\]]
		elseif (sub(value,#value) == [[\]] and soundInfo.custom.customType == "file") then
			value = sub(value,0,#value - 1)
		end
	end
	
	return value
end