--[[---------------------------------------
	VARIABLE CACHING
-----------------------------------------]]

-- Lua Functions
local floor = math.floor
local pairs, select, type, unpack = pairs, select, type, unpack
local tinsert, wipe = table.insert, table.wipe
-- WoW API Functions / Variables
local GetAddOnMetadata = GetAddOnMetadata
local GetSpecialization = GetSpecialization
local InterfaceOptionsFrame = InterfaceOptionsFrame
local IsShiftKeyDown = IsShiftKeyDown
local UnitClass = UnitClass

local SCA, Ace3, LSM = unpack(select(2,...))
Ace3.version = GetAddOnMetadata(..., 'Version')
_G["SCA_Ace3"] = Ace3
--[[---------------------------------------
	CONSTANTS
-----------------------------------------]]
local SOUND_SOURCE_OPTIONS = {
	["packs"] = "SCA Audio Packs",
	["shared"] = "SharedMedia",
	["custom"] = "Custom Sounds",
}

local SOUND_CHANNEL_OPTIONS = {
	["Master"] = MASTER,
	["SFX"] = SOUND,
	["Music"] = MUSIC_VOLUME,
	["Ambience"] = AMBIENCE_VOLUME,
	["Dialog"] = DIALOG_VOLUME,
}

local SOUND_CUSTOM_TYPES = {
	["file"] = "Single Audio File",
	["pack"] = "Custom Audio Pack",
}

local SOUND_CRIT_TRIGGERS = {
	['crit'] = "Crits Only",
	['nocrit'] = "Non-crits Only",
	['always'] = "Always On",
}

local toggle_options

local function GetToggleOptions()
	SCA.temp.selectedMenuItem = 1
	if (not toggle_options) then
		--local db = Ace3.db.profile
		local db = SCA.profile
		
		local damage = db.audio.spells.damage
		local heals = db.audio.spells.heals
		local specific = db.audio.spells.specific
		
		local defense = db.audio.defense
		local combat = db.audio.combat
		local looting = db.audio.looting
		
		local chat = db.audio.chat
		local move = db.audio.move
		
		toggle_options = {
			type = "group",
			order = 1,
			name = "",
			desc = '',
			guiInline = true,
			args = {
				DamageSpells = {
					name = "Damage Spells",
					order = 1,
					type = "group",
					args = Ace3:GroupBuilder(damage,"normal"),
				},
				HealingSpells = {
					name = "Healing Spells",
					order = 2,
					type = "group",
					args = Ace3:GroupBuilder(heals,"normal"),
				},
				SpecificSpells = {
					name = "Specific Spells",
					order = 3,
					type = "group",
					args = Ace3:GroupBuilder(specific,nil,true),
				},
				Defense = {
					name = "Defense Sounds",
					order = 4,
					type = "group",
					args = Ace3:GroupBuilder(defense,"half"),
				},
				Combat = {
					name = "Combat Sounds",
					order = 5,
					type = "group",
					args = Ace3:SubGroupBuilder(combat),
				},
				Looting = {
					name = "Looting Sounds",
					order = 5,
					type = "group",
					args = Ace3:SubGroupBuilder(looting),
				},
				Chat = {
					name = "Chat Sounds",
					order = 6,
					type = "group",
					args = Ace3:SubGroupBuilder(chat),
				},
				Movement = {
					name = "Movement Sounds",
					order = 6,
					type = "group",
					args = Ace3:SubGroupBuilder(move),
				},
			},
		}
	end
	
	return toggle_options
end

local function SubmitSpecificSound(newSound,db,event)
	local _,spellID,spellName = Ace3:CheckSpell('onSuccess',newSound.validSpell)
	local isOverload = 0
	local _,class = UnitClass("player")
	local spec = GetSpecialization()
	local spells = SCA.data.spells
	local specific = db.audio.spells.specific

	if (spells.overload[class]) then
		for k,v in pairs(spells.overload[class][spec]) do
			if (spellName ~= v and strfind(v,spellName)) then
				isOverload = k
				break
			end
		end
	end
	
	if (db.audio.spells.ignored[spellID]) then
		db.audio.spells.ignored[spellID] = nil
	end
	
	if (not specific[spellID]) then
		local events = {}
		
		for k,v in pairs(newSound.events) do
			if (Ace3:CheckSpell(k,newSound.validSpell)) then
				events[k] = v
			end
		end
		
		specific[spellID] = {
			isSeparate = newSound.isSeparate,
			selected = newSound.selected,
			keys = {
				[0] = "all",
				[1] = "onCast",
				[2] = "onSuccess",
				[3] = "onFail",
				[4] = "onDamage",
				[5] = "onHeal",
			},
			events = events,			
		}
	end
	
	specific[spellID][event] = {}
	specific[spellID][event].validAudio = newSound[event].validAudio
	specific[spellID][event].options = {}
	specific[spellID][event].options.source = newSound[event].options.source
	specific[spellID][event].options.channel = newSound[event].options.channel

	if (event == "onDamage" or event == "onHeal" or event == "all") then
		specific[spellID][event].options.trigger = newSound[event].options.trigger
		specific[spellID][event].options.threshold = newSound[event].options.threshold
	end
	
	if (newSound[event].options.source == "packs") then
		specific[spellID][event].packs = {}
		specific[spellID][event].packs.value = newSound[event].packs.value
		specific[spellID][event].packs.text = newSound[event].packs.text
		specific[spellID][event].packs.path = newSound[event].packs.path
		specific[spellID][event].packs.max = newSound[event].packs.max
		specific[spellID][event].packs.isRandom = {}
		specific[spellID][event].packs.isRandom.enabled = newSound[event].packs.isRandom.enabled
		specific[spellID][event].packs.isRandom.currentFile = newSound[event].packs.isRandom.currentFile
	elseif (newSound[event].options.source == "shared") then
		specific[spellID][event].shared = {}
		specific[spellID][event].shared.value = newSound[event].shared.value
	elseif (newSound[event].options.source == "custom") then
		specific[spellID][event].custom = {}
		specific[spellID][event].custom.path = newSound[event].custom.path
		specific[spellID][event].custom.max = newSound[event].custom.max
		specific[spellID][event].custom.customType = newSound[event].custom.customType
		specific[spellID][event].custom.isRandom = {}
		specific[spellID][event].custom.isRandom.enabled = newSound[event].custom.isRandom.enabled
		specific[spellID][event].custom.isRandom.currentFile = newSound[event].custom.isRandom.currentFile
	end
	
	if (isOverload > 0) then
		if (not specific[isOverload]) then
			specific[isOverload] = {
				isSeparate = newSound.isSeparate
			}
		end
		
		specific[isOverload][event] = {}
		specific[isOverload][event].options = {}
		specific[isOverload][event].options.source = newSound[event].options.source
		specific[isOverload][event].options.channel = newSound[event].options.channel

		if (event == "onDamage" or event == "onHeal" or event == "all") then
			specific[isOverload][event].options.trigger = newSound[event].options.trigger
			specific[isOverload][event].options.threshold = newSound[event].options.threshold
		end
		
		if (newSound[event].options.source == "packs") then
			specific[isOverload][event].packs = {}
			specific[isOverload][event].packs.value = newSound[event].packs.value
			specific[isOverload][event].packs.text = newSound[event].packs.text
			specific[isOverload][event].packs.path = newSound[event].packs.path
			specific[isOverload][event].packs.max = newSound[event].packs.max
			specific[isOverload][event].packs.isRandom = {}
			specific[isOverload][event].packs.isRandom.enabled = newSound[event].packs.isRandom.enabled
			specific[isOverload][event].packs.isRandom.currentFile = newSound[event].packs.isRandom.currentFile
		elseif (newSound[event].options.source == "shared") then
			specific[isOverload][event].shared = {}
			specific[isOverload][event].shared.value = newSound[event].shared.value
		elseif (newSound[event].options.source == "custom") then
			specific[isOverload][event].custom = {}
			specific[isOverload][event].custom.path = newSound[event].custom.path
			specific[isOverload][event].custom.max = newSound[event].custom.max
			specific[isOverload][event].custom.customType = newSound[event].custom.customType
			specific[isOverload][event].custom.isRandom = {}
			specific[isOverload][event].custom.isRandom.enabled = newSound[event].custom.isRandom.enabled
			specific[isOverload][event].custom.isRandom.currentFile = newSound[event].custom.isRandom.currentFile
		end
	end
end

local spell_options

local function GetSpellOptions()
	SCA.temp.selectedMenuItem = 2
	if (not spell_options) then
		--local db = Ace3.db.profile
		local db = SCA.profile
		
		local damage = db.audio.spells.damage
		local heals = db.audio.spells.heals
		local specific = db.audio.spells.specific
		local ignored = db.audio.spells.ignored
		
		local newSound = SCA.temp.newSound
		local newIgnore = SCA.temp.newIgnore
		
		spell_options = {
			type = "group",
			childGroups = "tab",
			order = 1,
			name = "",
			desc = '',
			args = {
				DamageTab = Ace3:SpellTabBuilder(damage,"Damage","onDamage"),
				HealingTab = Ace3:SpellTabBuilder(heals,"Healing","onHeal"),
				SpecificTab = {
					name = "Specific Spells",
					order = 3,
					type = "group",
					args = {
						disabledError = Ace3:DisabledError(specific,0),
						AddCancelSound = {
							order = function()
								if (SCA.temp.addSound) then
									return 9
								else
									return 1
								end
							end,
							type = "execute",
							name = function(self)
								if (SCA.temp.addSound) then
									self.options.args.SpecificTab.args.AddCancelSound.width = "half"
								else
									self.options.args.SpecificTab.args.AddCancelSound.width = "normal"
								end
								
								if (SCA.temp.addSound) then
									return "Cancel"
								else
									return "Add Sound"
								end
							end,
							func = function()
								if (SCA.temp.addSound) then
									if (SCA.SpellFrame:IsShown()) then
										SCA.SpellFrame:Hide()
									end
									Ace3:ResetDB(newSound,true,true)
								else
									SCA.temp.addSound = true
								end
								GameTooltip:Hide()
							end,
							width = "normal",
						},
						SpellConfig = {
							name = "Spell Config",
							hidden = function()
								return not SCA.temp.addSound
							end,
							order = 2,
							type = "group",
							guiInline = true,
							args = {
								spellInfo = {
									type = "input",
									order = 1,				
									name = "Spell ID / Name",
									desc = "Input spell id, or name, to be assigned a specific sound.\n\nSpell names and IDs that you input are checked against a pre-built table of spell ID and names. If there is a spell that is missing that you feel should be added, please message me via the addon page.\n\nClicking the \"Spells\" button will display a list of spells that currently exist for your current class/spec.",
									disabled = function()
										if (SCA.temp.editSound) then
											return true
										else
											return false
										end
									end,
									get = function(self)
										return newSound.validSpell
									end,
									set = function(self,value)
										local errors = self.options.args.SpecificTab.args.errors.args.messages
										
										Ace3:CheckSpecificSpell(newSound,errors,Ace3:CheckSpell('onSuccess',value))
										
										Ace3:UpdateTutorial(newSound)
										
										-- If no event is selected ("empty"), we'll need to refresh the config's contents.
										if (newSound.selected == "empty") then
											Ace3:NavigateInterfaceOptions(SCA.temp.selectedMenuItem,true)
										end
									end,
								},
								spells = {
									order = 2,
									type = "execute",
									name = "Spells",
									desc = "List of spells, and spell IDs, that pertain to your class and spec.\n\n|cFFFF9B00Orange Spells|r are spells that are currently on the ignore list.\n\nA |cFF00FF00Green Spell|r is the spell that is currently selected.\n\n|cFF999999Grey Spells|r are spells that already have sounds assigned to them.",
									disabled = function()
										if (SCA.temp.editSound) then
											return true
										else
											return false
										end
									end,
									func = function()
										local _,class = UnitClass("player")
										local spec = GetSpecialization()
										if (SCA.SpellFrame:IsShown()) then
											SCA.SpellFrame:Hide()
										else
											local spells = SCA.data.spells
											for k,_ in pairs(spells.spellNames[class][spec]) do
												if (SCA.SpellFrame[k]) then
													if (specific[spells.spellNames[class][spec][k]]) then
														SCA.SpellFrame[k]:Disable()
														SCA.SpellFrame[k]:SetDisabledFontObject(SCA.SpellFrame.dFont)
													elseif (ignored[spells.spellNames[class][spec][k]]) then
														if (not SCA.SpellFrame[k]:IsEnabled()) then
															SCA.SpellFrame[k]:Enable()
														end
														SCA.SpellFrame[k]:SetNormalFontObject(SCA.SpellFrame.cFont)
														SCA.SpellFrame[k]:SetHighlightFontObject(SCA.SpellFrame.CFont)
													elseif (k == newSound.validSpell) then
														if (not SCA.SpellFrame[newSound.validSpell]:IsEnabled()) then
															SCA.SpellFrame[newSound.validSpell]:Enable()
														end
														SCA.SpellFrame[newSound.validSpell]:SetNormalFontObject(SCA.SpellFrame.sFont)
														SCA.SpellFrame[newSound.validSpell]:SetHighlightFontObject(SCA.SpellFrame.sFont)
													else
														if (not SCA.SpellFrame[k]:IsEnabled()) then
															SCA.SpellFrame[k]:Enable()
														end
														SCA.SpellFrame[k]:SetNormalFontObject(SCA.SpellFrame.nFont)
														SCA.SpellFrame[k]:SetHighlightFontObject(SCA.SpellFrame.hFont)
													end
												end
											end
											SCA.SpellFrame:Show()
										end
									end,
									width = "half",
								},
								spacing1 = Ace3:Spacer(3,"half"),
								tutorial = Ace3:TutorialToggle(newSound,4,"normal"),
								onCast = Ace3:EventToggle(newSound,'onCast',5,newSound.errorName,"half","Start","Toggle whether or not a sound will play when "..newSound.onCast.desc,"SpecificTab",true),
								onSuccess = Ace3:EventToggle(newSound,'onSuccess',6,newSound.errorName,"half","Success","Toggle whether or not a sound will play when "..newSound.onSuccess.desc,"SpecificTab",true),
								onFail = Ace3:EventToggle(newSound,'onFail',7,newSound.errorName,"half","Failed","Toggle whether or not a sound will play when "..newSound.onFail.desc,"SpecificTab",true),
								onDamage = Ace3:EventToggle(newSound,'onDamage',9,newSound.errorName,"half","Damage","Toggle whether or not a sound will play when "..newSound.onDamage.desc,"SpecificTab",true),
								onHeal = Ace3:EventToggle(newSound,'onHeal',10,newSound.errorName,"half","Heal","Toggle whether or not a sound will play when "..newSound.onHeal.desc,"SpecificTab",true),
							},
						},
						soundList = Ace3:SoundList(newSound,3,true),
						errors = Ace3:Errors(newSound.errorName,4,true),
						SoundConfig = {
							name = "Audio Config",
							disabled = function()
								if (newSound.validSpell and newSound.numEvents > 0) then
									return false
								else
									return true
								end
							end,
							hidden = function()
								return not SCA.temp.addSound
							end,
							disabled = function()
								if (newSound.numEvents == 0) then
									return true
								else
									return false
								end
							end,
							order = 5,
							type = "group",
							guiInline = true,
							args = {
								source = Ace3:Source(newSound,1,newSound.errorName,SOUND_SOURCE_OPTIONS,true),
								channel = Ace3:Channel(newSound,2,SOUND_CHANNEL_OPTIONS,true),
								volume = Ace3:Volume(newSound,3),
								--[[-----------------------------
									SCA Sound Packs
								-------------------------------]]
								packSelect = Ace3:PackSelect(newSound,4,"SpecificTab",true),
								isSCARandom = Ace3:IsSCARandom(newSound,7,true),
								--[[-----------------------------
									SharedMedia
								-------------------------------]]
								sharedMedia = Ace3:SharedMedia(newSound,4,"SpecificTab",true),
								--[[-----------------------------
									Custom Audio
								-------------------------------]]
								audioSource = Ace3:AudioSource(newSound,4,SOUND_CUSTOM_TYPES,"SpecificTab",true),
								soundEvents = {
									order = 5,
									type = "select",
									name = "Spell Events",
									desc = "Choose whether your custom audio is a file or sound pack",
									disabled = function()
										if (newSound.isSeparate and newSound.validSpell and newSound.numEvents > 1) then
											return false
										else
											return true
										end
									end,
									get = function()
										return newSound.selected
									end,
									set = function(self,value)
										newSound.selected = value
										
										newSound.tutorial.flags.spellEvent = true
										
										Ace3:UpdateTutorial(newSound)
									end,
									values = function()
										local args,sortTable = {},{}
										
										sortTable = Ace3:GetRefreshedTable(newSound,true)

										for i=1,#sortTable do
											if (sortTable[i] ~= "empty") then
												args[i] = sortTable[i]
											end
										end
										
										if (newSound.numEvents == 0) then
											args[0] = "No spell events"
										elseif (Ace3:GetSelected(newSound) == "all") then
											args[0] = "Select a spell event"
										end
										
										return args
									end
								},
								toggleSeparate = Ace3:ToggleSeparate(newSound,6,'SpecificTab',true),
								audioLocation = Ace3:AudioLocation(newSound,7,true),
								numFiles = Ace3:NumFiles(newSound,8,newSound.errorName,'SpecificTab',true),
								test = Ace3:TestSound(newSound,9,newSound.errorName,'SpecificTab',true),
								isCustomRandom = Ace3:IsCustomRandom(newSound,10),
								reset = Ace3:SoftReset(newSound,11,'SpecificTab',true),
							},
						},
						DamageTriggerHeader = {
							order = 6,
							type = 'group',
							name = 'Damage Triggers',
							guiInline = true,
							hidden = function()
								if (SCA.temp.addSound and newSound.events.onDamage and Ace3:CheckSpell('onDamage',newSound.validSpell)) then
									return false
								else
									return true
								end
							end,
							args = {
								threshold = Ace3:Threshold(newSound,Ace3:GetEvent(newSound,'onDamage'),'Damage',newSound.errorName,'SpecificTab'),
								critSelect = Ace3:CritSelect(newSound,Ace3:GetEvent(newSound,'onDamage'),SOUND_CRIT_TRIGGERS),
							},
						},
						HealingTriggerHeader = {
							order = 7,
							type = 'group',
							name = 'Healing Triggers',
							guiInline = true,
							hidden = function()
								if (SCA.temp.addSound and newSound.events.onHeal and Ace3:CheckSpell('onHeal',newSound.validSpell)) then
									return false
								else
									return true
								end
							end,
							args = {
								threshold = Ace3:Threshold(newSound,Ace3:GetEvent(newSound,'onHeal'),'Healing',newSound.errorName,'SpecificTab'),
								critSelect = Ace3:CritSelect(newSound,Ace3:GetEvent(newSound,'onHeal'),SOUND_CRIT_TRIGGERS),
							},
						},
						submit = {
							order = 8,
							type = "execute",
							name = function()
								if (SCA.temp.editSound) then
									return "Edit Sound"
								else
									return "Add Sound"
								end
							end,
							disabled = function()
								if (newSound.isSeparate) then
									local numValidEvents = 0
									
									for k,v in pairs(newSound.events) do
										if (v and newSound[k].validAudio and Ace3:CheckSpell(k,newSound.validSpell)) then
											numValidEvents = numValidEvents + 1
										end
									end
									
									if (numValidEvents == newSound.numEvents) then
										return false
									else
										return true
									end
								else
									if (newSound.all.validAudio and Ace3:CheckSpell('onSuccess',newSound.validSpell)) then
										return false
									else
										return true
									end
								end
							end,
							hidden = function()
								return not SCA.temp.addSound
							end,
							func = function(self)
								if (not newSound.isSeparate) then
									SubmitSpecificSound(newSound,db,'all')
								else
									for k,v in pairs(newSound.events) do
										if (v) then
											if ((k ~= "onCast" and k ~= "onSuccess" and k ~= "onFail" and Ace3:CheckSpell(k,newSound.validSpell)) or (k == "onCast" and Ace3:CheckSpell('onCast',newSound.validSpell)) or (k == "onSuccess" and Ace3:CheckSpell('onSuccess',newSound.validSpell)) or (k == "onFail" and Ace3:CheckSpell('onFail',newSound.validSpell))) then
												SubmitSpecificSound(newSound,db,k)
											end
										end
									end
								end
								
								if (SCA.temp.errors.newSound) then
									SCA.temp.errors.newSound = nil
								end
								
								Ace3:ResetDB(newSound,true,true)
								
								self.options.args.SpecificTab.args.soundList.args.messages.name = ' '
								Ace3:ReloadSpellIgnoreList(GetSpellOptions())
								Ace3:ReloadSpellAssignList(GetSpellOptions(),db.audio.spells)
								
								-- The tooltip for this button lingers because the button disappears. This forces to hide it.
								GameTooltip:Hide()
							end,
						},
						spacing = Ace3:Spacer(10,"half"),
						reset = Ace3:HardReset(newSound,11,"SpecificTab",true),
						assignedSpells = {
							name = "Assigned Spells",
							order = 1,
							desc = "TESTING",
							hidden = function()
								return SCA.temp.addSound
							end,
							type = "group",
							guiInline = true,
							args = {
							},
						},
					},
				},
				ignoreTab = {
					name = IGNORE,
					order = 4,
					type = "group",
					args = {
						input = {
							name = "Spell Input",
							order = 0,
							type = "group",
							desc = "Click on a checkbox to remove a spell from the ignore list.",
							guiInline = true,
							args = {
								spellInfo = {
									type = "input",
									order = 1,				
									name = "Spell ID / Name",
									desc = "Input spell id, or name, to be ignored by the addon.\n\nSpell names and IDs that you input are checked against a pre-built table of spell ID and names. If there is a spell that is missing that you feel should be added, please message me via the addon page.",
									get = function()
											return newIgnore.validSpell
									end,
									set = function(self,value)
										local spellID,name
										local errors = self.options.args.ignoreTab.args.errors.args.messages
										
										Ace3:CheckIgnoreSpell(newIgnore,errors,Ace3:CheckSpell('onSuccess',value))
									end,
								},
								spells = {
									order = 2,
									type = "execute",
									name = "Spells",
									desc = "List of spells, and spell IDs, that pertain to your class and spec.\n\n|cFFFF9B00Orange Spells|r are spells that have sounds assigned to them.\n\nA |cFF00FF00Green Spell|r is the spell that is currently selected.\n\n|cFF999999Grey Spells|r are spells that are already being ignored.",
									func = function()
										local _,class = UnitClass("player")
										local spec = GetSpecialization()
										if (SCA.SpellFrame:IsShown()) then
											SCA.SpellFrame:Hide()
										else
											local spells = SCA.data.spells
											for k,_ in pairs(spells.spellNames[class][spec]) do
												if (SCA.SpellFrame[k]) then
													if (ignored[spells.spellNames[class][spec][k]]) then
														SCA.SpellFrame[k]:Disable()
														SCA.SpellFrame[k]:SetDisabledFontObject(SCA.SpellFrame.dFont)
													elseif (specific[spells.spellNames[class][spec][k]]) then
														if (not SCA.SpellFrame[k]:IsEnabled()) then
															SCA.SpellFrame[k]:Enable()
														end
														SCA.SpellFrame[k]:SetNormalFontObject(SCA.SpellFrame.cFont)
														SCA.SpellFrame[k]:SetHighlightFontObject(SCA.SpellFrame.CFont)
													elseif (k == SCA.temp.newIgnore.validSpell) then
														if (not SCA.SpellFrame[SCA.temp.newIgnore.validSpell]:IsEnabled()) then
															SCA.SpellFrame[SCA.temp.newIgnore.validSpell]:Enable()
														end
														SCA.SpellFrame[SCA.temp.newIgnore.validSpell]:SetNormalFontObject(SCA.SpellFrame.sFont)
														SCA.SpellFrame[SCA.temp.newIgnore.validSpell]:SetHighlightFontObject(SCA.SpellFrame.sFont)
													else
														if (not SCA.SpellFrame[k]:IsEnabled()) then
															SCA.SpellFrame[k]:Enable()
														end
														SCA.SpellFrame[k]:SetNormalFontObject(SCA.SpellFrame.nFont)
														SCA.SpellFrame[k]:SetHighlightFontObject(SCA.SpellFrame.hFont)
													end
												end
											end
											SCA.SpellFrame:Show()
										end
									end,
									width = "half",
								},
								submit = {
									order = 3,
									type = "execute",
									name = "Ignore",
									disabled = function()
										if (#newIgnore.validSpell > 0) then
											return false
										else
											return true
										end
									end,
									func = function(self)
										local _,spellID = Ace3:CheckSpell('onSuccess',newIgnore.validSpell)

										ignored[spellID] = name
										if (specific[spellID]) then
											specific[spellID] = nil
											Ace3:ReloadSpellAssignList(GetSpellOptions(),db.audio.spells)
										end
										
										self.options.args.ignoreTab.args.errors.args.messages.name = " "
										newIgnore.isValidSpell = false
										newIgnore.validSpell = ''
										ignored[spellID] = true

										if (SCA.temp.errors.ignored) then
											SCA.temp.errors.ignored = nil
										end
								
										Ace3:ReloadSpellIgnoreList(GetSpellOptions())
									end,
								},
							},
						},
						errors = Ace3:Errors(newIgnore.errorName,4),
						ignoreList = {
							name = "Ignored Spells",
							order = 5,
							type = "group",
							dialogInline = true,
							args = {
							},
						},
					},
				},
			},
		}
	end
	
	return spell_options
end

local defense_options

local function GetDefenseOptions()
	SCA.temp.selectedMenuItem = 3
	
	if (SCA.SpellFrame and SCA.SpellFrame:IsShown()) then
		SCA.SpellFrame:Hide()
	end
	
	if (not defense_options) then
		--local db = Ace3.db.profile
		local db = SCA.profile
		local defense = db.audio.defense
		
		defense_options = {
			type = "group",
			childGroups = "tab",
			order = 1,
			name = "",
			desc = '',
			args = {
				disabledError = Ace3:DisabledError(defense,0),
				options = {
					name = "Toggle Options",
					order = 1,
					type = "group",
					guiInline = true,
					args = Ace3:BuildToggleCheckboxes(defense,nil,true),
				},
				soundList = Ace3:SoundList(defense,2),
				errors = Ace3:Errors(defense.info.errorName,3),
				SoundConfig = {
					name = "Audio Config",
					order = 4,
					type = "group",
					guiInline = true,
					disabled = function()
					if (defense.numEvents == 0) then
							return true
						else
							return false
						end
					end,
					args = {
						source = Ace3:Source(defense,1,defense.info.errorName,SOUND_SOURCE_OPTIONS,nil,true),
						channel = Ace3:Channel(defense,2,SOUND_CHANNEL_OPTIONS),
						volume = Ace3:Volume(defense,3),
						--[[-----------------------------
							SCA Sound Packs
						-------------------------------]]
						packSelect = Ace3:PackSelect(defense,4),
						isSCARandom = Ace3:IsSCARandom(defense,7),
						--[[-----------------------------
							SharedMedia
						-------------------------------]]
						sharedMedia = Ace3:SharedMedia(defense,4),
						--[[-----------------------------
							Custom Audio
						-------------------------------]]
						audioSource = Ace3:AudioSource(defense,4,SOUND_CUSTOM_TYPES,nil,nil,true),
						events = {
							order = 5,
							type = "select",
							name = "Defense Event",
							desc = "Choose which event you want to edit",
							disabled = function()
								if (defense.isSeparate) then
									return false
								else
									return true
								end
							end,
							get = function()
								return defense.selected
							end,
							set = function(self,value)
								defense.selected = value
							end,
							values = function()
								local args,sortTable = {},{}

								sortTable = Ace3:GetRefreshedTable(defense)
								
								for i=1,#sortTable do
									if (sortTable[i] ~= "empty") then
										args[i] = defense[sortTable[i]].label
									end
								end
								
								if (defense.numEvents == 0) then
									args[0] = "No defense events"
								elseif (Ace3:GetSelected(defense) == "all") then
									args[0] = 'Select a defense event'
								end
									
								wipe(sortTable)

								return args
							end,
						},
						toggleSeparate = Ace3:ToggleSeparate(defense,6,nil,nil,true,true),
						audioLocation = Ace3:AudioLocation(defense,7,nil,true),
						numFiles = Ace3:NumFiles(defense,8,defense.info.errorName,nil,nil,true),
						test = Ace3:TestSound(defense,9,defense.info.errorName,nil,nil,true),
						isCustomRandom = Ace3:IsCustomRandom(defense,10),
						reset = Ace3:SoftReset(defense,11,defense.info.tabName),
					},
				},
				reset = Ace3:HardReset(defense,5,defense.info.tabName),
			},
		}
		
	end
	
	return defense_options
end

local combat_options

local function GetCombatOptions()
	SCA.temp.selectedMenuItem = 4
	
	if (SCA.SpellFrame and SCA.SpellFrame:IsShown()) then
		SCA.SpellFrame:Hide()
	end
	
	if (not combat_options) then
		--local db = Ace3.db.profile
		local db = SCA.profile
		local combat = db.audio.combat
		
		combat_options = {
			type = "group",
			childGroups = "tab",
			order = 1,
			name = "",
			desc = '',
			args = Ace3:CombatTabBuilder(),
		}
		
	end
	
	return combat_options
end

local loot_options

local function GetLootOptions()
	SCA.temp.selectedMenuItem = 5
	if (not loot_options) then
		--local db = Ace3.db.profile
		local db = SCA.profile
		
		local damage = db.audio.spells.damage
		local heals = db.audio.spells.heals
		local specific = db.audio.spells.specific
		local ignored = db.audio.spells.ignored
		
		local newSound = SCA.temp.newSound
		local newIgnore = SCA.temp.newIgnore
		
		loot_options = {
			type = "group",
			childGroups = "tab",
			order = 1,
			name = "",
			desc = '',
			args = Ace3:LootTabBuilder(),
		}
	end
	
	return loot_options
end

local chat_options

local function GetChatOptions()
	SCA.temp.selectedMenuItem = 6
	
	if (SCA.SpellFrame and SCA.SpellFrame:IsShown()) then
		SCA.SpellFrame:Hide()
	end
	
	if (not chat_options) then
		chat_options = {
			type = "group",
			childGroups = "tab",
			order = 1,
			name = "",
			desc = '',
			args = Ace3:ChatTabBuilder(),
		}
	end
	
	return chat_options
end

local move_options

local function GetMoveOptions()
	SCA.temp.selectedMenuItem = 7
	
	if (SCA.SpellFrame and SCA.SpellFrame:IsShown()) then
		SCA.SpellFrame:Hide()
	end
	
	if (not move_options) then
		move_options = {
			type = "group",
			childGroups = "tab",
			order = 1,
			name = "",
			desc = '',
			args = Ace3:MoveTabBuilder(),
		}
	end
	
	return move_options
end

local preserve_options

local function GetPreserveOptions()
	SCA.temp.selectedMenuItem = 8
	
	if (SCA.SpellFrame and SCA.SpellFrame:IsShown()) then
		SCA.SpellFrame:Hide()
	end
	
	--local db = Ace3.db.profile
	local db = SCA.profile
	
	if (not preserve_options) then
		preserve_options = {
			type = "group",
			order = 1,
			childGroups = "tab",
			name = "",
			desc = '',
			args = {
				backup_options = {
					type = "group",
					order = 2,
					name = "Backup",
					desc = '',
					args = {
						button = {
							type = "execute",
							order = 1,
							name = "Retrieve Sound Data",
							desc = "Build a string that represents the database of your current sound configuration",
							disabled = SCA.temp.backup.isRetrieved,
							func = function(this)
								SCA.temp.backup.isRetrieved = true
								SCA.temp.backup.data = Ace3:BackupRestore(db.audio.keys[SCA.temp.backup.selected])
							end,
						},
						selected = {
							type = "select",
							order = 2,
							name = "Backup Select",
							desc = "Select which data you want to backup",
							get = function()
								return SCA.temp.backup.selected
							end,
							set = function(_,value)
								SCA.temp.backup.selected = value
							end,
							values = {
								[0] = "All Data",
								[1] = "Spell Sounds",
								[2] = "Defense Sounds",
								[3] = "Combat Sounds",
								[4] = "Loot Sounds",
								[5] = "Chat Sounds",
								[6] = "Movement Sounds",
							},
							--[[values = function()
								local args,sortTable = {},{}
									
								sortTable = Ace3:GetRefreshedTable(db.audio,nil,true)
										
								for i=1,#sortTable do
									args[i] = db.audio[sortTable[i].label
								end

								args[0] = "All Data"
								
								return args
								
							end,]]
						},
						text = {
							type = "input",
							name = "Backup String",
							desc = '',
							order = 3,
							multiline = 28,
							get = function()
								return string.format([[%s]],SCA.temp.backup.data)
							end,
							set = function(_,value)
								--SCA.temp.backup = value
							end,
							width = "full",
						},
					},
				},
				restore_options = {
					type = "group",
					order = 2,
					name = "Restore",
					desc = '',
					args = {
						sumbit = {
							type = "execute",
							order = 1,
							name = "Restore Sound Data",
							desc = "Overwrites your current SavedVariables data from the backup string provided.\n\n|cFFFF4025Restoring Data will reload your UI|r",
							disabled = function()
								if (#SCA.temp.restore.data > 0) then
									return false
								else
									return true
								end
							end,
							func = function(this)
								local errorMsg = Ace3:BackupRestore(SCA.temp.restore.data,true)
								
								if (errorMsg) then
									this.options.args.restore_options.args.errorMsg.name = "|cFFFF0000"..tostring(errorMsg).."|r"
								else
									SCA.temp.restore.data = ''
								end
							end,
							width = "normal",
						},
						errorMsg = {
							order = 2,
							type = "description",
							name = ' ',
							width = "normal",
						},
						spacer = {
							order = 3,
							type = "description",
							name = ' ',
							width = "half",
						},
						help = {
							type = "execute",
							order = 4,
							name = "Help",
							desc = "Helpful information about restoring data.",
							func = function(this)
								SCA.WarnFrame.text:SetText("Please be careful when using the restore feature! If the backup string you submit corrupts your SavedVariables data, you will have to reset it.\n\n1. Log out of your WoW character.\n2. Navigate to your SavedVariables folder.\n  • \"WTF\\Account\\<account>\\SavedVariables\\\n3. Delete \"CustomAudio.lua\" & \"CustomAudio.lua.bak\"\n4.Log back into WoW.")
								SCA.WarnFrame:SetHeight(175)
								SCA.WarnFrame:Show()
							end,
							width = "half",
						},
						text = {
							type = "input",
							name = "Restore String",
							desc = '',
							order = 5,
							multiline = 30,
							get = function()
								return SCA.temp.restore.data
							end,
							set = function(_,value)
								SCA.temp.restore.data = string.format([[%s]],value)
								SCA.temp.restore.isReady = true
							end,
							width = "full",
						},
					},
				}
			},
		}
	end
	
	return preserve_options
end

local WarnFrame = CreateFrame('Frame',nil,InterfaceOptionsFrame)
WarnFrame:SetWidth(405)
WarnFrame:SetHeight(300)
WarnFrame:SetPoint('CENTER',InterfaceOptionsFrame,'CENTER',0,0)
WarnFrame:SetBackdrop(SCA.BackdropSB)
WarnFrame:SetBackdropColor(0.15,0.15,0.15,0.9)
WarnFrame:SetBackdropBorderColor(1,1,1,1)
WarnFrame:SetFrameStrata("TOOLTIP")
WarnFrame:Hide()

WarnFrame.text = WarnFrame:CreateFontString(nil, 'MEDIUM', 'GameFontHighlightLarge')
WarnFrame.text:SetPoint('TOPLEFT',WarnFrame,'TOPLEFT',5,-5);
WarnFrame.text:SetPoint('TOPRIGHT',WarnFrame,'TOPRIGHT',-5,-5);
WarnFrame.text:SetPoint('BOTTOMLEFT',WarnFrame,'BOTTOMLEFT',5,5);
WarnFrame.text:SetPoint('BOTTOMRIGHT',WarnFrame,'BOTTOMRIGHT',-5,5);
WarnFrame.text:SetTextColor(1,1,1,1)
WarnFrame.text:SetJustifyH('LEFT')
WarnFrame.text:SetJustifyV('TOP')
WarnFrame.text:SetFont("Fonts\\FRIZQT__.TTF", 14)
--WarnFrame.text:SetAllPoints(WarnFrame)
--WarnFrame.text:SetParent(WarnFrame)

WarnFrame.close = CreateFrame("Button",nil,WarnFrame)
WarnFrame.close:SetWidth(150)
WarnFrame.close:SetHeight(50)
WarnFrame.close:SetText("Close")
WarnFrame.close:SetPoint("BOTTOM",WarnFrame,"BOTTOM",0,-7)
WarnFrame.close:SetNormalFontObject(GameFontNormal)
WarnFrame.close:SetHighlightFontObject(GameFontHighlight)
WarnFrame.close:Show()
WarnFrame.close:SetScript("OnClick",function(self,button)
	self:GetParent():Hide()
end)


SCA.WarnFrame = WarnFrame
_G['SCA_WarnFrame'] = WarnFrame

function Ace3:SetupOptions()
	local ACFG = LibStub("AceConfigRegistry-3.0")
	ACFG:RegisterOptionsTable("CustomAudio Quick Toggle", GetToggleOptions)
	ACFG:RegisterOptionsTable("CustomAudio Spell Sounds", GetSpellOptions)
	ACFG:RegisterOptionsTable("CustomAudio Defense Sounds", GetDefenseOptions)
	ACFG:RegisterOptionsTable("CustomAudio Combat Sounds", GetCombatOptions)
	ACFG:RegisterOptionsTable("CustomAudio Loot Sounds", GetLootOptions)
	ACFG:RegisterOptionsTable("CustomAudio Chat Sounds", GetChatOptions)
	ACFG:RegisterOptionsTable("CustomAudio Movement Sounds", GetMoveOptions)
	ACFG:RegisterOptionsTable("CustomAudio Backup & Restore", GetPreserveOptions)
	ACFG:RegisterOptionsTable("CustomAudio Profiles", LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db,true))

	local ACD = LibStub("AceConfigDialog-3.0")
	ACD:AddToBlizOptions("CustomAudio Quick Toggle", "Quick Toggle", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Spell Sounds", "Spell Sounds", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Defense Sounds", "Defense Sounds", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Combat Sounds", "Combat Sounds", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Loot Sounds", "Loot Sounds", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Chat Sounds", "Chat Sounds", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Movement Sounds", "Movement Sounds", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Backup & Restore", "Backup & Restore", "CustomAudio")
	ACD:AddToBlizOptions("CustomAudio Profiles", "Profiles", "CustomAudio")
	
	_G["SCA_ACD"] = ACD
end

function Ace3:UpdateInterfaceSettings()
	--local db = self.db.profile.audio
	local db = SCA.profile.audio

	self:InitOptions(db.spells.damage,GetSpellOptions().args.DamageTab)
	self:InitOptions(db.spells.heals,GetSpellOptions().args.HealingTab)
	self:InitOptions(db.defense,GetDefenseOptions(),true)
	
	for k,_ in pairs(db.combat) do
		if (type(db.combat[k]) == "table") then
			self:InitOptions(db.combat[k],GetCombatOptions().args[k.."Tab"],true)
		end
	end
	
	for k,_ in pairs(db.looting) do
		if (type(db.looting[k]) == "table") then
			self:InitOptions(db.looting[k],GetLootOptions().args[k.."Tab"],true)
		end
	end
	
	for k,_ in pairs(db.chat) do
		if (type(db.chat[k]) == "table") then
			self:InitOptions(db.chat[k],GetChatOptions().args[k.."Tab"],true)
		end
	end
	
	for k,_ in pairs(db.move) do
		if (type(db.move[k]) == "table") then
			self:InitOptions(db.move[k],GetMoveOptions().args[k.."Tab"],true)
		end
	end
	
	self:ReloadSpellIgnoreList(GetSpellOptions())
	self:ReloadSpellAssignList(GetSpellOptions(),db.spells)
	
	self:CreateSpellFrame(GetSpellOptions())
end

InterfaceOptionsFrame:HookScript("OnUpdate",function(self)
	-- The purpose here is to refresh the contents of the addon's config panel; simulating a click to the left navigation items will do this.
	-- This will only be done if the SHIFT key is down (making sure to check if it's down only once, via the "isShiftDown" variable).
	-- We also want to make sure the keyboard doesn't have focus, as SHIFT is commonly pressed while chatting.
	if (IsShiftKeyDown() and not SCA.temp.isShiftDown and not GetCurrentKeyBoardFocus()) then
		Ace3:NavigateInterfaceOptions(SCA.temp.selectedMenuItem,true)
		SCA.temp.isShiftDown = true
	elseif (not IsShiftKeyDown() and SCA.temp.isShiftDown) then
		Ace3:NavigateInterfaceOptions(SCA.temp.selectedMenuItem,true)
		SCA.temp.isShiftDown = false
	end
end)

InterfaceOptionsFrame:HookScript("OnShow",function(self)
	Ace3:UpdateInterfaceSettings()
end)

InterfaceOptionsFrame:HookScript("OnHide",function(self)
	if (SCA.SpellFrame:IsShown()) then
		SCA.SpellFrame:Hide()
	end
end)