--------------------------------------------------------------------------
-- SmartError.lua 
--------------------------------------------------------------------------
--[[
Author: Zensunim of Dragonblight

Change Log:
	v1.0 
		- Initial Release.
	v1.1
		- Added items/chests in use to the "Object is busy" alert
		- Added SPELL_FAILED_TARGETS_DEAD to Bad Target
	v1.2
		- Converted WAV sounds to OGG for Cataclysm/4.0 support
		- Spam sound detection (fixes volume issues)
		- Added sound test buttons
	v1.3
		- Updated sounds to play on the Master channel instead of the SFX channel
	v1.3.1
		- Updated TOC to 4.1
	v1.4
		- Added SPELL_FAILED_TOO_CLOSE to "Out of Range"
		- Renamed "Object is Busy" to "Can't Loot"
		- Added ERR_INV_FULL to "Can't Loot"
	v1.4.1
		- Updated TOC to 4.2
	v1.4.2
		- Updated TOC to 4.3
	v1.4.3
		- Updated TOC to 5.0.4
	v1.4.4
		- Updated TOC to 5.1
		- Added SPELL_FAILED_VISION_OBSCURED to "Line of Sight"
	v1.4.5
		- Updated TOC to 5.2
	v1.4.6
		- Updated TOC to 5.3
	v1.4.7
		- Updated TOC to 5.4
	v1.4.8
		- Updated TOC to 6.0
	v1.4.9
		- Updated TOC to 6.2
	v1.5
		- Added support for Legion
		- Updated TOC to 7.0
	v1.5.1
		- Updated TOC to 7.1
	v1.5.2
		- Updated TOC to 7.2
	v1.5.3
		- Updated TOC to 7.3
	v1.6
		- Added support for BFA

]]--

SmartError = {
	Version = "1.6";
	VersionNumber = 10600;
	DataCode = "1";
	DebugMode = nil;
	IgnoreTimeAmount = .2; -- Number of seconds between alert sounds
	IgnoreTime = nil;
	BetaMode = nil;
	Events = {
		"NoMana",
		"OutOfRange",
		"NoLoS",
		"Moving",
		"BadTarget",
		"InCombat",
		"NotFacing",
		"CrowdControlled",
		"ObjectBusy",
		"Resurrect",
		"Summon",
	};
}

SmartErrorData = { };

if (select(4, GetBuildInfo()) > 80000) then
	SmartError.BetaMode = true;
end


function SmartError_OnLoad()
    SmartErrorFrame:RegisterEvent("VARIABLES_LOADED");
end

function SmartError_ChatPrint(str)
	DEFAULT_CHAT_FRAME:AddMessage("[SmartError] "..str, 0.25, 1.0, 0.25);
end

function SmartError_ErrorPrint(str)
	DEFAULT_CHAT_FRAME:AddMessage("[SmartError] "..str, 1.0, 0.5, 0.5);
end

function SmartError_DebugPrint(str)
	if (SmartError.DebugMode) then
		DEFAULT_CHAT_FRAME:AddMessage("[SE] "..str, 0.75, 1.0, 0.25);
	end
end

function SmartError_OnEvent(self, event, ...)
	if (event == "VARIABLES_LOADED") then
		if (SmartErrorData.DataCode ~= SmartError.DataCode) then
			SmartError_ChatPrint(SmartError.Version..": "..SMARTERROR_NEWDATABASE);
			SmartError_FullReset();
		end
		SmartError_RenderOptions();
		SmartError_ActivateMod();
		if (SmartErrorData.Active) then
			SmartError_ChatPrint(SmartError.Version..": "..SMARTERROR_LOADED);
		else
			SmartError_ChatPrint(SmartError.Version..": "..SMARTERROR_LOADED.." (|cFFFF1111"..SMARTERROR_SUSPENDED.."|r)");
		end
		return;
	end
	if (event == "UI_ERROR_MESSAGE") then
		local messageId = 0;
		local message;
		messageId, message = ...;
		SmartError_DebugPrint(tostring(messageId)..": "..tostring(message));
		
		if (message == ERR_SPELL_COOLDOWN) then
			--Annoyingly spammy!  I'll have to find a different sound...
			--SmartError_PlaySound(2);
			--return;
		end
		if (message == SPELL_FAILED_SPELL_IN_PROGRESS) then
			--Annoyingly spammy!  I'll have to find a different sound...
			--SmartError_PlaySound(2);
			--return;
		end
		if (message == ERR_SPELL_OUT_OF_RANGE and SmartErrorData.Events["OutOfRange"] == true) then
			SmartError_PlaySound(2);
			return;
		end
		if (message == SPELL_FAILED_TOO_CLOSE and SmartErrorData.Events["OutOfRange"] == true) then
			SmartError_PlaySound(2);
			return;
		end
		if (message == SPELL_FAILED_MOVING and SmartErrorData.Events["Moving"] == true) then
			SmartError_PlaySound(2);
			return;
		end
		if (SmartErrorData.Events["NoLoS"] == true) then
			if (message == SPELL_FAILED_LINE_OF_SIGHT) then
				SmartError_PlaySound(2);
				return;
			end
			if (message == SPELL_FAILED_VISION_OBSCURED) then
				SmartError_PlaySound(2);
				return;
			end
		end
		if (SmartErrorData.Events["BadTarget"] == true) then
			if (message == SPELL_FAILED_BAD_TARGETS) then
				SmartError_PlaySound(8);
				return;
			end
			if (message == ERR_INVALID_ATTACK_TARGET) then
				SmartError_PlaySound(8);
				return;
			end
			if (message == SPELL_FAILED_TARGETS_DEAD) then
				SmartError_PlaySound(8);
				return;
			end
		end
		if (message == SPELL_FAILED_AFFECTING_COMBAT and SmartErrorData.Events["InCombat"] == true) then
			SmartError_PlaySound(2);
			return;
		end
		if (message == ERR_OUT_OF_MANA and SmartErrorData.Events["NoMana"] == true) then
			SmartError_PlaySound(1);
			return;
		end
		if (SmartErrorData.Events["ObjectBusy"] == true) then
			if (message == ERR_OBJECT_IS_BUSY) then
				SmartError_PlaySound(7);
				return;
			end
			if (message == SPELL_FAILED_CHEST_IN_USE) then
				SmartError_PlaySound(7);
				return;
			end
			if (message == ERR_CHEST_IN_USE) then
				SmartError_PlaySound(7);
				return;
			end
			if (message == ERR_INV_FULL) then
				SmartError_PlaySound(7);
				return;
			end
		end
		if (SmartErrorData.Events["NotFacing"] == true) then
			if (message == SPELL_FAILED_UNIT_NOT_INFRONT) then
				SmartError_PlaySound(3);
				return;
			end
			if (message == ERR_BADATTACKFACING) then
				SmartError_PlaySound(3);
				return;
			end
			if (message == SPELL_FAILED_NOT_BEHIND) then
				SmartError_PlaySound(3);
				return;
			end
			if (message == SPELL_FAILED_UNIT_NOT_BEHIND) then
				SmartError_PlaySound(3);
				return;
			end
		end
		if (SmartErrorData.Events["CrowdControlled"] == true) then
			if (message == SPELL_FAILED_SILENCED) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == SPELL_FAILED_STUNNED) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == ERR_ATTACK_PACIFIED) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == ERR_ATTACK_CHARMED) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == ERR_ATTACK_CONFUSED) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == ERR_ATTACK_FLEEING) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == ERR_ATTACK_PREVENTED_BY_MECHANIC_S) then
				SmartError_PlaySound(4);
				return;
			end
			if (message == "Can't attack while polymorphed.") then
				SmartError_PlaySound(4);
				return;
			end
		end
		return;
	end
	if (event == "RESURRECT_REQUEST") then
		if (SmartErrorData.Events["Resurrect"] == true) then
			SmartError_PlaySound(5);
			return;
		end
	end
	if (event == "CONFIRM_SUMMON") then
		if (SmartErrorData.Events["Summon"] == true) then
			SmartError_PlaySound(6);
			return;
		end
	end
end

function SmartError_PlaySound(iSound)
	local currentTime = GetTime();
	local soundTable = { };
	if (SmartError.IgnoreTime) then
		if (currentTime < SmartError.IgnoreTime) then
			return;
		end
	end
	SmartError.IgnoreTime = currentTime + SmartError.IgnoreTimeAmount;

	soundTable = {
		"Interface\\AddOns\\SmartError\\Sounds\\no_mana.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\no_los.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\facing.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\silence.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\resurrect.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\summon.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\objectbusy.ogg",
		"Interface\\AddOns\\SmartError\\Sounds\\badtarget.ogg",
	};
	PlaySoundFile(soundTable[iSound], "Master");
end

function SmartError_FullReset()
	SmartErrorData = { 
		DataCode = SmartError.DataCode;
		Active = true;
		Events = { };
	};
	SmartError_SetDefaults();
end

function SmartError_SetDefaults()
	for index, option in pairs(SmartError.Events) do
		SmartErrorData.Events[option] = true;
	end
end

function SmartError_RenderOptions()
	local ConfigurationPanel = CreateFrame("FRAME","SmartError_MainFrame");
	ConfigurationPanel.name = "SmartError";
	InterfaceOptions_AddCategory(ConfigurationPanel);

	local EnabledButton = CreateFrame("CheckButton", "SmartError_EnabledButton", ConfigurationPanel, "ChatConfigCheckButtonTemplate");
	EnabledButton:SetPoint("TOPLEFT", 10, -15);
	EnabledButton.tooltip = "Unchecking this box will disable all sounds.";
	EnabledButton:SetChecked(SmartErrorData.Active);
	getglobal(EnabledButton:GetName().."Text"):SetText("Enable Sounds");

	SmartError_CreateEventButton("NoMana", ConfigurationPanel, 30, -45, "Out of Mana", " Alert when you don't have enough mana to cast the spell");
	SmartError_CreateTestButton("NoMana", ConfigurationPanel, 300, -45, 1);
	SmartError_CreateEventButton("OutOfRange", ConfigurationPanel, 30, -75, "Out of Range", " Alert when you are out of range of your target (special abilities only)");
	SmartError_CreateTestButton("OutOfRange", ConfigurationPanel, 300, -75, 2);
	SmartError_CreateEventButton("NoLoS", ConfigurationPanel, 30, -105, "No Line of Sight", " Alert when the attack failed because the target was not in line of sight");
	SmartError_CreateTestButton("NoLoS", ConfigurationPanel, 300, -105, 2);
	SmartError_CreateEventButton("Moving", ConfigurationPanel, 30, -135, "Moving", " Alert when the attack failed because you were moving");
	SmartError_CreateTestButton("Moving", ConfigurationPanel, 300, -135, 2);
	SmartError_CreateEventButton("BadTarget", ConfigurationPanel, 30, -165, "Bad Target", " Alert that you're attacking an invalid target");
	SmartError_CreateTestButton("BadTarget", ConfigurationPanel, 300, -165, 8);
	SmartError_CreateEventButton("InCombat", ConfigurationPanel, 30, -195, "In Combat", " Alert when your ability failed because you were in combat");
	SmartError_CreateTestButton("InCombat", ConfigurationPanel, 300, -195, 2);
	SmartError_CreateEventButton("NotFacing", ConfigurationPanel, 30, -225, "Wrong Direction", " Alert when your ability failed because you were either not facing the target or you're facing the wrong side of the target");
	SmartError_CreateTestButton("NotFacing", ConfigurationPanel, 300, -225, 3);
	SmartError_CreateEventButton("CrowdControlled", ConfigurationPanel, 30, -255, "Crowd Controlled (Stunned/Silenced)", " Alert when your ability failed because you're crowd controlled (stunned, silenced, feared, etc.)");
	SmartError_CreateTestButton("CrowdControlled", ConfigurationPanel, 300, -255, 4);
	SmartError_CreateEventButton("ObjectBusy", ConfigurationPanel, 30, -285, "Can't Loot", " Alert when you fail to pick up an item because the object was busy or your inventory was full");
	SmartError_CreateTestButton("ObjectBusy", ConfigurationPanel, 300, -285, 7);
	SmartError_CreateEventButton("Resurrect", ConfigurationPanel, 30, -315, "Resurrected", " Alert when prompted to accept a resurrection");
	SmartError_CreateTestButton("Resurrect", ConfigurationPanel, 300, -315, 5);
	SmartError_CreateEventButton("Summon", ConfigurationPanel, 30, -345, "Summoned", " Alert when prompted to accept a summon");
	SmartError_CreateTestButton("Summon", ConfigurationPanel, 300, -345, 6);

	local ActivateAllButton = CreateFrame("Button", "SmartError_ActivateAllButton", ConfigurationPanel, "OptionsButtonTemplate");
	ActivateAllButton:SetPoint("TOPLEFT", 200, -25);
	ActivateAllButton.tooltip = "Click this button to activate all event sounds.";
	ActivateAllButton:SetScript("OnClick",SmartError_SetEventOptionAll);
	getglobal(ActivateAllButton:GetName().."Text"):SetText("Check All");

	local DeactivateAllButton = CreateFrame("Button", "SmartError_DeactivateAllButton", ConfigurationPanel, "OptionsButtonTemplate");
	DeactivateAllButton:SetPoint("TOPLEFT", 300, -25);
	DeactivateAllButton.tooltip = "Click this button to deactivate all event sounds.";
	DeactivateAllButton:SetScript("OnClick",SmartError_SetEventOptionNone);
	getglobal(DeactivateAllButton:GetName().."Text"):SetText("Uncheck All");

	ConfigurationPanel.okay = 
		function (self)
			SmartError_Option_Active(EnabledButton:GetChecked());
			for index, option in pairs(SmartError.Events) do
				SmartError_SetEventOption(option, getglobal("SmartError_EventButton_"..option):GetChecked())
			end
		end
	ConfigurationPanel.cancel = 
		function (self)
			EnabledButton:SetChecked(SmartErrorData.Active);
			for index, option in pairs(SmartError.Events) do
				SmartError_SetEventOption(option, SmartErrorData.Events[option])
			end
		end
	ConfigurationPanel.default = 
		function (self)
			SmartError_SetDefaults();
			SmartError_Option_Active(true);
			for index, option in pairs(SmartError.Events) do
				SmartError_SetEventOption(option, SmartErrorData.Events[option])
			end
		end
end

function SmartError_Option_Active(state)
	if (state) then
		SmartErrorData.Active = true;
	else
		SmartErrorData.Active = nil;
	end
	getglobal("SmartError_EnabledButton"):SetChecked(state);
	SmartError_ActivateMod();
end

function SmartError_ActivateMod()
	if (SmartErrorData.Active) then
		SmartErrorFrame:RegisterEvent("UI_ERROR_MESSAGE");
		SmartErrorFrame:RegisterEvent("RESURRECT_REQUEST");
		SmartErrorFrame:RegisterEvent("CONFIRM_SUMMON");
	else
		SmartErrorFrame:UnregisterEvent("UI_ERROR_MESSAGE");
		SmartErrorFrame:UnregisterEvent("RESURRECT_REQUEST");
		SmartErrorFrame:UnregisterEvent("CONFIRM_SUMMON");
	end	
end

function SmartError_SetEventOption(option, state)
	if (state) then
		SmartErrorData.Events[option] = true;
	else
		SmartErrorData.Events[option] = nil;
	end
	getglobal("SmartError_EventButton_"..option):SetChecked(state);
end

function SmartError_SetEventOptionAll()
	for index, option in pairs(SmartError.Events) do
		getglobal("SmartError_EventButton_"..option):SetChecked(true);
	end
end

function SmartError_SetEventOptionNone()
	for index, option in pairs(SmartError.Events) do
		getglobal("SmartError_EventButton_"..option):SetChecked(nil);
	end
end

function SmartError_CreateEventButton(optionCodeName, configurationPanel, coordX, coordY, optionName, optionDescription)
	local button = CreateFrame("CheckButton", "SmartError_EventButton_"..optionCodeName, configurationPanel, "ChatConfigCheckButtonTemplate");
	button:SetPoint("TOPLEFT", coordX, coordY);
	button.tooltip = optionDescription;
	button:SetChecked(SmartErrorData.Events[optionCodeName]);
	getglobal(button:GetName().."Text"):SetText(optionName);
end

function SmartError_CreateTestButton(optionCodeName, configurationPanel, coordX, coordY, soundNumber, optionDescription)
	local button = CreateFrame("Button", "SmartError_TestButton_"..optionCodeName, configurationPanel, "OptionsButtonTemplate");
	button:SetPoint("TOPLEFT", coordX, coordY);
	button:SetScript("OnClick",function() SmartError_PlaySound(soundNumber) end);
	getglobal(button:GetName().."Text"):SetText("Test");
end
