--- Extends Objects With Additional Functionality
--
local Extentions = {}
local AddOnName, nsVars = ...
local ss = nsVars.SS

local type, pairs = type, pairs
local IsShiftKeyDown = IsShiftKeyDown

--- Calls SetVertexColor using table values.
-- @param obj The Frame object.
-- @param color The color table.
--
local function SetVertColor(obj, color)
  obj:SetVertexColor(ss.Utils:GetRGBAValues(color))
end

--- Animates an alpha change.
-- @param display The Frame.
-- @param target The target alpha.
-- @param onComplete Completion handler. (Optional)
-- @param onUpdate OnUpdate handler. (Optional)
--
local function Fade(display, target, onComplete, onUpdate)
  local fadeDuration  = 0.1
  target = target or 0
  
  if target == display.fadeTarget then
    return
  end
  
  display.fadeAnimGroup:SetScript("OnFinished", nil)
  display.fadeAnimGroup:SetScript("OnUpdate", nil)
  
  display.fadeTarget = target
  display.fadeOnComplete = onComplete
  display.fadeOnUpdate = onUpdate
  
  -- Retrieve current values to handle ongoing animations.
  local dur = (display.fadeAnimGroup.fade:GetProgress() * fadeDuration) or fadeDuration
  
  if not display.fadeAnimGroup:IsDone() then
    display.fadeAnimGroup:Stop()
  else
    dur = fadeDuration
  end
  
  -- Set animation values.
  display.fadeAnimGroup.fade:SetFromAlpha(display.fadeCurAlpha or display:GetAlpha())
  display.fadeAnimGroup.fade:SetToAlpha  (target)
  display.fadeAnimGroup.fade:SetDuration (dur)
  display.fadeAnimGroup:Play()
  
  display.fadeAnimGroup:SetScript("OnFinished", function()
    -- Apply the alpha change.
    display:SetAlpha(display.fadeTarget)
    display.fadeCurAlpha = display.fadeTarget
    
    if type(display.fadeOnComplete) == "function" then
      display.fadeOnComplete(display)
    end
    display.fadeOnComplete = nil
    display.fadeOnUpdate = nil
  end)
  
  display.fadeAnimGroup:SetScript("OnUpdate", function()
    -- Ensure the unit frames is responsive imediately.
    display.fadeCurAlpha = display:GetAlpha()
    
    if type(display.fadeOnUpdate) == "function" then
      display.fadeOnUpdate(display)
    end
  end)
end

--- Adds the SetVertColor functionality.
-- @param ... One or more Frames.
--
function Extentions:DelegateVertexColor(...)
  local obj = {...}

  for _, v in pairs(obj) do
    if not v.SetVertexColor then
      ss.Utils.throw(ss.Utils:stringf("Object (%s) does not implement SetVertexColor", type(v)), 2)
    else
      v.SetVertColor = SetVertColor
    end
  end
end

--- Makes a frame movable whike holding the selected key.
-- @param frame The target frame.
-- @param button The button to hold, defaults to "LeftButton".
-- @param listener optional callback for "OnDragStop".
--
function Extentions:MakeMovable(frame, button, listener)
  if type(frame) == "table" then
    frame:SetClampedToScreen(true)
    frame:SetMovable(true)
    frame:EnableMouse(true)
    frame:RegisterForDrag(button or "LeftButton")
  
    frame:SetScript("OnDragStart", function(self)
      if IsShiftKeyDown() then
        self:StartMoving()
      end
    end)
  
    frame:SetScript("OnDragStop", function(self)
      self:StopMovingOrSizing()
      if listener then
        listener(self)
      end
    end)
  else
    ss.Utils.throw(ss.Utils.Errors.ErrorInvalidArgs, "Extentions", "MakeMovable",
      {"frame", "Frame", frame}
    )
  end
end

--- Makes a frame movable whike holding the selected key.
-- @param frame The target frame.
-- @param button The button to hold, defaults to "LeftButton".
-- @param listener optional callback for "OnDragStop".
--
function Extentions:MakeFadable(...)
  local frames = {...}
  
  for _, frame in pairs(frames) do
    if type(frame) == "table" then
      frame.fadeCurAlpha = frame:GetAlpha()
      frame.fadeTarget = frame.fadeCurAlpha
      frame.fadeAnimGroup = frame.fadeAnimGroup or frame:CreateAnimationGroup()
      frame.fadeAnimGroup.fade = frame.fadeAnimGroup.fade or frame.fadeAnimGroup:CreateAnimation("Alpha")
      frame.Fade = Fade
    else
      ss.Utils.throw(ss.Utils.Errors.ErrorInvalidArgs, "Extentions", "MakeFadable",
        {"frame", "Frame", frame}
      )
    end
  end
end

ss.Utils.addModule("Extentions", Extentions)
