-----------------------
--Scrolling Chat Text--
--A World of Warcraft addon for displaying chat messages in a scrolling frame

local addonName, SCT = ...;
addonName = "ScrollingChatText";
_G["SCT"] = SCT;

-- Version detection and compatibility
local WOW_VERSION = select(4, GetBuildInfo());
local IS_RETAIL = WOW_PROJECT_ID == WOW_PROJECT_MAINLINE;
local IS_CLASSIC = WOW_PROJECT_ID == WOW_PROJECT_CLASSIC;
local IS_TBC = WOW_PROJECT_ID == WOW_PROJECT_BURNING_CRUSADE_CLASSIC;
local IS_WOTLK = WOW_PROJECT_ID == WOW_PROJECT_WRATH_CLASSIC;
local IS_CATA = WOW_PROJECT_ID == WOW_PROJECT_CATA_CLASSIC;
local IS_MOP = WOW_PROJECT_ID == WOW_PROJECT_MOP_CLASSIC;

-- Compatibility functions
local function CreateTimer(delay, callback)
    if C_Timer then
        C_Timer.After(delay, callback);
    else
        -- Classic/BC/WotLK fallback
        local frame = CreateFrame("Frame");
        frame:SetScript("OnUpdate", function(self, elapsed)
            self.elapsed = (self.elapsed or 0) + elapsed;
            if self.elapsed >= delay then
                callback();
                self:SetScript("OnUpdate", nil);
            end
        end);
    end
end

local function CreateTicker(duration, callback, iterations)
    if C_Timer then
        return C_Timer.NewTicker(duration, callback, iterations);
    else
        -- Classic/BC/WotLK fallback
        local frame = CreateFrame("Frame");
        local count = 0;
        frame:SetScript("OnUpdate", function(self, elapsed)
            self.elapsed = (self.elapsed or 0) + elapsed;
            if self.elapsed >= duration then
                count = count + 1;
                callback();
                if iterations and count >= iterations then
                    self:SetScript("OnUpdate", nil);
                end
                self.elapsed = 0;
            end
        end);
        return frame;
    end
end

local function GetPlayerClass(senderName, senderGUID)
    if GetPlayerInfoByGUID then
        local _, class = GetPlayerInfoByGUID(senderGUID);
        return class;
    else
        -- Classic fallback: try to get class from UnitClass if it's the player
        if senderName == UnitName("player") then
            local _, class = UnitClass("player");
            return class;
        else
            -- Default fallback
            return "PRIEST";
        end
    end
end

local function IsSettingsOpen()
    if SettingsPanel then
        return SettingsPanel:IsShown();
    else
        return InterfaceOptionsFrame:IsShown();
    end
end

-- Store compatibility functions in SCT (without self parameter)
SCT.CreateTimer = function(delay, callback)
    if type(delay) ~= "number" or delay <= 0 then
        SCT:print("Error: CreateTimer requires a positive number for delay");
        return;
    end
    if type(callback) ~= "function" then
        SCT:print("Error: CreateTimer requires a function for callback");
        return;
    end
    return CreateTimer(delay, callback);
end;
SCT.CreateTicker = function(duration, callback, iterations)
    if type(duration) ~= "number" or duration <= 0 then
        SCT:print("Error: CreateTicker requires a positive number for duration");
        return;
    end
    if type(callback) ~= "function" then
        SCT:print("Error: CreateTicker requires a function for callback");
        return;
    end
    return CreateTicker(duration, callback, iterations);
end;
SCT.GetPlayerClass = function(senderName, senderGUID)
    if type(senderName) ~= "string" then
        return "PRIEST"; -- Default fallback
    end
    return GetPlayerClass(senderName, senderGUID);
end;
SCT.IsSettingsOpen = function()
    return IsSettingsOpen();
end;
SCT.IS_RETAIL = IS_RETAIL;
SCT.IS_CLASSIC = IS_CLASSIC;
SCT.IS_TBC = IS_TBC;
SCT.IS_WOTLK = IS_WOTLK;
SCT.IS_CATA = IS_CATA;
SCT.IS_MOP = IS_MOP;

-- Constants
SCT.CONSTANTS = {
	-- Animation and timing
	DUPLICATE_EVENT_THRESHOLD = 0.5,  -- Seconds to prevent duplicate events
	FRAME_FADE_IN_DURATION = 0.1,     -- Frame fade in duration
	FRAME_FADE_OUT_DURATION = 0.2,    -- Frame fade out duration
	FRAME_FADE_DELAY = 0.1,           -- Delay after fade out before hiding
	
	-- UI colors and alpha
	FRAME_BACKGROUND_ALPHA = 0.2,     -- Main frame background alpha
	LINE_BACKGROUND_ALPHA = 0.4,      -- Line frame background alpha
	DISPLAY_TAB_ALPHA = 0.3,          -- Display tab alpha
	DISPLAY_TAB_BACKGROUND_ALPHA = 0.8, -- Display tab background alpha
	
	-- Text colors
	BUTTON_TEXT_RED = {1, 0.2, 0.2},  -- Red color for stop button
	BUTTON_TEXT_GRAY = {0.5, 0.5, 0.5}, -- Gray color for disabled button
	BUTTON_TEXT_WHITE = {1, 1, 1},    -- White color for normal button
	
	-- Test configuration
	TEST_DURATION = 30,               -- Test duration in seconds
	TEST_INTERVAL = 2,                -- Interval between test messages
	
	-- Default frame dimensions
	DEFAULT_FRAME_WIDTH = 30,         -- Default frame width
	DEFAULT_FRAME_HEIGHT = 200,       -- Default frame height
	DEFAULT_LINE_HEIGHT = 15,         -- Default line height
	DEFAULT_FRAME_X = -300,           -- Default frame X position
	DEFAULT_FRAME_Y = 80,             -- Default frame Y position
};

local GetAddOnMetadata = C_AddOns and C_AddOns.GetAddOnMetadata or GetAddOnMetadata;

-- Initialize libraries
SCT.LSM = LibStub("LibSharedMedia-3.0");
SCT.acr = LibStub:GetLibrary("AceConfigRegistry-3.0");
SCT.acd = LibStub:GetLibrary("AceConfigDialog-3.0");

-- Default options
SCT.optionDefaults = {
	global = {
		sctEnabled = true,
		sctScrollHeight = 200,
		sctLineFrameHeight = 15,
		sctAnimationSpeed = 50,
		sctGrowthDirection = 1,
		sctAlignment = 1,
		sctAlpha = 1,
		sctLineFrameScale = 1,
		sctShowBackground = false,
		lockAllFrames = false,
		sctFont = "SCT Default",
		sctFontOutline = "NONE",
		sctEnableEmojis = true,
		messageFilters = {
			{
				keyword = "[WorldBuffs]",
				matchType = "begins",
				enabled = true
			}
		}, -- Message filters with default filter
		filterMatchType = "contains", -- Default match type for new filters
		filterCaseSensitive = false, -- Case sensitivity for filters
		-- Chat type settings
		sctGuildEnabled = true,
		sctPartyEnabled = false,
		sctRaidEnabled = false,
		sctInstanceEnabled = false,
		sctSayEnabled = false,
		sctYellEnabled = false,
		sctWhisperEnabled = true,
		sctOfficerEnabled = false,

		sctShowChatPrefixes = false,
		sctShowMyMessages = true, -- Display user's own messages
	},
};

SCT.version = GetAddOnMetadata("ScrollingChatText", "Version") or 9999;
SCT.version = tonumber(SCT.version);
SCT.prefixColor = "|cFFFF6900";

-- Font setup
SCT.regionFont = "Fonts\\ARIALN.ttf";
SCT.regionFont2 = "Fonts\\FRIZQT__.TTF";
SCT.regionFontNumbers = "Fonts\\FRIZQT__.ttf";

function SCT:setRegionFont()
	if (LOCALE_koKR) then
     	SCT.regionFont = "Fonts\\2002.TTF";
     	SCT.regionFont2 = "Fonts\\2002.TTF";
     	SCT.regionFontNumbers = "Fonts\\K_Pagetext.ttf";
    elseif (LOCALE_zhCN) then
     	SCT.regionFont = "Fonts\\ARKai_T.ttf";
     	SCT.regionFont2 = "Fonts\\ARKai_T.ttf";
     	SCT.regionFontNumbers = "Fonts\\ARKai_T.ttf";
    elseif (LOCALE_zhTW) then
     	SCT.regionFont = "Fonts\\blei00d.TTF";
     	SCT.regionFont2 = "Fonts\\blei00d.TTF";
     	SCT.regionFontNumbers = "Fonts\\blei00d.ttf";
    elseif (LOCALE_ruRU) then
     	SCT.regionFont2 = "Fonts\\ARIALN.ttf";
     	SCT.regionFontNumbers = "Fonts\\MORPHEUS_CYR.ttf";
    	SCT.regionFontItalic = "Interface\\Addons\\ScrollingChatText\\Media\\ARIALNI.TTF";
    	SCT.regionFontBoldItalic = "Interface\\Addons\\ScrollingChatText\\Media\\ARIALNBI.TTF";
    else
    	SCT.regionFontItalic = "Interface\\Addons\\ScrollingChatText\\Media\\ARIALNI.TTF";
    	SCT.regionFontBoldItalic = "Interface\\Addons\\ScrollingChatText\\Media\\ARIALNBI.TTF";
    end
end

SCT:setRegionFont();
SCT.LSM:Register("font", "SCT Default", SCT.regionFont);
SCT.LSM:Register("font", "SCT Numbers", SCT.regionFontNumbers);

-- Utility functions
function SCT:print(msg)
	print(SCT.prefixColor .. "[SCT]|r " .. msg);
end



function SCT:RGBToHex(r, g, b)
	r = r <= 255 and r >= 0 and r or 0
	g = g <= 255 and g >= 0 and g or 0
	b = b <= 255 and b >= 0 and b or 0
	return string.format("%02x%02x%02x", r, g, b)
end

function SCT:getClassColor(class)
	local colors = {
		["WARRIOR"] = {0.78, 0.61, 0.43},
		["PALADIN"] = {0.96, 0.55, 0.73},
		["HUNTER"] = {0.67, 0.83, 0.45},
		["ROGUE"] = {1.00, 0.96, 0.41},
		["PRIEST"] = {1.00, 1.00, 1.00},
		["SHAMAN"] = {0.00, 0.44, 0.87},
		["MAGE"] = {0.25, 0.78, 0.92},
		["WARLOCK"] = {0.53, 0.53, 0.93},
		["DRUID"] = {1.00, 0.49, 0.04},
	};
	
	-- Add expansion-specific classes
	if SCT.IS_RETAIL then
		-- Retail: all classes
		colors["DEATHKNIGHT"] = {0.77, 0.12, 0.23};
		colors["MONK"] = {0.00, 1.00, 0.59};
		colors["DEMONHUNTER"] = {0.64, 0.19, 0.79};
	elseif SCT.IS_MOP then
		-- MoP: add Death Knight and Monk
		colors["DEATHKNIGHT"] = {0.77, 0.12, 0.23};
		colors["MONK"] = {0.00, 1.00, 0.59};
	elseif SCT.IS_CATA then
		-- Cata: add Death Knight
		colors["DEATHKNIGHT"] = {0.77, 0.12, 0.23};
	elseif SCT.IS_WOTLK then
		-- WotLK: add Death Knight
		colors["DEATHKNIGHT"] = {0.77, 0.12, 0.23};
	end
	
	if colors[class] then
		local r, g, b = unpack(colors[class]);
		return r, g, b, string.format("%02x%02x%02x", r * 255, g * 255, b * 255);
	end
	return 1, 1, 1, "FFFFFF";
end

function SCT:loadDatabase()
	SCT.db = LibStub("AceDB-3.0"):New("SCTdatabase", SCT.optionDefaults, "Default");
	
	-- Create config reference for backward compatibility
	SCT.config = SCT.db.global;
end



local function init()
    SCT:loadDatabase();
	SCT.prefixColor = "|cff" .. SCT:RGBToHex(255, 105, 0);
	
	-- Print version detection info
	local versionText = "Unknown";
	if SCT.IS_RETAIL then
		versionText = "Retail";
	elseif SCT.IS_MOP then
		versionText = "MoP Classic";
	elseif SCT.IS_CATA then
		versionText = "Cata Classic";
	elseif SCT.IS_WOTLK then
		versionText = "WotLK Classic";
	elseif SCT.IS_TBC then
		versionText = "TBC Classic";
	elseif SCT.IS_CLASSIC then
		versionText = "Classic Era";
	end
	SCT:print("Scrolling Chat Text loaded for " .. versionText .. " (Interface " .. WOW_VERSION .. ")");
	
	-- Ensure the default [WorldBuffs] filter exists
	if not SCT.db.global.messageFilters then
		SCT.db.global.messageFilters = {};
	end
	
	local hasDefaultFilter = false;
	for _, filter in ipairs(SCT.db.global.messageFilters) do
		if filter.keyword == "[WorldBuffs]" then
			hasDefaultFilter = true;
			break;
		end
	end
	
	-- Add default filter if it doesn't exist
	if not hasDefaultFilter then
		table.insert(SCT.db.global.messageFilters, {
			keyword = "[WorldBuffs]",
			matchType = "begins",
			enabled = true
		});
	end
	
	-- Register the reset confirmation popup
	StaticPopupDialogs["SCT_RESET_CONFIRM"] = {
		text = "Are you sure you want to reset all settings to defaults?\n\nThis will reset:\n• All general settings\n• All chat type settings\n• All message filters (except the default [WorldBuffs] filter)",
		button1 = "Reset",
		button2 = "Cancel",
		OnAccept = function()
			SCT:performResetToDefaults();
		end,
		timeout = 0,
		whileDead = true,
		hideOnEscape = true,
		preferredIndex = 3,
	};
	
	-- Load chat message frames after database is loaded
	SCT.CreateTimer(0.2, function()
		if SCT.loadScrollingChatFrames then
			SCT:loadScrollingChatFrames();
		end
	end)
	
	-- Rebuild filter list after database is fully initialized
	SCT.CreateTimer(0.5, function()
		if SCT.rebuildFilterList then
			SCT:rebuildFilterList();
		end
	end)
	
	-- Create animation update frame for animated emojis
	SCT.animationFrame = CreateFrame("Frame");
	SCT.animationFrame:SetScript("OnUpdate", function(self, elapsed)
		-- Run animation updates at full frame rate for smooth animated emojis
		if SCT.updateAnimatedEmojis then
			SCT:updateAnimatedEmojis(elapsed);
		end
	end);
end

local f = CreateFrame("Frame");
f:RegisterEvent("ADDON_LOADED");
f:SetScript("OnEvent", function(self, event, addon)
	if (event == "ADDON_LOADED" and addon == "ScrollingChatText") then
		init();
	end
end) 