--------------------------------------------
---Scrolling Chat Text Options-----------
--------------------------------------------

local addonName, SCT = ...;



function SCT:loadOptions()
	

	
	local options = {
		name = "Scrolling Chat Text",
		handler = SCT,
		type = "group",
		args = {
			general = {
				name = "General",
				type = "group",
				order = 1,
				args = {
					sctEnabled = {
						type = "toggle",
						name = "Enabled",
						desc = "Enable or disable scrolling chat text",
						order = 1,
						get = "getSctEnabled",
						set = "setSctEnabled",
					},
					spacer1 = {
						type = "description",
						name = "",
						order = 2,
					},
					sctScrollHeight = {
						type = "range",
						name = "Frame Height",
						desc = "Height of the scrolling frame",
						min = 50,
						max = 500,
						step = 10,
						order = 3,
						get = "getSctScrollHeight",
						set = "setSctScrollHeight",
					},
					sctLineFrameHeight = {
						type = "range",
						name = "Line Height",
						desc = "Height of each message line",
						min = 10,
						max = 30,
						step = 1,
						order = 4,
						get = "getSctLineFrameHeight",
						set = "setSctLineFrameHeight",
					},
					sctAnimationSpeed = {
						type = "range",
						name = "Animation Speed",
						desc = "Speed of the scrolling animation",
						min = 10,
						max = 200,
						step = 5,
						order = 5,
						get = "getSctAnimationSpeed",
						set = "setSctAnimationSpeed",
					},
					sctGrowthDirection = {
						type = "select",
						name = "Growth Direction",
						desc = "Direction messages grow from",
						values = {
							[1] = "Up",
							[2] = "Down",
						},
						order = 6,
						get = "getSctGrowthDirection",
						set = "setSctGrowthDirection",
					},
					sctAlignment = {
						type = "select",
						name = "Text Alignment",
						desc = "Alignment of text within the frame",
						values = {
							[1] = "Left",
							[2] = "Center",
							[3] = "Right",
						},
						order = 7,
						get = "getSctAlignment",
						set = "setSctAlignment",
					},
					sctAlpha = {
						type = "range",
						name = "Frame Alpha",
						desc = "Transparency of the frame",
						min = 0.1,
						max = 1.0,
						step = 0.1,
						order = 8,
						get = "getSctAlpha",
						set = "setSctAlpha",
					},
					sctLineFrameScale = {
						type = "range",
						name = "Text Scale",
						desc = "Scale of the text",
						min = 0.5,
						max = 2.0,
						step = 0.1,
						order = 9,
						get = "getSctLineFrameScale",
						set = "setSctLineFrameScale",
					},
					sctShowBackground = {
						type = "toggle",
						name = "Show Text Background",
						desc = "Show or hide the background color behind the text",
						order = 10,
						get = "getSctShowBackground",
						set = "setSctShowBackground",
					},
					spacer2 = {
						type = "description",
						name = "",
						order = 11,
					},
					lockAllFrames = {
						type = "toggle",
						name = "Lock Frames",
						desc = "Lock all frames to prevent moving",
						order = 12,
						get = "getLockAllFrames",
						set = "setLockAllFrames",
					},
					enableEmojis = {
						type = "toggle",
						name = "Enable Emoji Support",
						desc = "Enable support for TwitchEmotes and other emoji addons",
						order = 13,
						get = "getSctEnableEmojis",
						set = "setSctEnableEmojis",
					},
					spacer3 = {
						type = "description",
						name = "",
						order = 14,
					},
					resetDefaults = {
						type = "execute",
						name = "Reset to Defaults",
						desc = "Reset all settings to their default values",
						order = 15,
						func = "resetToDefaults",
					},
				},
			},
			chatTypes = {
				name = "Chat Types",
				type = "group",
				order = 2,
				args = {
					sctGuildEnabled = {
						type = "toggle",
						name = "Guild Chat",
						desc = "Display guild chat messages",
						order = 1,
						get = "getSctGuildEnabled",
						set = "setSctGuildEnabled",
					},
					sctPartyEnabled = {
						type = "toggle",
						name = "Party Chat",
						desc = "Display party chat messages",
						order = 2,
						get = "getSctPartyEnabled",
						set = "setSctPartyEnabled",
					},
					sctRaidEnabled = {
						type = "toggle",
						name = "Raid Chat",
						desc = "Display raid chat messages",
						order = 3,
						get = "getSctRaidEnabled",
						set = "setSctRaidEnabled",
					},
					sctInstanceEnabled = {
						type = "toggle",
						name = "Instance Chat",
						desc = "Display instance chat messages",
						order = 4,
						get = "getSctInstanceEnabled",
						set = "setSctInstanceEnabled",
					},
					sctSayEnabled = {
						type = "toggle",
						name = "Say",
						desc = "Display say messages",
						order = 5,
						get = "getSctSayEnabled",
						set = "setSctSayEnabled",
					},
					sctYellEnabled = {
						type = "toggle",
						name = "Yell",
						desc = "Display yell messages",
						order = 6,
						get = "getSctYellEnabled",
						set = "setSctYellEnabled",
					},
					sctWhisperEnabled = {
						type = "toggle",
						name = "Whisper",
						desc = "Display whisper messages",
						order = 7,
						get = "getSctWhisperEnabled",
						set = "setSctWhisperEnabled",
					},
					sctOfficerEnabled = {
						type = "toggle",
						name = "Officer Chat",
						desc = "Display officer chat messages",
						order = 8,
						get = "getSctOfficerEnabled",
						set = "setSctOfficerEnabled",
					},
					prefixesHeader = {
						type = "header",
						name = "Display Options",
						order = 9,
					},
					sctShowChatPrefixes = {
						type = "toggle",
						name = "Show Chat Type Prefixes",
						desc = "Show prefixes like [Guild], [Party], etc. before messages",
						order = 10,
						get = "getSctShowChatPrefixes",
						set = "setSctShowChatPrefixes",
					},
					sctShowMyMessages = {
						type = "toggle",
						name = "Display My Messages",
						desc = "Show your own messages in the scrolling chat frame",
						order = 11,
						get = "getSctShowMyMessages",
						set = "setSctShowMyMessages",
					},

				},
			},
			filters = {
				name = "Message Filters",
				type = "group",
				order = 3,
				args = {
					filtersDesc = {
						type = "description",
						name = "Filter out chat messages containing specific keywords. Messages matching any enabled filter will not be displayed.",
						order = 1,
					},
					inputGroup = {
						type = "group",
						name = "",
						inline = true,
						order = 2,
						args = {
							addFilter = {
								type = "input",
								name = "Add New Filter",
								desc = "Enter a keyword to filter and press Enter",
								order = 1,
								get = function() return ""; end,
								set = function(info, value) 
									if value and value ~= "" then
										SCT:addMessageFilterDirect(value);
									end
								end,
							},
						},
					},
					matchType = {
						type = "select",
						name = "Match Type",
						desc = "How to match the keyword",
						values = {
							["contains"] = "Contains",
							["begins"] = "Begins with",
							["ends"] = "Ends with",
							["exact"] = "Exact match",
						},
						order = 3,
						get = "getFilterMatchType",
						set = "setFilterMatchType",
					},

					caseSensitive = {
						type = "toggle",
						name = "Case Sensitive",
						desc = "Enable case sensitive matching",
						order = 5,
						get = "getFilterCaseSensitive",
						set = "setFilterCaseSensitive",
					},
					spacer1 = {
						type = "description",
						name = "",
						order = 6,
					},
					filterList = {
						type = "group",
						name = "Active Filters",
						inline = true,
						order = 6,
						args = {},
					},
				},
			},

		},
	};
	
	SCT.options = options;
	SCT.acr:RegisterOptionsTable("ScrollingChatText", options);
	SCT.acd:AddToBlizOptions("ScrollingChatText", "Scrolling Chat Text");
	
	-- Initialize the filter list
	SCT:rebuildFilterList();
end

-- Getter and setter functions
function SCT:getSctEnabled(info)
	return self.db.global.sctEnabled;
end

function SCT:setSctEnabled(info, value)
	self.db.global.sctEnabled = value;
	self:updateSctState();
end

function SCT:getSctScrollHeight(info)
	return self.db.global.sctScrollHeight;
end

function SCT:setSctScrollHeight(info, value)
	self.db.global.sctScrollHeight = value;
	self:sctUpdateSettings();
end

function SCT:getSctLineFrameHeight(info)
	return self.db.global.sctLineFrameHeight;
end

function SCT:setSctLineFrameHeight(info, value)
	self.db.global.sctLineFrameHeight = value;
	self:sctUpdateSettings();
end

function SCT:getSctAnimationSpeed(info)
	return self.db.global.sctAnimationSpeed;
end

function SCT:setSctAnimationSpeed(info, value)
	self.db.global.sctAnimationSpeed = value;
	self:sctUpdateSettings();
end

function SCT:getSctGrowthDirection(info)
	return self.db.global.sctGrowthDirection;
end

function SCT:setSctGrowthDirection(info, value)
	self.db.global.sctGrowthDirection = value;
	self:sctUpdateSettings();
end

function SCT:getSctAlignment(info)
	return self.db.global.sctAlignment;
end

function SCT:setSctAlignment(info, value)
	self.db.global.sctAlignment = value;
	self:sctUpdateSettings();
end

function SCT:getSctAlpha(info)
	return self.db.global.sctAlpha;
end

function SCT:setSctAlpha(info, value)
	self.db.global.sctAlpha = value;
	self:sctUpdateSettings();
end

function SCT:getSctLineFrameScale(info)
	return self.db.global.sctLineFrameScale;
end

function SCT:setSctLineFrameScale(info, value)
	self.db.global.sctLineFrameScale = value;
	self:sctUpdateSettings();
end

function SCT:getSctShowBackground(info)
	return self.db.global.sctShowBackground;
end

function SCT:setSctShowBackground(info, value)
	self.db.global.sctShowBackground = value;
	self:sctUpdateSettings();
end

function SCT:getLockAllFrames(info)
	return self.db.global.lockAllFrames;
end

function SCT:setLockAllFrames(info, value)
	self.db.global.lockAllFrames = value;
	self:sctUpdateFrameLocks();
	self:sctUpdateFrameLocksLayout();
end

function SCT:getSctEnableEmojis(info)
	return self.db.global.sctEnableEmojis;
end

function SCT:setSctEnableEmojis(info, value)
	self.db.global.sctEnableEmojis = value;
end

-- Chat type getters and setters
function SCT:getSctGuildEnabled(info)
	return self.db.global.sctGuildEnabled;
end

function SCT:setSctGuildEnabled(info, value)
	self.db.global.sctGuildEnabled = value;
end

function SCT:getSctPartyEnabled(info)
	return self.db.global.sctPartyEnabled;
end

function SCT:setSctPartyEnabled(info, value)
	self.db.global.sctPartyEnabled = value;
end

function SCT:getSctRaidEnabled(info)
	return self.db.global.sctRaidEnabled;
end

function SCT:setSctRaidEnabled(info, value)
	self.db.global.sctRaidEnabled = value;
end

function SCT:getSctInstanceEnabled(info)
	return self.db.global.sctInstanceEnabled;
end

function SCT:setSctInstanceEnabled(info, value)
	self.db.global.sctInstanceEnabled = value;
end

function SCT:getSctSayEnabled(info)
	return self.db.global.sctSayEnabled;
end

function SCT:setSctSayEnabled(info, value)
	self.db.global.sctSayEnabled = value;
end

function SCT:getSctYellEnabled(info)
	return self.db.global.sctYellEnabled;
end

function SCT:setSctYellEnabled(info, value)
	self.db.global.sctYellEnabled = value;
end

function SCT:getSctWhisperEnabled(info)
	return self.db.global.sctWhisperEnabled;
end

function SCT:setSctWhisperEnabled(info, value)
	self.db.global.sctWhisperEnabled = value;
end

function SCT:getSctOfficerEnabled(info)
	return self.db.global.sctOfficerEnabled;
end

function SCT:setSctOfficerEnabled(info, value)
	self.db.global.sctOfficerEnabled = value;
end

function SCT:getSctShowChatPrefixes(info)
	return self.db.global.sctShowChatPrefixes;
end

function SCT:setSctShowChatPrefixes(info, value)
	self.db.global.sctShowChatPrefixes = value;
end

function SCT:getSctShowMyMessages(info)
	return self.db.global.sctShowMyMessages;
end

function SCT:setSctShowMyMessages(info, value)
	self.db.global.sctShowMyMessages = value;
end

-- Message Filter Functions
function SCT:buildFilterList()
	local filterArgs = {};
	
	-- Check if database is loaded
	if not self.db or not self.db.global then
		return filterArgs; -- Return empty args if database not ready
	end
	
	-- Ensure messageFilters exists
	if not self.db.global.messageFilters then
		self.db.global.messageFilters = {};
	end
	
	-- Ensure the default [WorldBuffs] filter exists
	local hasDefaultFilter = false;
	for _, filter in ipairs(self.db.global.messageFilters) do
		if filter.keyword == "[WorldBuffs]" then
			hasDefaultFilter = true;
			break;
		end
	end
	
	-- Add default filter if it doesn't exist
	if not hasDefaultFilter then
		table.insert(self.db.global.messageFilters, {
			keyword = "[WorldBuffs]",
			matchType = "begins",
			enabled = true
		});
	end
	
	for i, filter in ipairs(self.db.global.messageFilters) do
		-- Add keyword display
		filterArgs["keyword" .. i] = {
			type = "description",
			name = function()
				local color = filter.enabled and "|cffffffff" or "|cff808080";
				return color .. "Keyword: " .. filter.keyword .. "|r";
			end,
			order = i * 4 - 3,
		};
		
		-- Add match type display
		filterArgs["matchType" .. i] = {
			type = "description",
			name = function()
				local color = filter.enabled and "|cffffffff" or "|cff808080";
				return color .. "Match Type: " .. filter.matchType .. "|r";
			end,
			order = i * 4 - 2,
		};
		
		-- Add enabled toggle
		filterArgs["enabled" .. i] = {
			type = "toggle",
			name = "Enabled",
			desc = "Enable or disable this filter",
			order = i * 4 - 1,
			get = function() return filter.enabled; end,
			set = function(info, value) 
				filter.enabled = value;
				self.acr:NotifyChange("ScrollingChatText");
			end,
		};
		
		-- Add delete button
		filterArgs["remove" .. i] = {
			type = "execute",
			name = "Delete",
			desc = "Delete this filter",
			order = i * 4,
			func = function() 
				self:removeMessageFilter(i);
			end,
		};
		
		-- Add separator after each filter (except the last one)
		if i < #self.db.global.messageFilters then
			filterArgs["separator" .. i] = {
				type = "description",
				name = "",
				order = i * 4 + 0.5,
			};
		end
	end
	
	if #self.db.global.messageFilters == 0 then
		filterArgs["noFilters"] = {
			type = "description",
			name = "No filters added yet.",
			order = 100,
		};
	end
	
	return filterArgs;
end

function SCT:getFilterMatchType(info)
	if not self.db or not self.db.global then
		return "contains"; -- Default value if database not ready
	end
	return self.db.global.filterMatchType;
end

function SCT:setFilterMatchType(info, value)
	if not self.db or not self.db.global then
		return; -- Don't set if database not ready
	end
	self.db.global.filterMatchType = value;
end

function SCT:getFilterCaseSensitive(info)
	if not self.db or not self.db.global then
		return false; -- Default value if database not ready
	end
	return self.db.global.filterCaseSensitive;
end

function SCT:setFilterCaseSensitive(info, value)
	if not self.db or not self.db.global then
		return; -- Don't set if database not ready
	end
	self.db.global.filterCaseSensitive = value;
end

function SCT:rebuildFilterList()
	-- Rebuild the filter list in the options
	if self.options and self.options.args and self.options.args.filters and self.options.args.filters.args and self.options.args.filters.args.filterList then
		self.options.args.filters.args.filterList.args = self:buildFilterList();
		-- Notify the options system that the table has changed
		self.acr:NotifyChange("ScrollingChatText");
	end
end

function SCT:addMessageFilterDirect(keyword)
	-- Check if database is loaded
	if not self.db or not self.db.global then
		return; -- Don't add if database not ready
	end
	
	-- Ensure messageFilters exists
	if not self.db.global.messageFilters then
		self.db.global.messageFilters = {};
	end
	
	-- Check for duplicates (case-insensitive)
	local keywordLower = string.lower(keyword);
	for _, existingFilter in ipairs(self.db.global.messageFilters) do
		if string.lower(existingFilter.keyword) == keywordLower then
			return;
		end
	end
	
	-- Add the filter to the list
	local newFilter = {
		keyword = keyword,
		matchType = self.db.global.filterMatchType,
		enabled = true
	};
	
	table.insert(self.db.global.messageFilters, newFilter);
	
	-- Rebuild the filter list and refresh
	self:rebuildFilterList();
	self.acr:NotifyChange("ScrollingChatText");
end

function SCT:removeMessageFilter(filterIndex)
	-- Check if database is loaded
	if not self.db or not self.db.global then
		return; -- Don't remove if database not ready
	end
	
	table.remove(self.db.global.messageFilters, filterIndex);
	-- Rebuild the filter list and refresh
	self:rebuildFilterList();
	self.acr:NotifyChange("ScrollingChatText");
end

-- Function to check if a message should be filtered
function SCT:shouldFilterMessage(message)
	-- Check if database is loaded
	if not self.db or not self.db.global then
		return false; -- Don't filter if database not ready
	end
	
	if not self.db.global.messageFilters then
		return false;
	end
	
	for i, filter in ipairs(self.db.global.messageFilters) do
		if filter.enabled then
			local matches = false;
			local messageText = message;
			local keywordText = filter.keyword;
			
			-- Apply case sensitivity
			if not self.db.global.filterCaseSensitive then
				messageText = string.lower(messageText);
				keywordText = string.lower(keywordText);
			end
			
			if filter.matchType == "contains" then
				matches = string.find(messageText, keywordText, 1, true);
			elseif filter.matchType == "begins" then
				-- Escape special regex characters for begins match
				local escapedKeyword = string.gsub(keywordText, "([%[%]%.%+%-%*%?%^%$%(%)%%])", "%%%1");
				matches = string.match(messageText, "^" .. escapedKeyword);
			elseif filter.matchType == "ends" then
				-- Escape special regex characters for ends match
				local escapedKeyword = string.gsub(keywordText, "([%[%]%.%+%-%*%?%^%$%(%)%%])", "%%%1");
				matches = string.match(messageText, escapedKeyword .. "$");
			elseif filter.matchType == "exact" then
				matches = messageText == keywordText;
			end
			
			if matches then
				return true; -- Filter this message
			end
		end
	end
	return false; -- Don't filter this message
end

function SCT:resetToDefaults()
	-- Show confirmation dialog
	StaticPopup_Show("SCT_RESET_CONFIRM");
end

function SCT:performResetToDefaults()
	-- Reset all settings to defaults
	self.db.global.sctEnabled = true;
	self.db.global.sctScrollHeight = 200;
	self.db.global.sctLineFrameHeight = 15;
	self.db.global.sctAnimationSpeed = 50;
	self.db.global.sctGrowthDirection = 1;
	self.db.global.sctAlignment = 1;
	self.db.global.sctAlpha = 1;
	self.db.global.sctLineFrameScale = 1;
	self.db.global.sctShowBackground = false;
	self.db.global.lockAllFrames = false;
	self.db.global.sctFont = "SCT Default";
	self.db.global.sctFontOutline = "NONE";
	self.db.global.sctEnableEmojis = true;
	-- Chat type settings
	self.db.global.sctGuildEnabled = true;
	self.db.global.sctPartyEnabled = false;
	self.db.global.sctRaidEnabled = false;
	self.db.global.sctInstanceEnabled = false;
	self.db.global.sctSayEnabled = false;
	self.db.global.sctYellEnabled = false;
	self.db.global.sctWhisperEnabled = true;
	self.db.global.sctOfficerEnabled = false;
	self.db.global.sctShowChatPrefixes = false;
	self.db.global.sctShowMyMessages = true;
	
	-- Reset filter settings to default filter only
	self.db.global.messageFilters = {
		{
			keyword = "[WorldBuffs]",
			matchType = "begins",
			enabled = true
		}
	};
	self.db.global.filterMatchType = "contains";
	self.db.global.filterCaseSensitive = false;
	
	-- Update the UI
	self:sctUpdateSettings();
	self:sctUpdateFrameLocks();
	self:sctUpdateFrameLocksLayout();
	
	-- Rebuild the filter list
	self:rebuildFilterList();
	
	-- Show confirmation message
	self:print("All settings have been reset to defaults.");
end

-- Slash command
SLASH_SCT1 = "/sct";
SLASH_SCT2 = "/scrollingchattext";
SlashCmdList["SCT"] = function(msg)
	-- Convert to lowercase for case-insensitive comparison
	local lowerMsg = string.lower(msg);
	
	if (lowerMsg == "config" or lowerMsg == "options") then
		if SCT.acd then
			SCT.acd:Open("ScrollingChatText");
		end
	elseif (lowerMsg == "test") then
		SCT:startSctTest();
	elseif (lowerMsg == "stop") then
		SCT:stopSctTest();
	else
		SCT:print("Scrolling Chat Text commands:");
		SCT:print("/sct config - Open options");
		SCT:print("/sct test - Start test");
		SCT:print("/sct stop - Stop test");
	end
end

-- Call loadOptions when the module is loaded
	SCT:loadOptions();

