--------------------------------------------
---Scrolling Chat Text Frames------------
--------------------------------------------

local addonName, SCT = ...;

-- Animation support variables
local SCT_TimeSinceLastUpdate = 0;
local SCT_T = 0;

function SCT:createAutoScrollingFrame(name, width, height, x, y, lineFrameHeight)
	local frame;
	if SCT.IS_RETAIL and BackdropTemplate then
		frame = CreateFrame("Frame", name, UIParent, "BackdropTemplate");
	else
		frame = CreateFrame("Frame", name, UIParent);
		-- Apply backdrop manually for older versions
		if frame.SetBackdrop then
			frame:SetBackdrop({
				bgFile = "Interface\\Buttons\\WHITE8x8",
				insets = {top = 0, left = 0, bottom = 0, right = 0},
			});
		end
	end
	frame.defaultX = x;
	frame.defaultY = y;
	frame.firstRun = true;
	frame:SetMovable(true);
	frame:EnableMouse(true);
	frame:SetUserPlaced(false);
	frame:SetSize(width, height);
	frame:SetPoint("CENTER", UIParent, x or 0, y or 0);
	-- Set backdrop only if the method exists
	if frame.SetBackdrop then
		frame:SetBackdrop({
			bgFile = "Interface\\Buttons\\WHITE8x8",
			insets = {top = 0, left = 0, bottom = 0, right = 0},
		});
		frame:SetBackdropColor(0, 0, 0, SCT.CONSTANTS.FRAME_BACKGROUND_ALPHA);
	else
		-- Fallback for older versions: create a background texture
		frame.bgTexture = frame:CreateTexture(nil, "BACKGROUND");
		frame.bgTexture:SetTexture("Interface\\Buttons\\WHITE8x8");
		-- Set initial visibility based on user setting
		if SCT.db.global and SCT.db.global.sctShowBackground then
			frame.bgTexture:SetVertexColor(0, 0, 0, SCT.CONSTANTS.FRAME_BACKGROUND_ALPHA);
			frame.bgTexture:Show();
		else
			frame.bgTexture:SetVertexColor(0, 0, 0, 0);
			frame.bgTexture:Hide();
		end
		frame.bgTexture:SetAllPoints();
	end
	
	frame.lineFrames = {};
	frame.lineFrameHeight = lineFrameHeight;
	frame.lineFrameScale = 1;
	frame.lineFrameFont = "SCT Default";
	frame.lineFrameFontSize = lineFrameHeight - 2;
	frame.lineFrameFontOutline = "NONE";
	
	frame.createLineFrame = function()
		local count = #frame.lineFrames + 1;
		local lineFrame;
		if SCT.IS_RETAIL and BackdropTemplate then
			lineFrame = CreateFrame("Frame", frame:GetName().. "LineFrame" .. count, frame, "BackdropTemplate");
		else
			lineFrame = CreateFrame("Frame", frame:GetName().. "LineFrame" .. count, frame);
			-- Apply backdrop manually for older versions
			if lineFrame.SetBackdrop then
				lineFrame:SetBackdrop({
					bgFile = "Interface\\Buttons\\WHITE8x8",
					insets = {top = 0, left = 0, bottom = 0, right = 0},
				});
			end
		end
		lineFrame.count = count;
		lineFrame:SetSize(100, frame.lineFrameHeight);
		if (frame.alignment == 1) then
			--Left.
			lineFrame:SetPoint("LEFT", frame, 1, 0);
		elseif (frame.alignment == 2) then
			--Middle.
			lineFrame:SetPoint("CENTER", frame, 0, 0);
		elseif (frame.alignment == 3) then
			--Right.
			lineFrame:SetPoint("RIGHT", frame, -1, 0);
		end
		-- Set backdrop only if the method exists
		if lineFrame.SetBackdrop then
			lineFrame:SetBackdrop({
				bgFile = "Interface\\Buttons\\WHITE8x8",
				insets = {top = 0, left = 0, bottom = 0, right = 0},
			});
			lineFrame:SetBackdropColor(0, 0, 0, SCT.CONSTANTS.LINE_BACKGROUND_ALPHA);
		else
			-- Fallback for older versions: create a background texture
			lineFrame.bgTexture = lineFrame:CreateTexture(nil, "BACKGROUND");
			lineFrame.bgTexture:SetTexture("Interface\\Buttons\\WHITE8x8");
			-- Set initial visibility based on user setting
			if SCT.db.global and SCT.db.global.sctShowBackground then
				lineFrame.bgTexture:SetVertexColor(0, 0, 0, SCT.CONSTANTS.LINE_BACKGROUND_ALPHA);
			else
				lineFrame.bgTexture:SetVertexColor(0, 0, 0, 0);
			end
			lineFrame.bgTexture:SetAllPoints();
			-- Hide the texture initially - it will be shown when needed
			lineFrame.bgTexture:Hide();
		end
		lineFrame.fs = lineFrame:CreateFontString(frame:GetName().. "LineFrameFS" .. count, "ARTWORK");
		lineFrame.fs:SetFont(SCT.LSM:Fetch("font", frame.lineFrameFont), frame.lineFrameFontSize, frame.lineFrameFontOutline);
		lineFrame.fs:SetJustifyH("LEFT");
		lineFrame.fs:SetTextColor(1, 1, 1, 1); -- Reset to white to ensure color codes work
		lineFrame.texture = lineFrame:CreateTexture(nil, "ARTWORK");
		lineFrame.texture:SetTexture("error");
		lineFrame.texture:SetPoint("LEFT", 0, 0);
		lineFrame.texture:SetSize(frame.lineFrameHeight - 0, frame.lineFrameHeight - 0);
		lineFrame.fs:SetPoint("LEFT", lineFrame.texture, "RIGHT", 2, 0);
		lineFrame:SetScale(frame.lineFrameScale);
		lineFrame:Hide();
		frame.lineFrames[count] = lineFrame;
		return lineFrame;
	end
	
	-- Object pool for line frames to reduce garbage collection
	local lineFramePool = {};
	
	frame.getLineFrame = function()
		local lineFrame;
		-- Try to reuse from pool first
		if #lineFramePool > 0 then
			lineFrame = table.remove(lineFramePool);
			lineFrame:SetParent(frame);
			lineFrame:Show();
			return lineFrame;
		end
		
		-- Try to find an unused frame
		for k, v in ipairs(frame.lineFrames) do
			if (not v.enabled) then
				lineFrame = v;
				break;
			end
		end
		if (lineFrame) then
			return lineFrame;
		else
			return frame.createLineFrame();
		end
	end
	
	frame.growthDirection = 1; --Down, can be changed in options.
	frame.clearAllPoints = function()
		for k, v in ipairs(frame.lineFrames) do
			v:ClearAllPoints();
		end
	end
	
	frame.queue = {};
	frame.addLine = function(text, icon)
		local lineFrame = frame.getLineFrame();
		lineFrame.enabled = true;
		local t = {
			text = text,
			icon = icon,
			lineFrame = lineFrame,
			y = 0,
			elapsed = GetTime(),
		};
		lineFrame.fs:SetText(text);
		lineFrame:SetScale(frame.lineFrameScale);
		
		-- Apply background setting
		if lineFrame.SetBackdropColor then
			if SCT.db.global and SCT.db.global.sctShowBackground then
				lineFrame:SetBackdropColor(0, 0, 0, SCT.CONSTANTS.LINE_BACKGROUND_ALPHA);
			else
				lineFrame:SetBackdropColor(0, 0, 0, 0);
			end
		elseif lineFrame.bgTexture then
			-- Fallback for older versions using background texture
			if SCT.db.global and SCT.db.global.sctShowBackground then
				lineFrame.bgTexture:SetVertexColor(0, 0, 0, SCT.CONSTANTS.LINE_BACKGROUND_ALPHA);
				lineFrame.bgTexture:Show();
			else
				lineFrame.bgTexture:SetVertexColor(0, 0, 0, 0);
				lineFrame.bgTexture:Hide();
			end
		end
		
		if (icon) then
			--if there's an icon specified then show the left texture and move the text right to suit.
			lineFrame.texture:SetTexture(icon);
			lineFrame.texture:Show();
			lineFrame.fs:ClearAllPoints();
			lineFrame.fs:SetPoint("LEFT", lineFrame.texture, "RIGHT", 2, 0);
			lineFrame:SetWidth(lineFrame.texture:GetWidth() + lineFrame.fs:GetWidth() + 4);
		else
			--Otherwise hide the texture and move the text left.
			lineFrame.texture:Hide();
			lineFrame.fs:ClearAllPoints();
			lineFrame.fs:SetPoint("LEFT", 0, 0);
			lineFrame:SetWidth(lineFrame.fs:GetWidth() + 4);
		end
		tinsert(frame.queue, 1, t);
		UIFrameFadeIn(lineFrame, SCT.CONSTANTS.FRAME_FADE_IN_DURATION, 0, 1);
		frame.sortLineFrames();
		lineFrame:Show();
	end
	
	frame.animationSpeed = 50;
	frame.growthDirection = 1;
	frame.alignment = 1;
	
	frame.updateAnimationSettings = function(growthDirection, animationSpeed, alignment)
		if (growthDirection) then
			frame.growthDirection = growthDirection;
		end
		if (animationSpeed) then
			frame.animationSpeed = animationSpeed;
		end
		if (alignment) then
			frame.alignment = alignment;
		end
		--Points are set again OnUpdate.
		for k, v in pairs(frame.lineFrames) do
			v:ClearAllPoints();
			v:SetScale(frame.lineFrameScale);
			if (frame.alignment == 1) then
				--Left.
				v:SetPoint("LEFT", frame, "LEFT", 1, 0);
			elseif (frame.alignment == 2) then
				--Middle.
				v:SetPoint("CENTER", frame, 0, 0);
			elseif (frame.alignment == 3) then
				--Right.
				v:SetPoint("RIGHT", frame, -1, 0);
			end
			v.fs:SetFont(SCT.LSM:Fetch("font", frame.lineFrameFont), frame.lineFrameFontSize, frame.lineFrameFontOutline);
		end
	end
	
	frame.sortLineFrames = function()
		local currentTime = GetTime();
		local frameHeight = frame:GetHeight();
		local lineHeight = frame.lineFrameHeight;
		local growthDir = frame.growthDirection;
		
		-- Process queue in reverse to avoid index shifting issues
		for i = #frame.queue, 1, -1 do
			local v = frame.queue[i];
			local lineFrame = v.lineFrame;
			local elapsed = currentTime - v.elapsed;
			local offset = elapsed * frame.animationSpeed;
			
			-- Remove if done animating
			if (v.y >= frameHeight - lineHeight) then
				lineFrame.enabled = false;
				UIFrameFadeOut(lineFrame, SCT.CONSTANTS.FRAME_FADE_OUT_DURATION, 1, 0);
				SCT.CreateTimer(SCT.CONSTANTS.FRAME_FADE_DELAY, function()
					lineFrame:Hide();
					-- Add to pool for reuse
					table.insert(lineFramePool, lineFrame);
				end)
				table.remove(frame.queue, i);
			else
				-- Update position
				if (growthDir == 1) then
					lineFrame:SetPoint("BOTTOM", frame, 0, offset);
				else
					lineFrame:SetPoint("TOP", frame, 0, -offset);
				end
				v.y = offset;
			end
		end
		
		-- Check for overlapping frames (only if we have multiple frames)
		if #frame.queue > 1 then
			for i = 2, #frame.queue do
				local v = frame.queue[i];
				local last = frame.queue[i - 1];
				local lineFrame = v.lineFrame;
				
				if (v.y - last.y < lineHeight) then
					v.y = last.y + lineHeight + 1;
					if (growthDir == 1) then
						lineFrame:SetPoint("BOTTOM", frame, 0, v.y);
					else
						lineFrame:SetPoint("TOP", frame, 0, -v.y);
					end
				end
			end
		end
	end
	
	frame:SetScript("OnUpdate", function(self, elapsed)
		-- Throttle updates to improve performance
		self.updateAccumulator = (self.updateAccumulator or 0) + elapsed;
		if self.updateAccumulator >= 0.016 then -- ~60 FPS
			frame.sortLineFrames();
			self.updateAccumulator = 0;
		end
	end)
	
	frame.OnMouseUpFunc = function(self, button)
		if (button == "LeftButton" and frame.isMoving) then
			frame:StopMovingOrSizing();
			frame.isMoving = false;
			if (frame.firstRun) then
				SCT:print("Type /sct config to lock frames once you're done dragging them.");
			end
			frame.firstRun = nil
			frame:SetUserPlaced(false);
			if SCT.db and SCT.db.global then
				SCT.db.global[frame:GetName() .. "_point"], _, SCT.db.global[frame:GetName() .. "_relativePoint"], 
						SCT.db.global[frame:GetName() .. "_x"], SCT.db.global[frame:GetName() .. "_y"] = frame:GetPoint();
		end
	end
	end
	
	frame.OnMouseDownFunc = function(self, button)
		if (button == "LeftButton" and not frame.isMoving) then
			if (not frame.locked) then
				frame:StartMoving();
				frame.isMoving = true;
			end
		end
	end
	
	frame.OnHideFunc = function(self, button)
		if (frame.isMoving) then
			frame:StopMovingOrSizing();
			frame.firstRun = nil
			frame.isMoving = false;
		end
	end
	
	frame:SetScript("OnMouseDown", function(self, button)
		frame.OnMouseDownFunc(self, button);
	end)
	
	frame:SetScript("OnMouseUp", function(self, button)
		frame.OnMouseUpFunc(self, button);
	end)
	
	frame:SetScript("OnHide", function(self)
		frame.OnHideFunc(frame);
	end)
	
	if (SCT.db and SCT.db.global and SCT.db.global[frame:GetName() .. "_point"]) then
		frame.ignoreFramePositionManager = true;
		frame:ClearAllPoints();
		frame:SetPoint(SCT.db.global[frame:GetName() .. "_point"], nil, SCT.db.global[frame:GetName() .. "_relativePoint"], 
				SCT.db.global[frame:GetName() .. "_x"], SCT.db.global[frame:GetName() .. "_y"]);
		frame:SetUserPlaced(false);
		frame.firstRun = nil;
	end
	
	frame.displayTab = CreateFrame("Frame", "$parentDisplayTab", frame);
	if SCT.IS_RETAIL and BackdropTemplate then
		-- BackdropTemplate is already applied in CreateFrame
	else
		-- Apply backdrop manually for older versions
		if frame.displayTab.SetBackdrop then
			frame.displayTab:SetBackdrop({
				bgFile = "Interface\\Buttons\\WHITE8x8",
				insets = {top = 0, left = 0, bottom = 0, right = 0},
				edgeFile = [[Interface/Buttons/WHITE8X8]],
				edgeSize = 1,
			});
		end
	end
	frame.displayTab:SetSize(width, height);
	-- Set backdrop color only if the method exists
	if frame.displayTab.SetBackdropColor then
		frame.displayTab:SetBackdropColor(0, 0, 0, 0.8);
		frame.displayTab:SetBackdropBorderColor(1, 1, 1, 1);
	end
	frame.displayTab:SetAllPoints();
	frame.displayTab:SetFrameStrata("HIGH");
	frame.displayTab.fs = frame.displayTab:CreateFontString("$parentFS", "ARTWORK");
	frame.displayTab.fs:SetPoint("CENTER", 0, 0);
	frame.displayTab.fs:SetFont(SCT.regionFont, frame:GetHeight() - 8);
	frame.displayTab:SetMovable(true);
	frame.displayTab:EnableMouse(true);
	frame.displayTab:SetUserPlaced(false);
	frame.displayTab:SetScript("OnMouseDown", function(self, button)
		frame.OnMouseDownFunc(self, button);
	end)
	frame.displayTab:SetScript("OnMouseUp", function(self, button)
		frame.OnMouseUpFunc(self, button);
	end)
	frame.displayTab:SetScript("OnHide", function(self)
		frame.OnHideFunc(self);
	end)
	
	frame.displayTab.top = SCT:createMoveMeFrame(frame, 200, 50);
	frame.displayTab:Hide();
	frame.displayTab.top:Hide();
	
	return frame;
end

function SCT:createMoveMeFrame(parent, width, height)
	local frame = CreateFrame("Frame", "$ParentTop", parent);
	if SCT.IS_RETAIL and BackdropTemplate then
		-- BackdropTemplate is already applied in CreateFrame
	else
		-- Apply backdrop manually for older versions
		if frame.SetBackdrop then
			frame:SetBackdrop({
				bgFile = "Interface\\Buttons\\WHITE8x8",
				edgeFile = "Interface\\Addons\\ScrollingChatText\\Media\\UI-Tooltip-Border-NoBottom",
				tileEdge = true,
				edgeSize = 16,
				insets = {top = 5, left = 2, bottom = 5, right = 2},
			});
		end
	end
	frame:SetSize(width, height);
	frame:SetPoint("BOTTOM", parent, "TOP", 0, -4);
	-- Set backdrop color only if the method exists
	if frame.SetBackdropColor then
		frame:SetBackdropColor(0, 0, 0, 0.8);
		frame:SetBackdropBorderColor(1, 1, 1, 0.8);
	end
	frame:SetFrameStrata("HIGH");
	
	frame.fs = frame:CreateFontString("$parentFS", "ARTWORK");
	frame.fs:SetPoint("TOP", frame, "TOP", 0, -8);
	frame.fs:SetFont(SCT.regionFont, 12);
	
	frame.fs2 = frame:CreateFontString("$parentFS2", "ARTWORK");
	frame.fs2:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", 6, 8);
	frame.fs2:SetFont(SCT.regionFont, 11);
	
	-- Create Config button (settings icon) - positioned at top right
	frame.configButton = CreateFrame("Button", "$parentConfigButton", frame, "UIPanelButtonTemplate");
	frame.configButton:SetSize(20, 16);
	frame.configButton:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -6, -8);
	
	-- Create the config icon using a Classic WoW compatible texture
	local configIcon = frame.configButton:CreateTexture(nil, "ARTWORK");
	configIcon:SetTexture("Interface\\Buttons\\UI-OptionsButton");
	configIcon:SetSize(10, 10);
	configIcon:SetPoint("CENTER", frame.configButton, "CENTER", 0, 0);
	configIcon:SetTexCoord(0.1, 0.9, 0.1, 0.9);
	-- Make the gear icon white
	configIcon:SetVertexColor(1, 1, 1);
	
	frame.configButton:SetScript("OnClick", function(self)
		if SCT.acd then
			SCT.acd:Open("ScrollingChatText");
		end
	end);
	
	-- Add tooltip to config button
	frame.configButton:SetScript("OnEnter", function(self)
		GameTooltip:SetOwner(self, "ANCHOR_TOP");
		GameTooltip:SetText("Open Settings");
		GameTooltip:Show();
	end);
	
	frame.configButton:SetScript("OnLeave", function(self)
		GameTooltip:Hide();
	end);
	
	-- Create lock button
	frame.lockButton = CreateFrame("Button", "$parentLockButton", frame, "UIPanelButtonTemplate");
	frame.lockButton:SetSize(35, 18);
	frame.lockButton:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -6, 8);
	frame.lockButton:SetText("Lock");
	-- Set initial text color to white
	local lockText = frame.lockButton:GetFontString();
	if lockText then
		lockText:SetTextColor(1, 1, 1);
	end
	frame.lockButton:SetScript("OnClick", function(self)
		if SCT.db.global.lockAllFrames then
			SCT:setLockAllFrames(nil, false);
		else
			SCT:setLockAllFrames(nil, true);
		end
	end)
	
	-- Create test button
	frame.testButton = CreateFrame("Button", "$parentTestButton", frame, "UIPanelButtonTemplate");
	frame.testButton:SetSize(35, 18);
	frame.testButton:SetPoint("BOTTOMRIGHT", frame.lockButton, "BOTTOMLEFT", -3, 0);
	frame.testButton:SetText("Test");
	-- Set initial text color to white
	local testText = frame.testButton:GetFontString();
	if testText then
		testText:SetTextColor(1, 1, 1);
	end
	frame.testButton:SetScript("OnClick", function(self)
		if SCT:getSctTestState() then
			SCT:stopSctTest();
		else
			SCT:startSctTest();
		end
	end)
	
	frame:SetMovable(true);
	frame:EnableMouse(true);
	frame:SetUserPlaced(false);
	
	frame:SetScript("OnMouseDown", function(self, button)
		parent.OnMouseDownFunc(parent, button);
	end)
	frame:SetScript("OnMouseUp", function(self, button)
		parent.OnMouseUpFunc(parent, button);
	end)
	frame:SetScript("OnHide", function(self)
		parent.OnHideFunc(parent);
	end)
	
	-- Initialize button states based on saved configuration
	frame.initializeButtonStates = function()
		if SCT.db.global and SCT.db.global.lockAllFrames then
			frame.lockButton:SetText("Unlock");
			parent.locked = true;
		else
			frame.lockButton:SetText("Lock");
			parent.locked = false;
		end
	end
	
	frame.updateTooltip = function(tooltip)
		frame:SetScript("OnEnter", function(self)
			GameTooltip:SetOwner(self, "ANCHOR_TOP");
			GameTooltip:SetText(tooltip);
				GameTooltip:Show();
		end)
		frame:SetScript("OnLeave", function(self)
			GameTooltip:Hide();
		end)
	end
	
	return frame;
end

-- Animation update function for animated emojis
function SCT:updateAnimatedEmojis(elapsed)
	if not SCT.db.global.sctEnableEmojis then
		return;
	end
	
	SCT_TimeSinceLastUpdate = SCT_TimeSinceLastUpdate + elapsed;
	
	-- Update every ~30fps (0.033 seconds)
	if SCT_TimeSinceLastUpdate >= 0.033 then
		SCT_T = SCT_T + SCT_TimeSinceLastUpdate;
		
		-- Update animated emotes in our scrolling chat frames
		if SCT.chatMessageFrame and SCT.chatMessageFrame.lineFrames then
			for _, lineFrame in ipairs(SCT.chatMessageFrame.lineFrames) do
				if lineFrame:IsShown() and lineFrame.fs then
					SCT:updateEmoteInFontString(lineFrame.fs, 28, 28);
				end
			end
		end
		
		SCT_TimeSinceLastUpdate = 0;
	end
end

-- Update animated emote in a font string (similar to TwitchEmotes)
function SCT:updateEmoteInFontString(fontstring, widthOverride, heightOverride)
	if not _G["TwitchEmotes_animation_metadata"] then
		return;
	end
	
	local txt = fontstring:GetText();
	if not txt then
		return;
	end
	
	for emoteTextureString in txt:gmatch("(|TInterface\\AddOns\\TwitchEmotes\\Emotes.-|t)") do
		local imagepath = emoteTextureString:match("|T(Interface\\AddOns\\TwitchEmotes\\Emotes.-.tga).-|t");
		
		if imagepath then
			local animdata = _G["TwitchEmotes_animation_metadata"][imagepath];
			if animdata then
				local framenum = SCT:getCurrentFrameNum(animdata);
				local nTxt;
				
				if widthOverride or heightOverride then
					nTxt = txt:gsub(SCT:escapePattern(emoteTextureString),
						_G["TwitchEmotes_BuildEmoteFrameStringWithDimensions"](
						imagepath, animdata, framenum, widthOverride, heightOverride));
				else
					nTxt = txt:gsub(SCT:escapePattern(emoteTextureString),
						_G["TwitchEmotes_BuildEmoteFrameString"](
						imagepath, animdata, framenum));
				end
				
				if nTxt ~= txt then
					fontstring:SetText(nTxt);
					txt = nTxt;
				end
			end
		end
	end
end

-- Get current frame number for animation
function SCT:getCurrentFrameNum(animdata)
	if animdata.pingpong then
		local vframen = math.floor((SCT_T * animdata.framerate) % ((animdata.nFrames * 2) - 1));
		if vframen > animdata.nFrames then
			vframen = animdata.nFrames - (vframen % animdata.nFrames);
		end
		return vframen;
	end
	
	return math.floor((SCT_T * animdata.framerate) % animdata.nFrames);
end

-- Escape pattern for string replacement
function SCT:escapePattern(x)
	return x:gsub('%+', '%%+'):gsub('%-', '%%-');
end