-- Emoji support for ScrollingChatText
-- Integrates with TwitchEmotes and other emoji addons

local SCT = SCT or {};

-- Check if TwitchEmotes is loaded
local function isTwitchEmotesLoaded()
	return _G["TwitchEmotes_defaultpack"] ~= nil;
end



-- Process emojis in text (TwitchEmotes style)
function SCT:processEmojis(text)
	if not text or text == "" then
		return text;
	end
	
	local processedText = text;
	
	-- Process TwitchEmotes if available
	if isTwitchEmotesLoaded() then
		processedText = SCT:processTwitchEmotes(processedText);
	end
	
	-- Process other emoji addons here if needed
	-- if isEmojiAddonLoaded("ElvUI_ChatTweaks") then
	--     processedText = SCT:processElvUIChatTweaks(processedText);
	-- end
	
	return processedText;
end

-- Cache for emoji processing to avoid repeated lookups
local emojiCache = {};

-- Process TwitchEmotes specifically
function SCT:processTwitchEmotes(text)
	if not _G["TwitchEmotes_defaultpack"] or not _G["TwitchEmotes_emoticons"] then
		return text;
	end
	
	-- Check cache first
	if emojiCache[text] then
		return emojiCache[text];
	end
	

	
	local delimiters = "%s,'<>?-!";
	local processedText = text;
	local emoticons = _G["TwitchEmotes_emoticons"];
	local defaultpack = _G["TwitchEmotes_defaultpack"];
	local animation_metadata = _G["TwitchEmotes_animation_metadata"];
	
	-- Process each word in the text
	for word in string.gmatch(text, "[^" .. delimiters .. "]+") do
		-- Check cache for this word
		if emojiCache[word] then
			processedText = string.gsub(processedText, word, emojiCache[word], 1);
		else
			-- Look up the word in TwitchEmotes_emoticons to get the canonical emote name
			local canonicalEmote = emoticons[word];
			
			if canonicalEmote then
				-- Look up the canonical name in TwitchEmotes_defaultpack to get the texture path
				local emote = defaultpack[canonicalEmote];
				
				if emote then
					-- Get the emote path and size
					local path_and_size = emote;
					local path = string.match(path_and_size, "(.*%.tga)");
					local size = string.match(path_and_size, ":(.*)");
					
					-- Check if this is an animated emote
					local animdata = animation_metadata and animation_metadata[path];
					
					if animdata then
						-- Use animated emote format (already includes |T and |t)
						path_and_size = _G["TwitchEmotes_BuildEmoteFrameString"](path, animdata, 0);
					else
						-- Use standard size (28:28) for static emotes
						if size and size ~= "28:28" then
							path_and_size = string.gsub(emote, size, "28:28");
						end
						-- Wrap static emotes in |T and |t tags
						path_and_size = "|T" .. path_and_size .. "|t";
					end
					
					-- Cache the result for this word
					emojiCache[word] = path_and_size;
					
					-- Replace the word with the texture
					processedText = string.gsub(processedText, word, path_and_size, 1);
				end
			end
		end
	end
	
	-- Cache the full result (limit cache size to prevent memory issues)
	if #emojiCache > 1000 then
		-- Clear cache if it gets too large
		emojiCache = {};
	end
	emojiCache[text] = processedText;
	
	return processedText;
end

-- Check if emoji support is enabled
function SCT:isEmojiSupportEnabled()
	return SCT.db.global and SCT.db.global.sctEnableEmojis;
end

 