--------------------------------------------
---Scrolling Chat Text-------------------
--------------------------------------------

local addonName, SCT = ...;
local chatMessageFrame;
local doSCT;
local lineFrameHeight = SCT.CONSTANTS.DEFAULT_LINE_HEIGHT;
local sctEventsFrame = CreateFrame("Frame", "SCTChatMessages");
local testRunning, testRunningTimer;

local GetTime = GetTime;

function SCT:loadScrollingChatFrames()
	if (not SCT.db.global or not SCT.db.global.sctEnabled) then
		SCT:updateSctState();
		return;
	end
	if (not chatMessageFrame) then
		local scrollHeight = SCT.db.global and SCT.db.global.sctScrollHeight or SCT.CONSTANTS.DEFAULT_FRAME_HEIGHT;
		local lineHeight = SCT.db.global and SCT.db.global.sctLineFrameHeight or SCT.CONSTANTS.DEFAULT_LINE_HEIGHT;
		chatMessageFrame = SCT:createAutoScrollingFrame("SCTChatMessages", SCT.CONSTANTS.DEFAULT_FRAME_WIDTH, scrollHeight, SCT.CONSTANTS.DEFAULT_FRAME_X, SCT.CONSTANTS.DEFAULT_FRAME_Y, lineHeight);
		SCT.chatMessageFrame = chatMessageFrame;
		lineFrameHeight = lineHeight;
		chatMessageFrame.lineFrameHeight = lineHeight;
		-- Set backdrop color only if the method exists
		if chatMessageFrame.SetBackdropColor then
			chatMessageFrame:SetBackdropColor(0, 0, 0, 0);
		end
		
		-- Initialize button states based on saved configuration
		if chatMessageFrame.displayTab and chatMessageFrame.displayTab.top then
			chatMessageFrame.displayTab.top:initializeButtonStates();
		end
		
		local point, _, relativePoint, x, y = chatMessageFrame:GetPoint();
		if (point == "CENTER" and x == chatMessageFrame.defaultX and y == chatMessageFrame.defaultY) then
			chatMessageFrame.firstRun = true;
			-- On first run, respect the saved lock state instead of forcing unlock
			SCT:sctUpdateFrameLocks();
		else
			SCT:sctUpdateFrameLocks();
		end
	end
	SCT:updateSctState();
	SCT:sctUpdateSettings();
	SCT:sctUpdateFrameLocks();
	SCT:sctUpdateFrameLocksLayout();
end

function SCT:openScrollingChatFrame()
	if (not chatMessageFrame) then
		SCT:loadScrollingChatFrames();
	end
	if (not chatMessageFrame) then
		return;
	end
	if (chatMessageFrame:IsShown()) then
		chatMessageFrame:Hide();
	else
		chatMessageFrame:Show();
		if (SCT.IsSettingsOpen()) then
			chatMessageFrame:SetFrameStrata("DIALOG");
		else
			chatMessageFrame:SetFrameStrata("MEDIUM");
		end
	end
end

function SCT:updateSctState()
	local enabled;
	if (not SCT.db.global or not SCT.db.global.sctEnabled) then
		if (chatMessageFrame) then
			chatMessageFrame:Hide();
		end
		doSCT = false;
		SCT.doSCT = false;
		return;
	end
	
	-- Check if any chat types are enabled
	local anyChatEnabled = false;
	if SCT.db.global then
		anyChatEnabled = SCT.db.global.sctGuildEnabled or 
						SCT.db.global.sctPartyEnabled or 
						SCT.db.global.sctRaidEnabled or 
						SCT.db.global.sctInstanceEnabled or 
						SCT.db.global.sctSayEnabled or 
						SCT.db.global.sctYellEnabled or 
						SCT.db.global.sctWhisperEnabled or 
						SCT.db.global.sctOfficerEnabled;
		

	end
	
	enabled = true;
	
	-- Always enable if test is running or it's first run
	if (testRunning or (chatMessageFrame and chatMessageFrame.firstRun)) then
		enabled = true;
	elseif (chatMessageFrame and chatMessageFrame.locked) then
		-- If frame is locked, only enable if chat types are enabled
		enabled = anyChatEnabled;
	else
		-- If frame is not locked, enable if chat types are enabled
		enabled = anyChatEnabled;
	end
	
	if (enabled) then
		if (chatMessageFrame) then
			chatMessageFrame:Show();
		end
		doSCT = true;
		SCT.doSCT = true;
	else
		if (chatMessageFrame) then
			chatMessageFrame:Hide();
		end
		doSCT = false;
		SCT.doSCT = false;
	end
end

function SCT:sctUpdateSettings()
	if (chatMessageFrame and SCT.db.global) then
		chatMessageFrame.updateAnimationSettings(SCT.db.global.sctGrowthDirection, SCT.db.global.sctAnimationSpeed, SCT.db.global.sctAlignment);
		lineFrameHeight = SCT.db.global.sctLineFrameHeight;
		chatMessageFrame.lineFrameHeight = SCT.db.global.sctLineFrameHeight;
		chatMessageFrame.lineFrameFont = (SCT.db and SCT.db.global and SCT.db.global.sctFont) or "SCT Default";
		chatMessageFrame.lineFrameScale = SCT.db.global.sctLineFrameScale;
		chatMessageFrame.lineFrameFontOutline = (SCT.db and SCT.db.global and SCT.db.global.sctFontOutline) or "NONE";
		chatMessageFrame:SetHeight(SCT.db.global.sctScrollHeight);
		chatMessageFrame:SetAlpha(SCT.db.global.sctAlpha);
		
		-- Update main frame background
		if chatMessageFrame.SetBackdropColor then
			if SCT.db.global.sctShowBackground then
				chatMessageFrame:SetBackdropColor(0, 0, 0, SCT.CONSTANTS.FRAME_BACKGROUND_ALPHA);
			else
				chatMessageFrame:SetBackdropColor(0, 0, 0, 0);
			end
		elseif chatMessageFrame.bgTexture then
			-- Fallback for older versions using background texture
			if SCT.db.global.sctShowBackground then
				chatMessageFrame.bgTexture:SetVertexColor(0, 0, 0, SCT.CONSTANTS.FRAME_BACKGROUND_ALPHA);
				chatMessageFrame.bgTexture:Show();
			else
				chatMessageFrame.bgTexture:SetVertexColor(0, 0, 0, 0);
				chatMessageFrame.bgTexture:Hide();
			end
		end
		
		-- Update background visibility for all line frames (only if background setting changed)
		if chatMessageFrame.lineFrames and chatMessageFrame.lastBackgroundSetting ~= SCT.db.global.sctShowBackground then
			chatMessageFrame.lastBackgroundSetting = SCT.db.global.sctShowBackground;
			for _, lineFrame in pairs(chatMessageFrame.lineFrames) do
				if lineFrame.SetBackdropColor then
					if SCT.db.global.sctShowBackground then
						lineFrame:SetBackdropColor(0, 0, 0, 0.4);
					else
						lineFrame:SetBackdropColor(0, 0, 0, 0);
					end
				elseif lineFrame.bgTexture then
					-- Fallback for older versions using background texture
					if SCT.db.global.sctShowBackground then
						lineFrame.bgTexture:SetVertexColor(0, 0, 0, 0.4);
						lineFrame.bgTexture:Show();
					else
						lineFrame.bgTexture:SetVertexColor(0, 0, 0, 0);
						lineFrame.bgTexture:Hide();
					end
				end
			end
		end
		
		chatMessageFrame.updateAnimationSettings();
	end
end

function SCT:sctUpdateFrameLocks()
	if (chatMessageFrame and SCT.db.global) then
		chatMessageFrame.locked = SCT.db.global.lockAllFrames;
	end
	SCT:sctUpdateFrameLocksLayout();
end

function SCT:sctUpdateFrameLocksLayout()
	SCT:updateSctState();
	if (chatMessageFrame) then
		local frame = chatMessageFrame;
		if (frame) then
			-- Update lock button text based on saved configuration (always)
			if frame.displayTab and frame.displayTab.top and frame.displayTab.top.lockButton then
				if SCT.db.global and SCT.db.global.lockAllFrames then
					frame.displayTab.top.lockButton:SetText("Unlock");
				else
					frame.displayTab.top.lockButton:SetText("Lock");
				end
			end
			
			if (frame.locked) then
				frame.displayTab:Hide();
				frame.displayTab.top:Hide();
				frame:EnableMouse(false);
			else
				frame.displayTab.top:ClearAllPoints();
				frame.displayTab.top:SetPoint("BOTTOM", frame, "TOP", 0, -1.5);
				-- Set backdrop only if the method exists
				if frame.displayTab.top.SetBackdrop then
					frame.displayTab.top:SetBackdrop({
						bgFile = "Interface\\Buttons\\WHITE8x8",
						edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
						tileEdge = true,
						edgeSize = 8,
						insets = {top = 3, left = 3, bottom = 3, right = 3},
					});
					frame.displayTab.top:SetBackdropColor(0, 0, 0, SCT.CONSTANTS.DISPLAY_TAB_BACKGROUND_ALPHA);
				else
					-- Fallback for older versions: create a background texture
					if not frame.displayTab.top.bgTexture then
						frame.displayTab.top.bgTexture = frame.displayTab.top:CreateTexture(nil, "BACKGROUND");
						frame.displayTab.top.bgTexture:SetTexture("Interface\\Buttons\\WHITE8x8");
						frame.displayTab.top.bgTexture:SetAllPoints();
					end
					frame.displayTab.top.bgTexture:SetVertexColor(0, 0, 0, SCT.CONSTANTS.DISPLAY_TAB_BACKGROUND_ALPHA);
				end
				frame.displayTab:SetAlpha(SCT.CONSTANTS.DISPLAY_TAB_ALPHA);
				frame.displayTab.top.fs:SetText("|cFFDEDE42Chat Messages|r");
				frame.displayTab.top.fs2:SetText("|cFF9CD6DE" .. "Drag Me" .. "|r");
				
				frame.displayTab:Show();
				frame.displayTab.top:Show();
				frame:EnableMouse(true);
			end
		end
	end
end

function SCT:sctLockFrames()
	if chatMessageFrame then
		SCT:setLockAllFrames(nil, true);
	end
end

function SCT:sctUnlockFrames()
	if chatMessageFrame then
		SCT:setLockAllFrames(nil, false);
	end
end

function SCT:getSctLockState()
	if (chatMessageFrame) then
		return chatMessageFrame.locked;
	end
end

function SCT:updateTestButtonStates()
	if (chatMessageFrame and chatMessageFrame.displayTab and chatMessageFrame.displayTab.top) then
		local testButton = chatMessageFrame.displayTab.top.testButton;
		local lockButton = chatMessageFrame.displayTab.top.lockButton;
		
		if testButton then
			if testRunning then
				testButton:SetText("Stop");
				-- Change text color to red to indicate "Stop" state
				local text = testButton:GetFontString();
				if text then
					text:SetTextColor(unpack(SCT.CONSTANTS.BUTTON_TEXT_RED));
				end
			else
				testButton:SetText("Test");
				-- Keep white color as default
				local text = testButton:GetFontString();
				if text then
					text:SetTextColor(unpack(SCT.CONSTANTS.BUTTON_TEXT_WHITE));
				end
			end
		end
		
		if lockButton then
			if testRunning then
				lockButton:Disable();
				-- Get the button's text font string and change its color
				local text = lockButton:GetFontString();
				if text then
					text:SetTextColor(unpack(SCT.CONSTANTS.BUTTON_TEXT_GRAY));
				end
			else
				lockButton:Enable();
				-- Get the button's text font string and change its color
				local text = lockButton:GetFontString();
				if text then
					text:SetTextColor(unpack(SCT.CONSTANTS.BUTTON_TEXT_WHITE));
				end
			end
		end
	end
end

local sentEventCache = {};
function SCT:sctSendEvent(text, icon, senderName)
	if (doSCT or testRunning) then
		if (chatMessageFrame) then
				--Events almost always include a player name so this text msg should be unique enough to be used as a cooldown check for duplicate events.
	if (not sentEventCache[text] or sentEventCache[text] < GetTime() - SCT.CONSTANTS.DUPLICATE_EVENT_THRESHOLD) then
		-- Limit cache size to prevent memory issues
		local cacheSize = 0;
		for _ in pairs(sentEventCache) do
			cacheSize = cacheSize + 1;
		end
		if cacheSize > 1000 then
			sentEventCache = {};
		end
				sentEventCache[text] = GetTime();
				chatMessageFrame.addLine(text, icon);
			end
		end
	end
end

function SCT:sctChatMessageEvent(senderName, senderClass, message, chatType, channelName)
	-- Early exit if addon is disabled
	if not SCT.db.global or not SCT.db.global.sctEnabled then
		return;
	end
	
	-- Early chat type filtering - check if this chat type is enabled
	local chatTypeEnabled = false;
	if chatType == "CHAT_MSG_GUILD" then
		chatTypeEnabled = SCT.db.global.sctGuildEnabled;
	elseif chatType == "CHAT_MSG_OFFICER" then
		chatTypeEnabled = SCT.db.global.sctOfficerEnabled;
	elseif chatType == "CHAT_MSG_PARTY" then
		chatTypeEnabled = SCT.db.global.sctPartyEnabled;
	elseif chatType == "CHAT_MSG_RAID" then
		chatTypeEnabled = SCT.db.global.sctRaidEnabled;
	elseif chatType == "CHAT_MSG_INSTANCE_CHAT" then
		chatTypeEnabled = SCT.db.global.sctInstanceEnabled;
	elseif chatType == "CHAT_MSG_SAY" then
		chatTypeEnabled = SCT.db.global.sctSayEnabled;
	elseif chatType == "CHAT_MSG_YELL" then
		chatTypeEnabled = SCT.db.global.sctYellEnabled;
	elseif chatType == "CHAT_MSG_WHISPER" then
		chatTypeEnabled = SCT.db.global.sctWhisperEnabled;
	end
	
	-- Early exit if chat type is disabled
	if not chatTypeEnabled then
		return;
	end
	
	-- Check if this is the player's own message and if we should show it
	local playerName = UnitName("player");
	if senderName == playerName and not SCT.db.global.sctShowMyMessages then
		return; -- Skip player's own messages if disabled
	end
	
	-- Check if message should be filtered (only if we have filters)
	if SCT.db.global.messageFilters and #SCT.db.global.messageFilters > 0 then
		if SCT:shouldFilterMessage(message) then
			return; -- Skip this message
		end
	end
	

	
	-- Remove server name from sender name (everything after the dash)
	local cleanName = senderName:match("^([^-]+)") or senderName;
	
	-- Process emojis in the message if enabled (only process the message part, not the entire text)
	local processedMessage = message;
	-- Process emojis in the message if enabled
	if SCT:isEmojiSupportEnabled() then
		processedMessage = SCT:processEmojis(message);
	end
	
	local _, _, _, classHex = SCT:getClassColor(senderClass);
	
	-- Get chat color based on chat type using lookup table for better performance
	local chatColorLookup = {
		["CHAT_MSG_GUILD"] = "|cff00ff00", -- Green
		["CHAT_MSG_OFFICER"] = "|cff00ff00", -- Green
		["CHAT_MSG_PARTY"] = "|cff00aaff", -- Blue
		["CHAT_MSG_RAID"] = "|cffff7f00", -- Orange
		["CHAT_MSG_INSTANCE_CHAT"] = "|cff00aaff", -- Blue
		["CHAT_MSG_SAY"] = "|cffffffff", -- White
		["CHAT_MSG_YELL"] = "|cffff0000", -- Red
		["CHAT_MSG_WHISPER"] = "|cffff00ff", -- Purple
	};
	local chatColor = chatColorLookup[chatType] or "|cffffffff"; -- Default white
	local chatPrefix = "";
	

	
	-- Only process prefixes if enabled
	if SCT.db.global.sctShowChatPrefixes then
		-- Use a lookup table for better performance
		local prefixLookup = {
			["CHAT_MSG_GUILD"] = "[Guild] ",
			["CHAT_MSG_OFFICER"] = "[Officer] ",
			["CHAT_MSG_PARTY"] = "[Party] ",
			["CHAT_MSG_RAID"] = "[Raid] ",
			["CHAT_MSG_INSTANCE_CHAT"] = "[Instance] ",
			["CHAT_MSG_SAY"] = "[Say] ",
			["CHAT_MSG_YELL"] = "[Yell] ",
			["CHAT_MSG_WHISPER"] = "[Whisper] ",
		};
		chatPrefix = prefixLookup[chatType] or "";
	end
	
	local text = chatPrefix .. "|cff" .. classHex .. cleanName .. "|r: " .. chatColor .. processedMessage .. "|r";
	SCT:sctSendEvent(text, nil, senderName);
end



local function chatEventUnfiltered(...)
	if (doSCT) then
		local args = {...};
		local event = args[1];
		
		-- Handle different chat types based on settings
		if (event == "CHAT_MSG_GUILD" and SCT.db.global.sctGuildEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_OFFICER" and SCT.db.global.sctOfficerEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_PARTY" and SCT.db.global.sctPartyEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_RAID" and SCT.db.global.sctRaidEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_INSTANCE_CHAT" and SCT.db.global.sctInstanceEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_SAY" and SCT.db.global.sctSayEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_YELL" and SCT.db.global.sctYellEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);
		elseif (event == "CHAT_MSG_WHISPER" and SCT.db.global.sctWhisperEnabled) then
			local messageContent = args[2];
			local senderName = args[3];
			local senderGUID = args[13];
			local senderClass = SCT.GetPlayerClass(senderName, senderGUID);
			SCT:sctChatMessageEvent(senderName, senderClass, messageContent, event);

		end
	end
end

-- Register all possible chat events (version-aware)
sctEventsFrame:RegisterEvent("PLAYER_ENTERING_WORLD");
sctEventsFrame:RegisterEvent("CHAT_MSG_GUILD");
sctEventsFrame:RegisterEvent("CHAT_MSG_OFFICER");
sctEventsFrame:RegisterEvent("CHAT_MSG_PARTY");
sctEventsFrame:RegisterEvent("CHAT_MSG_RAID");
sctEventsFrame:RegisterEvent("CHAT_MSG_SAY");
sctEventsFrame:RegisterEvent("CHAT_MSG_YELL");
sctEventsFrame:RegisterEvent("CHAT_MSG_WHISPER");

-- Register version-specific events
if SCT.IS_RETAIL then
    sctEventsFrame:RegisterEvent("CHAT_MSG_INSTANCE_CHAT");
end


sctEventsFrame:SetScript('OnEvent', function(self, event, ...)
	if (event == "CHAT_MSG_GUILD" or event == "CHAT_MSG_OFFICER" or event == "CHAT_MSG_PARTY" or 
		event == "CHAT_MSG_RAID" or event == "CHAT_MSG_SAY" or 
		event == "CHAT_MSG_YELL" or event == "CHAT_MSG_WHISPER") then
		chatEventUnfiltered(event, ...);
	elseif (SCT.IS_RETAIL and event == "CHAT_MSG_INSTANCE_CHAT") then
		chatEventUnfiltered(event, ...);
	elseif (event == "PLAYER_ENTERING_WORLD") then
		-- Clear event cache to prevent memory buildup
		sentEventCache = {};
		-- Limit cache size to prevent memory issues
		if SCT.db.global then
			SCT:updateSctState();
		end
	end
end)

local testTicker;
local function testSct(testID)
	local events = {
		[2] = {
			senderName = "Player1",
			senderClass = "DRUID",
			message = "Hello chat!",
			chatType = "CHAT_MSG_GUILD",
		},
		[5] = {
			senderName = "Player2",
			senderClass = "MAGE",
			message = "Anyone up for a dungeon?",
			chatType = "CHAT_MSG_PARTY",
		},
		[7] = {
			senderName = "Player3",
			senderClass = "WARRIOR",
			message = "Great raid tonight!",
			chatType = "CHAT_MSG_RAID",
		},
		[10] = {
			senderName = "Player4",
			senderClass = "PRIEST",
			message = "Thanks for the help everyone",
			chatType = "CHAT_MSG_SAY",
		},
		[13] = {
			senderName = "Player5",
			senderClass = "HUNTER",
			message = "Looking for group!",
			chatType = "CHAT_MSG_YELL",
		},
		[16] = {
			senderName = "Player6",
			senderClass = "PALADIN",
			message = "Whisper me for details",
			chatType = "CHAT_MSG_WHISPER",
		},

	};
	
	local data = events[testID];
	if (data) then
		local senderName = data.senderName;
		local senderClass = data.senderClass;
		local message = data.message;
		local chatType = data.chatType;
		
		SCT:sctChatMessageEvent(senderName, senderClass, message, chatType, nil);
	end
end

local testID = 0;
function SCT:startSctTicker()
	if (testTicker) then
		return;
	end
	testID = 0;
	testTicker = true;
	SCT:sctTicker();
end

function SCT:stopSctTicker()
	testTicker = nil;
end

function SCT:sctTicker()
	if (testTicker) then
		testID = testID + 1;
		testSct(testID);
		SCT.CreateTimer(SCT.CONSTANTS.TEST_INTERVAL, function()
			SCT:sctTicker();
		end)
	end
end

function SCT:startSctTest(quiet)
	if (not SCT.db.global or not SCT.db.global.sctEnabled) then
		SCT:print("Scrolling Chat Text is not enabled.");
		return;
	end
	if (testRunningTimer) then
		testRunningTimer:Cancel();
	end
	testRunningTimer = SCT.CreateTicker(SCT.CONSTANTS.TEST_DURATION, function()
		SCT:stopSctTest();
	end, 1)
	if (not quiet) then
		SCT:print("Started Scrolling Chat Text test for " .. SCT.CONSTANTS.TEST_DURATION .. " seconds.");
	end
	testRunning = true;
	SCT:sctUnlockFrames();
	SCT:startSctTicker();
	SCT:updateTestButtonStates();
	SCT.acr:NotifyChange("ScrollingChatText");
end

function SCT:stopSctTest()
	if (testRunningTimer) then
		testRunningTimer:Cancel();
	end
	if (not SCT.allFramesTestRunning) then
		SCT:print("|cFF00C800Stopped Scrolling Chat Text test.");
	end
	testRunning = nil;
	SCT:stopSctTicker();
	SCT:sctUpdateFrameLocks();
	SCT:updateTestButtonStates();
	SCT.acr:NotifyChange("ScrollingChatText");
end

function SCT:getSctTestState()
	return testRunning;
end 