-- Spell IDs for the Roll The Bones Buffs
local RollTheBones = 193316
local SharkInfestedWaters = "193357"
local TrueBearing = "193359"
local JollyRoger = "199603"
local Broadsides = "193356"
local BuriedTreasure = "199600"
local GrandMelee = "193358"

-- Number of buffs active. Used to determine if we have a 1, 2 or 5 roll. This is calculated after every Roll The Bones use.
local amount = 0

-- EditBox to show statistics which can be copied to the clipboard.
local TextFrame = CreateFrame("Frame", "RTBS_TextFrame", UIParent)
TextFrame:SetWidth(350)
TextFrame:SetHeight(400)
TextFrame:SetPoint("LEFT", UIParent, "LEFT", 3, 10)
TextFrame:SetFrameStrata("DIALOG")
TextFrame:SetBackdrop({
	bgFile = [[Interface\Buttons\WHITE8x8]],
	insets = {left = 3, right = 3, top = 4, bottom = 3}
})
TextFrame:SetBackdropColor(0, 0, 0, 0.7)

--Make movable
TextFrame:EnableMouse(true)
TextFrame:SetMovable(true)
TextFrame:RegisterForDrag("LeftButton")
TextFrame:SetScript("OnDragStart", TextFrame.StartMoving)
TextFrame:SetScript("OnDragStop", TextFrame.StopMovingOrSizing)

-- Create the EditBox
TextBox = CreateFrame("EditBox", "RTBS_TextBox", TextFrame)
TextBox:SetWidth(400)
TextBox:SetHeight(400)
TextBox:SetMaxLetters(99999)
TextBox:SetMultiLine(true)
TextBox:SetAutoFocus(false)
TextBox:SetFont("Fonts\\ARIALN.ttf", 14, "THINOUTLINE")
TextBox:SetScript("OnEscapePressed", function() TextBox:ClearFocus() end)

-- Create the ScrollFrame
local ScrollFrame = CreateFrame("ScrollFrame", "RTBS_TextBoxScroll", TextFrame, "UIPanelScrollFrameTemplate")
ScrollFrame:SetPoint("TOPLEFT", TextFrame, "TOPLEFT", 8, -30)
ScrollFrame:SetPoint("BOTTOMRIGHT", TextFrame, "BOTTOMRIGHT", -30, 8)
ScrollFrame:SetScrollChild(TextBox)

-- Event Frame definition
local EventFrame = CreateFrame("Frame")
EventFrame:RegisterEvent("ADDON_LOADED")

EventFrame:SetScript("OnEvent", function(self, event, ...)
	if(event == "ADDON_LOADED") then
		-- Only load addon if current class is rogue.
		classDisplayName, class, classID = UnitClass("player")
		if(classID == 4) then
			-- Set event handler for talent switch.
			EventFrame:RegisterEvent("ACTIVE_TALENT_GROUP_CHANGED")

			-- Initialize saved variables if called for the first time.
			SetStandardValues()

			-- Register event if the current specialization is outlaw.
			local currentSpecIndex = GetSpecialization()
			local currentSpecId = currentSpecIndex and select(1, GetSpecializationInfo(currentSpecIndex)) or "None"
			if(currentSpecId == 260) then
				EventFrame:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
			end
		end
	end

	-- Register/unregister events if the current specialization changes.
	if(event == "ACTIVE_TALENT_GROUP_CHANGED") then
		if(GetSpecializationInfo(GetSpecialization()) == 260) then
			EventFrame:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
		else
			EventFrame:UnregisterEvent("UNIT_SPELLCAST_SUCCEEDED")
		end
	end
	
	--[[
		Main event of the addon, each time called the player uses Roll The Bones.
		Here we check which buffs were rolled and add those to the statistics.
	--]]
	if(event == "UNIT_SPELLCAST_SUCCEEDED") then
		 -- Check if the player used Roll The Bones. Return if not.
		local unit = ...
		local unitID, spell, rank, lineID, spellID = select(1, ...)
		if(unit ~= "player" or spellID ~= RollTheBones) then
			return
		end

		RTBS_DB.rolls = RTBS_DB.rolls + 1
		RTBS_DB.overallRolls = RTBS_DB.overallRolls + 1

		-- Set all buffs to inactive
		for buff in pairs(RTBS_DB.buffs) do
			RTBS_DB.buffs[buff]["active"] = false
		end
		-- Buff detection
		for i = 1, 40 do
			local name, rank, icon, count, dispelType, duration, expires, caster, isStealable, nameplateShowPersonal, spellID, canApplyAura, isBossDebuff, _, nameplateShowAll, timeMod, value1, value2, value3 = UnitBuff("player" , i)
			spellID = tostring(spellID)
			if(RTBS_DB.buffs[spellID] ~= nil) then
				RTBS_DB.buffs[spellID]["active"] = true
				RTBS_DB.buffs[spellID]["timesRolled"] = RTBS_DB.buffs[spellID]["timesRolled"] + 1
				amount = amount + 1
			end
		end
		
		-- Ending a streak
		if(amount > 1) then
			if(RTBS_DB.rolls > RTBS_DB.rollsRecord) then
				RTBS_DB.rollsRecord = RTBS_DB.rolls
			end
			RTBS_DB.streakEnd[RTBS_DB.rolls] = RTBS_DB.streakEnd[RTBS_DB.rolls] + 1
			RTBS_DB.endingRolls = RTBS_DB.endingRolls + 1
			RTBS_DB.rolls = 0
		end

		-- Updating buff rolls.
		if(amount == 1) then
			RTBS_DB.num1Buffs = RTBS_DB.num1Buffs + 1
		elseif(amount == 2) then
			RTBS_DB.num2Buffs = RTBS_DB.num2Buffs + 1
		elseif(amount == 5) then
			RTBS_DB.num5Buffs = RTBS_DB.num5Buffs + 1
		end
		
		amount = 0

		-- Updating the textual output
		UpdateTextBox()
	end
end)

function UpdateTextBox()
	local output = ""
	if(RTBS_DB.text == "true") then
		
		output = "Overall Rolls: " .. RTBS_DB.overallRolls .. "\n"
		output = output .. "Single streak record: " .. RTBS_DB.rollsRecord .. "\n\n"
		output = output .. "Individual chances:\n"
		output = output .. "Shark Infested Waters: " .. RTBS_DB.buffs[SharkInfestedWaters]["timesRolled"] .. " (" .. round((RTBS_DB.buffs[SharkInfestedWaters]["timesRolled"] / RTBS_DB.overallRolls * 100), 2) .. "%)\n"
		output = output .. "True Bearing: " .. RTBS_DB.buffs[TrueBearing]["timesRolled"] .. " (" .. round((RTBS_DB.buffs[TrueBearing]["timesRolled"] / RTBS_DB.overallRolls * 100), 2) .. "%)\n"
		output = output .. "Jolly Roger: " .. RTBS_DB.buffs[JollyRoger]["timesRolled"] .. " (" .. round((RTBS_DB.buffs[JollyRoger]["timesRolled"] / RTBS_DB.overallRolls * 100), 2) .. "%)\n"
		output = output .. "Broadsides: " .. RTBS_DB.buffs[Broadsides]["timesRolled"] .. " (" .. round((RTBS_DB.buffs[Broadsides]["timesRolled"] / RTBS_DB.overallRolls * 100), 2) .. "%)\n"
		output = output .. "Buried Treasure: " .. RTBS_DB.buffs[BuriedTreasure]["timesRolled"] .. " (" .. round((RTBS_DB.buffs[BuriedTreasure]["timesRolled"] / RTBS_DB.overallRolls * 100), 2) .. "%)\n"
		output = output .. "Grand Melee: " .. RTBS_DB.buffs[GrandMelee]["timesRolled"] .. " (" .. round((RTBS_DB.buffs[GrandMelee]["timesRolled"] / RTBS_DB.overallRolls * 100), 2) .. "%)\n\n"
		output = output .. "Chances for number of buffs:\n"
		output = output .. "1 Buff: ".. RTBS_DB.num1Buffs .. " (" .. round(RTBS_DB.num1Buffs / RTBS_DB.overallRolls * 100, 2) .. "%)\n"
		output = output .. "2 Buff: ".. RTBS_DB.num2Buffs .. " (" .. round(RTBS_DB.num2Buffs / RTBS_DB.overallRolls * 100, 2) .. "%)\n"
		output = output .. "5 Buff: ".. RTBS_DB.num5Buffs .. " (" .. round(RTBS_DB.num5Buffs / RTBS_DB.overallRolls * 100, 2) .. "%)\n\n"
		output = output .. "Streak ends at:\n"

		for i=1,RTBS_DB.rollsRecord,1 do
			output = output .. i .. " rolls: " .. RTBS_DB.streakEnd[i] .. " (" .. round(RTBS_DB.streakEnd[i] / RTBS_DB.endingRolls * 100, 2) .. "%)\n"
		end
	elseif(RTBS_DB.json == "true") then
		output = "{\n"
		output = output .. "    \"overall_rolls\" : " .. RTBS_DB.overallRolls .. ",\n"
		output = output .. "    \"single_streak_record\" : " .. RTBS_DB.rollsRecord .. ",\n"
		output = output .. "    \"shark_infested_waters\" : " .. RTBS_DB.buffs[SharkInfestedWaters]["timesRolled"] .. ",\n"
		output = output .. "    \"true_bearing\" : " .. RTBS_DB.buffs[TrueBearing]["timesRolled"] .. ",\n"
		output = output .. "    \"jolly_roger\" : " .. RTBS_DB.buffs[JollyRoger]["timesRolled"] .. ",\n"
		output = output .. "    \"broadsides\" : " .. RTBS_DB.buffs[Broadsides]["timesRolled"] .. ",\n"
		output = output .. "    \"buried_treasure\" : " .. RTBS_DB.buffs[BuriedTreasure]["timesRolled"] .. ",\n"
		output = output .. "    \"grand_melee\" : " .. RTBS_DB.buffs[GrandMelee]["timesRolled"] .. ",\n"
		output = output .. "    \"1_buff\" : " .. RTBS_DB.num1Buffs .. ",\n"
		output = output .. "    \"2_buff\" : " .. RTBS_DB.num2Buffs .. ",\n"
		output = output .. "    \"5_buff\" : " .. RTBS_DB.num5Buffs .. ",\n"
		output = output .. "    \"streak_ends\" : {\n"
		for i=1,RTBS_DB.rollsRecord,1 do
			output = output .. "        \"" .. i .. "\" : " .. RTBS_DB.streakEnd[i]
			if(i < RTBS_DB.rollsRecord) then
				output = output .. ",\n"
			end
		end
		output = output .. "\n    }\n"
		output = output .. "}"
	end
	
	TextBox:SetText(output)
end

function round(num, numDecimalPlaces)
	local mult = 10 ^ (numDecimalPlaces or 0)
	return math.floor(num * mult + 0.5) / mult
end

-- Set standard saved variables if they don't exist.
function SetStandardValues()
	if(RTBS_DB == nil) then
		RTBS_DB = {}
	end
	--[[
		Buff Table. Indexes "active" and "abbr" are currently not used.
		"timesRolled" saves the total amount of times a buff was rolled.
	--]]
	if(not RTBS_DB.buffs) then
		RTBS_DB.buffs = {
			[SharkInfestedWaters] = {
				["active"] = false,
				["abbr"] = "Crit",
				["timesRolled"] = 0,
			},
			[TrueBearing] = {
				["active"] = false,
				["abbr"] = "CD Reduce",
				["timesRolled"] = 0,
			},
			[JollyRoger] = {
				["active"] = false,
				["abbr"] = "Saber%",
				["timesRolled"] = 0,
			},
			[Broadsides] = {
				["active"] = false,
				["abbr"] = "+CP",
				["timesRolled"] = 0,
			},
			[BuriedTreasure] = {
				["active"] = false,
				["abbr"] = "Energy Reg",
				["timesRolled"] = 0,
			},
			[GrandMelee] = {
				["active"] = false,
				["abbr"] = "AS + Leech",
				["timesRolled"] = 0,
			},
		}
	end
	--[[
		Prepopulated array for saving streak ends.
		Three single buff rolls resulting in a roll with more than one buff would add 1 to streakEnd[4].
		Having multiple two-buff rolls back to back is a "one-streak ender" resulting in +1 to streakEnd[1].
	--]]
	if(not RTBS_DB.streakEnd) then
		RTBS_DB.streakEnd = {}
		for i=1,100 do
			RTBS_DB.streakEnd[i] = 0
		end
	end
	-- Number of rolls since the last reset
	if(not RTBS_DB.overallRolls) then
		RTBS_DB.overallRolls = 0
	end
	-- Number of single-buff rolls in a row. Adds up every time you consecutively roll only one buff.
	if(not RTBS_DB.rolls) then
		RTBS_DB.rolls = 0
	end
	-- Number of highest single buff streak.
	if(not RTBS_DB.rollsRecord) then
		RTBS_DB.rollsRecord = 0
	end
	-- Number of rolls that ended a streak. A roll sequence of 1 > 1 > 1 > 2 > 1 > 3 > 2 has three streak ending rolls (2, 3 and 2).
	if(not RTBS_DB.endingRolls) then
		RTBS_DB.endingRolls = 0
	end
	-- Number of times 1, 2, 3 and 6 buffs were rolles.
	if(not RTBS_DB.num1Buffs) then
		RTBS_DB.num1Buffs = 0
	end
	if(not RTBS_DB.num2Buffs) then
		RTBS_DB.num2Buffs = 0
	end
	if(not RTBS_DB.num5Buffs) then
		RTBS_DB.num5Buffs = 0
	end
	-- Whether to show the text output or not
	if(not RTBS_DB.show) then
		RTBS_DB.show = "true"
	end
	-- Whether to show a textual or json representation of data
	if(not RTBS_DB.text) then
		RTBS_DB.text = "true"
	end
	if(not RTBS_DB.json) then
		RTBS_DB.json = "false"
	end
	
	ConfigureDisplay()
	UpdateTextBox()
end

-- Show/hide the TextFrame depending on the user's setting
function ConfigureDisplay()
	if(RTBS_DB.show == "true") then
		TextFrame:Show()
	else
		TextFrame:Hide()
	end
end

-- Setting everything to zero
function ResetStatistics()
	RTBS_DB.buffs = {
		[SharkInfestedWaters] = {
			["active"] = false,
			["abbr"] = "Crit",
			["timesRolled"] = 0,
		},
		[TrueBearing] = {
			["active"] = false,
			["abbr"] = "CD Reduce",
			["timesRolled"] = 0,
		},
		[JollyRoger] = {
			["active"] = false,
			["abbr"] = "Saber%",
			["timesRolled"] = 0,
		},
		[Broadsides] = {
			["active"] = false,
			["abbr"] = "+CP",
			["timesRolled"] = 0,
		},
		[BuriedTreasure] = {
			["active"] = false,
			["abbr"] = "Energy Reg",
			["timesRolled"] = 0,
		},
		[GrandMelee] = {
			["active"] = false,
			["abbr"] = "AS + Leech",
			["timesRolled"] = 0,
		},
	}

	for i=1,100 do
		RTBS_DB.streakEnd[i] = 0
	end

	RTBS_DB.overallRolls = 0
	RTBS_DB.rolls = 0
	RTBS_DB.rollsRecord = 0
	RTBS_DB.endingRolls = 0
	RTBS_DB.num1Buffs = 0
	RTBS_DB.num2Buffs = 0
	RTBS_DB.num5Buffs = 0
	RTBS_DB.show = "true"
	RTBS_DB.text = "true"
	RTBS_DB.json = "false"
end

-- Slash commands
SLASH_RTBS1 = '/rtbs'
function SlashCmdList.RTBS(msg, editbox)
	local command, rest = msg:match("^(%S*)%s*(.-)$");
	if(command == "show") then
		TextFrame:Show()
		RTBS_DB.show = "true"
	elseif(command == "hide") then
		TextFrame:Hide()
		RTBS_DB.show = "false"
	elseif(command == "reset") then
		ResetStatistics()
		UpdateTextBox()
	elseif(command == "json") then
		RTBS_DB.json = "true"
		RTBS_DB.text = "false"
		UpdateTextBox()
	elseif(command == "text") then
		RTBS_DB.json = "false"
		RTBS_DB.text = "true"
		UpdateTextBox()
	else
		print("This is not a command of Roll The Bones Statistics. Please consider reading the documentation at https://mods.curse.com/addons/wow/roll-the-bones-statistics.")
	end
end