﻿local LibStub = _G.LibStub
ReagentRecipeTooltip = LibStub("AceAddon-3.0"):NewAddon("ReagentRecipeTooltip", "AceConsole-3.0", "AceEvent-3.0", "AceHook-3.0", "AceSerializer-3.0", "AceTimer-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("ReagentRecipeTooltip")

local cache = {}

local debug = false

local options = {
    name = L["Reagent Recipe Tooltip"],
    handler = ReagentRecipeTooltip,
    type = "group",
    args = {
        threshold = {
            type = "select",
            name = L["Threshold"],
            values = {"Optimal", "Medium", "Easy", "Trivial"},
            desc = L["The minimum difficulty for a recipe to be shown"],
            get = "GetThreshold",
            set = "SetThreshold",
            order=1
        },
        tree = {
            type = "toggle",
            name = L["Tree View"],
            desc = L["Toggles between tree and list"],
            get = "IsUsedInTree",
            set = "ToggleUsedInTree",
            order=5,
            width="full"
        },
        showAlts = {
            type = "toggle",
            name = L["Show Alts"],
            desc = L["Toggles the display of alts"],
            get = "GetShowAlts",
            set = "SetShowAlts",
            order=4,
            width="full"
        },
        recipesListed = {
            type = "range",
            name = L["Recipes Listed"],
            desc = L["The number of recipes displayed per character per tree level"],
            get = "GetRecipesListed",
            set = "SetRecipesListed",
            step=1,
            min = 1,
            max = 30,
            order=2,
        },
        delete= {
            type = "input",
            name = L["Erase"],
            desc = L["Erase data for a character."],
            usage = L["<Character Name>"],
            set = "EraseData",
            order=6
        },
    },
}

local defaults = {
    profile = {
        threshold = "trivial",
        usedInTree = true,
        showAlts = true,
        showFriends = true,
        recipesListed = 20,
        version = false,
    },
}

function ReagentRecipeTooltip:OnInitialize()
    self.db = LibStub("AceDB-3.0"):New("ReagentRecipeTooltipDB", defaults, "Default")

    LibStub("AceConfig-3.0"):RegisterOptionsTable("ReagentRecipeTooltip",  options)
    self.optionsFrame = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("ReagentRecipeTooltip", "ReagentRecipeTooltip")
    if not self.db.factionrealm.player then
        self.db.factionrealm.player = {}
        self.db.factionrealm.friends = {}
    end
    if not self.db.factionrealm.characters then
        self.db.factionrealm.characters = {}
    end
    local characterName = UnitName("player")
    if not self.db.factionrealm.characters[characterName] then
        self.db.factionrealm.characters[characterName] = {
            enchanting = false,
            jewelcrafting = false,
            inscription = false,
        }
    end
    self:Print(self.db.profile.version)

    if not self.db.profile then
        self:Print("no version detected")
        for reagent, reagentData in pairs(self.db.factionrealm.player) do
            for character, data in pairs(reagentData) do
                for recipe, recipeData in pairs(data) do
                    if not recipeData.level then
                        self.db.factionrealm.player[reagent][character][recipe].level = 0
                    end
                end
            end
        end
    end
    self.db.profile.version = 1.02
    cache = self:BuildCache()
end


function ReagentRecipeTooltip:EraseData(info, character)
    self.db.factionrealm.characters[character] = {
        enchanting = false,
        jewelcrafting = false,
        inscription = false,
    }
    for a, types in pairs(self.db.factionrealm) do
        for b, reagents in pairs(types) do
            for name, recipes in pairs(reagents) do
                if (name == character) then
                    reagents[name] = nil
                end
            end
        end
    end
    cache = self:BuildCache()
end

function ReagentRecipeTooltip:GetCharacterList()
    local characters = {};
    for types in ipairs(self.db.factionrealm) do
        for reagents in ipairs(types) do
            for name in pairs(reagents) do
                self:Print(name);
                table.insert(characters, name);
            end
        end
    end
    return characters;
end


function ReagentRecipeTooltip:ChatCommand(input)
    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame(self.optionsFrame)
    else
        LibStub("AceConfigCmd-3.0").HandleCommand(ReagentRecipeTooltip, "rt", "ReagentRecipeTooltip", input)
    end
end

function ReagentRecipeTooltip:GetRecipesListed(info)
    return self.db.profile.recipesListed
end


function ReagentRecipeTooltip:SetRecipesListed(info, newValue)
    self.db.profile.recipesListed = newValue
end

function ReagentRecipeTooltip:GetShowAlts(info)
    return self.db.profile.showAlts
end

function ReagentRecipeTooltip:SetShowAlts(info, newValue)
    self.db.profile.showAlts = newValue
end

function ReagentRecipeTooltip:GetThreshold(info)
    local threshold = self.db.profile.threshold
    if (threshold == "optimal") then
        return 1
    end
    if (threshold == "medium") then
        return 2
    end
    if (threshold == "easy") then
        return 3
    end
    if (threshold == "trivial") then
        return 4
    end
end

function ReagentRecipeTooltip:SetThreshold(info, newValue)
    local db = self.db.profile
    if (newValue == 1) then        
        db.threshold = "optimal"
    elseif (newValue == 2) then
        db.threshold = "medium"
    elseif (newValue == 3) then
        db.threshold = "easy"
    elseif (newValue == 4) then
        db.threshold = "trivial"
    else
        self:Print(newValue .. " is not a valid threshold.")
    end
    self:Print("Threshold set to " .. self.db.profile.threshold)
end

function ReagentRecipeTooltip:IsUsedInTree(info)
    return self.db.profile.usedInTree
end

function ReagentRecipeTooltip:ToggleUsedInTree(info, value)
    self.db.profile.usedInTree = value
--~     self:Print("UsedInTree set to " .. self.usedInTree)
end

local rh_tradeskill = GameTooltip.SetRecipeReagentItem
function GameTooltip:SetRecipeReagentItem(...)
    local link = C_TradeSkillUI.GetRecipeReagentItemLink(...)
    if link then return self:SetHyperlink(link) end
    return rh_tradeskill(self, ...)
end

local function AttachTooltip(tooltip, ...)
    local a,b = tooltip:GetItem()
    ReagentRecipeTooltip:MakeTree(tooltip, select(1, tooltip:GetItem()), true, {})
    --~ 	return self.hooks[tooltip].OnTooltipSetItem(tooltip, ...)
end

function ReagentRecipeTooltip:OnEnable()
    self:RegisterEvent("TRADE_SKILL_SHOW")
    self:RegisterEvent("CHAT_MSG_SKILL")
    self:RegisterEvent("TRADE_SKILL_LIST_UPDATE")
    GameTooltip:HookScript("OnTooltipSetItem", AttachTooltip)
    ItemRefTooltip:HookScript("OnTooltipSetItem", AttachTooltip)
    ItemRefShoppingTooltip1:HookScript("OnTooltipSetItem", AttachTooltip)
    ItemRefShoppingTooltip2:HookScript("OnTooltipSetItem", AttachTooltip)
end



function ReagentRecipeTooltip:CHAT_MSG_SKILL(info, msg)
    if strfind(msg, "Inscription") or strfind(msg, "Jewelcrafting") then
        local skill
        for i = -4, -1 do
            skill = tonumber(strsub(msg, i))
            if skill then
                break
            end
            if i == -1 then
                skill = 1
            end
        end
    end
end

function ReagentRecipeTooltip:OnDisable()
    -- Called when the addon is disabled
end

function ReagentRecipeTooltip:QueueRecipes()
    local skill = C_TradeSkillUI.GetTradeSkillLine()
    if debug and not skill then print("no skill in queue") end
    if not skill then return end

    if ReagentRecipeTooltip.CR_Timer then return end -- we already have scheduled refresh, wait for it

    ReagentRecipeTooltip.CR_Timer = ReagentRecipeTooltip:ScheduleTimer(function() ReagentRecipeTooltip:ReadRecipes() end, 1)
end

function ReagentRecipeTooltip:ReadRecipes()
    if debug then print("Reading Recipes") end
    if ReagentRecipeTooltip.CR_Timer then ReagentRecipeTooltip:CancelTimer(ReagentRecipeTooltip.CR_Timer) ReagentRecipeTooltip.CR_Timer=nil end
    if not C_TradeSkillUI.IsTradeSkillReady() then return end -- prevents missing reagents in recipes
    local skill = C_TradeSkillUI.GetTradeSkillLine()
    if debug and not skill then print("no skill in read") end
    if not skill then return end
    if not C_TradeSkillUI.IsTradeSkillLinked() then
        if debug then print("about to get recipes") end
        local recipesIDs = C_TradeSkillUI.GetAllRecipeIDs()
        local recipeInfo = {};
        for key, recipeID in pairs(recipesIDs) do
            local recipeInfo = C_TradeSkillUI.GetRecipeInfo(recipeID)
            if (recipeInfo.learned) then

                if debug then print("adding".. recipeInfo.name) end
                for reagentIndex = 1, C_TradeSkillUI.GetRecipeNumReagents(recipeID) do
                    local reagentName, _, reagentCount = C_TradeSkillUI.GetRecipeReagentInfo(recipeID, reagentIndex)
                    if not reagentName then reagentName="Missing" end
                    local db = self.db.factionrealm.player
                    db[reagentName] = db[reagentName] or {}
                    local characterName = UnitName("player")
                    db[reagentName][characterName] = db[reagentName][characterName] or {}
                    db[reagentName][characterName][recipeInfo.name] = {
                        count = reagentCount,
                        difficulty = recipeInfo.difficulty,
                        level = 0
                    }
                end
            end

        end
        ReagentRecipeTooltip.LastRecipeCheckLine = skill

        cache = self:BuildCache()
    end
end
function ReagentRecipeTooltip:TRADE_SKILL_SHOW()

	self:QueueRecipes()
end

function ReagentRecipeTooltip:TRADE_SKILL_LIST_UPDATE()
    self:QueueRecipes()
end

function ReagentRecipeTooltip:TRADE_SKILL_CLOSE()

end

function ReagentRecipeTooltip:MakeTree(tooltip, itemName, top, exclude)
    local buffer = {}
    local useful = false
    if exclude[itemName] then
        return false
    end
    if cache and cache[itemName] then
        for characterName, recipe in pairs(cache[itemName]) do
            buffer[characterName] = {}
            for _, recipeData in ipairs(recipe) do
				local recipeName = recipeData.name
                local nested = false
                if ReagentRecipeTooltip:IsUsedInTree() then
                    exclude[itemName] = true
                    nested = ReagentRecipeTooltip:MakeTree(tooltip, recipeName, false, exclude)
                end
                if not buffer[characterName][recipeData.difficulty] then
                    buffer[characterName][recipeData.difficulty] = {}
                end
                if (IsAltKeyDown()or ReagentRecipeTooltip:compareDifficulty(recipeData.difficulty, self.db.profile.threshold) >= 0) or nested then
                    table.insert(buffer[characterName][recipeData.difficulty], {
						name = recipeName,
						count = recipeData.count,
						nested = nested,
						level = recipeData.level
					})
                    useful = true
                end
            end
        end
        if useful then
            if top then
                ReagentRecipeTooltip:displayTree(tooltip, buffer, 0)
            else
                return buffer
            end
        else
            return false
        end
    end
    return false
end


function ReagentRecipeTooltip:displayTree(tooltip, tree, depth, prevChar)
    if tree[UnitName("Player")] then
        self:displayCharacterTree(tooltip, tree[UnitName("Player")], depth, UnitName("Player"), prevChar)
    end
    if self:GetShowAlts() then
        for character, recipe in pairs(tree) do
            if character ~= UnitName("Player") then
                self:displayCharacterTree(tooltip, recipe, depth, character, prevChar)
            end
        end
    end
end

function ReagentRecipeTooltip:displayCharacterTree(tooltip, recipe, depth, characterName, prevChar)
    for i=0, depth do
        characterName = "   " .. characterName
    end
    local characterListed = false
    local recipes = 0
    for d=3, 0, -1 do
        local difficulty = ReagentRecipeTooltip:numberToDifficulty(d)
        local red, green, blue = ReagentRecipeTooltip:getDifficultyColor(difficulty)
        if recipe[difficulty] then
            for _, recipeData in ipairs(recipe[difficulty]) do
				local recipeName = recipeData.name
                if (recipes < tonumber(self:GetRecipesListed())) then
                    recipes = recipes + 1
                    recipeName = " " .. recipeName
                    for i=0, depth do
                        recipeName = "   " .. recipeName
                    end
                    if not characterListed then --added this variable and moved the printing of the character name to the loop in order to prevent names from appearing without recipes attached
                        characterListed = true
                        if not (depth == 0 and trim(characterName) == UnitName("Player")) and trim(characterName) ~= prevChar then
                            tooltip:AddLine(characterName)
                        end
                    end
					if (recipeData.level and recipeData.level > 0) then
						tooltip:AddDoubleLine(recipeName.." ("..recipeData.level..")", recipeData.count, red, green, blue, red, green, blue)
					else
						tooltip:AddDoubleLine(recipeName, recipeData.count, red, green, blue, red, green, blue)
					end
                    if recipeData.nested then
                        ReagentRecipeTooltip:displayTree(tooltip, recipeData.nested, depth + 1, trim(characterName))
                    end
                else
                    tooltip:AddLine("Max recipes listed")
                    break
                end
            end
        end
    end
end

-- remove trailing and leading whitespace from string.
-- http://en.wikipedia.org/wiki/Trim_(8programming)
function trim(s)
  -- from PiL2 20.4
  return (s:gsub("^%s*(.-)%s*$", "%1"))
end


--Returns a positive number if difficulty1 is harder than difficulty2, 0 if they're even, or a negative number otherwise
function ReagentRecipeTooltip:compareDifficulty(difficulty1, difficulty2)
    difficulty1 = ReagentRecipeTooltip:difficultyToNumber(difficulty1)
    difficulty2 = ReagentRecipeTooltip:difficultyToNumber(difficulty2)
    return difficulty1 - difficulty2
end

function ReagentRecipeTooltip:difficultyToNumber(difficulty)
    if difficulty == "trivial" then
        return 0
    elseif difficulty == "easy" then
        return 1
    elseif difficulty == "medium" then
        return 2
    elseif difficulty == "optimal" then
        return 3
    end
end

function ReagentRecipeTooltip:numberToDifficulty(x)
    if x == 0 then
        return "trivial"
    elseif x == 1 then
        return "easy"
    elseif x == 2 then
        return "medium"
    elseif x == 3 then
        return "optimal"
    end
end

function ReagentRecipeTooltip:getDifficultyColor(difficulty)
    if difficulty == "trivial" then
        return .7, .7, .7
    elseif difficulty == "easy" then
        return .05, .8, .05
    elseif difficulty == "medium" then
        return 1, 1, 0
    elseif difficulty == "optimal" then
        return 1, .5, 0
    end
end

function ReagentRecipeTooltip:BuildCache()
    local db = self.db.factionrealm.player
    local ans = {}
    for reagent, characters in pairs(db) do
        ans[reagent] = {}
        for character, recipes in pairs(characters) do
            ans[reagent][character] = {}
			local n = 1
            if recipes ~= true then
                for recipeName, recipeData in pairs(recipes) do
                    ans[reagent][character][n] = {
                        name = recipeName,
                        count = recipeData.count,
                        difficulty = recipeData.difficulty,
                        level = recipeData.level
                    }
                    n=n+1
                end
                table.sort(ans[reagent][character], function(a,b) return a.level > b.level end)
            end
        end
    end
    db = self.db.factionrealm.friends
    for reagent, characters in pairs(db) do
        if not ans[reagent] then
            ans[reagent] = {}
        end
        for character, recipes in pairs(characters) do
            ans[reagent][character] = {}
			local n=1
            for recipeName, recipeData in pairs(recipes) do
				ans[reagent][character][n] = {
					name = recipeName,
					count = recipeData.count,
					difficulty = recipeData.difficulty,
					level = recipeData.level
				}
				n=n+1
            end
            table.sort(ans[reagent][character], function(a,b) return a.level > b.level end)
        end
    end
    return ans
end
