-- ------------------------------------------------------------------------- --
-- Project: Lib Executive Assistant - Library for Group and Task Management
--   to provide plug-in accessibility to Executive Assistant as a secondary
--   information delivery vector.
--
-- Mange Groups / Tasks to addon criteria, manage completion of Tasks or 
-- 	 update their status (e.g., buy 10 x -> buy 3 more x) on progression,
--   provide some communication similarity (e.g., LEA_alert), and similar
--   overall function if ExecAssist is being canabalized for function logic
--   (string extenders)
-- 
-- This library DOES NOT provide 'Intern Module' functionality; however it
-- 	 does contain some functions common to them (lea_alert & lea_debug)
--
-- Author:  VincentSDSH				
-- Version: 1.0-14
--
-- API QuickReference
--	:LEA_msgSetup(name_alert, name_debug)
--  :LEA_enableDebug( boolean_state )
--	:LEA_debug(data, label, colorize)
--	:LEA_alert(msg, r, g, b)
--	:Create_Group(parent_groupID, groupName)	
--	:Create_Task(parent_groupID, taskName, resetType)
--	:Edit_Task(taskVariant, taskName, resetType, defaultEnabled, userOverride, isAcctWide, hideDays, forUser)
--	:Edit_Group(groupVariant, listName, defaultEnabled, userOverride, hideFromDisplay, hideDays, forUser)
--	:Delete_Task(taskVariant)
--	:Delete_Group(groupVariant)
--	:getChecked(taskID, user) 
--	:setChecked(taskID, markComplete, user) 
--	:getHandle(elementID)
--	:IsActive_forUser(elementID, user)
--	:IsAvailable_forUser(taskID, user)
--	:IterateAgglomeration()
--	:getCurrentUser()	
--	:isAvail_ExecAssist()
--	:isGroup(element ID string)
--	:isTask(element ID string)
--  :refreshElement(ElementVariant)
--
-- Conventions:
-- 		groupID, taskID, anythingID : STRING : contains the ID number of an item
--    hGroup, hTask, hElement : table pointer : handle to interior of *ID
-- 		ex: local hGroup = self:getHandle(groupID)
-- 
--[[ Group and Task Structures Explained (prototypes: active Groups and Tasks may contain additional data)
--	 Developers ** MAY ** add additional fields and data but please be kind to your users, follow safe naming practices, and do not store critical data in them.

	prototype = {
		["Group"] = {
	  	["id"] = "", -- self handle :: pass (hGroup) rather than (hGroup, groupID) when asking a parent to operate on the child's /reference/ not just the child (e.g., .DO ops, etc)
			["p"] = "", -- parent GroupID
			["listName"] = "Default New Group Name",
			["defaultEnabled"] = true, -- ENABLED
			["hideDays"] = {}, -- {Sun -> Sat} Primed b/c of multiselect control
			["DO"] = {} -- Display Order: tells the group who its children are and what order to display or operate on them
  	},
		["Task"]	= {
			["id"] = "", -- taskID
			["p"] = "",  -- parent GroupID
			["taskName"] = "Default New Task Name",
			["defaultEnabled"] = true, -- ENABLED
			["resetType"] = "Daily Automatic",	-- {Daily, Weekly, Other}
			["hideDays"] = {} -- {Sun -> Sat} Primed b/c of multiselect control					
		}
	}
--]]
-- ------------------------------------------------------------------------- --

-- [ To Be or Not To Be ] -------------------------------------------------- --
local PT_MAJOR, PT_MINOR = "LibExecAssist-1.0", 14
local lib, oldminor = LibStub:NewLibrary(PT_MAJOR, PT_MINOR)
local EXECASSIST = "Executive_Assistant"
if not lib then return end
-- -------------------------------------------------- [ To Be or Not To Be ] --

-- [ local vars ] ---------------------------------------------------------- --
-- Currently Logged In User, formatted for use with Executive Assistant
local currentUser = UnitName("player").." - "..GetRealmName()

-- :isGroup(var); :isTask(var) are useful when iterating the agglomeration ()
local string_sub = _G.string.sub
function lib:isGroup(data) if data and #data>0 then return (string_sub(data, 1,1) == "g") and true or false else return false end end
function lib:isTask(data)  if data and #data>0 then return (string_sub(data, 1,1) == "t") and true or false else return false end end

-- isAvail_ExecAssist, easy way to query if Executive Assistant is loaded
function lib:isAvail_ExecAssist(verbose) local isLoaded = IsAddOnLoaded(EXECASSIST) if IsAddOnLoaded(EXECASSIST) then return true else if verbose then DEFAULT_CHAT_FRAME:AddMessage("Executive Assistant is Required", 1, 0, 0) end return false end end

-- debug: allows easy value-tracing within the the lib; handy for verifying 'addon or library' problems or seeing library internals
local debug = function(a, b) DEFAULT_CHAT_FRAME:AddMessage("|cff00ffff:|cffff0066libExecAssist|cff00ffff:|r "..tostring(a)..(b and (" - "..tostring(b)) or "" ) ) end
-- ---------------------------------------------------------- [ local vars ] --

-- [ Housekeeping ] --------------------------------------------------- --
local Crayons = {["PTtan"] = "ff9966", ["PTblue"] = "6699ff"}
local function cr(color, msg) return "|cff"..color..(true and msg or "nil").."|r" end

	-- [ setup tags ] --------------------------------------------------- --
	-- Provides a 'context tag' for text output (e.g., MyAddon: message text)
	-- When callled from many addons/modules, differentiation is important.
	-- 
	-- If not using either LEA_alert() or LEA_debug() you DO NOT need to call this function
	--
	-- if name_alert is omitted (nil) then LEA_alert() calls will return w/o output
	-- if name_debug is omitted (nil) then LEA_debug() calls will return w/o output
function lib:LEA_msgSetup(name_alert, name_debug)
	self.LEA_name_alert = name_alert -- Long Name for Alerts -- users should understand
	self.LEA_name_debug = name_debug -- Shot Name for debug() function -- devs
end

	-- [ debug function ] ----------------------------------------------- --
	-- LEA_enableDebug(): boolean: tells LEA_debug() whether or not to do anything
	-- LEA_debug(data, label, colorize): Common function for Intern Modules
	-- 	data: is what you're looking for
	--  label: is what you wish to call it (note than 'is' is suffixed to the label)
	-- 			 : ex: for k, v in pairs(var) debug(v, k) end
	-- 
	--  If data or label is nil "nil" will be displayed
	--
function lib:LEA_enableDebug(enableDebug) self.LEA_debugEnabled = enableDebug end
function lib:LEA_debug(data, label, colorize)
	if self.LEA_debugEnabled and self.LEA_name_debug then 
		local r, b, g 
		if not colorize then r=1; g=1; b=.8 
		elseif (label and data) then r=0; g=1; b=0 
		else r=1; g=0; b=0 
		end; 
		local bigColon = cr(Crayons["PTblue"], "::")		
		DEFAULT_CHAT_FRAME:AddMessage( bigColon..cr(Crayons["PTtan"], self.LEA_name_debug)..bigColon .." ".. (label and (tostring(label).." is: ") or "").. tostring(data), r, g, b); 
	end
end

	-- [ addon Alert msg ] ---------------------------------------------- --	
	-- Used in Intern Modules to provide a similar message experience to ExecAssist message types
	-- requires LEA_msgSetup()
function lib:LEA_alert(msg, r, g, b)
	if self.LEA_name_alert then
		DEFAULT_CHAT_FRAME:AddMessage( cr(Crayons["PTtan"], self.LEA_name_alert)..cr(Crayons["PTblue"],": ")..tostring(msg), r or 1, g or 1, b or .8);
	end		
end

-- [ Creating ] ----------------------------------------------------- --
-- Executive Assistant uses an Agglomeration to contain pile of Goups and Tasks.
-- Each Group and Task knows its parent.
-- Each Group knows its children.
-- the 'garden' is the master parent group
-- The 'root' groups in the display list all grow from the 'garden'
-- 
-- conventions: 
--			.p: parent ID
--  	 .DO: Display Order of children (applies only to Groups)
--     .id: the ID of the branch (so a 'k, v in paris()' can pass v and v.id = k w/o having to pass k /and/ v, or hGroup, groupID, etc)

	-- if parent_groupID == nil, Group is created in root ("garden")
	-- if groupName == nil, the default new group name is used	
function lib:Create_Group(parent_groupID, groupName)	
	local hParent -- if parent_groupID == nil then hParent will remain nil and the Group will be created in the garden (root)
	if parent_groupID then hParent = lib:getHandle(parent_groupID) end

	local groupID = ExecAssist:AddGroup(true and groupName or "External Addon Target", hParent, true) -- name, parent, noProgeny
	lib:UpdateTaskWindow()
	
	return groupID
end

	-- if parent_groupID == nil, a default-named Group is created in root ("garden")
	-- taskName is a string
	-- Valid resetTypes are:	
	--  local eaL = LibStub("AceLocale-3.0"):GetLocale("Executive_Assistant")	
	--	eaL.DailyAutomatic
	--	eaL.WeeklyAutomatic
	--	eaL.Manual
	--	eaL.CalendarAutomatic
	--	eaL.Purgable
	--	eaL.Reminder
	--	eaL.AtEachLogin
function lib:Create_Task(parent_groupID, taskName, resetType) -- returns taskID, parent_group	
	if not parent_groupID then parent_groupID = lib:Create_Group() end
	local hGroup = lib:getHandle(parent_groupID)
	
	local hTask = ExecAssist:AddTask(hGroup, nil, true)
	hTask.extCreate = self:GetName() -- tags the task; forensics & custom-control loading key (for modules) 

	if taskName or resetType then lib:Edit_Task(hTask, taskName, resetType) end

	ExecAssist:semiSmart_regrow(hGroup.G)
	lib:UpdateTaskWindow()
	
	return hTask.id, parent_groupID
end
	-- ----------------------------------------------------- [ Creating ] --

-- Edits return True for good edit; False for missing ID
-- Due to the async nature of edits, passing a nil for any parameter (save the first) will ignore updating that parameter. 
-- Send a false rather than nil for boolean negative
-- taskVariant and groupVariant allow passing of an elementID or element Handle (hElement) so if you already have 
-- hTask then hTask doesn't have to be reobtained by passing taskID or hTask.id BUT you don't have to obtain the element handle
-- if all you have is the ID.

-- taskName = string. Name to display. Color Codes can be used, they are safely editable by the user.
-- resetType = valid resetType (see lib:Create_Task() above).  Nota Bene: Others may be used, from other Intern Modules etc but
--						 it's up to you to find the proper localization key
-- defaultEnabled: boolean  Default Enabled or Disabled State
-- userOverride: numeric 1=accept default, 2=enabled for user, 3=disabled for user  Overrides the default usage state
-- isAcctWide: boolean
-- hidedays: table. Partial tables cannot be sent, it's all or nothing.
-- 			hideDays = {
-- 				true, -- [sunday]
-- 				true, -- [monday]
-- 				true, -- [tuesday]
-- 				false, -- [wednesday]
-- 				false, -- [thursday]
-- 				true, -- [friday]
-- 				true, -- [saturday]
-- 			},
-- forUser: username, of omitted the current user is used
function lib:Edit_Task(taskVariant, taskName, resetType, defaultEnabled, userOverride, isAcctWide, hideDays, forUser)
	local hTask = type(taskVariant)=="string" and lib:getHandle(taskVariant) or taskVariant
	-- defaultEnabled: true=enabled; false=disabled;
	-- userOverride: 1 / default; 2=enabled; 3=disabled;
		
	if hTask then	
		if defaultEnabled ~= nil then hTask.defaultEnabled = defaultEnabled end
		if userOverride ~= nil then 
			if not forUser then forUser = currentUser end
			local charStem = ExecAssist.db.global.charStem[forUser]
			if userOverride==1 and charStem.agg[hTask.id] then charStem.agg[hTask.id].userOverride = nil -- 1:default / no override
			else
				if not charStem.agg[hTask.id] then charStem.agg[hTask.id] = {} end
				charStem.agg[hTask.id].userOverride = userOverride
			end
		end
		if taskName 	~= nil then hTask.taskName 	 = taskName end
		if resetType 	~= nil then hTask.resetType  = resetType end 
		if isAcctWide ~= nil then hTask.isAcctWide = isAcctWide end
		if hideDays   ~= nil and type(hideDays) == "table" then hTask.hideDays = hideDays end

		ExecAssist:semiSmart_regrow(lib:getHandle(hTask.p).G) 
		lib:UpdateTaskWindow()
		return true -- good edit
	else
		return false -- could not edit, missing task
	end
end 

-- listName = string. Name to display. Color Codes can be used, they are safely editable by the user.
-- defaultEnabled: boolean  Default Enabled or Disabled State
-- userOverride: numeric 1=accept default, 2=enabled for user, 3=disabled for user  Overrides the default usage state
-- hidedays: table. Partial tables cannot be sent, it's all or nothing.
-- 			hideDays = {
-- 				true, -- [sunday]
-- 				true, -- [monday]
-- 				true, -- [tuesday]
-- 				false, -- [wednesday]
-- 				false, -- [thursday]
-- 				true, -- [friday]
-- 				true, -- [saturday]
-- 			},
-- forUser: username, of omitted the current user is used
function lib:Edit_Group(groupVariant, listName, defaultEnabled, userOverride, hideFromDisplay, hideDays, forUser)
	local hGroup = type(groupVariant)=="string" and lib:getHandle(groupVariant) or groupVariant

	if hGroup then
		if defaultEnabled ~= nil then hGroup.defaultEnabled = defaultEnabled end
		if not forUser then forUser = currentUser end
		local charStem = ExecAssist.db.global.charStem[forUser]		
		if userOverride==1 and charStem.agg[hGroup.id] then charStem.agg[hGroup.id].userOverride = nil -- 1:default / no override
		else
			if not charStem.agg[hGroup.id] then charStem.agg[hGroup.id] = {} end
			charStem.agg[hGroup.id].userOverride = userOverride
		end
		if listName 	~= nil then hGroup.listName 	= listName end
		if hideFromDisplay ~= nil then hGroup.hideFromDisplay = hideFromDisplay end
		if hideDays		~= nil and type(hideDays) == "table" then hGroup.hideDays = hideDays end

		ExecAssist:semiSmart_regrow(hGroup.G)
		lib:UpdateTaskWindow()
		return true  -- good edit
	else
		return false -- could not edit, missing group
	end
end -- requires groupID

-- Deletes the task or group supplied
-- Accepts both ID strings and hElement handles.
-- RETURNS: true if a call to ExecAssist was sent, which means the item was active and the delete routines can be run
function lib:Delete_Task(taskVariant)
	local hTask
	if type(taskVariant)=="string" then hTask =lib:getHandle(taskVariant)
	else hTask = taskVariant
	end
	
	if hTask then ExecAssist:DeleteTask(hTask, lib:getHandle(hTask.p).G); return true end
end
-- Delets the group and all groups and tasks under it
-- RETURNS: true if a call to ExecAssist was sent, which means the item was active and the delete routines can be run
function lib:Delete_Group(groupVariant)
	local hGroup 	
	if type(groupVariant)=="string" then hGroup = lib:getHandle(groupVariant) 
	else hGroup = groupVariant
	end
	
	if hGroup then ExecAssist:DeleteGroup(hGroup, hGroup.G); return true end
end

-- Get the Completed State (is taskID Checked) for the supplied character name
-- TaskID is required, if charName is omitted
function lib:getChecked(taskID, user) 
	if ExecAssist.db.global.groupStem.agg[taskID] then return ExecAssist:getCheckedState(true and user or currentUser, taskID) end
end
-- Set the Completed State (is taskID Checked) for the supplied character name
-- TaskID is required, markComplete is a required boolean, if charName is omitted
function lib:setChecked(taskID, markComplete, user) 
	if not taskID then return end

	local sdbg = ExecAssist.db.global
	local hTask = sdbg.groupStem.agg[taskID]
	
	if hTask then	
		if not user then user = currentUser end
		if hTask.isAcctWide then
			local AWagg = sdbg.acctwideStem.agg
				
			if not AWagg[taskID] then 
				AWagg[taskID] = {["checked"] = markComplete}
			else
				AWagg[taskID].checked = markComplete
			end
		else
			local charStem = sdbg.charStem[user]
			if not charStem.agg[taskID] then 
				charStem.agg[taskID] = {["checked"] = markComplete}
			else
				charStem.agg[taskID].checked = markComplete
			end
		end

		lib:UpdateTaskWindow()
	end	
end

	-- Returns the handle for the elementID sent to it, also allows checking the existance of an ID
	-- local hGroup = getHandle(groupID)
	-- local hTask  = getHandle(taskID)
	-- if lib:getHandle(taskID) then goDoSomethingWith_taskID(taskID) end
function lib:getHandle(elementID) return ExecAssist.db.global.groupStem.agg[elementID] end

	-- Returns a boolean for the Group or Task ID, True of the user is using the Group or Task, false otherwise
	-- elementID is required (groupID or taskID), if user is omitted, the current user will be automatically supplied.
function lib:IsActive_forUser(elementID, user) -- Is not Disabled for User
	if not user then user = currentUser end 
	return ExecAssist:ID_isActive(user, elementID) 
end

	-- Groups can be made unavailble individually (see :IsActive_forUser() ) which affects all the groups and tasks inside them. 
	-- A Task may be Active for the User but be Disabled due to its parentage.  This return a boolean to that effect
	-- taskID is required, if user is omitted, the current user will be automatically supplied.
function lib:IsAvailable_forUser(taskID, user) -- Is not disabled for User by lineage (NB: runs ID_isActive)
	if not user then user = currentUser end
	return ExecAssist:hasCleanGenes(user, taskID)
end

	-- Returns an iterator for the Agglomeration of Groups and Tasks.
	-- returns k, v pairs() as k=taskID or groupID, and v=the internal Group or Task data
	-- 
	-- It's useful to use :isGroup(var) and :isTask(var) to tell which is which
	-- (e.g., for k, v in pairs( lib:IterateAgglomeration() ) do, if self:isGroup(v) then ... elseif self:isTask(v) then ... else ... debug("There is no else, the author is just being a smart-ass") end
function lib:IterateAgglomeration() return pairs(ExecAssist.db.global.groupStem.agg) end

	-- :UpdateTaskWindow() lets the addon author assume TaskWindow usage w/o worrying if it is actually used.
	-- In general practice, this would not be necessary but it may be useful if direct-edits are used (rather than this lib)
function lib:UpdateTaskWindow() if ExecAssist.taskWindow then ExecAssist:UpdateWindow() end end

	-- returns the current user, formatted for ExecAssist
function lib:getCurrentUser()	return currentUser end

	-- Will refresh an item in the ExecAssist Options Table.
	-- :refreshElement(ElementVariant) accept as variant for TaskID, hTask; GroupID, hGroup
function lib:refreshElement(hElement)	
	if type(elementID) == "string" then hElement = lib:getHandle(elementID) end
	if hElement and hElement.p then ExecAssist:semiSmart_regrow(ExecAssist.db.global.groupStem.agg[hElement.p].G) end
end

	-- Iterator for Characters
function lib:IterateCharacters() return pairs(ExecAssist.db.global.charStem) end 

function lib:isTaskUsed(variantTask)
	local taskID, isUsed
	if type(variantTask) == "string" then taskID = variantTask
	else taskID = variantTask.id
	end
	
	for userName, v in lib:IterateCharacters() do
		if self:IsActive_forUser(taskID, userName) then
			isUsed = true
			break;
		end
	end

	return isUsed
end

-- ----------------------------------------------------------- [ Core ] --

-- [ Embeds ] ----------------------------------------------------------- --
lib.embeds = lib.embeds or {} -- table containing objs lib is embedded in

local mixins = { -- functions to expose to addon
	"isAvail_ExecAssist",
	"isGroup", "isTask", "getHandle", "getCurrentUser",
	
	"Create_Task", "Create_Group",
	"Edit_Task",   "Edit_Group", 
	"Delete_Task", "Delete_Group",
	
	"getChecked", "setChecked", 
	
	"LEA_enableDebug",
	"LEA_msgSetup",
	"LEA_debug", "LEA_alert",
	
	"UpdateTaskWindow",
	"IterateAgglomeration",
	"IsActive_forUser",
	"IsAvailable_forUser",
	
	"refreshElement",
	"IterateCharacters",
	"isTaskUsed"
}
function lib:Embed( target )
	for k, v in pairs( mixins ) do target[v] = self[v] end
	self.embeds[target] = true
	return target
end

function lib:OnEmbedEnable( target ) end
function lib:OnEmbedDisable( target ) end
for addon in pairs(lib.embeds) do lib:Embed(addon) end
-- ----------------------------------------------------------- [ Embeds ] --
