local MAJOR, MINOR = "LibPetBreedInfo-1.0", 16
local lib, oldminor = LibStub:NewLibrary(MAJOR, MINOR)

if not lib then return end
lib.breedData = {}
--
-- GLOBALS:
--

local _G = _G
local hooksecurefunc, tinsert, pairs, wipe = _G.hooksecurefunc, _G.table.insert, _G.pairs, _G.wipe
local ipairs = _G.ipairs
local C_PetJournal,C_PetBattles = _G.C_PetJournal, _G.C_PetBattles
local _
local EMPTY_PET = "0x0000000000000000"
local epsilion = .005
local STATE_Mod_SpeedPrecent = 25
local STATE_MaxHealthBonus = 2
local STATE_Mod_MaxHealthPrercent = 99
local HEALTH,POWER,SPEED = 1,2,3
local BASE_HEALTH = 100
local HEALTH_PER_STAMINA = 5
local FALLBACK_BREEDS = {3,4,5,6,7,8,9,10,11,12}
local WILD_PET_HEALTH_MULTIPLIER = 1.2
local MAX_PETS_PER_TEAM = 3
--
-- Non Lib helper functions
--
local function round(num, idp)
  local mult = 10^(idp or 0)
  return math.floor(num * mult + 0.5) / mult
end

local function clamp(num,minVal,maxVal)
	return math.min(math.max(num,minVal),maxVal)
end

local function inverseStats(num)
	return num ~= 0 and 100/(num+100) or 1
end


--Gets the breed index for the given pet
--Returns arg1: petIndex that can be used by GetBreedName to return the breeds name. arg2: a confidence indicator, lower numbers are better. A good cutoff for high confidence is .15
--Returns nil if the species cannot battle or does not have a base stats profile or if the petID is not valid
function lib:GetBreedByPetID(petID)
	if petID then
		local speciesID, _, level, _, _, _,_ ,_, _, _, _, _, _, _, canBattle = C_PetJournal.GetPetInfoByPetID(petID)
		if not canBattle then return end
		local _, maxHealth, power, speed, rarity = C_PetJournal.GetPetStats(petID)
		
		return self:GetBreedByStats(speciesID,level,rarity,maxHealth,power,speed)
	end
end


--Gets the breed index for the given hypothetical pet
--Returns arg1: petIndex that can be used by GetBreedName to return the breeds name. arg2: a confidence indicator, lower numbers are better. A good cutoff for high confidence is .15
--Returns nil if the species cannot battle or does not have a base stats profile or if level, health power and speed are not numbers and rarity is not between 1 and 4
function lib:GetBreedByStats(speciesID,level,rarity,health,power,speed)
	
	if not self.breedData.speciesToBaseStatProfile[speciesID] or not self.breedData.qualityMultiplier[rarity] or  
		type(level) ~= "number" or type(health) ~= "number" or type(power) ~= "number" or type(speed) ~= "number" then 
		
		return 
	end
	
	local canBattle      = select(8,C_PetJournal.GetPetInfoBySpeciesID(speciesID))
	local baseStatsIndex = self.breedData.speciesToBaseStatProfile[speciesID]
	
	if not baseStatsIndex and not canBattle then return end
	
	local baseStats         = self.breedData.baseStatsProfiles[baseStatsIndex]
	local qualityMultiplier = self.breedData.qualityMultiplier[rarity]

	health = clamp(round(( ((health-BASE_HEALTH)/HEALTH_PER_STAMINA) / qualityMultiplier) - level * baseStats[HEALTH],1) / level,0,2)
	power  = clamp(round((									  power  / qualityMultiplier) - level * baseStats[POWER] ,1) / level,0,2)
	speed  = clamp(round((                                    speed  / qualityMultiplier) - level * baseStats[SPEED] ,1) / level,0,2)
	
	local breeds = self:GetAvailableBreeds(speciesID)
	breeds = breeds and breeds or FALLBACK_BREEDS
	
	local leastDiff = 100
	local leastDiff2 = 100
	local breedID = 0
	
	for i=1,#breeds do
		local breedIndex = breeds[i]
		local breedData = self.breedData.breeds[breedIndex]
		local diff = round(math.abs(health - breedData[HEALTH]) + math.abs(power - breedData[POWER]) + math.abs(speed - breedData[SPEED]),2) + epsilion
		
		if diff < leastDiff then
			leastDiff2 = leastDiff
			leastDiff  = diff
			breedID    = breedIndex
		elseif diff < leastDiff2 then
			leastDiff2 = diff
		end
		
	end
	return breedID, round(leastDiff2 / leastDiff ,2)
end



--Gets the breed index for the given pet
--!!!Results are not valid if computed while a round play back is happening!!!
--Returns arg1: petIndex that can be used by GetBreedName to return the breeds name. arg2: a confidence indicator, lower numbers are better. A good cutoff for high confidence is .15
--Returns nil if the species cannot battle or does not have a base stats profile, or an invalid petOwner/petID is used, or if the player is not in a pet battle

--/run print(C_PetBattles.GetName(1,1),LibStub("LibPetBreedInfo-1.0"):GetBreedByPetBattleSlot(1,1))
function lib:GetBreedByPetBattleSlot(petOwner,id)
	if not C_PetBattles.IsInBattle() then return end
	if petOwner ~= LE_BATTLE_PET_ALLY and petOwner ~= LE_BATTLE_PET_ENEMY then return end
	if id <= 0 or id > MAX_PETS_PER_TEAM and id <= C_PetBattles.GetNumPets(petOwner) then return end
	
	local speedMultiplier  = C_PetBattles.GetStateValue(petOwner,id,STATE_Mod_SpeedPrecent)
	local healthMultiplier = C_PetBattles.GetStateValue(petOwner,id,STATE_Mod_MaxHealthPrercent)
	local healthModifier   = C_PetBattles.GetStateValue(petOwner,id,STATE_MaxHealthBonus)
	
	if not speedMultiplier or not healthMultiplier or not healthModifier then return end
	
	speedMultiplier  = inverseStats(speedMultiplier)
	healthMultiplier = inverseStats(healthMultiplier)

	
	
	
	
	if C_PetBattles.IsWildBattle() and petOwner == LE_BATTLE_PET_ENEMY then
		healthMultiplier = healthMultiplier * WILD_PET_HEALTH_MULTIPLIER
	end
	
	local speciesID = C_PetBattles.GetPetSpeciesID(petOwner,id)
	local speed     = round(C_PetBattles.GetSpeed(petOwner,id) * speedMultiplier)
	local power     = C_PetBattles.GetPower(petOwner,id)
	local health    = (C_PetBattles.GetMaxHealth(petOwner,id) * healthMultiplier) - healthModifier
	local rarity    = C_PetBattles.GetBreedQuality(petOwner,id)
	local level     = C_PetBattles.GetLevel(petOwner,id)
	
	return self:GetBreedByStats(speciesID,level,rarity,health,power,speed)
end


--Gets the predicted stats for a species at a given level
--returns health,power,speed
--returns nil if the pet does not have a base stats profile, or if an invalid... breedID, rarity, or non numeric level is given.
function lib:GetPetPredictedStats(speciesID, breedID, rarity, level)
	if not self.breedData.breeds[breedID] or not self.breedData.speciesToBaseStatProfile[speciesID] or type(level) ~= "number" or not self.breedData.qualityMultiplier[rarity] then return end
	
	local baseStatsIndex = self.breedData.speciesToBaseStatProfile[speciesID]
	local baseStats      = self.breedData.baseStatsProfiles[baseStatsIndex]
	local multiplier     = self.breedData.qualityMultiplier[rarity]
	local breedStats     = self.breedData.breeds[breedID]
	
	local health = round((baseStats[HEALTH] + breedStats[HEALTH]) * multiplier * level * HEALTH_PER_STAMINA + BASE_HEALTH)
	local power  = round((baseStats[POWER]  + breedStats[POWER])  * multiplier * level) 
	local speed  = round((baseStats[SPEED]  + breedStats[SPEED])  * multiplier * level)
	
	return health, power , speed
end

--Gets a table containing the possible BreedID's for a given species
--Returns nil if no data is present for the given speciesID
--[[
function lib:GetAvailableBreeds(speciesID)
end]]

--Gets an iterator for all of the possible breed names
function lib:IterateBreedNames()
	return pairs(self.breedData.breedNames)
end

--Gets the string representation of a BreedID
--Returns nil for invalid breedID's
function lib:GetBreedName(breedID)
	return self.breedData.breedNames[breedID]
end


--Gets a table of possible breedId's for a given species
--BreedIDs are from 1-10
--Returns nil for invalid speciesID's
function lib:GetAvailableBreeds(speciesID)
	return self.breedData.speciesToAvailableBreeds[speciesID]
end

--/lb code LibStub("LibPetBreedInfo-1.0"):GetSpeciesWithoutProfiles()
--Gets a table of speciesIDs which are present in game but do not have a base stats profile
--This is a dev function to make updating this library easier
--Returns a table
--[[
function lib:GetSpeciesWithoutProfiles()
	local missingPets = {}
	for i=1,2000 do --2000 to catch new pet IDs
		name = C_PetJournal.GetPetInfoBySpeciesID(i)
		if name and not self.breedData.speciesToBaseStatProfile[i] then
			tinsert(missingPets,i)
		end
	end
	return missingPets
end]]

--/run LibStub("LibPetBreedInfo-1.0"):FindMissingSpeciesBreeds()
--[[function lib:FindMissingSpeciesBreeds()
	local lpj = LibStub("LibPetJournal-2.0")
	local availableBreeds = self.breedData.speciesToAvailableBreeds
	for _,v in lpj:IterateSpeciesIDs() do
		if not availableBreeds[v] then
			print(v,C_PetJournal.GetPetInfoBySpeciesID(v))
		end
	end
end]]

--/run LibStub("LibPetBreedInfo-1.0"):ExtractData()
--[[function lib:ExtractData()
	BPBID.InitializeArrays()
	local speciesToBaseStatProfile= self.breedData.speciesToBaseStatProfile
	for k,v in pairs(BPBID.BasePetStats) do
		if not speciesToBaseStatProfile[k] then 
			print(k)
		end
	end
end]]




lib.breedData.breeds = {
	[3]={.5,.5,.5}, --1
	[4]={0,2,0}, --2
	[5]={0,0,2}, --3
	[6]={2,0,0}, --4 
	[7]={.9,.9,0},--5
	[8]={0,.9,.9},--6
	[9]={.9,0,.9},--7
	[10]={.4,.9,.4},--8
	[11]={.4,.4,.9},--9
	[12]={.9,.4,.4}--10
}

lib.breedData.breedNames = {
	[3]="B/B", --1
	[4]="P/P", --2
	[5]="S/S", --3
	[6]="H/H", --4 
	[7]="H/P",--5
	[8]="P/S",--6
	[9]="H/S",--7
	[10]="P/B",--8
	[11]="S/B",--9
	[12]="H/B"--10
}

lib.breedData.qualityMultiplier = {1,1.1,1.2,1.3,1.4}


--stam,power,speed
lib.breedData.baseStatsProfiles = {
	{8.5, 7.5, 8}, --1
	{7, 8.5, 8.5},--2
	{6.5, 9, 8.5},--3
	{7, 9, 8},--4
	{7.5, 7.5, 9},--5
	{8, 7.5, 8.5},--6
	{9, 7, 8},--7
	{9, 8, 7},--8
	{8.5, 8, 7.5},--9
	{8, 8, 8},--10
	{8.5, 7, 8.5},--11
	{8.5, 9, 6.5},--12
	{8, 7, 9},--13
	{7.5, 8.5, 8},--14
	{8, 8.5, 7.5},--15
	{7, 7, 10},--16
	{8.5, 8.5, 7},--17
	{7.5, 8, 8.5},--18
	{7.5, 7, 9.5},--19
	{9, 7.5, 7.5},--20
	{9, 9, 6},--21
	{9, 7.5, 7.5},--22
	{7.5, 9, 7.5},--23
	{8, 9, 7},--24
	{9.5, 8, 6.5},--24
	{9.5, 8.5, 6},--25
	{8, 9.5, 6.5},--26
	{8, 8.75, 7.25},--27
	{7, 8, 9},--28
	{9, 6.5, 8.5},--29
	{6, 8, 10},--30
	{9, 6.5, 8.5},--31
	{8, 6.5, 9.5},--32
	{9, 8.5, 6.5},--33
	{9.5, 7.5, 7},--34
	{6.5, 8, 9.5},--35
	{8.5, 8.5, 8.5},--36
	{9, 9, 9},--37
	{6.75, 10.5, 6.75},--38
	{7.5, 8.5, 8.5}, --39
	{25.5, 11.5, 7.25},--40
	{510.5, 8, 9.5}, --41
	{7.75, 8, 8.25}, --42
	{7.75, 8.25, 8}, --43
	{8.25, 8.5, 7.25}, --44
	{8.25, 8, 7.75}, --45
	{8, 8.25, 7.75}, --46
	{11, 16, 7}, --47
	{8.25, 8, 7.75}, --45
	{7.25, 8.75, 8.25}, --48
}




lib.breedData.speciesToBaseStatProfile = {
	[2] = 41,
	[39] = 1,
	[40] = 2,
	[41] = 2,
	[42] = 3,
	[43] = 4,
	[44] = 2,
	[45] = 2,
	[46] = 5,
	[47] = 6,
	[49] = 7,
	[50] = 8,
	[51] = 9,
	[52] = 10,
	[55] = 11,
	[56] = 12,
	[57] = 1,
	[58] = 4,
	[59] = 9,
	[64] = 1,
	[65] = 1,
	[67] = 10,
	[68] = 10,
	[69] = 5,
	[70] = 6,
	[71] = 10,
	[72] = 13,
	[73] = 10,
	[74] = 5,
	[75] = 14,
	[77] = 10,
	[78] = 14,
	[83] = 1,
	[84] = 10,
	[85] = 1,
	[86] = 1,
	[87] = 15,
	[89] = 10,
	[90] = 2,
	[92] = 10,
	[93] = 3,
	[94] = 16,
	[95] = 10,
	[106] = 1,
	[107] = 10,
	[111] = 10,
	[114] = 17,
	[115] = 10,
	[116] = 10,
	[117] = 10,
	[118] = 10,
	[119] = 10,
	[120] = 10,
	[121] = 10,
	[122] = 10,
	[124] = 10,
	[125] = 10,
	[126] = 10,
	[127] = 6,
	[128] = 10,
	[130] = 10,
	[131] = 6,
	[132] = 8,
	[136] = 10,
	[137] = 13,
	[138] = 1,
	[139] = 14,
	[140] = 15,
	[141] = 5,
	[142] = 15,
	[143] = 14,
	[144] = 5,
	[145] = 1,
	[146] = 18,
	[149] = 19,
	[153] = 10,
	[155] = 10,
	[156] = 10,
	[157] = 10,
	[158] = 10,
	[159] = 10,
	[160] = 10,
	[162] = 15,
	[163] = 15,
	[164] = 15,
	[165] = 1,
	[166] = 17,
	[167] = 22,
	[168] = 1,
	[169] = 10,
	[170] = 10,
	[171] = 10,
	[172] = 14,
	[173] = 15,
	[174] = 15,
	[175] = 14,
	[179] = 9,
	[180] = 9,
	[183] = 10,
	[186] = 14,
	[187] = 17,
	[188] = 17,
	[189] = 10,
	[190] = 17,
	[191] = 1,
	[192] = 10,
	[193] = 15,
	[194] = 10,
	[195] = 5,
	[196] = 9,
	[197] = 18,
	[198] = 10,
	[199] = 10,
	[200] = 13,
	[201] = 10,
	[202] = 10,
	[203] = 13,
	[204] = 10,
	[205] = 10,
	[206] = 17,
	[207] = 10,
	[209] = 10,
	[210] = 10,
	[211] = 9,
	[212] = 10,
	[213] = 10,
	[214] = 10,
	[215] = 1,
	[216] = 10,
	[217] = 10,
	[218] = 1,
	[220] = 10,
	[224] = 2,
	[225] = 10,
	[226] = 10,
	[227] = 10,
	[228] = 10,
	[229] = 10,
	[231] = 9,
	[232] = 2,
	[233] = 2,
	[234] = 2,
	[235] = 2,
	[236] = 14,
	[237] = 2,
	[238] = 2,
	[239] = 10,
	[240] = 17,
	[241] = 10,
	[242] = 14,
	[243] = 21,
	[244] = 12,
	[245] = 10,
	[246] = 10,
	[247] = 10,
	[248] = 10,
	[249] = 17,
	[250] = 23,
	[251] = 17,
	[253] = 10,
	[254] = 10,
	[255] = 23,
	[256] = 24,
	[257] = 10,
	[258] = 12,
	[259] = 1,
	[260] = 15,
	[261] = 1,
	[262] = 1,
	[264] = 17,
	[265] = 25,
	[266] = 17,
	[267] = 10,
	[268] = 12,
	[270] = 17,
	[271] = 6,
	[272] = 22,
	[277] = 1,
	[278] = 10,
	[279] = 10,
	[280] = 10,
	[281] = 10,
	[282] = 10,
	[283] = 10,
	[285] = 24,
	[286] = 8,
	[287] = 6,
	[289] = 26,
	[291] = 22,
	[292] = 10,
	[293] = 10,
	[294] = 10,
	[296] = 10,
	[297] = 27,
	[298] = 10,
	[301] = 2,
	[302] = 17,
	[303] = 2,
	[306] = 2,
	[307] = 10,
	[308] = 10,
	[309] = 3,
	[310] = 10,
	[311] = 9,
	[316] = 1,
	[317] = 17,
	[318] = 14,
	[319] = 3,
	[320] = 1,
	[321] = 10,
	[323] = 10,
	[325] = 23,
	[328] = 10,
	[329] = 10,
	[330] = 10,
	[331] = 10,
	[332] = 10,
	[333] = 10,
	[335] = 22,
	[336] = 10,
	[337] = 10,
	[338] = 1,
	[339] = 10,
	[340] = 10,
	[341] = 10,
	[342] = 10,
	[343] = 2,
	[344] = 9,
	[345] = 15,
	[346] = 10,
	[347] = 28,
	[348] = 17,
	[354] = 10,
	[374] = 10,
	[375] = 10,
	[378] = 13,
	[379] = 6,
	[380] = 10,
	[381] = 10,
	[382] = 10,
	[383] = 6,
	[384] = 10,
	[385] = 6,
	[386] = 6,
	[387] = 18,
	[388] = 9,
	[389] = 1,
	[390] = 6,
	[391] = 13,
	[392] = 6,
	[393] = 11,
	[394] = 10,
	[395] = 9,
	[396] = 2,
	[397] = 10,
	[398] = 6,
	[399] = 18,
	[400] = 2,
	[401] = 9,
	[402] = 9,
	[403] = 18,
	[404] = 6,
	[405] = 18,
	[406] = 1,
	[407] = 2,
	[408] = 18,
	[409] = 29,
	[410] = 6,
	[411] = 10,
	[412] = 2,
	[414] = 10,
	[415] = 14,
	[416] = 10,
	[417] = 6,
	[418] = 18,
	[419] = 1,
	[420] = 1,
	[421] = 15,
	[422] = 14,
	[423] = 17,
	[424] = 11,
	[425] = 18,
	[427] = 2,
	[428] = 2,
	[429] = 14,
	[430] = 15,
	[431] = 18,
	[432] = 10,
	[433] = 18,
	[434] = 10,
	[437] = 10,
	[438] = 18,
	[439] = 9,
	[440] = 10,
	[441] = 13,
	[442] = 32,
	[443] = 10,
	[444] = 10,
	[445] = 10,
	[446] = 9,
	[447] = 6,
	[448] = 13,
	[449] = 10,
	[450] = 8,
	[452] = 10,
	[453] = 8,
	[454] = 6,
	[455] = 10,
	[456] = 15,
	[457] = 8,
	[458] = 17,
	[459] = 2,
	[460] = 14,
	[461] = 8,
	[462] = 10,
	[463] = 21,
	[464] = 10,
	[465] = 2,
	[466] = 18,
	[467] = 14,
	[468] = 14,
	[469] = 2,
	[470] = 2,
	[471] = 1,
	[472] = 1,
	[473] = 22,
	[474] = 31,
	[475] = 11,
	[476] = 5,
	[477] = 5,
	[478] = 9,
	[479] = 33,
	[480] = 15,
	[482] = 18,
	[483] = 1,
	[484] = 2,
	[485] = 22,
	[486] = 6,
	[487] = 10,
	[488] = 18,
	[489] = 15,
	[491] = 4,
	[492] = 15,
	[493] = 26,
	[494] = 10,
	[495] = 1,
	[496] = 26,
	[497] = 11,
	[498] = 9,
	[499] = 6,
	[500] = 10,
	[502] = 1,
	[503] = 5,
	[504] = 15,
	[505] = 18,
	[506] = 2,
	[507] = 14,
	[508] = 10,
	[509] = 10,
	[510] = 1,
	[511] = 18,
	[512] = 14,
	[513] = 10,
	[514] = 10,
	[515] = 10,
	[517] = 10,
	[518] = 34,
	[519] = 9,
	[521] = 4,
	[523] = 8,
	[525] = 10,
	[528] = 18,
	[529] = 10,
	[530] = 10,
	[532] = 12,
	[534] = 14,
	[535] = 10,
	[536] = 9,
	[537] = 9,
	[538] = 17,
	[539] = 6,
	[540] = 6,
	[541] = 11,
	[542] = 1,
	[543] = 14,
	[544] = 17,
	[545] = 10,
	[546] = 14,
	[547] = 13,
	[548] = 14,
	[549] = 10,
	[550] = 6,
	[552] = 14,
	[553] = 6,
	[554] = 14,
	[555] = 11,
	[556] = 14,
	[557] = 14,
	[558] = 10,
	[559] = 23,
	[560] = 1,
	[562] = 18,
	[564] = 35,
	[565] = 1,
	[566] = 5,
	[567] = 18,
	[568] = 26,
	[569] = 1,
	[570] = 10,
	[571] = 18,
	[572] = 9,
	[573] = 7,
	[626] = 18,
	[627] = 17,
	[628] = 9,
	[629] = 9,
	[630] = 10,
	[631] = 18,
	[632] = 18,
	[633] = 10,
	[634] = 2,
	[635] = 18,
	[637] = 10,
	[638] = 11,
	[639] = 6,
	[640] = 13,
	[641] = 13,
	[644] = 6,
	[645] = 10,
	[646] = 10,
	[647] = 10,
	[648] = 22,
	[649] = 22,
	[650] = 9,
	[652] = 10,
	[665] = 14,
	[666] = 10,
	[671] = 10,
	[675] = 6,
	[677] = 10,
	[678] = 10,
	[679] = 10,
	[680] = 10,
	[699] = 2,
	[702] = 1,
	[703] = 10,
	[705] = 15,
	[706] = 10,
	[707] = 10,
	[708] = 6,
	[709] = 6,
	[710] = 10,
	[711] = 10,
	[712] = 10,
	[713] = 22,
	[714] = 2,
	[715] = 10,
	[716] = 2,
	[717] = 14,
	[718] = 10,
	[722] = 14,
	[723] = 22,
	[724] = 10,
	[725] = 10,
	[726] = 2,
	[727] = 10,
	[728] = 10,
	[729] = 13,
	[730] = 13,
	[731] = 2,
	[732] = 15,
	[733] = 10,
	[737] = 36,
	[739] = 29,
	[740] = 6,
	[741] = 10,
	[742] = 10,
	[743] = 26,
	[744] = 11,
	[745] = 10,
	[746] = 12,
	[747] = 14,
	[748] = 15,
	[749] = 10,
	[750] = 10,
	[751] = 6,
	[752] = 22,
	[753] = 14,
	[754] = 10,
	[755] = 32,
	[756] = 11,
	[757] = 9,
	[758] = 14,
	[792] = 15,
	[800] = 10,
	[802] = 17,
	[817] = 9,
	[818] = 15,
	[819] = 14,
	[820] = 10,
	[821] = 10,
	[823] = 10,
	[824] = 10,
	[825] = 10,
	[826] = 10,
	[827] = 10,
	[828] = 10,
	[829] = 10,
	[830] = 10,
	[831] = 10,
	[832] = 10,
	[833] = 10,
	[834] = 9,
	[835] = 10,
	[836] = 19,
	[837] = 9,
	[838] = 10,
	[844] = 15,
	[845] = 10,
	[846] = 31,
	[847] = 10,
	[848] = 13,
	[849] = 10,
	[850] = 10,
	[851] = 18,
	[855] = 14,
	[856] = 10,
	[868] = 10,
	[872] = 10,
	[873] = 10,
	[874] = 10,
	[875] = 10,
	[876] = 10,
	[877] = 10,
	[878] = 10,
	[879] = 10,
	[880] = 10,
	[881] = 10,
	[882] = 10,
	[883] = 10,
	[884] = 10,
	[885] = 10,
	[886] = 10,
	[887] = 10,
	[888] = 10,
	[889] = 10,
	[890] = 10,
	[891] = 10,
	[892] = 10,
	[893] = 10,
	[894] = 10,
	[895] = 10,
	[896] = 10,
	[897] = 10,
	[898] = 10,
	[899] = 10,
	[900] = 10,
	[901] = 10,
	[902] = 10,
	[903] = 23,
	[904] = 10,
	[905] = 10,
	[906] = 10,
	[907] = 10,
	[908] = 10,
	[909] = 10,
	[911] = 15,
	[912] = 15,
	[913] = 15,
	[915] = 10,
	[916] = 10,
	[917] = 10,
	[921] = 10,
	[922] = 10,
	[923] = 1,
	[924] = 10,
	[925] = 10,
	[926] = 10,
	[927] = 10,
	[928] = 10,
	[929] = 10,
	[931] = 10,
	[932] = 10,
	[933] = 10,
	[934] = 10,
	[935] = 10,
	[936] = 10,
	[937] = 10,
	[938] = 10,
	[939] = 10,
	[941] = 10,
	[942] = 10,
	[943] = 10,
	[944] = 10,
	[945] = 10,
	[946] = 10,
	[947] = 10,
	[948] = 10,
	[949] = 10,
	[950] = 10,
	[951] = 10,
	[952] = 10,
	[953] = 10,
	[954] = 10,
	[955] = 10,
	[956] = 10,
	[957] = 10,
	[958] = 10,
	[959] = 10,
	[960] = 10,
	[961] = 10,
	[962] = 10,
	[963] = 10,
	[964] = 10,
	[965] = 10,
	[966] = 10,
	[967] = 10,
	[968] = 10,
	[969] = 10,
	[970] = 10,
	[971] = 10,
	[972] = 10,
	[973] = 10,
	[974] = 10,
	[975] = 10,
	[976] = 10,
	[977] = 37,
	[978] = 37,
	[979] = 37,
	[980] = 37,
	[981] = 37,
	[982] = 37,
	[983] = 37,
	[984] = 37,
	[985] = 37,
	[986] = 37,
	[987] = 37,
	[988] = 37,
	[989] = 10,
	[990] = 10,
	[991] = 10,
	[992] = 10,
	[993] = 10,
	[994] = 10,
	[995] = 10,
	[996] = 10,
	[997] = 10,
	[998] = 10,
	[999] = 10,
	[1000] = 10,
	[1001] = 10,
	[1002] = 38,
	[1003] = 38,
	[1004] = 10,
	[1005] = 10,
	[1006] = 10,
	[1007] = 10,
	[1008] = 10,
	[1009] = 10,
	[1010] = 10,
	[1011] = 10,
	[1012] = 10,
	[1013] = 22,
	[1039] = 5,
	[1040] = 10,
	[1042] = 39,
	[1061] = 10,
	[1062] = 6,
	[1063] = 17,
	[1065] = 17,
	[1066] = 10,
	[1067] = 1,
	[1068] = 10,
	[1073] = 10,
	[1117] = 39,
	[1124] = 10,
	[1125] = 10,
	[1126] = 10,
	[1127] = 14,
	[1128] = 10,
	[1129] = 10,
	[1130] = 10,
	[1131] = 10,
	[1132] = 10,
	[1133] = 10,
	[1134] = 10,
	[1135] = 10,
	[1136] = 10,
	[1137] = 10,
	[1138] = 10,
	[1139] = 10,
	[1140] = 10,
	[1141] = 10,
	[1142] = 1,
	[1143] = 10,
	[1144] = 10,
	[1145] = 10,
	[1146] = 10,
	[1147] = 10,
	[1149] = 10,
	[1150] = 10,
	[1151] = 10,
	[1152] = 14,
	[1153] = 10,
	[1154] = 10,
	[1155] = 15,
	[1156] = 10,
	[1157] = 10,
	[1158] = 10,
	[1159] = 14,
	[1160] = 9,
	[1161] = 15,
	[1162] = 4,
	[1163] = 8,
	[1164] = 18,
	[1165] = 15,
	[1166] = 15,
	[1167] = 15,
	--end 5.1
	[1168] = 10,
	[1168] = 10,
	[1174] = 14,
	[1175] = 10,
	[1176] = 10,
	[1177] = 8,
	[1178] = 10,
	[1179] = 15,
	[1180] = 2,
	[1181] = 18,
	[1182] = 1,
	[1183] = 10,
	[1184] = 2,
	[1185] = 10,
	[1187] = 40,
	[1188] = 40,
	[1189] = 40,
	[1190] = 40,
	[1191] = 40,
	[1192] = 40,
	[1193] = 40,
	[1194] = 40,
	[1195] = 40,
	[1196] = 10,
	[1197] = 10,
	[1198] = 10,
	[1200] = 2,
	[1201] = 2,
	[1202] = 6,
	[1204] = 1,
	[1205] = 2,
	[1206] = 18,
	[1207] = 39,
	[1208] = 18,
	[1209] = 18,
	[1211] = 2,
	[1212] = 2,
	[1213] = 2,
	--end 5.2
	[1226] = 14,
	[1227] = 9,
	[1229] = 42,
	[1231] = 10,
	[1232] = 43,
	[1233] = 44,
	[1234] = 45,
	[1235] = 10,
	[1237] = 46,
	[1238] = 17,
	[1244] = 10,
	[1245] = 10,
	[1247] = 47,
	[1248] = 45,
	[1255] = 48,
	--end 5.3
}



lib.breedData.speciesToAvailableBreeds = {
	[39] = {11},
	[40] = {8},
	[41] = {4},
	[42] = {7},
	[43] = {3},
	[44] = {5},
	[45] = {3},
	[46] = {12},
	[47] = {3, 5},
	[49] = {3},
	[50] = {11},
	[51] = {3, 5},
	[52] = {3, 10, 11, 12},
	[55] = {3, 5, 6, 9},
	[56] = {8},
	[57] = {4},
	[58] = {8},
	[59] = {9},
	[64] = {9},
	[65] = {12},
	[67] = {3, 8, 10},
	[68] = {3, 6, 7},
	[69] = {3, 10, 11},
	[70] = {3, 5, 12},
	[72] = {3, 5, 11},
	[74] = {11},
	[75] = {5},
	[77] = {3},
	[78] = {12},
	[83] = {8},
	[84] = {3, 10, 11, 12},
	[85] = {9},
	[86] = {7},
	[87] = {5},
	[89] = {6},
	[90] = {7},
	[92] = {3},
	[93] = {4},
	[94] = {5},
	[95] = {3},
	[106] = {10},
	[107] = {3},
	[111] = {12},
	[114] = {12},
	[116] = {12},
	[117] = {3},
	[118] = {3},
	[119] = {3},
	[120] = {3},
	[121] = {3},
	[122] = {5},
	[124] = {3},
	[125] = {5},
	[126] = {10},
	[127] = {7},
	[128] = {3},
	[130] = {3},
	[131] = {8},
	[132] = {6},
	[136] = {3, 7, 10, 12},
	[137] = {3, 5, 11},
	[138] = {3, 12},
	[139] = {3, 8, 12},
	[140] = {3, 4, 12},
	[141] = {3, 6, 12},
	[142] = {3, 5},
	[143] = {3, 5, 8},
	[144] = {3, 5, 11},
	[145] = {3, 5, 6, 8},
	[146] = {3, 5, 9, 12},
	[149] = {9},
	[153] = {7},
	[155] = {7},
	[156] = {3},
	[157] = {3},
	[158] = {3},
	[159] = {6},
	[160] = {4},
	[162] = {10},
	[163] = {8},
	[164] = {3},
	[165] = {9},
	[166] = {7},
	[167] = {3, 6, 9},
	[168] = {5},
	[169] = {5},
	[170] = {7},
	[171] = {7},
	[172] = {10},
	[173] = {7},
	[174] = {3},
	[175] = {5},
	[179] = {7},
	[180] = {7},
	[183] = {3},
	[186] = {3, 4, 5, 8},
	[187] = {8},
	[188] = {6},
	[189] = {3},
	[190] = {6, 7, 12},
	[191] = {3},
	[192] = {9},
	[193] = {4},
	[194] = {3, 5, 8, 9, 11},
	[195] = {5, 8, 9, 11},
	[196] = {7},
	[197] = {8},
	[198] = {11},
	[199] = {10},
	[200] = {5},
	[201] = {7},
	[202] = {7},
	[203] = {12},
	[204] = {3},
	[205] = {7},
	[206] = {3, 9, 11},
	[207] = {8},
	[209] = {3},
	[210] = {5},
	[211] = {6},
	[212] = {3},
	[213] = {5},
	[214] = {3},
	[215] = {10},
	[216] = {3},
	[217] = {7},
	[218] = {4},
	[220] = {9},
	[224] = {10},
	[225] = {3},
	[226] = {4},
	[227] = {4},
	[228] = {7},
	[229] = {3},
	[231] = {3},
	[232] = {8},
	[233] = {8},
	[234] = {10},
	[235] = {10},
	[236] = {6},
	[237] = {3},
	[238] = {11},
	[239] = {4},
	[240] = {3},
	[241] = {12},
	[242] = {8},
	[243] = {10},
	[244] = {7},
	[245] = {8},
	[246] = {3},
	[247] = {6},
	[248] = {8},
	[249] = {7},
	[250] = {8},
	[251] = {3},
	[253] = {3},
	[254] = {3, 5, 8},
	[255] = {6},
	[256] = {7},
	[258] = {6},
	[259] = {12},
	[260] = {8},
	[261] = {4},
	[262] = {5},
	[264] = {8},
	[265] = {6},
	[266] = {8},
	[267] = {3},
	[268] = {4},
	[270] = {9},
	[271] = {3, 9, 10},
	[272] = {6, 9},
	[277] = {11},
	[278] = {9},
	[279] = {8},
	[280] = {3},
	[281] = {3},
	[282] = {3},
	[283] = {3},
	[285] = {10},
	[286] = {7},
	[287] = {3},
	[289] = {6},
	[291] = {12},
	[292] = {3},
	[293] = {9},
	[294] = {8},
	[296] = {3},
	[297] = {4},
	[298] = {3},
	[301] = {11},
	[302] = {7},
	[303] = {5},
	[306] = {11},
	[307] = {3},
	[308] = {3},
	[309] = {8},
	[310] = {8},
	[311] = {3},
	[316] = {3},
	[317] = {3},
	[318] = {3},
	[319] = {10},
	[320] = {7},
	[321] = {6},
	[323] = {8},
	[325] = {5, 8, 10},
	[328] = {4},
	[329] = {8},
	[330] = {10},
	[331] = {3},
	[332] = {3},
	[333] = {11},
	[335] = {12},
	[336] = {3},
	[337] = {12},
	[338] = {7},
	[339] = {12},
	[340] = {3},
	[341] = {3},
	[342] = {3},
	[343] = {3, 8},
	[346] = {3},
	[347] = {3},
	[348] = {8},
	[374] = {7, 8, 9, 10, 11, 12},
	[378] = {3, 5, 7, 9, 11, 12},
	[379] = {3, 5, 10, 11, 12},
	[380] = {3, 12},
	[381] = {9},
	[383] = {8, 11},
	[385] = {3, 5, 8, 11, 12},
	[386] = {3, 5, 11},
	[387] = {3, 5},
	[388] = {3, 6, 7},
	[389] = {3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
	[391] = {3, 5, 9},
	[392] = {3, 5, 9, 11},
	[393] = {5, 6, 7, 9, 11, 12},
	[395] = {3, 6, 7},
	[396] = {8, 11},
	[397] = {3, 9, 11, 12},
	[398] = {3, 5, 9, 11},
	[399] = {3, 5, 8},
	[400] = {3, 4, 9, 11},
	[401] = {3, 6, 7},
	[402] = {3, 9, 12},
	[403] = {3, 5, 8, 10},
	[404] = {3, 12},
	[405] = {3, 4, 7},
	[406] = {3, 6, 7, 9},
	[407] = {3, 5, 8, 11},
	[408] = {3, 5, 11},
	[409] = {5, 8, 10},
	[410] = {3, 5, 9, 11},
	[411] = {3, 4},
	[412] = {3, 5, 11},
	[414] = {3, 8},
	[415] = {6, 7, 9},
	[416] = {3, 5, 8},
	[417] = {3, 5, 9, 11},
	[418] = {3, 11},
	[419] = {3, 11},
	[420] = {3, 7, 12},
	[421] = {3, 9, 12},
	[422] = {3, 5, 8},
	[423] = {3, 6, 9},
	[424] = {5, 6, 7, 9, 11, 12},
	[425] = {3, 8, 10},
	[427] = {3, 9},
	[428] = {3, 5, 8, 11},
	[429] = {4, 6, 7, 9},
	[430] = {5},
	[431] = {8, 9},
	[432] = {3, 8},
	[433] = {3, 5, 7},
	[437] = {3, 10},
	[438] = {6, 9},
	[439] = {6, 9},
	[440] = {3, 4, 8},
	[441] = {5, 7, 9},
	[442] = {6, 7, 9, 12},
	[443] = {3, 5, 7, 9, 11, 12},
	[445] = {6, 7, 9, 10},
	[446] = {3, 4, 6, 9},
	[447] = {3, 5, 9, 11, 12},
	[448] = {3, 5, 7, 9, 11, 12},
	[449] = {3, 5, 11},
	[450] = {3, 6},
	[452] = {3, 5, 10, 11, 12},
	[453] = {6, 7},
	[454] = {3, 5, 9, 11},
	[455] = {7, 8, 10},
	[456] = {3, 6, 7, 8},
	[457] = {6, 9},
	[458] = {3, 6},
	[459] = {3, 5, 8, 11},
	[460] = {3, 6, 9},
	[461] = {3},
	[463] = {6, 9},
	[464] = {3, 5, 9},
	[465] = {3, 4, 8},
	[466] = {3, 5, 7},
	[467] = {3, 6, 7, 9},
	[468] = {3, 6, 7, 9},
	[469] = {3, 4, 6, 7, 9},
	[470] = {3, 4, 7, 11},
	[471] = {3, 5, 6, 8, 9, 10},
	[472] = {3, 4, 5, 6, 7, 8, 10, 11},
	[473] = {3, 9},
	[474] = {5},
	[475] = {3, 12},
	[477] = {3, 11},
	[478] = {3, 9, 11, 12},
	[479] = {3, 11},
	[480] = {3, 6, 9},
	[482] = {3, 5},
	[483] = {3, 12},
	[484] = {3, 5, 8, 11},
	[485] = {6, 7, 9},
	[487] = {3, 5, 7, 10, 11, 12},
	[488] = {12},
	[489] = {4, 8, 10},
	[491] = {5, 8},
	[492] = {3, 6, 9},
	[493] = {3, 6, 7},
	[494] = {3, 6, 7},
	[495] = {3, 12},
	[496] = {3, 9, 12},
	[497] = {6, 7, 9, 12},
	[498] = {3, 8, 9},
	[499] = {7},
	[500] = {3, 4, 6, 7, 8, 10},
	[502] = {3, 12},
	[503] = {3, 5, 11},
	[504] = {3, 4, 10},
	[505] = {3, 5, 7, 11},
	[506] = {3, 6, 9, 11},
	[507] = {3, 9, 12},
	[508] = {3, 4, 7},
	[509] = {6, 7, 9},
	[511] = {3, 5, 11},
	[512] = {4, 6, 9, 12},
	[513] = {4, 5, 7, 10},
	[514] = {3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
	[515] = {3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
	[517] = {3, 8},
	[518] = {3, 6},
	[519] = {6, 7, 9},
	[521] = {10},
	[523] = {6, 9},
	[525] = {3, 5, 12},
	[528] = {9, 12},
	[529] = {4, 7},
	[530] = {3, 6, 7, 9},
	[532] = {4, 6},
	[534] = {3, 4, 5, 8, 10},
	[535] = {3, 9},
	[536] = {7, 9},
	[537] = {3, 4, 6, 7},
	[538] = {6, 8},
	[539] = {3, 5, 10, 11, 12},
	[540] = {3, 5, 10, 11, 12},
	[541] = {5, 6, 7, 9},
	[542] = {3, 12},
	[543] = {3, 5, 9},
	[544] = {5, 11, 12},
	[545] = {3, 5, 8},
	[546] = {6, 9, 12},
	[547] = {3, 5, 7, 9, 10, 11},
	[548] = {3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
	[549] = {3, 5, 10, 11, 12},
	[550] = {3, 5, 10, 11, 12},
	[552] = {3, 4, 6, 7, 11},
	[553] = {3, 5, 10, 11, 12},
	[554] = {3, 7, 9},
	[555] = {5, 6, 7, 9},
	[556] = {6, 9, 12},
	[557] = {4, 5, 7, 10, 11},
	[558] = {8},
	[559] = {4, 6, 7, 9},
	[560] = {3, 5, 10},
	[562] = {3, 5, 8},
	[564] = {3, 6, 12},
	[565] = {3, 12},
	[566] = {3, 9, 12},
	[567] = {5, 10},
	[568] = {3, 9},
	[569] = {3, 12},
	[570] = {3, 10},
	[571] = {3, 5},
	[572] = {4, 6},
	[573] = {3, 5, 10, 11},
	[626] = {4, 5, 7, 8, 10, 11},
	[627] = {7, 10},
	[628] = {3, 10},
	[629] = {3, 4, 7},
	[630] = {3, 5, 8, 10},
	[631] = {7, 9},
	[632] = {3, 10},
	[633] = {3, 7, 9, 12},
	[634] = {3, 9},
	[635] = {3, 8, 10},
	[637] = {3, 5, 11},
	[638] = {5, 6, 7, 9},
	[639] = {3, 5, 10, 11, 12},
	[640] = {3, 5, 7, 9, 11, 12},
	[641] = {3, 5, 7, 9, 11, 12},
	[644] = {3, 5, 10, 11, 12},
	[645] = {3, 8, 9},
	[646] = {3, 5, 11, 12},
	[647] = {3, 5, 8, 10, 11},
	[648] = {3, 6, 7, 12},
	[649] = {3, 7, 8},
	[650] = {8},
	[652] = {11},
	[665] = {11},
	[671] = {8},
	[675] = {3, 5, 10, 11, 12},
	[677] = {3, 5, 8},
	[678] = {6, 9},
	[679] = {5, 11},
	[680] = {5, 11},
	[699] = {3, 9, 11, 12},
	[702] = {3, 5, 12},
	[703] = {3, 11},
	[706] = {3, 5, 8},
	[707] = {3, 5},
	[708] = {3, 5, 10, 11, 12},
	[709] = {3, 5, 11, 12},
	[710] = {3, 9, 11},
	[711] = {3, 5, 11},
	[712] = {3, 5, 11},
	[713] = {3, 9},
	[714] = {3, 4, 10, 11},
	[716] = {7, 9},
	[717] = {5, 6, 9, 12},
	[718] = {3, 8, 9},
	[722] = {3, 9, 10, 12},
	[723] = {3, 6, 9},
	[724] = {5, 10},
	[725] = {3, 11},
	[726] = {3, 5, 7},
	[727] = {3, 5, 8, 11, 12},
	[728] = {3, 5, 11, 12},
	[729] = {3, 5, 7, 9, 11, 12},
	[730] = {3, 5, 9, 11, 12},
	[731] = {7, 9},
	[732] = {8, 9},
	[733] = {3, 5, 11},
	[737] = {5, 8},
	[739] = {5, 8},
	[740] = {3, 9, 10, 11, 12},
	[741] = {3, 12},
	[742] = {3, 5},
	[743] = {3, 5, 12},
	[744] = {5, 6, 7, 9, 11, 12},
	[745] = {3, 9},
	[746] = {4, 6},
	[747] = {3, 9, 10, 12},
	[748] = {3, 5, 7, 8, 9, 10, 11, 12},
	[749] = {5, 11},
	[750] = {3, 5, 11},
	[751] = {3, 9, 11, 12},
	[752] = {3, 4, 12},
	[753] = {3, 9},
	[754] = {3, 5, 7, 8, 9, 11, 12},
	[755] = {8, 10},
	[756] = {3, 5, 9},
	[757] = {5},
	[758] = {8},
	[792] = {3, 12},
	[802] = {4},
	[817] = {7},
	[818] = {8},
	[819] = {5},
	[820] = {12},
	[821] = {11},
	[823] = {3, 5, 9, 12},
	[834] = {7},
	[835] = {5},
	[836] = {5},
	[837] = {3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
	[838] = {3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
	[844] = {5},
	[845] = {3, 8, 10},
	[846] = {5},
	[847] = {12},
	[848] = {3, 5, 9},
	[849] = {3},
	[850] = {3},
	[851] = {3, 5, 7},
	[855] = {3},
	[856] = {9, 12},
	[868] = {9},
	[872] = {3},
	[873] = {3},
	[874] = {3},
	[875] = {8},
	[876] = {7},
	[877] = {9},
	[878] = {3},
	[879] = {7},
	[880] = {7},
	[881] = {8},
	[882] = {9},
	[883] = {6},
	[884] = {8},
	[885] = {8},
	[886] = {9},
	[887] = {8},
	[888] = {3},
	[889] = {3},
	[890] = {3},
	[891] = {5},
	[892] = {8},
	[893] = {3},
	[894] = {5},
	[895] = {8},
	[896] = {6},
	[897] = {9},
	[898] = {3},
	[899] = {8},
	[900] = {6},
	[901] = {6},
	[902] = {9},
	[903] = {10},
	[904] = {9},
	[905] = {8},
	[906] = {5},
	[907] = {9},
	[908] = {8},
	[909] = {5},
	[911] = {7},
	[912] = {4},
	[913] = {8},
	[915] = {7},
	[916] = {8},
	[917] = {6},
	[921] = {7},
	[922] = {4},
	[923] = {6},
	[924] = {9},
	[925] = {7},
	[926] = {8},
	[927] = {5},
	[928] = {8},
	[929] = {9},
	[931] = {9},
	[932] = {6},
	[933] = {5},
	[934] = {6},
	[935] = {8},
	[936] = {3},
	[937] = {8},
	[938] = {9},
	[939] = {7},
	[941] = {8},
	[942] = {6},
	[943] = {9},
	[944] = {6},
	[945] = {7},
	[946] = {5},
	[947] = {9},
	[948] = {5},
	[949] = {8},
	[950] = {6},
	[951] = {7},
	[952] = {7},
	[953] = {6},
	[954] = {3},
	[955] = {5},
	[956] = {6},
	[957] = {8},
	[958] = {3},
	[959] = {8},
	[960] = {4},
	[961] = {4},
	[962] = {8},
	[963] = {7},
	[964] = {3},
	[965] = {3},
	[966] = {9},
	[967] = {8},
	[968] = {7},
	[969] = {9},
	[970] = {8},
	[971] = {7},
	[972] = {9},
	[973] = {8},
	[974] = {8},
	[975] = {6},
	[976] = {7},
	[977] = {9},
	[978] = {8},
	[979] = {6},
	[980] = {6},
	[981] = {8},
	[982] = {5},
	[983] = {6},
	[984] = {9},
	[985] = {7},
	[986] = {4},
	[987] = {9},
	[988] = {8},
	[989] = {9},
	[990] = {5},
	[991] = {8},
	[992] = {6},
	[993] = {5},
	[994] = {8},
	[995] = {9},
	[996] = {6},
	[997] = {7},
	[998] = {9},
	[999] = {6},
	[1000] = {8},
	[1001] = {8},
	[1002] = {9},
	[1003] = {9},
	[1004] = {9},
	[1005] = {9},
	[1006] = {6},
	[1007] = {4},
	[1008] = {6},
	[1009] = {7},
	[1010] = {8},
	[1011] = {7},
	[1012] = {5},
	[1013] = {3, 6, 9},
	[1039] = {6},
	[1040] = {6},
	[1042] = {8},
	[1061] = {5},
	[1062] = {3, 9, 10, 12},
	[1063] = {8},
	[1065] = {8},
	[1066] = {10},
	[1067] = {7},
	[1068] = {3, 5, 8, 10},
	[1073] = {12},
	[1117] = {3},
	[1124] = {8},
	[1125] = {3},
	[1126] = {7},
	[1127] = {8},
	[1128] = {3, 12},
	[1129] = {7},
	[1130] = {8},
	[1131] = {11},
	[1132] = {5},
	[1133] = {10},
	[1134] = {9},
	[1135] = {12},
	[1136] = {3},
	[1137] = {6},
	[1138] = {9},
	[1139] = {8},
	[1140] = {3},
	[1141] = {7},
	[1142] = {11},
	[1143] = {8},
	[1144] = {9},
	[1145] = {5},
	[1146] = {7},
	[1147] = {4},
	[1149] = {8},
	[1150] = {9},
	[1151] = {7},
	[1152] = {6},
	[1153] = {4},
	[1154] = {9},
	[1155] = {6},
	[1156] = {8},
	[1157] = {3, 12},
	[1158] = {12},
	[1159] = {12},
	[1160] = {8},
	[1161] = {4, 8, 10},
	[1162] = {5, 8},
	[1163] = {6, 7},
	[1164] = {3, 5, 11},
	[1165] = {4, 8, 10},
	[1166] = {4, 8, 10},
	[1167] = {4, 8, 10},
	[1168] = {12},
	[1174] = {12},
	[1175] = {3, 12},
	[1176] = {3},
	[1177] = {6, 7, 9, 10},
	[1178] = {4},
	[1179] = {3},
	[1180] = {4, 5, 8, 10},
	[1181] = {6, 9},
	[1182] = {3, 12},
	[1183] = {4},
	[1184] = {4},
	[1185] = {9},
	[1187] = {4},
	[1188] = {3},
	[1189] = {9},
	[1190] = {11},
	[1191] = {6},
	[1192] = {10},
	[1193] = {12},
	[1194] = {8},
	[1195] = {5},
	[1196] = {9},
	[1197] = {7},
	[1198] = {8},
	[1200] = {8},
	[1201] = {10},
	[1202] = {3, 10},
	[1204] = {11}, -- CURRENTLY UNUSED Pierre
	[1205] = {9, 10},
	[1206] = {10},
	[1207] = {12},
	[1208] = {11},
	[1209] = {3},
	[1211] = {4, 5, 8, 10},
	[1212] = {4, 5, 8, 10},
	[1213] = {4, 5, 8, 10},
	[1226] = {8},
	[1227] = {6, 7},
	[1229] = {5, 10},
	[1231] = {3, 10},
	[1232] = {3},
	[1233] = {6, 8},
	[1234] = {7},
	[1235] = {5, 11},
	[1237] = {8},
	[1238] = {3, 6},
	[1244] = {7},
	[1245] = {3},
	[1248] = {3},
}

--[[
local a,b = 0,0
for k,v in pairs(lib.breedData.speciesToAvailableBreeds) do a = a+1 end
for k,v in pairs(lib.breedData.speciesToBaseStatProfile) do b = b+1 end

missingPets = ""
for k,v in pairs(lib.breedData.speciesToBaseStatProfile) do 
	if not lib.breedData.speciesToAvailableBreeds[k] then
		missingPets = missingPets..k..", "
		print(k)
	end
end


print(a,b)]]
