local L = LibStub("AceLocale-3.0"):GetLocale("Pokedex")

-- upvalues
local _G, Pokedex, LibStub, CreateFrame, GetBindingKey, InterfaceOptionsFrame_OpenToCategory, SetBindingClick, UIParent = _G, Pokedex, LibStub, CreateFrame, GetBindingKey, InterfaceOptionsFrame_OpenToCategory, SetBindingClick, UIParent
local ceil, floor, format, pairs, select, tonumber = ceil, floor, format, pairs, select, tonumber
local ctoy, GetAchievementInfo, GetItemInfo, GetSpellInfo, UnitFactionGroup = C_ToyBox, GetAchievementInfo, GetItemInfo, GetSpellInfo, UnitFactionGroup


local IS = Pokedex.Globals.Types.InitStates;     -- Initialization States
local SL = Pokedex.Globals.Types.SkillLevels;    -- minimum skill level to hold that rank of profession
local DL = Pokedex.Globals.Types.DebugLevels;    -- Debug Levels
local gc = Pokedex.Globals.Constants;
local gv = Pokedex.Globals.Variables;
local gs = Pokedex.Globals.SortMethods;
local gf = Pokedex.Globals.FormatMethods;


function Pokedex:LoadSavedSettings()
	-- saved settings and default values
	local defaults = {
		global = {
			rgDebugInfo = { MISC     = 0,
							MOUNTS   = 0,
							PETS     = 0,
							TITLES   = 0,
							DISMOUNT = 0,
			},
		},
		char = {
			mounts = {
				fEnableHotness = true,
				fNewHotness = true,
				idHot = 0,
				iHeat = 50,
				ranks = {},       -- { key(string) is name of mount, value(number) is rank }
			},
			pets = {
				fEnableHotness = true,
				fNewHotness = true,
				idHot = 0,
				iHeat = 50,
				ranks = {},       -- { key(string) is name of mount, value(number) is rank }
			},
			titles = {
				fEnableHotness = true,
				fNewHotness = true,
				idHot = -1,
				iHeat = 50,
				ranks = {},       -- { key(string) is name of mount, value(number) is rank }
			},
		},
		profile = {
			UseAsMount = {
				NagrandCorral = true,
				ItemFishingRaft = true,
				ItemRatstallionHarness = true,
				DruidForms = true,
				MonkRoll = false,
				ShamanGhostWolf = true,
				WarlockBurningRush = true,
			},
			CastWithMount = {
				DKPathOfFrost = true,
				DruidCenarionWard = false,
				MageBlazingBarrier = true,
				MageIceBarrier = true,
				MagePrismaticBarrier = true,
				RogueRemoveDisguise = true,
				ShamanWaterWalking = true,
			},
			CastWhenFalling = {
				DemonGlide = true,
				DruidFlap = true,
				EngrGoblinGlider = true,
				EngrFlexweaveUnderlay = true,
				ItemGoblinGliderKit = true,
				ItemGoldenGlider = true,
				ItemSkyhornKite = true,
				ItemSnowfallLager = true,
				MageSlowFall = true,
				MonkZenFlight = true,
				PaladinDivineShield = true,
				PaladinHandOfProtection = true,
				PriestLevitate = true,
				RogueLegendaryDaggers = true,
			},
			SafeDismount = {
				Enabled = false,
				ForCombat = false,
				ForAttack = false,
				ForGathering = false,
			},
			fAnnounce = true,
			iChannel = 1,
			fChangeTitleOnMount = false,
			fAutoSummonCompanion = true,
			fFavorGroundMounts = true,
			mounts = {
				iDefaultRank = 50,
				iMaximumRank = 100,
				ranks = {},       -- { key(string) is name of mount, value(number) is rank }
				sort = "ByName",
				format = "Name",
			},
			pets = {
				iDefaultRank = 50,
				iMaximumRank = 100,
				ranks = {},         -- { key(string) is name of pet species, value(number) is rank }
				sort = "ByName",
				format = "Name",
			},
			titles = {
				iDefaultRank = 50,
				iMaximumRank = 100,
				ranks = {},         -- { key(string) is name of title, value(number) is rank }
				sort = "ByName",
				format = "Name",
			},
		},
	}

	self.db = LibStub("AceDB-3.0"):New("PokedexDB", defaults, true) --, _G.DEFAULT)

	-- we should be able to migrate saved data formats immediately
	self:MigrateSavedData();

	self.db.RegisterCallback(self, "OnProfileChanged", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileCopied",  "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileReset",   "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileDeleted", "OnProfileDeleted")
end

function Pokedex:OnProfileChanged()
	-- the profile they changed to may still need migration
	self:MigrateSavedData();

	-- rebuild internals for changes
	-- REVIEW - does this cover it?
	self:UpdateDismountSettings();
	self:UpdateMountInfo()
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:OnProfileDeleted(_, _, name)
	for k, v in pairs(self.db.sv.profileKeys) do
		if v == name then
			self.db.sv.profileKeys[v] = nil
		end
	end
end

-- As the addon develops and ages, there may be changes to the structure 
-- and contents of the saved data. This function migrates data between
-- data format versions to preserve the users settings.
function Pokedex:MigrateSavedData()
	-- different sections of saved settings each need to be stamped with the
	-- current Format number, i.e. db.global, db.char, db.profile, etc
	local profiledb = self.db.profile
	local  globaldb = self.db.global
	local    chardb = self.db.char

	-- If the saved format value for profile is nil, then its a brand new profile
	-- and no migration will be needed. Treat it as being stamped current.
	local iFormatProfile = profiledb.iDataFormat or gc.iCurDataFormat

	-- If the saved format values for char and global are nil, set the value equal
	-- to iFormatProfile. This lets us skip migration when there is nothing to migrate,
	-- but does allow us to do some needed work initing the new sections as part of
	-- the big profile reorg done as part of formats 11-12.
	local iFormatGlobal = globaldb.iDataFormat or iFormatProfile
	local iFormatChar   =   chardb.iDataFormat or iFormatProfile

	-- note that in a given session you might only migrate one section, 
	-- like char, as global or profile could have already been migrated
	if (gc.iCurDataFormat == iFormatProfile and 
		gc.iCurDataFormat == iFormatGlobal and
		gc.iCurDataFormat == iFormatChar) then
		-- all sections up to date, exit
		profiledb.iDataFormat = gc.iCurDataFormat
		globaldb.iDataFormat  = gc.iCurDataFormat
		chardb.iDataFormat    = gc.iCurDataFormat
		return;
	end

	-- function assumes new default value is same as old
	local function Migrate(fromTable, fromName, toTable, toName)
		if fromTable and fromTable[fromName] ~= nil then
			toTable[toName] = fromTable[fromName];
			fromTable[fromName] = nil;
		end
	end

	-- no longer supporting migration of settings formats older than 8.
	-- some functions required in migrating some steps have been removed or changed
	-- too much over time to work in the way expected by some of the migrations steps
	if iFormatProfile < 8 then
		profiledb.idHotCritter = nil;
		profiledb.idHotCritterPref = nil;
		profiledb.rgCritterInfo = nil;
		profiledb.fChangeTitleOnLoad = nil;
		profiledb.rgDebugInfo = nil;
		profiledb.fCombineAllFastMounts = nil;
		profiledb.fCombineFastAndSlowMounts = nil;

		profiledb.rgMountInfo = {};
		profiledb.rgCompanionInfo = {};
		profiledb.rgTitleInfo = {};
	end

	-- migrate keybind from old togglemount action to new togglemount button
--[==[ REVIEW - this should really only happend if keybindindings are set to be per user
		TODO - find how to check that setting	
	if iFormatProfile < 9 then
		local bind1, bind2 = GetBindingKey("POKEDEXTOGGLEMOUNT")
		if (bind1) then SetBindingClick(bind1, "PokedexToggleMountButton", "LeftButton"); end
		if (bind2) then SetBindingClick(bind2, "PokedexToggleMountButton", "LeftButton"); end

		-- migration hasn't been working well so we'll also just toss up a warning
		local popup = CreateFrame("Frame", "PokedexFormWarning", UIParent);
		popup:SetPoint("CENTER", UIParent, "CENTER")
		popup:SetFrameStrata("DIALOG")
		popup:SetHeight(150)
		popup:SetWidth(300)
		popup:SetBackdrop({
			bgFile = "Interface/Tooltips/ChatBubble-Background",
			edgeFile = "Interface/Tooltips/ChatBubble-BackDrop",
			tile = true, tileSize = 32, edgeSize = 32,
			insets = { left = 32, right = 32, top = 32, bottom = 32 }
		})
		popup:SetBackdropColor(0,0,0, 1)

		popup.close = CreateFrame("Button", "", popup, "OptionsButtonTemplate")
		popup.close:SetText("Close")
		popup.close:SetPoint("BOTTOMRIGHT", popup, "BOTTOMRIGHT", -10, 10)
		popup.close:SetScript("OnClick", function() popup:Hide() end )

		popup.warning = popup:CreateFontString("", "OVERLAY", "GameFontNormal")
		popup.warning:SetJustifyH("LEFT")
		popup.warning:SetJustifyV("TOP")
		popup.warning:SetPoint("TOPLEFT", popup, "TOPLEFT", 10, -10)
		popup.warning:SetPoint("BOTTOMRIGHT", popup.close, "TOPRIGHT", 0, 5)
		popup.warning:SetText("POKEDEX - Please note that you will need to re-keybind the Toggle Mount command and/or edit any macros which use ToggleMount due to changes made to support new features like Druid forms. Please see the README.TXT file in the addon directory for more details.");

		popup:Show()
	end
--]==]

	-- migrate pet ranks into new format
	if iFormatProfile < 10 then
		profiledb.rgPetRanks = {}
		local iDefault = profiledb.pets.iDefaultRank
		for _, v in pairs(profiledb.rgCompanionInfo) do
			if iDefault ~= v.rank then
				profiledb.rgPetRanks[v.name] = v.rank;
			end
		end
	end


	-- reorganization of settings for profile sharing checkin
	-- new format for mount and title info
	if iFormatProfile < 11 then
		-- ensure subtables are there (future proofing the migration)
		profiledb.mounts = profiledb.mounts or { ranks = {} }
		profiledb.pets   = profiledb.pets   or { ranks = {} }
		profiledb.titles = profiledb.titles or { ranks = {} }

		profiledb.fFirstBoot = nil

		-- migrate mount ranks into new format
		local iDefault = profiledb.mounts.iDefaultRank
		for _, v in pairs(profiledb.rgMountInfo) do
			if iDefault ~= v.rank then
				profiledb.mounts.ranks[v.name] = v.rank;
			end
		end
		
		-- remove old mount info
		profiledb.rgMountInfo = nil;

		-- move location of pet ranks
		profiledb.pets.ranks = profiledb.rgPetRanks;
		profiledb.rgPetRanks = nil;

		-- remove old pet rank format and non-migratable hotid
		profiledb.rgCompanionInfo = nil;
		profiledb.idHotCompanion = nil;


		-- migrate title ranks into new format
		local iDefault = profiledb.titles.iDefaultRank
		for _, v in pairs(profiledb.rgTitleInfo) do
			if iDefault ~= v.rank then
				profiledb.titles.ranks[v.name] = v.rank;
			end
		end

		-- fix up the No Title entry
		local none = profiledb.rgTitleInfo[0]
		if none then
			profiledb.titles.ranks[none.name] = nil
			profiledb.titles.ranks[_G.PLAYER_TITLE_NONE] = none.rank
		else
			profiledb.titles.ranks[_G.PLAYER_TITLE_NONE] = 0
		end

		-- remove old title info
		profiledb.rgTitleInfo = nil;
	end

	-- reorganization of settings for profile sharing checkin
	-- new format for mount and title info
	if iFormatChar < 11 then
		-- ensure subtables are there (future proofing the migration)
		chardb.mounts = chardb.mounts or { ranks = {} }
		chardb.pets   = chardb.pets   or { ranks = {} }
		chardb.titles = chardb.titles or { ranks = {} }

		Migrate(chardb, "idHotPet", chardb.pets, "idHot")  -- move pet hot id
		chardb.idHotCompanion = nil;

		-- only migrate and clean up an existing char - realm profile that 
		-- matches the current character
		if self.db:GetCurrentProfile() == _G.format("%s - %s", _G.UnitName("player"), _G.GetRealmName()) then
			Migrate(profiledb, "fEnableHotness", chardb, "fEnableHotness") 
			Migrate(profiledb, "fNewHotness",    chardb, "fNewHotness")
			Migrate(profiledb, "idHotMount",        chardb.mounts, "idHot")
			Migrate(profiledb, "iHotMountPref",     chardb.mounts, "iHeat")
			Migrate(profiledb, "iHotCompanionPref", chardb.pets,   "iHeat")
			Migrate(profiledb, "idHotTitle",        chardb.titles, "idHot")
			Migrate(profiledb, "iHotTitlePref",     chardb.titles, "iHeat")
		end
	end

	-- Cause a Default profile to be created for the legacy user so that
	-- it will be in their dropdowns ready to be selected and copied into.
	if iFormatGlobal < 12 then
		local profiles = self.db:GetProfiles()
		if not profiles[_G.DEFAULT] then
			local current = self.db:GetCurrentProfile()
			self.db:SetProfile(_G.DEFAULT)
			self.db:SetProfile(current)
		end
	end

	-- enable hotness and new hotness with seperate settings for titles, pets and mounts
	if iFormatChar < 13 then
		if chardb.fEnableHotness ~= nil then
			chardb.titles.fEnableHotness = chardb.fEnableHotness
			chardb.mounts.fEnableHotness = chardb.fEnableHotness
			chardb.pets.fEnableHotness   = chardb.fEnableHotness
			chardb.fEnableHotness = nil
		end

		if chardb.fNewHotness ~= nil then
			chardb.titles.fNewHotness = chardb.fNewHotness
			chardb.mounts.fNewHotness = chardb.fNewHotness
			chardb.pets.fNewHotness   = chardb.fNewHotness
			chardb.fNewHotness = nil
		end
	end
	
	-- move some profile settings into subgroups
	if iFormatProfile < 13 then
		-- ensure subtables are there (future proofing the migration)
		profiledb.UseAsMount      = profiledb.UseAsMount      or {}
		profiledb.CastWithMount   = profiledb.CastWithMount   or {}
		profiledb.CastWhenFalling = profiledb.CastWhenFalling or {}
		profiledb.SafeDismount    = profiledb.SafeDismount    or {}

		Migrate(profiledb, "fDruidUseForms",         profiledb.UseAsMount, "DruidForms")
		Migrate(profiledb, "fMonkUseRoll",           profiledb.UseAsMount, "MonkRoll")
		Migrate(profiledb, "fShamanUseGhostWolf",    profiledb.UseAsMount, "ShamanGhostWolf")
		Migrate(profiledb, "fWarlockUseBurningRush", profiledb.UseAsMount, "WarlockBurningRush")

		Migrate(profiledb, "fDKCastHornWithMount", profiledb.CastWithMount, "DKHornOfWinter")
		Migrate(profiledb, "fRogueRemoveDisguise", profiledb.CastWithMount, "RogueRemoveDisguise")
		Migrate(profiledb, "fWarlockCastWaterWalkingWithMount", profiledb.CastWithMount, "WarlockWaterWalking")

		-- the default for shaman water walking is changing to true since there are no reagent/glyph requirements anymore
		profiledb.CastWithMount.ShamanWaterWalking = profiledb.fShamanCastWaterWalkingWithMount or false
		profiledb.fShamanCastWaterWalkingWithMount = nil

		Migrate(profiledb, "fMonkUseZenFlight", profiledb.CastWhenFalling, "MonkZenFlight")
		
		Migrate(profiledb, "fManageAutoDismount",   profiledb.SafeDismount, "Enabled")
		Migrate(profiledb, "fDismountForGathering", profiledb.SafeDismount, "ForGathering")
		Migrate(profiledb, "fDismountForCombat",    profiledb.SafeDismount, "ForCombat")
		Migrate(profiledb, "fDismountForAttack",    profiledb.SafeDismount, "ForAttack")
	end

	-- default and max rank moved from 5 and 10 to 50 and a 100, multiply all ranks by 10
	local bumpranks14 = function(t)
		for key, rank in pairs(t) do
			t[key] = 10 * rank
		end
	end
	
	if iFormatChar < 14 then
		bumpranks14(chardb.mounts.ranks)
		bumpranks14(chardb.pets.ranks)
		bumpranks14(chardb.titles.ranks)
	end

	if iFormatProfile < 14 then
		bumpranks14(profiledb.mounts.ranks)
		bumpranks14(profiledb.pets.ranks)
		bumpranks14(profiledb.titles.ranks)
	end
	
	-- WoD patch
	if iFormatProfile < 15 then
		-- Unending Breath can no longer be Soulburned to give Water Walking
		profiledb.CastWithMount.WarlockWaterWalking = nil
	end

	-- Legion patch
	if iFormatProfile < 16 then
		profiledb.CastWithMount.DruidCenarionWard = false
		profiledb.CastWithMount.DKHornOfWinter = nil
		profiledb.CastWithMount.PaladinSacredShield = nil

		profiledb.CastWhenFalling.DKPathOfFrost = nil
		profiledb.CastWhenFalling.PaladinFallingAvenger = nil
	end

	profiledb.iDataFormat = gc.iCurDataFormat
	globaldb.iDataFormat  = gc.iCurDataFormat
	chardb.iDataFormat    = gc.iCurDataFormat
end

local EnsureRankBounds = function(type)
	local maximum = Pokedex.db.profile[type].iMaximumRank
	local default = Pokedex.db.profile[type].iDefaultRank
	local prof_ranks = Pokedex.db.profile[type].ranks
	local char_ranks = Pokedex.db.char[type].ranks

	for name,rank in pairs(char_ranks) do
		if rank > maximum then char_ranks[name] = maximum end
	end

	for name,rank in pairs(prof_ranks) do
		if rank > maximum then prof_ranks[name] = maximum end
	end

	for name,rank in pairs(prof_ranks) do
		if rank == default then prof_ranks[name] = nil end
	end
end

--=========================================================================--
-- Options UI definition
--=========================================================================--

local function newCounter()
	local i = 0
	return function()
		i = i + 1
		return i
	end
end

local function GroupContentHidden(info)
	local name = info[#info]
	local args = info.option.args
	for k,_ in pairs(Pokedex.db.profile[name]) do
		if not args[k].hidden then
			return false
		end
	end
	return true
end

function Pokedex.GetUIOptions(uiType, uiName, appName)
	-- Pokedex:Print("GetUIOptions called", uiType, uiName, appName)

	if (not Pokedex.options) then
		if (gv.InitState ~= IS.INITIALIZED and Pokedex:Initialize("GetUIOptions") == IS.UNINITIALIZED) then
			return {
				name = gc.strVersionedTitle,
				handler = Pokedex,
				type = "group",
				childGroups = "tab",
				args = {
					CtrlInitError = {
						order = 1,
						type = "description",
						name = L["ERROR: Pokedex failed to initialize correctly. This is usually caused when WoW has invalidated its cache and hasn't finished rebuilding it. Please try this action again later."],
					},
					Padding_InitError = {
						order = 2,
						type = "description",
						name = "",
					},
					CtrlRetryInit = {
						order = 3,
						type = "execute",
						name = L["retry initialization"],
						desc = L["retry initialization"],
						func = function() InterfaceOptionsFrame_OpenToCategory(Pokedex.optionsFrame); end,
					}
				}
			}
		end

		Pokedex.options = {}

		-- start options.commands --- commands executable from command line
		Pokedex.options.commands = {
			name = gc.strVersionedTitle,
			handler = Pokedex,
			type = "group",
			args = {
				test = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["echoes out test info relevant to current feature in development"],
					desc = L["echoes out test info relevant to current feature in development"],
					func = function(info) Pokedex:EchoTest() end,
				},
				debug = {
					hidden = true,
					order = 0,
					type = "input",
					name = L["echoes out test info relevant to current feature in development"],
					desc = L["echoes out test info relevant to current feature in development"],
					get = function(info) return "" end,
					set = "SetDebug",
				},
				speed = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["echoes out current speed"],
					desc = L["echoes out current speed"],
					func = function(info) Pokedex:EchoSpeed() end,
				},
				zone = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["echoes out zone info"],
					desc = L["echoes out zone info"],
					func = function(info) Pokedex:EchoZone() end,
				},
				SummonMount = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon mount"],
					desc = L["summons a mount"],
					func = function(info) Pokedex:SummonMount(false) end,
				},
				sm = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon mount"],
					desc = L["summons a mount"],
					func = function(info) Pokedex:SummonMount(false) end,
				},
				DismissMount = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["dismiss mount"],
					desc = L["dismisses current mount"],
					func = function(info) Pokedex:DismissMount() end,
				},
				dm = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["dismiss mount"],
					desc = L["dismisses current mount"],
					func = function(info) Pokedex:DismissMount() end,
				},
				ToggleMount = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["toggle mount"],
					desc = L["toggles a mount"],
					func = function(info) Pokedex:ToggleMount() end,
				},
				tm = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["toggle mount"],
					desc = L["toggles a mount"],
					func = function(info) Pokedex:ToggleMount() end,
				},
				SummonNextMount = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon next mount"],
					desc = L["summons next mount in collection"],
					func = function(info) Pokedex:SummonNextMount() end,
				},
				snm = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon next mount"],
					desc = L["summons next mount in collection"],
					func = function(info) Pokedex:SummonNextMount() end,
				},
				SummonOtherMount = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon other mount"],
					desc = L["if summon mount would summon flyer, will summon ground mount and vice versa"],
					func = function(info) Pokedex:SummonOtherMount() end,
				},
				som = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon other mount"],
					desc = L["if summon mount would summon flyer, will summon ground mount and vice versa"],
					func = function(info) Pokedex:SummonOtherMount() end,
				},
				SummonCompanion = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon companion"],
					desc = L["summons a companion"],
					func = function(info) Pokedex:SummonCompanion(false) end,
				},
				sc = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon companion"],
					desc = L["summons a companion"],
					func = function(info) Pokedex:SummonCompanion(false) end,
				},
				DismissCompanion = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["dismiss companion"],
					desc = L["dismisses current companion"],
					func = function(info) Pokedex:DismissCompanion() end,
				},
				dc = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["dismiss companion"],
					desc = L["dismisses current companion"],
					func = function(info) Pokedex:DismissCompanion() end,
				},
				ToggleCompanion = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["toggle companion"],
					desc = L["toggles a companion"],
					func = function(info) Pokedex:ToggleCompanion() end,
				},
				tc = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["toggle companion"],
					desc = L["toggles a companion"],
					func = function(info) Pokedex:ToggleCompanion() end,
				},
				SummonNextCompanion = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon next companion"],
					desc = L["summons next companion in collection"],
					func = function(info) Pokedex:SummonCompanion(true) end,
				},
				snc = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon next companion"],
					desc = L["summons next companion in collection"],
					func = function(info) Pokedex:SummonCompanion(true) end,
				},
				train = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon train wrecker"],
					desc = L["summon a Lil' XT or Landro's Lil' XT to destroy a train set"],
					func = function(info) Pokedex:SummonTrainWrecker() end,
				},
				vendor = {  -- mammoth, yak or squire
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon vendor"],
					desc = L["summons a mount or companion with vendor capabilities"],
					func = function(info) Pokedex:SummonVendor() end,
				},
				squire = {  -- squire only
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon squire"],
					desc = L["summon argent squire or gruntling pet"],
					func = function(info) Pokedex:SummonVendor(false, true, true) end,
				},
				repair = {  -- mammoth or yak
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon repair mount"],
					desc = L["summons a mount with repair capabilities"],
					func = function(info) Pokedex:SummonVendor(true) end,
				},
				yak = {  -- yak only
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon Grand Expedition Yak"],
					desc = L["summon a Grand Expedition Yak for repairs or transmogs"],
					func = function(info) Pokedex:SummonVendor(true, true) end,
				},
				tmog = {  -- yak only
					hidden = true,
					order = 0,
					type = "execute",
					name = L["summon a transmorgifier"],
					desc = L["summons a mount or companion with transmog capabilities"],
					func = function(info) Pokedex:SummonVendor(true, true) end,
				},
				ChangeTitle = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["change title"],
					desc = L["change title"],
					func = function() Pokedex:ChangeTitle() end,
				},
				ct = {
					hidden = true,
					order = 0,
					type = "execute",
					name = L["change title"],
					desc = L["change title"],
					func = function() Pokedex:ChangeTitle() end,
				},
			},
		}  -- end options.commands --- commands executable from command line

		-- start options.gui --- controls to display in option guis
		local count = newCounter()
		Pokedex.options.gui = {
			name = gc.strVersionedTitle,
			handler = Pokedex,
			type = "group",
			childGroups = "tab",
			args = {
				CtrlReadMe = {
					order = count(),
					type = "description",
					name = L["Please see the README.TXT file in the Pokedex addon folder for more information on how to use Pokedex"],
				},
				CtrlGroupMounts = {
					order = count(),
					name = L["Mounts"],
					handler = Pokedex,
					type = "group",
					args = {
						CtrlGroupRanks = {
							order = count(),	
							name = L["Ranks"],
							handler = Pokedex,
							type = "group",
							dialogInline = true,
							args = {
								CtrlFavorGroundMounts = {
									order = count(),
									type = "toggle",
									name = L["favor ground mounts"],
									desc = L["ensures that in areas where flying is not allowed, we will filter out the flying mounts that are still usable (gryphons, wind riders, etc) in favor of ground only mounts (horses, kodos, etc)"],
									get = function(info) return Pokedex.db.profile.fFavorGroundMounts end,
									set = function(info, value) 
										Pokedex.db.profile.fFavorGroundMounts = value 
										Pokedex:UpdateMountInfo()
									end,
								},
								CtrlMountTypes = {
									order = count(),
									type = "select",
									name = L["select mount type"],
									desc = L["select mount type"],
									values = function(info) return gv.Mounts.Lists.UITypesDisplay end,
									disabled = function(info) return (gv.Mounts.count == 0) end,
									get = function(info) return gv.SelectedMountBucket end,
									set = function(info, value) 
											gv.SelectedMountBucket = value
											if not value:MountInBucket(gv.SelectedMount) then
												gv.SelectedMount = value:GetFirstMount()
											end
											-- chance gv.Mounts:UpdateDisplayNames();
										end,
								},
								Padding_MountTypes = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlMountsOfType = {
									order = count(),
									type = "select",
									width = "double",
									name = L["select mount for ranking"],
									desc = L["mount whose rank you can set"],
									values = function(info) return gv.SelectedMountBucket.mounts end,
									disabled = function(info) return (gv.Mounts.count == 0) end,
									get = function(info) return gv.SelectedMount end,
									set = function(info, value) gv.SelectedMount = value end,
								},
								CtrlRankingForMount = {
									order = count(),
									type = "range",
									name = L["mount rank"],
									desc = L["rank of current mount"],
									min = 0,
									max = Pokedex.db.profile.mounts.iMaximumRank,
									step = 1,
									disabled = function(info) return (gv.Mounts.count == 0) end,
									get = function(info) return gv.SelectedMount.profile_rank end,
									set = function(info, value) 
											gv.SelectedMount.profile_rank = value;
											gv.Mounts:UpdateDisplayNames(gv.SelectedMount);
										end,
								},
								Padding_MountRankings = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlMountOverrideRank = {
									order = count(),
									type = "toggle",
									width = "double",
									name = L["set rank override"],
									desc = L["set a character specific value for rank"],
									disabled = function(info) return (gv.Mounts.count == 0) end,
									get = function(info) return gv.SelectedMount.char_rank ~= nil end,
									set = function(info, value) 
											gv.SelectedMount.char_rank = value and gv.SelectedMount.rank or nil;
											if not value then gv.Mounts:UpdateDisplayNames(gv.SelectedMount) end;
										end,
								},
								CtrlRankingForMountOverride = {
									order = count(),
									type = "range",
									name = L["override rank"],
									desc = L["character specific rank"],
									min = 0,
									max = Pokedex.db.profile.mounts.iMaximumRank,
									step = 1,
									disabled = function(info) return (gv.Mounts.count == 0 or gv.SelectedMount.char_rank == nil) end,
									get = function(info) return gv.SelectedMount.char_rank end,
									set = function(info, value)
											gv.SelectedMount.char_rank = value;
											gv.Mounts:UpdateDisplayNames(gv.SelectedMount);
										end,
								},
								Padding_MountRankingOverride = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlMountSort = {
									order = count(),
									type = "select",
									name = L["sorting method"],
									desc = L["sorting method"],
									values = gs.DescByIndex,
									get = function(info) return gs.IndexByName[Pokedex.db.profile.mounts.sort] end,
									set = function(info, value)
											local name = gs.NameByIndex[value]
											Pokedex.db.profile.mounts.sort = name
											gs.Mounts = gs[name]
										 end,
								},
								CtrlMountFormat = {
									order = count(),
									type = "select",
									name = L["mount format"],
									desc = L["mount format"],
									values = gf.DescByIndex,
									get = function(info) return gf.IndexByName[Pokedex.db.profile.mounts.format] end,
									set = function(info, value)
											local name = gf.NameByIndex[value]
											Pokedex.db.profile.mounts.format = name
											gf.Mounts = gf[name]
											gv.Mounts:UpdateDisplayNames()
										 end,
								},
								Padding_MountDisplay = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlMountDefaultRank = {
									order = count(),
									type = "input",
									name = L["default rank"],
									desc = L["default rank"],
									get = function(info) return format("%u", Pokedex.db.profile.mounts.iDefaultRank) end,
									set = function(info, value)
											Pokedex.db.profile.mounts.iDefaultRank = tonumber(value) or Pokedex.db.profile.mounts.iDefaultRank
											EnsureRankBounds("mounts")
											gv.Mounts:UpdateDisplayNames()
										 end,
									validate = function(info, value)
											local iValue = tonumber(value)
											if not iValue or (floor(iValue) ~= ceil(iValue)) or iValue < 0 then
												local msg = L["ERROR: value must be a whole number"]
												Pokedex:Print(msg)
												return msg
											elseif iValue > Pokedex.db.profile.mounts.iMaximumRank then
												local msg = L["ERROR: default rank cannot be greater than the maximum rank"]
												Pokedex:Print(msg)
												return msg
											else
												return true
											end
										end,
								},
								CtrlMountMaximumRank = {
									order = count(),
									type = "input",
									name = L["maximum rank"],
									desc = L["maximum rank"],
									get = function(info) return format("%u", Pokedex.db.profile.mounts.iMaximumRank) end,
									set = function(info, value)
											local value = tonumber(value) or Pokedex.db.profile.mounts.iMaximumRank
											Pokedex.db.profile.mounts.iMaximumRank = value
											info.options.args.CtrlGroupMounts.args.CtrlRankingForMount.max = value
											info.options.args.CtrlGroupMounts.args.CtrlRankingForMountOverride.max = value
											EnsureRankBounds("mounts")
											gv.Mounts:UpdateDisplayNames()
										 end,
									validate = function(info, value)
											local iValue = tonumber(value)
											if not iValue or (floor(iValue) ~= ceil(iValue)) or iValue < 1 then
												local msg = L["ERROR: value must be a positive integer"]
												Pokedex:Print(msg)
												return msg
											elseif iValue < Pokedex.db.profile.mounts.iDefaultRank then
												local msg = L["ERROR: maximum rank cannot be less than the default rank"]
												Pokedex:Print(msg)
												return msg
											else
												return true
											end
										end,
								},
								Padding_MountRankBounds = {
									order = count(),
									type = "description",
									name = "",
								},
							},
						},
						CtrlGroupHotness = {
							order = count(),	
							name = L["Hotness"],
							handler = Pokedex,
							type = "group",
							dialogInline = true,
							args = {
								fEnableHotness = {
									order = count(),
									type = "toggle",
									name = L["enable hot mount"],
									desc = L["lets you turn on|off the hot mount subfeatures"],
									get = function(info) return Pokedex.db.char.mounts.fEnableHotness end,
									set = function(info, value) 
											Pokedex.db.char.mounts.fEnableHotness = value 
											-- chance or hot gv.Mounts:UpdateDisplayNames()
										end,
								},
								fNewHotness = {
									order = count(),
									type = "toggle",
									name = L["new hotness"],
									desc = L["always make newest mount the hot one"],
									disabled = function(info) return not Pokedex.db.char.mounts.fEnableHotness end,
									get = function(info) return Pokedex.db.char.mounts.fNewHotness end,
									set = function(info, value) Pokedex.db.char.mounts.fNewHotness = value end,
								},
								Padding_Hotness = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlHotMount = {
									order = count(),
									type = "select",
									width = "double",
									name = L["select hot mount"],
									desc = L["mount to become hot one"],
									values = function(info) return gv.Mounts.HotList end,
									disabled = function(info) return (not Pokedex.db.char.mounts.fEnableHotness or gv.Pets.count == 0) end,
									get = function(info) return gv.HotMountIndex end,
									set = function(info, value) 
										local name  = gv.Mounts.HotList[value]
										local mount = gv.Mounts.ByName[name] or gc.nomount									

										gv.HotMountIndex = value
										gv.HotMount = mount
										Pokedex.db.char.mounts.idHot = mount.id
										-- chance or hot gv.Mounts:UpdateDisplayNames()
									end,
								},
								CtrlMountHeat = {
									order = count(),
									type = "range",
									name = L["mount heat"],
									desc = L["set hotness as a percentage - 100% means the hot mount is the only one that will be chosen"],
									min = 0,
									max = 100,
									step = 5,
									disabled = function(info) return not Pokedex.db.char.mounts.fEnableHotness end,
									get = function(info) return Pokedex.db.char.mounts.iHeat end,
									set = function(info, value) 
										Pokedex.db.char.mounts.iHeat = value 
										-- chance or hot gv.Mounts:UpdateDisplayNames()
									end,
								},
								Padding_HotMounts = {
									order = count(),
									type = "description",
									name = "",
								},
							},
						},
						MountFeatures = {
							order = count(),
							type = "header",
							name = L["Mount Features"],
						},
						UseAsMount = {
							order = count(),
							name = L["Use As Mount"],
							type = "group",
							dialogInline = true,
							-- hidden = GroupContentHidden,
							get = function(info) return Pokedex.db.profile.UseAsMount[info[#info]] end,
							set = function(info, value) 
									Pokedex.db.profile.UseAsMount[info[#info]] = value 
									Pokedex:UpdateMountInfo()
								end,
							args = {
								CtrlUseAsMountDesc = {
									order = count(),
									type = "description",
									hidden = false,
									width = "full",
									name = L["This feature adds mount like forms and spells to the set of mounts Pokedex summons from. They are usually only selected when a normal mount cannot be summoned, such as when you are indoors or moving."],
								},
								NagrandCorral = {
									order = count(),
									type = "toggle",
									name = GetSpellInfo((UnitFactionGroup("player") == "Horde") and gc.idSpellCorralHorde or gc.idSpellCorralAlliance),
									image = select(3, GetSpellInfo((UnitFactionGroup("player") == "Horde") and gc.idSpellCorralHorde or gc.idSpellCorralAlliance)),
									desc = L["allow ToggleMount to make use of the Nagrand Corral garrison outpost mount"],
								},
								ItemFishingRaft = {
									order = count(),
									type = "toggle",
									name = select(2, ctoy.GetToyInfo(gc.idItemFishingRaft)) or "Anglers Fishing Raft",
									image = select(3, ctoy.GetToyInfo(gc.idItemFishingRaft)) or 774121,
									desc = L["allow ToggleMount to make use of the Angler's Fishing Raft"],
								},
								ItemRatstallionHarness = {
									order = count(),
									type = "toggle",
									name = GetItemInfo(gc.idItemRatstallionHarness) or "Ratstallion Harness",
									image = select(10, GetItemInfo(gc.idItemRatstallionHarness)) or 119938,
									desc = L["allow ToggleMount to make use of the Ratstallion Harness"],
								},
								DruidForms = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fDruid,
									image = select(3, GetSpellInfo(gc.idSpellTravelForm)),
									name = L["Druid forms"],
									desc = L["allow ToggleMount to select Druid forms when better than a mount or if a mount can't be summoned"],
								},
								MonkRoll = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fMonk,
									name = GetSpellInfo(gc.idSpellRoll),
									image = select(3, GetSpellInfo(gc.idSpellRoll)),
									desc = L["allow ToggleMount to make use of Roll in situations when a mount cannot be summoned"],
								},
								ShamanGhostWolf = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fShaman,
									name = GetSpellInfo(gc.idSpellGhostWolf),
									image = select(3, GetSpellInfo(gc.idSpellGhostWolf)),
									desc = L["allow ToggleMount to make use of Ghost Wolf form in situations when a mount cannot be summoned"],
								},
								WarlockBurningRush = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fWarlock,
									name = GetSpellInfo(gc.idSpellBurningRush),
									image = select(3, GetSpellInfo(gc.idSpellBurningRush)),
									desc = L["allow ToggleMount to make use of Burning Rush in situations when a mount cannot be summoned"],
								},
							},
						},	

						CastWithMount = {
							order = count(),
							name = L["Cast With Mount"],
							type = "group",
							dialogInline = true,
							-- hidden = GroupContentHidden,
							get = function(info) return Pokedex.db.profile.CastWithMount[info[#info]] end,
							set = function(info, value) 
									Pokedex.db.profile.CastWithMount[info[#info]] = value 
									Pokedex:UpdateMountInfo()
								end,
							args = {
								CtrlCastWithMountDesc = {
									order = count(),
									type = "description",
									hidden = false,
									width = "full",
									name = L["This feature tries to refresh or apply a useful short term buff while mounting. Mount summoning is not affected by the global cooldown which lets us cast an instant spell along with the mount."],
								},

								DKPathOfFrost = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fDeathKnight,
									name = GetSpellInfo(gc.idSpellPathOfFrost),
									image = select(3, GetSpellInfo(gc.idSpellPathOfFrost)),
									desc = L["cast Path of Frost while mounting"],
								},
								DruidCenarionWard = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fDruid,
									name = GetSpellInfo(gc.idSpellCenarionWard),
									image = select(3, GetSpellInfo(gc.idSpellCenarionWard)),
									desc = L["cast Cenarion Ward while mounting"],
								},
								MageBlazingBarrier = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fMage,
									name = GetSpellInfo(gc.idSpellBlazingBarrier),
									image = select(3, GetSpellInfo(gc.idSpellBlazingBarrier)),
									desc = L["cast Blazing Barrier while mounting"],
								},
								MageIceBarrier = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fMage,
									name = GetSpellInfo(gc.idSpellIceBarrier),
									image = select(3, GetSpellInfo(gc.idSpellIceBarrier)),
									desc = L["cast Ice Barrier while mounting"],
								},
								MagePrismaticBarrier = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fMage,
									name = GetSpellInfo(gc.idSpellPrismaticBarrier),
									image = select(3, GetSpellInfo(gc.idSpellPrismaticBarrier)),
									desc = L["cast Prismatic Barrier while mounting"],
								},
								RogueRemoveDisguise = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fRogue,
									name = L["remove Disguise"],
									image = select(3, GetSpellInfo(gc.idSpellDisguise)),
									desc = L["remove Disguise when mounting"],
								},
								ShamanWaterWalking = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fShaman,
									name = GetSpellInfo(gc.idSpellWaterWalking),
									image = select(3, GetSpellInfo(gc.idSpellWaterWalking)),
									desc = L["cast Water Walking while mounting"],
								},

								CtrlClassNotSupportedDesc = {
									order = count(),
									type = "description",
									hidden = not (gc.Player.fHunter or gc.Player.fMonk or gc.Player.fPaladin or gc.Player.fPriest or gc.Player.fWarlock or gc.Player.fWarrior),
									width = "full",
									name = "\n" .. L["This feature does not currently support any spells for this class."],
								},
							},
						},	

						CastWhenFalling = {
							order = count(),
							name = L["Cast When Falling"],
							type = "group",
							dialogInline = true,
							-- hidden = GroupContentHidden,
							get = function(info) return Pokedex.db.profile.CastWhenFalling[info[#info]] end,
							set = function(info, value) 
									Pokedex.db.profile.CastWhenFalling[info[#info]] = value 
									Pokedex:UpdateMountInfo()
								end,
							args = {
								CtrlCastWhenFallingDesc = {
									order = count(),
									type = "description",
									hidden = false,
									width = "full",
									name = L["This feature tries to use class abilities and items to prevent or reduce damage when falling. Because there is no macro condition to test for falling, this feature only works when you are out of combat."],
								},
								DemonGlide = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fDemonHunter,
									name = GetSpellInfo(gc.idSpellGlide),
									image = select(3, GetSpellInfo(gc.idSpellGlide)),
									desc = L["allow ToggleMount to cast Glide when falling"],
								},
								DruidFlap = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fDruid,
									name = GetSpellInfo(gc.idSpellFlap),
									image = select(3, GetSpellInfo(gc.idSpellFlap)),
									desc = L["allow ToggleMount to cast Flap when falling"],
								},
								MonkZenFlight = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fMonk,
									name = GetSpellInfo(gc.idSpellZenFlight),
									image = select(3, GetSpellInfo(gc.idSpellZenFlight)),
									desc = L["allow ToggleMount to cast Zen Flight when falling"],
								},
								PriestLevitate = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fPriest,
									name = GetSpellInfo(gc.idSpellLevitate),
									image = select(3, GetSpellInfo(gc.idSpellLevitate)),
									desc = L["allow ToggleMount to cast Levitate when falling"],
								},
								MageSlowFall = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fMage,
									name = GetSpellInfo(gc.idSpellSlowFall),
									image = select(3, GetSpellInfo(gc.idSpellSlowFall)),
									desc = L["allow ToggleMount to cast Slow Fall when falling"],
								},
								ItemSkyhornKite = {
									order = count(),
									type = "toggle",
									name = select(2, ctoy.GetToyInfo(gc.idItemSkyhornKite)) or "Rocfeather Skyhorn Kite",
									image = select(3, ctoy.GetToyInfo(gc.idItemSkyhornKite)) or 604252,
									desc = L["allow ToggleMount to use Rocfeather Skyhorn Kite when falling"],
								},
								EngrGoblinGlider = {
									order = count(),
									type = "toggle",
									hidden = not gv.Skills.fEngineer,
									name = GetSpellInfo(gc.idSpellGoblinGlider),
									image = select(3, GetSpellInfo(gc.idSpellGoblinGlider)),
									desc = L["allow ToggleMount to use Goblin Glider when falling"],
								},
								EngrFlexweaveUnderlay = {
									order = count(),
									type = "toggle",
									hidden = not gv.Skills.fEngineer,
									name = GetSpellInfo(gc.idSpellFlexweaveUnderlay),
									image = select(3, GetSpellInfo(gc.idSpellFlexweaveUnderlay)),
									desc = L["allow ToggleMount to use Flexweave Underlay when falling"],
								},
								ItemGoldenGlider = {
									order = count(),
									type = "toggle",
									name = GetSpellInfo(gc.idSpellGoldenGlider),
									image = select(3, GetSpellInfo(gc.idSpellGoldenGlider)),
									desc = L["allow ToggleMount to use Golden Glider when falling (Timeless Isle only)"],
								},
								ItemGoblinGliderKit = {
									order = count(),
									type = "toggle",
									name = GetItemInfo(gc.idItemGoblinGliderKit) or "Goblin Glider Kit",
									image = select(10, GetItemInfo(gc.idItemGoblinGliderKit)) or 133632,
									desc = L["allow ToggleMount to use Goblin Glider Kit when falling"],
								},
								ItemSnowfallLager = {
									order = count(),
									type = "toggle",
									name = GetItemInfo(gc.idItemSnowfallLager) or "Snowfall Lager",
									image = select(10, GetItemInfo(gc.idItemSnowfallLager)) or 132792,
									desc = L["allow ToggleMount to use Snowfall Lager when falling (Storm Peaks only)"],
								},
								RogueLegendaryDaggers = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fRogue,
									name = select(2, GetAchievementInfo(gc.idAchieveLegendaryDaggers)),
									-- image = select(3, GetSpellInfo(gc.)),
									desc = L["allow ToggleMount to equip (first click) and then use (second click) a legendary dagger when falling"],
								},
								PaladinHandOfProtection = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fPaladin,
									name = GetSpellInfo(gc.idSpellHandOfProtection),
									image = select(3, GetSpellInfo(gc.idSpellHandOfProtection)),
									desc = L["allow ToggleMount to cast Hand of Protection when falling"],
								},
								PaladinDivineShield = {
									order = count(),
									type = "toggle",
									hidden = not gc.Player.fPaladin,
									name = GetSpellInfo(gc.idSpellDivineShield),
									image = select(3, GetSpellInfo(gc.idSpellDivineShield)),
									desc = L["allow ToggleMount to cast Divine Shield when falling"],
								},
							},
						},	
						SafeDismount = {
							order = count(),
							name = L["Safe Dismount"],
							type = "group",
							dialogInline = true,
							-- hidden = function(info) return (gv.rsRidingSkill < SL.Expert) end,
							get = function(info) return Pokedex.db.profile.SafeDismount[info[#info]] end,
							set = function(info, value) Pokedex.db.profile.SafeDismount[info[#info]] = value end,
							args = {
								CtrlSafeDismountDesc = {
									order = count(),
									type = "description",
									width = "full",
									name = L["This feature works with and manages the games Auto Dismount in Flight setting to improve your experience with flying mounts. Auto dismount will be disabled unless in one of the selected conditions."],
								},
								Enabled = {
									order = count(),
									type = "toggle",
									width = "full",
									name = L["Manage Auto Dismount"],
									desc = L["Enables Pokedex's management of the Auto Dismount in Flight option"],
									get = function(info) return Pokedex.db.profile.SafeDismount.Enabled end,
									set = "SetManageAutoDismount",
								},
								Padding_Dismount1 = {
									order = count(),
									type = "description",
									width = "full",
									name = "",
								},
								Padding_Dismount2 = {
									order = count(),
									type = "description",
									width = "full",
									name = "",
								},
								Padding_Dismount3 = {
									order = count(),
									type = "description",
									width = "full",
									name = "",
								},
								ForCombat = {
									order = count(),
									type = "toggle",
									width = "full",
									name = L["Dismount for Combat"],
									desc = L["Enables Auto Dismount when in combat"],
									disabled = function(info) return (not Pokedex.db.profile.SafeDismount.Enabled) end,
								},
								ForAttack = {
									order = count(),
									type = "toggle",
									width = "full",
									name = L["Dismount to Attack"],
									desc = L["Enables Auto Dismount when targeting something attackable"],
									disabled = function(info) return (not Pokedex.db.profile.SafeDismount.Enabled) end,
								},
								ForGathering = {
									order = count(),
									type = "toggle",
									width = "full",
									name = L["Dismount for Gathering"],
									desc = L["Enables Auto Dismount when gathering a resource with mining, herbalism or skinning"],
									hidden = function(info) return not gv.Skills.fGatherer end,
									disabled = function(info) return (not Pokedex.db.profile.SafeDismount.Enabled) end,
								},
							},
						},
					},
				},

				CtrlGroupCompanions = {
					order = count(),
					name = L["Companions"],
					handler = Pokedex,
					type = "group",
					args = {
						CtrlGroupRanks = {
							order = count(),
							name = L["Ranks"],
							handler = Pokedex,
							type = "group",
							dialogInline = true,
							args = {
								CtrlCompanionForRanking = {
									order = count(),
									type = "select",
									width = "double",
									name = L["select companion for ranking"],
									desc = L["companion whose rank you can set"],
									values = function(info) return gv.Pets.All end,
									disabled = function(info) return (gv.Pets.count == 0) end,
									get = function(info) return gv.SelectedPet end,
									set = function(info, value) gv.SelectedPet = value end,
								},
								CtrlRankingForCompanion = {
									order = count(),
									type = "range",
									name = L["companion rank"],
									desc = L["rank of current companion"],
									min = 0,
									max = Pokedex.db.profile.pets.iMaximumRank,
									step = 1,
									disabled = function(info) return (gv.Pets.count == 0) end,
									get = function(info) return gv.SelectedPet.profile_rank end,
									set = function(info, value) 
											gv.SelectedPet.profile_rank = value;
											gv.Pets:UpdateDisplayNames(gv.SelectedPet);
										end,
								},
								Padding_CompanionRankings = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlCompanionOverrideRank = {
									order = count(),
									type = "toggle",
									width = "double",
									name = L["set rank override"],
									desc = L["set a character specific value for rank"],
									disabled = function(info) return (gv.Pets.count == 0) end,
									get = function(info) return gv.SelectedPet.char_rank ~= nil end,
									set = function(info, value) 
											gv.SelectedPet.char_rank = value and gv.SelectedPet.rank or nil;
											if not value then gv.Pets:UpdateDisplayNames(gv.SelectedPet) end;
										end,
								},
								CtrlRankingForCompanionOverride = {
									order = count(),
									type = "range",
									name = L["override rank"],
									desc = L["character specific rank"],
									min = 0,
									max = Pokedex.db.profile.pets.iMaximumRank,
									step = 1,
									disabled = function(info) return (gv.Pets.count == 0 or gv.SelectedPet.char_rank == nil) end,
									get = function(info) return gv.SelectedPet.char_rank end,
									set = function(info, value) 
											gv.SelectedPet.char_rank = value;
											gv.Pets:UpdateDisplayNames(gv.SelectedPet);
										end,
								},
								Padding_CompanionRankingOverride = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlCompanionSort = {
									order = count(),
									type = "select",
									name = L["sorting method"],
									desc = L["sorting method"],
									values = gs.DescByIndex,
									get = function(info) return gs.IndexByName[Pokedex.db.profile.pets.sort] end,
									set = function(info, value)
											local name = gs.NameByIndex[value]
											Pokedex.db.profile.pets.sort = name
											gs.Pets = gs[name]
										 end,
								},
								CtrlCompanionFormat = {
									order = count(),
									type = "select",
									name = L["companion format"],
									desc = L["companion format"],
									values = gf.DescByIndex,
									get = function(info) return gf.IndexByName[Pokedex.db.profile.pets.format] end,
									set = function(info, value)
											local name = gf.NameByIndex[value]
											Pokedex.db.profile.pets.format = name
											gf.Pets = gf[name]
											gv.Pets:UpdateDisplayNames()
										 end,
								},
								Padding_CompanionDisplay = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlCompanionDefaultRank = {
									order = count(),
									type = "input",
									name = L["default rank"],
									desc = L["default rank"],
									get = function(info) return format("%u", Pokedex.db.profile.pets.iDefaultRank) end,
									set = function(info, value)
											Pokedex.db.profile.pets.iDefaultRank = tonumber(value) or Pokedex.db.profile.pets.iDefaultRank
											EnsureRankBounds("pets")
											gv.Pets:UpdateDisplayNames()
										 end,
									validate = function(info, value)
											local iValue = tonumber(value)
											if not iValue or (floor(iValue) ~= ceil(iValue)) or iValue < 0 then
												local msg = L["ERROR: value must be a whole number"]
												Pokedex:Print(msg)
												return msg
											elseif iValue > Pokedex.db.profile.pets.iMaximumRank then
												local msg = L["ERROR: default rank cannot be greater than the maximum rank"]
												Pokedex:Print(msg)
												return msg
											else
												return true
											end
										end,
								},
								CtrlCompanionMaximumRank = {
									order = count(),
									type = "input",
									name = L["maximum rank"],
									desc = L["maximum rank"],
									get = function(info) return format("%u", Pokedex.db.profile.pets.iMaximumRank) end,
									set = function(info, value)
											local value = tonumber(value) or Pokedex.db.profile.pets.iMaximumRank
											Pokedex.db.profile.pets.iMaximumRank = value
											info.options.args.CtrlGroupCompanions.args.CtrlRankingForCompanion.max = value
											info.options.args.CtrlGroupCompanions.args.CtrlRankingForCompanionOverride.max = value
											EnsureRankBounds("pets")
											gv.Pets:UpdateDisplayNames()
										 end,
									validate = function(info, value)
											local iValue = tonumber(value)
											if not iValue or (floor(iValue) ~= ceil(iValue)) or iValue < 1 then
												local msg = L["ERROR: value must be a positive integer"]
												Pokedex:Print(msg)
												return msg
											elseif iValue < Pokedex.db.profile.pets.iDefaultRank then
												local msg = L["ERROR: maximum rank cannot be less than the default rank"]
												Pokedex:Print(msg)
												return msg
											else
												return true
											end
										end,
								},
								Padding_CompanionRankBounds = {
									order = count(),
									type = "description",
									name = "",
								},
							},
						},
						CtrlGroupHotness = {
							order = count(),	
							name = L["Hotness"],
							handler = Pokedex,
							type = "group",
							dialogInline = true,
							args = {
								fEnableHotness = {
									order = count(),
									type = "toggle",
									name = L["enable hot pet"],
									desc = L["lets you turn on|off the hot pet subfeatures"],
									get = function(info) return Pokedex.db.char.pets.fEnableHotness end,
									set = function(info, value) Pokedex.db.char.pets.fEnableHotness = value end,
								},
								fNewHotness = {
									order = count(),
									type = "toggle",
									name = L["new hotness"],
									desc = L["always make newest companion the hot one"],
									disabled = function(info) return not Pokedex.db.char.pets.fEnableHotness end,
									get = function(info) return Pokedex.db.char.pets.fNewHotness end,
									set = function(info, value) Pokedex.db.char.pets.fNewHotness = value end,
								},
								Padding_Hotness = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlHotCompanion = {
									order = count(),
									type = "select",
									width = "double",
									name = L["select hot companion"],
									desc = L["companion to become hot one"],
									values = function(info) return gv.Pets.HotList end,
									disabled = function(info) return (not Pokedex.db.char.pets.fEnableHotness or gv.Pets.count == 0) end,
									get = function(info) return gv.HotPetIndex end,
									set = function(info, value)
										local name = gv.Pets.HotList[value]
										local pet  = gv.Pets.ByName[name] or gc.nopet

										gv.HotPetIndex = value
										gv.HotPet = pet
										Pokedex.db.char.pets.idHot = pet.id
										-- chance or hot gv.Pets:UpdateDisplayNames()
									end,
								},
								CtrlCompanionHeat = {
									order = count(),
									type = "range",
									name = L["companion heat"],
									desc = L["set hotness as a percentage - 100% means the hot pet is the only one that will be chosen"],
									min = 0,
									max = 100,
									step = 5,
									disabled = function(info) return not Pokedex.db.char.pets.fEnableHotness end,
									get = function(info) return Pokedex.db.char.pets.iHeat end,
									set = function(info, value) 
										Pokedex.db.char.pets.iHeat = value 
										-- chance or hot gv.Pets:UpdateDisplayNames()
									end,
								},
								Padding_HotCompanions = {
									order = count(),
									type = "description",
									name = "",
								},
							},
						},
						fAnnounce = {
							order = count(),
							type = "toggle",
							name = L["announce"],
							desc = L["(on|off) let everyone know who *you* choose"],
							get = function(info) return Pokedex.db.profile.fAnnounce end,
							set = function(info, value) Pokedex.db.profile.fAnnounce = value end,
						},
						iChannel = {
							order = count(),
							type = "select",
							name = L["channel"],
							desc = L["channel to announce selection in"],
							values = gc.rgstrChannelDescs,
							get = function(info) return Pokedex.db.profile.iChannel end,
							set = function(info, value) Pokedex.db.profile.iChannel = value end,
						},
						Padding_Announcements = {
							order = count(),
							type = "description",
							name = "",
						},
						CtrlAutoSummonCompanion = {
							order = count(),
							type = "toggle",
							width = "full",
							name = L["Auto Summon Companion"],
							desc = L["Ensures a companion is accompanying you whenever you dismount or enter a new location"],
							get = function(info) return Pokedex.db.profile.fAutoSummonCompanion end,
							set = function(info, value) Pokedex.db.profile.fAutoSummonCompanion = value end,
						},
						Padding_AutoSummonCompanion = {
							order = count(),
							type = "description",
							name = "",
						},
					},
				},

				CtrlGroupTitles = {
					order = count(),
					name = L["Titles"],
					handler = Pokedex,
					type = "group",
					args = {
						CtrlGroupRanks = {
							order = count(),	
							name = L["Ranks"],
							handler = Pokedex,
							type = "group",
							dialogInline = true,
							args = {
								CtrlTitleForRanking = {
									order = count(),
									type = "select",
									width = "double",
									name = L["select title for ranking"],
									desc = L["title whose rank you can set"],
									values = function(info) return gv.Titles.All end,
									disabled = function(info) return (gv.Titles == 0) end,
									get = function(info) return gv.SelectedTitle end,
									set = function(info, value) gv.SelectedTitle = value; end,
								},
								CtrlRankingForTitle = {
									order = count(),
									type = "range",
									name = L["title rank"],
									desc = L["rank of current title"],
									min = 0,
									max = Pokedex.db.profile.titles.iMaximumRank,
									step = 1,
									disabled = function(info) return (gv.Titles.count == 0) end,
									get = function(info) return gv.SelectedTitle.profile_rank end,
									set = function(info, value) 
											gv.SelectedTitle.profile_rank = value 
											gv.Titles:UpdateDisplayNames(gv.SelectedTitle);
										end,
								},
								Padding_TitleRankings = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlTitleOverrideRank = {
									order = count(),
									type = "toggle",
									width = "double",
									name = L["set rank override"],
									desc = L["set a character specific value for rank"],
									disabled = function(info) return (gv.Titles.count == 0) end,
									get = function(info) return gv.SelectedTitle.char_rank ~= nil end,
									set = function(info, value) 
											gv.SelectedTitle.char_rank = value and gv.SelectedTitle.rank or nil;
											if not value then gv.Titles:UpdateDisplayNames(gv.SelectedTitle) end;
										end,
								},
								CtrlRankingForTitleOverride = {
									order = count(),
									type = "range",
									name = L["override rank"],
									desc = L["character specific rank"],
									min = 0,
									max = Pokedex.db.profile.titles.iMaximumRank,
									step = 1,
									disabled = function(info) return (gv.Titles.count == 0 or gv.SelectedTitle.char_rank == nil) end,
									get = function(info) return gv.SelectedTitle.char_rank end,
									set = function(info, value) 
											gv.SelectedTitle.char_rank = value;
											gv.Titles:UpdateDisplayNames(gv.SelectedTitle);
										end,
								},
								Padding_TitleRankingOverride = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlTitleSort = {
									order = count(),
									type = "select",
									name = L["sorting method"],
									desc = L["sorting method"],
									values = gs.DescByIndex,
									get = function(info) return gs.IndexByName[Pokedex.db.profile.titles.sort] end,
									set = function(info, value)
											local name = gs.NameByIndex[value]
											Pokedex.db.profile.titles.sort = name
											gs.Titles = gs[name]
										 end,
								},
								CtrlTitleFormat = {
									order = count(),
									type = "select",
									name = L["title format"],
									desc = L["title format"],
									values = gf.DescByIndex,
									get = function(info) return gf.IndexByName[Pokedex.db.profile.titles.format] end,
									set = function(info, value)
											local name = gf.NameByIndex[value]
											Pokedex.db.profile.titles.format = name
											gf.Titles = gf[name]
											gv.Titles:UpdateDisplayNames()
										 end,
								},
								Padding_TitleDisplay = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlTitleDefaultRank = {
									order = count(),
									type = "input",
									name = L["default rank"],
									desc = L["default rank"],
									get = function(info) return format("%u", Pokedex.db.profile.titles.iDefaultRank) end,
									set = function(info, value)
											Pokedex.db.profile.titles.iDefaultRank = tonumber(value) or Pokedex.db.profile.titles.iDefaultRank
											EnsureRankBounds("titles")
											gv.Titles:UpdateDisplayNames()
										 end,
									validate = function(info, value)
											local iValue = tonumber(value)
											if not iValue or (floor(iValue) ~= ceil(iValue)) or iValue < 1 then
												local msg = L["ERROR: value must be a whole number"]
												Pokedex:Print(msg)
												return msg
											elseif iValue > Pokedex.db.profile.titles.iMaximumRank then
												local msg = L["ERROR: default rank cannot be greater than the maximum rank"]
												Pokedex:Print(msg)
												return msg
											else
												return true
											end
										end,
								},
								CtrlTitleMaximumRank = {
									order = count(),
									type = "input",
									name = L["maximum rank"],
									desc = L["maximum rank"],
									get = function(info) return format("%u", Pokedex.db.profile.titles.iMaximumRank) end,
									set = function(info, value)
											local value = tonumber(value) or Pokedex.db.profile.titles.iMaximumRank
											Pokedex.db.profile.titles.iMaximumRank = value
											info.options.args.CtrlGroupTitles.args.CtrlRankingForTitle.max = value
											info.options.args.CtrlGroupTitles.args.CtrlRankingForTitleOverride.max = value
											EnsureRankBounds("titles")
											gv.Titles:UpdateDisplayNames()
										 end,
									validate = function(info, value)
											local iValue = tonumber(value)
											if not iValue or (floor(iValue) ~= ceil(iValue)) or iValue < 1 then
												local msg = L["ERROR: value must be a positive integer"]
												Pokedex:Print(msg)
												return msg
											elseif iValue < Pokedex.db.profile.titles.iDefaultRank then
												local msg = L["ERROR: maximum rank cannot be less than the default rank"]
												Pokedex:Print(msg)
												return msg
											else
												return true
											end
										end,
								},
								Padding_TitleRankBounds = {
									order = count(),
									type = "description",
									name = "",
								},
							},
						},
						CtrlGroupHotness = {
							order = count(),	
							name = L["Hotness"],
							handler = Pokedex,
							type = "group",
							dialogInline = true,
							args = {
								fEnableHotness = {
									order = count(),
									type = "toggle",
									name = L["enable hot title"],
									desc = L["lets you turn on|off the hot title subfeatures"],
									get = function(info) return Pokedex.db.char.titles.fEnableHotness end,
									set = function(info, value) Pokedex.db.char.titles.fEnableHotness = value end,
								},
								fNewHotness = {
									order = count(),
									type = "toggle",
									name = L["new hotness"],
									desc = L["always make newest title the hot one"],
									disabled = function(info) return not Pokedex.db.char.titles.fEnableHotness end,
									get = function(info) return Pokedex.db.char.titles.fNewHotness end,
									set = function(info, value) Pokedex.db.char.titles.fNewHotness = value end,
								},
								Padding_Hotness = {
									order = count(),
									type = "description",
									name = "",
								},
								CtrlHotTitle = {
									order = count(),
									type = "select",
									width = "double",
									name = L["select hot title"],
									desc = L["title to become hot one"],
									values = function(info) return gv.Titles.HotList end,
									disabled = function(info) return (not Pokedex.db.char.titles.fEnableHotness or gv.Titles.count == 0) end,
									get = function(info) return gv.HotTitleIndex end,
									set = function(info, value) 
										local name  = gv.Titles.HotList[value]
										local title = gv.Titles.ByName[name] or gc.notitle

										gv.HotTitleIndex = value
										gv.HotTitle = title
										Pokedex.db.char.titles.idHot = title.id
										-- chance or hot gv.Titles:UpdateDisplayNames()
									end,
								},
								CtrlTitleHeat = {
									order = count(),
									type = "range",
									name = L["title heat"],
									desc = L["set hotness as a percentage - 100% means the hot pet is the only one that will be chosen"],
									min = 0,
									max = 100,
									step = 5,
									disabled = function(info) return not Pokedex.db.char.titles.fEnableHotness end,
									get = function(info) return Pokedex.db.char.titles.iHeat end,
									set = function(info, value) 
										Pokedex.db.char.titles.iHeat = value 
										-- chance or hot gv.Titles:UpdateDisplayNames()
									end,
								},
								Padding_HotTitles = {
									order = count(),
									type = "description",
									name = "",
								},
							},
						},
						CtrlChangeTitleOnMount = {
							order = count(),
							type = "toggle",
							name = L["Change title on mount"],
							desc = L["Change title when a mount is summoned"],
							get = function(info) return Pokedex.db.profile.fChangeTitleOnMount end,
							set = function(info, value) Pokedex.db.profile.fChangeTitleOnMount = value end,
						},
						Padding_TitleButtons = {
							order = count(),
							type = "description",
							name = "",
						},
					},
				},
			},
		}  -- end options.commands --- commands executable from command line
	end

	return (uiType == "cmd") and Pokedex.options.commands or Pokedex.options.gui;
end
