--=========================================================================--
-- upvalues
--=========================================================================--

local _G, Pokedex, LibStub, cmj, cpj = _G, Pokedex, LibStub, C_MountJournal, C_PetJournal
local format, gmatch, gsub, strfind, strsub, strtrim, strupper = format, gmatch, gsub, strfind, strsub, strtrim, strupper
local assert, loadstring, tonumber, tostring, type = assert, loadstring, tonumber, tostring, type
local getmetatable, ipairs, next, pairs, rawget, rawset, select, setmetatable, sort, tinsert, tremove, unpack = getmetatable, ipairs, next, pairs, rawget, rawset, select, setmetatable, sort, tinsert, tremove, unpack
local band, bnot, bor, max, min, mrandom = bit.band, bit.bnot, bit.bor, max, min, math.random
local fband = function(...) return 0 ~= band(...) end

local CreateFont, CreateFrame, GameTooltip, GameTooltipText, GetCVarBool, SetCVar, UIParent = CreateFont, CreateFrame, GameTooltip, GameTooltipText, GetCVarBool, SetCVar, UIParent
local GetRunningMacro, GetTime, InterfaceOptionsFrame_OpenToCategory, SecureCmdOptionParse = GetRunningMacro, GetTime, InterfaceOptionsFrame_OpenToCategory, SecureCmdOptionParse
local Dismount, GetCompanionCooldown, VehicleExit = Dismount, GetCompanionCooldown, VehicleExit
local GetCurrentTitle, GetNumTitles, GetTitleName, IsTitleKnown = GetCurrentTitle, GetNumTitles, GetTitleName, IsTitleKnown
local GetActiveSpecGroup, GetProfessionInfo, GetProfessions = GetActiveSpecGroup, GetProfessionInfo, GetProfessions
local GetCurrentMapAreaID, GetCurrentMapContinent, GetMapContinents, GetRealZoneText, GetSubZoneText, GetWintergraspWaitTime, GetZoneText, SetMapToCurrentZone = GetCurrentMapAreaID, GetCurrentMapContinent, GetMapContinents, GetRealZoneText, GetSubZoneText, GetWintergraspWaitTime, GetZoneText, SetMapToCurrentZone
local GetNumGroupMembers, IsInRaid, SendChatMessage = GetNumGroupMembers, IsInRaid, SendChatMessage
local EquipItemByName, GetInventoryItemCooldown, GetInventoryItemID, GetInventoryItemLink, GetItemCooldown, GetItemCount, GetItemInfo, GetItemSpell, GetMirrorTimerProgress, GetSpellCooldown, GetSpellInfo, IsEquippedItem, IsSpellKnown, IsUsableSpell, OffhandHasWeapon, PlayerHasToy = EquipItemByName, GetInventoryItemCooldown, GetInventoryItemID, GetInventoryItemLink, GetItemCooldown, GetItemCount, GetItemInfo, GetItemSpell, GetMirrorTimerProgress, GetSpellCooldown, GetSpellInfo, IsEquippedItem, IsSpellKnown, IsUsableSpell, OffhandHasWeapon, PlayerHasToy
local InCombatLockdown, IsFalling, IsFlyableArea, IsFlying, IsIndoors, IsInInstance, IsMounted, IsOutdoors, IsSwimming = InCombatLockdown, IsFalling, IsFlyableArea, IsFlying, IsIndoors, IsInInstance, IsMounted, IsOutdoors, IsSwimming
local GetShapeshiftForm, GetShapeshiftFormID, GetUnitSpeed, UnitAffectingCombat, UnitBuff, UnitCanAttack, UnitCastingInfo, UnitClass, UnitDebuff, UnitFactionGroup, UnitInVehicle, UnitIsDead, UnitIsDeadOrGhost, UnitLevel, UnitOnTaxi, UnitRace = GetShapeshiftForm, GetShapeshiftFormID, GetUnitSpeed, UnitAffectingCombat, UnitBuff, UnitCanAttack, UnitCastingInfo, UnitClass, UnitDebuff, UnitFactionGroup, UnitInVehicle, UnitIsDead, UnitIsDeadOrGhost, UnitLevel, UnitOnTaxi, UnitRace
local INVSLOT_BACK, SPELL_POWER_SOUL_SHARDS = INVSLOT_BACK, SPELL_POWER_SOUL_SHARDS

local L = LibStub("AceLocale-3.0"):GetLocale("Pokedex")

--=========================================================================--
-- global variables, constants and types
--=========================================================================--

local IS = Pokedex.Globals.Types.InitStates;     -- Initialization States
local DL = Pokedex.Globals.Types.DebugLevels;    -- Debug Levels
local SL = Pokedex.Globals.Types.SkillLevels;    -- max rank of each skill level
local SI = Pokedex.Globals.Types.SkillIds;       -- skill identifiers
local MF = Pokedex.Globals.Types.MountFlags;     -- Mount Flags
local DC = {};  -- Debug Categories 
local gc = Pokedex.Globals.Constants;
local gv = Pokedex.Globals.Variables;
local gs = Pokedex.Globals.SortMethods;
local gf = Pokedex.Globals.FormatMethods;
local EventMonitor_PJListUpdate = Pokedex.Globals.Types.CEventMonitor:Create("PET_JOURNAL_LIST_UPDATE")

local MFS_ATV = bor(MF.Flyer, MF.NoFlyZone_Legal, MF.Underwater_Legal)
local maskClearNFZL = bnot(MF.NoFlyZone_Legal)

--=========================================================================--
-- Addon Management functions
--=========================================================================--

-- Called when the addon is loaded
function Pokedex:OnInitialize()
	Pokedex:LoadSavedSettings()

	LibStub("AceConfig-3.0"):RegisterOptionsTable("Pokedex", Pokedex.GetUIOptions)
	self.optionsFrame = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("Pokedex", "Pokedex")

	LibStub("AceConfig-3.0"):RegisterOptionsTable("Pokedex-Profiles", LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db, true))
	self.profilesFrame = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("Pokedex-Profiles", "Profiles", "Pokedex")

	self:RegisterChatCommand("pd", "ChatCommand")
	self:RegisterChatCommand("pokedex", "ChatCommand")
end


function Pokedex:ChatCommand(input)
	if (gv.InitState ~= IS.INITIALIZED and self:Initialize("ChatCommand") == IS.UNINITIALIZED) then
		self:Print(L["ERROR: Pokedex failed to initialize correctly. This is usually caused when WoW has invalidated its cache and hasn't finished rebuilding it. Please try this action again later."]);
		return
	end

	if (GetRunningMacro()) then
		if (input) then
			local orig = input;
			input = SecureCmdOptionParse(input);
			if (DC.MISC >= DL.AV) then self:Print("Pokedex called from macro, arguments were parsed"); end
			if (DC.MISC >= DL.AV) then self:Print("from:", orig); end
			if (DC.MISC >= DL.AV) then self:Print("to:", input); end
		end
		
		-- popping up options UI from within a macro is most likely not what the user really wanted, so just exit
		if (not input or input:trim() == "") then
			return
		end
	end

	if (not input or input:trim() == "") then
		InterfaceOptionsFrame_OpenToCategory(self.optionsFrame);
	else
		LibStub("AceConfigCmd-3.0").HandleCommand(Pokedex, "pd", "Pokedex", input);
	end
end


-- Called when the addon is enabled
function Pokedex:OnEnable()
	DC = self.db.global.rgDebugInfo;

	-- check that none of our actions are blocked
	self:RegisterEvent("ADDON_ACTION_BLOCKED");

	-- check that none of our actions are forbidden
	self:RegisterEvent("ADDON_ACTION_FORBIDDEN");

	-- init data tables and register events and hooks
	self:Initialize("OnEnable");
end


-- Called when the addon is disabled
function Pokedex:OnDisable()
	self:UnregisterEvent("ADDON_ACTION_BLOCKED");
	self:UnregisterEvent("ADDON_ACTION_FORBIDDEN");

	if (gv.InitState == IS.INITIALIZED) then
		self:UnregisterEvent("LEARNED_SPELL_IN_TAB");
		self:UnregisterEvent("PLAYER_TALENT_UPDATE");
		-- self:UnregisterEvent("PLAYER_SPECIALIZATION_CHANGED");
		
		self:UnregisterEvent("COMPANION_LEARNED");
		EventMonitor_PJListUpdate:Suspend()

		self:UnregisterEvent("KNOWN_TITLES_UPDATE");
		self:UnregisterEvent("NEW_TITLE_EARNED");
		self:UnregisterEvent("OLD_TITLE_LOST");

		-- skills
		self:UnregisterEvent("SKILL_LINES_CHANGED");

		if gv.Skills.Engineering > 0 then
			self:UnregisterEvent("PLAYER_EQUIPMENT_CHANGED");
		end

		-- dismount
		self:UnregisterEvent("CVAR_UPDATE");
		self:UnregisterEvent("PLAYER_REGEN_DISABLED");
		self:UnregisterEvent("PLAYER_REGEN_ENABLED");
		self:UnregisterEvent("PLAYER_TARGET_CHANGED");

		-- companion autosummon
		self:UnregisterEvent("ZONE_CHANGED_NEW_AREA");

		Pokedex:UnhookAll();
	end
end


-- Initialize all the pet data in the addon
function Pokedex:Initialize(strCaller)
	strCaller = (strCaller or "unknown caller");
	
	if (gv.InitState == IS.INITIALIZED) then 
		if (DC.MISC >= DL.EXCEP) then self:Print("attempt to reinitialize by " .. strCaller); end
		return IS.INITIALIZED;
	end

	if (gv.InitState == IS.INITIALIZING) then 
		if (DC.MISC >= DL.EXCEP) then self:Print("attempt to initialize during initialization by " .. strCaller); end
		return IS.INITIALIZING;
	end

	if (DC.MISC >= DL.EXCEP) then self:Print("initialization triggered by " .. strCaller); end
	
	local factionGroup = UnitFactionGroup("player")
	if factionGroup == "Alliance" then
		gc.iOffFaction = 0
	elseif factionGroup == "Horde" then
		gc.iOffFaction = 1
	end

	-- get skills
	self:InitSkills();

	-- At this point we used to return IS.UNINITIALIZED if we failed any one
	-- of the calls to UpdateMountInfo, UpdateTitleInfo or FUpdateCompanionInfo.
	-- A failure at this point was often due to bad cache states; not being able
	-- to get the names of the mounts for example. We now delay init the mount,
	-- title and pet tables until first access in a attempt to give the game 
	-- time. Any checks for valid data that would further delay init of those 
	-- classes can and should be handled there.


	self:UpdateDismountSettings();

	gv.InitState = IS.INITIALIZING;

	-- create secure action button to support forms
	self:CreateMountButton();

	-- update class based forms and mount features
	self:RegisterEvent("LEARNED_SPELL_IN_TAB");
	self:RegisterEvent("PLAYER_TALENT_UPDATE");
	-- self:RegisterEvent("PLAYER_SPECIALIZATION_CHANGED");

	-- update data when new mount or companion added
	self:RegisterEvent("COMPANION_LEARNED");
	EventMonitor_PJListUpdate:Resume();

	-- keep track of current mount or companion
	self:SecureHook(_G.C_MountJournal, "SummonByID", "CMJSummonHook");
	self:SecureHook(_G.C_MountJournal, "Dismiss", "CMJDismissHook");

	-- update data when new title added	
	self:RegisterEvent("KNOWN_TITLES_UPDATE");
	self:RegisterEvent("NEW_TITLE_EARNED");
	self:RegisterEvent("OLD_TITLE_LOST");

	-- keep track of current title
	self:SecureHook("SetCurrentTitle", "SetCurrentTitleHook");

	-- update dismount settings if the CVAR gets changed
	self:RegisterEvent("CVAR_UPDATE");
	
	-- allow flying dismount if in combat or if you've targeted something attackable
	self:RegisterEvent("PLAYER_REGEN_DISABLED");
	self:RegisterEvent("PLAYER_REGEN_ENABLED");
	self:RegisterEvent("PLAYER_TARGET_CHANGED");

	-- companion autosummon
	self:RegisterEvent("ZONE_CHANGED_NEW_AREA");

	-- update dismount for gathering if profession gained or lost
	-- LEARNED_SPELL_IN_TAB lets us know when spell is added to general tab, but not when we drop the skill
	-- SPELLS_CHANGED seems like it would get called *all* the time, we'd be a small perf drag on the game
	-- SKILL_LINES_CHANGED even though we look at spells now, this may still be the best way to know that we should
	self:RegisterEvent("SKILL_LINES_CHANGED");

	-- if character is an engineer then we need to monitor their cloak slot for parachutes or glider
	if gv.Skills.Engineering > 0 then
		self:RegisterEvent("PLAYER_EQUIPMENT_CHANGED");
	end

	-- hook tooltip to see if we're about to try to gather
	self:SecureHookScript(GameTooltip, "OnShow", "MainTooltipShow");
	self:SecureHookScript(GameTooltip, "OnHide", "MainTooltipHide");

	-- if (DC.MISC >= DL.BASIC) then self:EchoCounts(); end
	gv.InitState = IS.INITIALIZED;
end

function Pokedex:SafeCall(fn, ...)
	if (gv.InitState ~= IS.INITIALIZED and self:Initialize("SafeCall") == IS.UNINITIALIZED) then
		self:Print(L["ERROR: Pokedex failed to initialize correctly. This is usually caused when WoW has invalidated its cache and hasn't finished rebuilding it. Please try this action again later."]);
	else
		return fn(self, ...);
	end
end

--=========================================================================--
--=========================================================================--
--
-- DEBUG AND INFRASTRUCTURE FUNCTIONS
--
--=========================================================================--
--=========================================================================--

local function TestSpeed(fVerbose)
	local nSpeed = GetUnitSpeed("player");
	local nPercent = (nSpeed == 0) and 0 or ((nSpeed * 100) / 7);
	local strOut = "";

	if (nSpeed == 0) then
		strOut = format((fVerbose and "%i yards per second" or "%.1f YPS"), nSpeed);
	elseif (nSpeed == 7) then
		strOut = format((fVerbose and "%i yards per second, standard run speed" or "%.1f YPS"), nSpeed);
	elseif (nPercent > 100) then
		strOut = format((fVerbose and "%.2f yards per second, speed boosted by %.2f%%" or "%.1f YPS  %.2f%%"), nSpeed, nPercent);
	else
		strOut = format((fVerbose and "%.2f yards per second, speed reduced to %.2f%% of run speed" or "%.1f YPS  %.2f%%"), nSpeed, nPercent);
	end
	
	strOut = gsub(strOut, "%.00", "");
	return strOut;
end

--[===[@debug@
local LibQTip = nil
local ldbTestDataObj = nil
local ldbTestFrame = nil
local HeaderFont = nil
local LineFont = nil

local function UpdateSpeed(frame,elapsed)
	ldbTestDataObj.text = TestSpeed(false);
end

local function MFtoStr(flags)
	local strFlags = ""  --tostring(flags)
	for name, flag in pairs(MF) do
		if type(flag) == "number" and fband(flag, flags) then
			strFlags = format("%s  %s", strFlags, name)
		end
	end
	return strtrim(strFlags)
end

local function TestTip_OnEnter(self)
	local tooltip = LibQTip:Acquire("PokedexTestTooltip", 6)
	self.tooltip = tooltip 
	
	tooltip:SetHeaderFont(HeaderFont);
	tooltip:SetFont(LineFont);

	tooltip:AddLine(" ");

	tooltip:AddHeader("MapID", "Continent", "Zone", "Real Zone", "SubZone");

	SetMapToCurrentZone();
	local iContinent = GetCurrentMapContinent();
	local strContinent
	if iContinent > 0 then
		local rgstrContinents = { GetMapContinents() };
		strContinent = format("%i  %s", iContinent, (rgstrContinents[iContinent*2] or "unknown"));
	else
		strContinent = format("%i  %s", iContinent, "unknown");
	end
		
	tooltip:AddLine(GetCurrentMapAreaID(), strContinent, GetZoneText(), GetRealZoneText(), GetSubZoneText());
	
	tooltip:AddLine(" ");
	tooltip:AddSeparator();
	tooltip:AddLine(" ");

	tooltip:AddHeader("FlyableArea", "Swimming", "Outdoors", "Indoors", "Falling");
	tooltip:AddLine( tostring(IsFlyableArea()), tostring(IsSwimming()), tostring(IsOutdoors()), tostring(IsIndoors()), tostring(IsFalling()) );

	tooltip:AddLine(" ");
	tooltip:AddSeparator();
	tooltip:AddLine(" ");

	tooltip:AddHeader("Red Drake", "Red Cloud", "Riding Turtle", "Subdued Seahorse", "Abyssal Seahorse", "Red Scarab");
	tooltip:AddLine( tostring(IsUsableSpell(gc.idSpellRedDrake)), tostring(IsUsableSpell(gc.idSpellRedFlyingCloud)), tostring(IsUsableSpell(gc.idSpellRidingTurtle)), tostring(IsUsableSpell(gc.idSpellSubduedSeahorse)), tostring(IsUsableSpell(gc.idSpellAbyssalSeahorse)), tostring(IsUsableSpell(gc.idSpellRedScarab)) );

	tooltip:AddLine(" ");
	tooltip:AddSeparator();
	tooltip:AddLine(" ");

	local listname, flags = Pokedex:GetAreaInfo()
	tooltip:AddHeader("NoFlyZone", "Underwater", "Moving", "Indoors", "Combat");
	tooltip:AddLine(fband(flags, MF.NoFlyZone_Legal), fband(flags, MF.Underwater_Legal), fband(flags, MF.Moving_Legal), fband(flags, MF.Indoors_Legal), fband(flags, MF.Combat_Legal));

	tooltip:AddLine(" ");
	tooltip:AddSeparator();
	tooltip:AddLine(" ");

	tooltip:AddHeader(listname, "Summonable", "Rejected");
	local fulllist = gv.Mounts.Lists[listname]
	local filtered = fulllist:Filter(flags)

	local iFiltered = 1
	for iFull = iFiltered, #fulllist do
		if fulllist[iFull] ~= filtered[iFiltered] then
			local flagsFailed = band(bnot(fulllist[iFull].flagsSupported), flags)
			tooltip:AddLine(fulllist[iFull].name, nil, MFtoStr(flagsFailed))
			-- tooltip:AddLine(fulllist[iFull].name, nil, MFtoStr(band(flagsFailed,MF.NoFlyZone_Legal)), MFtoStr(band(flagsFailed,MF.Underwater_Legal)), MFtoStr(band(flagsFailed,MF.Moving_Legal)), MFtoStr(band(flagsFailed,MF.Indoors_Legal)))
		else
			local mount = fulllist[iFull]:SelectMount()
			if mount == nil then
				tooltip:AddLine(fulllist[iFull].name, nil, "no elligible mounts")
			else
				tooltip:AddLine(fulllist[iFull].name, mount.name);
				--tooltip:AddLine(fulllist[iFull].name, "X");
			end
			iFiltered = iFiltered + 1
		end
	end

	tooltip:AddLine(" ");
	tooltip:AddSeparator();
	tooltip:AddLine(" ");

	tooltip:SetCell(tooltip:AddHeader(), 1, "button macro", tooltip:GetColumnCount())
	Pokedex.ToggleMountPreClick(gv.btnToggleMount)
	tooltip:SetCell(tooltip:AddLine(), 1, gv.btnToggleMount:GetAttribute("macrotext"), tooltip:GetColumnCount())
	Pokedex.ToggleMountPostClick(gv.btnToggleMount)
	tooltip:AddLine(" ");
	tooltip:SetCell(tooltip:AddHeader(), 1, "combat macro", tooltip:GetColumnCount())
	tooltip:SetCell(tooltip:AddLine(), 1, gv.btnToggleMount:GetAttribute("macrotext"), tooltip:GetColumnCount())


	-- Use smart anchoring code to anchor the tooltip to our frame
	tooltip:SmartAnchorTo(self)

	-- Show it
	tooltip:Show()
end
 
local function TestTip_OnLeave(self)
	-- Release the tooltip
	LibQTip:Release(self.tooltip)
	self.tooltip = nil
end

function Pokedex:StartTestUI()
	if (not LibQTip) then
		LibQTip = LibStub('LibQTip-1.0');
	end

	if (not ldbTestFrame) then 
		ldbTestFrame = CreateFrame("Frame", "Pokedex_Test");
		ldbTestFrame:SetScript("OnUpdate", UpdateSpeed);
	end

	if (not ldbTestDataObj) then 
		ldbTestDataObj = LibStub("LibDataBroker-1.1"):NewDataObject("Pokedex_Test", {
			type	= "data source",
			icon	= "Interface\\Icons\\inv_pet_nurturedpenguinegg.png",
			label	= "Pokedex",
			text	= "0 YPS",
			OnEnter = TestTip_OnEnter,
			OnLeave = TestTip_OnLeave,
		});
	end
	
	if (not HeaderFont) then
		HeaderFont = CreateFont("PokedexHeaderFont");
		HeaderFont:SetFont(GameTooltipText:GetFont(), 17);
		HeaderFont:SetTextColor( 255/255, 215/255, 0/255 );
	end
	
	if (not LineFont) then
		LineFont = CreateFont("PokedexLineFont");
		LineFont:SetFont(GameTooltipText:GetFont(), 15);
		LineFont:SetTextColor( 255/255, 250/255, 205/255 );
	end
end
--@end-debug@]===]

function Pokedex:EchoTest()
	if (DC.DISMOUNT >= DL.AV) then 
		local SafeDismount = self.db.profile.SafeDismount
		self:Printf("SafeDismount  Enabled=%s  ForGathering=%s  ForCombat=%s  ForAttack=%s", 
			tostring(SafeDismount.Enabled),   tostring(SafeDismount.ForGathering), 
			tostring(SafeDismount.ForCombat), tostring(SafeDismount.ForAttack));
	end

--[===[@debug@
	-- table validation - verifies spellId and mount name match - helps catch typos in table
	-- will spam errors if on a taxi and we've ran test code just to get speed
	if (not UnitOnTaxi("player") and IsOutdoors()) then
		local listname, flagsArea = self:GetAreaInfo()

		-- self:Print("area flags", MFtoStr(flagsArea))
		local flagsArea = band(flagsArea, bnot(MF.Moving_Legal)) -- mounts return as usable spells while moving, so clear that flag if set
		local fSwimming = IsSwimming()

		local mounts = cmj.GetMountIDs()
		for _, idMount in ipairs(mounts) do
			local name, idSpell, icon, active, isUsable, sourceType, isFavorite, isFactionSpecific, faction, hideOnChar, isCollected = cmj.GetMountInfoByID(idMount)
			-- if isCollected and not hideOnChar and faction ~= gc.iOffFaction then
				local flags = self:GetMountAttributes(idMount)

	--			if (not fband(flags, MF.Vashjir) and not fband(flags, MF.Ahn_Qiraj)) then
				if (listname == "AhnQiraj" or not fband(flags, MF.Ahn_Qiraj)) then
					local fCanUse = IsUsableSpell(idSpell);
					
					-- add in the flags we leave off of some mounts to instead set on their buckets. 
					if fband(flags, MF.Swimmer)then
						flags = bor(flags, MF.Underwater_Legal, MF.NoFlyZone_Legal)
					end
					if fband(flags, MF.Walker) then
						flags = bor(flags, MF.Underwater_Legal, MF.NoFlyZone_Legal)
					end
					if fband(flags, MF.Ahn_Qiraj) then
						flags = bor(flags, MF.NoFlyZone_Legal)
					end
						

					if idSpell == 75207 then -- Abyssal Seahorse is buggy
						if fSwimming and not fCanUse then
							Pokedex:Printf("Abyssal Seahorse bug may be fixed")
						else
							assert(fCanUse == fSwimming, "Abyssal Seahorse doesn't match swimming")
						end
					elseif idSpell == 98718 then -- Subdued Seahorse is buggy
						assert(fCanUse or not fSwimming)
					elseif fCanUse then
						if (band(flags, flagsArea) ~= flagsArea) then
							-- mount is usable, but failed a flag check
							-- this is a mount we thought we couldn't summon, why?
							Pokedex:Printf("%6i %3i %s undersold without %s", idSpell, idMount, name, MFtoStr(band(bnot(flags), flagsArea)))
						end
					elseif gv.Skills.Riding ~= SL.None then
						-- don't bother to look for a reason unless we actually have riding skill
						if (band(flags, flagsArea) == flagsArea) then
							-- mount isn't usable, but we didn't fail any flag checks
							-- this is a mount we thought we could summon, why?
							Pokedex:Printf("%6i %3i %s oversold by %s", idSpell, idMount, name, MFtoStr(flagsArea))
						end
					end
				end
--			end
		end
		self:Printf("Mount IDs table contains %s entries", tostring(#mounts));
	end

	self:StartTestUI();
--@end-debug@]===]
end

function Pokedex:EchoCounts()
	self:Printf("Current counts are %i mounts, %i companions and %i titles", gv.Mounts.count, gv.Pets.count, gv.Titles.count);
end

function Pokedex:EchoSpeed()
	self:Print(TestSpeed(true));
end

function Pokedex:EchoZone()
	SetMapToCurrentZone();
	local iContinent = GetCurrentMapContinent();
	local strContinent
	if iContinent > 0 then
		local rgstrContinents = { GetMapContinents() };
		strContinent = format("%i  %s", iContinent, (rgstrContinents[iContinent*2] or "unknown"));
	else
		strContinent = format("%i  %s", iContinent, "unknown");
	end

	self:Printf("MapID: %i   Continent: %s   Zone: %s   Real Zone: %s   SubZone: %s", GetCurrentMapAreaID(), strContinent, GetZoneText(), GetRealZoneText(), GetSubZoneText());
	self:Printf("FlyableArea: %s   Swimming: %s   Outdoors: %s   Indoors: %s   Falling: %s", tostring(IsFlyableArea()), tostring(IsSwimming()), tostring(IsOutdoors()), tostring(IsIndoors()), tostring(IsFalling()) );

	local listname, flags = Pokedex:GetAreaInfo()
	self:Printf("NoFlyZone: %s   Underwater: %s   Moving: %s   Indoors: %s   Combat: %s", tostring(fband(flags, MF.NoFlyZone_Legal)), tostring(fband(flags, MF.Underwater_Legal)), tostring(fband(flags, MF.Moving_Legal)), tostring(fband(flags, MF.Indoors_Legal)), tostring(fband(flags, MF.Combat_Legal)));

	self:Print("")
	-- tooltip:AddHeader(listname, "Summonable", "Rejected");
	local fulllist = gv.Mounts.Lists[listname]
	local filtered = fulllist:Filter(flags)

	local iFiltered = 1
	for iFull = iFiltered, #fulllist do
		if fulllist[iFull] ~= filtered[iFiltered] then
			local flagsFailed = band(bnot(fulllist[iFull].flagsSupported), flags)
			self:Printf("%s   %s", fulllist[iFull].name, MFtoStr(flagsFailed))
			-- tooltip:AddLine(fulllist[iFull].name, nil, MFtoStr(band(flagsFailed,MF.NoFlyZone_Legal)), MFtoStr(band(flagsFailed,MF.Underwater_Legal)), MFtoStr(band(flagsFailed,MF.Moving_Legal)), MFtoStr(band(flagsFailed,MF.Indoors_Legal)))
		else
			local mount = fulllist[iFull]:SelectMount()
			if mount == nil then
				self:Printf("%s   %s", fulllist[iFull].name, "no elligible mounts")
			else
				self:Printf("%s   %s", fulllist[iFull].name, mount.name);
				--tooltip:AddLine(fulllist[iFull].name, "X");
			end
			iFiltered = iFiltered + 1
		end
	end
end

function Pokedex:ADDON_ACTION_BLOCKED(_, culprit, action, ...)
	if (culprit ~= "Pokedex") then return end
	self:Printf("ERROR: ADDON_ACTION_BLOCKED calling %s  %s", action, self:StrFromVarArg(nil, ...));
end

function Pokedex:ADDON_ACTION_FORBIDDEN(_, culprit, action, ...)
	if (culprit ~= "Pokedex") then return end
	self:Printf("ERROR: ADDON_ACTION_BLOCKED calling %s  %s", action, self:StrFromVarArg(nil, ...));
end


function Pokedex:SetDebug(info, value)
	local strName, strValue, index = self:GetArgs(value, 2);
	
	if (strName == nil) then
		self:DebugValues();
		return;
	end

	if (index ~= 1e9) then
		self:Print("ERROR: too many parameters passed");
		self:DebugUsage();
		return;
	end

	local iValue = tonumber(strValue)
	if (iValue == nil) then
		self:Print("ERROR: no value given for debug level");
		self:DebugUsage();
		return;
	elseif (iValue < DL.NONE or iValue > DL.MAX) then
		self:Printf("ERROR: level_value must be number between %i and %i", DL.NONE, DL.MAX);
		self:DebugUsage();
		return;
	end

	strName = strupper(strName);
	if (strName == "ALL") then
		for k in pairs(DC) do
			DC[k] = iValue;
		end
	elseif (DC[strName] ~= nil) then
			DC[strName] = iValue;
	else
		self:Print("ERROR: level_name not recognized. Must match an existing value or ALL.");
		self:DebugUsage();
		return;
	end

	self:DebugValues();
end

function Pokedex:DebugValues()
	self:Print("current debug levels:");
	for k,v in pairs(DC) do
		self:Printf("value: %i  name: %s", v, k);
		-- self:Printf("lvalue:%i pvalue:%i name:%s", DL[v.category], v, k);
	end
end

function Pokedex:DebugUsage()
	self:Print("to see current debug levels:");
	self:Print("  /pd debug ");
	self:Print("to set a level:");
	self:Print("  /pd debug level_name level_value");
end


function Pokedex:PrintTable(table, tableName)
	tableName = tableName or tostring(table);
	if (table == nil) then 
		self:Print("'table' is nil");
	elseif (type(table) ~= "table") then
		self:Printf("'table' is actually of type %s", type(table));
	else
		self:Printf("key/value pairs for table %s", tableName);
		for k,v in pairs(table) do
			self:Printf("key(%s): %s   value(%s): %s", type(k), tostring(k), type(v), tostring(v));
		end
	end
end

function Pokedex:StrFromVarArg(strDelim, ...)
	if (select("#", ...) == 0) then return ""; end
	strDelim = strDelim or "  "

	local strOut = tostring(select(1, ...));
	for i=2, select("#", ...), 1 do
		strOut = format("%s%s%s", strOut, strDelim, tostring(select(i, ...)));
	end

	return strOut;	
end

local function NewCounter(iStart)
	local i = 0
	return function()
		i = i + 1
		return i
	end
end

--=========================================================================--
--=========================================================================--
--
-- BUTTON FUNCTIONS
--
--=========================================================================--
--=========================================================================--

function Pokedex:LEARNED_SPELL_IN_TAB(event, spellID, ...)
	-- self:Print(event, spellID, ...)
	if (gc.rgidMonitoredSpell[spellID]) then 
		self:UpdateMountInfo(event);
		LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
	end
end

function Pokedex:PLAYER_SPECIALIZATION_CHANGED(...)
	-- self:Print(...)
	self:UpdateMountInfo("PLAYER_SPECIALIZATION_CHANGED");
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:PLAYER_TALENT_UPDATE(...)
	-- self:Print(...)
	self:UpdateMountInfo("PLAYER_TALENT_UPDATE");
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:PLAYER_EQUIPMENT_CHANGED(event, slot, hasItem, ...)
	-- self:Print(event, slot, hasItem, ...)
	if slot == INVSLOT_BACK then
		self:UpdateMountInfo("INVSLOT_BACK changed");
		LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
	end
end

function Pokedex:CreateMountButton()
	gv.btnToggleMount = CreateFrame("Button", "PokedexToggleMountButton", UIParent, "SecureActionButtonTemplate");
	if not gv.btnToggleMount then self:Print("btnToggleMount is nil") end

	_G["BINDING_NAME_CLICK PokedexToggleMountButton:LeftButton"] = L["Toggle Mount"];

	gv.btnToggleMount:SetAttribute("type", "macro");

	gv.btnToggleMount:SetScript("PreClick",  Pokedex.ToggleMountPreClick);
	gv.btnToggleMount:SetScript("PostClick", Pokedex.ToggleMountPostClick);
	Pokedex.ToggleMountPostClick(gv.btnToggleMount);

end

Pokedex.ToggleMountPreClick = function(btn)
	if InCombatLockdown() then return end
	if nil ~= UnitCastingInfo("player") then return end

	local player = Pokedex.Globals.Constants.Player
	local strMacro = nil

	if (Pokedex:IsMounted() and not IsFalling()) then	-- not true when in a mount form ... by design
		strMacro = player.strCombatMacro
	else
		local mount = Pokedex:SelectMount()
		strMacro = format("%s%s", 
			(mount.fOnGCD) and "" or player.strCastOnMount, 
			(mount.strMacro) and mount.strMacro or "")
	end

	btn:SetAttribute("macrotext", strMacro)
	-- Pokedex:Print("ToggleMount macro\n", strMacro)
end

Pokedex.ToggleMountPostClick = function(btn)
	if InCombatLockdown() then return end
	btn:SetAttribute("macrotext", Pokedex.Globals.Constants.Player.strCombatMacro);
end


--=========================================================================--
--=========================================================================--
--
-- SKILL FUNCTIONS
--
--=========================================================================--
--=========================================================================--

local function GetSkills()
	local function GetSkillRanks(...)
		local t = {}
		for i = 1, 2 do  -- first two values returned are the two professions
			local iProf = select(i, ...)
			if iProf then
				local name, _, rank, maxRank, _, _, idSkill = GetProfessionInfo(iProf);
				local skill = SI[idSkill]
				if skill then
					t[skill] = rank
				else
					assert(skill ~= nil, tostring(idSkill).."  "..tostring(name))
				end
			end
		end
		return t
	end

	-- find out what skills are known
	local skills = GetSkillRanks(GetProfessions());

	skills.fHerbalist = nil ~= skills.Herbalism
	skills.fSkinner   = nil ~= skills.Skinning
	skills.fMiner     = nil ~= skills.Mining
	skills.fGatherer  = skills.fHerbalist or skills.fSkinner or skills.fMiner

	skills.fEngineer  = nil ~= skills.Engineering

	-- determine our riding skill level
	for _,v in ipairs(gc.rgRidingSkills) do
		if (IsSpellKnown(v.id)) then
			skills.Riding = v.level
			break
		end
	end

	-- only one meta is created, the first time the function is called
	if (not gc.metaSkills) then 
		local t = {}
		for _, skill in pairs(SI) do
			t[skill] = 0
		end
		gc.metaSkills = { __index = t }
	end
	return setmetatable(skills, gc.metaSkills)
end

function Pokedex:InitSkills()
	-- find out what skills are known
	gv.Skills = GetSkills()

	if (DC.MISC >= DL.BASIC) then 
		local msg = format("SKILLS  Riding:%s  fGatherer:%s", gc.rgstrSkillRankName[gv.Skills.Riding], tostring(gv.Skills.fGatherer))
		for _, skill in pairs(SI) do
			local rank = rawget(gv.Skills, skill)
			if rank ~= nil then
				msg = format("%s  %s:%i", msg, skill, rank)
			end
		end
		self:Print(msg); 
	end
end

function Pokedex:SKILL_LINES_CHANGED(...)
	-- if (DC.MISC >= DL.AV) then self:Printf("SKILL_LINES_CHANGED  %s", self:StrFromVarArg(nil, ...)); end

	local oldSkills = gv.Skills;
	gv.Skills = GetSkills()

	-- did we gain a new level of riding skill
	if gv.Skills.Riding > oldSkills.Riding then
		if (DC.MISC >= DL.BASIC) then self:Printf("skill for %s riding earned", gc.rgstrSkillRankName[gv.Skills.Riding]); end
		self:UpdateMountInfo("Riding skill level changed");
		LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
	end

	if (gv.Skills.fGatherer ~= oldSkills.fGatherer) then
		local SafeDismount = self.db.profile.SafeDismount
		if (SafeDismount.ForGathering and not gv.Skills.fGatherer) then
			if (DC.MISC >= DL.BASIC) then self:Print("we no longer have a gathering skill and so will disable dismount for gathering"); end
			SafeDismount.ForGathering = false;
		end
		LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
	end

	if (gv.Skills.fEngineer ~= oldSkills.fEngineer) then
		if (gv.Skills.fEngineer) then
			self:RegisterEvent("PLAYER_EQUIPMENT_CHANGED");
		else
			self:UnregisterEvent("PLAYER_EQUIPMENT_CHANGED");
		end
	end
end


--=========================================================================--
--=========================================================================--
--
-- MOUNT AND COMPANION SHARED FUNCTIONS
--
--=========================================================================--
--=========================================================================--

function Pokedex:SummonVendor(fNoSquire, fNoMammoth, fNoYak)
	local strFaction = UnitFactionGroup("player")
	local idArgentMinion = gc.tblArgentMinion[strFaction]
	local idMammoth = gc.tblTravelersTundraMammoth[strFaction]

	local petArgentMinion = not fNoSquire and gv.Pets.ByID[idArgentMinion]
	local mountMammoth = not fNoMammoth and gv.Mounts.ByID[idMammoth]
	local mountYak = not fNoYak and gv.Mounts.ByID[gc.idSpellGrandExpeditionYak]

	if (IsOutdoors()) then 
		if (mountMammoth) then
			if (not mountMammoth.fActive) then self:SummonSelectedMount(mountMammoth.mid) end
		elseif (mountYak) then
			if (not mountYak.fActive) then self:SummonSelectedMount(mountYak.mid) end
		elseif (petArgentMinion) then
			if (petArgentMinion ~= self:GetCurrentPet()) then self:SummonSelectedPet(petArgentMinion); end
		else
			self:Print(L["ERROR: You have no mounts or pets with those capabilities"]);
		end
	else
		if (petArgentMinion) then
			if (petArgentMinion ~= self:GetCurrentPet()) then self:SummonSelectedPet(petArgentMinion); end
		elseif (mountYak) then
			self:Printf(L["ERROR: You cannot summon %s in this area"], mountYak.name);
		elseif (mountMammoth) then
			self:Printf(L["ERROR: You cannot summon %s in this area"], mountMammoth.name);
		else
			self:Print(L["ERROR: You have no mounts or pets with those capabilities"]);
		end
	end
end

function Pokedex:SummonTrainWrecker()
	local petLilXT        = gv.Pets.ByID[gc.idSpeciesLilXT]
	local petLandrosLilXT = gv.Pets.ByID[gc.idSpeciesLandrosLilXT]

	if petLilXT ~= nil then
		if (petLilXT ~= self:GetCurrentPet()) then self:SummonSelectedPet(petLilXT) end
	elseif petLandrosLilXT ~= nil then
		if (petLandrosLilXT ~= self:GetCurrentPet()) then self:SummonSelectedPet(petLandrosLilXT) end
	end
end

function Pokedex:COMPANION_LEARNED(...)
	if (DC.MOUNTS >= DL.BASIC or DC.PETS >= DL.BASIC) then self:Print("COMPANION_LEARNED  " .. self:StrFromVarArg(nil, ...)); end

	local CMounts = getmetatable(gv.Mounts)
	CMounts.numOwned = -1

	self:UpdateMountInfo("COMPANION_LEARNED MOUNT")
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:CMJSummonHook(id)
	gv.Mounts:SetCurrent(gv.Mounts.ByMID[id])
end

function Pokedex:CMJDismissHook()
	gv.Mounts:SetCurrent(gc.nomount)
end


function Pokedex:AnnounceSummon(strName)
	if (not self.db.profile.fAnnounce) then return end

	local iChannel = self.db.profile.iChannel

	-- if not in raid, try sending it to party
	if (3 == iChannel and not IsInRaid()) then
		iChannel = 2
	end
	-- if not in a party, just send it to yourself
	if (2 == iChannel and 0 == GetNumGroupMembers()) then
		iChannel = 1
	end

	local strOut
	if (iChannel == 4) then
		strOut = format(L["lets %s know that they have been chosen."], strName);
	else
		strOut = format(L["%s, I choose you!"], strName);
	end

	if (iChannel == 1) then
		self:Print(strOut);
	else
		SendChatMessage(strOut, gc.rgstrChannels[iChannel]);
	end

end


--=========================================================================--
--=========================================================================--
--
-- MOUNT FUNCTIONS
--
--=========================================================================--
--=========================================================================--

function Pokedex:IsMounted()
	return (IsMounted() or UnitInVehicle("player"));
end

function Pokedex:ToggleMount()
	if (Pokedex:IsMounted()) then
		self:DismissMount()
	else
		self:SummonMount()
	end
end

function Pokedex:DismissMount()
	if UnitInVehicle("player") then
		VehicleExit()
		if (self.db.profile.fAutoSummonCompanion and self:GetCurrentPet() == gc.nopet and gv.Pets.count > 0) then
			self:SummonCompanion()
		end
	elseif IsMounted() then
		-- TODO find if this goes through dismiss companion hook for tracking
		_G.Dismount()
		if (self.db.profile.fAutoSummonCompanion and self:GetCurrentPet() == gc.nopet and gv.Pets.count > 0) then
			self:SummonCompanion()
		end
	end
end

function Pokedex:SummonMount(...)
	local mount = self:SelectMount(true, ...);
	if (mount and mount.mid > 0) then
		self:SummonSelectedMount(mount.mid);
	end
end

function Pokedex:SummonNextMount()
	local CurMount = gv.Mounts:GetCurrent()
	local iCurrent = CurMount and CurMount.index or 0
	local rgMounts = gv.Mounts.ByIndex

	-- print(CurMount.name and CurMount.index)
	for i = iCurrent+1, #rgMounts do
		local mount = rgMounts[i]
		if mount.id and mount.id ~= gc.idSpellAbyssalSeahorse then
			if IsUsableSpell(mount.id) and not fband(mount.flags, MF.Unusable) then
				self:SummonSelectedMount(mount.mid)
				return
			end
		end
	end
	
	for i = 1, iCurrent-1 do
		local mount = rgMounts[i]
		if mount.id and mount.id ~= gc.idSpellAbyssalSeahorse then
			if IsUsableSpell(mount.id) then
				self:SummonSelectedMount(mount.mid)
				return
			end
		end
	end
end

function Pokedex:SummonOtherMount()
	local mounttypes = gv.Mounts.Types
	local ATVs = mounttypes.ATVs
	local runners = mounttypes.Runners

	local listname, flags = self:GetAreaInfo()
	-- Pokedex:Printf("%s  %x", listname, flags)
	local bucketlist = gv.Mounts.Lists[listname]:Filter(flags)
	local mount, bucket = bucketlist:SelectMount(true)
	-- print(mount, bucket)
	-- print(mount.name, bucket.name)
	mount = nil

	local function OverrideOrder(list, selected, desired)
		local iSelected, iDesired = nil, 0
		for i,bucket in ipairs(list) do
			if bucket == selected then iSelected = i end
			if bucket == desired then iDesired = i end
		end

		if not iSelected then 
			-- self:Print("error: selected bucket was not in list")
		elseif iSelected < iDesired then
			-- this is the case where we could have selected from the set we think the
			-- user may be asking for but we didn't because we selected from something
			-- that we thought was better before we got to it
			-- Pokedex:Printf("Overriding %s for %s", selected.name, desired.name)
			return desired:SelectMount(true)
		end
	end

	if IsSwimming() then 
		-- Pokedex:Print("bucket list before adjustments")
		-- for k,v in ipairs(bucketlist) do print(k,v.name) end

		local swimmers = mounttypes.Swimmers
		local vashjir = mounttypes.Vashjir
		local aquatic = gc.Player.AquaticForm

		if fband(flags, MF.Underwater_Legal) then
			-- this means user has a breath timer so we should be selecting a fast swimming mount or form
			if bucket == swimmers or bucket == vashjir or bucket == aquatic then
				-- we did, but doesn't want that then we should pull the swimming buckets from the list
				local iCur = 1
				while bucketlist[iCur] do
					local cur = bucketlist[iCur]
					if cur == swimmers or cur == vashjir or cur == aquatic then
						tremove(bucketlist, iCur)
					else
						iCur = iCur + 1
					end
				end

				-- Pokedex:Print("bucket list after adjustments")
				-- for k,v in ipairs(bucketlist) do print(k,v.name) end
				
				mount, bucket = bucketlist:SelectMount(true)
				Pokedex:Printf("Overriding; selecting %s from %s", mount.name, bucket.name)
			end
		else
			-- things that give you water breathing make it look like you're at the surface 
			-- so we should be selecting a flyer > water walker > fishing raft > swimmer > runner
			if not (bucket == swimmers or bucket == vashjir or bucket == aquatic) then
				-- user probably wants to stay underwater and so wants the faster swim speed
				while bucketlist[1] do
					local cur = bucketlist[1]
					if cur == swimmers or cur == vashjir or cur == aquatic then
						break
					else
						tremove(bucketlist, 1)
					end
				end

				-- Pokedex:Print("bucket list after adjustments")
				-- for k,v in ipairs(bucketlist) do print(k,v.name) end
				
				mount, bucket = bucketlist:SelectMount(true)
				-- Pokedex:Printf("Overriding; selecting %s from %s", mount.name, bucket.name)
			end
		end
	elseif not fband(flags, MF.NoFlyZone_Legal) and bucket ~= runners then
		-- they could fly but maybe want to show of their ground mounts
		mount = OverrideOrder(bucketlist, bucket, runners)
	end
	
	if not mount then
		-- whatever we gave them they weren't happy, give 'em something useful
		-- Pokedex:Printf("fell through to ATVS. underwater: %s  flyable: %s", tostring(fband(flags, MF.Underwater_Legal)), tostring(not fband(flags, MF.NoFlyZone_Legal)))
		mount = ATVs:SelectMount(true)
	end

	if (mount and mount.mid > 0) then
		self:SummonSelectedMount(mount.mid);
	end
end


function Pokedex:SummonPassengerMount()
	-- fPassengers, fWaterWalkers)
	Pokedex:SummonMount(true)
end

function Pokedex:SummonWaterWalker()
	-- fPassengers, fWaterWalkers)
	Pokedex:SummonMount(false, true)
end

function Pokedex:SummonSelectedMount(idMount)
	local mount = gv.Mounts.ByMID[idMount]
	if not mount then
		if (DC.MOUNTS >= DL.BASIC) then self:Print("ERROR - bad companion index"); end
		return
	end

	cmj.SummonByID(idMount)
	--self:AnnounceSummon(mount.name);

	-- change title on load, unless its heroic maloriak temporary title in which case revel in the glory
	if (self.db.profile.fChangeTitleOnMount and GetCurrentTitle() ~= gc.idTitleMinionSlayer) then 
		self:ChangeTitle(); 
	end
end

-- works through buckets to find a usable mount/form
function Pokedex:SelectMount(...)
	local listname, flags = self:GetAreaInfo()
	local buckets = gv.Mounts.Lists[listname]

	if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("selecting mount from %s bucket list", listname); end
	return buckets:Filter(flags):SelectMount(...) or gc.nomount
end


function Pokedex:TestMount()
	self:Print("test mount")
	local mount = self:SelectMount()
	self:Print(format("%s%s", 
			(mount.fOnGCD) and "" or gc.Player.strCastOnMount, 
			(mount.strMacro) and mount.strMacro or "/Pokedex ToggleMount"))
end

--=========================================================================--
--=========================================================================--
--
-- NEW MOUNT FUNCTIONS NEW NEW
--
--=========================================================================--
--=========================================================================--

local function UpdateMountFlags(mount)
	local flags = mount.rawflags

	-- user has no riding skill
	if (gv.Skills.Riding == SL.None and not (fband(flags, MF.Sidecar) or fband(flags, MF.Walker))) then
		flags = MF.Unusable;
	end

	-- check if its a mount that has a skill requirement
	if (mount.skill and gv.Skills[mount.skill.name] < mount.skill.rank) then
		flags = MF.Unusable;  -- failed required skill check
	end

	-- check if its a mount that has a class requirement
	if (mount.class and mount.class ~= select(2, UnitClass("player"))) then
		flags = MF.Unusable;  -- failed required class check
	end

	-- adjust flags of flying mounts
	if fband(flags, MF.Flyer) then
		if fband(flags, MF.NoFlyZone_Legal) then
			if (Pokedex.db.profile.fFavorGroundMounts) then
				-- remove ground flag from flying mounts if user favors ground mounts
				flags = band(flags, maskClearNFZL);
			end
		elseif (gv.Skills.Riding < SL.Expert) then
			-- user does not have flying skill and mount is flyer that cannot run in ground only zone
			flags = band(flags, MF.Unknown); -- keep the unknown flag if set
			flags = bor(flags, MF.Unusable);
		end
	end

	mount.flags = flags
end

-- returns: rawflags
function Pokedex:GetMountAttributes(idMount)
	local _, _, _, _, gameflags = cmj.GetMountInfoExtraByID(idMount)

	if gameflags == 230 then -- ground mount
		return bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)
	elseif gameflags == 269 then -- water striders
		return bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)
	elseif gameflags == 231 then -- turtles
		return bor(MF.Swimmer, MF.Walker)
	elseif gameflags == 284 then -- chauffered motorcylces
		return bor(MF.Sidecar, MF.NoFlyZone_Legal, MF.Underwater_Legal)
	elseif gameflags == 232 then -- vashjir
		return bor(MF.Vashjir)
	elseif gameflags == 241 then -- qiraji scarab
		return bor(MF.Ahn_Qiraj)
	elseif gameflags == 247 then -- flying cloud (hydrophobes)
		return bor(MF.Flyer, MF.NoFlyZone_Legal)
	elseif gameflags == 248 then -- other flyers (hydroplanes, actually ATV as there are no known underwater planes that are fly zone only)
		return bor(MF.Flyer, MF.NoFlyZone_Legal, MF.Underwater_Legal)
	elseif gameflags == 254 then -- subdued seahorse
		return bor(MF.Swimmer); -- if it can't fly or run but can swim, then assume it must do that well
	elseif gameflags == 284 then -- chauffered motorcycle with sidecar
		return bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)
	elseif gameflags == 242 then -- swift spectral gryphon
		return bor(MF.Unknown, MF.Unusable);
	else
		return bor(MF.Unknown, MF.Unusable);
	end
end

--=====================================================================--
-- CLASS DEFINITION FOR STANDARD MOUNTS AND SPECIAL MOUNTS
--=====================================================================--

local metaMount = {
	__lt = gs.pfMounts,
	__index = function(t,k)
			if (k == "rank") then
				return t.char_rank or t.profile_rank
			elseif (k == "char_rank") then
				return Pokedex.db.char.mounts.ranks[t.name]
			elseif (k == "profile_rank") then
				return Pokedex.db.profile.mounts.ranks[t.name] or Pokedex.db.profile.mounts.iDefaultRank
			elseif (k == "fActive") then
				return select(4, cmj.GetMountInfoByID(t.mid))
			elseif (k == "fHas") then
				return true
			elseif (k == "fCan") then
				-- return IsUsableSpell(t.id)
				return true
			elseif (k == "fOnGCD") then
				return false
			elseif (k == "strMacro") then
				return format("/run Pokedex:SummonSelectedMount(%i);", t.mid)
			else
				return nil
			end
		end,
	__newindex = function(t,k,v)
			if (k == "char_rank") then
				Pokedex.db.char.mounts.ranks[t.name] = v
			elseif (k == "profile_rank") then
				Pokedex.db.profile.mounts.ranks[t.name] = (Pokedex.db.profile.mounts.iDefaultRank ~= v) and v or nil
			else
				Pokedex:Print("unexpected table assignment", t.name, k, v)
			end
		end,
}

--=====================================================================--
-- CLASS DEFINITION FOR RUNNING WILD, WORGEN MOUNT FORM
--=====================================================================--

local function NewRunningWild()
	-- if player is Worgen this is the mount meta to use post initialization
	local metaRunningWild ={
		__lt = gs.pfMounts,
		__index = function(t,k)
				if (k == "rank") then
					return t.char_rank or t.profile_rank
				elseif (k == "char_rank") then
					return Pokedex.db.char.mounts.ranks[t.name]
				elseif (k == "profile_rank") then
					return Pokedex.db.profile.mounts.ranks[t.name] or Pokedex.db.profile.mounts.iDefaultRank
				elseif (k == "fActive") then
					return UnitBuff("player", t.name);
				elseif (k == "fHas") then
					if (IsSpellKnown(t.id)) then
						rawset(t, k, true);
						return true;
					else
						return false;
					end
				elseif (k == "fCan") then
					return not (UnitBuff("player", t.strCoinOfManyFaces) or UnitBuff("player", t.strDressedToKill) or UnitBuff("player", t.strPolyformicAcidPotion));
				end
				return nil;
			end,
		__newindex = function(t,k,v)
				if (k == "char_rank") then
					Pokedex.db.char.mounts.ranks[t.name] = v
				elseif (k == "profile_rank") then
					Pokedex.db.profile.mounts.ranks[t.name] = (Pokedex.db.profile.mounts.iDefaultRank ~= v) and v or nil
				else
					Pokedex:Print("unexpected table assignment for Running Wild", k, v)
				end
			end,
	}

	-- Delay inits table. Executes on first table access and replaces itself.
	local metaInit = { 
	__index = function(t,k)
			setmetatable( t, nil );
			local spellName = GetSpellInfo(t.id) or "Running Wild";
			t.name = spellName

			local _, race = UnitRace("player");
			if (race ~= "Worgen") then 
				t.fHas = false;  -- setting this alone should really be enough
				t.flags = 0
				return t[k];
			end

			t.strMacro = format("/cast %s", spellName)
			t.fOnGCD = false;
			t.flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)
			t.rawflags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)

			t.strCoinOfManyFaces = GetSpellInfo(gc.idSpellCoinOfManyFaces) or "Coin of Many Faces";
			t.strDressedToKill = GetSpellInfo(gc.idSpellDressedToKill) or "Dressed to Kill";
			t.strPolyformicAcidPotion = GetSpellInfo(gc.idSpellPolyformicAcidPotion) or "Polyformic Acid Potion";

			setmetatable( t, metaRunningWild )
			return t[k];
		end,
	}

	return setmetatable( { id = 87840 }, metaInit );
end

--=====================================================================--
-- CLASS DEFINITION FOR DRAGONWRATH, TARECGOSA'S VISAGE
--=====================================================================--

local function NewTarecgosa()
	local metaTarecgosa = {
		__lt = gs.pfMounts,
		__index = function(t,k)
				if (k == "rank") then
					return t.char_rank or t.profile_rank
				elseif (k == "char_rank") then
					return Pokedex.db.char.mounts.ranks[t.name]
				elseif (k == "profile_rank") then
					return Pokedex.db.profile.mounts.ranks[t.name] or Pokedex.db.profile.mounts.iDefaultRank
				elseif (k == "fActive") then
					return UnitBuff("player", t.name);
				elseif (k == "fCan") then 	-- true if item is equipped
					return (IsEquippedItem(t.iid));
				elseif (k == "fHas") then	-- true if item is in inventory or bank
					return (GetItemCount(t.iid, true) > 0);
				end
			return nil;
		end,
		__newindex = function(t,k,v)
				if (k == "char_rank") then
					Pokedex.db.char.mounts.ranks[t.name] = v
				elseif (k == "profile_rank") then
					Pokedex.db.profile.mounts.ranks[t.name] = (Pokedex.db.profile.mounts.iDefaultRank ~= v) and v or nil
				else
					Pokedex:Print("unexpected table assignment for Tarecgosa's Visage", k, v)
				end
			end,
	}

	-- Delay inits table. Called on first table access as the __index metamethod.
	local metaInit = {
		__index = function(t,k)
			setmetatable(t, nil);

			t.name = GetSpellInfo(t.id) or "Tarecgosa's Visage";
			t.strMacro = format("/use %s", GetItemInfo(t.iid) or "Dragonwrath, Tarecgosa's Rest")
			t.fOnGCD = false;
			t.flags = bor(MF.Flyer, MF.NoFlyZone_Legal, MF.Underwater_Legal)
			t.rawflags = bor(MF.Flyer, MF.NoFlyZone_Legal, MF.Underwater_Legal)

			setmetatable(t, metaTarecgosa);
			return t[k];
		end,
	}

	return setmetatable( { id = 101641, iid = 71086 }, metaInit );
end

--=====================================================================--
-- CLASS DEFINITION FOR MAGIC BROOM
--=====================================================================--

local function NewMagicBroom()
	local metaMagicBroom = {
		__lt = gs.pfMounts,
		__index = function(t,k)
				if (k == "rank") then
					return t.char_rank or t.profile_rank
				elseif (k == "char_rank") then
					return Pokedex.db.char.mounts.ranks[t.name]
				elseif (k == "profile_rank") then
					return Pokedex.db.profile.mounts.ranks[t.name] or Pokedex.db.profile.mounts.iDefaultRank
				elseif (k == "fHas") then	-- true if item is in inventory or bank
					return (GetItemCount(t.iid, true) > 0);
				elseif (k == "fCan") then 	-- true if item is in inventory only
					return (GetItemCount(t.iid, false) > 0);
				elseif (k == "fActive") then
					return UnitBuff("player", t.name);
				end
			return nil;
		end,
		__newindex = function(t,k,v)
				if (k == "char_rank") then
					Pokedex.db.char.mounts.ranks[t.name] = v
				elseif (k == "profile_rank") then
					Pokedex.db.profile.mounts.ranks[t.name] = (Pokedex.db.profile.mounts.iDefaultRank ~= v) and v or nil
				else
					Pokedex:Print("unexpected table assignment for Magic Broom", k, v)
				end
			end,
	}

	-- Delay inits table. Called on first table access as the __index metamethod.
	local metaInit = {
		__index = function(t,k)
			setmetatable(t, nil);

			t.name = GetSpellInfo(t.id) or "Magic Broom";
			t.strMacro = format("/use %s", GetItemInfo(t.iid) or "Magic Broom")
			t.fOnGCD = false;
			t.flags = bor(MF.Flyer, MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Moving_Legal)
			t.rawflags = bor(MF.Flyer, MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Moving_Legal)

			setmetatable(t, metaMagicBroom);
			return t[k];
		end,
	}

	return setmetatable( { id = 47977, iid = 37011 }, metaInit );
end

--=====================================================================--
-- CLASS DEFINITION FOR MOONFANG / SHIMMERING MOONSTONE
--=====================================================================--

local function NewMoonfang()
	local metaMoonfang = {
		__lt = gs.pfMounts,
		__index = function(t,k)
				if (k == "rank") then
					return t.char_rank or t.profile_rank
				elseif (k == "char_rank") then
					return Pokedex.db.char.mounts.ranks[t.name]
				elseif (k == "profile_rank") then
					return Pokedex.db.profile.mounts.ranks[t.name] or Pokedex.db.profile.mounts.iDefaultRank
				elseif (k == "fHas") then	-- true if item is in inventory or bank
					return (GetItemCount(t.iid, true) > 0);
				elseif (k == "fCan") then 	-- true if item is in inventory only
					return (GetItemCount(t.iid, false) > 0);
				elseif (k == "fActive") then
					return UnitBuff("player", t.name);
				end
			return nil;
		end,
		__newindex = function(t,k,v)
				if (k == "char_rank") then
					Pokedex.db.char.mounts.ranks[t.name] = v
				elseif (k == "profile_rank") then
					Pokedex.db.profile.mounts.ranks[t.name] = (Pokedex.db.profile.mounts.iDefaultRank ~= v) and v or nil
				else
					Pokedex:Print("unexpected table assignment for Moonfang", k, v)
				end
			end,
	}

	-- Delay inits table. Called on first table access as the __index metamethod.
	local metaInit = {
		__index = function(t,k)
			setmetatable(t, nil);

			t.name = GetSpellInfo(t.id) or "Moonfang";
			t.strMacro = format("/use %s", GetItemInfo(t.iid) or "Shimmering Moonstone")
			t.fOnGCD = false;
			t.flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)
			t.rawflags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal)

			setmetatable(t, metaMoonfang);
			return t[k];
		end,
	}

	return setmetatable( { id = 145133, iid = 101675 }, metaInit );
end

--=====================================================================--
-- CLASS DEFINITION FOR HOT ROD
--=====================================================================--

local function NewHotRod()
	local metaHotRod = {
		__lt = gs.pfMounts,
		__index = function(t,k)
				if (k == "rank") then
					return t.char_rank or t.profile_rank
				elseif (k == "char_rank") then
					return Pokedex.db.char.mounts.ranks[t.name]
				elseif (k == "profile_rank") then
					return Pokedex.db.profile.mounts.ranks[t.name] or Pokedex.db.profile.mounts.iDefaultRank
				elseif (k == "fHas") then	-- true if you're on Kezan island
					SetMapToCurrentZone();
					return GetCurrentMapAreaID() == 605;
				elseif (k == "fCan") then 	-- true if item is in inventory only
					return (GetItemCount(t.iid, false) > 0);
				elseif (k == "fActive") then
					return UnitBuff("player", t.name);
				end
			return nil;
		end,
		__newindex = function(t,k,v)
				if (k == "char_rank") then
					Pokedex.db.char.mounts.ranks[t.name] = v
				elseif (k == "profile_rank") then
					Pokedex.db.profile.mounts.ranks[t.name] = (Pokedex.db.profile.mounts.iDefaultRank ~= v) and v or nil
				else
					Pokedex:Print("unexpected table assignment for Hot Rod", k, v)
				end
			end,
	}

	-- Delay inits table. Called on first table access as the __index metamethod.
	local metaInit = {
		__index = function(t,k)
			setmetatable(t, nil);

			t.name = GetSpellInfo(t.id) or "Hot Rod";
			t.strMacro = format("/use %s", GetItemInfo(t.iid) or "Keys to the Hot Rod")
			t.fOnGCD = false;
			t.flags = bor(MF.NoFlyZone_Legal)
			t.rawflags = bor(MF.NoFlyZone_Legal)

			setmetatable(t, metaHotRod);
			return t[k];
		end,
	}

	return setmetatable( { id = 66392, iid = 46856 }, metaInit );
end

--=====================================================================--
-- MOUNT BUCKETS 
--=====================================================================--

local metaFormBucket = {}
metaFormBucket.__lt = gs.ByOrder
metaFormBucket.__index = metaFormBucket

local function MakeNewFormBucket(form)
	return setmetatable( { mount = form, name = form.name, flagsSupported = form.flags }, metaFormBucket )
end

function metaFormBucket:IsEmpty()
	return not self.mount.fHas
end

function metaFormBucket:IsNotEmpty()
	return self.mount.fHas
end

function metaFormBucket:MountQualifies()
	return false
end

function metaFormBucket:MountInBucket(mount)
	return self.mount == mount
end

function metaFormBucket:GetFirstMount()
	return self.mount
end

function metaFormBucket:SelectMount(fNoSpecials)
	if (not fNoSpecials and self.mount.fCan) then
		if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("selecting form %s", self.name); end
		return self.mount
	end

	if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("skipping over form %s", self.name); end
	return nil
end


local metaMountBucket = {}
metaMountBucket.__lt = gs.ByOrder
metaMountBucket.__index = metaMountBucket

local function MakeNewMountBucket(name, flagsIncluded, flagsSupported)
	return setmetatable( { name = name, flagsIncluded = flagsIncluded, flagsSupported = flagsSupported }, metaMountBucket )
end

function metaMountBucket:IsEmpty()
	return next(self.mounts) == nil
end

function metaMountBucket:IsNotEmpty()
	return next(self.mounts) ~= nil
end

function metaMountBucket:MountQualifies(mount)
	return self.flagsIncluded and self.flagsIncluded == band(mount.flags, self.flagsIncluded)
end

function metaMountBucket:MountInBucket(mount)
	return self.mounts[mount] ~= nil
end

function metaMountBucket:GetFirstMount()
	local first = next(self.mounts)  -- grab one so that we don't have to nil check in the loop
	for mount,_ in pairs(self.mounts) do
		first = gs.Mounts(first, mount) and first or mount
	end		

	return first
end

function metaMountBucket:SelectMount(fNoSpecials, fPassengers, fWaterWalkers)
	local strCurCast = UnitCastingInfo("player")
	if (strCurCast ~= nil) then
		if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("currently casting %s; summon will fail for this reason so ignore this summons attempt", strCurCast); end
		return;
	end

	local CurMount = gv.Mounts:GetCurrent();
	
	local rgFiltered = {};
	local cTotalRanks = 0;
	local fSkippedCur = false;
	local fSkippedHot = false;

	for mount,name in pairs(self.mounts) do 
		-- if a mount is flagging itself as disabled
		if (not mount.fCan) then
			if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("%s skipped for returning disabled", mount.name); end
		-- cannot call specials from inside any of the old summoning methods
		elseif (fNoSpecials and mount.mid == nil) then
			if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("%s skipped for not being summonable from this method", mount.name); end
		elseif (fPassengers and not gc.rgPassengerMounts[mount.id]) then
			if (DC.MOUNTS >= DL.AV) then Pokedex:Printf("%s skipped for not carrying passengers", mount.name); end
		elseif (fWaterWalkers and not gc.rgWaterWalkingMounts[mount.id]) then
			if (DC.MOUNTS >= DL.AV) then Pokedex:Printf("%s skipped for not water walking", mount.name); end
		-- if its ranked as 0, then it gets skipped
		elseif (mount.rank == 0) then
			if (DC.MOUNTS >= DL.AV) then Pokedex:Printf("%s skipped for having rank of 0", mount.name); end
		-- if its the current mount, then it is skipped
		elseif (mount == CurMount) then
			if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("%s skipped for being current mount", mount.name); end
			fSkippedCur = true;
		-- if hot mount and hotness is enabled then don't add to list, it will be handled seperately
		elseif (Pokedex.db.char.mounts.fEnableHotness and mount == gv.HotMount) then
			if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("%s skipped for being hot mount", mount.name); end
			fSkippedHot = true;
		-- else its put into the pool of summonables
		else
			tinsert(rgFiltered, mount)
			cTotalRanks = cTotalRanks + mount.rank;
			if (DC.MOUNTS >= DL.AV) then Pokedex:Printf("%s added to list of summonable mounts with a rank of %i. Total rank count is %i", mount.name, mount.rank, cTotalRanks); end
		end
	end
	if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("%s list built: %i mounts with total ranks of %i", self.name, #rgFiltered, cTotalRanks); end

	-- if we skipped the hot mount while building a list, nows the time for its heat check
	if (fSkippedHot) then
		local iHeatCheck = mrandom(1,100);
		if (Pokedex.db.char.mounts.iHeat >= iHeatCheck) then
			if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("%s passed heat check (rolled %i, needed %i or less) and will be summoned", gv.HotMount.name, iHeatCheck, Pokedex.db.char.mounts.iHeat); end
			return gv.HotMount;
		end

		if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("%s skipped for failing heat check (rolled %i, needed %i or less) as hot mount", gv.HotMount.name, iHeatCheck, Pokedex.db.char.mounts.iHeat); end
	end

	-- selection returned 0 mounts to choose from	
	if (#rgFiltered == 0 or cTotalRanks == 0) then 
		-- both values should be in sync
		if (#rgFiltered ~= cTotalRanks) then Pokedex:Printf("ERROR: only one of #rgFiltered and cTotalRanks was zero for %s", self.name); end

		if (fSkippedHot) then
			if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("hot mount %s failed heat check but is apparently only one summonable", gv.HotMount.name); end
			return gv.HotMount;
		end

		if (fSkippedCur) then
			if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("current mount %s is apparently only one summonable; doing nothing", CurMount.name); end
			return; -- only summonable mount is already summoned
		end

		return;
	end

	
	-- only one mount to choose from
	if (#rgFiltered == 1) then
		local mount = rgFiltered[1];
		if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("%s is apparently only one summonable", mount.name); end
		return rgFiltered[1];
	end		

	-- multiple mounts
	local cRank = mrandom(1,cTotalRanks);
	if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("random roll from 1 to %i produced %i", cTotalRanks, cRank); end
	for _, mount in ipairs(rgFiltered) do
		cRank = cRank - mount.rank;
		if (DC.MOUNTS >= DL.AV) then Pokedex:Printf("%s's rank of %i brings total down to %i", mount.name, mount.rank, cRank); end
		if (cRank <= 0) then -- found our slot
			if (DC.MOUNTS >= DL.BASIC) then Pokedex:Printf("random selection has chosen %s", mount.name); end
			return mount;
		end
	end

	if (cRank > 0) then Pokedex:Print(L["ERROR: selection error"]); end
end

--=====================================================================--
-- MOUNT BUCKETS COLLECTION
--=====================================================================--

local metaBucketList = {}
metaBucketList.__index = metaBucketList

function metaBucketList:AddBucket(bucket)
	if bucket:IsNotEmpty() then
		tinsert(self, bucket)
	end
end

--[==[
function metaBucketList:AddMount(mount)
	for _,bucket in ipairs(self) do
		if bucket:MountQualifies(mount) then
			bucket.mounts[mount] = mount.name
		end
	end
end
--]==]

function metaBucketList:SelectMount(...)
	for _,bucket in ipairs(self) do
		local mount = bucket:SelectMount(...)
		if mount then 
			-- Pokedex:Printf("selected %s from %s", mount.name, bucket.name)
			return mount, bucket
		end
	end

	Pokedex:Print(L["ERROR: You have no summonable mounts."]);
end

function metaBucketList:Filter(flagsRequired)
--	print("bucket list filter called with flags", flagsRequired)
	local filteredList = setmetatable( {}, metaBucketList )
	if type(flagsRequired) == "number" then
		for order,bucket in ipairs(self) do
			if bucket.flagsSupported and band(bucket.flagsSupported, flagsRequired) == flagsRequired then
				if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("mount bucket %i %s selected", order, bucket.name); end
				tinsert(filteredList, bucket)
			else
				if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("mount bucket %i %s rejected %s", order, bucket.name, MFtoStr(band(bnot(bucket.flagsSupported), flagsRequired))); end
			end
		end
	end
	return filteredList
end


--=====================================================================--
-- MOUNTS COLLECTION CLASS DEFINITION
--=====================================================================--

local CMounts = { 
	numOwned = -1, count = 0, idLastSeen = 0, 
	specials = { 
			RunningWild = NewRunningWild(), 
			Tarecgosa = NewTarecgosa(),
			Moonfang = NewMoonfang(),
		 }, 
	Types = {                                --   name                     flagsIncluded                                  flagsSupported  --
			        ATVs = MakeNewMountBucket( L["Reliable Mounts"],       nil,                                       bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			WaterWalkers = MakeNewMountBucket(   "WaterWalkers",           nil,                                       bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			      Flyers = MakeNewMountBucket( L["Flyers"],                MF.Flyer,                                      0                                        ),
			     Runners = MakeNewMountBucket( L["Runners"],               MF.NoFlyZone_Legal,                            MF.NoFlyZone_Legal                       ),
			     Scarabs = MakeNewMountBucket( L["Qiraji Scarabs"],        MF.Ahn_Qiraj,                              bor(MF.NoFlyZone_Legal)                      ),
			     Vashjir = MakeNewMountBucket( L["Vashj'ir Seahorses"],    MF.Vashjir,                                bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			    Swimmers = MakeNewMountBucket( L["Swimmers"],              MF.Swimmer,                                bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			 Hydroplanes = MakeNewMountBucket(   "Hydroplanes",        bor(MF.Underwater_Legal, MF.Flyer),                MF.Underwater_Legal                      ),
			 Hydroponies = MakeNewMountBucket(   "Hydroponies",        bor(MF.Underwater_Legal, MF.NoFlyZone_Legal),  bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			    Sidecars = MakeNewMountBucket(   "Sidecars",               MF.Sidecar,                                bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			     Walkers = MakeNewMountBucket(   "Walkers",                MF.Walker,                                 bor(MF.NoFlyZone_Legal, MF.Underwater_Legal) ),
			Unidentified = MakeNewMountBucket( L["Unidentified Mounts"],   MF.Unknown,                                    nil                                      ),
			    Unusable = MakeNewMountBucket( L["Unusable Mounts"],       MF.Unusable,                                   nil                                      ),
		},
}

gv.Mounts = setmetatable( {}, CMounts )

-- metamethod ensures that when we want mount information that it is up to date
CMounts.__index = function(t,k)
	local fUpdated = false

	-- local numOwned = CMounts:GetNumMounts() + CMounts:GetNumSpecialMounts();
	-- if (CMounts.numOwned ~= numOwned) then
	if (CMounts.numOwned == -1) then
		-- if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("Building mount list from %i mounts to %i mounts for key access %s", CMounts.numOwned, numOwned, tostring(k)); end
		fUpdated = true
		CMounts:UpdateMounts()
	end
	return (k == "fUpdated") and fUpdated or CMounts[k]
end

-- metamethod ensures nothing is set on outer table which would prevent future 
-- lookups from triggering the __index metamethod
--[===[@debug@
CMounts.__newindex = function(t,k,v)
	Pokedex:Print("Pokedex mounts error - attempt to set values on outer table", k, v)
end
--@end-debug@]===]

-- this number is the same shown in the journal (MountJournal.numOwned), but we can't rely on trying to pull it from there
function CMounts:GetNumMounts()
	local numOwned = 0
	-- LEGION - we could cache and reuse the table returned by GetMountIDs - especially if its always the same table
	local mounts = cmj.GetMountIDs()
	for _, idMount in ipairs(mounts) do
		local _, _, _, _, _, _, _, _, faction, hideOnChar, isCollected = cmj.GetMountInfoByID(idMount)
		if isCollected and not hideOnChar and faction ~= gc.iOffFaction then
			numOwned = numOwned + 1
		end
	end
	-- Pokedex:Print("GetNumMounts returning:" .. numOwned)
	return numOwned
end

function CMounts:GetNumSpecialMounts()
	local count = 0
	for name,mount in pairs(self.specials) do
		if mount.fHas then
			count = count + 1
		end
	end
	-- Pokedex:Print("GetNumSpecialMounts returning:" .. count)
	return count
end

function CMounts:GetMounts()
	local fInitialized  = (self.count > 0)  -- count instead of numOwned because you can own unusable mounts and we want that to be same as owning nothing

	self.ByID    = { [0] = gc.nomount }  -- key(number) is spell ID    value(table)  is mount
	self.ByMID   = { [0] = gc.nomount }  -- key(number) is mount ID    value(table)  is mount
	self.ByName  = { [0] = gc.nomount }  -- key(string) is mount name  value(table)  is mount
	self.ByIndex = { [0] = gc.nomount }  -- key(number) is name order  value(table)  is mount

	-- find and add standard mounts
	local numOwned = 0
	local mountIDs = cmj.GetMountIDs()
	for _, idMount in ipairs(mountIDs) do
		local name, idSpell, icon, active, isUsable, sourceType, isFavorite, isFactionSpecific, faction, hideOnChar, isCollected = cmj.GetMountInfoByID(idMount)

		if isCollected and not hideOnChar and faction ~= gc.iOffFaction then
			if (not fInitialized) then
				if (name == nil) then
					-- TODO: retest scenario where cache has been deleted
					if (DC.MISC >= DL.BASIC) then Pokedex:Print("unable to initialize, missing mount names"); end
					return false;
				end

				-- sanity checks that unique ids really are, only do on first pass
				assert(self.ByName[name] == nil, "duplicate mount name found: "..name)
				assert(self.ByMID[idMount] == nil, "duplicate mount mountID found: "..idMount)
				assert(self.ByID[idSpell] == nil, "duplicate mount spellID found: "..idSpell)
			end

			local mount = { name = name, mid = idMount, id = idSpell, index = 1 }  -- , chance = 0 }
			mount.class = gc.rgClassMounts[idSpell]
			mount.skill = gc.rgSkillMounts[idSpell]
			mount.rawflags = Pokedex:GetMountAttributes(idMount)
			mount.flags = mount.rawflags -- setting a value here so we don't have to worry about __newindex later
			mount = setmetatable( mount, metaMount )

			self.ByID[idSpell] = mount
			self.ByMID[idMount] = mount
			self.ByName[name] = mount
			self.ByIndex[#(self.ByIndex) + 1] = mount
			numOwned = numOwned + 1
		end
	end

	sort(self.ByIndex, gs.ByName)
	for index, mount in ipairs(self.ByIndex) do
		mount.index = index
	end

	for key, mount in pairs(self.specials) do
		if mount.fHas then
			numOwned = numOwned + 1
			self.ByID[mount.id] = mount
			self.ByName[mount.name] = mount
		end
	end

	self.numOwned = numOwned
	-- Pokedex:Print("GetMounts has set numOwned to:" .. self.numOwned)

	-- if first pass this session, build known table
	if not fInitialized then
		local strknown = Pokedex.db.char.mounts.known
		local known = {}
		
		-- if first pass ever then build known out of ByID
		if (strknown == nil) then
			local strknown = ""
			for idSpell in pairs(self.ByID) do
				known[idSpell] = true
				strknown = format("%i %s", idSpell, strknown)
			end
			Pokedex.db.char.mounts.known = strknown
		else
			for idSpell in gmatch(strknown, "(%d+)") do
				known[tonumber(idSpell)] = true
			end
		end
		
		self.known = known
	end
end

function CMounts:UpdateMounts()
--	if (self.numOwned ~= (self:GetNumMounts() + self:GetNumSpecialMounts())) then
	if (self.numOwned == -1) then
		self:GetMounts()
	end

	local idCurSelection = gv.SelectedMount.id
	local mountActive
	local mountNew

	gv.HotMount = self.ByID[Pokedex.db.char.mounts.idHot]
	if not gv.HotMount then
		gv.HotMount = gc.nomount
		Pokedex.db.char.mounts.idHot = 0
	end

	self.count   = 0   -- number of usable mounts
	local HotList = { [0] = L["no hot mount"] }  -- key(int) is display order  value(string) is mount name
	self.idLastSeen = 0

	for k, bucket in pairs(self.Types) do
		bucket.mounts = {}
	end

	for id, mount in pairs(self.ByID) do
		if id ~= 0 then
			-- update flags - skills or settings (favor ground mounts) may have changed
			UpdateMountFlags(mount)
			local fUsableMount = not fband(mount.flags, MF.Unusable)

			-- place mounts in appropriate buckets (flyer, hydroplane, etc)
			for k, bucket in pairs(self.Types) do
				if bucket:MountQualifies(mount) then
					-- Pokedex:Printf("%s added to bucket %s", mount.name, k)
					bucket.mounts[mount] = mount.name
				end
			end

			if fUsableMount then
				self.count = self.count + 1
				tinsert(HotList, mount.name)

				-- ATV list built out of rawflags
				if band(mount.rawflags, MFS_ATV) == MFS_ATV then
					self.Types.ATVs.mounts[mount] = mount.name
				end
			end

			if mount.fActive then
				mountActive = mount
				self.idLastSeen = id
			end
			
			if not self.known[id] then 
				if (DC.MOUNTS >= DL.BASIC) then Pokedex:Print("new mount found", mount.name, mount.mid, id); end
				Pokedex.db.char.mounts.known = format("%i %s", id, Pokedex.db.char.mounts.known)
				self.known[id] = true
				mountNew = mount

				if fUsableMount and Pokedex.db.char.mounts.fEnableHotness and Pokedex.db.char.mounts.fNewHotness then
					Pokedex.db.char.mounts.idHot = id
					gv.HotMount = mount
				end
			end
		end
	end

	-- WaterWalker bucket
	for id in pairs(gc.rgWaterWalkingMounts) do
		local mount = self.ByID[id]
		if mount and not fband(mount.flags, MF.Unusable) then
			self.Types.WaterWalkers.mounts[mount] = mount.name
		end
	end


	self:MakeLists()

	-- sort the HotList alphabetically and find the index for the HotMount
	local hotName = gv.HotMount.name
	sort(HotList)
	for i=0, #HotList do
		if HotList[i] == hotName then 
			gv.HotMountIndex = i
			break
		end
	end
	self.HotList = HotList

	-- new mount > what's already selected > nomount if we have no summonable mounts > first
	self:SetSelectedMount(mountNew or mountActive or self.ByID[idCurSelection])   -- will set gv.SelectedMountBucket and gv.SelectedMount

	-- add in extra formatting if needed
	self:UpdateDisplayNames()
end


--[==[
	okay ... maybe we start to trust IsFlyableArea, NoFlyZone_Legal will be not IsFlyableArea ..
	but the flag for underwater legal will be set off of RedDrake type fFlyable
	
	Underwater_Legal  -- asked for when not flyable and isswimming         -- could be underwater or at surface of no fly zone ... those should be same for our purposes
	NoFlyZone_Legal   -- asked for when not flyable and not IsFlyableArea  -- unless fFlyable is true in which case shouldn't we believe that? so (not(fFlyable or IsFlyableArea))
--]==]

function CMounts:MakeLists()
	local types = self.Types
	local forms = gc.Player

	self.Lists = {}

	local Falling = setmetatable( {}, metaBucketList )
	Falling:AddBucket(forms.FlyingBroom)
	Falling:AddBucket(forms.FlightForm)
	Falling:AddBucket(forms.FlappingOwl1)
	Falling:AddBucket(forms.Glide)
	Falling:AddBucket(forms.ZenFlight)
	Falling:AddBucket(forms.Levitate)
	Falling:AddBucket(forms.SlowFall)
	Falling:AddBucket(forms.SkyhornKite)
	Falling:AddBucket(forms.GoblinGlider)
	Falling:AddBucket(forms.GoldenGlider)
	Falling:AddBucket(forms.GoblinGliderKit)
	Falling:AddBucket(forms.FlexweaveUnderlay)
	-- Falling:AddBucket(forms.ParachuteCloak)
	Falling:AddBucket(forms.SnowfallLager)
	Falling:AddBucket(forms.FlappingOwl2)
	Falling:AddBucket(forms.LegendaryDaggers)
	Falling:AddBucket(forms.HandOfProtection)
	Falling:AddBucket(forms.DivineShield)
	Falling:AddBucket(forms.CatForm)
	self.Lists.Falling = Falling



	-- AhnQiraj
	local AhnQiraj = setmetatable( {}, metaBucketList )
	AhnQiraj:AddBucket(forms.StagForm)
	AhnQiraj:AddBucket(forms.FlightForm)
	AhnQiraj:AddBucket(forms.MagicBroom)
	AhnQiraj:AddBucket(types.Flyers)
	AhnQiraj:AddBucket(types.Scarabs)
	AhnQiraj:AddBucket(forms.Ratstallion)
	AhnQiraj:AddBucket(forms.NagrandCorral)
	AhnQiraj:AddBucket(forms.HotRod)
	AhnQiraj:AddBucket(types.Runners)
	AhnQiraj:AddBucket(types.ATVs)
	AhnQiraj:AddBucket(types.Sidecars)
	AhnQiraj:AddBucket(forms.TravelForm)
	AhnQiraj:AddBucket(forms.CatForm)
	AhnQiraj:AddBucket(forms.GhostWolf)
	AhnQiraj:AddBucket(forms.BurningRush)
	AhnQiraj:AddBucket(forms.MonkRoll)
	AhnQiraj:AddBucket(types.Walkers)
	AhnQiraj:AddBucket(forms.BobbingBerg)
	AhnQiraj:AddBucket(forms.FishingRaft)
	self.Lists.AhnQiraj = AhnQiraj


	-- DryOrder - coping AhnQiraj but omit the scarabs; easy way to keep order in sync between the two lists
	local DryOrder = setmetatable( {}, metaBucketList )
	for _, bucket in ipairs(AhnQiraj) do
		if bucket ~= types.Scarabs then
			tinsert(DryOrder, bucket)
		end
	end
	self.Lists.DryOrder = DryOrder



	-- bucket order for Vashjir 
	-- WoD has messed us up because we can't figure out whether we're underwater or at surface.
	local Vashjir = setmetatable( {}, metaBucketList )
	Vashjir:AddBucket(types.Vashjir)
	-- LEGION - is Aquatic Form slower than Seahorse?
	Vashjir:AddBucket(forms.AquaticForm)
	Vashjir:AddBucket(types.Swimmers)
	Vashjir:AddBucket(forms.FlightForm)		-- skipped by Underwater_Legal or NoFlyZone_Legal
	Vashjir:AddBucket(forms.FlyingBroom)	-- skipped by Underwater_Legal or NoFlyZone_Legal
	Vashjir:AddBucket(types.Flyers)			-- skipped by Underwater_Legal or NoFlyZone_Legal
	Vashjir:AddBucket(types.Hydroplanes)	-- skipped by NoFlyZone_Legal
	Vashjir:AddBucket(types.WaterWalkers)
	Vashjir:AddBucket(forms.MagicBroom)
	Vashjir:AddBucket(types.Runners)		-- skipped by Underwater_Legal
	Vashjir:AddBucket(types.Hydroponies)
	Vashjir:AddBucket(types.ATVs)
	Vashjir:AddBucket(forms.MonkRoll)
	self.Lists.Vashjir = Vashjir



	-- bucket order when underwater - filtered by Underwater_Legal and NoFlyZone_Legal flags
	-- WoD broke reliable underwater detection with change to flying cloud. We'll now only select
	-- this list if breath timer is up which means water breathing will always make you look like
	-- you're at the surface.
	local WetOrder = setmetatable( {}, metaBucketList )
	WetOrder:AddBucket(forms.FlightForm)	-- skipped by Underwater_Legal or NoFlyZone_Legal
	WetOrder:AddBucket(forms.FlyingBroom)	-- skipped by Underwater_Legal or NoFlyZone_Legal
	WetOrder:AddBucket(types.Flyers)		-- skipped by Underwater_Legal or NoFlyZone_Legal
	WetOrder:AddBucket(forms.AquaticForm)
	WetOrder:AddBucket(types.Swimmers)
	WetOrder:AddBucket(types.Hydroplanes)	-- skipped by NoFlyZone_Legal
	WetOrder:AddBucket(types.WaterWalkers)
	WetOrder:AddBucket(forms.BobbingBerg)
	WetOrder:AddBucket(forms.FishingRaft)
	WetOrder:AddBucket(forms.MagicBroom)
	WetOrder:AddBucket(types.Runners)		-- skipped by Underwater_Legal
	WetOrder:AddBucket(types.Hydroponies)
	WetOrder:AddBucket(types.ATVs)
	WetOrder:AddBucket(forms.MonkRoll)
	self.Lists.WetOrder = WetOrder



	-- bucket order at the surface, differs from underwater by putting water walkers in front of swimmers
	local Surface = setmetatable( {}, metaBucketList )
	Surface:AddBucket(forms.FlightForm)		-- skipped by Underwater_Legal or NoFlyZone_Legal
	Surface:AddBucket(forms.FlyingBroom)	-- skipped by Underwater_Legal or NoFlyZone_Legal
	Surface:AddBucket(types.Flyers)			-- skipped by Underwater_Legal or NoFlyZone_Legal
	Surface:AddBucket(types.Hydroplanes)	-- skipped by NoFlyZone_Legal
	Surface:AddBucket(types.WaterWalkers)
	Surface:AddBucket(forms.BobbingBerg)
	Surface:AddBucket(forms.FishingRaft)
	Surface:AddBucket(forms.AquaticForm)
	Surface:AddBucket(types.Swimmers)
	Surface:AddBucket(forms.MagicBroom)
	Surface:AddBucket(types.Runners)		-- skipped by Underwater_Legal
	Surface:AddBucket(types.Hydroponies)
	Surface:AddBucket(types.ATVs)
	Surface:AddBucket(forms.GhostWolf)
	Surface:AddBucket(forms.MonkRoll)
	self.Lists.Surface = Surface


	
	-- TODO: ocean floor list to deal with buffs that give fast run speed on ocean floor?


	-- UI list - sets the order of 
	local UITypesOrder = setmetatable( {}, metaBucketList )
	UITypesOrder:AddBucket(types.Flyers)
	UITypesOrder:AddBucket(types.Runners)
	UITypesOrder:AddBucket(types.Swimmers)
	UITypesOrder:AddBucket(types.Vashjir)
	UITypesOrder:AddBucket(types.Scarabs)
	-- UITypesOrder:AddBucket(types.ATVs)
	UITypesOrder:AddBucket(types.Unidentified)
	UITypesOrder:AddBucket(types.Unusable)
	if #UITypesOrder == 0 then  -- if we have a no displayable buckets, then add placeholder
		UITypesOrder[0] = setmetatable( { mounts = { [gc.nomount] = L["no mounts available"] }, name = L["no mounts available"], flagsSupported = 0, flagsIncluded = 0 }, metaMountBucket )
	end
	self.Lists.UITypesOrder = UITypesOrder
	
	-- now create list that can be shown in dropdown
	self.Lists.UITypesDisplay = {}
	for order, bucket in pairs(UITypesOrder) do
		bucket.order = order
		self.Lists.UITypesDisplay[bucket] = bucket.name
	end
end

function CMounts:UpdateDisplayNames(changed)
	for _, type in pairs(self.Lists.UITypesOrder) do
		local mounts = type.mounts
		if changed then
			if mounts[changed] then
				mounts[changed] = gf.Mounts(changed)
			end
		else
			for mount in pairs(mounts) do
				mounts[mount] = gf.Mounts(mount)
			end
		end
	end
end

--[===[ -- chance - code that calculates percentage chance of summons for UI purposes
function CMounts:UpdateDisplayNames(changed)
	local totalRanks = 0
	local HotMountScaler = 100
	local mounts = gv.SelectedMountBucket.mounts

	-- PASS 1 - COLLECTION
	
	-- add up the ranks of all mounts in the bucket
	for mount in pairs(mounts) do
		totalRanks = totalRanks + mount.rank
	end

	-- remove ranks from total for a special mount you have, but can't actually use
	-- for example, have Tarecgosa staff but it is not equipped
	for _,mount in pairs(self.specials) do
		if mounts[mount] and not mount.fCan then
			totalRanks = totalRanks - mount.rank
		end
	end

	-- if hot pet is in set, remove those ranks and figure out scalar for all other mounts
	if Pokedex.db.char.mounts.fEnableHotness and mounts[gv.HotMount] then
		totalRanks = totalRanks - gv.HotMount.rank
		HotMountScaler = 100 - Pokedex.db.char.mounts.iHeat
	end

	-- multiplying a rank by this result gives the result as a percentage
	HotMountScaler = HotMountScaler / max(totalRanks,1)
	
	-- PASS 2 - DECORATION

	-- calculate chance of summons and set the display name for all mounts
	for mount in pairs(mounts) do
		mount.chance = mount.rank * HotMountScaler
		mounts[mount] = gf.Mounts(mount.name, mount.rank, mount.chance)
	end

	-- fix any specials that can't be summoned to have a 0% chance
	-- for example, have Tarecgosa staff but it is not equipped
	for _,mount in pairs(self.specials) do
		if mounts[mount] and not mount.fCan then
			mount.chance = 0
			mounts[mount] = gf.Mounts(mount.name, mount.rank, mount.chance)
		end
	end

	-- if hot pet is in set, decoreate its name and set chance to hot pet chance
	if Pokedex.db.char.mounts.fEnableHotness and mounts[gv.HotMount] then
		local mount = gv.HotMount
		mount.chance = Pokedex.db.char.mounts.iHeat
		local spiffyName = format(gf.strFormatHotName, mount.name)
		mounts[mount] = gf.Mounts(spiffyName, mount.rank, mount.chance)
	end

end	
--]===]  -- chance - code that calculates percentage chance of summons for UI purposes

-- Hooking Summon and Dismiss ensures we'll usually know what 
-- the current state is, but if the user dismisses the mount by clicking off 
-- the buff or casting a spell, we may not get the update, so always get/set 
-- current pet through these functions.

function CMounts:SetCurrent(mount)
	if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Print(format("Current mount set to: %i %s", mount.id, mount.name)); end
	CMounts.idLastSeen = mount.id;
	self:SetSelectedMount(mount);
end

function CMounts:GetCurrent()
	local mount = self.ByID[self.idLastSeen]
	if mount.fActive then
		return mount
	end

	CMounts.idLastSeen = 0
	if IsMounted() then
		for id, mount in pairs(self.ByID) do
			if mount.fActive then
				CMounts.idLastSeen = id
				return mount
			end
		end
	end

	return gc.nomount;
end

function CMounts:SetSelectedMount(mount)
	if #self.Lists.UITypesOrder == 0 then
		gv.SelectedMountBucket = self.Lists.UITypesOrder[0]
		gv.SelectedMount = gc.nomount
		return
	end
	
	if mount and mount ~= gc.nomount then
		for _,bucket in ipairs(self.Lists.UITypesOrder) do
			if bucket:MountInBucket(mount) then
				gv.SelectedMountBucket = bucket
				gv.SelectedMount = mount
				return
			end
		end
	end
	
	-- mount was invalid for some reason or not given
	-- select first mount from first bucket
	gv.SelectedMountBucket = self.Lists.UITypesOrder[1]
	gv.SelectedMount = gv.SelectedMountBucket:GetFirstMount()
end


function Pokedex:UpdateMountInfo(event)
	-- if (DC.MOUNTS >= DL.EXCEP) then Pokedex:Printf("updating Mount info for", event); end
	gc.Player:UpdatePlayerInfo()
	CMounts:UpdateMounts()
	-- gv.Mounts:UpdateMounts()
end

--=========================================================================--
-- Easiest way to know if we can summon specific type of mount is to just
-- check the spell for summoning a specific example of that type and to
-- see if the spell is castable. IsUsableSpell does not care whether you 
-- actually have access to that mount or not.
-- returns listname, flags
--=========================================================================--
function Pokedex:GetAreaInfo()
	--=====================================================================--
	-- The IsUsableSpell checks will "fail" if the player is in a shapeshift
	-- form and the autoUnshift behavior is turned off. Work around that by
	-- temporarily enabling autoUnshift.
	--=====================================================================--
	local fHandleCVar = (GetShapeshiftForm() > 0) and not GetCVarBool("autoUnshift")
	if fHandleCVar then SetCVar("autoUnshift", 1) end
	
	--=====================================================================--
	-- The IsUsableSpell check for a flying mount will fail for druids if 
	-- they're in flight form and Auto-Dismount in Flight has been turned 
	-- off. However, if we know that we're currently flying, then it has to 
	-- be a flyable area, right?
	--=====================================================================--
--	local fFlyable    = (IsUsableSpell(gc.idSpellRedDrake) or IsFlying());
--	local fOutdoors   = (IsUsableSpell(gc.idSpellRidingTurtle) and IsOutdoors())
	local fSwimming   = (IsUsableSpell(gc.idSpellAbyssalSeahorse) or IsSwimming());
	local fVashjir    = (IsUsableSpell(gc.idSpellAbyssalSeahorse));
	local fAhnQiraj   = (IsUsableSpell(gc.idSpellRedScarab));

	local fUnderwater = (0 ~= GetMirrorTimerProgress("BREATH"))  -- or (fSwimming and fOutdoors and (not IsUsableSpell(gc.idSpellRedFlyingCloud)));
	local fWading = (IsUsableSpell(gc.idSpellSubduedSeahorse) and not fSwimming)

	local flagMoving  = (GetUnitSpeed("player") > 0) and MF.Moving_Legal or 0
	local flagIndoors = not IsOutdoors() and MF.Indoors_Legal or 0
	local flagCombat  =  InCombatLockdown() and MF.Combat_Legal or 0
	local flagUnderwater = fUnderwater and MF.Underwater_Legal or 0

	-- We were summoning off flying list for people who didn't have flying skill yet. I think the best
	-- place to handle that is here rather than futzing with the bucketing or bucket selection. If you
	-- don't have the riding skill to fly, we will always consider you to be in a no fly zone.
	local flagNoFlyZone = (gv.Skills.Riding < SL.Expert or (not IsFlying() and not IsFlyableArea())) and MF.NoFlyZone_Legal or 0
--	local flagNoFlyZone = (not fFlyable and not IsFlyableArea()) and MF.NoFlyZone_Legal or 0	-- no longer have a flyer that can't be used in ground zones as a way of checking

	if fHandleCVar then SetCVar("autoUnshift", 0) end

	--=====================================================================--
	-- Information returned by Blizzard is sometimes wrong depending on the
	-- mount and/or zone. Do fix ups based on current location.
	--=====================================================================--

	SetMapToCurrentZone();
	local continentID = GetCurrentMapContinent()
	local mapID = GetCurrentMapAreaID();

	-- Abyssal Seahorse incorrectly returns as usable outside of Vashjir.
	if (fVashjir) then
		fVashjir = mapID == 614 or -- Abyssal Depths
		           mapID == 610 or -- Kelp'thar Forest
		           mapID == 615;   -- Shimmering Expanse
	end

	-- IsFlyableArea incorrectly returns true inside some instances and scenarios
	if (flagNoFlyZone == 0 and
		(mapID == 773  or  -- Throne of the Four Winds
		 mapID == 874  or  -- Scarlet Monastery
		 mapID == 882  or  -- Unga Ingoo
		 mapID == 914  or  -- Dagger in the Dark
		 mapID == 970  or  -- Assault on the Dark Portal
		 mapID == 1022 or  -- Helheim
		 mapID == 1031 or  -- The Shattered Edge
		 mapID == 1035 or  -- Skyhold
		 mapID == 1044 or  -- The Wandering Isle
		 mapID == 1050 or  -- Dreadscar Rift
		 mapID == 1057 or  -- The Heart of Azeroth
		 mapID == 1078 or  -- Niskara
		 mapID == 1086 or  -- Malorne's Nightmare
		 mapID == 1114 or  -- Haustvald (Trial of Valor)
		 mapID == 1137 or  -- The Deadmines
		 mapID == 1174 or  -- Azuremyst Isle
		 mapID == 1188 or  -- Antorus, Broken Cliffs
		 mapID == 1198 or  -- Greater Invasion Point: Matron Folnuna
		 mapID == 1216 )   -- Telorgus Rift
		) then
		flagNoFlyZone = MF.NoFlyZone_Legal
	end


	local flags = bor(flagNoFlyZone, flagUnderwater, flagMoving, flagIndoors, flagCombat)

	if IsFalling() then
		return "Falling", flags
	elseif fSwimming then
		if fVashjir then
			return "Vashjir", flags
		elseif not fUnderwater then 
			return "Surface", flags
		else
			return "WetOrder", flags
		end
	else
		if fAhnQiraj then
			return "AhnQiraj", flags
		elseif fWading then
			return "Surface", flags
		else
			return "DryOrder", flags
		end
	end
end

--=========================================================================--
--=========================================================================--
--
-- PLAYER CLASS FUNCTIONS
--
--=========================================================================--
--=========================================================================--

local CPlayer = {}
gc.Player = setmetatable( CPlayer, { 
		__index = function(t,k)
			setmetatable(t,nil)
			
			local _, strClass = UnitClass("player");
			t.fDeathKnight = (strClass == "DEATHKNIGHT");
			t.fDemonHunter = (strClass == "DEMONHUNTER")
			t.fDruid       = (strClass == "DRUID");
			t.fHunter      = (strClass == "HUNTER");
			t.fMage        = (strClass == "MAGE");
			t.fMonk        = (strClass == "MONK");
			t.fPaladin     = (strClass == "PALADIN");
			t.fPriest      = (strClass == "PRIEST");
			t.fRogue       = (strClass == "ROGUE");
			t.fShaman      = (strClass == "SHAMAN");
			t.fWarlock     = (strClass == "WARLOCK");
			t.fWarrior     = (strClass == "WARRIOR");


			local MissingForm = setmetatable( { mount = { fHas = false } }, metaFormBucket )
			-- odd mounts
			t.FlyingBroom = MissingForm
			t.MagicBroom  = MissingForm
			t.HotRod      = MissingForm

			-- "mount" forms
			t.GhostWolf     = MissingForm
			t.MonkRoll      = MissingForm
			t.BurningRush   = MissingForm
			t.StagForm      = MissingForm
			t.FlightForm    = MissingForm
			t.AquaticForm   = MissingForm
			t.TravelForm    = MissingForm
			t.CatForm       = MissingForm
			t.BobbingBerg   = MissingForm
			t.FishingRaft   = MissingForm
			t.NagrandCorral = MissingForm
			t.Ratstallion   = MissingFrom

			-- fall protections
			t.DivineShield      = MissingForm
			t.FlappingOwl1      = MissingForm
			t.FlappingOwl2      = MissingForm
			t.FlexweaveUnderlay = MissingForm
			t.Glide             = MissingForm
			t.GoblinGlider      = MissingForm
			t.GoldenGlider      = MissingForm
			t.GoblinGliderKit   = MissingForm
			t.HandOfProtection  = MissingForm
			t.LegendaryDaggers  = MissingForm
			t.Levitate          = MissingForm
			t.ParachuteCloak    = MissingForm
			t.SkyhornKite       = MissingForm
			t.SlowFall          = MissingForm
			t.SnowfallLager     = MissingForm
			t.ZenFlight         = MissingForm

			t.getters = {}
			t.__index = function(tab,k)
					local func = tab.getters[k]
					if type(func) == "function" then
						return func(tab)
					end
					return nil
				end

			t.strCombatMacro = "/Pokedex DismissMount"
			t.strCastOnMount = ""  -- only prepended to mounts that don't use GCD -- REVIEW should we get a "regular mount" that needs a gcd then we'll have a problem working the /cancelform in right
			-- strCastOnDismount = function(self) return "" end,  -- is this needed or is always going to already be inside of Combat Macro and OnMount?
			t:UpdatePlayerInfo()
			
			setmetatable(t,t)
			return t[k];
		end,
	} );

function CPlayer:UpdatePlayerInfo()
	local MB = NewMagicBroom()
	self.FlyingBroom = setmetatable( { mount = MB, name = MB.name, flagsSupported = MF.Moving_Legal }, metaFormBucket )
	self.MagicBroom  = setmetatable( { mount = MB, name = MB.name, flagsSupported = MB.flags }, metaFormBucket )

	local HR = NewHotRod()
	self.HotRod = setmetatable( { mount = HR, name = HR.name, flagsSupported = HR.flags }, metaFormBucket )


	-- print("updating class info")
	local CastWhenFalling = Pokedex.db.profile.CastWhenFalling
	local CastWithMount = Pokedex.db.profile.CastWithMount
	local UseAsMount = Pokedex.db.profile.UseAsMount

	local get_rank = function(t) return Pokedex.db.profile.mounts.iDefaultRank end
	local get_fActive = function(t) return UnitBuff("player", t.name) end

	local metaFormShared = {
		__lt = gs.pfMounts,
		__index = function(t,k)
				local getter = t.getters[k]
				if (type(getter) == "function") then
					return getter(t,k)
				elseif (type(getter) == "table") then
					return getter[k]
				else
					return getter
				end
			end,
		__newindex = function(t,k,v)
				Pokedex:Print("unexpected table assignment", t.name, k, v)
			end,
	}			
	
	--=====================================================================--
	-- DEATH KNIGHT CLASS DEFINITIONS
	--=====================================================================--
	if self.fDeathKnight then
		if CastWithMount.DKPathOfFrost and IsSpellKnown(gc.idSpellPathOfFrost) then
			self.strCastOnMount = format("/cast %s\n", GetSpellInfo(gc.idSpellPathOfFrost))
		end
	end

	--=====================================================================--
	-- DEMON HUNTER CLASS DEFINITIONS
	--=====================================================================--
	if self.fDemonHunter then
		-- create the Glide Mount
		local strGlide = GetSpellInfo(gc.idSpellGlide) or "Glide"
		local Glide = setmetatable( { 
				id = gc.idSpellGlide,
				name = strGlide,
				fHas = CastWhenFalling.DemonGlide,
				fCan = true,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/dismount\n/cast %s\n", strGlide),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create the Glide  Bucket
		self.Glide = MakeNewFormBucket(Glide)
	end

	--=====================================================================--
	-- DRUID CLASS DEFINITIONS
	--=====================================================================--
	if self.fDruid then
		local fCenarionWard = CastWithMount.DruidCenarionWard and IsSpellKnown(gc.idSpellCenarionWard)
		local strCenarionWard = GetSpellInfo(gc.idSpellCenarionWard)

		-- find out which forms we have, stance numbers vary depending on what forms are known
		local siBear, siCat, siMoonkin, siStag, siTravel, siTreant
		for i=1, GetNumShapeshiftForms() do
			local _, name, _, _, spellId = GetShapeshiftFormInfo(i)
			if spellId == gc.idSpellBearForm    then siBear    = i elseif
			   spellId == gc.idSpellCatForm     then siCat     = i elseif
			   spellId == gc.idSpellFoNMoonkin  then siMoonkin = i elseif
			   spellId == gc.idSpellMoonkinForm then siMoonkin = i elseif
			   spellId == gc.idSpellStagForm    then siStag    = i elseif
			   spellId == gc.idSpellTravelForm  then siTravel  = i elseif
			   spellId == gc.idSpellTreantForm  then siTreant  = i 
			 end
			
		end

		-- spell names for use in macros
		local strCatForm      = GetSpellInfo(gc.idSpellCatForm);
		local strStagForm     = GetSpellInfo(gc.idSpellStagForm);
		local strTravelForm   = GetSpellInfo(gc.idSpellTravelForm);
		local strTreantForm   = GetSpellInfo(gc.idSpellTreantForm);

		local siGround = siTravel or siCat;
		local strGroundForm = siTravel and strTravelForm or strCatForm;

		-- strCastOnMount won't be called if we will be using a form, so it will only go
		-- into macro when getting on a regular mount. Use that to cancel out of any forms,
		-- like cat or bear, but let them stay in Moonkin form.
		-- /cancelform isn't working when in treant form which probably has to do with 
		-- the fact that in macros [stance:5] is false and [stance:0] is true :(
		self.strCastOnMount = format("%s/cancelform%s\n%s", 
			siTreant and format("/cancelaura %s\n", strTreantForm) or "",
			siMoonkin and format(" [nostance:%i]", siMoonkin) or "",
			fCenarionWard and format("/cast [nomounted, @player] %s\n", strCenarionWard) or "");

			-- this is the template used when a form is selected as the mount to be summoned
		local strMacroFormat = format("%s%s%s", 
			"/Pokedex DismissMount\n",
			"/cancelform [nostance:%i]\n",
			"/cast %s\n");

		-- create Stag Form Mount
		local StagForm = setmetatable( { 
				id = gc.idSpellStagForm,
				index = siStag,
				name = strStagForm,
				fHas = UseAsMount.DruidForms and siStag ~= nil,
				fOnGCD = true,
				flags =  bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format(strMacroFormat, siStag, strStagForm),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return select(3, GetShapeshiftFormInfo(t.index)) end,
				},
			}, metaFormShared )
		-- create Stag Form Bucket
		self.StagForm = MakeNewFormBucket(StagForm)

		-- create Flight Form Mount
		local FlightForm = setmetatable( { 
				id = idTravelForm,
				name = "Flight Form",
				fCan = true,
				fHas = UseAsMount.DruidForms and gv.Skills.Riding > SL.Journeyman, -- REVIEW - believe that player gets flight form when they learn flying riding skill)
				fOnGCD = true,
				flags = MF.Moving_Legal,
				strMacro = format(strMacroFormat, siTravel, strTravelForm),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create Flight Form Bucket
		self.FlightForm = MakeNewFormBucket(FlightForm)

		-- create Aquatic Form Mount
		local AquaticForm = setmetatable( { 
				id = gc.idSpellTravelForm,
				name = "Aquatic Form",
				fHas = UseAsMount.DruidForms and UnitLevel("player") > 17, -- REVIEW - believe that player gets fully function swim form at level 18
				fOnGCD = true,
				flags = bor(MF.Swimmer, MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format(strMacroFormat, siTravel, strTravelForm),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return IsSwimming() end,
				},
			}, metaFormShared )
		-- create Aquatic Form Bucket
		self.AquaticForm = MakeNewFormBucket(AquaticForm)
		-- LEGION - self.AquaticForm.glyphed = HasMinorGlyph(gc.idSpellGlyphOfAquaticForm)

		-- create Travel Form Mount
		local TravelForm = setmetatable( { 
				id = gc.idSpellTravelForm,
				name = strTravelForm,
				fHas = UseAsMount.DruidForms and siTravel ~= nil,
				fCan = true,
				fOnGCD = true,
				flags =  bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format(strMacroFormat, siTravel, strTravelForm),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create Travel Form Bucket
		self.TravelForm = MakeNewFormBucket(TravelForm)

		-- create Cat Form Mount
		local CatForm = setmetatable( { 
				id = gc.idSpellCatForm,
				name = strCatForm,
				fHas = UseAsMount.DruidForms and siCat ~= nil,
				fCan = true,
				fOnGCD = true,
				flags =  bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format(strMacroFormat, siCat, strCatForm),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create Cat Form Bucket
		self.CatForm = MakeNewFormBucket(CatForm)


		--=====================================================================--
		-- build big macro for use in combat
		--=====================================================================--
		local strCombatCancel = format("%s%s", 
			siCat     and format("[nostance:%i,noswimming,indoors]", siCat) or "",
			siGround  and format("[nostance:%i,noindoors,nomounted]", siGround) or "");

		local strCombatCast = format("%s%s", 
			siCat     and format("[indoors,noswimming]%s; ", strCatForm) or "",
--			siGround  and format("[%snomounted] %s", "", strGroundForm) or "");
			siGround  and format("[nomounted] %s", strGroundForm) or "");

		self.strCombatMacro = format("%s%s/Pokedex DismissMount",
			(strCombatCancel ~= "") and format("/cancelform %s\n", strCombatCancel) or "",
			(strCombatCast ~= "") and format("/cast %s\n", strCombatCast) or "");


		--=====================================================================--
		-- Glyph of the Flapping Owl
		--=====================================================================--
		if siMoonkin and IsSpellKnown(gc.idSpellFlap) then
			-- flapping owl 1 is for when you are already in moonkin form
			-- flapping owl 2 can put you into moonkin and flap with two presses
			local strFlap = GetSpellInfo(gc.idSpellFlap)
			local FlappingOwl1 = setmetatable( { 
					id = gc.idSpellFlap,
					name = strFlap,
					fHas = CastWhenFalling.DruidFlap,
					fOnGCD = true,
					flags =  bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal),
					strMacro = format("/cast [stance:%i] %s\n", siMoonkin, strFlap),
					getters = {
						rank = get_rank,
						fActive = get_fActive,
						fCan = function(t) local formId = GetShapeshiftFormID(); return formId == 31 or formId == 35 end,
					},
				}, metaFormShared )
			local FlappingOwl2 = setmetatable( { 
					id = gc.idSpellFlap,
					name = strFlap,
					fHas = CastWhenFalling.DruidFlap,
					fCan = true,
					fOnGCD = true,
					flags =  bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal),
					strMacro = format("%s%s%s", 
						format("/cancelform [nostance:%i]\n", siMoonkin),
						format("/cast [nostance:%i] %s\n", siMoonkin, GetSpellInfo(gc.idSpellMoonkinForm)),
						format("/cast [stance:%i] %s\n", siMoonkin, strFlap)),
					getters = {
						rank = get_rank,
						fActive = get_fActive,
					},
				}, metaFormShared )
			-- create Flapping Owl Bucket
			self.FlappingOwl1 = MakeNewFormBucket(FlappingOwl1)
			self.FlappingOwl2 = MakeNewFormBucket(FlappingOwl2)
		end
	end

	--=====================================================================--
	-- MAGE CLASS DEFINITIONS
	--=====================================================================--
	if self.fMage then
		local specID = GetSpecialization()
		if specID == 1 then
			local strCastMacro = format("/cast [nomounted, @player] %s\n", GetSpellInfo(gc.idSpellPrismaticBarrier))
			self.getters.strCastOnMount = function(t) return (GetSpellCooldown(gc.idSpellPrismaticBarrier) == 0) and strCastMacro or "" end
			self.strCastOnMount = not (CastWithMount.MagePrismaticBarrier and IsSpellKnown(gc.idSpellPrismaticBarrier)) and "" or nil
		elseif specID == 2 then
			local strCastMacro = format("/cast [nomounted, @player] %s\n", GetSpellInfo(gc.idSpellBlazingBarrier))
			self.getters.strCastOnMount = function(t) return (GetSpellCooldown(gc.idSpellBlazingBarrier) == 0) and strCastMacro or "" end
			self.strCastOnMount = not (CastWithMount.MageBlazingBarrier and IsSpellKnown(gc.idSpellBlazingBarrier)) and "" or nil
		elseif specID == 3 then
			local strCastMacro = format("/cast [nomounted, @player] %s\n", GetSpellInfo(gc.idSpellIceBarrier))
			self.getters.strCastOnMount = function(t) return (GetSpellCooldown(gc.idSpellIceBarrier) == 0) and strCastMacro or "" end
			self.strCastOnMount = not (CastWithMount.MageIceBarrier and IsSpellKnown(gc.idSpellIceBarrier)) and "" or nil
		end

		-- create the Slow Fall Mount
		local strSlowFall = GetSpellInfo(gc.idSpellSlowFall) or "Slow Fall"
		local SlowFall = setmetatable( { 
				id = gc.idSpellSlowFall,
				name = strSlowFall,
				fHas = CastWhenFalling.MageSlowFall and IsSpellKnown(gc.idSpellSlowFall),
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/cast [@player] %s\n", strSlowFall),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return IsUsableSpell(t.id) end,
				},
			}, metaFormShared )
		-- create the Slow Fall Bucket
		self.SlowFall = MakeNewFormBucket(SlowFall)
	end

	--=====================================================================--
	-- MONK CLASS DEFINITIONS
	--=====================================================================--
	if self.fMonk then
		-- create the Zen Flight Mount
		local strZenFlight = GetSpellInfo(gc.idSpellZenFlight)
		local ZenFlight = setmetatable( { 
				id = gc.idSpellZenFlight,
				name = strZenFlight,
				fHas = CastWhenFalling.MonkZenFlight and IsSpellKnown(gc.idSpellZenFlight),
				fOnGCD = true,
				flags = MF.Moving_Legal,
				strMacro = format("/cast %s\n", strZenFlight),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return IsUsableSpell(t.id) end,
				},
			}, metaFormShared )
		-- create the Zen Flight Bucket
		self.ZenFlight = MakeNewFormBucket(ZenFlight)

		-- create the Monk Roll Mount
		local strMonkRoll = GetSpellInfo(gc.idSpellRoll)
		local MonkRoll = setmetatable( { 
				id = gc.idSpellMonRoll,
				name = strMonkRoll,
				fHas = IsSpellKnown(gc.idSpellRoll),
				fCan = UseAsMount.MonkRoll,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Indoors_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format("/cast [nomounted] %s\n", strMonkRoll),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create the Monk Roll Bucket
		self.MonkRoll = MakeNewFormBucket(MonkRoll)

		-- create combat macro
		-- in combat we can roll, de-cloud or dismount
		self.strCombatMacro = format("%s%s/Pokedex DismissMount",
			(MonkRoll.fHas and MonkRoll.fCan) and MonkRoll.strMacro or "",
			format("/cancelaura %s\n", strZenFlight))
	end

	--=====================================================================--
	-- PALADIN CLASS DEFINITIONS
	--=====================================================================--
	if self.fPaladin then
		local strForbearance = GetSpellInfo(gc.idSpellForbearance) or "Forbearance"

		-- create the Divine Shield Mount
		local strDivineShield = GetSpellInfo(gc.idSpellDivineShield) or "Divine Shield"
		local DivineShield = setmetatable( { 
				id = gc.idSpellDivineShield,
				name = strDivineShield,
				fHas = CastWhenFalling.PaladinDivineShield and IsSpellKnown(gc.idSpellDivineShield),
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/cast [@player] %s\n", strDivineShield),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return 0 == GetSpellCooldown(t.id) and nil == UnitDebuff("player", strForbearance) end,
				},
			}, metaFormShared )
		-- create the Divine Shield Bucket
		self.DivineShield = MakeNewFormBucket(DivineShield)

		-- create the Hand of Protection Mount
		local strHandOfProtection = GetSpellInfo(gc.idSpellHandOfProtection) or "Blessing of Protection"
		local HandOfProtection = setmetatable( { 
				id = gc.idSpellHandOfProtection,
				name = strHandOfProtection,
				fHas = CastWhenFalling.PaladinHandOfProtection and IsSpellKnown(gc.idSpellHandOfProtection),
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/cast [@player] %s\n", strHandOfProtection),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return 0 == GetSpellCooldown(t.id) and nil == UnitDebuff("player", strForbearance) end,
				},
			}, metaFormShared )
		-- create the Hand of Protection Bucket
		self.HandOfProtection = MakeNewFormBucket(HandOfProtection)
	end

	--=====================================================================--
	-- PRIEST CLASS DEFINITIONS
	--=====================================================================--
	if self.fPriest then
		-- create the Levitate Mount
		local strLevitate = GetSpellInfo(gc.idSpellLevitate) or "Levitate"
		local Levitate = setmetatable( { 
				id = gc.idSpellLevitate,
				name = strLevitate,
				fHas = CastWhenFalling.PriestLevitate and IsSpellKnown(gc.idSpellLevitate),
				fCan = true,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/cast [@player] %s\n", strLevitate),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create the Levitate Bucket
		self.Levitate = MakeNewFormBucket(Levitate)
	end

	--=====================================================================--
	-- ROGUE CLASS DEFINITIONS
	--=====================================================================--
	if self.fRogue then
		local strCastMacro = format("/cancelaura %s\n", GetSpellInfo(gc.idSpellDisguise))
		self.strCastOnMount = CastWithMount.RogueRemoveDisguise and strCastMacro or ""

		-- create the Legendary Daggers Mount
		if GetItemCount(gc.idItemOHLegendaryDagger, true) > 0 or GetItemCount(gc.idItemMHLegendaryDagger, true) > 0 then
			-- TODO - when a user first acquires the daggers we won't create the falling object for it
			-- until their next reload ui. Eventual fix is item tracking infrastructure.
			local LegendaryDaggers = setmetatable( { 
					id = gc.idSpellLegendaryDaggers,
					name = GetSpellInfo(gc.idSpellLegendaryDaggers) or "Embrace of the Destroyer",
					mhand = { iid = gc.idItemMHLegendaryDagger, name = GetItemInfo(gc.idItemMHLegendaryDagger) or "Golad, Twilight of Aspects", slot = 16 },
					ohand = { iid = gc.idItemOHLegendaryDagger, name = GetItemInfo(gc.idItemOHLegendaryDagger) or "Tiriosh, Nightmare of Ages", slot = 17 },
					fHas = CastWhenFalling.RogueLegendaryDaggers, -- TODO - we should have a monitored items list like monitored spell, then can use inventory count for this test
					fOnGCD = true,
					flags = bor(MF.NoFlyZone_Legal, MF.Moving_Legal),
					getters = {
						rank = get_rank,
						fActive = get_fActive,
						fCan = function(t) 
								if IsFalling() then
									return (GetItemCount(t.ohand.iid, false) > 0 and GetItemCooldown(t.ohand.iid) == 0) or
										   (GetItemCount(t.mhand.iid, false) > 0 and GetItemCooldown(t.mhand.iid) == 0)
								end
								return false
							end,
						strMacro = function(t)
								local daggers = OffhandHasWeapon() and { t.ohand, t.mhand } or { t.mhand, t.ohand }
								for _,dagger in ipairs(daggers) do
									if GetItemCount(dagger.iid, false) > 0 then
										if not IsEquippedItem(dagger.iid) then
											dagger.restoreLink = GetInventoryItemLink("player", dagger.slot)
											EquipItemByName(dagger.name)
										end
										return format("/use %s\n", dagger.name)
									end
								end
							end,
						strCastOnMount = function(t)
								return format("%s%s", 
									(IsEquippedItem(t.mhand.iid) and t.mhand.restoreLink) and format("/equipslot %i %s\n", t.mhand.slot, GetItemInfo(t.mhand.restoreLink)) or "", 
									(IsEquippedItem(t.ohand.iid) and t.ohand.restoreLink) and format("/equipslot %i %s\n", t.ohand.slot, GetItemInfo(t.ohand.restoreLink)) or "" )
							end,
					},
				}, metaFormShared )
			-- create the Legendary Daggers Bucket
			self.LegendaryDaggers = MakeNewFormBucket(LegendaryDaggers)

			-- turn CastOnMount into a getter function where we will swap legendary daggers back out
			self.strCastOnMount = nil
			self.getters.strCastOnMount = function(t)
				return format("%s%s", LegendaryDaggers.strCastOnMount, CastWithMount.RogueRemoveDisguise and strCastMacro or "")
			end
		end			
	end

	--=====================================================================--
	-- SHAMAN CLASS DEFINITIONS
	--=====================================================================--
	if self.fShaman then
		local fWaterWalk = CastWithMount.ShamanWaterWalking and IsSpellKnown(gc.idSpellWaterWalking)
		local strGhostWolf = GetSpellInfo(gc.idSpellGhostWolf)
		local strWaterWalking = GetSpellInfo(gc.idSpellWaterWalking)

		-- create the Ghost Wolf Mount
		local GhostWolf = setmetatable( { 
				id = gc.idSpellGhostWolf,
				name = strGhostWolf,
				fHas = IsSpellKnown(gc.idSpellGhostWolf),
				fCan = UseAsMount.ShamanGhostWolf,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Indoors_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format("/stopcasting\n/cast [nomounted] %s\n/Pokedex DismissMount", strGhostWolf),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create the Ghost Wolf Bucket
		self.GhostWolf = MakeNewFormBucket(GhostWolf)

		if GhostWolf.fHas and GhostWolf.fCan then
			self.strCombatMacro = GhostWolf.strMacro
		end

		self.strCastOnMount = format("%s%s",
			(GhostWolf.fHas and GhostWolf.fCan) and "/cancelform\n" or "",
			fWaterWalk and format("/cast [nomounted, @player] %s\n", strWaterWalking) or "")
	end

	--=====================================================================--
	-- WARLOCK CLASS DEFINITIONS
	--=====================================================================--
	if self.fWarlock then
		-- LEGION - can warlock mounts run on water?
		local fMountsWaterWalk = false -- HasMinorGlyph(gc.idSpellGlyphOfNightmares)
		local strBurningRush = GetSpellInfo(gc.idSpellBurningRush) or "Burning Rush"
		local strMacroFormat = [==[
/cast [nomounted] %s
/cancelaura %s
/Pokedex DismissMount]==]

		-- when glyphed, warlock class mounts can walk on water
		gc.rgWaterWalkingMounts[23161] = fMountsWaterWalk or nil  -- Dreadsteed
		gc.rgWaterWalkingMounts[5784]  = fMountsWaterWalk or nil  -- Felsteed

		-- create the BurningRush Mount		
		local BurningRush = setmetatable( { 
				id = gc.idSpellBurningRush,
				name = strBurningRush,
				fHas = UseAsMount.WarlockBurningRush and IsSpellKnown(gc.idSpellBurningRush),
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Indoors_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format(strMacroFormat, strBurningRush, strBurningRush),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return not IsSwimming() end,
				},
			}, metaFormShared )

		-- create the BurningRush Bucket
		self.BurningRush = MakeNewFormBucket(BurningRush)
		
		if BurningRush.fHas and BurningRush.fCan then
			self.strCombatMacro = BurningRush.strMacro
			self.strCastOnMount = format("/cancelaura %s\n", strBurningRush)
		end
	end
	
	--=====================================================================--
	-- RATSTALLION HARNESS CLASS DEFINITION
	--=====================================================================--
	do
		local strSpellName = GetSpellInfo(gc.idSpellRatstallionHarness)
		local strItemName = GetItemInfo(gc.idItemRatstallionHarness) or "Ratstallion Harness"

		-- create the Ratstallion Mount
		local Ratstallion = setmetatable( { 
				id = gc.idSpellRatstallionHarness,
				iid = gc.idItemRatstallionHarness,
				name = strSpellName,
				fHas = UseAsMount.ItemRatstallionHarness, -- TODO - we should have a monitored items list like monitored spell, then can use inventory count for this test
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/use %s\n", strItemName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) 
							SetMapToCurrentZone()
							return GetItemCount(t.iid, false) > 0 and 8 == GetCurrentMapContinent() and 1014 == GetCurrentMapAreaID() and 11 == GetCurrentMapDungeonLevel()
						end,
				},
			}, metaFormShared )
		-- create the Ratstallion Bucket
		self.Ratstallion = MakeNewFormBucket(Ratstallion)
	end

	--=====================================================================--
	-- NAGRAND CORRAL CLASS DEFINITION
	--=====================================================================--
	do
		local idGarrison, strGarrison, idMount, strMount

		idGarrison = gc.idSpellGarrisonAbility
		strGarrison = GetSpellInfo(idGarrison) or "Garrison Ability"

		local faction = UnitFactionGroup("player")
		if "Alliance" == faction then
			idMount = gc.idSpellCorralAlliance
			strMount = GetSpellInfo(idMount) or "Telaari Talbuk"
		elseif "Horde" == faction then
			idMount = gc.idSpellCorralHorde
			strMount = GetSpellInfo(idMount) or "Frostwolf War Wolf"
		end

		if strGarrison and strMount then
			-- create the Nagrand Corral Mount
			local NagrandCorral = setmetatable( { 
					id = idGarrison,
					name = strMount,
					fHas = UseAsMount.NagrandCorral and IsSpellKnown(idGarrison), -- TODO - would like to find a better way of knowing if you made this out post choice or not
					fOnGCD = false,
					flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Moving_Legal, MF.Combat_Legal),
					strMacro = format("/cast %s\n", strGarrison),
					getters = {
						rank = get_rank,
						fActive = get_fActive,
						fCan = function(t) return GetSpellInfo(strGarrison) == t.name end,
					},
				}, metaFormShared )
			-- create the Nagrand Corral Bucket
			self.NagrandCorral = MakeNewFormBucket(NagrandCorral)
		end
	end

--=====================================================================--
-- CLASS DEFINITIONS FOR FISHING RAFT AND BOBBING BERG
--=====================================================================--

	if self.fMage then
		local strSpellName = GetSpellInfo(gc.idSpellBobbingBerg) or "Bipsi's Bobbing Berg"
		local strItemName = GetItemInfo(gc.idItemBobbingBerg) or "Bipsi's Bobbing Berg"

		-- create the Bipsi's Bobbing Berg
		local BobbingBerg = setmetatable( { 
				id = gc.idSpellBobbingBerg,
				iid = gc.idItemBobbingBerg,
				name = strSpellName,
				fHas = UseAsMount.ItemFishingRaft, -- TODO - we should have a monitored items list like monitored spell, then can use inventory count for this test
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/use %s\n", strItemName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return (GetItemCount(t.iid, false) > 0) and (GetItemCooldown(t.iid) == 0) end,
				},
			}, metaFormShared )
		-- create the Bobbing Berg Bucket
		self.BobbingBerg = MakeNewFormBucket(BobbingBerg)
	end

	do
		local strSpellName = GetSpellInfo(gc.idSpellFishingRaft) or "Anglers Fishing Raft"

		-- create the Angler's Fishing Raft
		local FishingRaft = setmetatable( { 
				id = gc.idSpellFishingRaft,
				iid = gc.idItemFishingRaft,
				name = strSpellName,
				fHas = UseAsMount.ItemFishingRaft and PlayerHasToy(gc.idItemFishingRaft) and select(3, GetFactionInfoByID(1302)) > 6,
				fCan = true,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/cast %s\n", strSpellName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
				},
			}, metaFormShared )
		-- create the Fishing Raft Bucket
		self.FishingRaft = MakeNewFormBucket(FishingRaft)
	end

	--=====================================================================--
	-- ROCFEATHER SKYHORN KITE
	--=====================================================================--
	do
		local strSpellName = GetSpellInfo(gc.idSpellSkyhornKite) or "Rocfeather Skyhorn Kite"

		-- create the Rocfeather Skyhorn Kite
		local SkyhornKite = setmetatable( { 
				id = gc.idSpellSkyhornKite,
				iid = gc.idItemSkyhornKite,
				name = strSpellName,
				fHas = CastWhenFalling.ItemSkyhornKite and PlayerHasToy(gc.idItemSkyhornKite),
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Underwater_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/cast %s\n", strSpellName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) 
							SetMapToCurrentZone();
							local continentID = GetCurrentMapContinent();
							if continentID == 8 then					
								local mapID = GetCurrentMapAreaID();
								local fDalaran = (mapID == 1014)
								local fGreatSea = (mapID == 1021) and (GetSubZoneText() == "")
								-- print("Skyhorn Kite", continentID, mapID, GetItemCooldown(t.iid))
								return not fDalaran and not fGreatSea and (GetItemCooldown(t.iid) == 0) 
							end
						end,
				},
			}, metaFormShared )
		-- create the Fishing Raft Bucket
		self.SkyhornKite = MakeNewFormBucket(SkyhornKite)
	end

	--=====================================================================--
	-- GOBLIN GLIDER KIT CLASS DEFINITION
	--=====================================================================--
	do
		local strSpellName = GetSpellInfo(gc.idSpellGoblinGliderKit)
		local strItemName = GetItemInfo(gc.idItemGoblinGliderKit) or "Goblin Glider Kit"

		-- create the Goblin Glider Mount
		local GoblinGliderKit = setmetatable( { 
				id = gc.idSpellGoblinGliderKit,
				iid = gc.idItemGoblinGliderKit,
				name = strSpellName,
				fHas = CastWhenFalling.ItemGoblinGliderKit, -- TODO - we should have a monitored items list like monitored spell, then can use inventory count for this test
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = format("/use %s\n", strItemName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return (not IsInInstance()) and (GetItemCount(t.iid, false) > 0) and (GetItemCooldown(t.iid) == 0) end,
				},
			}, metaFormShared )
		-- create the Goblin Glider Bucket
		self.GoblinGliderKit = MakeNewFormBucket(GoblinGliderKit)
	end

	--=====================================================================--
	-- GOLDEN GLIDER CLASS DEFINITION
	--=====================================================================--
	do
		local strSpellName = GetSpellInfo(gc.idSpellGoldenGlider)
		local strItemName = GetItemInfo(gc.idItemGoldenGlider) or "Golden Glider"

		-- create the Golden Glider Mount
		local GoldenGlider = setmetatable( { 
				id = gc.idSpellGoldenGlider,
				iid = gc.idItemGoldenGlider,
				name = strSpellName,
				fHas = CastWhenFalling.ItemGoldenGlider, -- TODO - we should have a monitored items list like monitored spell, then can use inventory count for this test
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format("/use %s\n", strItemName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return (GetCurrentMapAreaID() == 951) and (GetItemCount(t.iid, false) > 0) and (GetItemCooldown(t.iid) == 0) end,
				},
			}, metaFormShared )
		-- create the Golden Glider Bucket
		self.GoldenGlider = MakeNewFormBucket(GoldenGlider)
	end

	--=====================================================================--
	-- SNOWFALL LAGER CLASS DEFINITION
	--=====================================================================--
	do
		local strSpellName = GetSpellInfo(gc.idSpellSnowfallLager)
		local strItemName = GetItemInfo(gc.idItemSnowfallLager) or "Snowfall Lager"

		-- create the Snowfall Lager Mount
		local SnowfallLager = setmetatable( { 
				id = gc.idSpellSnowfallLager,
				iid = gc.idItemSnowfallLager,
				name = strSpellName,
				fHas = CastWhenFalling.ItemSnowfallLager, -- TODO - we should have a monitored items list like monitored spell, then can use inventory count for this test
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal),
				strMacro = format("/use %s\n", strItemName),
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return (GetCurrentMapAreaID() == 495) and (GetItemCount(t.iid, false) > 0) and (GetItemCooldown(t.iid) == 0) end,
				},
			}, metaFormShared )
		-- create the Snowfall Lager Bucket
		self.SnowfallLager = MakeNewFormBucket(SnowfallLager)
	end

	--=====================================================================--
	-- GOBLIN GLIDER CLASS DEFINITION
	--=====================================================================--
	if gv.Skills.Engineering >= 500 then
		-- create the Goblin Glider Mount
		local strSpellName = GetSpellInfo(gc.idSpellGoblinGlider) or "Goblin Glider"
		local GoblinGlider = setmetatable( { 
				id = gc.idSpellGoblinGlider,
				name = strSpellName,
				fHas = CastWhenFalling.EngrGoblinGlider,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = "/use 15\n",
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return not IsInInstance() and GetItemSpell(GetInventoryItemID("player", INVSLOT_BACK)) == strSpellName and GetInventoryItemCooldown("player", INVSLOT_BACK) == 0 end,
				},
			}, metaFormShared )
		-- create the Goblin Glider Bucket
		self.GoblinGlider = MakeNewFormBucket(GoblinGlider)
	end

	--=====================================================================--
	-- FLEXWEAVE UNDERLAY CLASS DEFINITION
	--=====================================================================--
	if gv.Skills.Engineering >= 350 then
		-- create the Flexweave Underlay Mount
		local strSpellName = GetSpellInfo(gc.idSpellFlexweaveUnderlay) or "Parachute"
		local FlexweaveUnderlay = setmetatable( { 
				id = gc.idSpellFlexweaveUnderlay,
				name = strSpellName,
				fHas = CastWhenFalling.EngrFlexweaveUnderlay and GetItemSpell(GetInventoryItemID("player", INVSLOT_BACK)) == strSpellName,
				fOnGCD = true,
				flags = bor(MF.NoFlyZone_Legal, MF.Combat_Legal, MF.Moving_Legal, MF.Indoors_Legal),
				strMacro = "/use 15\n",
				getters = {
					rank = get_rank,
					fActive = get_fActive,
					fCan = function(t) return not IsInInstance() and GetInventoryItemCooldown("player", INVSLOT_BACK) == 0 end,
				},
			}, metaFormShared )
		-- create the Flexweave Underlay Bucket
		self.FlexweaveUnderlay = MakeNewFormBucket(FlexweaveUnderlay)
	end
end


--=========================================================================--
--=========================================================================--
--
-- COMPANION FUNCTIONS
--
--=========================================================================--
--=========================================================================--

function Pokedex:ZONE_CHANGED_NEW_AREA(...)
	-- self:Printf("ZONE_CHANGED_NEW_AREA EVENT:  %s", self:StrFromVarArg(nil, ...));
	if (self.db.profile.fAutoSummonCompanion and self:GetCurrentPet() == gc.nopet and gv.Pets.count > 0) then
		if not (IsMounted() or UnitInVehicle("player") or UnitOnTaxi("player") or UnitIsDeadOrGhost("player")) then
			self:SummonCompanion()
		end
	end
end

function Pokedex:ToggleCompanion()
	if (self:GetCurrentPet() == gc.nopet) then
		self:SummonCompanion();
	else
		self:DismissCompanion();
	end
end

function Pokedex:DismissCompanion()
	local pid = cpj.GetSummonedPetGUID()
	if (pid) then cpj.SummonPetByGUID(pid) end
	self:SetSelectedPet(gc.nopet)
end

function Pokedex:SummonCompanion(fNext)
	local pet = self:SelectPet(fNext);
	if (pet ~= nil) then
		self:SummonSelectedPet(pet);
	end
end

function Pokedex:SummonSelectedPet(pet)
	assert(pet ~= nil)
	assert(#pet.pids > 0)
	local pidSummon, strName

	if (#pet.pids == 1) then
		pidSummon = pet.pids[1]
		local _, customName = cpj.GetPetInfoByPetID(pidSummon)
		strName = customName or pet.name
	else -- multiple copies of the pet, select the best
		local function petPicker(a,b)
			-- puts best at index 1
			if ((a.customName ~= nil) ~= (b.customName ~= nil)) then
				return a.customName ~= nil
			elseif (a.fave ~= b.fave) then
				return a.fave
			else
				return a.level > b.level
			end
		end

		local pets = {}
		for i,pid in ipairs(pet.pids) do
			local _, customName, level = cpj.GetPetInfoByPetID(pid)
			local fave = cpj.PetIsFavorite(pid)
			pets[i] = { pid = pid, customName = customName, fave = fave, level = level }
		end
		sort(pets, petPicker)
		pidSummon = pets[1].pid
		strName = pets[1].customName or pet.name
	end

	if (not UnitAffectingCombat("player")) then
		cpj.SummonPetByGUID(pidSummon)
		self:AnnounceSummon(strName);
		self:SetSelectedPet(pet)
		LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
	end		
end

function Pokedex:GetCurrentPet()
	local pid = cpj.GetSummonedPetGUID() or 0
	return gv.Pets.ByPID[pid]
end

function Pokedex:SetSelectedPet(pet)
	pet = pet or gc.nopet
	if (gv.Pets.All[pet] ~= nil) then
		-- nopet is only found in All when we actually own 0 pets
		gv.SelectedPet = pet
	elseif (gv.Pets.count > 0) then
		-- Should hit this case when we own pets, but there wasn't one that
		-- called out to be set for ranking. In that case, take the first.
		gv.SelectedPet = gv.Pets.ByIndex[1]
	else
		-- should never actually get here
		assert(false, pet.name)
		gv.SelectedPet = pet
	end
end

function Pokedex:SelectPet(fNext)
--[==[
	local isUsable, notEnoughMana = IsUsableSpell(gc.idSpellBattlePet)
	if (not isUsable or notEnoughMana) then
		if (DC.PETS >= DL.BASIC) then self:Print("IsUsableSpell failed for pets", isUsable, notEnoughMana); end
	end
--]==]
	if (GetSpellCooldown(gc.idSpellBattlePet) ~= 0) then
		if (DC.PETS >= DL.BASIC) then self:Print("in cooldown, ignore this summons attempt"); end
		return
	end

	local CurPet = self:GetCurrentPet();
	local rgPets = gv.Pets.ByIndex

	local fHasSnowballs = true;  -- used to check inventory for snowballs, this could be changed to be a calender check for Winter Veil
	local rgFiltered = {};
	local cTotalRanks = 0;
	local fSkippedCur = false;
	local fSkippedHot = false;

	--=====================================================================--
	-- FilterCompanions -- Local function that builds list of pets elligible
	-- for summoning as well as providing a sum of the ranks of those pets.
	-- Best done as seperate function to support different ways of calling,
	-- next pet versus random pet, and as local function we can reference 
	-- and modify variables in the scope of the calling function.
	-- NOTE: appends to rgFiltered, so multiple calls can build bigger list
	--=====================================================================--
	local function FilterCompanions(iStart, iEnd, fFirstOnly)
		if (iStart == 0 or iStart > iEnd) then
			if (DC.PETS >= DL.EXCEP) then self:Print("invalid search range"); end
			return;
		end 

		if (DC.PETS >= DL.AV) then self:Print("building list of eligible pets ..."); end
		for iPet = iStart, iEnd, 1 do
			local pet = rgPets[iPet]

			-- if its ranked as 0, then it gets skipped
			if (pet.rank == 0) then
				if (DC.PETS >= DL.AV) then self:Printf("%s skipped for having rank of 0", pet.name); end
			-- if we don't have snowballs and this pet requires one, then it is skipped
			elseif (not fHasSnowballs and gc.rgNeedsSnowball[pet.id]) then
				if (DC.PETS >= DL.AV) then self:Printf("%s skipped for lack of snowballs", pet.name); end
			-- if its the current companion, then it is skipped
			elseif (pet == CurPet) then
				if (DC.PETS >= DL.AV) then self:Printf("%s skipped for being current companion", pet.name); end
				fSkippedCur = true;
			-- if looking for first good value, then this is it
			elseif (fFirstOnly) then
				tinsert(rgFiltered, pet);
				cTotalRanks = cTotalRanks + pet.rank;
				if (DC.PETS >= DL.BASIC) then self:Printf("%s is Next eligible pet", pet.name); end
				return;
			-- if hot pet and hotness is enabled then don't add to list, it will be handled seperately
			elseif (self.db.char.pets.fEnableHotness and pet == gv.HotPet) then
				if (DC.PETS >= DL.AV) then self:Printf("%s skipped for being hot companion", pet.name); end
				fSkippedHot = true;
			-- else its put into the pool of summonables
			else
				tinsert(rgFiltered, pet);
				cTotalRanks = cTotalRanks + pet.rank;
				if (DC.PETS >= DL.AV) then self:Printf("%s added to list of summonable pets with a rank of %i. Total rank count is %i", pet.name, pet.rank, cTotalRanks); end
			end
		end

		if (DC.PETS >= DL.BASIC) then self:Printf("list built: %i pets with a total rank of %i", #rgFiltered, cTotalRanks); end
	end
	--=====================================================================--
	-- end of local function FilterCompanions
	-- resumption of parent function SummonCompanion
	--=====================================================================--

	-- generate filtered list of elligible pets and total ranks for list
	if (fNext and CurPet ~= gc.nopet) then
		-- We can optimize our search in this case by starting immediately after current
		FilterCompanions(CurPet.index + 1, #rgPets, true);

		-- if none were found in that portion of the list, then start new search from the front of the list
		if (cTotalRanks == 0) then FilterCompanions(1, CurPet.index, true); end
	else
		FilterCompanions(1, #rgPets, fNext);
	end


	-- if we skipped the hot pet while building a list, nows the time for its heat check
	if (fSkippedHot) then
		local iHeatCheck = mrandom(1,100);
		if (self.db.char.pets.iHeat >= iHeatCheck) then
			if (DC.PETS >= DL.BASIC) then self:Printf("%s passed heat check (rolled %i, needed %i or less) and will be summoned", gv.HotPet.name, iHeatCheck, self.db.char.pets.iHeat); end
			return gv.HotPet;
		end

		if (DC.PETS >= DL.BASIC) then self:Printf("%s skipped for failing heat check (rolled %i, needed %i or less) as hot pet", gv.HotPet.name, iHeatCheck, self.db.char.pets.iHeat); end
	end

	-- selection returned 0 pets to choose from	
	if (#rgFiltered == 0 or cTotalRanks == 0) then 
		-- both values should be in sync
		if (#rgFiltered ~= cTotalRanks) then self:Print("ERROR: only one of #rgFiltered and cTotalRanks was zero"); end

		if (fSkippedHot) then
			if (DC.PETS >= DL.BASIC) then self:Printf("hot pet %s failed heat check but is apparently only one summonable", gv.HotPet.name); end
			return gv.HotPet;
		end

		if (fSkippedCur) then
			if (DC.PETS >= DL.BASIC) then self:Printf("current companion %s is apparently only one summonable; doing nothing", CurPet.name); end
			return; -- only summonable pet is already summoned
		end

		self:Print(L["ERROR: You have no summonable companions."]);
		return;
	end


	-- only one pet to choose from
	if (#rgFiltered == 1) then
		local pet = rgFiltered[1];
		if (DC.PETS >= DL.BASIC) then self:Printf("%s is apparently only one summonable", pet.name); end
		return pet;
	end

	-- multiple pets
	local cRank = mrandom(1,cTotalRanks);
	if (DC.PETS >= DL.EXCEP) then self:Printf("random roll from 1 to %i produced %i", cTotalRanks, cRank); end
	for _, pet in ipairs(rgFiltered) do
		cRank = cRank - pet.rank;
		if (DC.PETS >= DL.AV) then self:Printf("%s's rank of %i brings total down to %i", pet.name, pet.rank, cRank); end
		if (cRank <= 0) then -- found our slot
			if (DC.PETS >= DL.BASIC) then self:Printf("random selection has chosen %s", pet.name); end
			return pet;
		end  
	end

	if (cRank > 0) then self:Print(L["ERROR: selection error"]); end
end


local cpjFilters = {}

cpjFilters.__index = cpjFilters
cpjFilters.ourFlags = {
		[_G.LE_PET_JOURNAL_FILTER_COLLECTED] = true, 
		[_G.LE_PET_JOURNAL_FILTER_NOT_COLLECTED] = false, }

function cpjFilters:New()
	local t = { 
		types = {},
		sources = {},
		userFlags = {},
		strSearch = _G.SEARCH,
	}

	return setmetatable(t, cpjFilters)
end

function cpjFilters:Cache(strFlags)
	-- since our search may trigger PET_JOURNAL_LIST_UPDATE events if 
	-- any of the filters are changed, we should temporarily unregister
	EventMonitor_PJListUpdate:Suspend()

	for flag, value in pairs(self.ourFlags) do
		local userValue = cpj.IsFilterChecked(flag)
		self.userFlags[flag] = userValue
		if (userValue ~= value) then
			cpj.SetFilterChecked(flag, value)
		end
	end

	for i=1, cpj.GetNumPetTypes() do
		if not cpj.IsPetTypeChecked(i) then
			self.types[i] = true
			cpj.SetPetTypeFilter(i, true)
		end
	end

	for i=1, cpj.GetNumPetSources() do
		if not cpj.IsPetSourceChecked(i) then
			self.sources[i] = true
			cpj.SetPetSourceChecked(i, true)
		end
	end

	self.strSearch = (_G.PetJournalSearchBox) and _G.PetJournalSearchBox:GetText() or _G.SEARCH
	if (self.strSearch ~= _G.SEARCH) then
		cpj.ClearSearchFilter()
	end
end

function cpjFilters:Restore()
	for flag,value in pairs(self.userFlags) do
		if value ~= self.ourFlags[flag] then
			cpj.SetFilterChecked(flag, value)
		end
	end

	for i,v in pairs(self.types) do
		if v then
			cpj.SetPetTypeFilter(i, false)
		end
	end

	for i,v in pairs(self.sources) do
		if v then
			cpj.SetPetSourceChecked(i, false)
		end
	end

	if (self.strSearch ~= _G.SEARCH) then
		cpj.SetSearchFilter(self.strSearch)
	end

	self.inuse = false

	-- register event again
	EventMonitor_PJListUpdate:Resume()
end


local metaPet = {
	__lt = gs.pfPets,
	__index = function(t,k)
			if (k == "rank") then
				return t.char_rank or t.profile_rank
			elseif (k == "char_rank") then
				return Pokedex.db.char.pets.ranks[t.name]
			elseif (k == "profile_rank") then
				return Pokedex.db.profile.pets.ranks[t.name] or Pokedex.db.profile.pets.iDefaultRank
			else
				return nil
			end
		end,
	__newindex = function(t,k,v)
			if (k == "char_rank") then
				Pokedex.db.char.pets.ranks[t.name] = v
			elseif (k == "profile_rank") then
				Pokedex.db.profile.pets.ranks[t.name] = (Pokedex.db.profile.pets.iDefaultRank ~= v) and v or nil
			else
				Pokedex:Print("unexpected table assignment caught by metaPet", t.name, k, v)
			end
		end,
}

local CPets = { numOwned = -1, count = 0 }

gv.Pets = setmetatable( {}, CPets )

CPets.__index = function(t,k)
	-- if the number of pets has changed, build lists
	local fUpdated = false
	local _, numOwned = cpj.GetNumPets(false);
	if (CPets.numOwned ~= numOwned) then
		if (DC.PETS >= DL.EXCEP) then Pokedex:Printf("Building Pet List from %i pets to %i pets for key access %s", CPets.numOwned, numOwned, tostring(k)); end
		fUpdated = true

		-- cache current filter and search state of PetJournal and then set flags for all user pets
		local filter = cpjFilters:New()
		filter:Cache()

		CPets:UpdatePets()

		-- restore pet journal settings	
		filter:Restore()
	end

	return (k == "fUpdated") and fUpdated or CPets[k]
end

function CPets:UpdatePets()
	local fInitialized  = (self.count > 0)  -- count instead of numOwned because you can own unusable pets and we want that to be same as owning nothing
	local fFirstRun = (Pokedex.db.char.pets.known == nil)

	local count   = 0                                -- number of pet names or species
	local All     = { [gc.nopet] = L["no companions available"] }   -- key(table)  is pet         value(string) is pet name
	local HotList = {        [0] = L["no hot companion"] }          -- key(int) is display order  value(string) is pet name

	local ByID    = { [0] = gc.nopet }  -- key(number) is speciesID   value(table)  is pet
	local ByCID   = { [0] = gc.nopet }  -- key(number) is creatureID  value(table)  is pet
	local ByPID   = { [0] = gc.nopet }  -- key(number) is petID       value(table)  is pet
	local ByName  = { [0] = gc.nopet }  -- key(string) is pet name    value(table)  is pet
	local ByIndex = { [0] = gc.nopet }  -- key(number) is name order  value(table)  is pet

	local idCurSelection = gv.SelectedPet.id
	local petNew
	
	local strknown = Pokedex.db.char.pets.known or ""
	local known = fInitialized and self.known or {}
	if not fInitialized and not fFirstRun then 
		for id in gmatch(strknown, "(%d+)") do
			known[tonumber(id)] = true
		end
	end

	-- local errorCount = 0
	local numPets, numOwned = cpj.GetNumPets(false)
	assert(numPets == numOwned, "ERROR pet journal list isn't filtered correctly")
	for i=1, numPets do
		local petID, speciesID, isOwned, customName, level, favorite, _, name, _, _, creatureID = cpj.GetPetInfoByIndex(i, false);
		-- assert(isOwned, tostring(name))
		if not isOwned then
			if (DC.PETS >= DL.EXCEP) then Pokedex:Print("skipping unowned pet", name); end
			-- if errorCount == 0 then print("not owned", tostring(name)) end
			-- errorCount = errorCount + 1
		elseif (not cpj.PetIsSummonable(petID)) then
			if (DC.PETS >= DL.EXCEP) then Pokedex:Print("skipping non-summonable owned pet", name); end
		else
			local pet = ByName[name];
			if (not pet) then
				assert(ByID[speciesID] == nil)
				assert(ByCID[creatureID] == nil)

				count = count + 1
				pet = setmetatable( { name = name, cid = creatureID, id = speciesID, index = count, pids = {} }, metaPet )

				All[pet] = gf.Pets(pet)
				tinsert(HotList, name)

				ByID[speciesID] = pet
				ByCID[creatureID] = pet
				ByName[name] = pet
				ByIndex[count] = pet

				if not known[speciesID] then
					known[speciesID] = true
					strknown = format("%i %s", speciesID, strknown)

					if not fFirstRun then
						if (DC.PETS >= DL.BASIC) then Pokedex:Print("new pet found", name, speciesID); end
						petNew = pet

						if (Pokedex.db.char.pets.fEnableHotness and Pokedex.db.char.pets.fNewHotness) then
							Pokedex.db.char.pets.idHot = speciesID
						end
					end
				end

--[===[
			elseif (not fInitialized) then
				-- sanity checks that unique ids really are, only do on first pass
				local petFromID = ByID[speciesID]
				local petFromCID = ByCID[creatureID]
				assert(pet == petFromID)
				assert(pet == petFromCID)
				assert(petFromID == petFromCID)
--]===]
			end

			assert(ByPID[petID] == nil)
			ByPID[petID] = pet;
			tinsert(pet.pids, petID)
		end
	end

	-- if we have pets to show in our dropdown, remove the placeholder
	if (count ~= 0) then
		All[gc.nopet] = nil
	end

	-- sort the HotList alphabetically and find the index for the HotPet
	gv.HotPet = ByID[Pokedex.db.char.pets.idHot] or gc.nopet
	local hotName = gv.HotPet.name
	sort(HotList)
	for i=0, #HotList do
		if HotList[i] == hotName then 
			gv.HotPetIndex = i
			break
		end
	end

	self.numOwned = numOwned
	self.count   = count
	
	self.All     = All
	self.HotList = HotList

	self.ByID    = ByID
	self.ByCID   = ByCID
	self.ByPID   = ByPID
	self.ByName  = ByName
	self.ByIndex = ByIndex
	
	self.known = known
	Pokedex.db.char.pets.known = strknown

	-- new pet > active pet > what's already selected > nopet if we have no summonable pets > first
	local pidActive = cpj.GetSummonedPetGUID()
	local petActive = pidActive and ByPID[pidActive]
	Pokedex:SetSelectedPet(petNew or petActive or ByID[idCurSelection])  -- will set gv.SelectedPet
end

function CPets:UpdateDisplayNames(changed)
	local pets = self.All
	if changed then
		if pets[changed] then
			pets[changed] = gf.Pets(changed)
		end
	else
		for pet in pairs(pets) do
			pets[pet] = gf.Pets(pet)
		end
	end
end	

local cIgnorePJLU = 2 -- number of events during login to ignore
function Pokedex:PET_JOURNAL_LIST_UPDATE(...)
	if (DC.PETS >= DL.AV) then self:Print(...); end
	-- This event is firing off *before* the pet journal returns accurate information.
	-- We're just going to assume that the first event is always bad and that 
	-- everything will be fine from the second event onwards.
	-- Also, we're going to ignore the second event since we want to delay 
	-- initialization until a user action forces it.
	if cIgnorePJLU > 0 then
		cIgnorePJLU = cIgnorePJLU - 1
		return
	end

	local _, numOwned = cpj.GetNumPets(false);
	if (numOwned > 0 and gv.Pets.fUpdated) then
		LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
	end
end


--=========================================================================--
--=========================================================================--
--
-- TITLE FUNCTIONS
--
--=========================================================================--
--=========================================================================--

function Pokedex:KNOWN_TITLES_UPDATE()
	if (DC.TITLES >= DL.BASIC) then self:Print("KNOWN_TITLES_UPDATE"); end
	self:UpdateTitleInfo();
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:NEW_TITLE_EARNED(...)
	if (DC.TITLES >= DL.BASIC) then self:Printf("NTE  %s", self:StrFromVarArg(nil, ...)); end
	self:UpdateTitleInfo();
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:OLD_TITLE_LOST(...)
	if (DC.TITLES >= DL.BASIC) then self:Printf("OTL  %s", self:StrFromVarArg(nil, ...)); end
	self:UpdateTitleInfo();
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end

function Pokedex:SetCurrentTitleHook(idTitle)
	gv.SelectedTitle = gv.Titles.ByID[idTitle] or gv.Titles.ByID[-1];
	if gv.SelectedTitle == nil then
		gv.SelectedTitle = gv.Titles.ByID[(GetCurrentTitle())]
	end
	if (DC.TITLES >= DL.BASIC) then 
		self:Printf("Current title set to: %i %s", idTitle, gv.SelectedTitle.name); 
	end
end

-- return first the title ID and then the sorted order index
function Pokedex:GetCurrentTitle()
	local idTitle = GetCurrentTitle();
	return idTitle, gv.Titles.ByID[idTitle];
end

function Pokedex:ChangeTitle()
	local _, CurTitle = self:GetCurrentTitle();

	local rgFiltered = {};
	local cTotalRanks = 0;
	local fSkippedCur = false;
	local fSkippedHot = false;

	-- Builds list of elligible titles for change. For mounts and pets, this
	-- is a local function; but we have no plans for NextTitle option or any
	-- other scenarios where we have to make multiple passes at the data.
	if (DC.TITLES >= DL.EXCEP) then self:Print("building list of eligible titles ..."); end
	for title in pairs(gv.Titles.All) do
		-- if its ranked as 0, then it gets skipped
		if (title.rank == 0) then
			if (DC.TITLES >= DL.EXCEP) then self:Printf("%s skipped for having rank of 0", title.name); end
		-- if its the current title, then it is skipped
		elseif (title == CurTitle) then
			if (DC.TITLES >= DL.EXCEP) then self:Printf("%s skipped for being current title", title.name); end
			fSkippedCur = true;
		-- if hot title and hotness is enabled then don't add to list, it will be handled seperately
		elseif (self.db.char.titles.fEnableHotness and title == gv.HotTitle and title ~= gc.notitle) then
			if (DC.TITLES >= DL.EXCEP) then self:Printf("%s skipped for being hot title", title.name); end
			fSkippedHot = true;
		-- else its put into the pool of summonables
		else
			tinsert(rgFiltered, title)
			cTotalRanks = cTotalRanks + title.rank;
			if (DC.TITLES >= DL.AV) then self:Printf("%s added to list of possible titles with a rank of %i. Total rank count is %i", title.name, title.rank, cTotalRanks); end
		end
	end

	if (DC.TITLES >= DL.BASIC) then self:Printf("list built: %i titles with a total rank of %i", #rgFiltered, cTotalRanks); end

	-- if we skipped the hot title while building a list, nows the time for its heat check
	if (fSkippedHot) then
		local iHeatCheck = mrandom(1,100);
		if (self.db.char.titles.iHeat >= iHeatCheck) then
			if (DC.TITLES >= DL.BASIC) then self:Printf("%s passed heat check (rolled %i, needed %i or less) and will be selected", gv.HotTitle.name, iHeatCheck, self.db.char.titles.iHeat); end
			_G.SetCurrentTitle(gv.HotTitle.id);
			return;
		end

		if (DC.TITLES >= DL.BASIC) then self:Printf("%s failed heat check (rolled %i, needed %i or less) and will not be selected", gv.HotTitle.name, iHeatCheck, self.db.char.titles.iHeat); end
	end

	-- selection returned 0 titles to choose from	
	if (#rgFiltered == 0 or cTotalRanks == 0) then 
		-- both values should be in sync
		if (#rgFiltered ~= cTotalRanks) then self:Print(L["ERROR: only one of #rgFiltered and cTotalRanks was zero"]); end

		if (fSkippedHot) then
			if (DC.TITLES >= DL.BASIC) then self:Printf("%s failed heat check but is only eligible title", gv.HotTitle.name); end
			_G.SetCurrentTitle(gv.HotTitle.id);
			return
		end

		if (fSkippedCur) then
			if (DC.TITLES >= DL.BASIC) then self:Printf("%s is only one eligible; doing nothing", CurTitle.name); end
			return; -- only selectable title is already displayed
		end

		self:Print(L["ERROR: You don't have any titles."]);
		return;
	end

	
	-- only one title to choose from
	if (#rgFiltered == 1) then
		local title = rgFiltered[1];
		if (DC.TITLES >= DL.BASIC) then self:Printf("%s is apparently only title known", title.name); end
		_G.SetCurrentTitle(title.id);
		return;
	end

	-- multiple titles
	local cRank = mrandom(1,cTotalRanks);
	if (DC.TITLES >= DL.EXCEP) then self:Printf("random roll from 1 to %i produced %i", cTotalRanks, cRank); end

	for _, title in ipairs(rgFiltered) do
		cRank = cRank - title.rank;
		if (DC.TITLES >= DL.AV) then self:Printf("%s's rank of %i brings total down to %i", title.name, title.rank, cRank); end
		if (cRank <= 0) then -- found our slot
			if (DC.TITLES >= DL.BASIC) then self:Printf("random selection has chosen %s", title.name); end
			_G.SetCurrentTitle(title.id);
			return;
		end
	end

	if (cRank > 0) then self:Print(L["ERROR: selection error"]); end
end

local metaTitle = {
	__lt = gs.pfTitles,
	__index = function(t,k)
			if (k == "rank") then
				return t.char_rank or t.profile_rank
			elseif (k == "char_rank") then
				return Pokedex.db.char.titles.ranks[t.name]
			elseif (k == "profile_rank") then
				return Pokedex.db.profile.titles.ranks[t.name] or Pokedex.db.profile.titles.iDefaultRank
			else
				return nil
			end
		end,
	__newindex = function(t,k,v)
			if (k == "char_rank") then
				Pokedex.db.char.titles.ranks[t.name] = v
			elseif (k == "profile_rank") then
				Pokedex.db.profile.titles.ranks[t.name] = (Pokedex.db.profile.titles.iDefaultRank ~= v) and v or nil
			else
				Pokedex:Print("unexpected table assignment caught by metaTitle", t.name, k, v)
			end
		end,
}

local CTitles = { numTitles = -1, count = 0 }

gv.Titles = setmetatable( {}, CTitles )

CTitles.__index = function(t,k)
	-- if the number of titles has changed, build lists
	local fUpdated = false
	local numTitles = GetNumTitles();
	if (CTitles.numTitles ~= numTitles) then
		if (DC.TITLES >= DL.EXCEP) then Pokedex:Printf("Building Title List from %i titles to %i titles for key access %s", CTitles.numTitles, numTitles, tostring(k)); end
		fUpdated = true
		CTitles:UpdateTitles()
	end

	return (k == "fUpdated") and fUpdated or CTitles[k]
end

function CTitles:UpdateTitles()
	local fInitialized  = (self.count > 0)  -- count instead of numTitles because you can own unusable titles and we want that to be same as owning nothing
	local fFirstRun = (Pokedex.db.char.titles.known == nil)

	gc.notitle = setmetatable( gc.notitle, metaTitle )        -- replace simple meta table, since No Title can be ranked
	local strNoTitle = _G.PLAYER_TITLE_NONE
	local count   = 0                                         -- number of titles known

	local All     = { [gc.notitle] = strNoTitle }             -- key(table)  is title       value(string) is title name
	local HotList = {          [0] = L["no hot title"] }      -- key(int) is display order  value(string) is title name

	local ByID    = { [gc.notitle.id]  = gc.notitle }   -- key(number) is titleID       value(table)  is title
	local ByName  = {     [strNoTitle] = gc.notitle }   -- key(string) is title name    value(table)  is title
	-- local ByIndex = { [0] = gc.notitle }  -- key(number) is name order  value(table)  is title

	local idCurSelection = gv.SelectedTitle.id
	local titleNew
	
	local strknown = Pokedex.db.char.titles.known or ""
	local known = fInitialized and self.known or {}
	if not fInitialized and not fFirstRun then 
		for id in gmatch(strknown, "(%d+)") do
			known[tonumber(id)] = true
		end
	end

	local numTitles = GetNumTitles()
	for titleID = 1, numTitles do
		if (IsTitleKnown(titleID)) then
			count = count + 1
			local name = strtrim((GetTitleName(titleID)))
			local title = setmetatable( { name = name, id = titleID }, metaTitle )
			
			All[title] = gf.Titles(title)
			tinsert(HotList, name)

			ByID[titleID] = title
			ByName[name] = title
			
			if not known[titleID] then
				known[titleID] = true
				strknown = format("%i %s", titleID, strknown)

				if not fFirstRun then
					if (DC.TITLES >= DL.BASIC) then Pokedex:Print("new title found", name, titleID); end
					titleNew = title

					if (Pokedex.db.char.titles.fEnableHotness and Pokedex.db.char.titles.fNewHotness) then
						Pokedex.db.char.titles.idHot = titleID
					end
				end
			end
		end
	end

	-- if there use does not have any titles, change the string shown in the dropdown
	if (count == 0) then
		All[gc.notitle] = L["no titles available"]
	end

	-- sort the HotList alphabetically and find the index for the HotTitle
	gv.HotTitle = ByID[Pokedex.db.char.titles.idHot] or gc.notitle
	local hotName = gv.HotTitle.name
	sort(HotList)
	for i=0, #HotList do
		if HotList[i] == hotName then 
			gv.HotTitleIndex = i
			break
		end
	end

	self.numTitles = numTitles
	self.count     = count
	
	self.All       = All
	self.HotList   = HotList

	self.ByID      = ByID
	self.ByName    = ByName
	
	self.known = known
	Pokedex.db.char.titles.known = strknown

	-- new title > active title > what's already selected > notitle if we have no summonable titles > first
	gv.SelectedTitle = titleNew or ByID[GetCurrentTitle()] or ByID[idCurSelection]  -- will set gv.SelectedTitle
end

function CTitles:UpdateDisplayNames(changed)
	local titles = self.All
	if changed then
		if titles[changed] then
			titles[changed] = gf.Titles(changed)
		end
	else
		for title in pairs(titles) do
			titles[title] = gf.Titles(title)
		end
	end
end	

function Pokedex:UpdateTitleInfo()
	gv.Titles:UpdateTitles()
end


--=========================================================================--
--=========================================================================--
--
-- AUTO-DISMOUNT FUNCTIONS
--
--=========================================================================--
--=========================================================================--

function Pokedex:CVAR_UPDATE(_, glstr, ...)
	if (DC.DISMOUNT >= DL.EXCEP) then self:Printf("CVAR_UPDATE EVENT:  %s %s", tostring(glstr), self:StrFromVarArg(nil, ...)); end
	if (glstr == "AUTO_DISMOUNT_FLYING_TEXT") then
		self:UpdateDismountSettings();
	end
end


function Pokedex:SetManageAutoDismount(info, value)
	if (gv.iAutoDismountFlying == 1 and value) then
		if (DC.DISMOUNT >= DL.EXCEP) then self:Print("Safe Dismount enabled, disabling Auto Dismount in Flight accordingly."); end
		gv.iAutoDismountFlying = 0;
		SetCVar("autoDismountFlying", 0); --, "AUTO_DISMOUNT_FLYING_TEXT")
	end

	Pokedex.db.profile.SafeDismount.Enabled = value;
end


function Pokedex:UpdateDismountSettings()
	if (GetCVarBool("autoDismountFlying")) then 
		-- even though we may be reseting it back to 0, our internal state should match reality
		gv.iAutoDismountFlying = 1;

		if (self.db.profile.SafeDismount.Enabled) then
			if (DC.DISMOUNT >= DL.EXCEP) then self:Print("Safe Dismount is reverting someones change to the Auto Dismount in Flight setting."); end
			SetCVar("autoDismountFlying", 0); --, "AUTO_DISMOUNT_FLYING_TEXT")
		end
	else
		gv.iAutoDismountFlying = 0;
	end
	LibStub("AceConfigRegistry-3.0"):NotifyChange("Pokedex");
end


function Pokedex:MainTooltipShow(...)
	-- if we have gathering skills and the feature is turned on
	-- if we're not currently flying then we don't have to worry about monkeying with the setting at all
	local SafeDismount = self.db.profile.SafeDismount
	if (SafeDismount.Enabled and SafeDismount.ForGathering and IsFlying()) then
		-- if dismount was turned on for a different condition, tooltip scraping will only cause errors
		if ((not SafeDismount.ForCombat or not gv.fCanDismountForCombat) and
		    (not SafeDismount.ForAttack or not gv.fCanDismountForAttack)) then

			local fGatherable = false;
			for i=1,GameTooltip:NumLines() do
				local mytext = _G["GameTooltipTextLeft" .. i];
				local text = mytext:GetText();
				fGatherable = (gv.Skills.fMiner     and strfind(text, L["Requires"]) and strfind(text, L["Mining"])) or
				              (gv.Skills.fHerbalist and strfind(text, L["Requires"]) and strfind(text, L["Herbalism"])) or
				              (gv.Skills.fSkinner   and strfind(text, L["Skinnable"]));
			end

			if (fGatherable and gv.iAutoDismountFlying == 0) then
				if (DC.DISMOUNT >= DL.BASIC) then self:Print("turning on dismount, mousing over gatherable"); end
				gv.iAutoDismountFlying = 1;
				SetCVar("autoDismountFlying", 1); --, "AUTO_DISMOUNT_FLYING_TEXT")
			elseif (fGatherable and gv.iAutoDismountFlying == 1) then
				if (DC.DISMOUNT >= DL.EXCEP) then self:Print("WEIRD - new show, but can already dismount"); end
			elseif (not fGatherable and gv.iAutoDismountFlying == 1) then
				if (DC.DISMOUNT >= DL.BASIC) then self:Print("ERROR - can dismount, but tooltip doesn't match"); end
				gv.iAutoDismountFlying = 0;
				SetCVar("autoDismountFlying", 0); --, "AUTO_DISMOUNT_FLYING_TEXT")
			end
		end
	end
end


function Pokedex:MainTooltipHide(...)
	-- we only care if gathering feature is turned on
	local SafeDismount = self.db.profile.SafeDismount
	if (SafeDismount.Enabled and SafeDismount.ForGathering) then
		-- if dismount is already off, we don't have to do anything
		if (gv.iAutoDismountFlying == 1) then
			-- if dismount was turned on for a different condition, tooltip scraping will only cause errors
			if ((not SafeDismount.ForCombat or not gv.fCanDismountForCombat) and
			    (not SafeDismount.ForAttack or not gv.fCanDismountForAttack)) then
				if (DC.DISMOUNT >= DL.BASIC) then self:Print("dismount turned off, not mousing over gatherable"); end
				gv.iAutoDismountFlying = 0;
				SetCVar("autoDismountFlying", 0); --, "AUTO_DISMOUNT_FLYING_TEXT")
			end
		end
	end	
end


function Pokedex:PLAYER_TARGET_CHANGED()
	local SafeDismount = self.db.profile.SafeDismount
	-- if feature is turned on
	if (SafeDismount.Enabled and SafeDismount.ForAttack) then
		-- if we had an attackable target
		if (gv.fCanDismountForAttack) then
			-- then only need to change if we lost one
			if (not UnitCanAttack("player", "target") or UnitIsDead("target")) then
				gv.fCanDismountForAttack = false;

				-- if in combat, don't clear based on target
				if (not SafeDismount.ForCombat or not gv.fCanDismountForCombat) then
					if (DC.DISMOUNT >= DL.BASIC) then self:Print("dismount turned off, target not attackable"); end
					gv.iAutoDismountFlying = 0;
					SetCVar("autoDismountFlying", 0); --, "AUTO_DISMOUNT_FLYING_TEXT")
				end
			end
		-- if not already tracking, then don't start unless we're mounted - hopeful optimization for instances
		elseif (IsMounted()) then
			-- if we have an attackable target
			if (UnitCanAttack("player", "target") and not UnitIsDead("target")) then
				gv.fCanDismountForAttack = true;
				
				-- don't need to set if already done for being in combat
				if (not SafeDismount.ForCombat or not gv.fCanDismountForCombat) then
					if (DC.DISMOUNT >= DL.BASIC) then self:Print("turning on dismount, attackable target"); end
					gv.iAutoDismountFlying = 1;
					SetCVar("autoDismountFlying", 1); --, "AUTO_DISMOUNT_FLYING_TEXT")
				end
			end
		end
	end
end


function Pokedex:PLAYER_REGEN_DISABLED()
	local SafeDismount = self.db.profile.SafeDismount
	-- if feature is turned on
	-- if we're not currently mounted then we don't have to worry about monkeying with the setting at all
	if (SafeDismount.Enabled and SafeDismount.ForCombat and IsMounted()) then
		gv.fCanDismountForCombat = true;

		if (gv.iAutoDismountFlying == 0) then
			if (DC.DISMOUNT >= DL.BASIC) then self:Print("turning on dismount, in combat"); end
			gv.iAutoDismountFlying = 1;
			SetCVar("autoDismountFlying", 1); --, "AUTO_DISMOUNT_FLYING_TEXT")
		end
	end
end


function Pokedex:PLAYER_REGEN_ENABLED()
	local SafeDismount = self.db.profile.SafeDismount
	-- if feature is turned on
	if (SafeDismount.Enabled and SafeDismount.ForCombat) then
		-- out of combat, so this will no longer be the reason why we can dismount
		gv.fCanDismountForCombat = false;

		-- if can dismount is already 0, then there's nothing left to change
		if (gv.iAutoDismountFlying == 1) then
			-- if dismount can occur for attackable target, don't clear for out of combat
			if (not SafeDismount.ForAttack or not gv.fCanDismountForAttack) then
				if (DC.DISMOUNT >= DL.BASIC) then self:Print("dismount turned off, out of combat"); end
				gv.iAutoDismountFlying = 0;
				SetCVar("autoDismountFlying", 0); --, "AUTO_DISMOUNT_FLYING_TEXT")
			end
		end
	end	
end
