local MogIt,mog = ...;
_G["MogIt"] = mog;
local L = mog.L;

local ItemInfo = LibStub("LibItemInfo-1.0");

LibStub("Libra"):Embed(mog);

mog.frame = CreateFrame("Frame","MogItFrame",UIParent,"ButtonFrameTemplate");
mog.list = {};

function mog:Error(msg)
	DEFAULT_CHAT_FRAME:AddMessage("MogIt: "..msg,0.9,0.5,0.9);
end

--// Slash Commands
function mog:ToggleFrame()
	ToggleFrame(mog.frame);
end

function mog:TogglePreview()
	ToggleFrame(mog.view);
end
--//


--// Bindings
SLASH_MOGIT1 = "/mog";
SLASH_MOGIT2 = "/mogit";
SlashCmdList["MOGIT"] = mog.ToggleFrame;

BINDING_HEADER_MogIt = "MogIt";
BINDING_NAME_MogIt = L["Toggle Mogit"];
BINDING_NAME_MogItPreview = L["Toggle Preview"];
--//


--// LibDataBroker
mog.LDBI = LibStub("LibDBIcon-1.0");
mog.mmb = LibStub("LibDataBroker-1.1"):NewDataObject("MogIt",{
	type = "launcher",
	icon = "Interface\\Icons\\INV_Enchant_EssenceCosmicGreater",
	OnClick = function(self,btn)
		if btn == "RightButton" then
			mog:TogglePreview();
		else
			mog:ToggleFrame();
		end
	end,
	OnTooltipShow = function(self)
		if not self or not self.AddLine then return end
		self:AddLine("MogIt");
		self:AddLine(L["Left click to toggle MogIt"],1,1,1);
		self:AddLine(L["Right click to toggle the preview"],1,1,1);
	end,
});
--//


--// Module API
mog.moduleVersion = 3;
mog.modules = {};
mog.moduleList = {};

function mog:GetModule(name)
	return mog.modules[name];
end

function mog:GetActiveModule()
	return mog.active;
end

function mog:RegisterModule(name,version,data)
	if mog.modules[name] then
		--mog:Error(L["The \124cFFFFFFFF%s\124r module is already loaded."]:format(name));
		return mog.modules[name];
	--elseif type(version) ~= "number" or version < mog.moduleVersion then
		--mog:Error(L["The \124cFFFFFFFF%s\124r module needs to be updated to work with this version of MogIt."]:format(name));
		--return;
	--elseif version > mog.moduleVersion then
		--mog:Error(L["The \124cFFFFFFFF%s\124r module requires you to update MogIt for it to work."]:format(name));
		--return;
	end
	data = data or {};
	data.version = version;
	data.name = name;
	mog.modules[name] = data;
	table.insert(mog.moduleList,data);
	if mog.menu.active == mog.menu.modules then
		mog.menu:Rebuild(1);
	end
	return data;
end

function mog:SetModule(module,text)
	if mog.active and mog.active ~= module and mog.active.Unlist then
		mog.active:Unlist(module);
	end
	mog.active = module;
	mog:BuildList(true);
	mog:FilterUpdate();
	mog.frame.path:SetText(text or module.label or module.name or "");
end

function mog:BuildList(top,module)
	if (module and mog.active and mog.active.name ~= module) then return end;
	mog.list = mog.active and mog.active.BuildList and mog.active:BuildList() or {};
	mog:SortList(nil,true);
	mog:UpdateScroll(top and 1);
	mog.filt.models:SetText(#mog.list);
end
--//


--// Item Cache
local itemCacheCallbacks = {
	BuildList = mog.BuildList;
	ModelOnEnter = function()
		local owner = GameTooltip:GetOwner();
		if owner and GameTooltip[mog] then
			owner:OnEnter();
		end
	end,
	ItemMenu = function()
		mog.Item_Menu:Rebuild(1);
	end,
	SetMenu = function()
		mog.Set_Menu:Rebuild(1);
	end,
};

local pendingCallbacks = {};

for k in pairs(itemCacheCallbacks) do
	pendingCallbacks[k] = {};
end

function mog:AddItemCacheCallback(name, func)
	itemCacheCallbacks[name] = func;
	pendingCallbacks[name] = {};
end

function mog:GetItemInfo(id, callback)
	if not callback then return ItemInfo[id] end
	if ItemInfo[id] then
		-- clear pending items when they are cached
		pendingCallbacks[callback][id] = nil;
		return ItemInfo[id];
	elseif itemCacheCallbacks[callback] then
		-- add to pending items for this callback if not cached
		pendingCallbacks[callback][id] = true;
	end
end

function mog.ItemInfoReceived()
	for k, callback in pairs(pendingCallbacks) do
		-- execute the callback if any items are pending for it
		if next(callback) then
			itemCacheCallbacks[k]();
		end
	end
end

ItemInfo.RegisterCallback(mog, "OnItemInfoReceivedBatch", "ItemInfoReceived");
--//

local sourceItemLink = {}

function mog:GetItemLinkFromSource(source)
	if not sourceItemLink[source] then
		local _, _, _, _, _, link = C_TransmogCollection.GetAppearanceSourceInfo(source)
		sourceItemLink[source] = link
	end
	return sourceItemLink[source]
end

local itemSourceID = {}

local model = CreateFrame("DressUpModel")
model:SetAutoDress(false)

function mog:GetSourceFromItem(item)
	if not itemSourceID[item] then
		local visualID, sourceID = C_TransmogCollection.GetItemInfo(item)
		itemSourceID[item] = sourceID
		if not itemSourceID[item] then
			model:SetUnit("player")
			model:Undress()
			model:TryOn(item)
			for i = 1, 19 do
				local source = model:GetSlotTransmogSources(i)
				if source ~= 0 then
					itemSourceID[item] = source
					break
				end
			end
		end
	end
	return itemSourceID[item]
end

function mog:HasItem(sourceID, includeAlternate)
	if not sourceID then return end
	local found = false;
	local sourceInfo = C_TransmogCollection.GetSourceInfo(sourceID)
	if not sourceInfo then return end
	found = sourceInfo.isCollected
	if includeAlternate then
		local sources = C_TransmogCollection.GetAllAppearanceSources(sourceInfo.visualID)
		for i, sourceID in ipairs(sources) do
			local sourceInfo = C_TransmogCollection.GetSourceInfo(sourceID)
			if sourceInfo.isCollected then
				found = true
				break
			end
		end
	end
	return found
end


--// Events
local defaults = {
	profile = {
		tooltipItemID = false,
		alwaysShowCollected = true,
		tooltipAlwaysShowOwned = true,
		wishlistCheckAlts = true,
		tooltipWishlistDetail = true,
		loadModulesDefault = false,
		
		noAnim = false,
		url = "Battle.net",
		
		dressupPreview = false,
		singlePreview = false,
		previewUIPanel = false,
		previewFixedSize = false,
		previewConfirmClose = true,
		
		sortWishlist = false,
		loadModulesWishlist = false,
		
		tooltip = true,
		tooltipWidth = 300,
		tooltipHeight = 300,
		tooltipMouse = false,
		tooltipDress = false,
		tooltipRotate = true,
		tooltipMog = true,
		tooltipMod = "None",
		tooltipCustomModel = false,
		tooltipRace = 1,
		tooltipGender = 0,
		tooltipAnchor = "vertical",
		
		minimap = {},
		
		point = "CENTER",
		gridWidth = 600,
		gridHeight = 400,
		rows = 2;
		columns = 3,
		gridDress = "preview",
		sync = true,
		previewProps = {
			["*"] = {
				w = 335,
				h = 385,
				point = "CENTER",
			}
		},
		
		slotLabels = {},
	}
}

function mog.LoadSettings()
	mog:UpdateGUI();
	
	if mog.db.profile.minimap.hide then
		mog.LDBI:Hide("MogIt");
	else
		mog.LDBI:Show("MogIt");
	end
	
	mog.tooltip:SetSize(mog.db.profile.tooltipWidth, mog.db.profile.tooltipHeight);
	mog.tooltip.rotate:SetShown(mog.db.profile.tooltipRotate);
	
	mog:UpdateScroll();
	
	mog:SetSinglePreview(mog.db.profile.singlePreview);
end

function mog:LoadBaseModules()
	for i, module in ipairs(self.baseModules) do
		if GetAddOnEnableState(myName, module) > 0 and not IsAddOnLoaded(module) then
			LoadAddOn(module)
		end
	end
end

mog.frame:RegisterEvent("ADDON_LOADED");
mog.frame:RegisterEvent("PLAYER_LOGIN");
mog.frame:RegisterEvent("GET_ITEM_INFO_RECEIVED");
mog.frame:RegisterEvent("PLAYER_EQUIPMENT_CHANGED");
mog.frame:RegisterEvent("TRANSMOG_SEARCH_UPDATED");
mog.frame:SetScript("OnEvent", function(self, event, ...)
	return mog[event] and mog[event](mog, ...)
end);

function mog:ADDON_LOADED(addon)
	if addon == MogIt then
		local AceDB = LibStub("AceDB-3.0")
		mog.db = AceDB:New("MogItDB", defaults, true)
		mog.db.RegisterCallback(mog, "OnProfileChanged", "LoadSettings")
		mog.db.RegisterCallback(mog, "OnProfileCopied", "LoadSettings")
		mog.db.RegisterCallback(mog, "OnProfileReset", "LoadSettings")

		if not mog.db.global.version then
			mog:Error(L["MogIt has loaded! Type \"/mog\" to open it."]);
		end
		mog.db.global.version = GetAddOnMetadata(MogIt,"Version");
		
		mog.LDBI:Register("MogIt",mog.mmb,mog.db.profile.minimap);
		
		
		for name,module in pairs(mog.moduleList) do
			if module.MogItLoaded then
				module:MogItLoaded()
			end
		end
		
		C_TransmogCollection.SetShowMissingSourceInItemTooltips(mog.db.profile.alwaysShowCollected)

		if mog.db.profile.loadModulesDefault then
			mog:LoadBaseModules()
		end
	elseif mog.modules[addon] then
		mog:LoadDB(addon)
		mog.modules[addon].loaded = true;
		if mog.menu.active == mog.menu.modules then
			mog.menu:Rebuild(1)
		end
	elseif addon == "Blizzard_Collections" then
		for i, model in ipairs(WardrobeCollectionFrame.ItemsCollectionFrame.Models) do
			model:SetScript("OnMouseDown", function(self, button)
				if IsControlKeyDown() and button == "RightButton" then
					local link
					local sources = WardrobeCollectionFrame_GetSortedAppearanceSources(self.visualInfo.visualID)
					if WardrobeCollectionFrame.tooltipSourceIndex then
						local index = WardrobeUtils_GetValidIndexForNumSources(WardrobeCollectionFrame.tooltipSourceIndex, #sources)
						link = select(6, C_TransmogCollection.GetAppearanceSourceInfo(sources[index].sourceID))
					end
					mog:AddToPreview(link)
					return
				end
				self:OnMouseDown(button)
			end)
		end
		local orig_OnMouseUp = WardrobeCollectionFrame.SetsCollectionFrame.ScrollFrame.buttons[1]:GetScript("OnMouseUp")
		for i, button in ipairs(WardrobeCollectionFrame.SetsCollectionFrame.ScrollFrame.buttons) do
			button:SetScript("OnMouseUp", function(self, button)
				if IsControlKeyDown() and button == "RightButton" then
					local preview = mog:GetPreview()
					for source in pairs(C_TransmogSets.GetSetSources(self.setID)) do
						mog:AddToPreview(select(6, C_TransmogCollection.GetAppearanceSourceInfo(source)), preview)
					end
					return
				end
				orig_OnMouseUp(self, button)
			end)
		end
		-- WardrobeCollectionFrame.SetsCollectionFrame.DetailsFrame.itemFramesPool.resetterFunc = function(self, obj) obj:RegisterForDrag("LeftButton", "RightButton") end
	end
end


local SLOTS = {
	[LE_TRANSMOG_COLLECTION_TYPE_HEAD] = "Head",
	[LE_TRANSMOG_COLLECTION_TYPE_SHOULDER] = "Shoulder",
	[LE_TRANSMOG_COLLECTION_TYPE_BACK] = "Back",
	[LE_TRANSMOG_COLLECTION_TYPE_CHEST] = "Chest",
	[LE_TRANSMOG_COLLECTION_TYPE_SHIRT] = "Shirt",
	[LE_TRANSMOG_COLLECTION_TYPE_TABARD] = "Tabard",
	[LE_TRANSMOG_COLLECTION_TYPE_WRIST] = "Wrist",
	[LE_TRANSMOG_COLLECTION_TYPE_HANDS] = "Hands",
	[LE_TRANSMOG_COLLECTION_TYPE_WAIST] = "Waist",
	[LE_TRANSMOG_COLLECTION_TYPE_LEGS] = "Legs",
	[LE_TRANSMOG_COLLECTION_TYPE_FEET] = "Feet",
	[LE_TRANSMOG_COLLECTION_TYPE_WAND] = "Wand",
	[LE_TRANSMOG_COLLECTION_TYPE_1H_AXE] = "1H-axe",
	[LE_TRANSMOG_COLLECTION_TYPE_1H_SWORD] = "1H-sword",
	[LE_TRANSMOG_COLLECTION_TYPE_1H_MACE] = "1H-mace",
	[LE_TRANSMOG_COLLECTION_TYPE_DAGGER] = "Dagger",
	[LE_TRANSMOG_COLLECTION_TYPE_FIST] = "Fist",
	[LE_TRANSMOG_COLLECTION_TYPE_SHIELD] = "Shield",
	[LE_TRANSMOG_COLLECTION_TYPE_HOLDABLE] = "Holdable",
	[LE_TRANSMOG_COLLECTION_TYPE_2H_AXE] = "2H-axe",
	[LE_TRANSMOG_COLLECTION_TYPE_2H_SWORD] = "2H-sword",
	[LE_TRANSMOG_COLLECTION_TYPE_2H_MACE] = "2H-mace",
	[LE_TRANSMOG_COLLECTION_TYPE_STAFF] = "Staff",
	[LE_TRANSMOG_COLLECTION_TYPE_POLEARM] = "Polearm",
	[LE_TRANSMOG_COLLECTION_TYPE_BOW] = "Bow",
	[LE_TRANSMOG_COLLECTION_TYPE_GUN] = "Gun",
	[LE_TRANSMOG_COLLECTION_TYPE_CROSSBOW] = "Crossbow",
	[LE_TRANSMOG_COLLECTION_TYPE_WARGLAIVES] = "Warglaives",
}

local SLOT_MODULES = {
	[LE_TRANSMOG_COLLECTION_TYPE_BACK] = "Other",
	[LE_TRANSMOG_COLLECTION_TYPE_SHIRT] = "Other",
	[LE_TRANSMOG_COLLECTION_TYPE_TABARD] = "Other",
	[LE_TRANSMOG_COLLECTION_TYPE_WAND] = "Ranged",
	[LE_TRANSMOG_COLLECTION_TYPE_1H_AXE] = "OneHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_1H_SWORD] = "OneHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_1H_MACE] = "OneHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_DAGGER] = "OneHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_FIST] = "OneHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_SHIELD] = "Other",
	[LE_TRANSMOG_COLLECTION_TYPE_HOLDABLE] = "Other",
	[LE_TRANSMOG_COLLECTION_TYPE_2H_AXE] = "TwoHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_2H_SWORD] = "TwoHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_2H_MACE] = "TwoHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_STAFF] = "TwoHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_POLEARM] = "TwoHanded",
	[LE_TRANSMOG_COLLECTION_TYPE_BOW] = "Ranged",
	[LE_TRANSMOG_COLLECTION_TYPE_GUN] = "Ranged",
	[LE_TRANSMOG_COLLECTION_TYPE_CROSSBOW] = "Ranged",
	[LE_TRANSMOG_COLLECTION_TYPE_WARGLAIVES] = "OneHanded",
}

mog.relevantCategories = {}

function mog:TRANSMOG_SEARCH_UPDATED()
	-- local t = debugprofilestop()
	
	local ARMOR_CLASSES = {
		WARRIOR = "Plate",
		DEATHKNIGHT = "Plate",
		PALADIN = "Plate",
		MONK = "Leather",
		PRIEST = "Cloth",
		SHAMAN = "Mail",
		DRUID = "Leather",
		ROGUE = "Leather",
		MAGE = "Cloth",
		WARLOCK = "Cloth",
		HUNTER = "Mail",
		DEMONHUNTER = "Leather",
	}
	
	local FACTIONS = {
		["Alliance"] = 1,
		["Horde"] = 2,
		-- hack for neutral pandaren, the items they can see are for both factions
		["Neutral"] = 3,
	}
	
	local _, playerClass = UnitClass("player")
	local faction = UnitFactionGroup("player")
	
	local armorClass = ARMOR_CLASSES[playerClass]
	
	mog.relevantCategories[armorClass] = true
	
	LoadAddOn("MogIt_"..armorClass)
	LoadAddOn("MogIt_Other")
	LoadAddOn("MogIt_OneHanded")
	LoadAddOn("MogIt_TwoHanded")
	LoadAddOn("MogIt_Ranged")
	
	local ArmorDB = _G["MogIt_"..armorClass.."DB"] or {}
	MogIt_OtherDB = MogIt_OtherDB or {}
	MogIt_OneHandedDB = MogIt_OneHandedDB or {}
	MogIt_TwoHandedDB = MogIt_TwoHandedDB or {}
	MogIt_RangedDB = MogIt_RangedDB or {}
	
	_G["MogIt_"..armorClass.."DB"] = ArmorDB
	
	local GetAppearanceSources = C_TransmogCollection.GetAppearanceSources
	local GetAppearanceSourceDrops = C_TransmogCollection.GetAppearanceSourceDrops
	local bor = bit.bor
	
	for i = 1, NUM_LE_TRANSMOG_COLLECTION_TYPES do
		local name, isWeapon, canEnchant, canMainHand, canOffHand = C_TransmogCollection.GetCategoryInfo(i)
		if name then
			name = SLOTS[i]
			local db = db
			if isWeapon then
				mog.relevantCategories[name] = true
			end
			if SLOT_MODULES[i] then
				db = _G["MogIt_"..SLOT_MODULES[i].."DB"]
			else
				db = ArmorDB
			end
			db[name] = db[name] or {}
			for i, appearance in ipairs(C_TransmogCollection.GetCategoryAppearances(i)) do
				if not appearance.isHideVisual then
					local v = db[name][appearance.visualID] or {}
					db[name][appearance.visualID] = v
					if v[1] and v[1].sourceID then
						db[name][appearance.visualID] = {}
					end
					for i, source in ipairs(GetAppearanceSources(appearance.visualID)) do
						local s = v[source.sourceID] or {}
						v[source.sourceID] = s
						s.sourceType = source.sourceType
						s.drops = GetAppearanceSourceDrops(source.sourceID)
						s.classes = bor(s.classes or 0, L.classBits[playerClass])
						s.faction = bor(s.faction or 0, FACTIONS[faction])
					end
				end
			end
		end
	end
	
	self:LoadDB("MogIt_"..armorClass)
	self:LoadDB("MogIt_Other")
	self:LoadDB("MogIt_OneHanded")
	self:LoadDB("MogIt_TwoHanded")
	self:LoadDB("MogIt_Ranged")
	
	self.frame:UnregisterEvent("TRANSMOG_SEARCH_UPDATED")
	
	-- print(format("MogIt modules loaded in %d ms.", debugprofilestop() - t))
end


function mog:LoadDB(addon)
	if not IsAddOnLoaded(addon) then return end
	local SOURCE_TYPES = {
		[1] = 1,
		[2] = 3,
		[3] = 4,
		[4] = 1,
		[5] = 6,
		[6] = 5,
	}
	
	local module = mog:GetModule(addon)
	local moduleDB = _G[addon.."DB"]
	
	-- won't exist if module was never loaded
	if not moduleDB then return end
	
	for slot, appearances in pairs(moduleDB) do
		local list = {}
		module.slots[slot] = {
			label = slot,
			list = list,
		}
		wipe(module.slotList)
		for visualID, appearance in pairs(appearances) do
			for sourceID, source in pairs(appearance) do
				local id = source.sourceID or sourceID
				tinsert(list, id)
				mog:AddData("item", id, "display", visualID)
				-- mog:AddData("item", id, "level", lvl)
				mog:AddData("item", id, "faction", source.faction)
				mog:AddData("item", id, "class", source.classes)
				mog:AddData("item", id, "source", SOURCE_TYPES[source.sourceType])
				-- mog:AddData("item", id, "sourceid", sourceid)
				mog:AddData("item", id, "sourceinfo", source.drops)
				-- mog:AddData("item", id, "zone", zone)
			end
		end
	end
	
	for i = 1, NUM_LE_TRANSMOG_COLLECTION_TYPES do
		local slotID = SLOTS[i]
		if moduleDB[slotID] then
			tinsert(module.slotList, slotID)
		end
	end
end


function mog:TRANSMOG_COLLECTION_SOURCE_ADDED(sourceID)
end


function mog:PLAYER_LOGIN()
	C_Timer.After(1, function()
		-- this function doesn't yield correct results immediately, so we delay it
		for slot, v in pairs(mog.mogSlots) do
			local isTransmogrified, _, _, _, _, _, _, visibleItemID = C_Transmog.GetSlotInfo(slot, LE_TRANSMOG_TYPE_APPEARANCE);
			if isTransmogrified then
				-- we need an item ID here if we still need to cache these at all
				-- mog:GetItemInfo(visibleItemID);
			end
		end
	end)
	
	for k, slot in pairs(SLOTS) do
		local name = C_TransmogCollection.GetCategoryInfo(k)
		if name then
			mog.db.profile.slotLabels[slot] = name
		end
	end
	
	mog:LoadSettings();
	self.frame:SetScript("OnSizeChanged", function(self, width, height)
		mog.db.profile.gridWidth = width;
		mog.db.profile.gridHeight = height;
		mog:UpdateGUI(true);
	end)
end

function mog:PLAYER_EQUIPMENT_CHANGED(slot, hasItem)
	local slotName = mog.mogSlots[slot];
	local item = GetInventoryItemLink("player", slot);
	if slotName then
		local baseSourceID, baseVisualID, appliedSourceID, appliedVisualID = C_Transmog.GetSlotVisualInfo(slot, LE_TRANSMOG_TYPE_APPEARANCE);
		local isTransmogrified, _, _, _, _, _, isHideVisual, texture = C_Transmog.GetSlotInfo(slot, LE_TRANSMOG_TYPE_APPEARANCE);
		if isTransmogrified then
			-- we need an item ID here if we still need to cache these at all
			-- mog:GetItemInfo(visibleItemID);
			item = appliedSourceID;
			itemAppearanceModID = visibleItemAppearanceModID;
		end
	end
	-- don't do anything if the slot is not visible (necklace, ring, trinket)
	if mog.db.profile.gridDress == "equipped" then
		for i, frame in ipairs(mog.models) do
			if frame.data.item then
				if hasItem then
					frame:TryOn(item, slotName, itemAppearanceModID);
				else
					frame:UndressSlot(slot);
				end
				frame:TryOn(frame.data.item);
			end
		end
	end
end
--//


--// Data API
mog.data = {};

function mog:AddData(data, id, key, value)
	if not (data and id and key) then return end;
	
	--if data == "item" then
	--	id = mog:ItemToString(id);
	--end
	
	if not mog.data[data] then
		mog.data[data] = {};
	end
	if not mog.data[data][key] then
		mog.data[data][key] = {};
	end
	mog.data[data][key][id] = value;
	return value;
end

function mog:DeleteData(data, id, key)
	if not mog.data[data] then return end;
	if id and key then
		mog.data[data][key][id] = nil;
	elseif id then
		for k,v in pairs(mog.data[data]) do
			v[id] = nil;
		end
	elseif key then
		mog.data[data][key] = nil;
	else
		mog.data[data] = nil;
	end
end

function mog:GetData(data, id, key)
	return mog.data[data] and mog.data[data][key] and mog.data[data][key][id];
end

mog.itemStringShort = "item:%d:0";
mog.itemStringLong = "item:%d:0::::::::::%d:1:%d";

function mog:ToStringItem(id, bonus, diff)
	-- itemID, enchantID, instanceDifficulty, numBonusIDs, bonusID1
	if (bonus and bonus ~= 0) or (diff and diff ~= 0) then
		return format(mog.itemStringLong, id, diff or 0, bonus or 0);
	else
		return format(mog.itemStringShort, id);
	end
end

local bonusDiffs = {
	-- MoP
	[451] = true, -- Raid Finder
	[449] = true, -- Heroic (Raid)
	[450] = true, -- Mythic (Raid)
	-- WoD
	[518] = true, -- dungeon-level-up-1
	[519] = true, -- dungeon-level-up-2
	[520] = true, -- dungeon-level-up-3
	[521] = true, -- dungeon-level-up-4
	[522] = true, -- dungeon-normal
	[524] = true, -- dungeon-heroic
	[525] = true, -- trade-skill (tier 1)
	[526] = true, -- trade-skill (armor tier 2)
	[527] = true, -- trade-skill (armor tier 3)
	[558] = true, -- trade-skill (weapon tier 2)
	[559] = true, -- trade-skill (weapon tier 3)
	[566] = true, -- raid-heroic
	[567] = true, -- raid-mythic
	[593] = true, -- trade-skill (armor tier 4)
	[594] = true, -- trade-skill (weapon tier 4)
	[615] = true, -- timewalker
	[617] = true, -- trade-skill (armor tier 5)
	[618] = true, -- trade-skill (armor tier 6)
	[619] = true, -- trade-skill (weapon tier 5)
	[620] = true, -- trade-skill (weapon tier 6)
	[642] = true, -- dungeon-mythic
	[648] = true, -- baleful (675)
	[651] = true, -- baleful empowered (695)
	[1798] = true, -- ???
	[1799] = true, -- ???
	[1805] = true, -- raid-heroic
	[1806] = true, -- raid-mythic
	[3379] = true, -- ???
	[3444] = true, -- ???
	[3445] = true, -- ???
	[3446] = true, -- ???
	
	[3524] = true, -- magical bonus ID for items that instead use the instance difficulty ID parameter
};

mog.itemStringPattern = "item:(%d+):%d*:%d*:%d*:%d*:%d*:%d*:%d*:%d*:%d*:%d*:(%d*):%d*:([%d:]+)";

function mog:ToNumberItem(item)
	if type(item) == "string" then
		local id, diff, bonus = item:match(mog.itemStringPattern);
		-- bonus ID can also be warforged, socketed, etc
		-- if there is more than one bonus ID, need to check all
		if bonus then
			if not tonumber(bonus) then
				for bonusID in gmatch(bonus, "%d+") do
					if bonusDiffs[tonumber(bonusID)] then
						bonus = bonusID;
						break;
					end
				end
			elseif not bonusDiffs[tonumber(bonus)] then
				bonus = nil;
			end
		end
		id = id or item:match("item:(%d+)");
		return tonumber(id), tonumber(bonus), tonumber(diff);
	elseif type(item) == "number" then
		return item;
	end
end

function mog:NormaliseItemString(item)
	return self:ToStringItem(self:ToNumberItem(item));
end
--//


--// Slot Conversion
mog.slots = {
	"HeadSlot",
	"ShoulderSlot",
	"BackSlot",
	"ChestSlot",
	"ShirtSlot",
	"TabardSlot",
	"WristSlot",
	"HandsSlot",
	"WaistSlot",
	"LegsSlot",
	"FeetSlot",
	"MainHandSlot",
	"SecondaryHandSlot",
};

mog.slotsType = {
	INVTYPE_HEAD = "HeadSlot",
	INVTYPE_SHOULDER = "ShoulderSlot",
	INVTYPE_CLOAK = "BackSlot",
	INVTYPE_CHEST = "ChestSlot",
	INVTYPE_ROBE = "ChestSlot",
	INVTYPE_BODY = "ShirtSlot",
	INVTYPE_TABARD = "TabardSlot",
	INVTYPE_WRIST = "WristSlot",
	INVTYPE_HAND = "HandsSlot",
	INVTYPE_WAIST = "WaistSlot",
	INVTYPE_LEGS = "LegsSlot",
	INVTYPE_FEET = "FeetSlot",
	INVTYPE_2HWEAPON = "MainHandSlot",
	INVTYPE_WEAPON = "MainHandSlot",
	INVTYPE_WEAPONMAINHAND = "MainHandSlot",
	INVTYPE_WEAPONOFFHAND = "SecondaryHandSlot",
	INVTYPE_RANGED = "MainHandSlot",
	INVTYPE_RANGEDRIGHT = "MainHandSlot",
	INVTYPE_SHIELD = "SecondaryHandSlot",
	INVTYPE_HOLDABLE = "SecondaryHandSlot",
};

-- all slot IDs that can be transmogrified
mog.mogSlots = {
	[INVSLOT_HEAD] = "HeadSlot",
	[INVSLOT_SHOULDER] = "ShoulderSlot",
	[INVSLOT_BACK] = "BackSlot",
	[INVSLOT_CHEST] = "ChestSlot",
	[INVSLOT_BODY] = "ShirtSlot",
	[INVSLOT_TABARD] = "TabardSlot",
	[INVSLOT_WRIST] = "WristSlot",
	[INVSLOT_HAND] = "HandsSlot",
	[INVSLOT_WAIST] = "WaistSlot",
	[INVSLOT_LEGS] = "LegsSlot",
	[INVSLOT_FEET] = "FeetSlot",
	[INVSLOT_MAINHAND] = "MainHandSlot",
	[INVSLOT_OFFHAND] = "SecondaryHandSlot",
}

function mog:GetSlot(id)
	return mog.slots[id] or mog.slotsType[id];
end
--//
