--[[
MobileVault
Author: Michael Joseph Murray aka Lyte of Lothar(US)
Please see license.txt for details.
$Revision: 240 $
$Date: 2015-04-02 16:21:00 +0000 (Thu, 02 Apr 2015) $
Project Version: 3.3.01
contact: codemaster2010 AT gmail DOT com

Copyright (c) 2007-2011 Michael J. Murray aka Lyte of Lothar(US)
All rights reserved unless otherwise explicitly stated.
]]

MobileVault = LibStub("AceAddon-3.0"):NewAddon("MobileVault", "AceConsole-3.0", "AceEvent-3.0", "AceTimer-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("MobileVault")

--deal with keybinding stuff
BINDING_HEADER_MOBILEVAULT = "Mobile Vault"
BINDING_NAME_MGVTOGGLE = L["Toggle Frame"]

--create our LDB launcher
LibStub:GetLibrary("LibDataBroker-1.1"):NewDataObject("MobileVault", {
	type = "launcher",
	label = "MobileVault",
	OnClick = function(_, button)
		if button == "LeftButton" then
			--open vault frame
			MobileVault:ShowVaultFrame()
		elseif button == "RightButton" then
			--open config
			LoadAddOn("MobileVault_Options")
			LibStub("AceConfigDialog-3.0"):Open("MobileVault")
		end
	end,
	icon = [[Interface\Icons\achievement_guildperk_mobilebanking]],
	OnTooltipShow = function(tooltip)
		if not tooltip or not tooltip.AddLine then return end
		tooltip:AddLine("MobileVault")
		tooltip:AddLine("|cffffff00" .. L["Click|r to toggle the Vault window"])
		tooltip:AddLine("|cffffff00" .. L["Right-click|r to open the options menu"])
	end,
})

function MobileVault:OnInitialize()
	local defaults = {
		profile = {
			position = {},
			guildTooltips = {
				['*'] = true,
			},
			frameColor = {0, 0, 0},
			highlightColor = {0, 0, 1},
			itemBorderColor = {1, 1, 1},
			selectedColor = {0, 1, 1},
			tabBorderColor = {1, 1, 1},
			textColor = {1, 1, 1},
			frameAlpha = 1,
			tooltipColor = "eda55f",
			itemPadding = 0,
			tabButtonSpacing = 4,
			strata = "LOW",
			scale = 1,
		},
		realm = {
			['*'] = { --guildName as key
				['*'] = { --page # as key
					links = {},
					stacks = {},
					name = "UNKNOWN",
					texture = "UNKNOWN",
					canView = true,
				},
				moneyLog = {},
				hasData = false,
			},
		},
	}
	
	self.db = LibStub("AceDB-3.0"):New("MobileVaultDB30", defaults, "Default")
	self.guilds = self.db.realm
	self.db.RegisterCallback(self, "OnProfileReset", "Update")
	self.db.RegisterCallback(self, "OnProfileCopied", "Update")
	self.db.RegisterCallback(self, "OnProfileChanged", "Update")
	
	_G["SlashCmdList"]["MOBILEVAULT_MAIN"] = function(s)
		if string.lower(s) == "show" then
			--open vault frame
			MobileVault:ShowVaultFrame()
		else
			LoadAddOn("MobileVault_Options")
			LibStub("AceConfigDialog-3.0"):Open("MobileVault")
		end
	end
	_G["SLASH_MOBILEVAULT_MAIN1"] = "/mgv"
	_G["SLASH_MOBILEVAULT_MAIN2"] = "/mobilevault"
	
	
	local spy = CreateFrame("FRAME", nil, UIParent)
	spy.name = "MobileVault"
	spy.addon = "MobileVault_Options"
	spy:Hide()
	spy:SetScript("OnShow", function(self)
		--remove the dummy entry
		for i, f in ipairs(INTERFACEOPTIONS_ADDONCATEGORIES) do
			if f == self.name or f.name == self.name then
				tremove(INTERFACEOPTIONS_ADDONCATEGORIES, i)
				break
			end
		end
		self:Hide()
		
		--load the config
		LoadAddOn(self.addon)
		
		--refresh the screen
		InterfaceOptionsFrame_OpenToCategory(self.name)
	end)
	InterfaceOptions_AddCategory(spy)
end

function MobileVault:OnEnable()
	self:RegisterEvent("GUILDBANKFRAME_OPENED")
	self:RegisterEvent("GUILDBANK_UPDATE_MONEY")
	self:RegisterEvent("GUILDBANK_UPDATE_TABS")
	self:RegisterEvent("GUILDBANKBAGSLOTS_CHANGED", "ScanPage")
	self:RegisterEvent("PLAYER_GUILD_UPDATE", "ObtainGuildName")
	
	self.activePage = 0
	self:ScheduleTimer("ObtainGuildName", 0.4)
end

function MobileVault:Update()
	if not self.frame then return end
	--set the main frame properties
	self.frame:SetWidth((600 + (13*self.db.profile.itemPadding)) * self.db.profile.scale)
	self.frame:SetHeight((375 + (6*self.db.profile.itemPadding)) * self.db.profile.scale)
	self.frame.actionbutton:SetWidth(100*self.db.profile.scale)
	self.frame.actionbutton:SetHeight(22*self.db.profile.scale)
	self.frame.text:SetWidth(600*self.db.profile.scale)
	self.frame.text:SetHeight(20*self.db.profile.scale)
	self.frame.subtext:SetWidth(600*self.db.profile.scale)
	self.frame.subtext:SetHeight(20*self.db.profile.scale)
	self.frame.scan:SetHeight(20*self.db.profile.scale)
	self.frame.gold:SetHeight(20*self.db.profile.scale)
	self.frame:SetFrameStrata(self.db.profile.strata)
	self.frame:SetAlpha(self.db.profile.frameAlpha)
	
	--set the main frame position
	if self.db.profile.position.x then
		self.frame:ClearAllPoints()
		self.frame:SetPoint(self.db.profile.position.point, UIParent, self.db.profile.position.anchor, self.db.profile.position.x, self.db.profile.position.y)
	else
		self.frame:SetPoint("CENTER", UIParent, "CENTER", 0, 200)
	end
	
	--set the main frame colors
	self.frame:SetBackdropColor(unpack(self.db.profile.frameColor))
	self.frame.text:SetTextColor(unpack(self.db.profile.textColor))
	self.frame.subtext:SetTextColor(unpack(self.db.profile.textColor))
	self.frame.scan:SetTextColor(unpack(self.db.profile.textColor))
	self.frame.gold:SetTextColor(unpack(self.db.profile.textColor))
	
	--iterate the item slots
	for i = 1, 98 do
		--sizes
		self.frame.items[i]:SetWidth(40*self.db.profile.scale)
		self.frame.items[i]:SetHeight(40*self.db.profile.scale)
		self.frame.items[i].tex:SetWidth(32*self.db.profile.scale)
		self.frame.items[i].tex:SetHeight(32*self.db.profile.scale)
		self.frame.items[i].count:SetWidth(32*self.db.profile.scale)
		self.frame.items[i].count:SetHeight(32*self.db.profile.scale)
		
		--colors
		self.frame.items[i]:SetBackdropBorderColor(unpack(self.db.profile.itemBorderColor))
		self.frame.items[i].count:SetTextColor(unpack(self.db.profile.textColor))
		
		self.frame.items[i]:ClearAllPoints()
		--adjust padding
		if i == 1 then
			self.frame.items[i]:SetPoint("TOPLEFT", self.frame, "TOPLEFT", 20, -60)
		else
			if (i % 7 > 1) or (i % 7 == 0) then
				self.frame.items[i]:SetPoint("TOP", self.frame.items[i-1], "BOTTOM", 0, -self.db.profile.itemPadding)
			elseif (i % 7 == 1) then
				self.frame.items[i]:SetPoint("LEFT", self.frame.items[i-7], "RIGHT", self.db.profile.itemPadding, 0)
			end
		end
	end
	--iterate the tab buttons
	for i = 1, MAX_GUILDBANK_TABS do
		--sizes
		self.frame.tabButtons[i]:SetWidth(40*self.db.profile.scale)
		self.frame.tabButtons[i]:SetHeight(40*self.db.profile.scale)
		self.frame.tabButtons[i].tab:SetWidth(32*self.db.profile.scale)
		self.frame.tabButtons[i].tab:SetHeight(32*self.db.profile.scale)
		
		self.frame.tabButtons[i]:ClearAllPoints()
		--adjust padding between buttons
		if i == 1 then
			self.frame.tabButtons[i]:SetPoint("LEFT", self.frame, "RIGHT", 3, 160)
		else
			self.frame.tabButtons[i]:SetPoint("TOP", self.frame.tabButtons[i-1], "BOTTOM", 0, -self.db.profile.tabButtonSpacing)
		end
		--colors
		self.frame.tabButtons[i]:SetBackdropBorderColor(unpack(self.db.profile.tabBorderColor))
		self.frame.tabButtons[i]:SetBackdropColor(unpack(self.db.profile.frameColor))
		
		--make sure the selected tab's color is updated too
		if self.frame.tabButtons[i].clicked then
			self.frame.tabButtons[i]:SetBackdropBorderColor(unpack(self.db.profile.selectedColor))
		end
	end
end

--take the given guild and page and show that page's items in the frame
function MobileVault:Populate(guild, page)
	if not self.guilds[guild] then return end
	if page == 0 then
		page = next(self.guilds[guild].links)
		self.frame.subtext:SetText(self.guilds[guild][page].name or L["Tab"] .. " "..page)
		self.frame.tabButtons[page].clicked = true
		self.frame.tabButtons[page]:SetBackdropBorderColor(unpack(self.db.profile.selectedColor))
		self.activePage = page
	end
	
	local itemName, itemLink, itemTexture
	for k, v in pairs(self.guilds[guild][page].links) do
		itemName, itemLink, _, _, _, _, _, _, _, itemTexture = GetItemInfo(v)
		self.frame.items[k].tex:SetTexture(itemTexture)
		self.frame.items[k].tex:SetTexCoord(0, 1, 0, 1)
		if self.guilds[guild][page].stacks[k] > 1 then
			self.frame.items[k].count:SetText(self.guilds[guild][page].stacks[k])
		end
		self.frame.items[k].item = itemLink
	end
end

--clear the currently displayed items
function MobileVault:Unpopulate()
	if self.frame.moneyLogFrame:IsVisible() then self.frame.moneyLogFrame:Hide() end
	for i = 1, 98 do
		self.frame.items[i].tex:SetTexture(nil)
		self.frame.items[i].count:SetText("")
		self.frame.items[i].item = nil
	end
end

function MobileVault:SetupTabs(guild)
	local numtabs = self.guilds[guild].numtabs
	for i = 1, numtabs do
		self.frame.tabButtons[i].tab:SetTexture(self.guilds[guild][i].texture)
		self.frame.tabButtons[i]:Show()
	end
	
	self.frame.moneyButton:Show()
end

function MobileVault:ObtainGuildName()
	if IsInGuild() then
		self.guildName = GetGuildInfo("player")
		self.activeGuild = self.guildName
	else
		self.guildName = next(self.guilds)
		self.activeGuild = self.guildName
	end
end

do
	local SEC_HOUR = 60*60
	local SEC_DAY = SEC_HOUR * 24
	local SEC_MONTH = SEC_DAY * 30
	local SEC_YEAR = SEC_DAY * 365
	function MobileVault:GetMoneyTransactions()
		local num = GetNumGuildBankMoneyTransactions()
		
		local trans, name, amount, years, months, days, hours
		local timestamp
		local now = time()
		for i = num, 1, -1 do
			trans, name, amount, years, months, days, hours = GetGuildBankMoneyTransaction(i)
			if not name then break end
			timestamp = now - (years * SEC_YEAR)
			timestamp = timestamp - (months * SEC_MONTH)
			timestamp = timestamp - (days * SEC_DAY)
			timestamp = timestamp - (hours * SEC_HOUR)
			
			self.guilds[self.guildName].moneyLog[i] = format("%s,%s,%d,%d", trans, name, amount, timestamp)
		end
	end
end

function MobileVault:GUILDBANKFRAME_OPENED()
	self.guilds[self.guildName].guildfunds = GetGuildBankMoney()
	
	QueryGuildBankLog(MAX_GUILDBANK_TABS+1)
	self:ScheduleTimer("GetMoneyTransactions", 1)
	
	self:GUILDBANK_UPDATE_TABS()
end

function MobileVault:GUILDBANK_UPDATE_MONEY()
	if not self.guildName then return end
	self.guilds[self.guildName].guildfunds = GetGuildBankMoney()
end

function MobileVault:GUILDBANK_UPDATE_TABS()
	if not self.guildName then return end
	self.guilds[self.guildName].numtabs = GetNumGuildBankTabs()
	
	--save the tab information
	local name, icon, view
	for page = 1, self.guilds[self.guildName].numtabs do
		name, icon, view = GetGuildBankTabInfo(page)
		
		self.guilds[self.guildName][page].texture = icon
		self.guilds[self.guildName][page].name = name
		self.guilds[self.guildName][page].canView = view
	end
end

local num, itemString
function MobileVault:ScanPage()
	if (GuildBankFrame and GuildBankFrame:IsVisible()) or (BagnonFrameguildbank and BagnonFrameguildbank:IsVisible()) then
		local page = GetCurrentGuildBankTab()
		--98 slots on a page
		for slot = 1, 98 do
			--see if the slot has an item in it
			if GetGuildBankItemLink(page, slot) then
				--if there is link, parse the ID and place that in table w/ key = slot
				_, _, itemString = string.find(GetGuildBankItemLink(page, slot), "^|c%x+|H(.+)|h%[.+%]")
				self.guilds[self.guildName][page].links[slot] = itemString
				
				--get item info too, need for stack sizes in image
				_, num = GetGuildBankItemInfo(page, slot)
				self.guilds[self.guildName][page].stacks[slot] = num
			else
				--if GetGuildBankItemLink returns nil the slot is empty
				--so nil the entry in the DB for the current slot
				self.guilds[self.guildName][page].links[slot] = nil
				self.guilds[self.guildName][page].stacks[slot] = nil
			end
			num, itemString = nil, nil
		end
		
		if not self.guilds[self.guildName].hasData then
			self.guilds[self.guildName].hasData = true
		end
		
		--save a timestamp per page
		self.guilds[self.guildName][page].lastScan = time()
	end
end

function MobileVault:GetDatestampString(guild, page)
	return date(L["date_format"], self.guilds[guild][page].lastScan)
end

function MobileVault:GetMoneyString(copper)
	local gold = (copper - (copper % 10000)) / 10000
	copper = copper % 10000
	local silver = (copper - (copper % 100)) / 100
	copper = copper % 100
	
	return string.format("%s|cff%s%s|r %s|cff%s%s|r %s|cff%s%s|r", gold, "ffd700", L["g"], silver, "c7c7cf", L["s"], copper, "eda55f", L["c"])
end

----------------------------------------------------------------------------------------------
--				TOOLTIP FUNCTIONS				     --
----------------------------------------------------------------------------------------------

local function AddItemCounts(frame, itemLink)
	local vaultID
	local _, _, itemString = strfind(itemLink, "^|c%x+|H(.+)|h%[.+%]")
	local color = MobileVault.db.profile.tooltipColor
	
	if itemString then
		local _, itemID = strsplit(":", itemString)
		for guild, data in pairs(MobileVault.guilds) do
			if MobileVault.db.profile.guildTooltips[guild] and (data.numtabs and type(data.numtabs) == "number") then
				local gBankCount = 0
				for page = 1, data.numtabs do
					for k, v in pairs(data[page].links) do
						_, vaultID = strsplit(":", v)
						if vaultID == itemID then
							gBankCount = gBankCount + data[page].stacks[k]
						end
					end
				end
				if gBankCount > 0 then
					frame:AddDoubleLine(format("|cff%s%s|r", color, guild), format("|cff%s%s %d|r", color, L["Vault:"], gBankCount))
				end
			end
		end
	end
	
	frame:Show()
end

local function HookIt(tt)
	tt:HookScript("OnTooltipSetItem", function(self, ...)
		local itemLink = select(2, self:GetItem())
		if itemLink and GetItemInfo(itemLink) then
			AddItemCounts(self, itemLink)
		end
	end)
end

HookIt(GameTooltip)
HookIt(ItemRefTooltip)
