--[[
	Whom is doing what with this library
	$Date: 2018-04-16 02:08:28 +0000 (Mon, 16 Apr 2018) $:
	$Revision: 27 $:
	$Author: myrroddin $:
	$URL: https://repos.wowace.com/wow/libutilities-1-0/trunk/LibUtilities-1.0/LibUtilities-1.0.lua $:
	$Id: LibUtilities-1.0.lua 27 2018-04-16 02:08:28Z myrroddin $:
	$Header: https://repos.wowace.com/wow/libutilities-1-0/trunk/LibUtilities-1.0/LibUtilities-1.0.lua 27 2018-04-16 02:08:28Z myrroddin $:
]]--

-- upvalue global constants for faster look up ------------
local _G = getfenv(0)
local string_match = _G.string.match
local string_split = _G.string.split
local string_format = _G.string.format
local string_gsub = _G.string.gsub
local string_sub = _G.string.sub
local string_trim = _G.string.trim
local string_find = _G.string.find
local to_number = _G.tonumber
local currResolution = _G.GetCurrentResolution
local getResolutions = _G.GetScreenResolutions
local f_mod = _G.math.fmod
local math_floor = _G.math.floor
local table_insert = _G.table.insert
local table_wipe = _G.wipe
local select = _G.select

local MAJOR = "LibUtilities-1.0"
local MINOR = to_number(string_match("$Revision: 27 $", "%d*")) or 90000
local lib, oldminor = LibStub:NewLibrary(MAJOR, MINOR)

if not lib then
	return -- no upgrade necessary
end

-- embedded functions beginning of file code
lib.embeds = lib.embeds or {}
--- Embeds lib on target AddOn
-- So you can call LibStub("LibUtilities-1.0").Embed(myAddOn)
-- @param target AddOn table in which to embed
-- @usage
-- local AddOnName, AddOn = ...
-- LibStub("LibUtilities-1.0"):Embed(AddOn)
-- -- OR, if using Ace3
-- -- you do not explicitly call :Embed
-- local MyAddOn = LibStub("AceAddon-3.0"):NewAddon("MyAddOn", "LibUtilities-1.0")
do
	local mixins = {
		"PixelPerfect",
		"VisualData",
		"NumberToHex",
		"DecodeItemLink",
		"Round",
		"TitleCase"
	}

	function lib:Embed(target)
		for k, v in pairs(mixins) do
			target[v] = self[v]
		end
		self.embeds[target] = true
		return target
	end
end -- end of do

--- Rounds a number to n decimal places
-- @paramsig number[, decimal]
-- @param number That you want rounded
-- @param decimal The number of decimal places
-- decimal defaults to 0 places
-- @return Your newly rounded number
-- @usage local myNumber = 123.456
-- myNumber = self:Round(myNumber, 2)
-- print(myNumber)
-- -- 123.46
function lib:Round(number, decimal)
	local mult = 10^(decimal or 0)
	return math_floor(number * mult + 0.5) / mult
end

-- local function, not external API -------------
local function titleCase(first, rest)
	return first:upper() .. rest:lower()
end

--- Creates Title Case for strings
-- @param str The string you want Title Cased
-- @return str now fixed
-- @usage local myString = "obi-wan kenobi"
-- myString = self:TitleCase(myString)
-- print(myString)
-- -- Obi-Wan Kenobi
function lib:TitleCase(str)
	str = string.gsub(str, "(%a)([%w_']*)", titleCase)
	return str
end

--- Scales an element to pixel perfection
-- @param number The height, width, or scale you want adjusted
-- @usage myFrame:SetHeight(myAddOn:PixelPerfect(100))
function lib:PixelPerfect(number)
	local uiScale = lib:VisualData()
	return uiScale * number
end

--- Useful visual data
-- Probably good to call if user changes screen resolutions
-- @usage local uiScale, resolutionX, resolutionY = myAddOn:VisualData()
-- @usage UIParent:SetScale(uiScale)
-- @return uiScale Number. The corrected UI scale (768/resolutionY)
-- @return resolutionX Number. Your screen's X dimension. IE// 1920
-- @return resolutionY Number. Your screen's Y dimension. IE// 1080
function lib:VisualData()
	local uiScale, resolutionX, resolutionY = 1, 800, 600
	local resolutionNow = resolutionNow or {}
	table_wipe(resolutionNow)
	resolutionNow = ({getResolutions()})[currResolution()]
	resolutionX, resolutionY = string_split("x", resolutionNow, 2)
	resolutionX, resolutionY = to_number(resolutionX), to_number(resolutionY)
	uiScale = 768/resolutionY
	return uiScale, resolutionX, resolutionY
end

--- Utility function to break down returns
-- for an item link like any other API
-- Note that numbers default to 0 rather than nil
-- if they normally would be empty strings ""
-- @param itemLink What you want parsed
-- can be a quest, talent, spell, enchant, or item link
-- @usage local bonusTable = select(17, myAddOn:DecodeItemLink(itemLink))
-- for i = 1, #bonusTable do
--     Print(i)
-- end
-- @return color String. "cff0070dd" as an example
-- @return linkType String. "item", "spell", "enchant", "talent", or "quest"
-- @return itemID Number suitable for searching Wowhead
-- @return enchantID Number. Permanent enchant applied to an item
-- See http://wow.gamepedia.com/EnchantId for a list
-- @return gemID1 Embedded gem, if socketed. Re-uses enchantID indicies
-- @return gemID2 Embedded gem, if socketed. Re-uses enchantID indicies
-- @return gemID3 Embedded gem, if socketed. Re-uses enchantID indicies
-- @return gemID4 Embedded gem, if socketed. Re-uses enchantID indicies
-- @return suffixID Number. Random enchantment ID. May be negative
-- See http://wow.gamepedia.com/SuffixId for a list
-- @return uniqueID Number. Data pertaining to a specific instance of the item
-- @return linkLevel Number. Level of the character supplying the link
-- @return specializationID Number. Specialization ID of the character supplying the link
-- See GetInspectSpecialization API for details
-- @return upgradeTypeID Number. Used with upgradeID
-- @return upgradeID Number.
-- @return instanceDifficultyID Number. Dungeon or raid difficultyID item was obtained
-- @return numBonuses Number. The number of bonusIDs
-- @return bonusIDs Table. A list of the bonuses by number or random suffix
function lib:DecodeItemLink(itemLink)
	local bonusIDs = bonusIDs or {}
	table_wipe(bonusIDs)

	local linkType, itemID, enchantID, gemID1, gemID2, gemID3, gemID4, suffixID, uniqueID, linkLevel, specializationID, upgradeTypeID, instanceDifficultyID, numBonuses, affixes = string_split(":", itemLink, 15)

	-- clean it up
	local color = string_match(linkType, "^c?f?f?(%x*)")
	linkType = string_gsub(linkType, "|?c?f?f?(%x*)|?H?", "")
	itemID = to_number(itemID) or 0
	enchantID = to_number(enchantID) or 0
	gemID1 = to_number(gemID1) or 0
	gemID2 = to_number(gemID2) or 0
	gemID3 = to_number(gemID3) or 0
	gemID4 = to_number(gemID4) or 0
	suffixID = to_number(suffixID) or 0
	uniqueID = to_number(uniqueID) or 0
	linkLevel = to_number(linkLevel) or 0
	specializationID = to_number(specializationID) or 0
	upgradeTypeID = to_number(upgradeTypeID) or 0
	instanceDifficultyID = to_number(instanceDifficultyID) or 0
	numBonuses = to_number(numBonuses) or 0

	if numBonuses >= 1 then
		for i = 1, numBonuses do
			local bonusID = select(i, string_split(":", affixes))
			table_insert(bonusIDs, to_number(bonusID))
		end
	end

	-- more clean up
	local upgradeID = select(numBonuses + 1, string_split(":", affixes))
	upgradeID = string_match(upgradeID, "%d*")
	upgradeID = to_number(upgradeID) or 0

	return color, itemType, itemID, enchantID, gemID1, gemID2, gemID3, gemID4, suffixID, uniqueID, linkLevel, specializationID, upgradeTypeID, upgradeID, instanceDifficultyID, numBonuses, bonusIDs
end

--- Converts a number into hexidecimal
-- @param number Number. Like 1234
-- @return hexString String. Like "04d2"
function lib:NumberToHex(number)
	local hexString = "0123456789abcdef"
	local s = ""
	while number > 0 do
		local mod = f_mod(number, 16)
		s = string_sub(hexString, mod + 1, mod + 1) .. s
		number = math_floor(number / 16)
	end
	if s == "" then s = "0" end
	return s
end

for AddOn in pairs(lib.embeds) do
	lib:Embed(AddOn)
end
