local title = "LazyCurve"

local coreFrame = CreateFrame("Frame");
local core = LibStub("AceAddon-3.0"):NewAddon(coreFrame, title, "AceHook-3.0") 
LazyCurve = {}
LazyCurve.core = core

local gsub = gsub
local table = table
local pairs = pairs
local string_find = string.find
local strupper = strupper
local GetAchievementLink = GetAchievementLink

local prefix = "<LazyCurve> "
LazyCurveDB = {
    avertise = true
}
local downloadlink = "https://www.curseforge.com/wow/addons/lazycurve"

local defaultCurve = 12110 --latest curve
local defaultEdge = 12111 -- latest edge
local arenaMaster = 1174
local curveTable
local PvPTable
LazyCurve.promote = LazyCurveDB.advertise
LazyCurve.defaultCurve = defaultCurve
LazyCurve.defaultEdge = defaultEdge
LazyCurve.arenaMaster = arenaMaster
LazyCurve.curveTable = curveTable
LazyCurve.PvPTable = PvPTable

--[[
creating ordered_table:
tablevariable = ordered_table.new{
    key1, value1, (!!so not key=val; but key, val!!)
    key2, value2,
}
--]]
local masterTable = {}
LazyCurve.masterTable = masterTable

masterTable.raids = {
    "ABT",
    "TOS",
    "NH",
    "TOV",
    "EN",
}

masterTable.finalBoss = {
    ABT = "argus",
    TOS = "kiljaeden",
    NH = "guldan",
    TOV = "helya",
    EN = "xavius"
}

masterTable.ABT = { --first 3 are nm, curve and edge; rest is mythic
    12266, --nm/lfr argus kill
    12110, --curve
    12111, --edge
    11992, --garothi worldbreaker
    11993, --hounds
    11994, --high command
    11995, --portal keeper
    11996, --eonar
    11997, --imonar
    11998, --kin'garoth
    11999, -- varimathras
    12000, --coven
    12001, --aggramar
    12002, --argus
    12111, --edge
}

masterTable.TOS = { --first 3 are nm, curve and edge; rest is mythic
    11790, --nm/lfr KJ kill
    11874, --curve
    11875, --edge
    11767, --goroth
    11774, --demonic inq
    11775, --harja
    11776, --mistress
    11777, --sisters
    11778, --host
    11779, --maiden
    11780, --avatar
    11781, --KJ
    11875, -- edge
}

masterTable.NH = { --first 3 are nm, curve and edge; rest is mythic
    10839, --nm/lfr guldan
    11195, -- curve
    11192, --edge
    10840, --skorpyron
    10842, --chronomatic anomaly
    10843, --trilliax
    10844, --spellblade aluriel
    10845, --star augur
    10846, --botanist
    10847, --tichondrius
    10848, --krosus
    10849, --elisande
    10850, --guldan
    11192, --edge
}

masterTable.TOV = { --first 3 are nm, curve and edge; rest is mythic
    11394, --nm/lfr helya
    11581, -- curve
    11580, --edge
    11396, --odig
    11397, --guarm
    11398, --helya
    11580, --edge
}

masterTable.EN = { --first 3 are nm, curve and edge; rest is mythic
    10820, --nm/lfr xavius
    11194, --curve
    11191, --edge
    10821, --nythendra
    10822, --elerethe renferal
    10823, --il'gynoth
    10824, --ursoc
    10825, --dragons
    10826, --cenarius
    10827, --xavius
    11191, --edge
}

--PVP chievos
masterTable["2V2"] = {--just the rating
    399, --1550 rating
    400, --1750 rating
    401, --2000 rating
    1159, --2200 rating
}

masterTable["3V3"] = {--just the rating
    402, --1550 rating
    403, --1750 rating
    405, --2000 rating
    1160, --2200 rating
    5266, --2400 rating
    5267, --2700 rating
}

masterTable.ArenaWins = {--just the wincount
    397, --1 win
    408, --10 in a row
    398, --100
    875, --200
    876, --300
}

masterTable.ArenaTitles = {--top x % of players
    2090, --challenger
    2093, --rival
    2092, --duelist
    2091, --gladiator
}

masterTable.AllianceRBGPercentage = {--top x % of players
    --[[9995, 10104, 10120, WOD--]]11024, 11036, 11049, 11050, 12040, 12179, 12189, --soldier of the alliance http://www.wowhead.com/title=449/soldier-of-the-alliance#reward-from-achievement
    --[[9996, 10106, 10118, WOD--]]11022, 11034, 11045, 11054, 12039, 12175, 12195, --defender of the alliance http://www.wowhead.com/title=448/defender-of-the-alliance#reward-from-achievement
    --[[9997, 10108, 10116, WOD--]]11020, 11032, 11047, 11052, 12038, 12177, 12191, --guardian of the alliance http://www.wowhead.com/title=447/guardian-of-the-alliance#reward-from-achievement
    6942, --hero of the alliance (any season)
}

masterTable.HordeRBGPercentage = {--top x % of players
    --[[9998, 10105, 10121, WOD--]]11025, 11035, 11048, 11051, 12044, 12178, 12190, --soldier of the horde http://www.wowhead.com/title=452/soldier-of-the-horde#reward-from-achievement
    --[[10001, 10107, 10119, WOD--]]11023, 11033, 11044, 11055, 12043, 12174, 12194, --defender of the horde http://www.wowhead.com/title=451/defender-of-the-horde#reward-from-achievement
    --[[10000, 10109, 10117, WOD--]]11021, 11031, 11046, 11053, 12042, 12176, 12192, --guardian of the horde http://www.wowhead.com/title=450/guardian-of-the-horde#reward-from-achievement
    6941, --hero of the alliance (any season)
}

masterTable.AllianceRBGRating = {--just the rating
    5330, --private
    5331,
    5332, --sergeant
    5333,
    5335, --knight
    5336,
    5337, --knight-captain
    5359,
    5339, --lieutenant commander
    5340,
    5341, --marshal
    5357,
    5343, --grand marshal
}

masterTable.HordeRBGRating = {--just the rating
    5345, --scout
    5346,
    5347, --sergeant
    5348,
    5349, --stone guard
    5351,
    5352, --legionnaire
    5338,
    5353, --champion
    5354,
    5355, --general
    5342,
    5356, --high warlord
}

masterTable.AllianceRBGWins = {--just the wincount
    5268, --1 win
    5322, --10
    5327, --25
    5328, --75
    5823, --150
    5329, --300
}

masterTable.HordeRBGWins = {--just the wincount
    5269, --1 win
    5323, --10
    5324, --25
    5325, --75
    5824, --150
    5326, --300
}

function core:getHighestMythicAchievement(raidname)
    local retID = masterTable[raidname][3]
    if(masterTable[raidname] ~= nil) then
        for k, chievoID in pairs(masterTable[raidname]) do
            local _, _, _, completed, _ = GetAchievementInfo(chievoID)
            if(completed and k > 3) then
                retID = chievoID
            end
        end
    end
    return retID
end

function core:getCurveTable(filter)
    t = ordered_table.new{}
    if(filter == "nofilter" or filter == "raids" or filter == "all") then
        for k,raidname in pairs(masterTable.raids) do
            local _, chievoName, _, completed, _ = GetAchievementInfo(masterTable[raidname][2])
            if(not completed or filter == "nofilter") then -- don't show nm/lfr chievo unless using a KEYWORD or you don't have curve
                t[raidname.."normal"] = masterTable[raidname][1]
            end
            t[raidname.."curve"] = masterTable[raidname][2]
            t[raidname.."edge"] = masterTable[raidname][3]
            t[raidname.."mythic"] = core.getHighestMythicAchievement(self,raidname)
            
            --nickname
            if(filter == "nofilter") then
                t[masterTable.finalBoss[raidname].."normal"] = masterTable[raidname][1]
                t[masterTable.finalBoss[raidname].."curve"] = masterTable[raidname][2]
                t[masterTable.finalBoss[raidname].."edge"] = masterTable[raidname][3]
                t[masterTable.finalBoss[raidname].."mythic"] = core.getHighestMythicAchievement(self,raidname)
            end
        end
        if(filter == "nofilter") then
            t["curve"] = defaultCurve
            t["edge"] = defaultEdge
        end
    end
    
    if(filter == "nofilter" or filter == "dungeons" or filter == "all") then
        t["KEY15"] = 11162
        t["KEY10"] = 11185
        t["KEY5"] = 11184
        t["KEY2"] =11183
    end
    
    return t
end

function  core:getHighestPvPAchievement(category)
    local t
    local faction, _ = UnitFactionGroup("player") -- options: Horde, Alliance, nil
    if faction == nil then
        --neutral panda or major error
        print("There has been a major error; or you're playing a neutral panda. Go get yourself a faction and /reloadui! :)")
        return
    end
    if(category == "2V2") then
        t = masterTable["2V2"]
    elseif(category == "3V3") then
        t = masterTable["3V3"]
    elseif(category == "Arena%") then
        t = masterTable.ArenaTitles
    elseif(category == "ArenaWins") then
        t = masterTable.ArenaWins
    elseif(category == "RBGRating") then
        t = masterTable[faction.."RBGRating"]
    elseif(category == "RBGWins") then
        t = masterTable[faction.."RBGWins"]
    elseif(category == "RBG%") then
        t = masterTable[faction.."RBGPercentage"]
    end
    local retID = t[#t]
    --tprint(t)
    for k, chievoID in pairs(t) do
            local _, _, _, completed, _ = GetAchievementInfo(chievoID)
            if(completed) then
                retID = chievoID
            end
        end
        
        return retID
end

function core:getPvPTable(filter)
    t = ordered_table.new{}
    if(filter == "nofilter" or filter == "arena") then 
        t["Arena2V2"] = core.getHighestPvPAchievement(self, "2V2")
        t["Arena3V3"] = core.getHighestPvPAchievement(self, "3V3")
        t["ArenaTop"] = core.getHighestPvPAchievement(self, "Arena%")
        t["ArenaWins"] = core.getHighestPvPAchievement(self, "ArenaWins")
        t["ArenaMaster"] = arenaMaster
    end
    if(filter == "nofilter" or filter == "bg") then
        t["RBGRating"] = core.getHighestPvPAchievement(self, "RBGRating")
        t["RBGTop"] = core.getHighestPvPAchievement(self, "RBG%")
        t["RBGWins"] = core.getHighestPvPAchievement(self, "RBGWins")
    end
    return t
end

local raidActivityTable = {
    [483] = "ABT",
    [482] = "ABT",
    
    [492] = "TOS",
    [478] = "TOS",
    [479] = "TOS",
    
    [481] = "NH",
    [416] = "NH",
    [415] = "NH",
    
    [457] = "TOV",
    [480] = "TOV",
    [456] = "TOV",
    
    [468] = "EN",
    [414] = "EN",
    [413] = "EN",
}

local dungeonCatID = 2
local raidCatID = 3
local pvpCatID = {}
pvpCatID["RankedArena"] = 4
pvpCatID["Arena"] = 7
pvpCatID["BG"] = 8
pvpCatID["RankedBG"] = 9

-- Ace3 Functions
function core:OnInitialize()
    self:RawHook("SendChatMessage", true)
    self:RawHook("LFGListUtil_GetSearchEntryMenu", true)
    self:RawHook("BNSendWhisper", true)
    curveTable = core.getCurveTable(self, "nofilter")
    PvPTable = core.getPvPTable(self, "nofilter")
    
    if(LazyCurveDB == nil) then
        LazyCurveDB = {
            avertise = true
        }
    end
    if LazyCurveDB.advertise == nil then
        LazyCurveDB.advertise = true
    end
    print("LazyCurve: loaded")
end

-- update tools
local function tprint( data, level )
	level = level or 0
	local ident=strrep('    ', level)
	if level>5 then return end

	if type(data)~='table' then print(tostring(data)) return end;

	for index,value in pairs(data) do repeat
		if type(value)~='table' then
			print( ident .. '['..index..'] = ' .. tostring(value) .. ' (' .. type(value) .. ')' );
			break;
		end
		print( ident .. '['..index..'] = {')
        tprint(value, level+1)
        print( ident .. '}' );
	until true end
end

function core:getAllCategoryInfo() --/run LazyCurve.getAllCategoryInfo()
    local tempTable ={}; 
    for i=1,10 do 
        tempTable[i] = C_LFGList.GetCategoryInfo(i) 
    end;
    tprint(tempTable)
end

function core:getAllActivityInfo(dungeonCatID, raidCatID) --/run LazyCurve.getAllActivityInfo(_, dungeonCatID, raidCatID)
    local dungeonActivities = C_LFGList.GetAvailableActivities(dungeonCatID)
    for key, val in pairs(dungeonActivities) do
        fullName, shortName, catID, _, _, _, minLevel, _, displayType, _ = C_LFGList.GetActivityInfo(val)
        if minLevel >= 110 then
            print("activityID " .. val .. ":" .. fullName .. "(" .. shortName .. ") minlvl:" .. minLevel .. "; displaytype:" .. displayType)
        end
    end
    print("-------------")
    local raidActivities = C_LFGList.GetAvailableActivities(raidCatID)
    for key, val in pairs(raidActivities) do
        fullName, shortName, catID, _, _, _, minLevel, _, displayType, _ = C_LFGList.GetActivityInfo(val)
        if minLevel >= 110 then 
            print("activityID " .. val .. ":" .. fullName .. "(" .. shortName .. ") minlvl:" .. minLevel .. "; displaytype:" .. displayType)
        end
    end
end
-- end update tools

------------------------------------------------------------------
function core:processMsg(msg)
    msg, _ = gsub(msg, "LAZYCURVE", downloadlink)
    local unchanged = msg
    
    for keyword, chievoID in ordered_table.pairs(curveTable) do
        msg = core.replaceKeywordWithAchievementLink(self, msg, keyword, chievoID)
    end
    for keyword, chievoID in ordered_table.pairs(PvPTable) do
        msg = core.replaceKeywordWithAchievementLink(self, msg, keyword, chievoID)
    end
    if(unchanged ~= msg and LazyCurveDB.advertise) then
        msg = prefix .. msg
    end
    return msg
end

function core:BNSendWhisper(...)
    local id, msg = ...;
    msg = core.processMsg(self, msg)
    core.hooks.BNSendWhisper(id, msg)
end

function core:SendChatMessage(...)
    -- Our hook to SendChatMessage. If msg contains a tracked keyword, we insert the achievement.
    local msg, chatType, language, channel = ...;
    msg = core.processMsg(self, msg)
    core.hooks.SendChatMessage(msg, chatType, language, channel);
end

function core:replaceKeywordWithAchievementLink(msg, keyword, chievoID)
    keyword = strupper(keyword)
    if string_find(msg, keyword) then --keyword found
        local found, _ = string_find(msg, keyword)
        while(found ~= nil) do --repeat untill all keywords are replaced
            msg, _ = gsub(msg, keyword, GetAchievementLink(chievoID))
            found, _ = string_find(msg, keyword)
        end
    end
    return msg
end

function core:SendAchievement(name, chievoID)
    local msg = GetAchievementLink(chievoID)
    if(LazyCurveDB.advertise) then
        msg = prefix .. msg
    end
    core.hooks.SendChatMessage(msg, "WHISPER", nil, name)
end

function core:getFilteredTable(activityID, override)
    fullName, difficulty, catID, _, _, _, minLevel, _ = C_LFGList.GetActivityInfo(activityID)
    local supportedActivity = false
    if(catID == dungeonCatID) then --dungeon chievos
        if(minLevel == 110) then
            supportedActivity = true
            filteredTable = core.getCurveTable(self,"dungeons")
        end
    elseif(catID == raidCatID) then --raid chievos
        supportedActivity = true
        if(raidActivityTable[activityID] ~= nil) then
            filteredTable = ordered_table.new{
                raidActivityTable[activityID].."normal", curveTable[raidActivityTable[activityID].."normal"],
                raidActivityTable[activityID].."curve", curveTable[raidActivityTable[activityID].."curve"],
                raidActivityTable[activityID].."edge", curveTable[raidActivityTable[activityID].."edge"],
                raidActivityTable[activityID].."mythic", curveTable[raidActivityTable[activityID].."mythic"]
            }
            local _, chievoName, _, completed, _ = GetAchievementInfo(curveTable[raidActivityTable[activityID].."curve"])
            if(completed) then
                filteredTable[raidActivityTable[activityID].."normal"] = nil
            end
            if(curveTable[raidActivityTable[activityID].."curve"] ~= curveTable.curve)then
                filteredTable.curve = curveTable.curve
            end
            if(curveTable[raidActivityTable[activityID].."edge"] ~= curveTable.edge)then
                filteredTable.edge = curveTable.edge
            end
        else
            filteredTable = core.getCurveTable(self,"raids")
        end
    end
    
    if(not supportedActivity) then --check pvpCatID
        for catName, catNumber in pairs(pvpCatID) do
            if(catID == catNumber) then
                if(string_find(catName, "BG")) then
                    filteredTable = core.getPvPTable(self, "bg")
                else
                    filteredTable = core.getPvPTable(self, "arena")
                end
                return filteredTable
            end
        end
        
        filteredTable = core.getCurveTable(self,"all")
    end
    return filteredTable
end

function core:getExtraMenuList(activityID, leaderName)
    local tempMenuList = {}
    
    filteredTable = core.getFilteredTable(self, activityID)
    for keyword, chievoID in ordered_table.pairs(filteredTable) do
        local _, chievoName, _, completed, _ = GetAchievementInfo(chievoID)
        if(completed) then
            table.insert(tempMenuList, {
                text =chievoName,
                func = function(_, name) core.SendAchievement(self, name, chievoID); end,
                notCheckable = true,
                arg1 = leaderName, --Leader name goes here
                disabled = not leaderName, --Disabled if we don't have a leader name yet or you haven't applied
            })
            if(string_find(keyword, "KEY")) then
                break
            end
        end
    end
    if(table.getn(tempMenuList)>1) then
        return { 
                text = prefix .. "Link Achievement to Leader",
                hasArrow = true,
                notCheckable = true,
                disabled = not leaderName,
                menuList = tempMenuList,
        };
    end
    if(table.getn(tempMenuList) == 1) then
        tempMenuList[1].text = prefix .. "Link" .. tempMenuList[1].text .. "  to Leader"
        return tempMenuList[1]
    end
    if(table.getn(tempMenuList) == 0) then
        --no achievements
        return {
            text = prefix .. "You haven't completed any relevant achievements yet :(",
            notCheckable = true,
            disabled = true,
        }
    end
end

function core:LFGListUtil_GetSearchEntryMenu(resultID)
    local _, activityID, _, _, _, _, _, _, _, _, _, _, leaderName = C_LFGList.GetSearchResultInfo(resultID);
    local LFG_LIST_SEARCH_ENTRY_MENU = nil
    LFG_LIST_SEARCH_ENTRY_MENU = core.hooks.LFGListUtil_GetSearchEntryMenu(resultID)
    LFG_LIST_SEARCH_ENTRY_MENU[2].tooltipTitle = nil
    LFG_LIST_SEARCH_ENTRY_MENU[2].tooltipText = nil
    
    local found = false
    for i, item in pairs(LFG_LIST_SEARCH_ENTRY_MENU) do
         if(string_find(item.text, prefix)) then
                local menuIndex = i
                found = true
                break
            end
    
    end
    if(not found) then
        menuIndex = #LFG_LIST_SEARCH_ENTRY_MENU
        table.insert(LFG_LIST_SEARCH_ENTRY_MENU, menuIndex, core.getExtraMenuList(self, activityID, leaderName))
    else
        LFG_LIST_SEARCH_ENTRY_MENU[menuIndex] = core.getExtraMenuList(self, activityID, leaderName)
    end
    
    table.insert(LFG_LIST_SEARCH_ENTRY_MENU, lastItem) -- put the cancel button back in it's place
    return LFG_LIST_SEARCH_ENTRY_MENU; --and all is well that ends well :)
end

function core:printInfo()
    print("Type \"/lazycurve togglepromote\" to start/stop self-promotion.")
    print("Please leave this on if you like my addon")
end

SLASH_LAZYCURVE1="/lazycurve"
SlashCmdList["LAZYCURVE"] =
	function(msg)
		local a1, a2 = strsplit(" ", strlower(msg), 2)
        if (a1 == "") then 
            core.printInfo()
        elseif (a1 == "info")  then 
            core.printInfo()
        elseif(a1 == "togglepromote") then
            LazyCurveDB.advertise = (not LazyCurveDB.advertise)
            if(LazyCurveDB.advertise) then
                print(prefix .. "self-promotion when linking an achievement is now turned ON")
            else
                print(prefix .. "self-promotion when linking an achievement is now turned OFF")
                print("Please consider turning this back on if you like my addon :)")
            end
        end
    end   


