-- SavedVariables setup
LatencyDisplaySaved = LatencyDisplaySaved or {
    locked = false,
    showFPS = true,
    compactMode = false,
    chatChannel = "SAY",
    position = nil,
    scale = 1,
}

local frame = CreateFrame("Frame", "LatencyDisplayFrame", UIParent, BackdropTemplateMixin and "BackdropTemplate")
if BackdropTemplateMixin and not frame.SetBackdrop then
    Mixin(frame, BackdropTemplateMixin)
end

-- Frame setup
frame:SetSize(10, 10)
frame:SetPoint("CENTER")
frame:SetBackdrop({
    bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
    tile = true, tileSize = 16, edgeSize = 16,
    insets = { left = 4, right = 4, top = 4, bottom = 4 },
})
frame:SetBackdropColor(0, 0, 0, 0.8)
frame:SetBackdropBorderColor(0, 0, 0, 1)
frame:SetScale(LatencyDisplaySaved.scale or 1)

local text = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
text:SetPoint("CENTER")
text:SetTextColor(1, 1, 1, 1)

frame:SetMovable(true)
frame:EnableMouse(true)
frame:RegisterForDrag("LeftButton")

local clickStartTime, clickStartX, clickStartY

frame:SetScript("OnMouseDown", function(self, button)
    if button == "LeftButton" and IsShiftKeyDown() and not LatencyDisplaySaved.locked then
        clickStartTime = GetTime()
        clickStartX, clickStartY = GetCursorPosition()
        self:StartMoving()
    end
end)

frame:SetScript("OnMouseUp", function(self, button)
    if button == "LeftButton" and IsShiftKeyDown() and not LatencyDisplaySaved.locked then
        self:StopMovingOrSizing()

        local clickDuration = GetTime() - (clickStartTime or 0)
        local endX, endY = GetCursorPosition()
        local movedDistance = math.sqrt((endX - clickStartX)^2 + (endY - clickStartY)^2)

        if clickDuration < 0.2 and movedDistance < 5 then
            local _, _, latHome, latWorld = GetNetStats()
            local msg = format("My latency: Home %d ms, World %d ms", latHome, latWorld)
            SendChatMessage(msg, LatencyDisplaySaved.chatChannel:upper())
        else
            local point, _, relativePoint, xOfs, yOfs = self:GetPoint()
            LatencyDisplaySaved.position = {point, relativePoint, xOfs, yOfs}
        end
    end
end)

-- Hover opacity
frame:SetScript("OnEnter", function(self)
    self:SetAlpha(1)
    local _, _, latHome, latWorld = GetNetStats()
    local fps = GetFramerate()
    GameTooltip:SetOwner(self, "ANCHOR_CURSOR")
    GameTooltip:SetText("Performance Info")
    GameTooltip:AddLine("Home Latency: " .. latHome .. " ms", 1, 1, 1)
    GameTooltip:AddLine("World Latency: " .. latWorld .. " ms", 1, 1, 1)
    if LatencyDisplaySaved.showFPS then
        GameTooltip:AddLine("FPS: " .. math.floor(fps), 1, 1, 1)
    end
    GameTooltip:AddLine("Chat Channel: " .. LatencyDisplaySaved.chatChannel, 0.8, 0.8, 0.8)
    GameTooltip:AddLine("Shift + Drag to move", 0.5, 0.5, 0.5)
    GameTooltip:AddLine("Shift + Click to announce", 0.5, 0.5, 0.5)
    GameTooltip:AddLine("Right Click for menu", 0.5, 0.5, 0.5)
    GameTooltip:AddLine("Addon by BRHAM", 1, 1, 1)
    GameTooltip:Show()
end)

frame:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
    frame:SetAlpha(0.7)
end)

-- Update Display
function frame:UpdateDisplay()
    local _, _, latHome, latWorld = GetNetStats()
    local fps = GetFramerate()
    local colorHome, colorWorld

    if latHome < 100 then
        colorHome = {0,1,0}
    elseif latHome < 250 then
        colorHome = {1,1,0}
    else
        colorHome = {1,0,0}
    end

    if latWorld < 100 then
        colorWorld = {0,1,0}
    elseif latWorld < 250 then
        colorWorld = {1,1,0}
    else
        colorWorld = {1,0,0}
    end

    local displayText
    if LatencyDisplaySaved.compactMode then
        displayText = format("L:%d W:%d%s", latHome, latWorld, LatencyDisplaySaved.showFPS and format(" | FPS:%d", math.floor(fps)) or "")
    else
        displayText = format("By BRHAM — Home:%d | World:%d%s", latHome, latWorld, LatencyDisplaySaved.showFPS and format(" | FPS:%d", math.floor(fps)) or "")
    end

    text:SetText(displayText)
    text:SetTextColor(unpack(colorHome))

    local padding = 10
    local w = text:GetStringWidth() + padding*2
    local h = text:GetStringHeight() + padding*2
    frame:SetSize(w,h)
end

-- Update loop
frame:SetScript("OnUpdate", function(self, elapsed)
    self._timer = (self._timer or 0) + elapsed
    if self._timer >= 1 then
        self._timer = 0
        self:UpdateDisplay()
    end
end)

-- Restore position on login
frame:RegisterEvent("PLAYER_ENTERING_WORLD")
frame:SetScript("OnEvent", function(self)
    if LatencyDisplaySaved.position then
        self:SetPoint(unpack(LatencyDisplaySaved.position))
    end
    frame:SetScale(LatencyDisplaySaved.scale or 1)
    self:UpdateDisplay()
end)

-- Slash Commands
SLASH_LATENCYDISPLAY1 = "/latencydisplay"
SLASH_LATENCYDISPLAY2 = "/ld"
SlashCmdList["LATENCYDISPLAY"] = function(msg)
    msg = string.lower(msg or "")
    if msg == "reset" then
        frame:ClearAllPoints()
        frame:SetPoint("CENTER")
        LatencyDisplaySaved.position = nil
        print("LatencyDisplay: Position reset.")
    elseif msg == "hide" then
        frame:Hide()
        print("LatencyDisplay hidden.")
    elseif msg == "show" then
        frame:Show()
        print("LatencyDisplay shown.")
    elseif msg == "lock" then
        LatencyDisplaySaved.locked = true
        print("LatencyDisplay: Frame locked.")
    elseif msg == "unlock" then
        LatencyDisplaySaved.locked = false
        print("LatencyDisplay: Frame unlocked.")
    elseif msg == "fps" then
        LatencyDisplaySaved.showFPS = not LatencyDisplaySaved.showFPS
        print("LatencyDisplay: FPS display " .. (LatencyDisplaySaved.showFPS and "enabled." or "disabled."))
    elseif msg == "compact" then
        LatencyDisplaySaved.compactMode = not LatencyDisplaySaved.compactMode
        print("LatencyDisplay: Compact mode " .. (LatencyDisplaySaved.compactMode and "enabled." or "disabled."))
    elseif msg:match("^scale%s+") then
        local scale = tonumber(msg:match("^scale%s+([%d%.]+)"))
        if scale then
            LatencyDisplaySaved.scale = scale
            frame:SetScale(scale)
            print("LatencyDisplay: Scale set to " .. scale)
        end
    elseif msg:match("^channel%s+") then
        local channel = msg:match("^channel%s+(%a+)")
        if channel then
            LatencyDisplaySaved.chatChannel = string.upper(channel)
            print("LatencyDisplay: Chat channel set to " .. LatencyDisplaySaved.chatChannel)
        end
    else
        print("|cffffcc00LatencyDisplay Commands:")
        print("/ld show - Show the display")
        print("/ld hide - Hide the display")
        print("/ld reset - Reset position")
        print("/ld lock - Lock movement")
        print("/ld unlock - Unlock movement")
        print("/ld fps - Toggle FPS display")
        print("/ld compact - Toggle compact mode")
        print("/ld scale [number] - Set frame scale")
        print("/ld channel [say/party/guild/etc] - Set output channel")
    end
    frame:UpdateDisplay()
end
