----------------------------------------------------------------
-- KiwiPlates: core
----------------------------------------------------------------

local addon = KiwiPlates

local Media  = LibStub("LibSharedMedia-3.0", true)

local next    = next
local pairs   = pairs
local ipairs  = ipairs
local unpack  = unpack
local select  = select
local strsub  = strsub
local format  = string.format
local tinsert = table.insert
local tremove = table.remove
local tconcat = table.concat
local format  = string.format

local UNKNOWNOBJECT = UNKNOWNOBJECT
local UnitGUID = UnitGUID
local UnitName = UnitName
local UnitLevel = UnitLevel
local UnitClass = UnitClass
local UnitIsUnit = UnitIsUnit
local UnitHealth = UnitHealth
local UnitReaction = UnitReaction
local UnitHealthMax = UnitHealthMax
local UnitIsTapDenied = UnitIsTapDenied
local UnitClassification = UnitClassification
local CastingBarFrame_SetUnit = CastingBarFrame_SetUnit
local C_GetNamePlateForUnit  = C_NamePlate.GetNamePlateForUnit
local C_SetNamePlateSelfSize = C_NamePlate.SetNamePlateSelfSize
local DifficultyColor = addon.DIFFICULTY_LEVEL_COLOR
local IsInInstance = IsInInstance

local InstanceType
local InCombat
local pixelScale
local targetFrame
local targetExists
local mouseFrame

local GetPlateSkin
local ConditionFields = {}
local activeWidgets = {}
local activeStatuses = {}

local cfgAlpha1
local cfgAlpha2
local cfgAlpha3
local cfgAdjustAlpha
local cfgReactionColor       
local cfgHealthColor1
local cfgHealthColor2
local cfgHealthColor3
local cfgHealthThreshold1
local cfgHealthThreshold2
local cfgClassColorReaction = {} 

local NamePlates = {}
local NamePlatesByUnit = {}
local NamePlatesAll = {}

local Types = { a = 'Player', e = 'Creature', t = 'Pet', m = 'GameObject', h = 'Vehicle', g = 'Vignette' }

local Reactions = { 'hostile', 'hostile', 'hostile', 'neutral', 'friendly', 'friendly',	'friendly',	'friendly' }

local Classifications = { elite = '+', rare = 'r', rareelite = 'r+', boss = 'b' }

local ClassColors = { UNKNOWN = {1,1,1,1} }
for class,color in pairs(RAID_CLASS_COLORS) do
	ClassColors[class] = { color.r, color.g, color.b, 1 }
end

local ColorTransparent = { 0,0,0,0 }

local ColorBlack = { 0,0,0,1 }

local ColorWhite = { 1,1,1,1 }

local ColorDefault = {1,1,1,1}

local ColorWidgets  = {
	kHealthBar    = "Health Bar",
	kHealthBorder = "Health Border",
	kHealthText   = "Health Text",
	kLevelText    = "Level Text",
	kNameText     = "Name Text",
}

local ColorStatuses = {
	color    = "Custom Color",
	health   = "Health Percent",
	reaction = "Unit Reaction",
	class    = "Class Color",
	level    = "Unit Level",
}

local WidgetNames = { 
	'kHealthBar', 
	'kHealthBorder',
	'kHealthText', 
	'kNameText', 
	'kLevelText', 
	'ClassificationFrame',
	'RaidTargetFrame',
	kHealthBar    = 'kkHealthBar',
	kHealthBorder = 'kkHealthBorder',
	kHealthText   = 'kkHealthText',
	kNameText     = 'kkNameText',
	kLevelText    = 'kkLevelText',
	RaidTargetFrame = 'RaidTargetFrame',
	ClassificationFrame = 'ClassificationFrame',
}

local FontCache = setmetatable({}, {__index = function(t,k) local v = Media:Fetch('font',      k or 'Roboto Condensed Bold'); t[k or 'Roboto Condensed Bold'] = v; return v end})
local TexCache  = setmetatable({}, {__index = function(t,k) local v = Media:Fetch('statusbar', k or 'Minimalist');            t[k or 'Minimalist'] = v;            return v end})

----------------------------------------------------------------
-- 
----------------------------------------------------------------

addon.defaults = {
	general     = {
		highlight     = true,
		classColor    = {},
		healthColor   = { threshold1 = .9, threshold2 = .3, color1 = { .6,1,.8,1 }, color2 = { 1,1,1,1 }, color3 = { 1,.4,.3,1 } },
		reactionColor = { hostile = {.7,.2,.1,1}, neutral = {1,.8,0,1}, friendly = {.2,.6,.1,1}, tapped = {.5,.5,.5,1}, playerfriendly = {.2,.6,.1,1}, playerhostile = {.7,.2,.1,1}	},
	},
	skins       = { { __skinName = 'Default', kHealthBar_enabled = true, kHealthBorder_enabled = true, kNameText_enabled = true, kLevelText_enabled = true, kHealthText_enabled = true, RaidTargetFrame_enabled = true } },
	rules       = { { 'and'  } },
	minimapIcon = {},
}

----------------------------------------------------------------
-- Used to reparent disabled/unused textures
----------------------------------------------------------------

local HiddenFrame = CreateFrame("Frame") 
HiddenFrame:Hide()

----------------------------------------------------------------
-- Highlight texture 
----------------------------------------------------------------

local HighlightTex = HiddenFrame:CreateTexture(nil, "OVERLAY")
HighlightTex:SetColorTexture(1,1,1,.2)
HighlightTex:SetVertexColor(1,1,1,1)
HighlightTex:SetBlendMode('ADD')

----------------------------------------------------------------
-- Register media stuff early
----------------------------------------------------------------

Media:Register('font', 'Yanone Kaffesatz Bold', "Interface\\Addons\\KiwiPlates\\media\\yanone.ttf" )
Media:Register('font', 'FrancoisOne', "Interface\\Addons\\KiwiPlates\\media\\francois.ttf" )
Media:Register('font', 'Roboto Condensed Bold', "Interface\\Addons\\KiwiPlates\\media\\roboto.ttf" )
Media:Register("font", "Accidental Presidency", "Interface\\Addons\\KiwiPlates\\media\\accid___.ttf" )
Media:Register("statusbar", "Minimalist", "Interface\\Addons\\KiwiPlates\\media\\Minimalist")
Media:Register("statusbar", "Gradient", "Interface\\Addons\\KiwiPlates\\media\\gradient")
Media:Register("statusbar", "Blizzard Solid White", "Interface\\Buttons\\white8x8")
Media:Register("statusbar", "Blizzard NamePlate", "Interface\\TargetingFrame\\UI-TargetingFrame-BarFill")

----------------------------------------------------------------
-- Table that caches settings for each skin to update widgets 
-- colors & values, example:
-- WidgetUpdate[skin] = {
--   -- functions to update widgets texts and statusbars
--   [1] = WidgetMethods.kHealthBar, [2] = WidgetMethods.kLevelTetx, ...
--   -- user defined colors & active widgets
--	 ['kHealthBar'] = customColor1, ['kNameText']  = customColor2, ['ClassificationFrame'] = true, ...
--   -- statuses
--	 methods  = {	['kHealthBar'] = UpdateColorReaction, ['kLevelText'] = UpdateColorCustom, ['kNameText'] = UpdateColorCustom }, 
--	 reaction = { 'kHealthBar' },
--	 color    = { 'kNameText', 'kLevelText' }, -- color = customColor = statusName
-- } 
----------------------------------------------------------------

local WidgetUpdate = {}

----------------------------------------------------------------
-- Statuses color painting management
----------------------------------------------------------------

local ColorStatusDefaults = {
	kHealthBar    = 'reaction',
	kHealthBorder = 'color',
	kHealthText   = 'color',
	kNameText     = 'color',
	kLevelText    = 'level',
}

local ColorDefaults = {
	kHealthBorder = ColorBlack,
	kHealthBar    = ColorWhite,
	kHealthText   = ColorWhite,
	kNameText     = ColorWhite,
	kLevelText    = ColorWhite,
}

local ColorMethods = {
	blizzard = function(UnitFrame)
		return ColorTransparent
	end,
	reaction = function(UnitFrame)
		if UnitFrame.__type == "Player" then
			if cfgClassColorReaction[UnitFrame.__reaction] then
				return ClassColors[UnitFrame.__class] or ColorWhite
			elseif UnitFrame.__reaction == 'friendly' then
				return cfgReactionColor.playerfriendly or ColorWhite
			else
				return cfgReactionColor.playerhostile or ColorWhite
			end		
		else
			return (UnitIsTapDenied(UnitFrame.unit) and cfgReactionColor.tapped) or cfgReactionColor[UnitFrame.__reaction] or ColorWhite  
		end
	end,
	health = function(UnitFrame, _, per)
		per = per or UnitHealth(UnitFrame.unit) / UnitHealthMax(UnitFrame.unit)
		return (per>=cfgHealthThreshold1 and cfgHealthColor1) or (per>=cfgHealthThreshold2 and cfgHealthColor2) or cfgHealthColor3 or ColorWhite
	end,
	class = function(UnitFrame)
		return ClassColors[UnitFrame.__class] or ClassColors.UNKNOWN
	end,
	level = function(UnitFrame)
		return DifficultyColor[UnitFrame.__level] or DifficultyColor[-1] 
	end,
	color = function(UnitFrame, widgetName)
		return UnitFrame.__update[widgetName] or ColorWhite
	end,
}

local function UpdatePlateColors(UnitFrame)
	local update = UnitFrame.__update
	for widgetName,func in pairs(update.methods) do
		local widget = UnitFrame[widgetName]
		widget:SetWidgetColor( unpack(widget.colorOverride or func(UnitFrame, widgetName)) )
	end
end

local function UpdateWidgetColor(UnitFrame, widgetName)
	local widget = UnitFrame[widgetName]
	widget:SetWidgetColor( unpack( widget.colorOverride or UnitFrame.__update.methods[widgetName](UnitFrame, widgetName) ) )
end

local function UpdateWidgetStatusColor(UnitFrame, statusName)
	local widgets = UnitFrame.__update[statusName]
	local count   = #widgets
	if count>0 then
		local func = ColorMethods[statusName]
		for i=count,1,-1 do
			local widgetName = widgets[i]
			local widget = UnitFrame[widgetName]
			widget:SetWidgetColor( unpack( widget.colorOverride or func(UnitFrame, widgetName) ) )
		end		
	end
end

----------------------------------------------------------------
-- Widgets values assignment management
----------------------------------------------------------------

local WidgetMethods = {
	kHealthBar = addon.Dummy, -- no update needed because we are using blizzard health bar
	kHealthText = function(UnitFrame)
		local unit = UnitFrame.unit
		local per  = UnitHealth(unit) / UnitHealthMax(unit)
		UnitFrame.kHealthText:SetText( format('%.0f%%',per*100) )
	end,
	kNameText = function(UnitFrame)
		UnitFrame.kNameText:SetText( UnitFrame.__name )
	end,
	kLevelText = function(UnitFrame)
		local level = UnitFrame.__level
		local class = UnitFrame.__classification
		local text  = level<0 and '??' or level .. (Classifications[class] or '')
		UnitFrame.kLevelText:SetText( text )
	end,
	ClassificationFrame = function(UnitFrame)
		local ClassificationFrame = UnitFrame.ClassificationFrame
		if UnitFrame.__classification=='boss' then
			local classificationIndicator = ClassificationFrame.classificationIndicator
			classificationIndicator:SetTexture("Interface\\TargetingFrame\\UI-TargetingFrame-Skull")
			classificationIndicator:SetTexCoord(0.1, 0.9, 0, 1)
		end
	end,
}

function UpdatePlateValues(UnitFrame)
	local widgets = UnitFrame.__update
	for i=#widgets,1,-1 do
		widgets[i](UnitFrame)
	end
end

----------------------------------------------------------------
-- Health update for health text widget & health status 
----------------------------------------------------------------

local UpdateColorHealth = ColorMethods.health
local HealthFrame = CreateFrame("Frame") 
HealthFrame:SetScript("OnEvent", function(_, _, unit)
	local UnitFrame = NamePlatesByUnit[unit]
	if UnitFrame then
		local kHealthText = UnitFrame.kHealthText
		if kHealthText then
			local per = UnitHealth(unit) / UnitHealthMax(unit)
			kHealthText:SetText( format('%.0f%%',per*100) )
		end	
		local widgets = UnitFrame.__update.health
		for i=#widgets,1,-1 do
			local widget = UnitFrame[widgets[i]]
			widget:SetWidgetColor( unpack( UpdateColorHealth(UnitFrame, nil, per) ) )
		end
	end	
end )

----------------------------------------------------------------
-- Disable blizzard stuff
----------------------------------------------------------------

local function DisableBlizzardStuff(UnitFrame)
	local healthBar = UnitFrame.healthBar
	healthBar.barTexture:SetColorTexture(0,0,0,0)
	local textures = healthBar.border.Textures
	for i=#textures,1,-1 do
		textures[i]:SetVertexColor(0,0,0,0)
		textures[i]:SetColorTexture(0,0,0,0)
		textures[i]:Hide()
	end
	local level = UnitFrame.castBar:GetFrameLevel()+1
	UnitFrame.RaidTargetFrame:SetFrameLevel(level)
	UnitFrame.ClassificationFrame:SetFrameLevel(level)
end

----------------------------------------------------------------
-- kHealthBar widget creation
----------------------------------------------------------------

local function CreateHealthBar(UnitFrame)
	local healthBar = UnitFrame.healthBar
	local layer, level = healthBar.barTexture:GetDrawLayer()
	local kHealthBar = healthBar:CreateTexture(nil, layer, nil, level+1)
	kHealthBar:SetPoint("TOPLEFT", healthBar.barTexture, "TOPLEFT")
	kHealthBar:SetPoint("BOTTOMRIGHT", healthBar.barTexture, "BOTTOMRIGHT")
	kHealthBar.SetWidgetColor = kHealthBar.SetVertexColor
	UnitFrame.kkHealthBar = kHealthBar
end

----------------------------------------------------------------
-- kHealthBorder widget creation
----------------------------------------------------------------

local CreateHealthBorder
do
	local BORDER_POINTS = {
		{ "TOPRIGHT",    "TOPLEFT",    0, 1, "BOTTOMRIGHT", "BOTTOMLEFT",  0, -1, "SetWidth" },
		{ "TOPLEFT",     "TOPRIGHT",   0, 1, "BOTTOMLEFT",  "BOTTOMRIGHT", 0, -1, "SetWidth" },
		{ "TOPRIGHT",    "BOTTOMLEFT", 0, 0, "TOPLEFT",     "BOTTOMRIGHT", 0, 0 , "SetHeight" },
		{ "BOTTOMRIGHT", "TOPRIGHT",   0, 0, "BOTTOMLEFT",  "TOPLEFT",     0, 0 , "SetHeight" },
	}

	local function SetBorderColor(border,r,g,b,a)
		border[1]:SetVertexColor(r,g,b,a)
		border[2]:SetVertexColor(r,g,b,a)
		border[3]:SetVertexColor(r,g,b,a)
		border[4]:SetVertexColor(r,g,b,a)
	end

	local function SetBorderSize(border,size)
		if border.size ~= size then
			local frame = border[1]:GetParent()
			for i=1,4 do
				local p = BORDER_POINTS[i]
				local t = border[i]
				t:ClearAllPoints()
				t:SetPoint( p[1], frame, p[2], p[3]*size, p[4]*size )
				t:SetPoint( p[5], frame, p[6], p[7]*size, p[8]*size )
				t[ p[9] ](t, size)
			end
			border.size = size
		end
	end

	local function Hide(border)
		SetBorderColor(border,0,0,0,0)
	end
	
	function CreateHealthBorder(UnitFrame)
		local frame = UnitFrame.healthBar
		local border = { widgetName= 'kHealthBorder', size=1 }
		for i=1,4 do
			local p = BORDER_POINTS[i]
			local t = frame:CreateTexture(nil, "BACKGROUND")
			t:SetPoint( p[1], frame, p[2], p[3], p[4] )
			t:SetPoint( p[5], frame, p[6], p[7], p[8] )
			t[ p[9] ](t,1)
			t:SetColorTexture(1,1,1,1)
			border[i] = t
		end
		border.SetWidgetColor = SetBorderColor
		border.SetWidgetSize  = SetBorderSize
		border.Hide = Hide
		UnitFrame.kkHealthBorder = border
		return border
	end
end

----------------------------------------------------------------
-- kLevelText widget creation
----------------------------------------------------------------

local function CreateLevelText(UnitFrame)
	local text  = UnitFrame.RaidTargetFrame:CreateFontString(nil, "BORDER")
	text.SetWidgetColor = text.SetTextColor
	text:SetShadowOffset(1,-1)
	text:SetShadowColor(0,0,0, 1)
	UnitFrame.kkLevelText = text
end

----------------------------------------------------------------
-- kNameText widget creation
----------------------------------------------------------------

local function CreateNameText(UnitFrame)
	local text = UnitFrame.RaidTargetFrame:CreateFontString(nil, "BORDER")
	text.SetWidgetColor = text.SetTextColor
	text:SetShadowOffset(1,-1)
	text:SetShadowColor(0,0,0, 1)
	UnitFrame.kkNameText = text
	UnitFrame.name:SetParent(HiddenFrame)
end

----------------------------------------------------------------
-- kHealthText widget creation
----------------------------------------------------------------

local function CreateHealthText(UnitFrame)
	local text = UnitFrame.RaidTargetFrame:CreateFontString(nil, "BORDER")
	text.SetWidgetColor = text.SetTextColor
	text:SetShadowOffset(1,-1)
	text:SetShadowColor(0,0,0, 1)
	UnitFrame.kkHealthText = text
end

----------------------------------------------------------------
-- Skin a nameplate
----------------------------------------------------------------

local SkinPlate
do
	local SkinMethods = {
		kHealthBar = function(UnitFrame, frameAnchor, db, enabled)
			local kHealthBar = UnitFrame.kkHealthBar
			if enabled then
				UnitFrame.kHealthBar = kHealthBar
				local healthBar = UnitFrame.healthBar
				if db.kHealthBar_color_status~='blizzard' then
					healthBar.barTexture:SetColorTexture(0,0,0,0)
					kHealthBar:SetTexture( TexCache[db.healthBarTexture] )
					kHealthBar:Show()
				else
					healthBar.barTexture:SetTexture( TexCache[db.healthBarTexture] )
					kHealthBar:Hide()
				end
			elseif kHealthBar then
				UnitFrame.kHealthBar = nil
				kHealthBar:Hide()
			end
		end,
		kHealthBorder = function(UnitFrame, frameAnchor, db, enabled)
			local kHealthBorder = UnitFrame.kkHealthBorder
			if enabled then
				kHealthBorder:SetWidgetSize( db.borderSize or 1 )
				UnitFrame.kHealthBorder = kHealthBorder
			elseif kHealthBorder then
				UnitFrame.kHealthBorder = nil
				kHealthBorder:Hide()
			end
		end,
		kNameText = function(UnitFrame, frameAnchor, db, enabled)
			local kNameText = UnitFrame.kkNameText
			if enabled then
				kNameText:SetPoint("BOTTOM", frameAnchor, "TOP", db.nameOffsetX or 0, db.nameOffsetY or -1);
				kNameText:SetFont( FontCache[db.nameFontFile or 'Roboto Condensed Bold'], db.nameFontSize or 12, db.nameFontFlags or 'OUTLINE' )
				UnitFrame.kNameText = kNameText
				kNameText:Show()
			elseif kNameText then
				UnitFrame.kNameText = nil
				kNameText:Hide()
			end
		end,
		kLevelText = function(UnitFrame, frameAnchor, db, enabled)
			local kLevelText = UnitFrame.kkLevelText
			if enabled then
				kLevelText:SetPoint( db.levelAnchorPoint or "LEFT", frameAnchor, "CENTER", db.levelOffsetX or -62, db.levelOffsetY or -4);
				kLevelText:SetFont( FontCache[db.levelFontFile or 'Accidental Presidency'], db.levelFontSize or 14, db.levelFontFlags or 'OUTLINE' )
				UnitFrame.kLevelText = kLevelText
				kLevelText:Show()
			elseif kLevelText then
				UnitFrame.kLevelText = nil
				kLevelText:Hide()
			end
		end,
		kHealthText = function(UnitFrame, frameAnchor, db, enabled)
			local kHealthText = UnitFrame.kkHealthText
			if enabled then
				kHealthText:ClearAllPoints()
				kHealthText:SetPoint( db.healthTextAnchorPoint or 'RIGHT', frameAnchor, 'CENTER', db.healthTextOffsetX or 66, db.healthTextOffsetY or -4)
				kHealthText:SetFont( FontCache[db.healthTextFontFile or 'Accidental Presidency'], db.healthTextFontSize or 14, db.healthTextFontFlags or 'OUTLINE' )
				UnitFrame.kHealthText = kHealthText
				kHealthText:Show()
			elseif kHealthText then
				UnitFrame.kHealthText = nil
				kHealthText:Hide()
			end
		end,
		ClassificationFrame = function(UnitFrame, frameAnchor, db, enabled)
			local ClassificationFrame = UnitFrame.ClassificationFrame
			if enabled then
				ClassificationFrame:SetSize( db.classIconSize or 14, db.classIconSize or 14 )
				ClassificationFrame:SetPoint('RIGHT', frameAnchor, 'LEFT', db.classIconOffsetX or 0, db.classIconOffsetY or 0)
				ClassificationFrame:Show()
			else
				ClassificationFrame:Hide()
			end
		end,
		RaidTargetFrame = function(UnitFrame, frameAnchor, db, enabled)
			local RaidTargetFrame = UnitFrame.RaidTargetFrame
			if enabled then
				RaidTargetFrame.RaidTargetIcon:SetParent(RaidTargetFrame)
				RaidTargetFrame:SetPoint("RIGHT", frameAnchor, "LEFT", db.raidTargetOffsetX or 154, db.raidTargetOffsetY or 0);
				RaidTargetFrame:SetSize( db.raidTargetSize or 20, db.raidTargetSize or 20 )
				RaidTargetFrame:Show()
			else
				RaidTargetFrame.RaidTargetIcon:SetParent(HiddenFrame) -- we cannot simply Hide() this frame because our widgets are parented to it
			end
		end,
	}

	function SkinPlate(plateFrame, UnitFrame, UnitAdded)
		-- calculate skin
		local db = addon.db.skins[ GetPlateSkin(UnitFrame, InCombat, InstanceType) ]
		-- opacity & frame level
		local target = UnitFrame.__target
		local mouse =  UnitFrame.__mouseover
		UnitFrame:SetFrameStrata( (target or mouse) and "HIGH" or "MEDIUM" )
		UnitFrame:SetAlpha( (mouse and 1) or (target and cfgAlpha1) or (not targetExists and cfgAlpha3) or cfgAlpha2 )
		local Reskin = (db ~= UnitFrame.__skin)
		if Reskin or UnitAdded then -- blizzard code resets these settings, so we need to reapply them even if our skin has not changed.
			-- UnitFrame
			UnitFrame:ClearAllPoints()
			UnitFrame:SetPoint( 'BOTTOM', plateFrame, 'BOTTOM', 0, db.plateOffsetY or 6 )
			UnitFrame:SetPoint( 'TOP', plateFrame, 'TOP', 0, 0 )
			UnitFrame:SetWidth( db.healthBarWidth or 156 )
			-- blizzard healthBar
			local healthBar = UnitFrame.healthBar
			healthBar:SetHeight( db.healthBarHeight or 12 )
			healthBar:SetShown(activeWidgets.kHealthBar)
			-- blizzard castBar
			local castBar = UnitFrame.castBar
			castBar:SetHeight( db.castBarHeight or 10 )
			if not ( db.castBarHidden or (UnitFrame.__reaction=='friendly')==db.castBarHiddenFriendly ) then
				local cbHeight = db.castBarHeight or 10
				castBar.BorderShield:SetSize(cbHeight, cbHeight)
				castBar.Icon:SetSize(cbHeight, cbHeight)
				castBar.Icon:SetTexCoord( 0.1, 0.9, 0.1, 0.9 )
				castBar:SetStatusBarTexture( TexCache[db.castBarTexture] )
				castBar.Text:SetFont( FontCache[db.castBarFontFile or 'Roboto Condensed Bold'], db.castBarFontSize or 8, db.castBarFontFlags or 'OUTLINE' )
			elseif UnitAdded then
				CastingBarFrame_SetUnit(castBar, nil)
			end
		end
		if Reskin then
			local update = WidgetUpdate[db]
			-- save skin & update stuff
			UnitFrame.__skin = db
			UnitFrame.__update = update
			-- skin widgets
			local frameAnchor = UnitFrame.healthBar
			for i=1,#WidgetNames do
				local widgetName = WidgetNames[i]
				SkinMethods[widgetName]( UnitFrame, frameAnchor, db, update[widgetName] )
			end
			-- update widgets values & color
			UpdatePlateValues(UnitFrame)
			UpdatePlateColors(UnitFrame)
			-- notify that a plate has been skinned to other modules
			addon:SendMessage('PLATE_SKINNED',UnitFrame, db)
		elseif UnitAdded then
			-- update widgets values & color
			UpdatePlateValues(UnitFrame)
			UpdatePlateColors(UnitFrame)
		end
	end

end

----------------------------------------------------------------
-- Fix Unknown entities in plate names
----------------------------------------------------------------

local WatchUnitFrame
do
	local timer
	local plates = {}
	timer = addon:CreateTimer(function()
		for plateFrame, UnitFrame in next,plates do
			local watch
			if UnitFrame == NamePlates[plateFrame] then
				local name = UnitName(UnitFrame.unit)
				if name ~= UNKNOWNOBJECT then
					UnitFrame.__name = name
					local text = UnitFrame.kNameText
					if text then 
						text:SetText(name) 
					end
					if ConditionFields.names then
						SkinPlate(plateFrame, UnitFrame)
					end
				else
					watch = true
				end
			end
			if not watch then plates[plateFrame] = nil end
		end
		if not next(plates) then timer:Stop() end
	end, .25, false)
	function WatchUnitFrame(plateFrame, UnitFrame)
		if not next(plates) then timer:Play() end
		plates[plateFrame] = UnitFrame
	end
end

----------------------------------------------------------------
-- Reskin visible nameplates
----------------------------------------------------------------

local function ReskinPlates()
	for plateFrame, UnitFrame in pairs(NamePlates) do
		SkinPlate(plateFrame, UnitFrame)
	end
end

----------------------------------------------------------------
-- Highlights nameplate
----------------------------------------------------------------

local function HighlightSet(UnitFrame)
	if UnitFrame then
		if UnitFrame.kHealthBar then
			HighlightTex:SetParent(UnitFrame.healthBar)
			HighlightTex:SetAllPoints()
		end	
	else
		HighlightTex:SetParent(HiddenFrame)
	end
end

----------------------------------------------------------------
-- Mouseover management
----------------------------------------------------------------

do
	local timer = addon:CreateAnimationGroup()
	timer:CreateAnimation():SetDuration(.2)
	timer:SetLooping("REPEAT")
	timer:SetScript("OnLoop", function()
		if not (mouseFrame and UnitIsUnit('mouseover', mouseFrame.UnitFrame.unit)) then
			timer:Stop()
			addon:UPDATE_MOUSEOVER_UNIT()
		end
	end )
	function addon:UPDATE_MOUSEOVER_UNIT()
		local plateFrame = C_GetNamePlateForUnit('mouseover')
		if plateFrame~=mouseFrame then
			if mouseFrame then
				local UnitFrame = mouseFrame.UnitFrame
				UnitFrame.__mouseover = nil
				SkinPlate(mouseFrame, UnitFrame)
				mouseFrame = nil
				HighlightSet(nil)
				timer:Stop()
			end	
			if plateFrame and NamePlates[plateFrame] then
				local UnitFrame = plateFrame.UnitFrame
				UnitFrame.__mouseover = true
				mouseFrame = plateFrame
				SkinPlate(mouseFrame, UnitFrame)
				HighlightSet(UnitFrame)
				timer:Play()
			end
		end	
	end
end

----------------------------------------------------------------
-- Opacity adjust
----------------------------------------------------------------

local function UpdatePlatesOpacity()
	local alpha = targetExists and cfgAlpha2 or cfgAlpha3
	for plateFrame, UnitFrame in pairs(NamePlates) do
		if plateFrame~=targetFrame and plateFrame~=mouseFrame then 
			UnitFrame:SetAlpha( alpha )
		end	
	end
end

----------------------------------------------------------------
-- Personal resource bar
----------------------------------------------------------------

local function PersonalBarAdded(plateFrame)
	-- undo some custom changes to avoid displaying a messed personal bar
	local UnitFrame = plateFrame.UnitFrame
	UnitFrame.healthBar.barTexture:SetTexture("Interface\\TargetingFrame\\UI-StatusBar")
	for i=1,#WidgetNames do
		local widget = UnitFrame[WidgetNames[i]]
		if widget then widget:Hide() end
	end
	UnitFrame.__skin = nil
end

local function PersonalBarRemoved(plateFrame)
	-- redo our custom changes once the personal resource bar is hidden
	local UnitFrame = plateFrame.UnitFrame
	UnitFrame.healthBar.barTexture:SetColorTexture(0,0,0,0)
end

----------------------------------------------------------------
-- Player target management
----------------------------------------------------------------

function addon:PLAYER_TARGET_CHANGED()
	local plateFrame = C_GetNamePlateForUnit('target')
	if plateFrame ~= targetFrame then
		if targetFrame then
			local UnitFrame = targetFrame.UnitFrame
			UnitFrame.__target = nil
			SkinPlate(targetFrame, UnitFrame)
			targetFrame = nil
		end
		if plateFrame and NamePlates[plateFrame] then
			local UnitFrame = plateFrame.UnitFrame
			UnitFrame.__target = true
			SkinPlate(plateFrame, UnitFrame)
			targetFrame = plateFrame
			self:SendMessage('PLAYER_TARGET_ACQUIRED', plateFrame, 'target' )
		end
	end
	if targetExists ~= UnitExists('target') then
		targetExists = not targetExists
		if cfgAdjustAlpha then
			UpdatePlatesOpacity()
		end	
	end
end

---------------------------------------------------------------
-- Nameplate created event
----------------------------------------------------------------

local CreateNamePlate
do
	local CreateMethods = {
		kHealthBar    = CreateHealthBar,
		kHealthBorder = CreateHealthBorder,
		kHealthText   = CreateHealthText,
		kLevelText    = CreateLevelText,
		kNameText     = CreateNameText,
	}
	function CreateNamePlate(UnitFrame)
		for i=1,#activeWidgets do
			local widgetName = activeWidgets[i]
			if not UnitFrame[WidgetNames[widgetName]] then
				CreateMethods[widgetName](UnitFrame)
			end	
		end
		UnitFrame.__skin = nil
	end
	function addon:NAME_PLATE_CREATED(plateFrame)
		local UnitFrame = plateFrame.UnitFrame
		DisableBlizzardStuff(UnitFrame)
		CreateNamePlate(UnitFrame)
		NamePlatesAll[#NamePlatesAll+1] = UnitFrame
	end	
end

---------------------------------------------------------------
-- Nameplate added event
----------------------------------------------------------------
	
function addon:NAME_PLATE_UNIT_ADDED(unit)
	local plateFrame = C_GetNamePlateForUnit(unit)
	if UnitIsUnit(unit,'player') then return PersonalBarAdded(plateFrame) end
	if plateFrame then
		local UnitFrame = plateFrame.UnitFrame
		NamePlates[plateFrame] = UnitFrame
		NamePlatesByUnit[unit] = UnitFrame
		UnitFrame:SetParent(WorldFrame)
		UnitFrame:SetScale(pixelScale)
		UnitFrame.__class  = select(2,UnitClass(unit))
		UnitFrame.__type   = Types[ strsub( UnitGUID(unit), 3,3 ) ]
		UnitFrame.__reaction = Reactions [ UnitReaction( unit, "player") or 1 ]		
		UnitFrame.__level  = UnitLevel( unit )		
		UnitFrame.__classification = UnitClassification(unit) or 'unknow'
		if UnitFrame.__level==-1 or UnitFrame.__classification=='worldboss' then	
			UnitFrame.__classification = 'boss'
		end
		UnitFrame.__name = UnitName( unit )
		if UnitFrame.__name == UNKNOWNOBJECT then
			WatchUnitFrame(plateFrame, UnitFrame)
		end
		UnitFrame.__target = UnitIsUnit( unit, 'target' ) or nil
		if UnitFrame.__target then
			if targetFrame then -- unmark&reskin old target frame
				targetFrame.UnitFrame.__target = nil
				SkinPlate(targetFrame, targetFrame.UnitFrame)			
			end
			targetFrame = plateFrame
		end
		SkinPlate( plateFrame, UnitFrame, true )
		self:SendMessage("NAME_PLATE_UNIT_ADDED", UnitFrame, unit)
	end
end

----------------------------------------------------------------
-- Nameplate removed event
----------------------------------------------------------------

function addon:NAME_PLATE_UNIT_REMOVED(unit)
	local plateFrame = C_GetNamePlateForUnit(unit)
	if UnitIsUnit(unit,'player') then return PersonalBarRemoved(plateFrame) end
	local UnitFrame = plateFrame.UnitFrame
	UnitFrame.__threat = nil
	UnitFrame.__target = nil
	UnitFrame.__mouseover = nil
	if plateFrame == targetFrame then
		targetFrame = nil
	end
	if plateFrame == mouseFrame then
		mouseFrame = nil
	end
	UnitFrame:SetParent(plateFrame)
	UnitFrame:SetScale(1)
	UnitFrame:ClearAllPoints()
	UnitFrame:SetAllPoints()
	NamePlates[plateFrame] = nil
	NamePlatesByUnit[unit] = nil
	self:SendMessage("NAME_PLATE_UNIT_REMOVED", UnitFrame, unit)
end

----------------------------------------------------------------
-- Events triggering status 'reaction' color update
----------------------------------------------------------------

function addon:UNIT_FACTION(unit)
	local UnitFrame = NamePlatesByUnit[unit]
	if UnitFrame then
		UpdateWidgetStatusColor(UnitFrame, 'reaction')
	end
end
addon.UNIT_TARGETABLE_CHANGED = UNIT_FACTION

----------------------------------------------------------------
-- Combat Start
----------------------------------------------------------------

function addon:PLAYER_REGEN_DISABLED()
	InCombat = true
	self:SendMessage('COMBAT_START')
	if ConditionFields.combat then
		ReskinPlates()
		for plateFrame, UnitFrame in pairs(NamePlates) do
			SkinPlate(plateFrame, UnitFrame)
		end
	end
end

----------------------------------------------------------------
-- Combat End
----------------------------------------------------------------

function addon:PLAYER_REGEN_ENABLED()
	InCombat = false
	self:SendMessage('COMBAT_END')
	if ConditionFields.combat then
		ReskinPlates()
	end
end

----------------------------------------------------------------
-- Zone Changed, reskin plates if necessary
----------------------------------------------------------------

function addon:ZONE_CHANGED_NEW_AREA(event)
	local _, type = IsInInstance()
	if type ~= InstanceType then
		InstanceType = type
		if ConditionFields.instance then
			ReskinPlates()
		end
	end
end
addon.PLAYER_ENTERING_WORLD = ZONE_CHANGED_NEW_AREA

----------------------------------------------------------------
-- Compile a function to calculate nameplate skin
----------------------------------------------------------------

local CompileSkinCheckFunction
do
	local function NamesToList(names)
		local lines = {}
		local t = { strsplit("\n",names) }
		for i=1,#t do -- Remove comments: any text starting with #@\/-[ characters.
			local s = strtrim( (strsplit( "#@\\\/\-\[", t[i] )) ) -- Don't remove strsplit extra brackets.
			if #s>0 then tinsert( lines, format('["%s"]=true',s) ) end
		end
		return tconcat( lines , ',')
	end
	local function MakeSkinCheckWithClosure(names, source)
		local lines = { "return function()" }
		for i=1,#names do
			tinsert( lines, format("local names%d = {%s}",i, NamesToList(names[i]) ) )
		end
		tinsert( lines,  source )
		tinsert( lines, 'end' )
		return assert(loadstring(tconcat( lines, "\n")))()()
	end
	function CompileSkinCheckFunction(rules, default, fields)
		if fields then wipe(fields) end
		local count, handler, lines, names, iff = 0, { 'return function(p, combat, instance) ' }, {}, {}, 'if'
		for j=1,#rules do
			local rule = rules[j]
			if #rule>1 then 
				for i=2,#rule do
					local c = rule[i]
					if c[1] == 'names' then
						tinsert( names, c[3] )
						tinsert( lines, format("names%d[p.__name]",#names) )
					else
						local line
						local typ   = type(c[3])
						local field = gsub(c[1], "@", 'p.__')
						if c[3] == 'nil' then
							line = format( '%s == nil', field )
						elseif typ == 'string' then
							line = format( '%s %s "%s"', field, c[2], c[3] )
						elseif typ == 'boolean' then
							line = format( 'not %s %s %s', field, c[2], tostring(not c[3]) )
						else
							line = format( '%s %s %s', field, c[2], tostring(c[3]) )
						end
						tinsert( lines, line )
					end
					if fields then fields[c[1]] = true end
				end
				tinsert( handler, format( '%s %s then return %d', iff, tconcat(lines,format(' %s ',rule[1])), j) )
				iff = "elseif"
				wipe(lines)
				count = count + 1
			end	
		end
		tinsert( handler, format(count>0 and 'else return %d end end' or 'return %d end', default or 1) )
		if #names>0 then
			GetPlateSkin = MakeSkinCheckWithClosure( names, tconcat(handler,"\n") )
		else
			GetPlateSkin = assert(loadstring(tconcat(handler,"\n")))()
		end
	end
end

----------------------------------------------------------------
-- Register/Unregister events
----------------------------------------------------------------

local function UpdateEventRegister( self, enabled, ... )
	local method = enabled and "RegisterEvent" or "UnregisterEvent"
	for i=select('#',...),1,-1 do
		self[method]( self, select(i,...) )
	end	
end

----------------------------------------------------------------
-- Update addon when database config or profile changes
----------------------------------------------------------------

do
	local function UpdateGeneral()
		local cfg = addon.db.general
		cfgAlpha1 = cfg.alpha1 or 1
		cfgAlpha2 = cfg.alpha2 or .4
		cfgAlpha3 = cfg.alpha3 or 1
		cfgAdjustAlpha  = math.abs(cfgAlpha3 - cfgAlpha2)>0.01
		cfgReactionColor = cfg.reactionColor
		cfgClassColorReaction.hostile  = cfg.classColorHostilePlayers
		cfgClassColorReaction.friendly = cfg.classColorFriendlyPlayers
		cfgHealthColor1 = cfg.healthColor.color1
		cfgHealthColor2 = cfg.healthColor.color2
		cfgHealthColor3 = cfg.healthColor.color3
		cfgHealthThreshold1 = cfg.healthColor.threshold1
		cfgHealthThreshold2 = cfg.healthColor.threshold2
		pixelScale = 768/select(2,GetPhysicalScreenSize())/WorldFrame:GetScale()
		if not InCombatLockdown() then
			SetCVar("nameplateGlobalScale", 1)
			SetCVar("nameplateSelectedScale", 1)
			SetCVar("nameplateMinScale", 1)
		end
	end

	local function UpdateSkins()
		wipe(activeWidgets)
		wipe(activeStatuses)
		wipe(WidgetUpdate)
		for _,skin in ipairs(addon.db.skins) do
			local update = { methods = {}, blizzard = {} }
			for statusName in pairs(ColorStatuses) do
				update[statusName] = {}
			end
			for _,widgetName in ipairs(WidgetNames) do
				if skin[widgetName..'_enabled'] then
					activeWidgets[widgetName] = true
					tinsert( activeWidgets, widgetName )
					local statusName = skin[widgetName..'_color_status'] or ColorStatusDefaults[widgetName]
					if statusName then
						update.methods[widgetName] = ColorMethods[statusName]
						tinsert( update[statusName], widgetName )
						update[widgetName] = skin[widgetName..'_color_default'] or ColorDefaults[widgetName]
						activeStatuses[statusName] = true
					else
						update[widgetName] = true -- hackish, we use this to detect enabled widgets inside each skin
					end
					local func = WidgetMethods[widgetName]
					if func then tinsert( update , WidgetMethods[widgetName] ) end
				end
			end
			WidgetUpdate[skin] = update
		end
	end

	function addon:Update()

		UpdateGeneral()
		
		UpdateSkins()
		
		CompileSkinCheckFunction(addon.db.rules, addon.db.defaultSkin, ConditionFields)

		UpdateEventRegister( HealthFrame, activeWidgets.kHealthText or activeStatuses.health , "UNIT_HEALTH_FREQUENT", "UNIT_MAXHEALTH" )
		UpdateEventRegister( self, self.db.general.highlight or ConditionFields['@mouseover'], "UPDATE_MOUSEOVER_UNIT" )
		UpdateEventRegister( self, activeStatuses.reaction, "UNIT_FACTION", "UNIT_TARGETABLE_CHANGED" )
		
		-- updating modules before reskining the plates, be careful in threat/auras modules UPDATE callbacks,
		-- do not call direct/indirect to the function SkinPlate() in these callbacks, do not use plates skin data.
		self:SendMessage('UPDATE')
		
		-- Create nameplates widgets if necessary and mark all nameplates to be reskinned, even the unused nameplates
		for _,UnitFrame in ipairs(NamePlatesAll) do
			CreateNamePlate(UnitFrame)
		end
		
		-- Not using the function C_NamePlate.GetNamePlates() to iterate the plates because that function 
		-- can return the personal resource bar if visible, and the addon must not skin that bar. 
		for plateFrame, UnitFrame in pairs(NamePlates) do
			SkinPlate(plateFrame, UnitFrame, true) -- true = Fake UNIT_ADDED to force a full update
		end
		
		UpdatePlatesOpacity()

	end
end

----------------------------------------------------------------
-- Database profile changed
----------------------------------------------------------------

function addon:OnProfileChanged()
	self:SendMessage('PROFILE_CHANGED')
	self:Update() 
end

----------------------------------------------------------------
-- Initialization
----------------------------------------------------------------

addon:RegisterMessage('INITIALIZE', function()
	InstanceType = select(2, IsInInstance())
	addon:RegisterEvent("NAME_PLATE_CREATED")
	addon:RegisterEvent("NAME_PLATE_UNIT_ADDED")
	addon:RegisterEvent("NAME_PLATE_UNIT_REMOVED")
	addon:RegisterEvent("PLAYER_TARGET_CHANGED")	
	addon:RegisterEvent("PLAYER_REGEN_DISABLED")
	addon:RegisterEvent("PLAYER_REGEN_ENABLED")
	addon:RegisterEvent("ZONE_CHANGED_NEW_AREA")
	addon:RegisterEvent("PLAYER_ENTERING_WORLD")
end )

----------------------------------------------------------------
-- Run
----------------------------------------------------------------

addon:RegisterMessage('ENABLE', function()
	addon:Update()
end )

----------------------------------------------------------------
-- Publish some stuff
----------------------------------------------------------------

addon.NamePlates          = NamePlates
addon.NamePlatesByUnit    = NamePlatesByUnit
addon.ClassColors         = ClassColors
addon.ColorDefault        = ColorDefault
addon.ColorDefaults       = ColorDefaults
addon.ColorWidgets        = ColorWidgets
addon.ColorStatuses       = ColorStatuses
addon.ColorStatusDefaults = ColorStatusDefaults
addon.UpdateWidgetColor   = UpdateWidgetColor

