--[[
Copyright 2018, Quarq
This file is part of GoPost.
GoPost is distributed under a BSD License.
It is provided AS-IS and all warranties, express or implied, including, but not
limited to, the implied warranties of merchantability or fitness for a particular
purpose, are disclaimed.  See the LICENSE file for full information.
--]]

local Addon = GoPost
local util = Addon.util


Addon:RegisterEvent("MAIL_SHOW")
Addon:RegisterEvent("MAIL_INBOX_UPDATE")
Addon:RegisterEvent("MAIL_CLOSED")


function Addon : MAIL_SHOW(arg)
	
	if (not self.initialized) then
		self:SetFrameStrata("DIALOG")
	
		self:CreateUI()
		assert(self.initialized)
	end
	
	-- show stuff
	self.expanded = {}			-- empty array means no sections expanded
	self.first_update = true	-- handle MAIL_INBOX_UPDATE without delay
	
	-- may GoPost current, depending on setting
	if (Addon.DB.auto) then
		MailFrameTab_OnClick(self.tab1, self.tab1:GetID())
	end

	-- while we're shown, we want to handle these
	-- Looter relies on these to advance it through looting a message (and, therefore, a section)
	-- if one of these is delayed "too long", then this file (Core.lua) will update the scroller while the Looter is still "busy",
	-- leaving the loot buttons disabled.  When the delayed event finally fires, the Looter goes idle, but the buttons are still disabled.
	-- So, we handle these and update the scroller [possibly unnecessarily] when they fire (subject to the same throttling as MAIL_INBOX_UPDATE)
	self:RegisterEvent("BAG_UPDATE_DELAYED")	-- signals well and truely done putting an item in your bag, e.g. looted from mail, and other sources
	self:RegisterEvent("PLAYER_MONEY")			-- signals a change to the player's gold balance, e.g. when looting from mail, and other sources
	self.BAG_UPDATE_DELAYED = function () Addon:UpdateMessages_Throttled() end
	self.PLAYER_MONEY = function () Addon:UpdateMessages_Throttled() end
end



function Addon : MAIL_CLOSED(arg)
	
	self.Looter:Abort(nil)
	self:Hide()
	
	-- ignore these events now
	self.BAG_UPDATE_DELAYED = nil
	self.PLAYER_MONEY = nil
end



function Addon : MAIL_INBOX_UPDATE(mb, arg2_always_false)

	if (mb) then return end	-- a mail item was clicked with this mouse button

	--print("handling MAIL_INBOX_UPDATE")
	
	if (self.first_update) then
		self.first_update = false
		self:UpdateMessages()
	else
		self:UpdateMessages_Throttled()
	end
end



function Addon : UpdateMessages_Throttled()
	if (1) then

		-- this version throttles UI updates to once every 0.5 seconds
		if (self.update_timer == nil) then
			self.update_timer = C_Timer.NewTimer(0.50,
												 function()
													 Addon.update_timer = nil
													 Addon:UpdateMessages()
												 end)
		end

	else

		-- this version resets the timer on each event, so it only updates messages after the last event
		if (self.update_timer) then
			self.update_timer:Cancel()
		end
		self.update_timer = C_Timer.NewTimer(0.50,
											 function()
												 Addon.update_timer = nil
												 Addon:UpdateMessages()
											 end)

	end
end



function Addon : CreateUI()
	
	assert(not self.initialized)

	local debug_offset = 0 --800

--	self:SetParent(InboxFrame)
	self:SetParent(MailFrame) -- if I have my own tab
	
	self:SetWidth(384-50)
	self:SetHeight(512-1)
	self:SetPoint("TOPLEFT", InboxFrame, "TOPLEFT", debug_offset + 0, -1)
	self:SetFrameStrata("DIALOG")
	
	-- filter controls at top
	self.openall = CreateFrame("Button", nil, self, "GPSmallButton")
	self.openall:SetText("Open All")
	self.openall:SetWidth(100)
	self.openall:SetHeight(28)
	self.openall:SetPoint("TOPRIGHT", self, "TOPRIGHT", -20, -27)
	self.openall:SetScript("OnClick",
						   function()
							   Addon.Looter:LootSection("*")
							   assert(Addon.Looter:IsBusy())
							   Addon:UpdateScroller() -- to disable loot buttons
						   end)
	
	-- move this to the left to make room for my "open all" button
	InboxTooMuchMail:SetPoint("TOP", InboxTooMuchMail:GetParent(), "TOP", -58-2, -25) -- Blizz XML has its top at 0, -25
	
	-- bottom has a scroller for messages
	self.scroller = CreateFrame("ScrollFrame", nil, self, "SectionScrollerTemplate")
	self.scroller:SetPoint("TOPLEFT", self, "TOPLEFT", 7, -61-5)
	self.scroller:SetPoint("BOTTOMRIGHT", self, "BOTTOMRIGHT", -7, 94)
	self.scroller.delegate = {
		SectionScrollerDelegate_NumberOfSections = function(ss) return self:Scroller_NumberOfSections() end,
		SectionScrollerDelegate_GetRowForSection = function(ss, section) return self:Scroller_GetRowForSection(section) end,
		SectionScrollerDelegate_NumberOfItemsInSection = function(ss, section) return self:Scroller_NumberOfItemsInSection(section) end,
		SectionScrollerDelegate_GetRowForItem = function(ss, section, item) return self:Scroller_GetRowForItem(section, item) end,
	}
	
	local bg = self.scroller:CreateTexture(nil, "BACKGROUND")
	bg:SetTexture("Interface/MailFrame/UI-MailFrameBG")
	bg:SetTexCoord(0.03, 0.625, 0, 0.68)
	bg:SetPoint("TOPLEFT", self.scroller, "TOPLEFT", 0, 5)
	bg:SetPoint("BOTTOMRIGHT", self.scroller, "BOTTOMRIGHT", 0, 0)

	
	-- add a tab for myself (https://github.com/tomrus88/BlizzardInterfaceCode/blob/master/Interface/SharedXML/SharedUIPanelTemplates.lua)
	local tabID = MailFrame.numTabs + 1
	local tab = CreateFrame("Button", "MailFrameTab"..tabID, MailFrame, "FriendsFrameTabTemplate")
	tab:SetID(tabID)
	tab:SetText("GoPost")
	tab:SetPoint("LEFT", _G["MailFrameTab"..(tabID-1)], "RIGHT", -8, 0);
	PanelTemplates_SetNumTabs(MailFrame, tabID);
	PanelTemplates_EnableTab(MailFrame, tabID);
	self.tab1 = tab
	
	self.orig_MailFrameTab_OnClick = MailFrameTab_OnClick;
	MailFrameTab_OnClick = function (tab, index) Addon:MailFrameTab_OnClick(tab, index) end
	
	self.tab1:SetScript("OnClick", function() MailFrameTab_OnClick(tab, tabID) end)

	
	self.initialized = true
end



function Addon : MailFrameTab_OnClick(tab, index)
	
	if (tab == self.tab1) then
		-- switch to Inbox tab, show myself, make my tab look like it's current
		self.orig_MailFrameTab_OnClick(MailFrameTab1, 1)
		self:Show()
		PanelTemplates_SetTab(MailFrame, self.tab1:GetID());
	else
		self.orig_MailFrameTab_OnClick(tab, index)
		self:Hide()
	end
	
	index = index or tab:GetID()
	-- co-exist with "Postal"
	if (index) then
		-- "Postal" puts two button it the top margin, but they remain visible when my tab is shown
		-- (probably "hide only on Send tab" instead of "show only on Inbox tab")
		-- so, I hide them on all but the Inbox tab
		local inbox = (index == 1)
		if (PostalSelectOpenButton) then PostalSelectOpenButton:SetShown(inbox) end
		if (PostalSelectReturnButton) then PostalSelectReturnButton:SetShown(inbox) end
	end
end



function Addon : SectionForMessage(m)
	
	-- figure out what section owns the message at index m
	local _, _, sender, subject, money, amountCOD, _, _, _, wasReturned, _, _, isGM = GetInboxHeaderInfo(m)
	--print("(message", m, "sender", sender, "subject", subject .. ")")

	if (amountCOD == nil) then
		amountCOD = 0
	end
	
	if (sender == nil) then
		return "Others", amountCOD -- sender will probably be available in a near-future MAIL_INBOX_UPDATE event
	end
	
	if (isGM or sender:find("The Postmaster") or sender:find("Thaumaturge Vashreen")) then	-- GM, or unlooted items (npc=34337), or bonus roll w/ bags full (npc=54441)
		return "System", amountCOD
	elseif (sender == "Auction House") then
		if (subject == nil) then
			return "Others", amountCOD -- subject will probably be available in a near-future MAIL_INBOX_UPDATE event
		end
		
		if (string.match(subject, "^Auction successful:")) then
			return "Sales", amountCOD
		elseif (string.match(subject, "^Auction won:")) then
			return "Purchases", amountCOD
		elseif (string.match(subject, "^Auction cancelled:")) then
			return "Cancelled", amountCOD
		elseif (string.match(subject, "^Auction expired:")) then
			return "Expired", amountCOD
		else
			return "Others", amountCOD 	-- e.g. subject is "Retrieving data" on first MAIL_INBOX_UPDATE after client startup
		end
	else
		return "Others", amountCOD
	end
end



function Addon : UpdateMessages()
	
	-- an index of messages used by my SectionScroller delegate
	self.sections = {}
	self.sections["Sales"] = {}
	self.sections["Purchases"] = {}
	self.sections["Cancelled"] = {}
	self.sections["Expired"] = {}
	self.sections["Others"] = {}
	self.sections["System"] = {}
			
	-- scan the inbox to fill the index
	local n = GetInboxNumItems()
	for m = 1, n do
		-- classify the message
		local section = self:SectionForMessage(m)
		assert(section)
		
		-- save display info about it
		-- packageIcon, stationeryIcon, sender, subject, money, CODAmount, daysLeft, hasItem, wasRead, wasReturned, textCreated, canReply, isGM = GetInboxHeaderInfo(m)
		local packageIcon, _, sender, subject, money, amountCOD, _, _, opened, _, _, _, _ = GetInboxHeaderInfo(m)
		local info = {}
		info.section = section
		info.messageID = m
		info.sender = sender
		info.subject = subject
		info.money = money or 0
		info.attachments = 0
		info.opened = opened
		info.cod = amountCOD
		
		if (section == "Sales") then
			-- item name is in the invoice
			local invoiceType, itemName, otherPlayer, bid, buyout, deposit, commission = GetInboxInvoiceInfo(m)
			info.itemName = itemName

			-- must parse the subject to get the stack size
			info.itemCount = string.match(info.subject, "%(([0-9]+)%)$") -- does it end with a stack size?
			if (info.itemCount) then
				info.itemCount = 0 + info.itemCount	-- force conversion to number
			else
				info.itemCount = 1	-- no stack size, so assume quantity is 1
			end
			
			-- see if we can get a texture from the item name
			local _, itemLink, _, _, _, _, _, _, _, itemTexture, _, _, _, _, _, _, _ = GetItemInfo(itemName)
			info.itemLink = itemLink
			info.itemTexture = itemTexture
			
			-- if no texture and it might be a battle pet, try to get a pet texture
			if (info.itemTexture == nil and (info.itemLink == nil or info.itemLink:find("|Hbattlepet:"))) then
				local speciesID, petID = C_PetJournal.FindPetIDByName(info.itemName)
				if (speciesID) then
					assert(info.itemTexture == nil)
					info.itemTexture = select(2, C_PetJournal.GetPetInfoBySpeciesID(speciesID)) --  name, icon, petType = C_PetJournal.GetPetInfoBySpeciesID(speciesID)
				end
			end
			
		else
	
			-- other types may have attachments... count them up
			for a = 1, ATTACHMENTS_MAX_RECEIVE do
				local itemName, itemID, texture, count, _, _ = GetInboxItem(m, a)
				if (itemName) then
					info.attachments = info.attachments + 1
					
					if (info.itemName == nil) then
						info.itemID = itemID
						info.itemName = itemName
						info.itemCount = count
						info.itemTexture = texture
						info.itemLink = GetInboxItemLink(m, a)
					end
				end
			end
		end
		
		-- save it with others in the same section
		--print("message", info.messageID, "is in", section)
		assert(self.sections)
		assert(self.sections[section])
		table.insert(self.sections[section], info)
	end
	
	-- update the scroller contents
	self:UpdateScroller()
	
end



function Addon : UpdateScroller()
	
	SectionScroller_UpdateContent(self.scroller)
	
	-- this button is disabled unless we have at least one lootable message in any section except System
	local enabled = false
	for section,messages in pairs(self.sections) do
		if (not (section == "System")) then
			for _,msg in ipairs(messages) do
				enabled = (msg.money > 0 or msg.attachments > 0)
				if (enabled) then break end
			end
			if (enabled) then break end
		end
	end
	
	enabled = enabled and (not self.Looter:IsBusy())

	self.openall:SetEnabled(enabled)
end



local ROW_MARGIN = 16	-- width of the scrollbar, about

local SECTION_NAMES = Addon.groups

function Addon : Scroller_NumberOfSections()
	assert(self.sections)
	return #SECTION_NAMES
end


function Addon : Scroller_GetRowForSection(s)
	
	-- get or create a row for this section
	local section = SECTION_NAMES[s]	-- like "Sales" or "Expired"
	local reuseKey = section
	local row = SectionScroller_GetReusableRow(self.scroller, reuseKey)
	if (row == nil) then
		row = self:HeaderRow_Create(self.scroller:GetWidth() - ROW_MARGIN)
		row.reuseKey = reuseKey
		row:SetScript("OnClick", function() self:HeaderRow_Click(row) end)
	end
	
	-- populate it with the current set of messages
	row.section = section
	row.Populate(self.sections[section])
	
	return row
end


function Addon : Scroller_NumberOfItemsInSection(s)
	
	local section = SECTION_NAMES[s]	-- like "Sales" or "Expired"
	if (self.expanded[section]) then
		local messages = self.sections[section]
		assert(messages)
		return #messages
	else
		return 0
	end
	
end


function Addon : Scroller_GetRowForItem(s, item)

	local section = SECTION_NAMES[s]	-- like "Sales" or "Expired"
	local messages = self.sections[section]
	local message = messages[item]

	local width = self.scroller:GetWidth() - ROW_MARGIN

	local reuseKey
	if (message.sender == "Auction House" and not message.subject:find("Outbid on")) then
		reuseKey = "AuctionMessage"
	else
		reuseKey = "OtherMessage"
	end
	
	local row = SectionScroller_GetReusableRow(self.scroller, reuseKey)
	if (row == nil) then
		local fn = reuseKey .. "_Create" -- like "AuctionMessage_Create"
		row = self[reuseKey.."_Create"](self, self.scroller:GetWidth() - ROW_MARGIN)
		row.reuseKey = reuseKey
	end
	
	row.Populate(message)
	
	return row
end


function Addon : HeaderRow_Click(row)
	
	if (self.expanded[row.section]) then
		self.expanded[row.section] = nil
	else
		self.expanded[row.section] = true
	end
	
	self:UpdateMessages()
end





