local L = LibStub("AceLocale-3.0"):GetLocale("GarrisonJukeBox")
local GJB = LibStub("AceAddon-3.0"):GetAddon("GarrisonJukeBox")

-- Lua APIs
local tinsert, tconcat, tremove, tsort = table.insert, table.concat, table.remove, table.sort
local fmt, tostring = string.format, tostring
local select, pairs, next, type, unpack = select, pairs, next, type, unpack
local loadstring, assert, error = loadstring, assert, error
local setmetatable, getmetatable, rawset, rawget = setmetatable, getmetatable, rawset, rawget
local mrand = math.random

 -- ------------------------------------------------------------
function GJB:ShowConfig()
	-- Open the jukebox, then the profiles tab and then the jukebox again so the menu expands
	InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.JukeBox)
	InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.Profiles)
	InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.JukeBox)
end

-- ------------------------------------------------------------
function GJB:RegisterModuleOptions(name, optionTbl, displayName)
	if GJB.moduleOptions then
		GJB.moduleOptions[name] = optionTbl
	else
		self.options.args[name] = (type(optionTbl) == "function") and optionTbl() or optionTbl
	end
	self.optionsFrames[name] = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("GarrisonJukeBoxDB", displayName, L["ADDON_NAME"], name)
end

-- ------------------------------------------------------------
function GJB.GenerateOptions()
	if not GJB.options then
		GJB.GenerateOptionsInternal()
		GJB.GenerateOptionsInternal = nil
		GJB.moduleOptions = nil
	end
	return GJB.options
end

-- ------------------------------------------------------------
function GJB.GenerateOptionsInternal()
	-- Create the options variable to use in the config screen --
	GJB.options = {
		name = L["OPT_JUKEBOX_OPTIONS"],
		handler = GJB,
		type = "group",
		args = {}
	}

	GJB.options.args["JukeBox"] = GJB.JukeBox
	GJB.options.args["Zones"] = GJB.Zones
	GJB.options.args["Advanced"] = GJB.Advanced
	GJB.options.args["STMode"] = GJB.STMode
	GJB.options.args["Settings"] = GJB.Settings
	GJB.options.args["Audio"] = GJB.Audio
	GJB.options.args["About"] = GJB.About

	for k, v in pairs(GJB.moduleOptions) do
		GJB.options.args[k] = (type(v) == "function") and v() or v
	end

	-- Add ordering data to the option table generated by AceDBOptions-3.0
	GJB.options.args.Profiles.order = -2
end

-- =================================
-- Options table functions 
-- ------------------------------------------------------------
-- JUKEBOX
-- ------------------------------------------------------------
-- Get a table of all expansion/category names
function GJB:GetExpansions()
	return	{	
		GJB.Expansions[GJB.EXP_WOW]["name"], 
		GJB.Expansions[GJB.EXP_BC]["name"], 
		GJB.Expansions[GJB.EXP_WOTLK]["name"], 
		GJB.Expansions[GJB.EXP_CATA1]["name"], 
		GJB.Expansions[GJB.EXP_CATA2]["name"], 
		GJB.Expansions[GJB.EXP_MOP]["name"], 
		GJB.Expansions[GJB.EXP_WOD]["name"], 
		GJB.Expansions[GJB.EXP_EVENTS]["name"],
		GJB.Expansions[GJB.EXP_BLIZZJB]["name"],
		GJB.Expansions[GJB.EXP_LEGION]["name"],
	}
end
-- ------------------------------------------------------------
function GJB:GetExpac(info) return GJB.expac end
-- ------------------------------------------------------------
function GJB:GetTitle(info) return GJB.title end
-- ------------------------------------------------------------
-- Gets a list of the currently selected titles for a specific expac
function GJB:GetCurExpacTitleList()
	local t = {}
	for k, v in pairs( self.musictable[self:GetExpac()] ) do
		tinsert( t, v.name )
	end
	return t
end
-- ------------------------------------------------------------
-- Gets a list of the files within currently selected expac and title (for use with sample song)
function GJB:getSampleSongList()
	local t = {}
	local f = self.musictable[self:GetExpac()][self:GetTitle()].files
	for k, v in pairs( f ) do
		t[k] = v[GJB.MT_MUSICFILE]
	end
	return t
end
-- ------------------------------------------------------------
-- Get a list of songs within selected expac and title that can be excluded
function GJB:getExcludedSongList()
	local t = {}
	if self.db.profile.musiclist[GJB.jukebox] ~= nil then
		local xid = self.db.profile.musiclist[GJB.jukebox].expac
		local tid = self.db.profile.musiclist[GJB.jukebox].title
		local tindex = 0
		
		for k, v in pairs( self.musictable[xid] ) do
			if v.id == tid then
				tindex = k
				break
			end
		end 
		
		if tindex > 0 then
			for k2, v2 in pairs( self.musictable[xid][tindex].files ) do
				t[ v2[GJB.MT_MUSICID] ] = v2[GJB.MT_MUSICFILE]
			end
		end
	end
	
	return t
end
-- ------------------------------------------------------------
function GJB:SetExcludedSong(info, k, v)
	local xl = self.db.profile.musiclist[self.jukebox].xlist
	if xl ~= nil and k ~= nil then
		if v == true then
			xl[k] = v
		else
			xl[k] = nil
		end

		local xid = self.db.profile.musiclist[GJB.jukebox].expac
		local tid = self.db.profile.musiclist[GJB.jukebox].title
		local tindex = 0
		
		for k, v in ipairs( self.musictable[xid] ) do
			if v.id == tid then
				tindex = k
				break
			end
		end 
		
		if tindex > 0 then
			if table.getn( self.musictable[xid][tindex].files) == table.getn(xl) then
				xl[k] = nil
				self:Print("|cffff0000Excluding all music from this title is not permitted. Remove the title instead.|r")
			end
		end
	end
end

-- ------------------------------------------------------------
function GJB:GetExcludedSong(info, k)
	local xl = self.db.profile.musiclist[GJB.jukebox].xlist
	if xl ~= nil and k ~= nil then
		return xl[k]
	end
	return nil
end

-- ------------------------------------------------------------
-- ZONES
-- ------------------------------------------------------------
-- Get a table of all continent, world and category names
function GJB:GetContinents()
	return {
		GJB.Continents[GJB.ZONE_KALIMDOR]["name"], 
		GJB.Continents[GJB.ZONE_EASTERN]["name"], 
		GJB.Continents[GJB.ZONE_OUTLAND]["name"], 
		GJB.Continents[GJB.ZONE_NORTH]["name"], 
		GJB.Continents[GJB.ZONE_CATA]["name"], 
		GJB.Continents[GJB.ZONE_PANDA]["name"], 
		GJB.Continents[GJB.ZONE_DRAENOR]["name"], 
		GJB.Continents[GJB.ZONE_RAIDS]["name"], 
		GJB.Continents[GJB.ZONE_DUNGEONS]["name"], 
		GJB.Continents[GJB.ZONE_BATTLEGROUNDS]["name"], 
		GJB.Continents[GJB.ZONE_SCENARIOS]["name"], 
		GJB.Continents[GJB.ZONE_LEGION]["name"], 
	}
end
-- ------------------------------------------------------------
function GJB:GetContinent(info) return GJB.continent end
-- ------------------------------------------------------------
function GJB:GetZone(info) return GJB.zone end
-- ------------------------------------------------------------
-- Get a list of zones within the selected continent/category
function GJB:GetCurContinentZoneList()
	local tbl = self.zonetable[self:GetContinent()]
	local t = {}
	for k, v in ipairs(tbl) do
		tinsert(t, v[GJB.ZT_ZONENAME])
	end
	return t
end
-- ------------------------------------------------------------
-- Get a list of zone within the selected continent
function GJB:GetZoneList()
	local tbl = self.db.profile.zonelist
	local t = {}
	for k, v in ipairs(tbl) do
		tinsert(t, v[GJB.ZL_ZONENAME])
	end
	return t
end

-- ------------------------------------------------------------
-- ADVANCED
-- ------------------------------------------------------------
function GJB:GetAdvExpac(info) return GJB.advexpac end
function GJB:GetAdvTitle(info) return GJB.advtitle end
function GJB:GetAdvContinent(info) return GJB.advcont end
function GJB:GetAdvZone(info) return GJB.advzone end

-- ------------------------------------------------------------
function GJB:GetAdvCurContZoneList()
	local tbl = self.zonetable[self:GetAdvContinent()]
	local t = {}
	for k, v in ipairs(tbl) do
		tinsert(t, v[GJB.ZT_ZONENAME])
	end
	return t
end

-- ------------------------------------------------------------
function GJB:GetAdvZoneList()
	local zlist = GJB.db.profile.advzonelist
	local t = {}
	for k, v in pairs(zlist) do
		tinsert(t, v.name)
	end
	return t
end

-- ------------------------------------------------------------
function GJB:GetAdvTitlesForZone()
	local t = {}
	if GJB.advzonebox ~= nil then
		local zlist = GJB.db.profile.advzonelist
		if zlist[GJB.advzonebox] ~= nil then
			local cid = zlist[GJB.advzonebox].cont
			local zid = zlist[GJB.advzonebox].zone
			
			for k, v in ipairs(zlist[GJB.advzonebox].titles) do
				tinsert(t, v.name)
			end
		end
	end
	return t
end

-- ------------------------------------------------------------
function GJB:GetAdvCurExpacTitleList()
	local tbl = self.musictable[self:GetAdvExpac()]
	local t = {}
	for k, v in ipairs(tbl) do
		tinsert(t, v.name)
	end
	return t
end

-- ------------------------------------------------------------
function GJB:GetAdvExcludedSongList()
	local t = {}
	if GJB.advzonebox ~= nil and GJB.advtitlebox ~= nil then
		local ztbl = self.db.profile.advzonelist[GJB.advzonebox]
		
		if ztbl ~= nil then
			local ttbl = ztbl.titles[GJB.advtitlebox]

			if ttbl ~= nil then
				local xid, tid = 0, 0
				for k, v in ipairs(self.Continents) do
					if v.id == ttbl.expac then
						xid = k
						break
					end
				end
				
				if xid > 0 then
					for k2, v2 in ipairs(self.musictable[xid]) do
						if v2.id == ttbl.title then
							tid = k2
							break
						end
					end
				
					if tid > 0 then
						for k3, v3 in ipairs( self.musictable[xid][tid].files ) do
							t[ v3[GJB.MT_MUSICID] ] = v3[GJB.MT_MUSICFILE]
						end
					end
				end
			end
		end
	end
	return t
end

-- ------------------------------------------------------------
function GJB:SetAdvExcludedSong(info, k, v)
	local sl = GJB.db.profile.advzonelist[GJB.advzonebox].titles[GJB.advtitlebox].xlist
	if sl ~= nil and k ~= nil then
		if v == true then
			sl[k] = v
		else
			sl[k] = nil
		end
	end
end

-- ------------------------------------------------------------
function GJB:GetAdvExcludedSong(info, k)
	local sl = GJB.db.profile.advzonelist[GJB.advzonebox].titles[GJB.advtitlebox].xlist
	if sl ~= nil and k ~= nil then
		return sl[k]
	end
	return nil
end

-- ------------------------------------------------------------
-- SINGLE TRACK MODE
-- ------------------------------------------------------------
function GJB:GetSTMExpac(info) return GJB.stm_expac end
function GJB:GetSTMTitle(info) return GJB.stm_title end

-- ------------------------------------------------------------
function GJB:getSTMSongList()
	local t = {}
	local f = self.musictable[GJB.stm_expac][GJB.stm_title].files
	for k,v in ipairs(f) do
		t[k] = v[GJB.MT_MUSICFILE]
	end
	return t
end

-- ------------------------------------------------------------
function GJB:GetCurSTMExpacMusicList()
	local tbl = self.musictable[self:GetSTMExpac()]
	local t = {}
	for k, v in ipairs(tbl) do
		tinsert(t, v.name)
	end
	return t
end

-- ------------------------------------------------------------
function GJB:GetSTMMusicList()
	local tbl = GJB.db.profile.stm_files
	local t = {}
	for k, v in ipairs(tbl) do
		local filename = self:ExtractMP3Filename(v.file)
		local expac = self.Expansions[v.expac].name
		local title = self.musictable[v.expac][self.mtid[v.expac][v.title].key].name
		tinsert( t, expac .. " / " .. title .. " : " .. filename )
	end
	return t
end

--===================================
-- CORE FUNCTIONS
--===================================
-- -------------------------------------------
-- function called to determine if music needs to be played or stopped
-- Priority is given in this order: Advanced (per zone), STM (file mode), Jukebox (regular)
function GJB:ProcessInfo()
	local playstate = false
	local stopmus = false
	--self:Print("ProcessInfo gCurZoneID: " .. tostring(gCurZoneID))
	
	-- check if advanced is enabled
	if self.db.profile.advenabled == true then
		if self:IsInAdvZoneList(self.gCurZoneID) then
			playstate = true
		else
			stopmus = true		
		end
	elseif self.db.profile.allzones == false then
		if self:IsInZoneList(self.gCurZoneID) then
			if self.gPetBattleEnabled == true then
				if self.db.profile.petbattle == false then
					playstate = true
				end
			else
				playstate = true
			end
		end
	else
		playstate = true
	end
	
	if playstate == true then
		if self.db.profile.advenabled == true then
			self.gMaxTitles = 0
			local found = false
			for k, v in ipairs(self.db.profile.advzonelist) do
				if v.zone == self.gCurZoneID then
					self.gMaxTitles = table.getn(v.titles)
				end
			end
		elseif self.db.profile.stmode == true then
			self.gMaxTitles = table.getn(self.db.profile.stm_files)
		else
			self.gMaxTitles = table.getn(self.db.profile.musiclist)
		end

		self:PrepareToPlay()

	elseif self.gMusicPlaying == true and self.gPetBattleEnabled == false then
		stopmus = true
	elseif self.gMusicPlaying == true and self.gPetBattleEnabled == true and self.db.profile.petbattle == false then
		stopmus = true
	end
	
	if stopmus then
		StopMusic()
		self.gMusicPlaying = false
		self:CancelTimer(self.musicTimer)
		self:SetNowPlayingText( "..." )
	end
end

-- -------------------------------------------
-- verifies if the submitted zone name is in the selected zones (zonelist)
-- -------------------------------------------
function GJB:IsInZoneList(id)
	for _, v in ipairs(self.db.profile.zonelist) do
		if v.zone == id then
			return true
		end
	end
	return false
end

-- -------------------------------------------
-- verifies if the submitted zone name is in the selected zones (zonelist)
-- -------------------------------------------
function GJB:IsInAdvZoneList(id)
	for _, v in ipairs(self.db.profile.advzonelist) do
		if v.zone == id then
			return true
		end
	end
	return false
end

-- -------------------------------------------
-- function to verify if item is excluded
-- -------------------------------------------
function GJB:SongInExcludeList(data)
	local t = self.db.profile.excludelist
	if t == nil then return false end

	local id = self.musictable[data.xpac][data.title].files[data.song][self.MT_MUSICID]

	for k, v in pairs(t) do
		if k == id then
			return true
		end
	end
	return false
end

-- -------------------------------------------
-- function to verify if item is excluded
-- -------------------------------------------
function GJB:SongInAdvExcludeList(ztbl, music, song)
	local t = ztbl.blist[music]
	if t == nil then return false end

	-- check if all are not to be played.
	local c = 0
	local tot = 0
	for k,v in pairs(t) do
		if v == true then
			c = c + 1
		end
	end
	
	local files = self:MusicListFromJukeBoxItem(music)
	if c ==  #files then
		self:Print(LTC:setColor(L["MSG_ALLEXCLUDED_PLAY"] .. " (" .. music .. ")", LTC_RED))
		return false
	end
	
	for k2, v2 in pairs(t) do
		if k2 == song and v2 == true then
			return true
		end
	end
	return false
end

-- -------------------------------------------
-- function to play in Single Track Mode
-- -------------------------------------------
function GJB:PlaySTMMusic()
	-- make sure there's something to play.
	local files = self.db.profile.stm_files
	if files == nil or table.getn(files) == 0 then return end
	
	self.gMaxMusic = table.getn(files) -- Maximum number of songs in stm_files

	if self.db.profile.mshuffle == true then
		if self.gMaxMusic > 1 then
			self.gMusicIndex = mrand(1, self.gMaxMusic)
			while self.gMusicIndex == self.gMusicIndexPrev do
				self.gMusicIndex = mrand(1, self.gMaxMusic)
			end
		end
	else
		if self.gMusicIndexPrev == 0 then -- first pass!
			self.gMusicIndex = 1
		end
	end
	
	StopMusic()
	self:CancelTimer(self.musicTimer)
	self:SetNowPlayingText( "..." )

	local xindex = files[self.gMusicIndex].expac
	local tindex = self.mtid[xindex][files[self.gMusicIndex].title].key
	local mindex = self.mtid[xindex][files[self.gMusicIndex].title][files[self.gMusicIndex].index]
	local song = self.musictable[xindex][tindex].files[mindex]
	
	local ptime = song[self.MT_MUSICLEN]
	local mins = floor(ptime/60)
	local secs = ptime - (mins * 60)

	if self.db.profile.history.enabled then
		local histdata = {}
		histdata.expac = xindex
		histdata.title = tindex
		histdata.index = mindex
		histdata.xname = GJB.Expansions[xindex].name
		histdata.tname = song[self.MT_MUSICFILE]
		histdata.file = song[self.MT_MUSICFILE]
		histdata.time = ptime
		GJB:AddToHistory(histdata)
		GJB:UpdateHistory()
	end
	
	PlayMusic(song[GJB.MT_MUSICFILE])
	GJB:SetNowPlayingText( GJB:ExtractMP3Filename( song[GJB.MT_MUSICFILE] ) )

	if GJB.db.profile.chatoutput then
		self:Print(L["PLAYING"] .. "(STMode)" .. song[self.MT_MUSICFILE] .. " (" .. files[self.gMusicIndex].index .. ")  " .. mins .. ":" .. fmt("%02d", secs))
	end
	
	gMusicPlaying = true
	self.musicTimer = self:ScheduleTimer("TimerEventMusic", song[self.MT_MUSICLEN])
	
	self.gMusicIndexPrev = self.gMusicIndex
	self.gMusicIndex = self.gMusicIndex + 1

	if self.gMusicIndex > self.gMaxMusic then
		self.gMusicIndex = 1
	end
end

-- -------------------------------------------
-- function to play a random title and music in conjunction with the "All Music" option
-- -------------------------------------------
function GJB:PlayAllMusic()
	self:CancelTimer(self.musicTimer)
	-- play a random song from the musictable
	local xpac = mrand(1, #self.Expansions)
	local title = mrand(1, #self.musictable[xpac])
	local music = mrand(1, #self.musictable[xpac][title].files)
	local song = self.musictable[xpac][title].files[music]
	
	local ptime = song[GJB.MT_MUSICLEN]
	local tmins = floor(ptime/60)
	local tsecs = ptime - (tmins * 60)

	if self.db.profile.history.enabled then
		local histdata = {}
		histdata.expac = xpac
		histdata.title = title
		histdata.index = song
		histdata.xname = GJB.Expansions[xpac].name .. " / " .. self.musictable[xpac][title].name
		histdata.tname = song[GJB.MT_MUSICFILE]
		histdata.file = song[GJB.MT_MUSICFILE]
		histdata.time = ptime
		GJB:AddToHistory(histdata)
		GJB:UpdateHistory()
	end
	
	PlayMusic(song[GJB.MT_MUSICFILE])
	GJB:SetNowPlayingText( GJB:ExtractMP3Filename( song[GJB.MT_MUSICFILE] ) )

	if self.db.profile.chatoutput then
		self:Print(L["PLAYING"] .. "#" .. title .. " - " .. GJB.Expansions[xpac].name .. " / " .. self.musictable[xpac][title].name .. " \013(" .. song[GJB.MT_MUSICFILE] .. ") " .. tmins .. ":" .. fmt("%02d", tsecs))
	end
	
	self.gMusicPlaying = true
	self.musicTimer = self:ScheduleTimer("TimerEventMusic", ptime)
end

-- -------------------------------------------
-- function to filter out excluded files from the title's files.
-- -------------------------------------------
function GJB:GetNonExcludedSongList( songs, xlist )
	local t = {}
	for k, v in ipairs(songs) do
		local found = false
		for k2, v2 in pairs(xlist) do
			if k2 == v[GJB.MT_MUSICID] then
				found = true
				break
			end
		end
		if found == false then
			tinsert(t, v)
		end
	end
	return t
end

function GJB:PlaySilence()
	self:CancelTimer(self.musicTimer)
	StopMusic()
	GJB:SetNowPlayingText("...")
	PlayMusic("Interface\\AddOns\\GarrisonJukeBox\\media\\Empty_Sound_5s.ogg")
	gMusicPlaying = true
	GJB.silentpausestate = true
	self.musicTimer = self:ScheduleTimer("TimerEventMusic", tonumber(self.db.profile.silentpause))
end

-- -------------------------------------------
-- function get a list of titles to be played within a specific zone
-- -------------------------------------------
function GJB:GetAdvTitlesForZoneID(zid)
	local t = {}
	
	for k, v in ipairs(self.db.profile.advzonelist) do
		if v.zone == zid then
			t = v.titles
			break
		end
	end
	
	return t
end

-- -------------------------------------------
-- function to play an item from the jukebox
-- -------------------------------------------
function GJB:PlayJukebox(adv)
	-- make sure there's something to play.
	local ml = {}
	
	-- set appropriate musiclist
	if adv == true then
		ml = self:GetAdvTitlesForZoneID(self.gCurZoneID)
	else
		ml = self.db.profile.musiclist
	end
	
	if ml == nil or table.getn(ml) == 0 then 
		self:SetNowPlayingText( "..." )
		return
	end
	
	self.gMaxTitles = table.getn(ml) -- Maximum number of titles in musiclist

	-- select the title to be played
	if self.db.profile.tshuffle then
		if self.gMaxTitles > 1 then
			self.gTitleIndex = mrand(1, self.gMaxTitles)
			while self.gTitleIndex == self.gTitleIndexPrev do
				self.gTitleIndex = mrand(1, self.gMaxTitles)
			end
		end
	else
		if self.gTitleIndexPrev == 0 then -- first pass!
			self.gTitleIndex = 1
		end
	end
	
	-- Get the song to be played using fast indexing.
	local xindex = ml[self.gTitleIndex].expac
	local tindex = self.mtid[xindex][ml[self.gTitleIndex].title].key
	local title = self.musictable[xindex][tindex]
	-- local title = self:FindTitleInMusicList( ml[self.gTitleIndex].expac, ml[self.gTitleIndex].title )
	local nxt = self:GetNonExcludedSongList( title.files, ml[self.gTitleIndex].xlist)
	
	self.gMaxMusic = table.getn( nxt )
	if self.gMaxMusic == 0 then return end
	
	if self.db.profile.mshuffle then
		if self.gMaxMusic > 1 then
			self.gMusicIndex = mrand(1, self.gMaxMusic)
			while self.gMusicIndex == self.gMusicIndexPrev do
				self.gMusicIndex = mrand(1, self.gMaxMusic)
			end
		else
			self.gMusicIndex = 1 -- make sure index = 1
		end
	else
		if self.gMusicIndexPrev == 0 then -- first pass!
			self.gMusicIndex = 1
		end
	end

	local nxf = nxt[self.gMusicIndex]
	local mid = nxf[GJB.MT_MUSICID]
	local mfile = nxf[GJB.MT_MUSICFILE]
	local mtime = nxf[GJB.MT_MUSICLEN]

	-- check if we send music to other party members
	if self.db.profile.shareparty and UnitIsGroupLeader("player") then
		local data = {}
		data.expac = xindex
		data.title = tindex
		data.index = self.gMusicIndex 
		self:sendMusicToParty(data)
	end
					
	if self.db.profile.history.enabled then
		local histdata = {}
		histdata.expac = xindex
		histdata.title = tindex
		histdata.index = self.gMusicIndex
		histdata.xname = ml[self.gTitleIndex].name 
		histdata.tname = nxf[GJB.MT_MUSICFILE]
		histdata.file = nxf[GJB.MT_MUSICFILE]
		histdata.time = mtime
		GJB:AddToHistory(histdata)
		GJB:UpdateHistory()
	end
	
	PlayMusic(nxf[GJB.MT_MUSICFILE])
	GJB:SetNowPlayingText( GJB:ExtractMP3Filename( nxf[GJB.MT_MUSICFILE] ) )

	local ptime = mtime
	local tmins = floor(ptime/60)
	local tsecs = ptime - (tmins * 60)

	if self.db.profile.chatoutput then
		self:Print(L["PLAYING"] .. "#" .. self.gTitleIndex .. " - " .. ml[self.gTitleIndex].name .. " \013(" .. mfile .. ") " .. tmins .. ":" .. fmt("%02d", tsecs))
	end
	
	self.gMusicPlaying = true
	self.musicTimer = self:ScheduleTimer("TimerEventMusic", mtime)

	self.gTitleIndexPrev = self.gTitleIndex
	self.gMusicIndexPrev = self.gMusicIndex

	self.gMusicIndex = self.gMusicIndex + 1

	if self.gMusicIndex > self.gMaxMusic then
		self.gMusicIndex = 1
		self.gTitleIndex = self.gTitleIndex + 1

		if self.gTitleIndex > self.gMaxTitles then
			self.gTitleIndex = 1
		end
	end
end

-- -------------------------------------------
-- function to process what music is to be played
-- -------------------------------------------
function GJB:PrepareToPlay()
	-- do we play a silence?
	if self.db.profile.silentpauseon == true and GJB.silentpausestate == false then
		self:PlaySilence()
	else
		-- make sure we turn off the silent pause state in preparation for the next song
		self.silentpausestate = false
		
		--self.gErrorPlay = false
		self:CancelTimer(self.musicTimer)

		-- What type are we going to play
		if self.db.profile.advenabled == true then
			-- Advanced (per zone config)
			self:PlayJukebox(true) -- pass true to tell PlayJukebox to use the advmusiclist
		elseif self.db.profile.stmode == true then
			-- Single Track Mode
			self:PlaySTMMusic()
		elseif self.db.profile.allmusic == true then -- disregard the jukebox list and play a random file
			-- All Music enabled
			self:PlayAllMusic()
		else
			-- play normal jukebox 
			self:PlayJukebox(false) -- pass fasle to tell PlayJukebox to use the musiclist
		end
	end
end
