----------------------------
--      Constants         --
----------------------------

local _

local CASTING_SOUND_FILE = "Interface\\AddOns\\FocusInterruptSounds\\casting.ogg"
local CC_SOUND_FILE = "Interface\\AddOns\\FocusInterruptSounds\\cc.ogg"
local INTERRUPTED_SOUND_FILE = "Interface\\AddOns\\FocusInterruptSounds\\interrupted.ogg"
local POLYMORPH_SOUND_FILE = "Interface\\AddOns\\FocusInterruptSounds\\sheep.ogg"
local INNERVATE_SOUND_FILE = "Interface\\AddOns\\FocusInterruptSounds\\innervate.ogg"

local SCHOOL_PHYSICAL	= 0x01;
local SCHOOL_HOLY	= 0x02;
local SCHOOL_FIRE	= 0x04;
local SCHOOL_NATURE	= 0x08;
local SCHOOL_FROST	= 0x10;
local SCHOOL_SHADOW	= 0x20;
local SCHOOL_ARCANE	= 0x40;
local SCHOOL_ALL	= 0x7F;

local DEFAULT_GLOBAL_OVERRIDES = 
[[
Sigryn->Blood of the Father
Tugar Bloodtotem->Fel Burst
]];

local DEFAULT_BLACKLIST = 
[[
Constellar Designate->Starblast
Inquisitor Variss->Mind Rend
Corrupting Shadows->Seed of Darkness
]];

local DEFAULT_PLAYER_INTERRUPT_SPELLS =
[[
Avenger's Shield
Command Demon|Spell Lock
Consume Magic
Counter Shot
Counterspell
Disrupting Shout
Kick
Mind Freeze
Muzzle
Pummel
Rebuke
Silence
Silencing Shot
Skull Bash
Solar Beam
Spear Hand Strike
Strangulate
Wind Shear
]];

local DEFAULT_PET_INTERRUPT_SPELLS =
[[
Nether Shock
Optical Blast
Pummel
Spell Lock
Shadow Lock
]];

local DEFAULT_AURA_BLACKLIST = 
[[Nature's Swiftness -> *
Presence of Mind -> *
Divine Shield -> *
Dark Fury -> *
]];

local DEFAULT_INCOMING_CC = 
[[Cyclone
Fear
Polymorph
Seduction
]];

local DEFAULT_INCOMING_CC_LASHBACK = 
[[Polymorph
Seduction
]];

local DEFAULT_PARTNER_CC_MAGIC = 
[[Polymorph
Repentance
Seduction
]];

local DEFAULT_PARTNER_CC_POISON = 
[[Wyvern Sting
]];


local DEFAULT_ARENA_PURGE = 
[[Innervate
]];

local DEFAULT_PVE_PURGE = 
[[Pyrogenics
Remedy
Rune Shield
Runic Barrier
]];

------------------------------
--      Initialization      --
------------------------------

FocusInterruptSounds = LibStub("AceAddon-3.0"):NewAddon("FocusInterruptSounds", "AceEvent-3.0", "AceConsole-3.0")

local options = {
	type = "group",
	name = "FocusInterruptSounds",
	get = function(info) return FocusInterruptSounds.db.profile[ info[#info] ] end,
	set = function(info, value) FocusInterruptSounds.db.profile[ info[#info] ] = value end,
	args = {
		General = {
			order = 1,
			type = "group",
			name = "General Settings",
			desc = "General Settings",
			args = {
				intro = {
					order = 1,
					type = "description",
					name = "FocusInterruptSounds plays sounds when hostile targets cast interruptable spells. "
							.. "It also has other special sound events for PvP and PvE.  Found a bug?  Send a "
							.. "in-game mail to Corg, Horde, Detheroc, US PvP (chances are I won't be watching "
							.. "the Ace forums.",
				},

				fEnableText = {
					type = "toggle",
					name = "Enable text",
					desc = "Enable/disable chatframe text from FocusInterruptSounds.",
					order = 2,
				},
				fEnableSound = {
					type = "toggle",
					name = "Enable sound",
					desc = "Enable/disable sounds in FocusInterruptSounds.",
					order = 2,
				},
				fIgnoreMute = {
					type = "toggle",
					name = "Ignore mute",
					desc = "Ignores/respects WoW's mute setting (ctrl-n).  Only applicable if sound is enabled.",
					order = 3,
				},
				fTargetFallback = {
					type = "toggle",
					name = "Target fallback",
					desc = "Allow interrupt sounds on target if not focus is set.",
					order = 4,
				},
				fPositiveReinforcement = {
					type = "toggle",
					name = "Positive reinforcement",
					desc = "Play a victory sound and show a victory message when you successfully interrupt.",
					order = 5,
				},
				fCheckSpellAvailability = {
					type = "toggle",
					name = "Check spell availability",
					desc = "Require that the interrupt or anti-CC spell can be cast before generating a warning.",
					order = 6,
				},
				fDisableInVehicle = {
					type = "toggle",
					name = "Disable in vehcile",
					desc = "Turns off sounds while in a vehicle.",
					order = 7,
				},
				fAnnounceInterrupts = {
					type = "toggle",
					name = "Announce interrupts",
					desc = "Announce successful interrupts to Party/Raid",
					order = 8,
				},
				iMinimumCastTime = {
					type = "range",
					name = "Minimum cast time (ms)",
					desc = "Spells casting faster than this will not generate a warning.  Time interval specified in milliseconds.",
					order = 9,
					softMin = 0,
					softMax = 3000,
					min = 0,
					max = 20000,
					bigStep = 100,
				},


				strGlobalOverrides = {
					type = "input",
					name = "Global Overrides",
					desc = "List of casters and spells should always generate a warning (even if they are not "
							.. "your target or focus), separated by a \"->\".  Use \"*\" to match any caster "
							.. "or any spell (but not both!).",
					order = 10,
					multiline = true,
					width = "double",
				},

				strBlacklist = {
					type = "input",
					name = "Caster -> Spell Blacklist",
					desc = "List of spells that should be ignored for a given caster, separated by a \"->\".  "
							.. "Use \"*\" to match any caster or any spell (but not both!).",
					order = 11,
					multiline = true,
					width = "double",
				},
				fIgnorePhysical = {
					type = "toggle",
					name = "Blacklist physical spells",
					desc = "Ignore spells classified as \"Physical.\"",
					order = 12,
				},
				fEnableBlizzardBlacklist = {
					type = "toggle",
					name = "Blizzard API blacklist",
					desc = "Ignore casts that UnitCastingInfo() classifies as uninterruptable.",
					order = 13,
				},
				strAuraBlacklist = {
					type = "input",
					name = "Aura -> Spell Blacklist",
					desc = "List of spells that should be ignored for a given aura, separated by a \"->\".  "
							.. "Use \"*\" to match any spell.",
					order = 14,
					multiline = true,
					width = "double",
				},

				strPlayerInterruptSpells = {
					type = "input",
					name = "Player Interrupt Spells",
					desc = "List of interrupt spells available to the player.  "
							.. "Only used if \"Check spell availability\" is enabled.",
					order = 15,
					multiline = true,
					width = "double",
				},

				strPetInterruptSpells = {
					type = "input",
					name = "Pet Interrupt Spells",
					desc = "List of interrupt spells available to the player's pet.  "
							.. "Only used if \"Check spell availability\" is enabled.",
					order = 16,
					multiline = true,
					width = "double",
				},

				strIncomingCC = {
					type = "input",
					name = "PvP Incoming CC Spells",
					desc = "List of spells that should sound a warning for incoming CC in arenas or nearby.",
					order = 17,
					multiline = true,
					width = "double",
				},

				strPartnerCC = {
					type = "input",
					name = "Arena Parner CC Debuffs",
					desc = "List of debuffs that should sound a warning if applied to your arena partner.",
					order = 18,
					multiline = true,
					width = "double",
				},

				strArenaPurge = {
					type = "input",
					name = "Arena Purge Buffs",
					desc = "List of buffs that should sound a warning when gained by an arena opponent.",
					order = 19,
					multiline = true,
					width = "double",
				},

				strPvePurge = {
					type = "input",
					name = "PvE Purge Buffs",
					desc = "List of buffs that should be purged from NPCs.",
					order = 20,
					multiline = true,
					width = "double",
				},
			},
		},
	},
};


function FocusInterruptSounds:OnInitialize()

	local strGlobalOverrides = DEFAULT_GLOBAL_OVERRIDES;
	local strAuraBlacklist = DEFAULT_AURA_BLACKLIST;
	local strPlayerInterruptSpells = DEFAULT_PLAYER_INTERRUPT_SPELLS;
	local strPetInterruptSpells = DEFAULT_PET_INTERRUPT_SPELLS;
	local strIncomingCC = "";
	local strPartnerCC = "";
	local strPvePurge = "";

	_, self.strClassName = UnitClass("player");

	self.fAntiCCIsLashback = false;
	self.fHasPurge = false;
	self.fCanDispel = false;
	self.fCanDepoison = false;

	if ("WARLOCK" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_SHADOW;
		self.str30YardSpellName = "Shoot";
		self.fHasPurge = true;
		self.fCanDispel = true;
	elseif ("MAGE" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_ARCANE;
		self.str30YardSpellName = "Shoot";
		self.fHasPurge = true;
	elseif ("SHAMAN" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_NATURE;
		self.strAntiCCSpellName = "Grounding Totem";
		self.str30YardSpellName = "Lightning Bolt";
		self.fHasPurge = true;
		self.fCanDepoison = true;
	elseif ("WARRIOR" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_PHYSICAL;
		self.str30YardSpellName = "Shoot";
	elseif ("ROGUE" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_PHYSICAL;
		self.strAntiCCSpellName = "Cloak of Shadows";
		self.str30YardSpellName = "Shoot";
	elseif ("PRIEST" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_SHADOW;
		self.strAntiCCSpellName = "Shadow Word: Death";
		self.fAntiCCIsLashback = true;
		self.str30YardSpellName = "Shadow Word: Pain";
		self.fHasPurge = true;
		self.fCanDispel = true;
	elseif ("HUNTER" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_PHYSICAL;
		self.strAntiCCSpellName = "Feign Death";
		self.str30YardSpellName = "Arcane Shot";
	elseif ("DRUID" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_PHYSICAL;
		self.str30YardSpellName = "Faerie Fire";
		self.fCanDepoison = true;
	elseif ("DEATHKNIGHT" == self.strClassName) then
		self.iInterruptSchool = SCHOOL_FROST;
		self.strAntiCCSpellName = "Anti-Magic Shell";
		self.str30YardSpellName = "Strangulate";
	elseif ("PALADIN" == self.strClassName) then
		self.fCanDispel = true;
		self.fCanDepoison = true;
	elseif ("MONK" == self.strClassName) then
		self.fCanDispel = true;
		self.fCanDepoison = true;
	end

	-- Add additional auras for classes with physical interrupts
	if (self.iInterruptSchool == SCHOOL_PHYSICAL) then
		strAuraBlacklist = strAuraBlacklist .. "Hand of Protection -> *\n";
	end

	-- Set up incoming CC warning defaults
	if (nil ~= self.strAntiCCSpellName) then
		if (self.fAntiCCIsLashback) then
			strIncomingCC = DEFAULT_INCOMING_CC_LASHBACK;
		else
			strIncomingCC = DEFAULT_INCOMING_CC;
		end
	end

	-- Set up partner CC defaults
	if (self.fCanDispel) then
		strPartnerCC = strPartnerCC .. DEFAULT_PARTNER_CC_MAGIC;
	end

	if (self.fCanDepoison) then
		strPartnerCC = strPartnerCC .. DEFAULT_PARTNER_CC_POISON;
	end

	-- Set up purge defaults
	if (self.fHasPurge) then
		strPvePurge = DEFAULT_PVE_PURGE;
	end

	-- Build the default settings array
	local DEFAULTS = {
		profile = {
			fEnableText = true,
			fEnableSound = true,
			fIgnoreMute = true,
			fTargetFallback = true,
			fPositiveReinforcement = true,
			fCheckSpellAvailability = true,
			fDisableInVehicle = true,
			fAnnounceInterrupts = true,

			iMinimumCastTime = 800,
			strGlobalOverrides = strGlobalOverrides,
			strBlacklist = DEFAULT_BLACKLIST,
			fIgnorePhysical = false,
			fEnableBlizzardBlacklist = true,
			strAuraBlacklist = strAuraBlacklist,
			strPlayerInterruptSpells = strPlayerInterruptSpells,
			strPetInterruptSpells = strPetInterruptSpells,
			strIncomingCC = strIncomingCC,
			strPartnerCC = strPartnerCC,
			strArenaPurge = DEFAULT_ARENA_PURGE,
			strPvePurge = strPvePurge,
		}
	};
	self.db = LibStub("AceDB-3.0"):New("FocusInterruptSoundsDB", DEFAULTS, self.strClassName)

	options.args.Profile = LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db)
	LibStub("AceConfig-3.0"):RegisterOptionsTable("FocusInterruptSounds", options)
	LibStub("AceConfigDialog-3.0"):SetDefaultSize("FocusInterruptSounds", 640, 480)
	LibStub("AceConfigDialog-3.0"):AddToBlizOptions("FocusInterruptSounds", nil, nil, "General")
	LibStub("AceConfigDialog-3.0"):AddToBlizOptions("FocusInterruptSounds", "Profile", "FocusInterruptSounds", "Profile")
	self:RegisterChatCommand("fis", function() LibStub("AceConfigDialog-3.0"):Open("FocusInterruptSounds") end)

end


function FocusInterruptSounds:OnEnable()
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED");
	self:CheckAndPrintMessage("Add-on activated for the class " .. self.strClassName);
end

function FocusInterruptSounds:OnDisable()
	self:UnregisterEvent("COMBAT_LOG_EVENT_UNFILTERED");
end

------------------------------
--        Functions         --
------------------------------

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:CheckAndPrintMessage
--
--		Prints a message, only if the options permit it.
--
function FocusInterruptSounds:CheckAndPrintMessage(strMsg)

	if (self.db.profile.fEnableText) then
		DEFAULT_CHAT_FRAME:AddMessage("|cff7fff7fFIS|r: " .. tostring(strMsg));
	end

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:CheckAndPlaySound
--
--		Plays a sound, only if the options permit it.
--
function FocusInterruptSounds:CheckAndPlaySound(strFile)

	if (self.db.profile.fEnableSound) then
		local strChannel = "SFX";
		if (self.db.profile.fIgnoreMute) then
			strChannel = "MASTER"; 
		end
		PlaySoundFile(strFile, strChannel);
	end

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsSourceFocusOrTarget
--
--		Returns true if the source flags are for the target we're making sounds for.
--
function FocusInterruptSounds:FIsSourceFocusOrTarget(iSourceFlags)

	-- Filter out non-hostile sources
	if (0 == bit.band(iSourceFlags, COMBATLOG_OBJECT_REACTION_HOSTILE)) then
		return false;
	end

	-- We want to react to focus actions
	if (0 ~= bit.band(iSourceFlags, COMBATLOG_OBJECT_FOCUS)) then
		return true;
	end

	-- If there is no hostile focus, allow fallback on the current target
	if (0 ~= bit.band(iSourceFlags, COMBATLOG_OBJECT_TARGET)
			and self.db.profile.fTargetFallback
			and not UnitCanAttack("player", "focus")
	) then
		return true;
	end

	return false;
end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:StrEscapeForRegExp
--
--		Returns the string escaped for use with LUA regular expressions.
--
function FocusInterruptSounds:StrEscapeForRegExp(str)

	-- Special characters: ^$()%.[]*+-?
	str = string.gsub(str, "%^", "%%%^");
	str = string.gsub(str, "%$", "%%%$");
	str = string.gsub(str, "%(", "%%%(");
	str = string.gsub(str, "%)", "%%%)");
	str = string.gsub(str, "%%", "%%%%");
	str = string.gsub(str, "%.", "%%%.");
	str = string.gsub(str, "%[", "%%%[");
	str = string.gsub(str, "%]", "%%%]");
	str = string.gsub(str, "%*", "%%%*");
	str = string.gsub(str, "%+", "%%%+");
	str = string.gsub(str, "%-", "%%%-");
	str = string.gsub(str, "%?", "%%%?");

	return str;

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FInList
--
--		Returns true if the given element is in the given newline-delimited list.
--
function FocusInterruptSounds:FInList(strElement, strList)

	--self:CheckAndPrintMessage("Looking for " .. strElement);

	return string.find("\n" .. strList .. "\n", "\n%s*" .. self:StrEscapeForRegExp(strElement) .. "%s*\n");

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FInMap
--
--		Returns true if the given key and value are in the given newline-delimited list.
--
function FocusInterruptSounds:FInMap(strKey, strValue, strMap)

	local strKeyEscaped;
	local strValueEscaped;

	if (nil == strKey) then
		strKeyEscaped = ".*";
	else
		strKeyEscaped = self:StrEscapeForRegExp(strKey);
	end

	if (nil == strValue) then
		strValueEscaped = ".*";
	else
		strValueEscaped = self:StrEscapeForRegExp(strValue);
	end

	return string.find("\n" .. strMap .. "\n", "\n%s*" .. strKeyEscaped
				.. "%s*%->%s*" .. strValueEscaped .. "%s*\n");

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsCasterOrSpellGlobalOverride
--
--		Returns true if the given spell (or cast+spell combo) is in the global override list.
--
function FocusInterruptSounds:FIsCasterOrSpellGlobalOverride(strMobName, iMobFlags, strSpellId, strSpellName, iSpellSchool)

	-- Is the spell in the global override?
	if (self:FInMap("*", strSpellName, self.db.profile.strGlobalOverrides)) then
		return true;
	end

	-- Only allow caster overrides for NPCs
	if (0 ~= bit.band(iMobFlags, COMBATLOG_OBJECT_CONTROL_NPC)) then
		-- Is the caster blacklisted?
		if (self:FInMap(strMobName, "*", self.db.profile.strGlobalOverrides)) then
			return true;
		end

		-- Is the caster+spell combo blacklisted?
		if (self:FInMap(strMobName, strSpellName, self.db.profile.strGlobalOverrides)) then
			return true;
		end
	end

	return false;

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsCasterOrSpellBlacklisted
--
--		Returns true if the given spell (or cast+spell combo) is blacklisted.
--
function FocusInterruptSounds:FIsCasterOrSpellBlacklisted(strMobName, iMobFlags, strSpellId, strSpellName, iSpellSchool)

	--- Blacklist based on UnitCastingInfo() API
	if (self.db.profile.fEnableBlizzardBlacklist or self.db.profile.iMinimumCastTime > 0) then
		local strMobId = "target";

		if (0 ~= bit.band(iMobFlags, COMBATLOG_OBJECT_FOCUS)) then
			strMobId = "focus";
		end

		local strSpellNameVerify, _, _, _, iBFAEndTime, iEndTime, _, fBFAInterruptImmune, fInterruptImmune = UnitCastingInfo(strMobId);

		-- Is this a channel rather than a cast?
		if (nil == strSpellNameVerify) then
			strSpellNameVerify, _, _, _, iBFAEndTime, iEndTime, fBFAInterruptImmune, fInterruptImmune = UnitChannelInfo(strMobId);
		end

		-- BFA lost a return value somewhere, so check what we got back and adjust parameters if necessary
		if (type(fInterruptImmune) ~= "boolean") then
			fInterruptImmune = fBFAInterruptImmune;
		end

		if (type(iEndTime) ~= "number") then
			iEndTime = iBFAEndTime;
		end

		if (nil == strSpellNameVerify) then
			-- If the caster is no longer casting, it was probably a really fast cast (e.g. Nature's Swiftness)
			return true;
		elseif (strSpellNameVerify ~= strSpellName) then
			self:CheckAndPrintMessage("Error: UnitCastingInfo verification failed: strSpellNameVerify="
				.. strSpellNameVerify .. " strSpellName=" .. strSpellName);
		else
			if (self.db.profile.fEnableBlizzardBlacklist and fInterruptImmune) then
				return true;
			end

			if (iEndTime - GetTime() * 1000 < self.db.profile.iMinimumCastTime) then
				return true;
			end
			
		end
	end

	-- Blacklist physical spells
	if (self.db.profile.fIgnorePhysical and 0 ~= bit.band(iSpellSchool, SCHOOL_PHYSICAL)) then
		return true;
	end

	-- Is the spell blacklisted?
	if (self:FInMap("*", strSpellName, self.db.profile.strBlacklist)) then
		return true;
	end

	-- Only allow caster blacklists for NPCs
	if (0 ~= bit.band(iMobFlags, COMBATLOG_OBJECT_CONTROL_NPC)) then
		-- Is the caster blacklisted?
		if (self:FInMap(strMobName, "*", self.db.profile.strBlacklist)) then
			return true;
		end

		-- Is the caster+spell combo blacklisted?
		if (self:FInMap(strMobName, strSpellName, self.db.profile.strBlacklist)) then
			return true;
		end
	end

	return false;

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsAuraBlacklisted
--
--		Returns true if the given spell (or cast+spell combo) is blacklisted.
--
function FocusInterruptSounds:FIsAuraBlacklisted(strAura, strSpellId, strSpellName, iSpellSchool)

	-- self:CheckAndPrintMessage("id = " .. strSpellId .. "; spell name = " .. strSpellName);

	-- Is the aura blacklisted?
	if (self:FInMap(strAura, "*", self.db.profile.strAuraBlacklist)) then
		return true;
	end

	-- Is the aura+spell combo blacklisted?
	if (self:FInMap(strAura, strSpellName, self.db.profile.strAuraBlacklist)) then
		return true;
	end

	return false;

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsSpellCastStart
--
--		Returns true if the given event is the start of a spell cast.  Note that for channeled
--		spells, this is actually going to be SPELL_CAST_SUCCESS.
--
function FocusInterruptSounds:FIsSpellCastStart(strEventType, iMobFlags, strSpellId, strSpellName, iSpellSchool)

	if ("SPELL_CAST_START" == strEventType) then
		return true;
	elseif ("SPELL_CAST_SUCCESS" == strEventType) then
		local strMobId = "target";

		if (0 ~= bit.band(iMobFlags, COMBATLOG_OBJECT_FOCUS)) then
			strMobId = "focus";
		end

		local strSpellNameVerify, _, _, _, _, _, _, _ = UnitChannelInfo(strMobId);
		return strSpellNameVerify == strSpellName;
	end

	return false;

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsCCSpell
--
--		Returns true if the given event is the start of a CC.
--
function FocusInterruptSounds:FIsCCSpell(strSpellId, strSpellName, iSpellSchool)

	return self:FInList(strSpellName, self.db.profile.strIncomingCC);

end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FHasBlacklistedAura
--
--		Returns true if the focus/target has an aura that will make the caster immune to
--		interrupts or will make the cast instant.
--
function FocusInterruptSounds:FHasBlacklistedAura(iSourceFlags, strSpellId, strSpellName, iSpellSchool)

	-- Go through recently cast buffs
	if (nil ~= self.lastInstacastSelfBuffName
		and GetTime() - self.lastInstacastSelfBuffTime < 1
		and self:FIsAuraBlacklisted(self.lastInstacastSelfBuffName, strSpellId, strSpellName, iSpellSchool)
	) then
		return true;
	end

	-- Go through the current buffs
	for i = 1, 40 do
		local strBuffName;

		if (0 ~= bit.band(iSourceFlags, COMBATLOG_OBJECT_FOCUS)) then
			strBuffName, _, _, _, _, _ = UnitBuff("focus", i);
		elseif (0 ~= bit.band(iSourceFlags, COMBATLOG_OBJECT_TARGET)) then
			strBuffName, _, _, _, _, _ = UnitBuff("target", i);
		end

		if (nil ~= strBuffName and self:FIsAuraBlacklisted(strBuffName, strSpellId, strSpellName, iSpellSchool)) then
			return true;
		end
	end

	return false;
end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsPetSpellAvailable
--
--		Returns true if the pet can cast the given spell.
--
function FocusInterruptSounds:FIsPetSpellAvailable(strSpellName)

	-- Make sure the user wants these extra checks
	if (not self.db.profile.fCheckSpellAvailability) then
		return true;
	end

	-- Make sure that there is a spell
	if (nil == strSpellName) then
		return false;
	end

	-- Verify that the pet can act (i.e. isn't feared)
	if (not GetPetActionsUsable()) then
		return false;
	end

	-- Verify that the spell isn't on cooldown (also checks existence)
	local iStartTime, _, fSpellEnabled = GetSpellCooldown(strSpellName, BOOKTYPE_PET);
	if (iStartTime ~= 0 or not fSpellEnabled) then
		return false;
	end

	-- Verify mana/energy
	local _, _, _, iCost, _, _, _, _, _ = GetSpellInfo(strSpellName);
	local iPetMana = UnitPowerType("playerpet");
	if (nil == iCost or nil == iPetMana or iPetMana < iCost) then
		return false;
	end

	return true;
end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsPlayerSpellAvailable
--
--		Returns true if you can cast the given spell.
--
function FocusInterruptSounds:FIsPlayerSpellAvailable(strSpellName)

	local strSpellDisplayNameVerify = nil;

	-- Is there a | special character?
	local iBarIndex = strSpellName:find("|");
	if (nil ~= iBarIndex) then
		strSpellDisplayNameVerify = strSpellName:sub(iBarIndex + 1);
		strSpellName = strSpellName:sub(0, iBarIndex - 1);
	end

	-- Make sure the user wants these extra checks
	if (not self.db.profile.fCheckSpellAvailability) then
		return true;
	end

	-- Make sure that there is a spell
	if (nil == strSpellName) then
		return false;
	end

	-- Verify that the spell isn't on cooldown
	local iStartTime, _, fSpellEnabled = GetSpellCooldown(strSpellName);
	if (iStartTime ~= 0 or not fSpellEnabled) then
		return false;
	end

	-- Verify display name (if applicable) and mana/energy
	local strSpellDisplayName, _, _, iCost, _, _, _, _, _ = GetSpellInfo(strSpellName);
	if (nil ~= strSpellDisplayNameVerify and strSpellDisplayNameVerify ~= strSpellDisplayName) then
		return false
	elseif (UnitPowerType("player") < iCost) then
		return false;
	end

	return true;
end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:FIsInterruptAvailable
--
--		Returns true if you can cast any interrupt.
--
function FocusInterruptSounds:FIsInterruptAvailable()

	-- Make sure the user wants these extra checks
	if (not self.db.profile.fCheckSpellAvailability) then
		return true;
	end

	for strSpell in string.gmatch(self.db.profile.strPlayerInterruptSpells, "[^%s][^\r\n]+[^%s]") do
		if (self:FIsPlayerSpellAvailable(strSpell)) then
			return true;
		end
	end

	if (GetPetActionsUsable()) then
		for strSpell in string.gmatch(self.db.profile.strPetInterruptSpells, "[^%s][^\r\n]+[^%s]") do
			if (self:FIsPetSpellAvailable(strSpell)) then
				return true;
			end
		end
	end

	return false;
end

---------------------------------------------------------------------------------------------------
--	FocusInterruptSounds:COMBAT_LOG_EVENT_UNFILTERED
--
--		Handler for combat log events.
--
function FocusInterruptSounds:COMBAT_LOG_EVENT_UNFILTERED(event, iTimestamp, strEventType, fHideCaster, strSourceGuid, strSourceName, iSourceFlags, iSourceFlags2, strDestGuid, strDestName, iDestFlags, iDestFlags2, varParam1, varParam2, varParam3, varParam4, varParam5, varParam6, ...)

	-- In BFA, all parameters after the first have been moved to a separate call
	if (nil == iTimestamp) then
		iTimestamp, strEventType, fHideCaster, strSourceGuid, strSourceName, iSourceFlags, iSourceFlags2, strDestGuid, strDestName, iDestFlags, iDestFlags2, varParam1, varParam2, varParam3, varParam4, varParam5, varParam6 = CombatLogGetCurrentEventInfo()
	end

	local fHandled = false;

	-- Short circuit this processing if we're essentially disabled
	if (not self.db.profile.fEnableText and not self.db.profile.fEnableSound) then
		return
	end

	-- Track instacast buffs
	if (self:FIsSourceFocusOrTarget(iSourceFlags)
			and "SPELL_CAST_SUCCESS" == strEventType
			and self:FInMap(varParam2, nil, self.db.profile.strAuraBlacklist)
	) then
		self.lastInstacastSelfBuffName = varParam2;
		self.lastInstacastSelfBuffTime = GetTime();
	end

	-- Turn off all notifications while in a vehicle
	if (self.db.profile.fDisableInVehicle and UnitInVehicle("player")) then
		return
	end

	-- Global override sounds
	if (not fHandled
			and self:FIsSpellCastStart(strEventType, iSourceFlags, varParam1, varParam2, varParam3)
			and self:FIsCasterOrSpellGlobalOverride(strSourceName, iSourceFlags, varParam1, varParam2, varParam3)
	) then
		self:CheckAndPrintMessage(strSourceName .. " is casting |cffff4444" .. varParam2 .. "|r!");
		self:CheckAndPlaySound(CASTING_SOUND_FILE);
		fHandled = true;
	end

	-- Your partner is sheeped, play a sound
	if (not fHandled
			and 0 ~= bit.band(iDestFlags, COMBATLOG_OBJECT_AFFILIATION_PARTY)
			and 0 ~= bit.band(iDestFlags, COMBATLOG_OBJECT_REACTION_FRIENDLY)
			and "SPELL_AURA_APPLIED" == strEventType
			and self:FInList(varParam2, self.db.profile.strPartnerCC)
			and IsActiveBattlefieldArena()
	) then
		self:CheckAndPrintMessage(strDestName .. " is Sheeped!");
		self:CheckAndPlaySound(POLYMORPH_SOUND_FILE);
		fHandled = true;
	end

	-- Enemy player in an arena is innervated, play a sound
	if (not fHandled
			and 0 ~= bit.band(iDestFlags, COMBATLOG_OBJECT_REACTION_HOSTILE)
			and "SPELL_AURA_APPLIED" == strEventType
			and ((IsActiveBattlefieldArena()
					and self:FInList(varParam2, self.db.profile.strArenaPurge))
				or 0 ~= bit.band(iDestFlags, COMBATLOG_OBJECT_CONTROL_NPC)
					and self:FInList(varParam2, self.db.profile.strPvePurge))
	) then
		self:CheckAndPrintMessage(strDestName .. " has " .. varParam2 .. "!");
		self:CheckAndPlaySound(INNERVATE_SOUND_FILE);
		fHandled = true;
	end

	-- Play a sound when the Focus starts casting
	if (not fHandled
			and self:FIsSourceFocusOrTarget(iSourceFlags)
			and self:FIsSpellCastStart(strEventType, iSourceFlags, varParam1, varParam2, varParam3)
			and not self:FIsCasterOrSpellBlacklisted(strSourceName, iSourceFlags, varParam1, varParam2, varParam3)
			and not self:FHasBlacklistedAura(iSourceFlags, varParam1, varParam2, varParam3)
			and self:FIsInterruptAvailable()
	) then
		self:CheckAndPrintMessage(strSourceName .. " is casting |cffff4444" .. varParam2 .. "|r!");
		self:CheckAndPlaySound(CASTING_SOUND_FILE);
		fHandled = true;
	end

	-- Play a sound when a hostile player is attempting to CC you
	if (0 ~= bit.band(iSourceFlags, COMBATLOG_OBJECT_REACTION_HOSTILE)
			and 0 == bit.band(iSourceFlags, COMBATLOG_OBJECT_CONTROL_NPC)
			and self:FIsSpellCastStart(strEventType, iSourceFlags, varParam1, varParam2, varParam3)
			and self:FIsCCSpell(varParam1, varParam2, varParam3)
	) then
		if ((nil ~= self.strAntiCCSpellName or self:FIsSpellAvailable(self.strAntiCCSpellName))
				and (IsActiveBattlefieldArena() or 1 == IsSpellInRange(self.str30YardSpellName, strTarget))
		) then
			self:CheckAndPrintMessage(strSourceName .. " is casting CC: |cffffcc44" .. varParam2 .. "|r.");
			if (not fHandled) then
				self:CheckAndPlaySound(CC_SOUND_FILE);
				fHandled = true;
			end
		else
			self:CheckAndPrintMessage(strSourceName .. " is casting CC: |cffffcc44" .. varParam2 .. "|r (far away/inactionable).");
		end
	end

	-- Play sound when you interrupt a hostile target
	if (not fHandled
			and "SPELL_INTERRUPT" == strEventType
			and 0 ~= bit.band(iSourceFlags, COMBATLOG_OBJECT_AFFILIATION_MINE)
			and 0 ~= bit.band(iDestFlags, COMBATLOG_OBJECT_REACTION_HOSTILE)
	) then
		self:CheckAndPrintMessage("Successfully interrupted |cffaaffff" .. varParam5 .. "|r.");
		if (self.db.profile.fPositiveReinforcement) then
			self:CheckAndPlaySound(INTERRUPTED_SOUND_FILE);
		end
		if (self.db.profile.fAnnounceInterrupts) then
			local strChannel = nil;
			local fInInstance, instanceType = IsInInstance();

			if (IsInGroup(LE_PARTY_CATEGORY_INSTANCE) or IsInRaid(LE_PARTY_CATEGORY_INSTANCE) or instanceType == "pvp" or instanceType == "arena") then
				strChannel = "INSTANCE_CHAT";
			elseif (IsInRaid(LE_PARTY_CATEGORY_HOME)) then
				strChannel = "RAID";
			elseif (IsInGroup(LE_PARTY_CATEGORY_HOME)) then
				strChannel = "PARTY";
			end

			if (nil ~= strChannel) then
				SendChatMessage("[FIS] Interrupted " .. strDestName .. "'s " .. GetSpellLink(varParam4), strChannel);
			end
		end
		fHandled = true;
	end
end
