--[[ 
FireBuy - Versión Final v1.10
Addon de compras automáticas
]]

local ADDON_NAME, FB = ...
local LDB = LibStub("LibDataBroker-1.1", true)
local LibDBIcon = LibStub("LibDBIcon-1.0", true)

-- Función auxiliar para obtener una clave única para el personaje actual
local function GetPlayerKey()
    return UnitName("player") .. "-" .. GetRealmName()
end

-- Configuración persistente
-- La parte global se comparte entre personajes, mientras que la parte "char" se indexa por personaje
FB_DB = FB_DB or {
    global = {
        enabled = true,
        minimap = { hide = false },
        classConfigs = {}
    },
    char = {} -- Cada personaje tendrá su configuración guardada aquí, usando su playerKey
}

-- Colores personalizados
local COLORS = {
    red = "|cFFFF0000",
    yellow = "|cFFFFFF00",
    green = "|cFF00FF00",
    white = "|cFFFFFFFF"
}

-- Variables de control
local colaCompras = {}
local comprando = false

-- Crear objeto DataBroker para el minimapa
local fbbutton = LDB:NewDataObject("FireBuy", {
    type = "launcher",
    text = "FB",
    icon = "Interface\\Icons\\INV_Misc_Coin_01",
    
    OnClick = function(_, button)
        if button == "LeftButton" then
            FB:ShowConfigWindow()
        elseif button == "RightButton" then
            FB_DB.global.enabled = not FB_DB.global.enabled
            local status = FB_DB.global.enabled and COLORS.green.."ACTIVATED" or COLORS.red.."DEACTIVATED"
            print(COLORS.yellow.."FireBuy: "..status.."|r")
        end
    end,
})

-- Inicializar y mostrar ícono
if LibDBIcon then
    LibDBIcon:Register("FireBuy", fbbutton, FB_DB.global.minimap)
    LibDBIcon:Show("FireBuy")
end

-- Sistema de temporizador compatible
local function CreateTimer(duration, callback)
    local timer = CreateFrame("Frame")
    timer.elapsed = 0
    timer:SetScript("OnUpdate", function(self, elapsed)
        self.elapsed = self.elapsed + elapsed
        if self.elapsed >= duration then
            callback()
            self:SetScript("OnUpdate", nil)
        end
    end)
end

-- Ventana de configuración mejorada
function FB:ShowConfigWindow()
    if _G["FBConfigWindow"] then _G["FBConfigWindow"]:Hide() end
    
    local frame = CreateFrame("Frame", "FBConfigWindow", UIParent, "BasicFrameTemplate")
    frame:SetSize(650, 550)
    frame:SetPoint("CENTER")
    
    -- Hacerla movible
    frame:SetMovable(true)
    frame:EnableMouse(true)
    frame:RegisterForDrag("LeftButton")
    frame:SetScript("OnDragStart", frame.StartMoving)
    frame:SetScript("OnDragStop", frame.StopMovingOrSizing)
    
    -- Cerrar con ESC
    tinsert(UISpecialFrames, "FBConfigWindow")
    
    -- Título actualizado
    frame.title = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    frame.title:SetPoint("CENTER", frame.TitleBg, "CENTER", 0, 0)
    frame.title:SetText("FireBuy - Configuration")

    -- Texto informativo derecho
    local infoText = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    infoText:SetPoint("TOPLEFT", 320, -50)
    infoText:SetSize(300, 60)
    infoText:SetJustifyH("RIGHT")
    infoText:SetText("FireBuy v1.10\n\nDrag the items into the available boxes\nand write the desired quantity as stock")

    -- Agradecimiento a la guild
    local thanksText = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    thanksText:SetPoint("BOTTOM", 0, 15)
    thanksText:SetText("Thanks to the guild Song Of Chaos (Benediction)")

    -- Obtener la clave del jugador para la configuración por personaje
    local playerKey = GetPlayerKey()
    FB_DB.char[playerKey] = FB_DB.char[playerKey] or { selectedClass = "Mage" }
    
    -- Selector de clase (DropDown)
    local classDropdown = CreateFrame("Frame", "FBClassDropdown", frame, "UIDropDownMenuTemplate")
    classDropdown:SetPoint("TOPLEFT", 20, -50)
    UIDropDownMenu_SetWidth(classDropdown, 150)
    
    local function ClassDropdown_OnClick(_, class)
        local key = GetPlayerKey()
        FB_DB.char[key].selectedClass = class
        classDropdown.selectedValue = class
        UIDropDownMenu_SetText(classDropdown, class)
        UIDropDownMenu_Refresh(classDropdown)
        FB:UpdateConfigUI()
    end
    
    UIDropDownMenu_Initialize(classDropdown, function(self, level, menuList)
        for _, class in ipairs({
            "Mage", "Paladin", "Warrior", "Priest", "Warlock", "Rogue", 
            "Hunter", "Druid", "Death Knight", "Shaman", "Monk", "Demon Hunter"
        }) do
            local info = UIDropDownMenu_CreateInfo()
            info.text = class
            info.func = ClassDropdown_OnClick
            info.arg1 = class
            local key = GetPlayerKey()
            info.checked = (FB_DB.char[key].selectedClass == class)
            UIDropDownMenu_AddButton(info)
        end
    end)
    
    -- Establecer la clase actual correctamente en el dropdown:
    local key = GetPlayerKey()
    FB_DB.char[key].selectedClass = FB_DB.char[key].selectedClass or "Mage"
    classDropdown.selectedValue = FB_DB.char[key].selectedClass
    UIDropDownMenu_SetText(classDropdown, FB_DB.char[key].selectedClass)
    UIDropDownMenu_Refresh(classDropdown)

    -- Configuración de items
    FB.itemFrames = {}
    for i = 1, 12 do
        local itemFrame = CreateFrame("Frame", nil, frame)
        itemFrame:SetSize(550, 25)
        itemFrame:SetPoint("TOPLEFT", 20, -90 - (i-1)*30)
        
        -- Icono
        itemFrame.icon = itemFrame:CreateTexture(nil, "ARTWORK")
        itemFrame.icon:SetSize(24, 24)
        itemFrame.icon:SetPoint("LEFT")
        itemFrame.icon:SetTexture(134400)
        
        -- Nombre
        itemFrame.nameBox = CreateFrame("EditBox", nil, itemFrame, "InputBoxTemplate")
        itemFrame.nameBox:SetSize(200, 18)
        itemFrame.nameBox:SetPoint("LEFT", 30, 0)
        itemFrame.nameBox:SetAutoFocus(false)
        itemFrame.nameBox:SetEnabled(false)
        
        -- Cantidad
        itemFrame.qtyBox = CreateFrame("EditBox", nil, itemFrame, "InputBoxTemplate")
        itemFrame.qtyBox:SetSize(50, 18)
        itemFrame.qtyBox:SetPoint("LEFT", 250, 0)
        itemFrame.qtyBox:SetNumeric(true)
        itemFrame.qtyBox:SetMaxLetters(4)
        itemFrame.qtyBox:SetScript("OnTextChanged", function(self)
            FB:SaveItemConfig(i, itemFrame.itemID, self:GetText())
        end)
        
        -- Botón de eliminar
        itemFrame.deleteButton = CreateFrame("Button", nil, itemFrame)
        itemFrame.deleteButton:SetSize(20, 20)
        itemFrame.deleteButton:SetPoint("LEFT", 310, 0)
        itemFrame.deleteButton:SetNormalTexture("Interface\\Buttons\\UI-GroupLoot-Pass-Up")
        itemFrame.deleteButton:SetHighlightTexture("Interface\\Buttons\\UI-GroupLoot-Pass-Highlight")
        itemFrame.deleteButton:SetScript("OnClick", function()
            itemFrame.itemID = nil
            itemFrame.icon:SetTexture(134400)
            itemFrame.nameBox:SetText("")
            itemFrame.qtyBox:SetText("")
            FB:SaveItemConfig(i, nil, nil)
        end)
        
        -- Drag & Drop mejorado
        itemFrame:EnableMouse(true)
        itemFrame:RegisterForDrag("LeftButton")
        itemFrame:SetScript("OnReceiveDrag", function(self)
            local type, id = GetCursorInfo()
            if type == "item" then
                local itemID = id
                local itemName, _, _, _, _, _, _, _, _, itemTexture = GetItemInfo(itemID)
                
                if itemName then
                    itemFrame.itemID = itemID
                    itemFrame.icon:SetTexture(itemTexture)
                    itemFrame.nameBox:SetText(itemName)
                    FB:SaveItemConfig(i, itemID, itemFrame.qtyBox:GetText() or 0)
                    ClearCursor()
                    ResetCursor()
                end
            end
        end)
        
        itemFrame:SetScript("OnDragStop", function()
            ClearCursor()
            ResetCursor()
        end)
        
        FB.itemFrames[i] = itemFrame
    end
    
    FB:UpdateConfigUI()
    frame:Show()
end

-- Funciones de guardado mejoradas
function FB:SaveItemConfig(slot, itemID, quantity)
    local key = GetPlayerKey()
    FB_DB.char[key] = FB_DB.char[key] or { selectedClass = "Mage" }
    local selectedClass = FB_DB.char[key].selectedClass
    FB_DB.global.classConfigs[selectedClass] = FB_DB.global.classConfigs[selectedClass] or {}
    
    if itemID and quantity then
        FB_DB.global.classConfigs[selectedClass][slot] = {
            itemID = tonumber(itemID),
            quantity = tonumber(quantity) or 0
        }
    else
        FB_DB.global.classConfigs[selectedClass][slot] = nil
    end
end

function FB:UpdateConfigUI()
    local key = GetPlayerKey()
    FB_DB.char[key] = FB_DB.char[key] or { selectedClass = "Mage" }
    local selectedClass = FB_DB.char[key].selectedClass
    FB_DB.global.classConfigs[selectedClass] = FB_DB.global.classConfigs[selectedClass] or {}
    
    for i = 1, 12 do
        local config = FB_DB.global.classConfigs[selectedClass][i] or {}
        local itemFrame = FB.itemFrames[i]
        if itemFrame then
            itemFrame.itemID = config.itemID
            local itemName, _, _, _, _, _, _, _, _, itemTexture = GetItemInfo(config.itemID or 0)
            
            itemFrame.nameBox:SetText(itemName or "")
            itemFrame.qtyBox:SetText(config.quantity or "")
            itemFrame.icon:SetTexture(itemTexture or 134400)
            itemFrame.deleteButton:SetShown(config.itemID ~= nil)
        end
    end
end

-- Sistema de compras mejorado
local function ComprarItem(slot, itemID, cantidad)
    local comprado = 0
    
    for intento = 1, 30 do
        local link = GetMerchantItemLink(slot)
        if not link or tonumber(link:match("item:(%d+)")) ~= itemID then break end
        
        local _, precio, stackSize, stock = GetMerchantItemInfo(slot)
        if not precio then break end
        
        local comprarAhora = math.min(
            cantidad - comprado,
            stackSize > 0 and stackSize or 1,
            stock == 0 and 1000 or stock
        )
        
        if comprarAhora <= 0 then break end
        
        BuyMerchantItem(slot, comprarAhora)
        comprado = comprado + comprarAhora
    end
    
    return comprado
end

local function ProcesarCola()
    if #colaCompras == 0 then
        comprando = false
        print(COLORS.yellow.."Firebuy - Completed purchases!!")
        return
    end
    
    local datos = table.remove(colaCompras, 1)
    local comprado = ComprarItem(datos.slot, datos.itemID, datos.cantidad)
    
    CreateTimer(3.0, function()
        local current = GetItemCount(datos.itemID)
        local nuevoNecesario = math.max(datos.cantidad - current, 0)
        
        if nuevoNecesario > 0 then
            table.insert(colaCompras, {
                slot = datos.slot,
                itemID = datos.itemID,
                cantidad = nuevoNecesario
            })
        end
        
        ProcesarCola()
    end)
end

local function IniciarCompras()
    if comprando or not FB_DB.global.enabled then return end
    comprando = true
    print(COLORS.yellow.."Firebuy - Starting to buy")
    
    wipe(colaCompras)
    
    local key = GetPlayerKey()
    FB_DB.char[key] = FB_DB.char[key] or { selectedClass = "Mage" }
    local selectedClass = FB_DB.char[key].selectedClass
    local classConfig = FB_DB.global.classConfigs[selectedClass]
    if not classConfig then return end
    
    for i = 1, 12 do
        local config = classConfig[i]
        if config and config.itemID and config.quantity and config.quantity > 0 then
            local current = GetItemCount(config.itemID)
            local needed = math.max(config.quantity - current, 0)
            
            if needed > 0 then
                for slot = 1, GetMerchantNumItems() do
                    local link = GetMerchantItemLink(slot)
                    if link and tonumber(link:match("item:(%d+)")) == config.itemID then
                        table.insert(colaCompras, {
                            slot = slot,
                            itemID = config.itemID,
                            cantidad = needed
                        })
                        break
                    end
                end
            end
        end
    end
    
    ProcesarCola()
end

-- Eventos principales
local frame = CreateFrame("Frame")
frame:RegisterEvent("ADDON_LOADED")
frame:RegisterEvent("MERCHANT_SHOW")
frame:SetScript("OnEvent", function(self, event, addon)
    if event == "ADDON_LOADED" and addon:lower() == "firebuy" then
        -- Migración de datos antigua (si existiera)
        if FB_DB and not FB_DB.global then
            local key = GetPlayerKey()
            FB_DB = {
                global = {
                    enabled = true,
                    minimap = { hide = false },
                    classConfigs = FB_DB.classConfigs or {}
                },
                char = {}
            }
            FB_DB.char[key] = { selectedClass = FB_DB.selectedClass or "Mage" }
        end
        
        -- Inicialización garantizada
        FB_DB.global = FB_DB.global or { enabled = true, minimap = { hide = false }, classConfigs = {} }
        FB_DB.char = FB_DB.char or {}
        local key = GetPlayerKey()
        FB_DB.char[key] = FB_DB.char[key] or { selectedClass = "Mage" }
        
        -- Validar clase seleccionada
        local validClasses = {
            Mage = true, Paladin = true, Warrior = true, Priest = true,
            Warlock = true, Rogue = true, Hunter = true, Druid = true,
            ["Death Knight"] = true, Shaman = true, Monk = true, ["Demon Hunter"] = true
        }
        
        if not validClasses[FB_DB.char[key].selectedClass] then
            FB_DB.char[key].selectedClass = "Mage"
        end
        
        if LibDBIcon then
            LibDBIcon:Refresh("FireBuy", FB_DB.global.minimap)
            LibDBIcon:Show("FireBuy")
        end
        
        print(COLORS.yellow.."FireBuy "..COLORS.green.."Ready! |cFF00FF00(Status: "..(FB_DB.global.enabled and "ACTIVATED" or "DEACTIVATED")..")")
    elseif event == "MERCHANT_SHOW" then
        CreateTimer(1.0, IniciarCompras)
    end
end)

-- Comando de chat
SLASH_FIREBUY1 = "/firebuy"
SlashCmdList["FIREBUY"] = function() FB:ShowConfigWindow() end
-- Gloria a la República de Yucatán