local function GetInstanceNameByActivity(id)
	local name,_ = C_LFGList.GetActivityInfo(id)
	return name
end

local DungeonList = {}
local DungeonActivities = {}

local function AddDungeon(groupID,activityList,imageID,name,texCoord)
	local entry = {}
	entry["checked"] = false
	entry["groupID"] = groupID
	entry["activityList"] = activityList
	entry["imageID"] = imageID
	if name then
		entry["name"] = name
	end
	if texCoord then
		entry["texCoord"] = texCoord
	end

	table.insert(DungeonList,entry)
end

-- Classic
-- Blackfathom Deeps	-- Blackrock Depths		-- Deadmines 			-- Dire Maul
AddDungeon(0,{54},227)	AddDungeon(0,{63},228)	AddDungeon(0,{18},63)	AddDungeon(0,{65},230)
-- Gnomeregan			-- Lower BRS 			-- Maraudon				-- Ragefire Chasm
AddDungeon(0,{56},231)	AddDungeon(0,{64},229)	AddDungeon(0,{61},232)	AddDungeon(0,{52},226)
-- Razorfen Downs		-- Razorfen Kraul		-- Scarlet Halls		-- Scarlet Monastery
AddDungeon(0,{58},233)	AddDungeon(0,{57},234)	AddDungeon(0,{77},311)	AddDungeon(0,{78},316)
-- Scholomance			-- Shadowfang Keep		-- Stratholme			-- The Stockade
AddDungeon(0,{51},246)	AddDungeon(0,{53},64)	AddDungeon(0,{66},236)	AddDungeon(0,{55},238)
-- Sunken Temple		-- Uldaman				-- Wailing Caverns		-- Zul'Farrak
AddDungeon(0,{62},237)	AddDungeon(0,{59},239)	AddDungeon(0,{50},240)	AddDungeon(0,{60},241)

-- TBC
-- Auchenai Crypts			-- Hellfire Ramparts		-- Magisters' Terrace		-- Mana-Tombs
AddDungeon(1,{74,84},247)	AddDungeon(1,{67,94},248)	AddDungeon(1,{99,100},249)	AddDungeon(1,{73,85},250)
-- Old Hillsbrad			-- Sethekk Halls 			-- Shadow Labyrinth			-- The Arcatraz
AddDungeon(1,{79,89},251)	AddDungeon(1,{75,86},252)	AddDungeon(1,{76,87},253)	AddDungeon(1,{83,96},254)
-- The Black Morass			-- The Blood Furnace		-- The Botanica				-- The Mechanar
AddDungeon(1,{80,88},255)	AddDungeon(1,{68,93},256)	AddDungeon(1,{82,97},257)	AddDungeon(1,{81,98},258)
-- Shattered Halls			-- The Slave Pens			-- The Steamvault			-- The Underbog
AddDungeon(1,{69,95},259)	AddDungeon(1,{70,90},260)	AddDungeon(1,{72,91},261)	AddDungeon(1,{71,92},262)

-- WotLK
-- Ahn'kahet				-- Azjol-Nerub				-- Drak'Tharon Keep			-- Gundrak
AddDungeon(2,{110,124},271)	AddDungeon(2,{103,127},272)	AddDungeon(2,{108,122},273)	AddDungeon(2,{109,123},274)
-- Halls of Lightning		-- Halls of Reflection 		-- Halls of Stone			
AddDungeon(2,{105,120},275)	AddDungeon(2,{116,132},276)	AddDungeon(2,{106,121},277)
-- Pit of Saron				-- Stratholme				-- The Forge of Souls		-- The Nexus
AddDungeon(2,{115,131},278)	AddDungeon(2,{107,118},279)	AddDungeon(2,{114,130},280)	AddDungeon(2,{112,126},281)
-- The Oculus				-- The Violet Hold			-- Trial of Champion		-- Utgarde Keep
AddDungeon(2,{104,119},282)	AddDungeon(2,{111,125},283)	AddDungeon(2,{113,129},284)	AddDungeon(2,{101,128},285)
-- Utgarde Pinnacle
AddDungeon(2,{102,117},286)

-- Cataclysm
-- Blackrock Caverns		-- Deadmines				-- End Time					-- Grim Batol
AddDungeon(3,{134,144},66)	AddDungeon(3,{148},63)		AddDungeon(3,{152},184)		AddDungeon(3,{135,143},71)
-- Halls of Origination		-- Hour of Twilight 		-- Lost City of the Tol'vir	-- Shadowfang Keep
AddDungeon(3,{136,142},70)	AddDungeon(3,{154},186)		AddDungeon(3,{139,147},69)	AddDungeon(3,{149},64)
-- The Stonecore			-- The Vortex Pinnacle		-- Throne of the Tides		-- Well of Eternity
AddDungeon(3,{137,141},67)	AddDungeon(3,{138,140},68)	AddDungeon(3,{133,146},65)	AddDungeon(3,{153},185)
-- Zul'Aman					-- Zul'Gurub
AddDungeon(3,{151},77)		AddDungeon(3,{150},76)

-- MoP
-- Gate of the Setting Sun	-- Mogu'shan Palace			-- Scarlet Halls			-- Scarlet Monastery
AddDungeon(4,{160,167},303)	AddDungeon(4,{158,166},321)	AddDungeon(4,{170},311)		AddDungeon(4,{169},316)
-- Scholomance				-- Shado-Pan Monastery 		-- Siege of Niuzao Temple	-- Stormstout Brewery
AddDungeon(4,{168},246)		AddDungeon(4,{157,165},312)	AddDungeon(4,{159,171},324)	AddDungeon(4,{156,164},302)
-- Temple of Jade Serpent	-- Random Pandaria Dungeon
AddDungeon(4,{155,163},313)	AddDungeon(4,{363,364},322, LFG_TYPE_RANDOM_DUNGEON.." ("..EXPANSION_NAME4..")")

-- WoD
-- Auchindoun					-- Bloodmaul Slag Mines			-- Grimrail Depot				-- Iron Docks
AddDungeon(5,{23,31,403},547)	AddDungeon(5,{21,29,401},385)	AddDungeon(5,{25,33,405},536)	AddDungeon(5,{22,30,402},558)
-- Shadowmoon Burial Grounds	-- Skyreach 					-- The Everbloom				-- Upper Blackrock Spire
AddDungeon(5,{27,35,407},537)	AddDungeon(5,{24,32,404},476)	AddDungeon(5,{26,34,406},556)	AddDungeon(5,{28,36,408},559)
-- Random Draenor Dungeon
AddDungeon(5,{395,396},557, LFG_TYPE_RANDOM_DUNGEON.." ("..EXPANSION_NAME5..")")

-- Legion Normal + Heroic
-- Assault on Violet Hold							-- Black Rook Hold 										-- Court of Stars									-- Darkheart Thicket
AddDungeon(-2,{429,439},777,nil,{0.1,0.4,0.3,0.7})	AddDungeon(-2,{430,440},740,nil,{0.15,0.45,0.3,0.7})	AddDungeon(-2,{433,443},800,nil,{0.2,0.6,0.1,0.5})	AddDungeon(-2,{426,436},762,nil,{0.2,0.6,0.3,0.6})
-- Eye of Azshara									-- Halls of Valor										-- Maw of Souls										-- Neltharion's Lair
AddDungeon(-2,{425,435},716,nil,{0.2,0.6,0.1,0.4})	AddDungeon(-2,{427,437},721,nil,{0.2,0.6,0.17,0.47}) 	AddDungeon(-2,{432,442},727,nil,{0,0.4,0.2,0.5})	AddDungeon(-2,{428,438},767,nil,{0.2,0.6,0.3,0.6})
-- The Arcway										-- Vault of the Wardens									-- Return to Karazhan Lower
AddDungeon(-2,{434,444},726,nil,{0,0.4,0.2,0.5})	AddDungeon(-2,{431,441},707,nil,{0.02,0.22,0.2,0.35})	AddDungeon(-2,{470},860,GetInstanceNameByActivity(470),{0.25,0.65,0.4,0.7})
-- Return to Karazhan Upper 												-- Cathedral of Eternal Night 						-- Seat of the Triumvirate
AddDungeon(-2,{472},860,GetInstanceNameByActivity(472),{0.05,0.45,0.1,0.4})	AddDungeon(-2,{474},900,nil,{0.15,0.45,0.4,0.7})	AddDungeon(-2,{484},945,nil,{0.15,0.45,0.4,0.7})
-- Random Legion Dungeon
AddDungeon(-2,{417,418},822, LFG_TYPE_RANDOM_DUNGEON.." ("..EXPANSION_NAME6..")")

-- Legion Mythic
-- Assault on Violet Hold						-- Black Rook Hold 									-- Court of Stars								-- Darkheart Thicket
AddDungeon(-3,{449},777,nil,{0.1,0.4,0.3,0.7})	AddDungeon(-3,{450},740,nil,{0.15,0.45,0.3,0.7})	AddDungeon(-3,{453},800,nil,{0.2,0.6,0.1,0.5})	AddDungeon(-3,{446},762,nil,{0.2,0.6,0.3,0.6})
-- Eye of Azshara								-- Halls of Valor									-- Maw of Souls									-- Neltharion's Lair
AddDungeon(-3,{445},716,nil,{0.2,0.6,0.1,0.4})	AddDungeon(-3,{447},721,nil,{0.2,0.6,0.17,0.47}) 	AddDungeon(-3,{452},727,nil,{0,0.4,0.2,0.5})	AddDungeon(-3,{448},767,nil,{0.2,0.6,0.3,0.6})
-- The Arcway									-- Vault of the Wardens								-- Return to Karazhan							-- Cathedral of Eternal Night
AddDungeon(-3,{454},726,nil,{0,0.4,0.2,0.5})	AddDungeon(-3,{451},707,nil,{0.02,0.22,0.2,0.35})	AddDungeon(-3,{455},860,nil,{0.05,0.45,0.4,0.7}) AddDungeon(-3,{475},900,nil,{0.15,0.45,0.4,0.7})
-- Seat of the Triumvirate
AddDungeon(-3,{485},945,nil,{0.15,0.45,0.4,0.7})

-- Legion Mythic+
-- Black Rook Hold 									-- Court of Stars								-- Darkheart Thicket
AddDungeon(-4,{463},740,nil,{0.15,0.45,0.3,0.7})	AddDungeon(-4,{466},800,nil,{0.2,0.6,0.1,0.5})	AddDungeon(-4,{460},762,nil,{0.2,0.6,0.3,0.6})
-- Eye of Azshara								-- Halls of Valor									-- Maw of Souls									-- Neltharion's Lair
AddDungeon(-4,{459},716,nil,{0.2,0.6,0.1,0.4})	AddDungeon(-4,{461},721,nil,{0.2,0.6,0.17,0.47}) 	AddDungeon(-4,{465},727,nil,{0,0.4,0.2,0.5})	AddDungeon(-4,{462},767,nil,{0.2,0.6,0.3,0.6})
-- The Arcway									-- Vault of the Wardens
AddDungeon(-4,{467},726,nil,{0,0.4,0.2,0.5})	AddDungeon(-4,{464},707,nil,{0.02,0.22,0.2,0.35})
-- Return to Karazhan L
AddDungeon(-4,{471},860,GetInstanceNameByActivity(471),{0.25,0.65,0.4,0.7})
-- Return to Karazhan U
AddDungeon(-4,{473},860,GetInstanceNameByActivity(473),{0.05,0.45,0.1,0.4})
-- Cathedral of Eternal Night
AddDungeon(-4,{476},900,nil,{0.15,0.45,0.4,0.7})
-- Seat of the Triumvirate
AddDungeon(-4,{486},945,nil,{0.15,0.45,0.4,0.7})


-- TEMPLATE

-- AddDungeon(
-- -4, -> Group ID (Internal)
-- {486}, -> LFG Instance ID (WOW)
-- 945, -> Instance ID (WOW)
-- nil, -> Forced name (Internal)
-- {0.15,0.45,0.4,0.7} -> IDFK (Internal)
-- )

-- Macro for searching Instance ID:
-- /run     for i=1, 2000 do        local instance = EJ_GetInstanceInfo(i)        if instance then            print(i)            print(instance)        end    end

-- /TEMPLATE 


-- MODULE FOR SEARCHING FRAME 
local var = hopAddon.var
local customOptions = hopAddon.optionsFrame.customSearchOptionsFrame

local searchUpdated = 2.5
local ResultFrames = {}
local SearchString = ""
local SearchKeywords = {}
local FilteredGroupsList = {}
local CustomSearchBlackList = {}

-- Holder
hopAddon.searchFrame = CreateFrame("Frame",nil,hopAddon)
hopAddon.searchFrame:SetSize(HOPADDON_WIDTH,HOPADDON_HEIGHT)
hopAddon.searchFrame:SetPoint("BOTTOM",0,0)

hopAddon.searchFrame.background = hopAddon.searchFrame:CreateTexture(nil,"BORDER")
hopAddon.searchFrame.background:SetSize(HOPADDON_WIDTH+106,HOPADDON_HEIGHT-10)
hopAddon.searchFrame.background:SetPoint("BOTTOM",1,5)
hopAddon.searchFrame.background:SetTexture("Interface\\Challenges\\challenges-besttime-bg")

-- Searching animation block
hopAddon.searchFrame.Waitdot2 = hopAddon.searchFrame:CreateTexture("Waitdot2","ARTWORK")
hopAddon.searchFrame.Waitdot2:SetPoint("BOTTOM",0,45)
hopAddon.searchFrame.Waitdot2:SetAtlas("groupfinder-waitdot",true)
hopAddon.searchFrame.Waitdot2:SetVertexColor(1,1,1,0)

hopAddon.searchFrame.Waitdot1 = hopAddon.searchFrame:CreateTexture("Waitdot2","ARTWORK")
hopAddon.searchFrame.Waitdot1:SetPoint("CENTER", "Waitdot2", -16,0)
hopAddon.searchFrame.Waitdot1:SetAtlas("groupfinder-waitdot",true)
hopAddon.searchFrame.Waitdot1:SetVertexColor(1,1,1,0)

hopAddon.searchFrame.Waitdot3 = hopAddon.searchFrame:CreateTexture("Waitdot2","ARTWORK")
hopAddon.searchFrame.Waitdot3:SetPoint("CENTER", "Waitdot2",16,0)
hopAddon.searchFrame.Waitdot3:SetAtlas("groupfinder-waitdot",true)
hopAddon.searchFrame.Waitdot3:SetVertexColor(1,1,1,0)

hopAddon.searchFrame.anim = hopAddon.searchFrame:CreateAnimationGroup("Searching")
hopAddon.searchFrame.anim:SetLooping("REPEAT")

SH_AddAlphaAnimation(hopAddon.searchFrame.anim,"Waitdot1",0.50,0.15,1,0,1)
SH_AddAlphaAnimation(hopAddon.searchFrame.anim,"Waitdot2",0.50,0.15,2,0,1)
SH_AddAlphaAnimation(hopAddon.searchFrame.anim,"Waitdot3",0.50,0.15,3,0,1)
SH_AddAlphaAnimation(hopAddon.searchFrame.anim,"Waitdot1",0.50,0.15,4,1,0)
SH_AddAlphaAnimation(hopAddon.searchFrame.anim,"Waitdot2",0.50,0.15,4,1,0)
SH_AddAlphaAnimation(hopAddon.searchFrame.anim,"Waitdot3",0.50,0.15,4,1,0)


-- Roles
local function CreateRoleBtn(point,x,y,role)
	local btn = CreateFrame("Button", role.."button", hopAddon.searchFrame)
	btn:SetSize(30,30)
	btn:SetPoint(point,x,y)
	btn.Icon = btn:CreateTexture("butOptionsTex","OVERLAY")
	btn.Icon:SetAllPoints(btn)
	btn.Icon:SetTexture("Interface\\LFGFrame\\UI-LFG-ICON-ROLES")
	btn.Icon:SetTexCoord(GetTexCoordsForRole(role))
	btn.CheckButton = CreateFrame("CheckButton", role.."CheckBtn",btn,"ChatConfigCheckButtonTemplate")
	btn.CheckButton:SetHitRectInsets(20,20,20,20)
	btn.CheckButton:SetPoint("LEFT",-10,0)

	btn:SetScript("OnClick", function(self)
		if self.CheckButton:IsEnabled() then
			self.CheckButton:Click()
		end
	end)
	return btn
end

SHtankButton = CreateRoleBtn("TOP",-45,-16,"TANK")
SHhealerButton = CreateRoleBtn("TOP",0,-16,"HEALER")
SHdamagerButton = CreateRoleBtn("TOP",45,-16,"DAMAGER")

-- Dungeons drop list mush be shown only when dungeons selected
local menuFrame = CreateFrame("Frame", "ExampleMenuFrame", hopAddon, "UIDropDownMenuTemplate")
menuFrame.lastClick = nil

hopAddon.buttonPlus = CreateFrame("Button","Filter_buttonPlus",hopAddon,"BrowserButtonTemplate")
hopAddon.buttonPlus:SetSize(25,25)
hopAddon.buttonPlus:SetPoint("CENTER",0,0)
hopAddon.buttonPlus.Icon = hopAddon.buttonPlus:CreateTexture("helpTex","OVERLAY")
hopAddon.buttonPlus.Icon:SetSize(20,20)
hopAddon.buttonPlus.Icon:SetPoint("CENTER",0,0)
hopAddon.buttonPlus.Icon:SetTexture("Interface\\Minimap\\UI-Minimap-ZoomInButton-Up")
hopAddon.buttonPlus:Hide()

-- Raids drop list must be shown only when raids selected
hopAddon.searchFrame.raidsDrop = CreateFrame("Frame", "raidsDrop", hopAddon.searchFrame, "UIDropDownMenuTemplate")
local rDrop = hopAddon.searchFrame.raidsDrop
rDrop:SetPoint("BOTTOM",0,35)
rDrop.activeValue = 0
rDrop:Hide()

-- Dropdown to select category to search in

hopAddon.searchFrame.dropDown = CreateFrame("Frame", "CategoryDrop", hopAddon.searchFrame, "UIDropDownMenuTemplate")
hopAddon.searchFrame.dropDown:SetPoint("BOTTOM",-10,8)
UIDropDownMenu_SetWidth(hopAddon.searchFrame.dropDown,180)
hopAddon.searchFrame.dropDown.text = hopAddon.searchFrame.dropDown:CreateFontString(nil, "OVERLAY", "GameFontNormal")
hopAddon.searchFrame.dropDown.text:SetSize(136,12)
hopAddon.searchFrame.dropDown.text:SetPoint("RIGHT",hopAddon.searchFrame.dropDown,"RIGHT",-42,3)
hopAddon.searchFrame.dropDown.text:SetJustifyH("RIGHT")

-- custom search dropdown init
local CategoryList = {}

hopAddon.searchFrame.dropDown.initialize = function(self, level)

    if not level then return end
    wipe(CategoryList)
    if level == 1 then
		local categories = C_LFGList.GetAvailableCategories();
		for i=1, #categories do
			local categoryID = categories[i];
			local name, separateRecommended, autoChoose, preferCurrentArea = C_LFGList.GetCategoryInfo(categoryID);

			CategoryList.text = name;
			CategoryList.value = categoryID;
			CategoryList.func = hopAddon.searchFrame.dropDown_OnClick;
			CategoryList.checked = self.activeValue == CategoryList.value;
			--CategoryList.tooltipTitle = "bla"
			--CategoryList.tooltipText = "blabla"
			--CategoryList.tooltipOnButton = 1
			UIDropDownMenu_AddButton(CategoryList, 1);
		end
	end
	
end

UIDropDownMenu_SetAnchor(hopAddon.searchFrame.dropDown,0,10,"TOPRIGHT",hopAddon.searchFrame.dropDown,"BOTTOMRIGHT")

-- Category filter options

hopAddon.searchFrame.holderCatFilters = CreateFrame("Button",nil,hopAddon.searchFrame)
local h = hopAddon.searchFrame.holderCatFilters
h:SetSize(210,90)
h:SetPoint("CENTER",-10,-5)
h:SetBackdrop({bgFile = "Interface\\FrameGeneral\\UI-Background-Rock", 
					edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border", 
					tile = true, tileSize = 256, edgeSize = 24, 
					insets = { left = 4, right = 4, top = 4, bottom = 4 }});
h:SetFrameLevel(10)
h:Hide()

h.timeEdit = CreateFrame("EditBox","searchFrameTimeEdit",h,"InputBoxInstructionsTemplate")
h.timeEdit:SetSize(40,22)
h.timeEdit:SetPoint("TOPLEFT",140,-15)
h.timeEdit:SetAutoFocus(false)
h.timeEdit:SetNumeric(true)
h.timeEdit:SetMaxLetters(4)
h.timeEdit:SetText("90")
h.timeEdit:SetScript("OnEnterPressed",function(self)
	if h.timeEdit:GetText() == "" then
		h.timeEdit:SetText("90")
	end
	EditBox_ClearFocus(self)
end)
h.timeEdit:SetScript("OnEscapePressed",function(self)
	EditBox_ClearFocus(self)
end)
h.timeEdit:SetScript("OnDisable", function(self)
	self:SetTextColor(0.5,0.5,0.5,1)
end)
h.timeEdit:SetScript("OnEnable", function(self)
	self:SetTextColor(1,1,1,1)
end)
h.timeEdit:Disable()

h.timeCheck = CreateFrame("CheckButton","searchFrameTimeCheck",h,"ChatConfigCheckButtonTemplate")
h.timeCheck:SetPoint("TOPLEFT",15,-15)
h.timeCheck:SetHitRectInsets(0,-90,0,0)
getglobal(h.timeCheck:GetName() .. 'Text'):SetText(HOPADDON_TIMEOPTION)
h.timeCheck.tooltip = HOPADDON_TIMETOOLTIP
h.timeCheck:SetChecked(false)
h.timeCheck:SetScript("OnClick",function(self)
	if h.timeEdit:IsEnabled() then
		h.timeEdit:Disable()
	else
		h.timeEdit:Enable()
	end
	PlaySound(PlaySoundKitID and "igMainMenuOptionCheckBoxOn" or 856)	
end)

h.ilvlEdit = CreateFrame("EditBox","searchFrameilvlEdit",h,"InputBoxInstructionsTemplate")
h.ilvlEdit:SetSize(40,22)
h.ilvlEdit:SetPoint("TOPLEFT",140,-35)
h.ilvlEdit:SetAutoFocus(false)
h.ilvlEdit:SetNumeric(true)
h.ilvlEdit:SetMaxLetters(3)
h.ilvlEdit:SetText("630")
h.ilvlEdit:SetScript("OnEnterPressed",function(self)
	if h.ilvlEdit:GetText() == "" then
		h.ilvlEdit:SetText("630")
	end
	EditBox_ClearFocus(self)
end)
h.ilvlEdit:SetScript("OnEscapePressed",function(self)
	EditBox_ClearFocus(self)
end)
h.ilvlEdit:SetScript("OnDisable", function(self)
	self:SetTextColor(0.5,0.5,0.5,1)
end)
h.ilvlEdit:SetScript("OnEnable", function(self)
	self:SetTextColor(1,1,1,1)
end)
h.ilvlEdit:Disable()

h.ilvlCheck = CreateFrame("CheckButton","searchFrameilvlCheck",h,"ChatConfigCheckButtonTemplate")
h.ilvlCheck:SetPoint("TOPLEFT",15,-35)
h.ilvlCheck:SetHitRectInsets(0,-90,0,0)
getglobal(h.ilvlCheck:GetName() .. 'Text'):SetText(HOPADDON_ITEMLEVEL)
h.ilvlCheck.tooltip = HOPADDON_ITEMLEVELTOOLTIP
h.ilvlCheck:SetChecked(false)
h.ilvlCheck:SetScript("OnClick",function(self)
	if h.ilvlEdit:IsEnabled() then
		h.ilvlEdit:Disable()
	else
		h.ilvlEdit:Enable()
	end
	PlaySound(PlaySoundKitID and "igMainMenuOptionCheckBoxOn" or 856)	
end)

h.autoInvCheck = CreateFrame("CheckButton","searchFrameacceptCheck",h,"ChatConfigCheckButtonTemplate")
h.autoInvCheck:SetPoint("TOPLEFT",15,-55)
getglobal(h.autoInvCheck:GetName() .. 'Text'):SetText(HOPADDON_SEARCH_AUTOINVITE)
h.autoInvCheck.tooltip = HOPADDON_SEARCHAUTOINVITETOOLTIP
h.autoInvCheck:SetChecked(false)

hopAddon.searchFrame.buttonCatFilters = CreateFrame("Button","catFiltersBtn",hopAddon.searchFrame,"BrowserButtonTemplate")
local b = hopAddon.searchFrame.buttonCatFilters
b:SetSize(24,24)
b:SetPoint("RIGHT",hopAddon.searchFrame.dropDown,8,3)
b.Icon = b:CreateTexture("butOptionsTex","OVERLAY")
b.Icon:SetSize(14,14)
b.Icon:SetPoint("CENTER",0,0)
b.Icon:SetTexture("Interface\\Buttons\\UI-OptionsButton")
b.tooltip = HOPADDON_ADDITIONALFILTERS
-- override on enter, onhide event is inside template
b:SetScript("OnClick", function(self,button)
	if h:IsShown() then
		h:Hide()
	else
		h:Show()
	end
	PlaySound(PlaySoundKitID and "igMainMenuOptionCheckBoxOn" or 856)	
end)

b:SetScript("OnEnter", function(self)
	GameTooltip:SetOwner(self, "ANCHOR_RIGHT", 0, -25)
	GameTooltip:SetText(self.tooltip, 1, 1, 1, true)
	GameTooltip:Show()
end)

-- Search box
hopAddon.searchFrame.searchBox = CreateFrame("EditBox", "ServerHop_SearchBox", hopAddon.searchFrame, "SearchBoxTemplate")
hopAddon.searchFrame.searchBox:SetPoint("TOPLEFT", hopAddon, "BOTTOMLEFT", 10, 2)
hopAddon.searchFrame.searchBox:SetAutoFocus(false)
hopAddon.searchFrame.searchBox:EnableMouse(true)
hopAddon.searchFrame.searchBox:SetSize(HOPADDON_WIDTH-34, 24)
hopAddon.searchFrame.searchBox:SetMaxLetters(1023)

hopAddon.searchFrame.searchBox:SetScript("OnEditFocusGained", function(frame)
	hopAddon.searchFrame.SearchString = frame:GetText()
end)
hopAddon.searchFrame.searchBox:SetScript("OnEscapePressed", function(frame)
	frame:ClearFocus()
	frame:SetText(hopAddon.searchFrame.SearchString)
end)

-- search box clear button
hopAddon.searchFrame.searchBox.clearButton:Show()
hopAddon.searchFrame.searchBox.clearButton:SetScript("OnEnter", function(self)
	GameTooltip:SetOwner(self, "ANCHOR_TOP", 0, 0)
	GameTooltip:SetText(HOPADDON_CLEARSEARCH, 1, 1, 1, true)	
	GameTooltip:Show()
end)

hopAddon.searchFrame.searchBox.clearButton:SetScript("OnLeave", function(self)
	GameTooltip:Hide()
end)

-- search refresh button
hopAddon.searchFrame.refreshButton = CreateFrame("Button", "SH_SearchBtn", hopAddon.searchFrame,"BrowserButtonTemplate")
hopAddon.searchFrame.refreshButton:SetSize(25,25)
hopAddon.searchFrame.refreshButton:SetFrameLevel(5)
hopAddon.searchFrame.refreshButton:SetPoint("LEFT",hopAddon.searchFrame.searchBox,-7,0)
hopAddon.searchFrame.refreshButton.Icon = hopAddon.searchFrame.refreshButton:CreateTexture(nil,"OVERLAY")
hopAddon.searchFrame.refreshButton.Icon:SetSize(14,14)
hopAddon.searchFrame.refreshButton.Icon:SetPoint("CENTER",0,0)
hopAddon.searchFrame.refreshButton.Icon:SetTexture("Interface\\Buttons\\UI-RefreshButton")

hopAddon.searchFrame.refreshButton.anim = hopAddon.searchFrame.refreshButton:CreateAnimationGroup("SH_PressSearch")
hopAddon.searchFrame.refreshButton.anim:SetLooping("REPEAT")

SH_AddTranslationAnimation(hopAddon.searchFrame.refreshButton.anim,"SH_SearchBtn",0.15,0.15,1,0,4)
SH_AddTranslationAnimation(hopAddon.searchFrame.refreshButton.anim,"SH_SearchBtn",0,0.15,2,0,-4)
SH_AddScaleAnimation(hopAddon.searchFrame.refreshButton.anim,"SH_SearchBtn",0,0.15,3,1.2,1.2)
SH_AddScaleAnimation(hopAddon.searchFrame.refreshButton.anim,"SH_SearchBtn",0.15,0.15,4,0.83,0.83)

-- search results scroll
local function UpdateSearchFrame(frame,value)
	-- update left frames
	frame.result = FilteredGroupsList[value]
	
	local id, activityID, name, comment, voiceChat, iLvl, honorLevel, age, numBNetFriends, numCharFriends, numGuildMates, isDelisted, leaderName, numMembers, autoinv = C_LFGList.GetSearchResultInfo(frame.result)
	
	-- party name
	frame.textName:SetText(name)
	
	-- autoinvite text color indication
	if autoinv then
		--frame:SetBackdropColor(0.07,0.24,0,1)
		frame:SetBackdropColor(0.30,0.6,0.28,1)
		--frame:SetBackdropColor(0.63,0.67,0.23,1)
	else
		frame:SetBackdropColor(0.18,0.16,0.11,1)
	end
	
	-- party members count
	frame.textCount:SetText(numMembers)
	
	-- party members icon
	frame.icon:SetVertexColor(1,1,1,1)				
	if (numBNetFriends > 0) then
		frame.icon:SetAtlas("groupfinder-icon-friend")		
	elseif (numCharFriends > 0) then
		frame.icon:SetAtlas("groupfinder-icon-friend")		
	elseif (numGuildMates > 0 ) then
		frame.icon:SetVertexColor(0,1,0,1)	
	else
		frame.icon:SetAtlas("groupfinder-waitdot")
	end
	
	-- frame backdrop
	frame.icon:Show()
	frame.iconRedx:Hide()
	frame.iconGreenChck:Hide()
	frame.spinner:Hide()
	local id, appStatus, pendingStatus, appDuration = C_LFGList.GetApplicationInfo(frame.result)

	if pendingStatus == "applied" or appStatus == "applied" then
		frame:SetBackdropColor(0.25,0.75,0.25,1)
		frame.icon:Hide()
		frame.spinner:Show()
		frame.spinner.Anim:Play()
	elseif pendingStatus == "invited" or appStatus == "invited" then
		frame:SetBackdropColor(0.25,0.75,0.25,1)
		frame.iconGreenChck:Show()
	elseif appStatus == "invitedeclined" or appStatus == "declined" or appStatus == "timedout" then
		frame.iconRedx:Show()
	end
	
end

----------------------------------------------------------------
local ROW_HEIGHT = 28   -- How tall is each row?
local MAX_ROWS = 5      -- How many rows can be shown at once?
----------------------------------------------------------------
-- Create the frame:
local scrolltooltip = 0

local scroll = CreateFrame("Frame", "SH_SearchResultsScroll", hopAddon.searchFrame)
hopAddon.searchFrame.scrollframe = scroll
scroll:SetPoint("TOP",hopAddon.searchFrame,"BOTTOM",0,-22)
scroll:SetSize(HOPADDON_WIDTH, (ROW_HEIGHT * MAX_ROWS))

scroll:EnableMouse(true)
scroll:SetMovable(true)
--[[scroll:SetBackdrop({
	bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background", tile = true, tileSize = 16,
	edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", edgeSize = 16,
	insets = { left = 4, right = 4, top = 4, bottom = 4 },
})]]

scroll:Hide()

-- results attached to search frame
local function ResultFrame_OnEnter(self)
	local resultID = self.result;
	scrolltooltip = self.index
	local id, activityID, name, comment, voiceChat, iLvl, honorLevel, age, numBNetFriends, numCharFriends, numGuildMates, isDelisted, leaderName, numMembers, autoinv = C_LFGList.GetSearchResultInfo(resultID)
	local fullName, shortName, categoryID, groupID, iLevel, filters, minLevel, maxPlayers, displayType, orderIndex, useHonorLevel = C_LFGList.GetActivityInfo(activityID);
	local memberCounts = C_LFGList.GetSearchResultMemberCounts(resultID);
	GameTooltip:SetOwner(self, "ANCHOR_RIGHT", 10, -40);
	GameTooltip:SetText(name);	
	GameTooltip:AddLine(fullName,1,1,1);
	if autoinv then
		GameTooltip:AddLine(HOPADDON_AUTOINVITE, 0.25, 0.75, 0.25, true)
	end	
	if ( comment ~= "" ) then
		GameTooltip:AddLine(string.format(LFG_LIST_COMMENT_FORMAT, comment), LFG_LIST_COMMENT_FONT_COLOR.r, LFG_LIST_COMMENT_FONT_COLOR.g, LFG_LIST_COMMENT_FONT_COLOR.b, true);
	end
	GameTooltip:AddLine(" ");
	if ( iLvl > 0 ) then
		GameTooltip:AddLine(string.format(LFG_LIST_TOOLTIP_ILVL, iLvl));
	end
	if ( voiceChat ~= "" ) then
		GameTooltip:AddLine(string.format(LFG_LIST_TOOLTIP_VOICE_CHAT, voiceChat), nil, nil, nil, true);
	end
	if ( iLvl > 0 or voiceChat ~= "" ) then
		GameTooltip:AddLine(" ");
	end

	if ( leaderName ) then
		GameTooltip:AddLine(string.format(LFG_LIST_TOOLTIP_LEADER, leaderName));
	end
	if ( age > 0 ) then
		GameTooltip:AddLine(string.format(LFG_LIST_TOOLTIP_AGE, SecondsToTime(age, false, false, 1, false)));
	end

	if ( leaderName or age > 0 ) then
		GameTooltip:AddLine(" ");
	end

	GameTooltip:AddLine(string.format(LFG_LIST_TOOLTIP_MEMBERS, numMembers, memberCounts.TANK, memberCounts.HEALER, memberCounts.DAMAGER));

	if ( numBNetFriends + numCharFriends + numGuildMates > 0 ) then
		GameTooltip:AddLine(" ");
		GameTooltip:AddLine(LFG_LIST_TOOLTIP_FRIENDS_IN_GROUP);
		GameTooltip:AddLine(LFGListSearchEntryUtil_GetFriendList(resultID), 1, 1, 1, true);
	end
	
	local completedEncounters = C_LFGList.GetSearchResultEncounterInfo(resultID);
	if ( completedEncounters and #completedEncounters > 0 ) then
		GameTooltip:AddLine(" ");
		GameTooltip:AddLine(LFG_LIST_BOSSES_DEFEATED);
		for i=1, #completedEncounters do
			GameTooltip:AddLine(completedEncounters[i], RED_FONT_COLOR.r, RED_FONT_COLOR.g, RED_FONT_COLOR.b);
		end
	end
	
	local id, appStatus, pendingStatus = C_LFGList.GetApplicationInfo(id);

	GameTooltip:AddLine(" ");
	
	if pendingStatus == "applied" or appStatus == "applied" then
		GameTooltip:AddDoubleLine(HOPADDON_CANCELAPP,HOPADDON_RMB,0.75, 0.25, 0);
	elseif pendingStatus == "invited" or appStatus == "invited" then
		GameTooltip:AddDoubleLine(HOPADDON_DECLINEAPP,HOPADDON_RMB,0.75, 0.25, 0);
	else
		GameTooltip:AddDoubleLine(HOPADDON_APPLY,HOPADDON_LMB,0.25, 0.75, 0.25);
		GameTooltip:AddDoubleLine(HOPADDON_BLACKLIST,HOPADDON_RMB,0.75, 0.25, 0);
	end

	GameTooltip:Show();
end

function scroll:Update()
	local maxValue = #FilteredGroupsList
	FauxScrollFrame_Update(self.scrollBar, maxValue, MAX_ROWS, ROW_HEIGHT)

	local offset = FauxScrollFrame_GetOffset(self.scrollBar)
	for i = 1, MAX_ROWS do
		local value = i + offset
		if value <= maxValue then
			local row = self.rows[i]
			UpdateSearchFrame(row,value)
			row:Show()
			if i == scrolltooltip then
				ResultFrame_OnEnter(row)
			end
		else
			self.rows[i]:Hide()
		end
	end
end

local bar = CreateFrame("ScrollFrame", "$parentScrollBar", scroll, "FauxScrollFrameTemplate")
bar:SetPoint("TOPLEFT", 0, -4)
bar:SetPoint("BOTTOMRIGHT", -25, 2)

bar:SetScript("OnVerticalScroll", function(self, offset)
	self.offset = math.floor(offset / ROW_HEIGHT + 0.5)

	scroll:Update()
end)

bar:SetScript("OnShow", function()
	scroll:Update()
end)

scroll.scrollBar = bar

local function MakeResultFrame(index)
	local frame = CreateFrame("Button",nil,scroll)
	frame:SetSize(220,ROW_HEIGHT)
	if index == 1 then
		--frame:SetPoint("TOPLEFT", hopAddon.searchFrame.searchBox, "BOTTOMLEFT",-4,0)
		frame:SetPoint("TOPLEFT",0,0)
	else
		frame:SetPoint("TOP", ResultFrames[index-1], "BOTTOM")	
	end

	frame:SetBackdrop({bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", 
						edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border", 
						tile = true, tileSize = 16, edgeSize = 16, 
						insets = { left = 2, right = 2, top = 2, bottom = 2 }});
	frame:SetBackdropColor(0,0,0,1);
	
	local highlightTexture = frame:CreateTexture()
	highlightTexture:SetPoint("TOPLEFT",-2,5)
	highlightTexture:SetPoint("BOTTOMRIGHT",2,-7)
	highlightTexture:SetTexture("Interface\\PaperDollInfoFrame\\UI-Character-Tab-Highlight")
	frame:SetHighlightTexture(highlightTexture,"ADD")


	frame.iconRedx	= frame:CreateTexture(nil,"OVERLAY")
	frame.iconRedx:SetSize(20,20)
	frame.iconRedx:SetPoint("RIGHT", frame, "RIGHT", -8,0)
	frame.iconRedx:SetAtlas("groupfinder-icon-redx")

	frame.iconGreenChck	= frame:CreateTexture(nil,"OVERLAY")
	frame.iconGreenChck:SetSize(20,20)
	frame.iconGreenChck:SetPoint("RIGHT", frame, "RIGHT", -8,0)
	frame.iconGreenChck:SetAtlas("Tracker-Check")
	
	frame.spinner = CreateFrame("Frame",nil,frame,"LoadingSpinnerTemplate")
	frame.spinner:SetSize(40,40)
	frame.spinner:SetPoint("RIGHT", frame, "RIGHT", 0,0)
	
	frame.textCount = frame:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
	frame.textCount:SetSize(18,16)
	frame.textCount:SetPoint("LEFT",frame,"LEFT",7,0)
	frame.textCount:SetJustifyH("CENTER")
	frame.textCount:SetText("")
	
	frame.icon	= frame:CreateTexture(nil,"OVERLAY")
	frame.icon:SetSize(16,16)
	frame.icon:SetPoint("LEFT", frame, 24,0)
	
	frame.textName = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
	frame.textName:SetSize(178,18)
	frame.textName:SetPoint("LEFT",frame.icon,"RIGHT",4,0)
	frame.textName:SetJustifyH("LEFT")
	frame.textName:SetText("")
	
	frame.result = 0
	frame.index = index
	
	frame:SetScript("OnEnter", ResultFrame_OnEnter)
	frame:SetScript("OnLeave", function(self) scrolltooltip = 0 GameTooltip:Hide() end);	
	frame:RegisterForClicks("AnyUp")
	frame:SetScript("OnClick", function(self,button,down)
	
		if button == "LeftButton" then
			local apps = C_LFGList.GetApplications();
			local total = 0
			for i=1, #apps do
				local id, appStatus, pendingStatus,appDuration = C_LFGList.GetApplicationInfo(apps[i]);
				if appDuration == 0 and pendingStatus == "applied" then
					-- phantom groups fix
					-- phantom groups have 0 duration and applied pending
				else
					if pendingStatus == "applied" or appStatus == "applied"
					or pendingStatus == "invited" or appStatus == "invited" then
						total = total + 1
					end
				end
			end
			
			local active, activityID, iLevel, honorLevel, name, comment, voiceChat, duration, autoAccept  = C_LFGList.GetActiveEntryInfo();	
			
			if total < 5 and ((not IsInGroup() and not IsInRaid()) or UnitIsGroupLeader('player')) and not name then
				PlaySound(PlaySoundKitID and "igMainMenuOptionCheckBoxOn" or 856)
				local tank, heal, dd = C_LFGList.GetAvailableRoles()
				tank = tank and SHtankButton.CheckButton:GetChecked()
				heal = heal and SHhealerButton.CheckButton:GetChecked()
				dd = dd and SHdamagerButton.CheckButton:GetChecked()
				C_LFGList.ApplyToGroup(self.result, "", tank, heal, dd)

				table.remove(FilteredGroupsList,index)
				scroll:Update()
			else
				PlaySound(PlaySoundKitID and "ui_garrison_architecttable_buildingplacementerror" or 47355, "Master")
			end
		end
		
		if button == "RightButton" then
			local id, activityID, name, comment, voiceChat, iLvl, honorLevel, age, numBNetFriends, numCharFriends, numGuildMates, isDelisted, leaderName, numMembers, autoinv = C_LFGList.GetSearchResultInfo(self.result)
			local id, appStatus, pendingStatus = C_LFGList.GetApplicationInfo(self.result);
			
			if appStatus == "applied" or pendingStatus == "applied" then
				C_LFGList.CancelApplication(self.result)
			elseif pendingStatus == "invited" or appStatus == "invited" then
				LFGListInviteDialog.DeclineButton:Click()
			else
				local t={}
				t[name]=GetTime()-age
				--print("Blacklisted "..name)
				table.insert(CustomSearchBlackList,t)
				table.remove(FilteredGroupsList,index)
			end
			
			if IsShiftKeyDown() then
				table.insert(hopAddon.tables.LeadersBL,leaderName)
			end
			
			scroll:Update()
			PlaySound(PlaySoundKitID and "igMainMenuOptionCheckBoxOn" or 856)
		end
	end)
	
	frame:Hide()
	ResultFrames[index]=frame
end

for i=1,MAX_ROWS do MakeResultFrame(i) end
scroll.rows = ResultFrames


-- Hyperlink for search results

local origChatFrame_OnHyperlinkShow = ChatFrame_OnHyperlinkShow

ChatFrame_OnHyperlinkShow = function(...)
	local chatFrame, link, text, button = ...
	if type(text) == "string" and text:match("sehopR") and not IsModifiedClick() then
		SH_ShowResultChatInfo(text)
	else
		return origChatFrame_OnHyperlinkShow(...)
	end
end

function SH_ShowResultChatInfo(text)
	local _,grname,grdesc,_ = strsplit("*",text)
	local header = string.match(text,'%[(.+)%]')
	
	ShowUIPanel(ItemRefTooltip)
	if (not ItemRefTooltip:IsVisible()) then
		ItemRefTooltip:SetOwner(UIParent, "ANCHOR_PRESERVE")
	end
	ItemRefTooltip:ClearLines()
	ItemRefTooltip:AddLine(header)
	ItemRefTooltip:AddLine(" ",1,1,1,1)
	ItemRefTooltip:AddLine(grname,1,1,1,1)
	if grdesc ~= "" then
		ItemRefTooltip:AddLine(grdesc,1,1,1,1)
	end
	ItemRefTooltip:Show()
end


-- MAIN FUNCTIONS --

local function SplitKeywords(keyword)
	SearchKeywords = {}
	for i in string.gmatch(keyword, "%S+") do
		if i ~= "-" then
			table.insert(SearchKeywords,i)
		end
	end
end

local function CustomSearch()
	hopAddon.searchFrame.scrollframe:Hide()
	
	local languages = C_LFGList.GetLanguageSearchFilter()
	C_LFGList.Search(hopAddon.searchFrame.dropDown.activeValue, LFGListSearchPanel_ParseSearchTerms(""), 0, 0, languages)
end

local function FoundInDelisted(appID)
	local id, activityID, name, comment, voiceChat, iLvl, honorLevel, age, numBNetFriends, numCharFriends, numGuildMates, isDelisted, leaderName, numMembers, autoinv = C_LFGList.GetSearchResultInfo(appID)
	local id, appStatus, pendingStatus, appDuration = C_LFGList.GetApplicationInfo(appID)
	
	-- if your selected group has the same leader as you - it's your group
	if C_LFGList.GetActiveEntryInfo() then
		local index = GetNumGroupMembers()
		local name = GetUnitName("player")
		
		if IsInRaid() then
			for i=1,index do
				if UnitIsGroupLeader("raid"..i) then
					name = GetUnitName("raid"..i,true)
					break
				end
			end
		else
			for i=1,4 do
				if UnitIsGroupLeader("party"..i) then
					name = GetUnitName("party"..i,true)
					break
				end
			end
		end

		if name == leaderName then return true end
	end
	
	if isDelisted or appStatus == "invitedeclined" or appStatus == "declined" or appStatus == "timedout"
	or appStatus == "cancelled" or pendingStatus == "cancelled" or appStatus == "failed" 
	then
		return true
	end
	
	return false
end

local function RemoveDelisted()
	local removed = false

	local i=#FilteredGroupsList
	while i > 0 do		
		if FoundInDelisted(FilteredGroupsList[i]) then
			table.remove(FilteredGroupsList,i)
			removed = true
		end
		i = i - 1
	end

	if removed then	scroll:Update() end
end

local function FoundInBlackList(caption,ctime)
	local epsilon = 2 -- blacklisting by name and creation time, time can give a small error
	
	for k,v in pairs(CustomSearchBlackList) do
		for key,value in pairs(v) do
			if key == caption then
				if (value <= GetTime()-ctime+epsilon) and (value >= GetTime()-ctime-epsilon) then
					--print("Blacklist match: "..key.." "..value)
					return true
				end
			end				
		end
		
	end

	return false
end

--[[
	Force launches when user changes filter
	Launches by searchresultrecieved event
]]

local function ResultMeetsFilters(appID,minusKeywords,plusKeywords,fullKeywords)
	local id, appStatus, pendingStatus, appDuration = C_LFGList.GetApplicationInfo(appID)
	local id, activityID, name, comment, voiceChat, iLvl, honorLevel, age, numBNetFriends, numCharFriends, numGuildMates, isDelisted, leaderName, numMembers, autoinv = C_LFGList.GetSearchResultInfo(appID)
	
	--[[
			!!!GROUP INFO FILTERING!!!
	]]
	-- group is not delisted or player in that group
	if FoundInDelisted(appID) then return end
	-- it is a Server Hop hosted group
	if string.find(name,"[Server Hop]",1,true) then return end
	-- group leader is not in blacklist
	for i=1,#hopAddon.tables.LeadersBL do
		if leaderName == hopAddon.tables.LeadersBL[i] then return end
	end
	-- group is not in blacklist
	if FoundInBlackList(name,age) then return end
	
	-- time since created check
	if hopAddon.searchFrame.holderCatFilters.timeCheck:GetChecked() and age > tonumber(hopAddon.searchFrame.holderCatFilters.timeEdit:GetText()) then
		return
	end
	-- group must have higher ilevel than filter
	if hopAddon.searchFrame.holderCatFilters.ilvlCheck:GetChecked() and iLvl < tonumber(hopAddon.searchFrame.holderCatFilters.ilvlEdit:GetText()) then
		return
	end
	-- group must have autoinvite on
	if hopAddon.searchFrame.holderCatFilters.autoInvCheck:GetChecked() and not autoinv then
		return
	end
	
	-- if particular instance is selected in second dropdown, detect it
	local InstanceIdFilter = 0
	-- dungeon category
	if hopAddon.searchFrame.dropDown.activeValue == 2 then
		-- check for roles here
		local memberCounts = C_LFGList.GetSearchResultMemberCounts(appID);
		if (SHtankButton.CheckButton:GetChecked() and memberCounts.TANK == 0)
		or (SHhealerButton.CheckButton:GetChecked() and memberCounts.HEALER == 0)
		or (SHdamagerButton.CheckButton:GetChecked() and memberCounts.DAMAGER < 3)
		then else return end

		if #DungeonActivities > 0 then
			local foundActivity = false
			for k,v in pairs(DungeonActivities) do
				if v == activityID then
					foundActivity = true
					break
				end
			end
			if not foundActivity then return end
		end
	-- raid category
	elseif hopAddon.searchFrame.dropDown.activeValue == 3 then
		InstanceIdFilter = rDrop.activeValue
	end

	-- if there's instance selected
	if InstanceIdFilter ~= 0 then
		-- then check if it meets activity of the group
		if activityID ~= InstanceIdFilter then return end
	end

	--[[	
			!!!FORM THAT SEARCHSTRING!!!
	]]
	
	-- add activityID name to searchstring for quests (dungeons and raids for rework)
	local actionName = ""
	if hopAddon.searchFrame.dropDown.activeValue == 1 
	--or hopAddon.searchFrame.dropDown.activeValue == 2 
	--or hopAddon.searchFrame.dropDown.activeValue == 3
	then
		-- for quests that's zone
		actionName = C_LFGList.GetActivityInfo(activityID)
	end
	
	-- this is the string we're searching in
	local app = actionName:lower().." "..name:lower().." "..comment:lower();
	
	--[[
			!!!WORD FILTERING!!!
	]]
	-- if there's "-keyword", remove it
	for k,word in pairs(minusKeywords) do
		if string.find(app,word,1,true) then
			return
		end
	end
	
	-- if there's nothing to search for, just add a group since it passed all filters
	if #plusKeywords == 0 and #fullKeywords == 0 then
		table.insert(FilteredGroupsList,id)
		return
	end
	
	-- if user entered any keywords, check them
	if #plusKeywords > 0 then
		for k,word in pairs(plusKeywords) do
			if string.find(app,word,1,true) then
				table.insert(FilteredGroupsList,id)
				return
			end
		end
	end
	
	if #fullKeywords > 0 then
		local wordlist = {}
		for word in app:gmatch("%S+") do
			table.insert(wordlist,word)
		end

		for k,keyword in pairs(fullKeywords) do
			for i,word in pairs(wordlist) do
				if word == keyword then
					table.insert(FilteredGroupsList,id)
					return
				end
			end
		end
	end
end

local function FilterCustomSearchResults()
	FilteredGroupsList = {}
	
	-- manage keywords once
	local minusKeywords = {}
	local plusKeywords = {}
	local fullKeywords = {}
	for k,word in pairs(SearchKeywords) do
		if string.sub(word,1,1)=="-" then
			table.insert(minusKeywords,string.sub(word,2))
		elseif string.sub(word,1,1)=="#" then
			table.insert(fullKeywords,string.sub(word,2))
		else
			table.insert(plusKeywords,word)
		end
	end
		
	local count, list = C_LFGList.GetSearchResults()
	if count > HOPADDON_MAX_RESULTS then count = HOPADDON_MAX_RESULTS end
	
	if hopAddon.searchFrame.dropDown.activeValue == 2 then
		DungeonActivities = {}
		for k,v in pairs(DungeonList) do
			if v["checked"] then
				for k1,v1 in pairs(v["activityList"]) do
					table.insert(DungeonActivities,v1)
				end
			end
		end
	end

	for i = 1,count do
		-- checking for overlapping bugs for each element
		if list[i] ~= nil then
			ResultMeetsFilters(list[i],minusKeywords,plusKeywords,fullKeywords)
		end
	end
	-- if there're groups that meet conditions, show them
	if #FilteredGroupsList > 0 then
		hopAddon.searchFrame.anim:Stop()
		
		hopAddon.searchFrame.scrollframe:Show()
		_G[hopAddon.searchFrame.searchBox:GetName().."SearchIcon"]:Hide()
		
		if false then
			-- add a setting for it here
			hopAddon:Show()
		end

	end
	scroll:Update()

	if #FilteredGroupsList == 0 then
		CustomSearch()
	end
end

function FinishCustomSearch()
	var.addonSearchRequest = false

	SearchKeywords = {}
	DoSearch = false
	FilteredGroupsList = {}
	CustomSearchBlackList = {}
	hopAddon.searchFrame.anim:Stop()
	hopAddon.searchFrame.scrollframe:Hide()
end

-- LATE FUNCTIONS --

function hopAddon.searchFrame.dropDown_OnClick(self, arg1, arg2, checked)
	local prevValue = hopAddon.searchFrame.dropDown.activeValue
	hopAddon.searchFrame.dropDown.activeValue = self.value;
	CloseDropDownMenus();

	if prevValue ~= self.value then

		hopAddon.buttonPlus:Hide()
		rDrop:Hide()
		rDrop.activeValue = 0
		if self.value == 2 then	hopAddon.buttonPlus:Show()
		elseif self.value == 3 then rDrop:Show()
		end
		
		hopAddon.searchFrame.dropDown.text:SetText(C_LFGList.GetCategoryInfo(self.value))

		hopAddon.searchFrame.scrollframe:Hide()	
	end
end

-- Groups dropdown list set up
local function droponclick(self,arg1,arg2,checked)
	if self.value == 0 then
	
	elseif self.value == -2 or self.value == -3 or self.value == -4 then
		for k,v in pairs(DungeonList) do
			if v["groupID"] == self.value then
				v["checked"] = true
			else
				v["checked"] = false
			end
		end
	else
		local id = tonumber(self.value)
		
		if id ~= nil then
			local value = DungeonList[id]
			value["checked"] = not value["checked"]

			--[[

			local f
			if menuFrame.lastClick ~= nil then
				f = menuFrame.lastClick
			else
				-- activate a new frame
				activeFrames = activeFrames + 1
				f = InstanceFrames[activeFrames]
			end
			
			f.Activities = arg1
			f.bgImage:SetTexture(arg2)

			if DungeonList[id]["texCoord"] then
				f.bgImage:SetTexCoord(value["texCoord"][1],value["texCoord"][2],value["texCoord"][3],value["texCoord"][4])
			else
				f.bgImage:SetTexCoord(0.2,0.5,0.15,0.45)
			end

			f:Show()

			f.heroic:Hide()
			f.mythic:Hide()
			f.challenge:Hide()

			if DungeonList[id]["groupID"] == -2 then
				f.heroic:Show()
			elseif DungeonList[id]["groupID"] == -3 then
				f.mythic:Show()
			elseif DungeonList[id]["groupID"] == -4 then
				f.challenge:Show()
			end
			
			]]

		end
	end
end

local function dropinitialize(self,level,menuList)
	local info = UIDropDownMenu_CreateInfo()

	if (not level or level == 1) then
		
		local h = 0
		local m = 0
		local c = 0
		local w = 0
		for k,v in pairs(DungeonList) do
			if v["checked"] then
				local group = v["groupID"]
				if group == -2 then
					h=h+1
				elseif group == -3 then
					m=m+1
				elseif group == -4 then
					c=c+1
				else
					w=w+1
				end
			end
		end

		local hstring = "" if h>0 then hstring = " ("..h..")" end
		local mstring = "" if m>0 then mstring = " ("..m..")" end
		local cstring = "" if c>0 then cstring = " ("..c..")" end
		local wstring = "" if w>0 then wstring = " ("..w..")" end
		local kstring = "Mythic Keystone" if GetItemInfo(138019) ~= nil then kstring = tostring(GetItemInfo(138019)) end

		-- mythic key
		info.text = kstring..cstring
		info.value = -4
		info.func = droponclick
		info.hasArrow = true
		info.notCheckable = true
		UIDropDownMenu_AddButton(info,level)
		-- mythic
		info.text = PLAYER_DIFFICULTY6..mstring
		info.value = -3
		info.func = droponclick
		info.hasArrow = true
		info.notCheckable = true
		UIDropDownMenu_AddButton(info,level)		
		-- heroic
		info.text = PLAYER_DIFFICULTY2.."/"..PLAYER_DIFFICULTY1..hstring
		info.value = -2
		info.func = droponclick
		info.hasArrow = true
		info.notCheckable = true
		UIDropDownMenu_AddButton(info,level)	
		-- old content
		info.text = HOPADDON_OLDCONTENT..wstring
		info.value = "OLDCONTENT"
		info.func = nil
		info.hasArrow = true
		info.notCheckable = true
		info.keepShownOnClick = true
		UIDropDownMenu_AddButton(info,level)

		-- no filter
		info.text = CANCEL;
		info.value = 0;	
		info.func = droponclick
		info.hasArrow = false
		info.notCheckable = true
		info.keepShownOnClick = false		
		UIDropDownMenu_AddButton(info);

	elseif (level == 2) then
		local raidAct = C_LFGList.GetAvailableActivities(2, nil, 0,"")

		for k,v in pairs(DungeonList) do
			groupID = v["groupID"]
			if UIDROPDOWNMENU_MENU_VALUE == groupID then
				local name, descr, bgImage,buttonImage = EJ_GetInstanceInfo(v["imageID"])
				if v["name"] then name = v["name"] end
				info.text = name
				info.value = ""..k
				info.arg1 = v["activityList"]
				info.arg2 = buttonImage
				info.func = droponclick
				info.hasArrow = false
				info.checked = v["checked"]
				info.keepShownOnClick = true
				UIDropDownMenu_AddButton(info,level)
			end
		end
		
		-- creating expansions
		if UIDROPDOWNMENU_MENU_VALUE == "OLDCONTENT" then
			for i=5,0,-1 do
				info.text = _G["EXPANSION_NAME"..i]
				info.value = "EXPANSION"..i
				info.func = nil
				info.hasArrow = true
				info.notCheckable = true
				info.keepShownOnClick = true				
				UIDropDownMenu_AddButton(info,level)
			end	
		end
	elseif (level == 3) then

		for k,v in pairs(DungeonList) do
			expac = v["groupID"]
			if UIDROPDOWNMENU_MENU_VALUE == "EXPANSION"..expac then
				local name, descr, bgImage,buttonImage = EJ_GetInstanceInfo(v["imageID"])
				if v["name"] then name = v["name"] end

				info.text = name
				info.value = ""..k
				info.arg1 = v["activityList"]
				info.arg2 = buttonImage
				info.func = droponclick
				info.hasArrow = false
				info.checked = v["checked"]
				info.keepShownOnClick = true
				UIDropDownMenu_AddButton(info,level)
			end
		end

	end
end

UIDropDownMenu_Initialize(menuFrame, dropinitialize)

-- override on enter, onhide event is inside template
hopAddon.buttonPlus:SetScript("OnClick", function(self)
	CloseDropDownMenus()
	menuFrame.lastClick = nil
	ToggleDropDownMenu(1, nil, menuFrame, self:GetName(), 0, 0, menuFrame.menuList, self, nil)
end)


-- Raids dropdown list set up
function hopAddon.searchFrame.raidsDrop_OnClick(self,arg1,arg2,checked)
	rDrop.activeValue = self.value

	if self.value == 0 then
		UIDropDownMenu_SetText(rDrop,NONE)
	else
		UIDropDownMenu_SetText(rDrop,C_LFGList.GetActivityInfo(self.value))
	end
	CloseDropDownMenus()
	hopAddon_searchBox_OnEnter(hopAddon.searchFrame.searchBox)
end

local rDropList = {
	-- classic
	[9] = 0,[293] = 0,[294] = 0,[295] = 0,
	-- bc
	[45] = 1,[296] = 1,[297] = 1,[298] = 1,
	[299] = 1,[300] = 1,[301] = 1,
	-- wotlk
	[43] = 2,[44] = 2,[46] = 2,[47] = 2,
	[48] = 2,[49] = 2,[302] = 2,[303] = 2,
	[304] = 2,[305] = 2,[306] = 2,[307] = 2,
	[308] = 2,[309] = 2,[310] = 2,[311] = 2,
	-- cata
	[313] = 3,[316] = 3,[317] = 3,[318] = 3,
	[319] = 3,[320] = 3,[321] = 3,[322] = 3,
	[323] = 3,[324] = 3,[325] = 3,[326] = 3,
	[327] = 3,[328] = 3,[329] = 3,[330] = 3,
	[331] = 3,[332] = 3,[333] = 3,[334] = 3,
	-- pandaria
	[397] = 4,
	[4] = 4,[41] = 4,[42] = 4,[335] = 4,
	[336] = 4,[337] = 4,[338] = 4,[339] = 4,
	[340] = 4,[341] = 4,[342] = 4,[343] = 4,
	[344] = 4,[345] = 4,[346] = 4,[347] = 4,
	[348] = 4,[349] = 4,[350] = 4,
	-- draenor
	[398] = 5,
	[37] = 5,[38] = 5,[39] = 5,
	[40] = 5,[399] = 5,[400] = 5,
	[409] = 5,[410] = 5,[412] = 5,
	-- legion
	[413] = 100, [414] = 100, [468] = 100, -- raid 1: emerald nightmare
	[456] = 200, [457] = 200, [480] = 200, -- raid 7.1: helya
	[415] = 300, [416] = 300, [481] = 300, -- raid 2: nighthold
	[478] = 400, [479] = 400, [492] = 400, -- raid 3: tomb of sargeras
	[482] = 500, [483] = 500, [493] = 500 -- raid 4: antorus, the burning throne
}

-- TEMPLATE

-- [482] = 500 -- raid 4: antorus, the burning throne
-- )

-- Macro for searching Instance ID:
-- /run     for i=1, 2000 do        local instance = C_LFGList.GetActivityInfo(i)        if instance then            print(i)            print(instance)        end    end

-- /TEMPLATE 

function rDrop.initialize(self,level)
	local info = UIDropDownMenu_CreateInfo()

	if (not level or level == 1) then
		-- no filter
		info.text = NONE;
		info.value = 0;	
		info.func = hopAddon.searchFrame.raidsDrop_OnClick
		info.hasArrow = false;
		info.checked = false
		UIDropDownMenu_AddButton(info);
		-- old content
		info.text = HOPADDON_OLDCONTENT
		info.value = "1"
		info.func = nil
		info.hasArrow = true
		info.checked = false
		UIDropDownMenu_AddButton(info,level)

		-- outdoor bosses
		info.text = C_LFGList.GetActivityInfo(458)
		info.value = 458
		info.func = hopAddon.searchFrame.raidsDrop_OnClick
		info.hasArrow = false;
		info.checked = (self.activeValue == info.value)
		UIDropDownMenu_AddButton(info,level)

		-- nighmare raid
		info.text = GetMapNameByID(1094)
		info.value = 100
		info.func = nil
		info.hasArrow = true
		info.checked = false
		UIDropDownMenu_AddButton(info,level)
		-- thorim raid
		info.text = GetMapNameByID(1114)
		info.value = 200
		info.func = nil
		info.hasArrow = true
		info.checked = false
		UIDropDownMenu_AddButton(info,level)		
		-- elf raid
		info.text = GetMapNameByID(1088)
		info.value = 300
		info.func = nil
		info.hasArrow = true
		info.checked = false
		UIDropDownMenu_AddButton(info,level)
		-- tomb of sargeras
		info.text = GetMapNameByID(1147)
		info.value = 400
		info.func = nil
		info.hasArrow = true
		info.checked = false
		UIDropDownMenu_AddButton(info,level)
		-- Antorus, the Burning Throne
		info.text = GetMapNameByID(1188)
		info.value = 500
		info.func = nil
		info.hasArrow = true
		info.checked = false
		UIDropDownMenu_AddButton(info,level)
	elseif (level == 2) then
		local raidAct = C_LFGList.GetAvailableActivities(3, nil, 0,"")
		
		-- creating expansions
		if UIDROPDOWNMENU_MENU_VALUE == "1" then
			for i=0,5 do
				info.text = _G["EXPANSION_NAME"..i]
				info.value = "3"..i
				info.func = nil
				info.hasArrow = true
				info.checked = false
				UIDropDownMenu_AddButton(info,level)
			end	
		end
		
		for i=1,#raidAct do
			local activityID = raidAct[i];
			local fullName, shortName, categoryID, groupID, itemLevel, filters, minLevel, maxPlayers, displayType = C_LFGList.GetActivityInfo(activityID);
			
			local expac = rDropList[activityID]
			if expac then
				if UIDROPDOWNMENU_MENU_VALUE == expac then
					info.text = fullName
					info.value = activityID
					info.func = hopAddon.searchFrame.raidsDrop_OnClick
					info.hasArrow = false;
					info.checked = (self.activeValue == info.value)
					UIDropDownMenu_AddButton(info,level)
				end
			end
		end

	elseif (level == 3) then
		local raidAct = C_LFGList.GetAvailableActivities(3, nil, 0,"")
		-- low level raids
		for i=1, #raidAct do
			local activityID = raidAct[i];
			local fullName, shortName, categoryID, groupID, itemLevel, filters, minLevel, maxPlayers, displayType = C_LFGList.GetActivityInfo(activityID);
			
			local expac = rDropList[activityID]
			if expac then
				if UIDROPDOWNMENU_MENU_VALUE == "3"..expac then
					info.text = fullName
					info.value = activityID
					info.func = hopAddon.searchFrame.raidsDrop_OnClick
					info.hasArrow = false;
					info.checked = (self.activeValue == info.value)
					UIDropDownMenu_AddButton(info,level)
				end
			end
		end
	end
end

UIDropDownMenu_Initialize(rDrop, rDrop.initialize)
UIDropDownMenu_SetSelectedID(rDrop, 0)
UIDropDownMenu_SetText(rDrop,NONE)
UIDropDownMenu_SetWidth(rDrop,180)


function hopAddon_searchBox_OnEnter(frame)
	frame:ClearFocus()
		
	SplitKeywords(frame:GetText():lower())
	if #SearchKeywords == 0 then
		hopAddon.searchFrame.searchBox.clearButton:Show()
	end
	
	-- first, check if we have available groups in current results
	FilterCustomSearchResults()
end

hopAddon.searchFrame.searchBox:SetScript("OnEnterPressed", hopAddon_searchBox_OnEnter)

hopAddon.searchFrame.searchBox.clearButton:SetScript("OnClick", function(btn)
	-- also clear field
	SearchBoxTemplateClearButton_OnClick(btn)
	hopAddon.searchFrame.SearchString = ""
	FinishCustomSearch()
	btn:Hide()
end)

hopAddon.searchFrame.refreshButton:SetScript("OnClick", function(self)
	CustomSearch()
	PlaySound(PlaySoundKitID and "igMainMenuOptionCheckBoxOn" or 856)
end)


-- EVENT FUNCTIONS --

local function SearchFrame_EventSystem(self, event, ...)
	local arg1 = ...

	if event == "LFG_LIST_SEARCH_RESULTS_RECEIVED" then
		searchUpdated = 0
		FilterCustomSearchResults()
	end
			
	if event == "LFG_LIST_SEARCH_RESULT_UPDATED" then
		if #FilteredGroupsList > 0 then
			RemoveDelisted()
		end
	end
	
	if event == "GROUP_ROSTER_UPDATE" then
		-- Stopping search in a group
		if var.addonSearchRequest then
			if customOptions.endlessCheckBox:GetChecked() then
				FinishCustomSearch()
			end
		end		
	end
	if event == "PLAYER_LEVEL_UP" then
		UIDropDownMenu_Initialize(rDrop, rDrop.initialize)
		UIDropDownMenu_Initialize(dDrop, dDrop.initialize)
	end
end

local backgroundScanner = CreateFrame("Frame")
--Events
backgroundScanner:SetScript("OnEvent", SearchFrame_EventSystem)
backgroundScanner:RegisterEvent("LFG_LIST_SEARCH_RESULT_UPDATED")
backgroundScanner:RegisterEvent("LFG_LIST_SEARCH_RESULTS_RECEIVED")
backgroundScanner:RegisterEvent("LFG_LIST_JOINED_GROUP")
backgroundScanner:RegisterEvent("GROUP_ROSTER_UPDATE")
backgroundScanner:RegisterEvent("PLAYER_LEVEL_UP")

local function OnUpdate(self,elapsed)
	searchUpdated = searchUpdated + elapsed
		
	-- HARDTESTED: blizzard allows to throw 2 search results per 5 seconds	
	-- for customSearch we just get results when they should be available
	if searchUpdated >= 2.5 and (#FilteredGroupsList == 0 or not hopAddon.searchFrame.scrollframe:IsShown()) then
		hopAddon.searchFrame.refreshButton.anim:Play()
	else
		hopAddon.searchFrame.refreshButton.anim:Stop()
	end
end

backgroundScanner:SetScript("OnUpdate",OnUpdate)



--[[
local f = CreateFrame("Button",nil,UIParent)
f:SetPoint("CENTER",0,0)
f:SetSize(80,20)
f:SetHighlightTexture("UI-EJ-DungeonButton-Highlight")

local instanceID, name, description, _, buttonImage, _, _, _, link = EJ_GetInstanceByIndex(1, false)

f.bgImage = f:CreateTexture(nil, "OVERLAY")
f.bgImage:SetPoint("TOPLEFT",0,0)
f.bgImage:SetPoint("BOTTOMRIGHT",0,0)
f.bgImage:SetTexCoord(0,0.68359375,0,0.7421875)


f.bgImage:SetTexture(1537284);

f:Show()


/run local index = 1 local id,name = EJ_GetInstanceByIndex(index, false) while id do print(id, name) index = index + 1 id,name = EJ_GetInstanceByIndex(index, false) end 



EJ_GetInstanceInfo(instanceID)

707 Казематы Стражей
726 Катакомбы Сурамара
800 Квартал Звезд
740 Крепость Черной Ладьи
767 Логово Нелтариона
716 Око Азшары
900 Собор Вечной Ночи
727 Утроба душ
762 Чаща Темного Сердца
721 Чертоги Доблести
777 Штурм Аметистовой крепост

768 Изумрудный Кошмар
861 Испытание доблести
786 Цитадель Ночи
875 Гробница Саргерас





local dFilter = CreateFrame("Frame",nil,UIParent)
hopAddon.dFilter = dFilter

dFilter:SetSize(100,100)
dFilter:SetPoint("CENTER",0,0)

--dFilter.Texture = dFilter:CreateTexture("kektex","OVERLAY")
--dFilter.Texture:SetAllPoints(dFilter)
--dFilter.Texture:SetTexture("Interface\\Common\\mini-hourglass")



local function CreateInstanceButton(row,column)
	local width = 44
	local height = 24

	local f = CreateFrame("Button","IB"..row..column,dFilter)
	f:SetSize(width,height)
	f:SetPoint("BOTTOMLEFT",(column-1)*width,(row-1)*height)
	
	f.bgImage = f:CreateTexture("tt","OVERLAY")
	f.bgImage:SetPoint("TOPLEFT",0,0)
	f.bgImage:SetPoint("BOTTOMRIGHT",0,0)
	f.bgImage:SetTexture(1537284)
	f.bgImage:SetTexCoord(0.2,0.5,0.15,0.45)

	f.heroic = f:CreateTexture("heroic","OVERLAY",nil,2)
	f.heroic:SetSize(12,13)
	f.heroic:SetPoint("LEFT",0,0)
	f.heroic:SetTexture("Interface\\GuildFrame\\GuildDifficulty")
	f.heroic:SetTexCoord(0.67968750,0.77343750,0.65625000,0.85937500)
	f.heroic:Hide()

	f.mythic = f:CreateTexture("mythic","OVERLAY",nil,2)
	f.mythic:SetSize(12,13)
	f.mythic:SetPoint("LEFT",0,0)
	f.mythic:SetTexture("Interface\\GuildFrame\\GuildDifficulty")
	f.mythic:SetTexCoord(0.77343750,0.8671875,0.65625000,0.85937500)
	f.mythic:Hide()

	f.challenge = f:CreateTexture("challenge","OVERLAY",nil,2)
	f.challenge:SetSize(12,12)
	f.challenge:SetPoint("LEFT",0,0)
	f.challenge:SetTexture("Interface\\Common\\mini-hourglass")
	f.challenge:Hide()

	f.Activities = {}
	f.Status = 1

	f:RegisterForClicks("AnyUp")
	f:SetScript("OnClick", function(self,button,down)
		CloseDropDownMenus()
		
		for k,v in pairs(f.Activities) do
			local name = C_LFGList.GetActivityInfo(v)
			print(name)
		end

		menuFrame.lastClick = self
		
		ToggleDropDownMenu(1, nil, menuFrame, self:GetName(), 0, 0, menuFrame.menuList, self, nil)

	end)

	f:Hide()
	return f
end

InstanceFrames = {}
for i=1,2 do
	for j=1,7 do
		table.insert(InstanceFrames,CreateInstanceButton(i,j))
	end
end


local activeFrames = 0


]]

--[[

-- Auchindoun					-- Bloodmaul Slag Mines			-- Grimrail Depot				-- Iron Docks
AddDungeon(5,{23,31,403},547)	AddDungeon(5,{21,29,401},385)	AddDungeon(5,{25,33,405},536)	AddDungeon(5,{22,30,402},558)
-- Shadowmoon Burial Grounds	-- Skyreach 					-- The Everbloom				-- Upper Blackrock Spire
AddDungeon(5,{27,35,407},537)	AddDungeon(5,{24,32,404},476)	AddDungeon(5,{26,34,406},556)	AddDungeon(5,{28,36,408},559)
-- Random Draenor Dungeon
AddDungeon(5,{395,396},557, LFG_TYPE_RANDOM_DUNGEON.." ("..EXPANSION_NAME5..")")




/run InstanceFrames[1].bgImage:SetTexCoord(0,1,0,1)

/run local index = 1 local id,name = EJ_GetInstanceByIndex(index, false) while id do print("-- "..name) print("AddDungeon(4,{},"..id..")") index = index + 1 id,name = EJ_GetInstanceByIndex(index, false) end 

/run LST={	[434] = 100,[444] = 200,[454] = 300,[455] = 300, [470] = 200,[471]=400,[472] = 200,[473]=400,[474] = 200,[475] = 300,[476]=400,[417] = 100,[418] = 200}
/run for k,v in pairs(LST) do name = C_LFGList.GetActivityInfo(k) print(k.." "..name) end




/run for i=1,100 do name = C_LFGList.GetActivityInfo(i) if name ~= nil then print(i.." "..name) end end

	-- Legion
	[425] = 100,[435] = 200,[445] = 300,[459]=400,
	[426] = 100,[436] = 200,[446] = 300,[460]=400,
	[427] = 100,[437] = 200,[447] = 300,[461]=400,
	[428] = 100,[438] = 200,[448] = 300,[462]=400,
	[429] = 100,[439] = 200,[449] = 300,[463]=400,
	[430] = 100,[440] = 200,[450] = 300,[464]=400,
	[431] = 100,[441] = 200,[451] = 300,[465]=400,
	[432] = 100,[442] = 200,[452] = 300,[466]=400,
	[433] = 100,[443] = 200,[453] = 300,[467]=400,
	[434] = 100,[444] = 200,[454] = 300,
							[455] = 300, 
				[470] = 200,			[471]=400,
				[472] = 200,			[473]=400,
				[474] = 200,[475] = 300,[476]=400,		
	[417] = 100,[418] = 200, -- random



]]