local addonName, farmerVars = ...

local currentVersion = 0113000

local rarityList = {
  [0] = 'Poor',
  'Common',
  'Uncommon',
  'Rare',
  'Epic',
  'Legendary',
  'Artifact',
  'Heirloom',
  'WoW Token'
}

local checkButtonList = {}
local sliderList = {}
local events = {}

farmerVars.rarityColors = {}
for i = 0, 8 do
  farmerVars.rarityColors[i] = {GetItemQualityColor(i)}
end

local farmerOptionsFrame = CreateFrame('Frame', 'farmerOptionsFrame', UIParent)
farmerOptionsFrame.name = 'Farmer'
InterfaceOptions_AddCategory(farmerOptionsFrame)

local function setDefaultPosition ()
  local frame = farmerVars.frame
  frame:ClearAllPoints()
  frame:SetPoint('BOTTOM', nil, 'CENTER', 0, 35)
end

local function storePosition (frame)
  local icon = GetItemIcon(114978)

  icon = ' |T' .. icon .. farmerVars.iconOffset
  farmerOptions.anchor = {frame:GetPoint()}
  frame:EnableMouse(false)
  frame:SetMovable(false)
  frame:SetFading(true)
  frame:Clear()
  frame:AddMessage(icon)
  frame:StopMovingOrSizing()
  frame:SetScript('OnDragStart', nil)
  frame:SetScript('OnReceiveDrag', nil)
end

local function moveFrame ()
  local frame = farmerVars.frame
  local icon = GetItemIcon(3334)

  icon = ' |T' .. icon .. farmerVars.iconOffset
  frame:RegisterForDrag('LeftButton')
  frame:SetFading(false)
  frame:Clear()
  frame:AddMessage(icon)
  frame:EnableMouse(true)
  frame:SetMovable(true)
  frame:SetScript('OnDragStart', function (self)
    if (self:IsMovable() == true) then
      self:StartMoving()
    end
  end)
  frame:SetScript('OnReceiveDrag', function (self)
    storePosition(self)
  end)
end

local function displayRarity (edit, rarity)
  local colorHex
  colorHex = farmerVars.rarityColors[rarity][4]
  edit:SetText('|c' .. colorHex .. rarityList[rarity])
  edit:SetCursorPosition(0)
end

local function setFontSize (size)
  farmerVars.font:SetFont('Fonts\\FRIZQT__.TTF', size, 'thickoutline')
  farmerVars.font:SetSpacing(size / 9)
  farmerVars.iconOffset = ':0:0:0:-' .. (size / 6)  .. '|t '
end

local function createCheckButton (name, anchorFrame, xOffset, yOffset, text, anchor, parentAnchor)
  local checkButton = CreateFrame('CheckButton', name .. 'CheckButton', farmerOptionsFrame, 'OptionsCheckButtonTemplate')

  if (anchor == nil) then
    anchor = 'TOPLEFT'
  end
  if (parentAnchor == nil) then
    parentAnchor = 'BOTTOMLEFT'
  end

  checkButton:SetPoint(anchor, anchorFrame, parentAnchor, xOffset, yOffset)
  _G[name .. 'CheckButtonText']:SetText(text)
  checkButtonList[name] = checkButton

  return checkButton
end

local function createButton (name, anchorFrame, xOffset, yOffset, text, anchor, parentAnchor, onClick)
  local button = CreateFrame('Button', name .. 'Button', farmerOptionsFrame, 'OptionsButtonTemplate')

  if (anchor == nil) then
    anchor = 'TOPLEFT'
  end
  if (parentAnchor == nil) then
    parentAnchor = 'BOTTOMLEFT'
  end

  button:SetPoint(anchor, anchorFrame, parentAnchor, xOffset, yOffset)
  button:SetWidth(150)
  button:SetHeight(25)
  button:SetText(text)

  if (onClick ~= nil) then
    button:SetScript('OnClick', onClick)
  end

  return button
end

local function createSlider (name, anchorFrame, xOffset, yOffset, text, min, max, lowText, highText, anchor, parentAnchor, onChange)
  local slider = CreateFrame('Slider', name .. 'Slider', farmerOptionsFrame, 'OptionsSliderTemplate')

  if (anchor == nil) then
    anchor = 'TOPLEFT'
  end
  if (parentAnchor == nil) then
    parentAnchor = 'BOTTOMLEFT'
  end

  slider:SetPoint(anchor, anchorFrame, parentAnchor, xOffset, yOffset)
  slider:SetOrientation('HORIZONTAL')
  slider:SetMinMaxValues(min, max)
  slider:SetValueStep(1)
  slider:SetObeyStepOnDrag(true)
  _G[name .. 'SliderText']:SetText(text)
  _G[name .. 'SliderLow']:SetText(lowText)
  _G[name .. 'SliderHigh']:SetText(highText)
  slider:SetScript('OnValueChanged', function (self, value)
    self.edit:SetText(value)
    self.edit:SetCursorPosition(0)
    if (onChange ~= nil) then
      onChange(self, value)
    end
  end)
  sliderList[name] = slider
  anchor = slider
  edit = CreateFrame('EditBox', name .. 'EditBox', farmerOptionsFrame)
  edit:SetAutoFocus(false)
  edit:Disable()
  edit:SetPoint('TOP', anchor, 'BOTTOM', 0, 0)
  edit:SetFontObject('ChatFontNormal')
  edit:SetHeight(20)
  edit:SetWidth(slider:GetWidth())
  edit:SetTextInsets(8, 8, 0, 0)
  edit:SetJustifyH('CENTER')
  edit:Show()
  -- edit:SetBackdrop(slider:GetBackdrop())
  -- edit:SetBackdropColor(0, 0, 0, 0.8)
  -- edit:SetBackdropBorderColor(1, 1, 1, 1)
  slider.edit = edit

  return slider, edit
end

local function initPanel ()
  local anchor = farmerOptionsFrame

  anchor = createCheckButton('fastLoot', farmerOptionsFrame, 300, -15, 'enable fast autoloot', 'TOPLEFT', 'TOPLEFT')
  anchor = createCheckButton('hideLootToasts', anchor, 0, -5, 'hide loot and item roll toasts')
  anchor = createCheckButton('hideInArena', anchor, 0, -5, 'don\'t display items in arena')
  anchor = createCheckButton('showTotal', anchor, 0, -5, 'show total count for stackable items')
  anchor = createCheckButton('showBags', anchor, 0, -5, 'show bag count for stackable items')

  anchor = createCheckButton('rarity', farmerOptionsFrame, 15, -15, 'show items based on rarity', 'TOPLEFT', 'TOPLEFT')
  _, anchor = createSlider('minimumRarity', anchor, 20, -20, 'minimum rarity', 0, 8, '', '', 'TOPLEFT', 'BOTTOMLEFT', function (self, value)
    displayRarity(self.edit, value)
  end)
  anchor = createCheckButton('special', anchor, -20, -10, 'always show farming items')
  anchor = createCheckButton('reagents', anchor, 0, -5, 'always show reagents')
  anchor = createCheckButton('artifactPower', anchor, 0, -5, 'always show artifact power items')
  anchor = createCheckButton('currency', anchor, 0, -25, 'show currencies')
  anchor = createCheckButton('money', anchor, 0, -5, 'show money')

  anchor = createButton ('move', farmerOptionsFrame, 10, 12, 'move frame', 'BOTTOMLEFT', 'BOTTOMLEFT', function (self)
    moveFrame()
  end)
  createButton ('resetPosition', anchor, 20, 0, 'reset position', 'LEFT', 'RIGHT', function (self)
    setDefaultPosition()
    storePosition(farmerVars.frame)
  end)
  anchor = createSlider('fontSize', anchor, 3, 40, 'font size', 8, 64, '8', '64', 'BOTTOMLEFT', 'TOPLEFT', function (self, value)
    setFontSize(value)
  end)
  createSlider('displayTime', anchor, 23, 0, 'display time', 1, 10, '1', '10', 'LEFT', 'RIGHT', function (self, value)
    farmerVars.frame:SetTimeVisible(value - farmerVars.frame:GetFadeDuration())
  end)
end

local function applyOptions ()
  if (farmerOptions.hideLootToasts == true) then
    AlertFrame:UnregisterEvent('SHOW_LOOT_TOAST')
    AlertFrame:UnregisterEvent('LOOT_ITEM_ROLL_WON')
    AlertFrame:UnregisterEvent('SHOW_LOOT_TOAST_UPGRADE')
  else
    AlertFrame:RegisterEvent('SHOW_LOOT_TOAST')
    AlertFrame:RegisterEvent('LOOT_ITEM_ROLL_WON')
    AlertFrame:RegisterEvent('SHOW_LOOT_TOAST_UPGRADE')
  end

  if (farmerOptions.money == true) then
    farmerVars.moneyStamp = GetMoney()
  end

  setFontSize(farmerOptions.fontSize)
  farmerVars.frame:SetTimeVisible(farmerOptions.displayTime - farmerVars.frame:GetFadeDuration())
  -- farmerVars.frame:SetTimeVisible(farmerOptions.displayTime)
end

local function loadOptions ()
  for k, v in pairs(checkButtonList) do
    v:SetChecked(farmerOptions[k])
  end
  for k, v in pairs(sliderList) do
    v:SetValue(farmerOptions[k])
  end
end

local function saveOptions ()
  for k, v in pairs(checkButtonList) do
    farmerOptions[k] = v:GetChecked()
  end
  for k, v in pairs(sliderList) do
    farmerOptions[k] = v:GetValue()
  end

  applyOptions()
end

farmerOptionsFrame.okay = saveOptions
farmerOptionsFrame.refresh = loadOptions
farmerOptionsFrame.cancel = applyOptions

function checkOption (name, default)
  if (farmerOptions[name] == nil) then
    farmerOptions[name] = default
  end
end

function events:ADDON_LOADED (addon)
  if (addon ~= 'Farmer') then
    return
  end

  initPanel()

  if (farmerOptions == nil) then
    farmerOptions = {}
    farmerOptions.version = currentVersion
  end

  if (farmerOptions.version == nil) then
    print('You seem to have used an old Version of Farmer\nCheck out all the new features in the options!')
  elseif (farmerOptions.version < currentVersion) then
    local version = GetAddOnMetadata(addonName, 'version')
    local text

    text = 'New in ' .. addonName .. ' version ' .. version .. ':\n' ..
           'You can show bag counts for stackable items'
    print(text)
  end
  farmerOptions.version = currentVersion

  checkOption('fastLoot', true)
  checkOption('rarity', true)
  checkOption('minimumRarity', 3)
  checkOption('currency', true)
  checkOption('reagents', true)
  checkOption('artifactPower', true)
  checkOption('money', false)
  checkOption('special', true)
  checkOption('fontSize', 24)
  checkOption('displayTime', 3)
  checkOption('hideLootToasts', false)
  checkOption('hideInArena', true)
  checkOption('showTotal', true)
  checkOption('showBags', false)

  if (farmerOptions.anchor == nil) then
    setDefaultPosition()
  else
    farmerVars.frame:SetPoint(unpack(farmerOptions.anchor))
  end

  applyOptions()
end

function events:PLAYER_LOGIN ()
  local money = GetMoney()

  if (earningStamp == nil) then
    earningStamp = money
  end

  if (farmerOptions.money == true) then
    farmerVars.moneyStamp = money
  end

  LootFrame:SetAlpha(0)
end

local function eventHandler (self, event, ...)
  events[event](self, ...)
end

farmerOptionsFrame:SetScript('OnEvent', eventHandler)

for k, v in pairs(events) do
  farmerOptionsFrame:RegisterEvent(k)
end

--[[
///#############################################################################
/// slash commands
///#############################################################################
--]]

local slashCommands = {}

slashCommands['move'] = moveFrame

function slashCommands:reset ()
  setDefaultPosition()
  storePosition(farmerVars.frame)
end

function slashCommands:gold (param)
  if (param == 'reset') then
    earningStamp = GetMoney()
    print('Money counter was reset')
    return
  end
  local difference = GetMoney() - earningStamp
  local amount = math.abs(difference)
  local text

  text = (GOLD_AMOUNT_TEXTURE .. ' ' ..
          SILVER_AMOUNT_TEXTURE .. ' ' ..
          COPPER_AMOUNT_TEXTURE):format(amount / 10000, 0, 0,
                                        (amount / 100) % 100, 0, 0,
                                        amount % 100, 0, 0)
  if (difference >= 0) then
    print('Money earned this session: ' .. text)
  else
    print('You lost money this session: ' .. text)
  end
end

function slashCommands:default ()
  InterfaceOptionsFrame_Show()
  InterfaceOptionsFrame_OpenToCategory(farmerOptionsFrame)
end

local function slashHandler (input)
  local command = string.match(input, '^([^ ]+)')
  local _, param = string.match(input, '^([^ ]+) ([^ ]*)')
  if (command ~= nil) then
    command = string.lower(command)
  else
    command = 'default'
  end

  if (param ~= nil) then
    param = string.lower(param)
  end

  if (slashCommands[command] ~= nil) then
    slashCommands[command](nil, param)
    return
  end
  print('Farmer: unknown command "' .. input .. '"')
end

SLASH_FARMER1 = '/farmer'
SlashCmdList['FARMER'] = slashHandler
