local ember = CreateFrame("Frame","Ember")
ember:Hide()

local atBank = false -- true while at a bank
local inCombat = false -- true while in combat

-- enums for state of a set
ember.WORN = 1 -- player is wearing the full set
ember.MISSING = 2 -- an item in the set is missing
ember.BANKED = 3 -- at least part of set is in bank
ember.BAGGED = 4 -- at least part of set is in bags (rest worn, if any)

-- equipset locations to never overlay
local locationsToIgnore = {
	[EQUIPMENTFLYOUT_IGNORESLOT_LOCATION] = true,
	[EQUIPMENTFLYOUT_UNIGNORESLOT_LOCATION] = true,
}

-- for quickly determining if the mouse is over an equipment set in the character window
local equipSetButtonLookup = {} -- populated during PLAYER_LOGIN

-- table of overlay buttons indexed by the flyout button, filled during GetFlyoutOverlay()
-- (can't add any properties to flyout buttons or it tains them and bugs swapping in combat)
ember.flyoutOverlayButtons = {}

-- translate inventory slot number to a name for tooltip purposes
ember.slotNames = { "Head", "Neck", "Shoulder", "Shirt", "Chest", "Waist", "Legs", "Feet",
						  "Wrist", "Hands", "Top Finger", "Bottom Finger", "Top Trinket",
						  "Bottom Trinket", "Cloak", "Main Hand", "Off Hand", "Ranged", "Tabard" }


-- free space by set (bank vs bag) and then by bag in that set
-- within each bag it's indexed by slot number, true if the slot is free/available
ember.freeSpace = {
	bank = {[-1]={},[5]={},[6]={},[7]={},[8]={},[9]={},[10]={},[11]={}},
	bags = {[0]={},[1]={},[2]={},[3]={},[4]={}}
}

-- order of bags to place items
ember.bagOrder = { bank={-1,5,6,7,8,9,10,11}, bags={4,3,2,1,0} }
-- hold SHIFT to move stuff in reverse bag order
ember.reverseBagOrder = { bank={-1,11,10,9,8,7,6,5}, bags={0,1,2,3,4} }

--[[ Event Handling ]]

-- event dispatch
ember:SetScript("OnEvent",function(self,event,...)
	ember[event](self,...)
end)
ember:RegisterEvent("PLAYER_LOGIN")

-- stuff to do only once when the player logs in
function ember:PLAYER_LOGIN()
	ember:RegisterEvent("BANKFRAME_OPENED")
	ember:RegisterEvent("BANKFRAME_CLOSED")
	ember:RegisterEvent("PLAYER_REGEN_DISABLED")
	ember:RegisterEvent("PLAYER_REGEN_ENABLED")
	-- when character slot flyout updates, update ember's overlays 
	hooksecurefunc("EquipmentFlyout_UpdateItems",function() ember:UpdateFlyoutOverlays(true) end)
	-- replace equipment set tooltip with ember's
	hooksecurefunc(GameTooltip,"SetEquipmentSet",ember.SetEquipmentSetTooltip)
	-- when equipment set names are colored, recolor set names
	for _,button in pairs(PaperDollEquipmentManagerPane.buttons) do
		hooksecurefunc(button.text,"SetTextColor",ember.RecolorSetName)
		equipSetButtonLookup[button] = true -- while here, note the button for focus lookup
		button:HookScript("OnClick",ember.ShowDoubleClickFrame)
	end
	-- create a Deposit/Withdraw button on the equip panel
	ember:CreateBankButton()
	-- only watch for items moving while equipset pane is open
	PaperDollEquipmentManagerPane:HookScript("OnShow",function(self)
		ember:RegisterEvent("BAG_UPDATE_DELAYED")
		ember:RegisterEvent("PLAYER_EQUIPMENT_CHANGED")
	end)
	-- and stop watching when equipset pane closes
	PaperDollEquipmentManagerPane:HookScript("OnHide",function(self)
		ember:UnregisterEvent("BAG_UPDATE_DELAYED")
		ember:UnregisterEvent("PLAYER_EQUIPMENT_CHANGED")
	end)
end

-- most of what this addon does, it does while the bank is open
function ember:BANKFRAME_OPENED()
	atBank = true
	ember:UpdateFlyoutOverlays()
	-- force refresh of colors since some may be discoverable in bank now
	ember:RefreshSetNameColors()
	-- show bank button if panel is open
	ember:UpdateBankButton()
end

-- close unnecessary stuff while bank is closed
function ember:BANKFRAME_CLOSED()
	atBank = false
	ember:UpdateFlyoutOverlays()
	-- force refresh of colors since some may be missing and in bank now
	-- (fires twice usually but maybe not always, wait for second)
	C_Timer.After(0.1,ember.RefreshSetNameColors)
	-- hide bank button
	ember:UpdateBankButton()
end

-- entering combat, need to hide all overlays; they can't do anything in combat
function ember:PLAYER_REGEN_DISABLED()
	inCombat = true
	ember:UpdateFlyoutOverlays() -- only does work while flyout is open
end

-- leaving combat, can restore overlays
function ember:PLAYER_REGEN_ENABLED()
	inCombat = false
	ember:UpdateFlyoutOverlays() -- only does work while flyout is open
end

-- BAG_UPDATE_DELAYED and PLAYER_EQUIPMENT_CHANGED perform identical behaviors, and
-- can sometimes spam many events at once. To minimize work, anytime either event
-- fires, it will reset a timer and wait 0.25 seconds of no event to do any work.
function ember:BAG_UPDATE_DELAYED()
	ember.timer = 0
	ember:Show()
end
ember.PLAYER_EQUIPMENT_CHANGED = ember.BAG_UPDATE_DELAYED

-- wait a quarter second after BAG_UPDATE_DELAYED or PLAYER_EQUIPMENT_CHANGED to do anything
ember:SetScript("OnUpdate",function(self,elapsed)
	self.timer = self.timer + elapsed
	if self.timer > 0.25 then
		self:Hide()
		-- if an equipment set tooltip is on screen, update it
		local focus = GetMouseFocus() --- it's on screen if mouse is over a set
		if equipSetButtonLookup[focus] then
			ember:SetEquipmentSetTooltip(focus.setID)
		end
		if PaperDollEquipmentManagerPane:IsVisible() then
			ember:UpdateBankButton() -- in case anything moved from/to bank
		end
	end
end)

--[[ Bag Space ]]

-- populates ember.freeSpace with available empty bag,slots
-- this should be called before ember:GetFreeSpace()
function ember:ScanFreeSpace(forBank)
	local free = forBank and ember.freeSpace.bank or ember.freeSpace.bags
	for bag in pairs(free) do
		local numSlots = GetContainerNumSlots(bag)
		-- if this is a special bag (ore, gem, herb, etc bag) pretend it has no room
		if select(2,GetContainerNumFreeSlots(bag))~=0 then
			numSlots = 0
		end
		wipe(free[bag]) -- wipe free slots
		-- go through all bags and mark empty slots as true in freeSpace["bank" or "bags"]
		for slot=1,numSlots do
			if not GetContainerItemLink(bag,slot) then
				free[bag][slot] = true
			end
		end
	end
end

-- returns Pickup'able bag,slot of next free spot in bags (or bank if true)
-- run ember:ScanFreeSpace before running this, especially on multi-item swaps
function ember:GetFreeSpace(forBank)
	local free = forBank and ember.freeSpace.bank or ember.freeSpace.bags
	local orderDirection = IsShiftKeyDown() and "reverseBagOrder" or "bagOrder"
	local order = forBank and ember[orderDirection].bank or ember[orderDirection].bags
	local slot
	for _,bag in pairs(order) do
		for slotCandidate in pairs(free[bag]) do
			-- find the earliest available slot (remember these are unordered tables)
			if not slot or slotCandidate<slot then
				slot = slotCandidate
			end
		end
		if slot then
			free[bag][slot] = nil -- found a space, clear its freeSpace entry 
			return bag,slot -- and return it
		end
	end
end

-- returns a bag,slot that's Pickup'able (safe for normal bags too)
function ember:ConvertBankToPickup(bag,slot)
	if not bag then
		bag = -1 -- the main bank area is bag -1
		slot = slot - 47 -- and offset by 47 (as of 1.1.2 3/17/18; was 39 in prior versions)
	end
	return bag,slot
end

function ember:NotEnoughSpace()
	UIErrorsFrame:AddMessage("Not enough space to perform that swap.",1,.1,.1,1)
end

--[[ Flyout Overlays ]]

-- When the user holds ALT over a character slot, a flyout of equippable items
-- appears out of the slot. This addons adds an overlay over each flyout button to:
-- - Color a border to describe where the item is (green=worn, blue=bank)
-- - Redirect items in the bank to bags, and items on person/in bags to the bank

-- hooksecurefunc for EquipmentFlyout_UpdateItems: updates overlays and border colors
-- buttons only clickable while at the bank and not in combat
-- force is true if it updates regardless whether flyout is open (in the hook itself)
function ember:UpdateFlyoutOverlays(force)
	if force or EquipmentFlyoutFrame:IsVisible() then
		local buttons = EquipmentFlyoutFrame.buttons
		for i=1,min(EquipmentFlyoutFrame.totalItems,EQUIPMENTFLYOUT_ITEMS_PER_PAGE) do
			local button = buttons[i] -- flyout button
			local overlay = ember:GetFlyoutOverlay(button)
			local border = overlay.border
			local location = button.location
			local player,bank,bags,_,slot,bag = EquipmentManager_UnpackLocation(location)
			if inCombat then -- hide overlay while in combat (it will be unclickable)
				border:SetVertexColor(0,0,0,0) 
			elseif player and not bag then -- worn
				border:SetVertexColor(0,0.8,0,1)
			elseif location==EQUIPMENTFLYOUT_PLACEINBAGS_LOCATION then
				border:SetVertexColor(0,0,0,0) -- hide, but keep overlay clickable (send equipped item to bank)
			elseif locationsToIgnore[location] then -- always pass through ignore/unignore clicks
				overlay:EnableMouse(false)
				border:SetVertexColor(0,0,0,0)
			elseif bank then -- item banked
				border:SetVertexColor(0.1,0.65,1,1)
			else -- item in bags
				border:SetVertexColor(0,0,0,0)
			end
		end
	end
end

-- gets the button.emberOverlay for the given flyout button, creating it if needed
-- when a button is fetched, it's made clickable only if at bank and not in combat
function ember:GetFlyoutOverlay(flyoutButton)
	if not ember.flyoutOverlayButtons[flyoutButton] then
		ember.flyoutOverlayButtons[flyoutButton] = CreateFrame("Button",nil,flyoutButton)
		local button = ember.flyoutOverlayButtons[flyoutButton]
		button:SetAllPoints(true)
		button:SetScript("OnEnter",ember.FlyoutButtonOnEnter)
		button:SetScript("OnLeave",ember.FlyoutButtonOnLeave)
		button:SetScript("OnClick",ember.FlyoutButtonOnClick)
		button.border = button:CreateTexture(nil,"ARTWORK")
		button.border:SetAllPoints(true)
		button.border:SetTexture("Interface\\Buttons\\UI-ActionButton-Border")
		button.border:SetTexCoord(0.2,0.8,0.2,0.8)
		button.border:SetBlendMode("ADD")
		button:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square","ADD")
	end
	ember.flyoutOverlayButtons[flyoutButton]:EnableMouse(atBank and not inCombat)
	return ember.flyoutOverlayButtons[flyoutButton]
end

-- pass flyout overlay's OnEnter to parent button
function ember:FlyoutButtonOnEnter()
	self:GetParent():GetScript("OnEnter")(self:GetParent())
end

-- pass flyout overlay's OnEnter to parent button
function ember:FlyoutButtonOnLeave()
	self:GetParent():GetScript("OnLeave")(self:GetParent())
end

-- flyout buttons are only clickable if at the bank; move item to or from bank
function ember:FlyoutButtonOnClick(mouseButton)
	local button = self:GetParent()
	local player,bank,bags,_,slot,bag = EquipmentManager_UnpackLocation(button.location)
	if _DEBUG then
		print("loc:",button.location,", player:",player,", bank:",bank,", bags:",bags,", slot:",slot,", bag:",bag)
		return
	end
	local freeBag,freeSlot
	ClearCursor()
	if button.location==EQUIPMENTFLYOUT_PLACEINBAGS_LOCATION then -- clicked "Place in Bags" button
		bag = nil
		slot = button.id -- pretend this is on person at slot parent.id
	end
	if player and not bag then -- on person, send to bank
		ember:ScanFreeSpace(true) -- find a free slot in the bank
		freeBag,freeSlot = ember:GetFreeSpace(true)
		if freeBag then
			PickupInventoryItem(slot)
			PickupContainerItem(freeBag,freeSlot)
		end
	else -- in bank or bags, bank is true if item is in bank
		ember:ScanFreeSpace(not bank)
		bag,slot = ember:ConvertBankToPickup(bag,slot)
		freeBag,freeSlot = ember:GetFreeSpace(not bank)
		if freeBag then
			PickupContainerItem(bag,slot)
			PickupContainerItem(freeBag,freeSlot)
		end
	end
	if not freeBag then
		ember:NotEnoughSpace()
	end
end

--[[ Set Tooltip ]]

-- replacement for GameTooltip:SetEquipmentSet(setID)
-- displays the items in the set and whether they're worn, in bags, or in bank
function ember:SetEquipmentSetTooltip(setID)
	if setID then
		local locations = C_EquipmentSet.GetItemLocations(setID)
		local setName = C_EquipmentSet.GetEquipmentSetInfo(setID)
		GameTooltip:ClearLines()
		GameTooltip:AddLine("|cFFCCCCCCSet: |cFFF8F8F8"..setName)
		local slotName,itemID,itemName,itemIcon,loc = ""
		local red,green,blue,white = "|cFFFF2020","|cFF20FF20","|cFF40C0FF","|cFFFFD200"
		for i=1,19 do
			loc = locations[i]
			if loc and loc~=1 and i~=18 then
				slotName = "|cFFCCCCCC"..ember.slotNames[i]..": "
				local player, bank, bags, _, slot, bag = EquipmentManager_UnpackLocation(loc)
				if loc==EQUIPMENT_SET_EMPTY_SLOT then
					GameTooltip:AddLine("\124TInterface\\PaperDoll\\UI-Backpack-EmptySlot:16\124t "..slotName..(GetInventoryItemLink("player",i) and white or green).."(empty)")
				elseif loc==EQUIPMENT_SET_ITEM_MISSING then
					GameTooltip:AddLine("\124TInterface\\RAIDFRAME\\ReadyCheck-NotReady:16\124t "..slotName..red.."(missing)")
				else
					itemID,itemName,itemIcon = EquipmentManager_GetItemInfoByLocation(loc)
					itemName = itemName or ""
					itemIcon = itemIcon and ("\124T"..itemIcon..":16\124t ") or " "
					if bank then
						GameTooltip:AddLine(itemIcon..slotName..blue..itemName)
					elseif bags then
						GameTooltip:AddLine(itemIcon..slotName..white..itemName)
					else
						GameTooltip:AddLine(itemIcon..slotName..green..itemName)
					end
				end
			end
		end
		GameTooltip:Show()
	end
end

--[[ EquipmentSet List Recoloring ]]

local recoloring = false -- semaphore for fontString:SetTextColor()

-- each PaperDollEquipmentManagerPane.buttons[].text FontString has RecolorSetName
-- in a hooksecurefunc to avoid tainting scrollFrame's .update
function ember:RecolorSetName()
	if not recoloring then
		recoloring = true -- to prevent the SetTextColor below from triggering another call

		local setID = self:GetParent().setID
		if setID then
			local status = ember:GetSetStatus(setID)
			if status==ember.WORN then
				self:SetTextColor(0.125,1,0.125,1)
			elseif status==ember.MISSING or (status==ember.BANKED and not atBank) then
				self:SetTextColor(1,0.125,0.125,1)
			elseif status==ember.BANKED then
				self:SetTextColor(0.25,0.75,1)
			end
		end

		recoloring = false
	end
end

-- opening bank doesn't do an update of the scrollFrame; this will recolor each list button
function ember:RefreshSetNameColors()
	if PaperDollEquipmentManagerPane:IsVisible() then
		for _,button in pairs(PaperDollEquipmentManagerPane.buttons) do
			local r,g,b,a = button.text:GetTextColor()
			ember.RecolorSetName(button.text,r,g,b,a) -- note the . notation (need button.text to be self)
		end
	end
end

-- returns a numerical status of a set
-- returns ember.MISSING if any piece is unavailable regardless rest
-- set ignoreMissing to see if any items are banked (bank set swaps)
function ember:GetSetStatus(setID,ignoreMissing)
	if setID then
		local _,_,_,isEquipped,numItems,numEquipped,numInInventory,numLost = C_EquipmentSet.GetEquipmentSetInfo(setID)
		--local worn,valid,_,_,missing = select(3,GetEquipmentSetInfoByName(setName))
		if numItems==0 then
			return nil
		elseif isEquipped then
			return ember.WORN
		elseif numLost>0 and not ignoreMissing then
			return ember.MISSING
		elseif not atBank then
			return ember.BAGGED
		else -- it's likely in the bank'
			local locations = C_EquipmentSet.GetItemLocations(setID)
			for _,location in pairs(locations) do
				if location and select(2,EquipmentManager_UnpackLocation(location)) then
					return ember.BANKED
				end
			end
			return ember.BAGGED
		end
	end
end

--[[ EquipmentSet List Overlay ]]

-- We want double-clicks of sets to consistently bank, but we can't get involved in
-- clicking the equip pane or equips taint. Solution: let first clicks go through and
-- make a clickable overlay appear for a second to intercept what would be a double click.

-- HookScript of list button's OnClicks; note the dot notation (parent is the list button/self)
function ember.ShowDoubleClickFrame(parent)
	if atBank then -- doubleClickFrame only appears at the bank (elsewhere don't alter behavior)
		local frame = ember:GetDoubleClickFrame()
		frame:SetParent(parent)
		frame:SetAllPoints(true)
		frame.timer = 0
		frame:Show()
		-- a set is being selected, update bank button to show Deposit or Withdraw
		ember:UpdateBankButton()
	end
end

-- this is called in ShowDoubleClickFrame to create the intercepting frame if it's not made already
function ember:GetDoubleClickFrame()
	if not ember.doubleClickFrame then
		ember.doubleClickFrame = CreateFrame("Button",nil,UIParent)
		local frame = ember.doubleClickFrame
		frame:Hide()
		frame:SetScript("OnClick",ember.DoubleClickFrameOnClick)
		frame:SetScript("OnLeave",frame.Hide)
		frame:SetScript("OnUpdate",ember.DoubleClickFrameOnUpdate)
	end
	return ember.doubleClickFrame
end

-- doubleClickFrame only stays up for a second before hiding itself
function ember:DoubleClickFrameOnUpdate(elapsed)
	self.timer = self.timer + elapsed
	if self.timer > 1 then
		self:Hide()
	end
end

-- when the doubleClickFrame is clicked, it's simulating a double click on a list button;
-- do the bank swap stuff
function ember:DoubleClickFrameOnClick()
	ember:EquipBankSet() -- will pull setID from selected setID
	self:Hide()
end

--[[ Sets At The Bank]]

-- while at the bank:
-- "equipping" a set with any banked items will pull the banked items to bags
-- "equipping" a set with no banked items will push the items to the bank
--   EXCEPT! when other sets contain the same item, unless Ctrl is held
--   (otherwise part of a set in bags would get pushed to bank)

-- this is called from the doubleClickFrame and the equip panel button
function ember:EquipBankSet()
	local setID = PaperDollEquipmentManagerPane.selectedSetID
	local status = ember:GetSetStatus(setID,true)
	if status==ember.BANKED then
		ember:PullFromBankToBags(setID)
	elseif status==ember.BAGGED or status==ember.WORN then
		ember:PushFromBagsToBank(setID)
	end
end

-- when pulling from the bank we don't care how many items are shared among sets;
-- if it's in the bank, bring it to bags
function ember:PullFromBankToBags(setID)
	ember:ScanFreeSpace() -- looking for free bag slots
	for _,location in pairs(C_EquipmentSet.GetItemLocations(setID)) do
		local player,bank,bags,_,slot,bag = EquipmentManager_UnpackLocation(location)
		if bank then
			bag,slot = ember:ConvertBankToPickup(bag,slot) -- convert to Pickup'able if needed
			local freeBag,freeSlot = ember:GetFreeSpace()
			if freeBag then
				PickupContainerItem(bag,slot)
				PickupContainerItem(freeBag,freeSlot)
			else
				return ember:NotEnoughSpace() -- stop pulling items from bank
			end
		end
	end
end

-- when pushing to bank, we need to avoid banking an item shared by another set
function ember:PushFromBagsToBank(setID)
	ember:ScanFreeSpace(true) -- looking for free bank slots
	for _,location in pairs(C_EquipmentSet.GetItemLocations(setID)) do
		if location>1 and ember:GetNumSetsWithLocation(location)==1 then
			local player,bank,bags,_,slot,bag = EquipmentManager_UnpackLocation(location)
			if player and not bank and not bag then -- if worn
				local freeBag,freeSlot = ember:GetFreeSpace(true)
				if freeBag then
					PickupInventoryItem(slot)
					PickupContainerItem(freeBag,freeSlot)
				else
					return ember:NotEnoughSpace() -- stop pushing items to bank
				end
			elseif not bank and bag and slot then -- if in bags
				local freeBag,freeSlot = ember:GetFreeSpace(true)
				if freeBag then
					PickupContainerItem(bag,slot)
					PickupContainerItem(freeBag,freeSlot)
				else
					return ember:NotEnoughSpace()
				end
			end
		end
	end
end

-- takes an equipset location (search) and returns the number of worn or bagged
-- sets that contain this item
-- if CTRL key is down, returns 1 (don't care how many and must be 1 to be searched)
function ember:GetNumSetsWithLocation(search)
	if IsControlKeyDown() then
		return 1 -- if CTRL is down, bank regardless of how many sets share this
	end
	local total = 0
	for _,setID in pairs(C_EquipmentSet.GetEquipmentSetIDs()) do
		local status = ember:GetSetStatus(setID,true)
		if status==ember.BAGGED or status==ember.WORN then -- only care about sets worn/in bags
			for _,location in pairs(C_EquipmentSet.GetItemLocations(setID)) do
				if location==search then
					total = total + 1
					break
				end
			end
		end
	end
	return total
end

--[[ Bank Button ]]

-- The default button PaperDollEquipmentManagerPaneEquipSet is a red panel button
-- that sits at the top of the set list to equip a set. Changing this button's behavior
-- would guarantee taint, so this addon creates its own button and displays it ontop
-- of the default while at the bank.

function ember:CreateBankButton()
	ember.bankButton = CreateFrame("Button",nil,PaperDollEquipmentManagerPane,"UIPanelButtonTemplate")
	local button = ember.bankButton
	button:SetPoint("TOPLEFT",PaperDollEquipmentManagerPaneEquipSet)
	button:SetPoint("BOTTOMRIGHT",PaperDollEquipmentManagerPaneEquipSet)
	button:SetScript("OnClick",ember.EquipBankSet) -- defined up in Push/Pull code
	PaperDollEquipmentManagerPane:HookScript("OnShow",ember.UpdateBankButton)
end

-- can be called anytime, updates the state of the bank button; disabled if no set selected,
-- Withdraw if banked set selected, Deposit otherwise
function ember:UpdateBankButton()
	local button = ember.bankButton
	if not atBank then
		button:Hide() -- not at bank, don't care
	elseif PaperDollEquipmentManagerPane:IsVisible() then
		-- at bank, show the button and make it higher than default Equip button
		button:SetFrameLevel(PaperDollEquipmentManagerPaneEquipSet:GetFrameLevel()+2)
		button:Show()
		-- update its status depending on what set is selected
		local setID = PaperDollEquipmentManagerPane.selectedSetID
		button:SetEnabled(setID and true)
		-- change button to Withdraw if a banked set, Deposit otherwise
		button:SetText(ember:GetSetStatus(setID)==ember.BANKED and "Withdraw" or "Deposit")
	end
end
