--------------------------------------------------------------------------
--DurabilityStatus.lua
--------------------------------------------------------------------------
--[[
	DurabilityStatus version 4.1.4
		Lets you know your current durability percentage and cost

	Original author: Derkyle http://www.manaflux.com
	Formally Maintained by Yarko
	Currently maintained by: Meeanda
	]]--

DurabilityStatus = {};

DSValues_Armor = { value=0, max=0, cost=0 };
DSValues_Inven = { value=0, max=0, cost=0 };
DSResetLocations = false;
DSIsMoving = false;

local Old_InterfaceOptionsFrameOkay_OnClick;

local BrokenItem = false;
local tmrCount;
local WarningDisplayed = false;

local curX, curY, startX, startY;

local DurStatus_Slots = {
	"Head",
	"Shoulder",
	"Chest",
	"Wrist",
	"Hands",
	"Waist",
	"Legs",
	"Feet",
	"MainHand",
	"SecondaryHand"
}

-- Define a new money type for DS frames - static but show lower denominations when zero
MoneyTypeInfo["DURABILITYSTATUS"] = {
	UpdateFunc = function(self)
		return self.staticMoney;
	end,

	collapse = 1,
	showSmallerCoins = "Backpack",
};

-- Durability status warning dialog
StaticPopupDialogs["DURABILITYSTATUS_WARNING"] = {
	text = "",
	button1 = OKAY,
	timeout = 0,
	showAlert = 1,
	whileDead = 1,
	exclusive = 1,
	hideOnEscape = 1
};


---------------------------------------------------
-- DurabilityStatus.OnLoad
---------------------------------------------------
function DurabilityStatus.OnLoad(self)
	--DEFAULT_CHAT_FRAME:AddMessage("|cffFF4500Durability Status|cff00ff00 v6.0 loaded successfully.");
	local ver = GetAddOnMetadata("DurabilityStatus","Version") or '1.0'
	DEFAULT_CHAT_FRAME:AddMessage(string.format("|cffFF4500%s|r [v|cff00ff00%s|r] Loaded", "DurabilityStatus", ver or "1.0"));
	self:RegisterEvent("VARIABLES_LOADED");
	self:RegisterEvent("PLAYER_LOGIN");
	--bagupdate
	self:RegisterEvent("PLAYER_DEAD");--bagupdate
	self:RegisterEvent("PLAYER_UNGHOST"); --bagupdate
	self:RegisterEvent("UPDATE_INVENTORY_ALERTS"); --update inventory
	self:RegisterEvent("PLAYER_LEAVING_WORLD");
	self:RegisterEvent("PLAYER_ENTERING_WORLD");
	self:RegisterEvent("UPDATE_INVENTORY_DURABILITY");
	
	-- Now updates the durability percentage when the player leaves combat and when the player
	-- spends money at a vendor who can repair.
	self:RegisterEvent("PLAYER_REGEN_ENABLED"); --update inventory
	self:RegisterEvent("PLAYER_MONEY"); --update inventory

	-- Now updates the durability percentage when the player switches gear
	self:RegisterEvent("UNIT_INVENTORY_CHANGED"); --update inventory

	--merchant
	self:RegisterEvent("MERCHANT_SHOW");
	self:RegisterEvent("MERCHANT_CLOSED");

end



---------------------------------------------------
-- DurabilityStatus.OnEvent
---------------------------------------------------
function DurabilityStatus.OnEvent(self, event, ...)
	local arg1 = ...;
	
	--DEFAULT_CHAT_FRAME:AddMessage(event.." - arg1: "..(arg1 or "nil")..", arg2: "..(arg2 or "nil")..", arg3: "
	--	..(arg3 or "nil")..", arg4: "..(arg4 or "nil")..", arg5: "..(arg5 or "nil"));
	
	if (event == "VARIABLES_LOADED") then
		
		if (not DurStatusOpt) then 
			DurabilityStatus.SetDefaults();
		end
		
		if(DurStatusOpt.showWindow ) then
			DS_Window:Show(); 
		else 
			DS_Window:Hide(); 
		end

		if (DurStatusOpt.showInChar) then
			DS_CharFrame:Show();
		else
			DS_CharFrame:Hide();
		end
		
		if (not DurStatusOpt.charX) then
			DurStatusOpt.charX = 0;
			DurStatusOpt.charY = 0;
		end
		
		DurabilityStatus.SetPosition();
		
		--DS_Window:SetUserPlaced(true); not necessary
		DurabilityStatus.BagUpdate();
		
		Old_InterfaceOptionsFrameOkay_OnClick = InterfaceOptionsFrameOkay_OnClick;
		InterfaceOptionsFrameOkay_OnClick = DurabilityStatus.InterfaceOptionsFrameOkayOnClick;
		
		DurabilityStatus_OptionsPanelWarnLevel:SetNumeric(true);
		
		InterfaceOptions_AddCategory(DurabilityStatus_OptionsPanel);
		
		-- Hook into merchant guild bank repair button so that DurabilityStatus_BagUpdate will be called after clicking
		MerchantGuildBankRepairButton:HookScript("OnClick", function() DurabilityStatus_Core.bagupdate = .5; end);
	end

	-- Handle additional events for player leaving combat and spending at repair vendor
	if (event == "PLAYER_ENTERING_WORLD" or event == "UPDATE_INVENTORY_ALERTS" or event == "PLAYER_REGEN_ENABLED" 
			or event == "PLAYER_UNGHOST" or event == "PLAYER_DEAD" or event == "UPDATE_INVENTORY_DURABILITY" 
			or (CanMerchantRepair() and event == "PLAYER_MONEY") 
			or (event == "UNIT_INVENTORY_CHANGED" and arg1 == "player")) then

		-- Update quickly if at a merchant, and update a bit less quickly when leaving combat or 
		-- spending money at a repair vendor.
		if ((CanMerchantRepair() and DSValues_Inven.cost + DSValues_Armor.cost > 0) or event == "PLAYER_REGEN_ENABLED" 
				or event == "UNIT_INVENTORY_CHANGED" or event == "UPDATE_INVENTORY_DURABILITY") then
			if (event == "PLAYER_MONEY" or event == "PLAYER_REGEN_ENABLED" or event == "UNIT_INVENTORY_CHANGED") then
				DurabilityStatus_Core.bagupdate = .5;
			else
				DurabilityStatus_Core.bagupdate = 0;
			end
		else
			DurabilityStatus_Core.bagupdate = .5;
		end

	end

	if (event == "PLAYER_ENTERING_WORLD") then
		self:RegisterEvent("UPDATE_INVENTORY_ALERTS");
		return;
	end

	if (event == "PLAYER_LEAVING_WORLD") then
		self:UnregisterEvent("UPDATE_INVENTORY_ALERTS");
		return;
	end

	if(event == "MERCHANT_SHOW") then

		if (DurStatusOpt.showMerchant) then

			DurabilityStatus_Core.bagupdate = 0;
			DurabilityStatus.ShowMerchantRepair();

		end

	elseif(event == "MERCHANT_CLOSED") then

		-- Update the durability percent after leaving vendor
		DurabilityStatus_Core.bagupdate = .5;

		DS_Window_Repair:Hide();
	end

end


---------------------------------------------------
-- DurabilityStatus.SetDefaults
---------------------------------------------------
function DurabilityStatus.SetDefaults()
	DurStatusOpt =
		{
			showMerchant = true,
			showReport = true,
			showWindow = true,
			showAverage = false,
			showInChar = true,
			showWarning = false,
			warnLevel = 30,
			charX = 0;
			charY = 0;
		}; 
end


---------------------------------------------------
-- DurabilityStatus.OptionsDefault
---------------------------------------------------
function DurabilityStatus.OptionsDefault()
	DurabilityStatus.SetDefaults();
	DSResetLocations = true;
	DurabilityStatus.UpdateInterface();
end


---------------------------------------------------
-- DurabilityStatus.OptionsRefresh
---------------------------------------------------
function DurabilityStatus.OptionsRefresh(self)
	DurabilityStatus_OptionsPanelMerchant:SetChecked(DurStatusOpt.showMerchant);
	DurabilityStatus_OptionsPanelShow:SetChecked(DurStatusOpt.showInChar);
	DurabilityStatus_OptionsPanelWindow:SetChecked(DurStatusOpt.showWindow);
	DurabilityStatus_OptionsPanelReport:SetChecked(DurStatusOpt.showReport);
	DurabilityStatus_OptionsPanelAverage:SetChecked(DurStatusOpt.showAverage);
	DurabilityStatus_OptionsPanelWarning:SetChecked(DurStatusOpt.showWarning);
	DurabilityStatus_OptionsPanelWarnLevel:SetNumber(DurStatusOpt.warnLevel);
	DurabilityStatus_OptionsPanelWarnLevel:SetCursorPosition(0);
end


---------------------------------------------------
-- DurabilityStatus.InterfaceOptionsFrameOkayOnClick
---------------------------------------------------
function DurabilityStatus.InterfaceOptionsFrameOkayOnClick(self, button, apply)
	-- Check for valid entries
	local level = DurabilityStatus_OptionsPanelWarnLevel:GetNumber();

	if (level < 0 or level > 99) then
		StaticPopupDialogs["DURABILITYSTATUS_WARNING"].text = DURABILITYSTATUS_CONFIG_WARNLEVEL_WARNING;
		StaticPopup_Show("DURABILITYSTATUS_WARNING");
		DurabilityStatus_OptionsPanelWarnLevel:SetFocus();
		return;
	end
	
	Old_InterfaceOptionsFrameOkay_OnClick(self, button, apply);
end


---------------------------------------------------
-- DurabilityStatus.OptionsOkay
---------------------------------------------------
function DurabilityStatus.OptionsOkay()
	DurStatusOpt.showMerchant = ((DurabilityStatus_OptionsPanelMerchant:GetChecked() and true) or false);
	DurStatusOpt.showInChar = ((DurabilityStatus_OptionsPanelShow:GetChecked() and true) or false);
	DurStatusOpt.showWindow = ((DurabilityStatus_OptionsPanelWindow:GetChecked() and true) or false);
	DurStatusOpt.showReport = ((DurabilityStatus_OptionsPanelReport:GetChecked() and true) or false);
	DurStatusOpt.showAverage = ((DurabilityStatus_OptionsPanelAverage:GetChecked() and true) or false);
	DurStatusOpt.showWarning = ((DurabilityStatus_OptionsPanelWarning:GetChecked() and true) or false);
	DurStatusOpt.warnLevel = DurabilityStatus_OptionsPanelWarnLevel:GetNumber();
	
	DurabilityStatus.UpdateInterface();
end


---------------------------------------------------
-- DurabilityStatus.UpdateInterface
---------------------------------------------------
function DurabilityStatus.UpdateInterface()
	if (DurStatusOpt.showInChar) then
		DS_CharFrame:Show();
	else
		DS_CharFrame:Hide();
	end
	
	if (DurStatusOpt.showWindow) then
		DS_Window:Show();
	else
		DS_Window:Hide();
	end;
	
	if (DSResetLocations) then
		DurabilityStatus.ResetFramePositions();
		DSResetLocations = false;
	end
	
	DurabilityStatus.BagUpdate();
end


---------------------------------------------------
-- DurabilityStatus.ResetFramePositions
---------------------------------------------------
function DurabilityStatus.ResetFramePositions()
	DS_Window:ClearAllPoints();
	DS_Window:SetPoint("TOPRIGHT", "MinimapCluster", "BOTTOMLEFT", 50, 25);

	DS_Window_Repair:ClearAllPoints();
	DS_Window_Repair:SetPoint("CENTER", "UIParent", "CENTER", 0, 0);

	DurStatusOpt.charX = 0;
	DurStatusOpt.charY = 0;
	
	--DurabilityStatus.SetPosition();
end


---------------------------------------------------
-- DurabilityStatus.GetPosition
---------------------------------------------------
function DurabilityStatus.GetPosition(flag)
	if (flag == 1) then
		_, _, _, curX, curY = DS_CharFrame:GetPoint();
	elseif (flag == 2) then
		_, _, _, startX, startY = DS_CharFrame:GetPoint();
	else
		local endX, endY;
		
		_, _, _, endX, endY = DS_CharFrame:GetPoint();
		
		DurStatusOpt.charX = curX + (endX - startX);
		DurStatusOpt.charY = curY + (endY - startY);
	end
end


---------------------------------------------------
-- DurabilityStatus.SetPosition
---------------------------------------------------
function DurabilityStatus.SetPosition()
	DS_CharFrame:ClearAllPoints();
	DS_CharFrame:SetPoint("TOPRIGHT", "CharacterModelFrame", "TOPRIGHT", DurStatusOpt.charX, DurStatusOpt.charY);
end


---------------------------------------------------
-- DurabilityStatus.Command_Toggle
---------------------------------------------------
function DurabilityStatus.Command_Toggle(args, switch, func)
	local newValue;
	if (string.len(args) == 0) then
		newValue = not DurStatusOpt[switch];
	elseif( string.lower(args) == "on" ) then
		newValue = true;
	elseif( string.lower(args) == "off" ) then
		newValue = false;
	else
		DEFAULT_CHAT_FRAME:AddMessage(DURABILITYSTATUS_INVALID_ARGUMENT);
		return;
	end
	DurStatusOpt[switch] = newValue;

	if ( func ) then
		func(DurStatusOpt[switch]);
	end
end



---------------------------------------------------
-- DurabilityStatus.OnUpdate
---------------------------------------------------
function DurabilityStatus.OnUpdate(self, elapsed)

	--BAG_UPDATE
	if(DurabilityStatus_Core.bagupdate) then

		--update the lapse
		DurabilityStatus_Core.bagupdate = DurabilityStatus_Core.bagupdate - elapsed;

		--check if the timer ran out
		if ( DurabilityStatus_Core.bagupdate <= 0 ) then

			DurabilityStatus.BagUpdate();
			DurabilityStatus_Core.bagupdate = nil;

		end

	end

end



---------------------------------------------------
-- DurabilityStatus.BagUpdate
---------------------------------------------------
function DurabilityStatus.BagUpdate()
	--DEFAULT_CHAT_FRAME:AddMessage("DurabilityStatus.BagUpdate()");
	
	BrokenItem = false;
	
	--inventory
	DSValues_Inven.value = 0;
	DSValues_Inven.max = 0;
	DSValues_Inven.cost = 0;


	for bag = 0, 4 do
		for slot = 1, GetContainerNumSlots(bag) do

			local sPercent, value, max, cost = DurabilityStatus.GetStatus(slot, bag);

			if (cost ~= nil) then
				DSValues_Inven.value = DSValues_Inven.value + value;
				DSValues_Inven.max = DSValues_Inven.max + max;
				DSValues_Inven.cost = DSValues_Inven.cost + cost;
			end
		end
	end


	--ARMOR
	DSValues_Armor.value = 0;
	DSValues_Armor.max = 0;
	DSValues_Armor.cost = 0;

	for i = 1, #DurStatus_Slots do

		local sPercent, value, max, cost = DurabilityStatus.GetStatus(i);

		if (cost ~= nil) then
			if DurStatusOpt.showAverage then
				DSValues_Armor.value = DSValues_Armor.value + value;
				DSValues_Armor.max = DSValues_Armor.max + max;
			else
				local current = DurabilityStatus.GetStatusPercent(value, max)
				local minimum = DurabilityStatus.GetStatusPercent(DSValues_Armor.value, DSValues_Armor.max)
				if (current < minimum) or (DSValues_Armor.max == 0) then
					DSValues_Armor.value = value;
					DSValues_Armor.max = max;
				end
			end
			DSValues_Armor.cost = DSValues_Armor.cost + cost;
		end

	end


	--update the percentage
	DurabilityStatus.UpdatePercent();

	--update the merchant window
	if(DS_Window_Repair:IsVisible()) then
		DurabilityStatus.ShowMerchantRepair();
		
		-- Close the DS repair window if all repairs have been made
		if (DSValues_Inven.cost + DSValues_Armor.cost <= 0) then
		   DS_Window_Repair:Hide();
		end
	end
	
	-- Disable the merchant repair buttons if all repairs have been made
	if (CanMerchantRepair() and (MerchantRepairAllButton:IsEnabled() or MerchantGuildBankRepairButton:IsEnabled()) 
	      and DSValues_Inven.cost + DSValues_Armor.cost <= 0) then
	   MerchantFrame_UpdateCanRepairAll();
	   MerchantFrame_UpdateGuildBankRepair();
	end
end


---------------------------------------------------
-- DurabilityStatus.GetStatus
---------------------------------------------------
function DurabilityStatus.GetStatus(index, bag)
	local val = 0;
	local max = 0;
	local cost = 0;
	local hasItem, repairCost

	DSCHKTT:SetOwner(WorldFrame, "ANCHOR_NONE");

	if (bag) then
		local _, lRepairCost = DSCHKTT:SetBagItem(bag, index);

		repairCost = lRepairCost;
		hasItem = 1;
	else
		local slotName = DurStatus_Slots[index] .. "Slot";

		local id = GetInventorySlotInfo(slotName);
		local lHasItem, _, lRepairCost = DSCHKTT:SetInventoryItem("player", id);
		hasItem = lHasItem;
		repairCost = lRepairCost;
	end

	if (hasItem) then
		if (repairCost) then
			cost = repairCost;
		end

		for i = 1, 30 do
			local field = _G["DSCHKTTTextLeft" .. i];
			if (field ~= nil) then
				local text = field:GetText();
				if (text) then
					-- find durability
					local _, _, f_val, f_max = string.find(text, DURABILITYSTATUS_TEXTCHK);

					if (f_val) then
						val = tonumber(f_val);
						max = tonumber(f_max);
					end
				end
			end

		end

	end
	
	-- Set that an item is broken
	if (cost and cost > 0 and val <= 0) then
		BrokenItem = true;
	end

	DSCHKTT:Hide();
	return DurabilityStatus.GetStatusPercent(val, max), val, max, cost;

end


---------------------------------------------------
-- DurabilityStatus.GetStatusPercent
---------------------------------------------------
function DurabilityStatus.GetStatusPercent(val, max)
	if (max > 0) then
		return (val / max);
	end
	return 1.0;
end




---------------------------------------------------
-- DurabilityStatus.UpdatePercent
---------------------------------------------------
function DurabilityStatus.UpdatePercent()
	local green = GREEN_FONT_COLOR;
	local yellow = NORMAL_FONT_COLOR;
	local red = RED_FONT_COLOR;

	local percent = DurabilityStatus.GetStatusPercent(DSValues_Armor.value, DSValues_Armor.max);

	local color = {};

	if (percent == 1.0) then
		color = green;
	elseif (percent == 0.5) then
		color = yellow;
	elseif (percent == 0.0) then
		color = red;
	elseif (percent > 0.5) then
		local pct = (1.0 - percent) * 2;
		color.r =(yellow.r - green.r)*pct + green.r;
		color.g = (yellow.g - green.g)*pct + green.g;
		color.b = (yellow.b - green.b)*pct + green.b;
	elseif (percent < 0.5) then
		local pct = (0.5 - percent) * 2;
		color.r = (red.r - yellow.r)*pct + yellow.r;
		color.g = (red.g - yellow.g)*pct + yellow.g;
		color.b = (red.b - yellow.b)*pct + yellow.b;
	end

	--make sure we don't go overboard
	if( ceil(percent*100) > 100) then
		percent = 100;
	else 
		percent = ceil(percent*100);
	end
	
	local brokenChar = ((BrokenItem and (RED_FONT_COLOR_CODE.."!")) or "");

	DS_WindowText:SetText(percent.."%"..brokenChar);
	DS_WindowText:SetTextColor(color.r, color.g, color.b);
	
	-- Update character frame as well
	DS_CharFrameText:SetText(percent.."%"..brokenChar);
	DS_CharFrameText:SetTextColor(color.r, color.g, color.b);
	
	-- Display warning that percentage has gone below warning level
	if (DurStatusOpt.showWarning) then
		if (not WarningDisplayed and percent <= DurStatusOpt.warnLevel) then
			StaticPopupDialogs["DURABILITYSTATUS_WARNING"].text = format(DURABILITYSTATUS_WARNING, percent);
			StaticPopup_Show("DURABILITYSTATUS_WARNING");
			WarningDisplayed = true;
		end
		
		if (WarningDisplayed and percent > DurStatusOpt.warnLevel) then
			WarningDisplayed = false;
		end
	end
end


---------------------------------------------------
-- DurabilityStatus.ShowTooltip
---------------------------------------------------
function DurabilityStatus.ShowTooltip(self)
	local width;
	local maxWidth = 0;
	
	if (self:GetName() == "DS_CharFrame") then
		GameTooltip:SetOwner(self, "ANCHOR_TOPLEFT");
	else
		GameTooltip_SetDefaultAnchor(GameTooltip, UIParent);
	end
	
	GameTooltip:AddLine(HIGHLIGHT_FONT_COLOR_CODE..DURABILITYSTATUS_CONFIG_HEADER);
	
	if (BrokenItem) then
		GameTooltip:AddLine(RED_FONT_COLOR_CODE..DURABILITYSTATUS_BROKEN);
	end
	
	if (DSValues_Armor.cost > 0) then
		SetTooltipMoney(GameTooltip, DSValues_Armor.cost, "DURABILITYSTATUS");
		_G["GameTooltipTextLeft"..GameTooltip:NumLines()]:SetText(NORMAL_FONT_COLOR_CODE
			..format(DURABILITYSTATUS_LABEL, DURABILITYSTATUS_ARMOR));
		maxWidth = _G["GameTooltipTextLeft"..GameTooltip:NumLines()]:GetWidth() 
			+ _G["GameTooltipMoneyFrame"..GameTooltip.shownMoneyFrames]:GetWidth();
	end
		
	if (DSValues_Inven.cost > 0) then
		SetTooltipMoney(GameTooltip, DSValues_Inven.cost, "DURABILITYSTATUS");
		_G["GameTooltipTextLeft"..GameTooltip:NumLines()]:SetText(NORMAL_FONT_COLOR_CODE
			..format(DURABILITYSTATUS_LABEL, DURABILITYSTATUS_INVEN));
		width = _G["GameTooltipTextLeft"..GameTooltip:NumLines()]:GetWidth() 
			+ _G["GameTooltipMoneyFrame"..GameTooltip.shownMoneyFrames]:GetWidth();
		
		if (width > maxWidth) then
			maxWidth = width;
		end
	end
	
	if (DSValues_Inven.cost > 0 and DSValues_Armor.cost > 0) then
		SetTooltipMoney(GameTooltip, DSValues_Inven.cost + DSValues_Armor.cost, "DURABILITYSTATUS");
		_G["GameTooltipTextLeft"..GameTooltip:NumLines()]:SetText(NORMAL_FONT_COLOR_CODE
			..format(DURABILITYSTATUS_LABEL, DURABILITYSTATUS_TOTAL));
		width = _G["GameTooltipTextLeft"..GameTooltip:NumLines()]:GetWidth() 
			+ _G["GameTooltipMoneyFrame"..GameTooltip.shownMoneyFrames]:GetWidth();
		
		if (width > maxWidth) then
			maxWidth = width;
		end
	end
	
	if (maxWidth > 0) then
		-- Adjust tooltip frame size to fit max moneyframe width
		GameTooltip:SetMinimumWidth(maxWidth);
		
		local textIndex = GameTooltip:NumLines() - GameTooltip.shownMoneyFrames;
		
		-- Relocate the money frames so that they are right-aligned in the tooltip
		for t = 1, GameTooltip.shownMoneyFrames do
			local frameName = "GameTooltipMoneyFrame"..t;
			local frame = _G[frameName];
			
			--frame:ClearAllPoints();		
			frame:SetPoint("LEFT", _G["GameTooltipTextLeft"..(t + textIndex)], "LEFT", 
				(maxWidth + 12) - frame:GetWidth(), 0);
		end
	end
	
	if (DSValues_Inven.cost == 0 and DSValues_Armor.cost == 0) then
		GameTooltip:AddLine(DURABILITYSTATUS_UNDAMAGED);
	end
		
	GameTooltip:Show();

end


---------------------------------------------------
-- DurabilityStatus.ReturnSplitMoney
---------------------------------------------------
function DurabilityStatus.ReturnSplitMoney(sMoney)

	--convert the long money value into seperate variables
	local gold = floor(sMoney / (COPPER_PER_SILVER * SILVER_PER_GOLD));
	-- local silver = floor((sMoney - (gold * COPPER_PER_SILVER * SILVER_PER_GOLD)) / COPPER_PER_SILVER);
	local silver = floor(sMoney / COPPER_PER_SILVER) % SILVER_PER_GOLD;
	local copper = sMoney % COPPER_PER_SILVER;

	return gold, silver, copper;
end



---------------------------------------------------
-- DurabilityStatus.GetMoneyString
---------------------------------------------------
function DurabilityStatus.GetMoneyString(money)
	local moneytext = "";
	
	local gold = floor(money / (COPPER_PER_SILVER * SILVER_PER_GOLD));
	local silver = floor((money - (gold * COPPER_PER_SILVER * SILVER_PER_GOLD)) / COPPER_PER_SILVER);
	local copper = mod(money, COPPER_PER_SILVER);

	if (copper > 0) then
		moneytext = format(COPPER_AMOUNT, copper);
	end
	
	if (silver > 0) then
		if (copper > 0) then
			moneytext = ", "..moneytext;
		end
		
		moneytext = format(SILVER_AMOUNT, silver)..moneytext
	end

	if (gold > 0) then
		if (copper > 0 or silver > 0) then
			moneytext = ", "..moneytext;
		end
		
		moneytext = format(GOLD_AMOUNT, gold)..moneytext
	end
	
	return moneytext;
end





---------------------------------------------------
-- DurabilityStatus.ShowMerchantRepair
---------------------------------------------------
function DurabilityStatus.ShowMerchantRepair()

	if (CanMerchantRepair()) then

		if (DSValues_Inven.cost > 0 or DSValues_Armor.cost > 0) then
			
			MoneyFrame_Update("DS_Window_RepairArmorMoneyFrame", DSValues_Armor.cost);
			MoneyFrame_Update("DS_Window_RepairInvenMoneyFrame", DSValues_Inven.cost);
			MoneyFrame_Update("DS_Window_RepairAllMoneyFrame", DSValues_Inven.cost + DSValues_Armor.cost);
			
			if (CanGuildBankRepair()) then
				DS_Window_Repair:SetHeight(215);
				DS_Window_RepairTextGuild:Show();
				DS_Window_RepairGuildMoneyFrame:Show();
				DS_Window_RepairGuildButton:Show();

				local amount = GetGuildBankWithdrawMoney();
				local guildBankMoney = GetGuildBankMoney();
				if ( amount == -1 ) then
					-- Guild leader shows full guild bank amount
					amount = guildBankMoney;
				else
					amount = min(amount, guildBankMoney);
				end
				MoneyFrame_Update("DS_Window_RepairGuildMoneyFrame", amount);
				
				--if (DSValues_Armor.cost + DSValues_Inven.cost <= amount) then
				--	DS_Window_RepairGuildButton:Enable();
				--else
				--	DS_Window_RepairGuildButton:Disable();
				--end
			else
				DS_Window_RepairTextGuild:Hide();
				DS_Window_RepairGuildMoneyFrame:Hide();
				DS_Window_RepairGuildButton:Hide();
				DS_Window_Repair:SetHeight(150);
			end

			local userCash     = GetMoney();

			--check for money
			if(DSValues_Inven.cost <= userCash and DSValues_Inven.cost > 0) then
				DS_Window_RepairInvenButton:Enable();
			else
				DS_Window_RepairInvenButton:Disable();
			end

			--check for money
			if(DSValues_Armor.cost <= userCash and DSValues_Armor.cost > 0) then
				DS_Window_RepairArmorButton:Enable();
			else
				DS_Window_RepairArmorButton:Disable();
			end


			--check for money
			if( (DSValues_Inven.cost + DSValues_Armor.cost) <= userCash ) then
				DS_Window_RepairAllButton:Enable();
			else
				DS_Window_RepairAllButton:Disable();
			end


			DS_Window_Repair:Show();

		end

	end


end


---------------------------------------------------
-- DurabilityStatus.RepairIventory
---------------------------------------------------
function DurabilityStatus.RepairIventory()

	local userCash     = GetMoney();

	--check that we have cash
	if(DSValues_Inven.cost <= userCash and DSValues_Inven.cost > 0) then

		ShowRepairCursor();

		local bag, slot;

		for bag = 0,4,1 do

			for slot = 1, GetContainerNumSlots(bag) do

				local _, repairCost = DSCHKTT:SetBagItem(bag,slot);

				if (repairCost and repairCost > 0) then

					PickupContainerItem(bag,slot);
				end
			end
		end

		HideRepairCursor();

	end


	if( DurStatusOpt.showReport and DEFAULT_CHAT_FRAME ) then
		DEFAULT_CHAT_FRAME:AddMessage("|cffffff00"..DURABILITYSTATUS_INVENTORY_COST .. ": " 
			.. DurabilityStatus.GetMoneyString(DSValues_Inven.cost));
	end

	-- Wait a second before updating percentage
	DurabilityStatus_Core.bagupdate = .5;

end


---------------------------------------------------
-- DurabilityStatus.RepairAll
---------------------------------------------------
function DurabilityStatus.RepairAll()
	local userCash     = GetMoney();
	
	if(DSValues_Armor.cost + DSValues_Inven.cost <= userCash
			and DSValues_Armor.cost + DSValues_Inven.cost > 0) then
		
		RepairAllItems();
		
	end
	
	if( DurStatusOpt.showReport and DEFAULT_CHAT_FRAME ) then
		local msg = "|cffffff00"..DURABILITYSTATUS_ALL_COST .. ": " 
			.. DurabilityStatus.GetMoneyString(DSValues_Armor.cost + DSValues_Inven.cost);
		DEFAULT_CHAT_FRAME:AddMessage(msg);
	end

	-- Wait a second before updating percentage
	DurabilityStatus_Core.bagupdate = .5;

end


---------------------------------------------------
-- DurabilityStatus.RepairEquipment
---------------------------------------------------
function DurabilityStatus.RepairEquipment()

	local userCash     = GetMoney();

	if(DSValues_Armor.cost <= userCash and DSValues_Armor.cost > 0) then

		ShowRepairCursor();

		for i = 1, #DurStatus_Slots do

			local slotName = DurStatus_Slots[i] .. "Slot";

			local id = GetInventorySlotInfo(slotName);
			local _, _, repairCost = DSCHKTT:SetInventoryItem("player", id);

			if (repairCost and repairCost > 0) then

				PickupInventoryItem(id);
			end

		end

		HideRepairCursor();
	end

	if( DurStatusOpt.showReport and DEFAULT_CHAT_FRAME ) then
		DEFAULT_CHAT_FRAME:AddMessage("|cffffff00"..DURABILITYSTATUS_ARMOR_COST .. ": " 
			.. DurabilityStatus.GetMoneyString(DSValues_Armor.cost));
	end

	-- Wait a second before updating percentage
	DurabilityStatus_Core.bagupdate = .5;

end


---------------------------------------------------
-- DurabilityStatus.RepairGuild
---------------------------------------------------
function DurabilityStatus.RepairGuild()
	if (not CanGuildBankRepair()) then
		return;
	end
	
	-- Guild amount is cached and is sometimes 0 so don't block guild repairs based on it (sigh)
	--local amount = GetGuildBankWithdrawMoney();
	--local guildBankMoney = GetGuildBankMoney();
	--if ( amount == -1 ) then
		-- Guild leader shows full guild bank amount
	--	amount = guildBankMoney;
	--else
	--	amount = min(amount, guildBankMoney);
	--end
	
	--if(DSValues_Armor.cost + DSValues_Inven.cost <= amount
	--		and DSValues_Armor.cost + DSValues_Inven.cost > 0) then
		
		RepairAllItems(1);
		
	--end
	
	-- No way to determine if the repair was successful so forget about reporting until API works better (sigh again)
	--if( DurStatusOpt.showReport and DEFAULT_CHAT_FRAME ) then
	--	local msg = "|cffffff00"..DURABILITYSTATUS_ALL_COST .. ": " 
	--		.. DurabilityStatus_GetMoneyString(DSValues_Armor.cost + DSValues_Inven.cost).." "
	--		..DURABILITYSTATUS_USED_GUILD_FUNDS;
	--	DEFAULT_CHAT_FRAME:AddMessage(msg);
	--end

	-- Wait a second before updating percentage
	DurabilityStatus_Core.bagupdate = .5;

end
