----------------------------------------------------
-- Dice Companion - Roll Tracker (Final Stable - Duplicate Safe)
----------------------------------------------------

local DC = _G.DiceCompanion or {}
_G.DiceCompanion = DC
local DiceCompanion = DC

----------------------------------------------------
-- INITIALIZE ROLL TRACKER
----------------------------------------------------
function DiceCompanion:InitializeRollTracker()
    local rollTracker = _G["DiceCompanion_RollTrackerFrame"] or CreateFrame("Frame", "DiceCompanion_RollTrackerFrame", DiceCompanion_MainFrame)
    rollTracker:SetAllPoints(DiceCompanion_MainFrame)
    rollTracker:Hide()

    ----------------------------------------------------
    -- DATABASE PREP
    ----------------------------------------------------
    DiceCompanion.db.rollHistory = DiceCompanion.db.rollHistory or {}

    ----------------------------------------------------
    -- INSTRUCTION TEXT
    ----------------------------------------------------
    local instruction = rollTracker:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    instruction:SetPoint("TOP", rollTracker, "TOP", 0, -35)
    instruction:SetText("Select a casino from the drop-down menu to start keeping track of that casino's dice rolls.")
    instruction:SetJustifyH("CENTER")
    instruction:SetWidth(360)

    ----------------------------------------------------
    -- TEXT FRAMES (LEFT / RIGHT)
    ----------------------------------------------------
    local leftFrame = CreateFrame("Frame", nil, rollTracker, "BackdropTemplate")
    leftFrame:SetSize(180, 300)
    leftFrame:SetPoint("TOPLEFT", 30, -105)
    leftFrame:SetClipsChildren(true)

    local rightFrame = CreateFrame("Frame", nil, rollTracker, "BackdropTemplate")
    rightFrame:SetSize(180, 300)
    rightFrame:SetPoint("TOPRIGHT", 0, -105)
    rightFrame:SetClipsChildren(true)

    -- 🩵 Left Text (under left dropdown)
    local leftText = leftFrame:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    leftText:SetPoint("TOPLEFT", leftFrame, "TOPLEFT", 0, 0)
    leftText:SetWidth(leftFrame:GetWidth())
    leftText:SetJustifyH("LEFT")
    leftText:SetSpacing(2)
    leftText:SetWordWrap(true)

    -- 🩵 Right Text (aligned directly under right dropdown)
    local rightText = rightFrame:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    rightText:SetPoint("TOPRIGHT", rightFrame, "TOPRIGHT", 0, 0)
    rightText:SetWidth(rightFrame:GetWidth())
    rightText:SetJustifyH("LEFT")
    rightText:SetSpacing(2)
    rightText:SetWordWrap(true)

    ----------------------------------------------------
    -- REALM FIX HELPERS
    ----------------------------------------------------
    local function NormalizeName(name)
        if not name then return nil end
        local base, realm = name:match("([^%-]+)%-?(.*)")
        realm = realm ~= "" and realm or GetRealmName()
        return base .. "-" .. realm
    end

    ----------------------------------------------------
    -- CREATE CASINO DROPDOWNS
    ----------------------------------------------------
    local function CreateCasinoDropdown(name, parent)
        local dropdown = CreateFrame("Frame", name, parent, "UIDropDownMenuTemplate")
        UIDropDownMenu_SetWidth(dropdown, 140)
        _G[name .. "Text"]:SetText("Select Casino")

        UIDropDownMenu_Initialize(dropdown, function(self, level)
            local casinoList = DiceCompanion.db.casinoList or {}
            for _, casinoName in ipairs(casinoList) do
                local info = UIDropDownMenu_CreateInfo()
                info.text = casinoName
                info.value = casinoName
                info.func = function()
                    UIDropDownMenu_SetSelectedValue(dropdown, casinoName)
                    _G[name .. "Text"]:SetText(casinoName)
                    CloseDropDownMenus()
                    DiceCompanion:UpdateRollDisplays()
                end
                UIDropDownMenu_AddButton(info)
            end
        end)

        return dropdown
    end

    local dropdownLeft = CreateCasinoDropdown("DiceCompanion_CasinoDropdownLeft", rollTracker)
    dropdownLeft:SetPoint("TOPLEFT", 5, -70)

    local dropdownRight = CreateCasinoDropdown("DiceCompanion_CasinoDropdownRight", rollTracker)
    dropdownRight:SetPoint("TOPRIGHT", -5, -70)

    ----------------------------------------------------
    -- DISPLAY UPDATER (clamped inside frames)
    ----------------------------------------------------
    function DiceCompanion:UpdateRollDisplays()
        local leftSel = UIDropDownMenu_GetSelectedValue(_G["DiceCompanion_CasinoDropdownLeft"])
        local rightSel = UIDropDownMenu_GetSelectedValue(_G["DiceCompanion_CasinoDropdownRight"])

        local function FormatRolls(casino)
            if not casino or not DiceCompanion.db.rollHistory[casino] then return "" end
            local rolls = DiceCompanion.db.rollHistory[casino]
            local lines, maxVisible = {}, 30
            for i = #rolls, math.max(#rolls - maxVisible + 1, 1), -1 do
                local r = rolls[i]
                table.insert(lines, string.format("%d-%d = %d (%s)", r.roll1, r.roll2, r.total, r.label))
            end
            return table.concat(lines, "\n")
        end

        local leftTextValue = FormatRolls(leftSel)
        local rightTextValue = FormatRolls(rightSel)

        leftText:SetText(leftTextValue)
        rightText:SetText(rightTextValue)

        -- 🧩 Clamp text vertically if it exceeds the frame height
        local function ClampHeight(fontString, parent)
            local textHeight = fontString:GetStringHeight() or 0
            if textHeight > parent:GetHeight() then
                fontString:SetHeight(parent:GetHeight())
                fontString:SetMaxLines( math.floor(parent:GetHeight() / fontString:GetLineHeight()) )
            else
                fontString:SetHeight(textHeight)
                fontString:SetMaxLines(0)
            end
        end

        ClampHeight(leftText, leftFrame)
        ClampHeight(rightText, rightFrame)
    end

    ----------------------------------------------------
    -- REFRESH DROPDOWNS
    ----------------------------------------------------
    function DiceCompanion:RefreshCasinoDropdowns()
        local casinoList = DiceCompanion.db.casinoList or {}
        for _, dropdownName in pairs({ "DiceCompanion_CasinoDropdownLeft", "DiceCompanion_CasinoDropdownRight" }) do
            local dropdown = _G[dropdownName]
            if dropdown then
                _G[dropdownName .. "Text"]:SetText("Select Casino")
                UIDropDownMenu_Initialize(dropdown, function(self, level)
                    for _, casinoName in ipairs(casinoList) do
                        local info = UIDropDownMenu_CreateInfo()
                        info.text, info.value = casinoName, casinoName
                        info.func = function()
                            UIDropDownMenu_SetSelectedValue(dropdown, casinoName)
                            _G[dropdownName .. "Text"]:SetText(casinoName)
                            CloseDropDownMenus()
                            DiceCompanion:UpdateRollDisplays()
                        end
                        UIDropDownMenu_AddButton(info)
                    end
                end)
            end
        end
    end

    DiceCompanion:RefreshCasinoDropdowns()

    ----------------------------------------------------
    -- CLASSIFY ROLL
    ----------------------------------------------------
    local function ClassifyRoll(r1, r2)
        local total, tags = r1 + r2, {}
        if r1 == 1 and r2 == 1 then table.insert(tags, "Snake Eyes") end
        if r1 == 6 and r2 == 6 then table.insert(tags, "Boxcars") end
        if total == 2 or total == 12 then table.insert(tags, "Craps") end
        if r1 == r2 then table.insert(tags, "Doubles") end
        if total < 7 then
            table.insert(tags, "Under")
        elseif total == 7 then
            table.insert(tags, "7")
        else
            table.insert(tags, "Over")
        end
        return table.concat(tags, "/"), total
    end

    ----------------------------------------------------
    -- SAVE ROLL
    ----------------------------------------------------
    local function SaveCasinoRoll(casino, r1, r2)
        if not casino then return end
        DiceCompanion.db.rollHistory[casino] = DiceCompanion.db.rollHistory[casino] or {}
        local label, total = ClassifyRoll(r1, r2)
        table.insert(DiceCompanion.db.rollHistory[casino], { roll1 = r1, roll2 = r2, total = total, label = label })
        while #DiceCompanion.db.rollHistory[casino] > 100 do
            table.remove(DiceCompanion.db.rollHistory[casino], 1)
        end
        DiceCompanion:UpdateRollDisplays()
    end

    ----------------------------------------------------
    -- CHAT EVENT HANDLER (Singleton + Echo Safe)
    ----------------------------------------------------
    if DiceCompanion.RollListener then
        DiceCompanion.RollListener:UnregisterAllEvents()
    else
        DiceCompanion.RollListener = CreateFrame("Frame")
    end

    local listener = DiceCompanion.RollListener
    listener:RegisterEvent("CHAT_MSG_SYSTEM")

    local rollBuffer = {}
    local recentPairs = {}

    listener:SetScript("OnEvent", function(_, _, msg)
        local roller, roll = msg:match("^(.-) rolls (%d+) %(%d+%-%d+%)$")
        if not roller or not roll then return end
        roll = tonumber(roll)
        if roll < 1 or roll > 6 then return end

        local rollerFull = NormalizeName(roller)
        local leftSel  = UIDropDownMenu_GetSelectedValue(_G["DiceCompanion_CasinoDropdownLeft"])
        local rightSel = UIDropDownMenu_GetSelectedValue(_G["DiceCompanion_CasinoDropdownRight"])
        if not leftSel and not rightSel then return end

        local leftFull  = leftSel and NormalizeName(leftSel)
        local rightFull = rightSel and NormalizeName(rightSel)

        local matchedCasino
        if rollerFull == leftFull then
            matchedCasino = leftSel
        elseif rollerFull == rightFull then
            matchedCasino = rightSel
        else
            return
        end

        rollBuffer[rollerFull] = rollBuffer[rollerFull] or {}
        table.insert(rollBuffer[rollerFull], roll)

        if #rollBuffer[rollerFull] >= 2 then
            local r1, r2 = rollBuffer[rollerFull][1], rollBuffer[rollerFull][2]
            local pairKey = rollerFull .. "-" .. r1 .. "-" .. r2

            if not recentPairs[pairKey] then
                SaveCasinoRoll(matchedCasino, r1, r2)
                recentPairs[pairKey] = true
                C_Timer.After(2, function() recentPairs[pairKey] = nil end)
            end

            rollBuffer[rollerFull] = {}
        end
    end)

    ----------------------------------------------------
    -- LINK TO MAIN FRAME
    ----------------------------------------------------
    if DiceCompanion_MainFrame and DiceCompanion_MainFrame.contentFrames then
        DiceCompanion_MainFrame.contentFrames["RollTracker"] = rollTracker
    end
    DiceCompanion.RollTracker = rollTracker
end

----------------------------------------------------
-- INITIALIZATION ON LOGIN
----------------------------------------------------
local init = CreateFrame("Frame")
init:RegisterEvent("PLAYER_LOGIN")
init:SetScript("OnEvent", function()
    if DiceCompanion and DiceCompanion_MainFrame then
        DiceCompanion:InitializeRollTracker()
    end
end)
