----------------------------------------------------
-- Dice Companion - Profit Tracker (Full Build)
-- Globally Linked Module
----------------------------------------------------

local DC = _G.DiceCompanion or {}
_G.DiceCompanion = DC
local DiceCompanion = DC

----------------------------------------------------
-- CONFIG / CONSTANTS
----------------------------------------------------
-- Simple lists: formatting (icons/brackets) is stripped automatically
local BET_KEYWORDS   = { "bet", "bets", "[bet]", "[bets]", "[BET]", "[BETS]" }
local WIN_KEYWORDS = { "win", "wins", "won", "winner", "winning", "[wins]", "[WINS]", "[Winner]", "[won]", "[WON]" }

-- Guess words we’ll try to pick up from your SAY/WHISPER when you make a call.
local GUESS_WORDS = {
    ["over"] = true,
    ["under"] = true,
    ["7"] = true,
    ["snake eyes"] = true,
    ["craps"] = true,
    ["boxcars"] = true,
    ["doubles"] = true,
}

-- Max UI sections to render per refresh (sensible upper bound)
local MAX_SECTIONS = 100

----------------------------------------------------
-- HELPERS
----------------------------------------------------

-- Returns true if msg contains any of the keyword variants (case-insensitive)
local function MessageHasAnyKeyword(msg, keywords)
    if not msg or not keywords then return false end

    -- Lowercase for consistency
    local lmsg = string.lower(msg)

    -- Remove any WoW raid icons like {circle}, {star}, {skull}, {triangle}, etc.
    lmsg = lmsg:gsub("{%s*(circle|square|star|cross|skull|triangle|diamond|moon)%s*}", "")

    -- Strip punctuation, spaces, and brackets for easier matching
    lmsg = lmsg:gsub("[%p%s%[%]]", "")

    -- Compare against cleaned keywords
    for _, kw in ipairs(keywords) do
        local cleanKW = string.lower(kw):gsub("[%p%s%[%]]", "")
        if lmsg:find(cleanKW, 1, true) then
            return true
        end
    end
    return false
end

----------------------------------------------------
-- HELPERS
----------------------------------------------------
local function NormalizeName(name)
    if not name or name == "" then return nil end

    -- Trim spaces and remove WoW-style brackets or punctuation
    name = name:gsub("[%[%]%s]", "")

    -- Convert any Unicode dash variations (– — −) to standard hyphen (-)
    name = name:gsub("[–—−]", "-")

    -- Extract base and realm parts
    local base, realm = name:match("([^%-]+)%-?(.*)")

    -- Default realm if none provided
    realm = (realm and realm ~= "") and realm or GetRealmName()

    -- Return consistent normalized format
    return string.format("%s-%s", base, realm)
end

local function PlayerNames()
    local base = UnitName("player")
    local full = NormalizeName(base)
    return base, full
end

local function ci_find(haystack, needle)
    if not haystack or not needle then return nil end
    return string.find(string.lower(haystack), string.lower(needle), 1, true)
end

local function escape_pattern(s) -- for safe gsub/pattern usage
    return s and s:gsub("([^%w])", "%%%1") or s
end

-- Safely find a casino name inside a message by scanning the user's saved casino list.
-- Matches both short and Full-Name (Name-Realm) in a case-insensitive way.
local function FindCasinoInMessage(msg)
    if not DiceCompanion or not DiceCompanion.db then return nil end
    local list = DiceCompanion.db.casinoList or {}
    if not msg or msg == "" then return nil end

    for _, name in ipairs(list) do
        -- 🧩 Safety check: skip bad entries (numbers, nils, etc.)
        if type(name) == "string" and name ~= "" then
            local full = NormalizeName(name)
            local base = name:match("^[^%-]+") or name -- strip realm if any

            -- check base (short), saved, and normalized forms
            if ci_find(msg, base) or ci_find(msg, name) or ci_find(msg, full) then
                return name
            end
        end
    end

    return nil
end

-- Parse a gold amount like "100g", "100 Gold", "1,000 G", "12 gold", etc.
local function ParseGold(msg)
    if not msg or msg == "" then return nil end

    -- Look for a number before g/G or the word "gold" (case-insensitive)
    local num = msg:match("([%d,]+%.?%d*)%s*[gG]") or msg:match("([%d,]+%.?%d*)%s*[Gg][oO][lL][dD]")
    if not num then return nil end

    local clean = num:gsub(",", "")
    return tonumber(clean)
end

-- Returns true if msg contains the player's name (base or full)
local function MessageMentionsPlayer(msg)
    if not msg then return false end
    local base, full = PlayerNames()
    if ci_find(msg, base) or ci_find(msg, full) then
        return true
    end
    return false
end

-- Try to detect a guess word from a message (e.g., from your SAY/WHISPER)
local function DetectGuess(msg)
    if not msg or msg == "" then return nil end
    local l = msg:lower()
    -- multi-word first to avoid "snake eyes" breaking into pieces
    if l:find("snake%s*eyes") then return "Snake Eyes" end
    for word in pairs(GUESS_WORDS) do
        -- exact "7"
        if word == "7" then
            -- treat any isolated '7' (bounded by non-word or string edges) as a guess
            if l:match("(^7$)") or l:match("%f[%D]7%f[%D]") then
                return "7"
            end
        else
            -- word boundary style presence
            local pat = "%f[%a]" .. escape_pattern(word) .. "%f[%A]"
            if l:match(pat) then
                -- Title Case
                if word == "craps" then return "Craps" end
                if word == "boxcars" then return "Boxcars" end
                if word == "doubles" then return "Doubles" end
                if word == "over" then return "Over" end
                if word == "under" then return "Under" end
            end
        end
    end
    return nil
end

----------------------------------------------------
-- DB SHAPE
-- DiceCompanion.db.profitHistory[casino] = {
--   totalBets  = number,
--   totalWon   = number,
--   totalLost  = number,
--   -- derived: net = totalWon - totalLost
--   guessStats = {
--     [guess] = { wins = 0, losses = 0, net = 0 } -- net tracked for profitability
--   },
--   pendingBet = { amount = number, guess = "Over"/..., ts = time() } or nil,
--   lastSeenTS = time()
-- }
----------------------------------------------------

local function EnsureCasino(casino)
    DiceCompanion.db.profitHistory = DiceCompanion.db.profitHistory or {}
    local ph = DiceCompanion.db.profitHistory
    ph[casino] = ph[casino] or {
        totalBets = 0,
        totalWon = 0,
        totalLost = 0,
        guessStats = {},
        pendingBet = nil,
        lastSeenTS = time(),
    }
    return ph[casino]
end

local function AddGuessWin(casinoData, guess, amount)
    if not guess or guess == "" then return end
    local g = casinoData.guessStats[guess] or { wins = 0, losses = 0, net = 0 }
    g.wins = g.wins + 1
    g.net = g.net + (amount or 0)
    casinoData.guessStats[guess] = g
end

local function AddGuessLoss(casinoData, guess, amount)
    if not guess or guess == "" then return end
    local g = casinoData.guessStats[guess] or { wins = 0, losses = 0, net = 0 }
    g.losses = g.losses + 1
    g.net = g.net - (amount or 0)
    casinoData.guessStats[guess] = g
end

local function ResolvePendingAsLoss(casinoData)
    if casinoData.pendingBet then
        local amt = casinoData.pendingBet.amount or 0
        local guess = casinoData.pendingBet.guess
        casinoData.totalLost = (casinoData.totalLost or 0) + amt
        AddGuessLoss(casinoData, guess, amt)
        casinoData.pendingBet = nil
    end
end

----------------------------------------------------
-- MOST/LEAST PROFITABLE GUESS
----------------------------------------------------
local function DetermineBestWorstGuess(casinoData)
    local bestGuess, worstGuess = nil, nil
    local bestNet, worstNet = nil, nil
    for guess, stats in pairs(casinoData.guessStats or {}) do
        if stats and type(stats.net) == "number" then
            if bestNet == nil or stats.net > bestNet then
                bestNet = stats.net
                bestGuess = guess
            end
            if worstNet == nil or stats.net < worstNet then
                worstNet = stats.net
                worstGuess = guess
            end
        end
    end
    return bestGuess, worstGuess
end

----------------------------------------------------
-- UI
----------------------------------------------------
function DiceCompanion:InitializeProfitTracker()
    local profitTracker = _G["DiceCompanion_ProfitTrackerFrame"] or CreateFrame("Frame", "DiceCompanion_ProfitTrackerFrame", DiceCompanion_MainFrame)
    profitTracker:SetAllPoints(DiceCompanion_MainFrame)
    profitTracker:Hide()

    -- ensure DB
    DiceCompanion.db.profitHistory = DiceCompanion.db.profitHistory or {}

    ------------------------------------------------
    -- Hide Profit tab when main frame hides (fixes lingering tab block)
    ------------------------------------------------
    if DiceCompanion_MainFrame and not profitTracker._tabHideHooked then
        hooksecurefunc(DiceCompanion_MainFrame, "Hide", function()
            if _G["DiceCompanion_ProfitTrackerTab"] then
                _G["DiceCompanion_ProfitTrackerTab"]:Hide()
            end
        end)
        profitTracker._tabHideHooked = true
    end

------------------------------------------------
-- Scroll area (Guaranteed-working version for MoP Classic)
------------------------------------------------
local scrollFrame = CreateFrame("ScrollFrame", "DiceCompanion_ProfitScrollFrame", profitTracker, "UIPanelScrollFrameTemplate")
scrollFrame:SetPoint("TOPLEFT", profitTracker, "TOPLEFT", 20, -45)
scrollFrame:SetPoint("BOTTOMRIGHT", profitTracker, "BOTTOMRIGHT", -28, 20)

-- Scroll child
local content = CreateFrame("Frame", "DiceCompanion_ProfitContent", scrollFrame)
scrollFrame:SetScrollChild(content)
content:SetSize(1, 1)
content:SetPoint("TOPLEFT")

------------------------------------------------
-- 🔧 Scrollbar hookup (works on all client builds)
------------------------------------------------
local scrollbar   = _G["DiceCompanion_ProfitScrollFrameScrollBar"]
local upButton    = _G["DiceCompanion_ProfitScrollFrameScrollBarScrollUpButton"]
local downButton  = _G["DiceCompanion_ProfitScrollFrameScrollBarScrollDownButton"]

-- Keep range synced with content height
local function UpdateRange()
    if not scrollbar then return end
    local range = scrollFrame:GetVerticalScrollRange() or 0
    scrollbar:SetMinMaxValues(0, range)
    if range > 0 then
        scrollbar:Enable()
        if upButton then upButton:Enable() end
        if downButton then downButton:Enable() end
    else
        scrollbar:Disable()
        if upButton then upButton:Disable() end
        if downButton then downButton:Disable() end
    end
end

-- Mouse wheel
scrollFrame:EnableMouseWheel(true)
scrollFrame:SetScript("OnMouseWheel", function(self, delta)
    local step = 40
    local newValue = scrollbar:GetValue() - (delta * step)
    newValue = math.max(0, math.min(newValue, scrollbar:GetMinMaxValues()))
    scrollbar:SetValue(newValue)
end)

-- Arrows
if upButton then
    upButton:SetScript("OnClick", function()
        local v = scrollbar:GetValue()
        scrollbar:SetValue(math.max(0, v - 40))
    end)
end
if downButton then
    downButton:SetScript("OnClick", function()
        local v = scrollbar:GetValue()
        local max = select(2, scrollbar:GetMinMaxValues())
        scrollbar:SetValue(math.min(max or 0, v + 40))
    end)
end

-- Dragging thumb scrolls content
scrollbar:SetScript("OnValueChanged", function(self, value)
    scrollFrame:SetVerticalScroll(value or 0)
end)

-- Keep range fresh
hooksecurefunc(scrollFrame, "UpdateScrollChildRect", UpdateRange)
scrollFrame:SetScript("OnScrollRangeChanged", UpdateRange)

-- First-show sync
scrollFrame:SetScript("OnShow", function(self)
    self:UpdateScrollChildRect()
    scrollbar:SetValue(0)
    self:SetVerticalScroll(0)
    UpdateRange()
end)

------------------------------------------------
-- Store and width tracking
------------------------------------------------
profitTracker.scrollFrame = scrollFrame
profitTracker.content     = content
profitTracker.sections    = {}

scrollFrame:SetScript("OnSizeChanged", function(sf, width)
    if profitTracker and profitTracker.content then
        profitTracker.content:SetWidth(math.max(1, width - 18))
    end
    if sf.UpdateScrollChildRect then sf:UpdateScrollChildRect() end
end)

------------------------------------------------
-- Section Builder
------------------------------------------------
local function CreateSection(parent)
    local f = CreateFrame("Frame", nil, parent, "BackdropTemplate")
    f:SetSize(parent:GetWidth(), 100)

    f.divTop = f:CreateTexture(nil, "ARTWORK")
    f.divTop:SetColorTexture(0.9, 0.9, 0.9, 0.8)
    f.divTop:SetPoint("TOPLEFT", f, "TOPLEFT", 5, -2)
    f.divTop:SetPoint("TOPRIGHT", f, "TOPRIGHT", -5, -2)
    f.divTop:SetHeight(1)

    f.divBottom = f:CreateTexture(nil, "ARTWORK")
    f.divBottom:SetColorTexture(0.2, 0.2, 0.2, 0.8)
    f.divBottom:SetPoint("BOTTOMLEFT", f, "BOTTOMLEFT", 5, 0)
    f.divBottom:SetPoint("BOTTOMRIGHT", f, "BOTTOMRIGHT", -5, 0)
    f.divBottom:SetHeight(1)

    f.title = f:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    f.title:SetPoint("TOPLEFT", f, "TOPLEFT", 8, -14)
    f.title:SetJustifyH("LEFT")

    f.line1 = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    f.line1:SetPoint("TOPLEFT", f, "TOPLEFT", 12, -36)
    f.line1:SetJustifyH("LEFT")

    f.line2 = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    f.line2:SetPoint("TOPLEFT", f, "TOPLEFT", 12, -54)
    f.line2:SetJustifyH("LEFT")

    f.line3 = f:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    f.line3:SetPoint("TOPLEFT", f, "TOPLEFT", 12, -76)
    f.line3:SetJustifyH("LEFT")

    return f
end

------------------------------------------------
-- UI Update
------------------------------------------------
function DiceCompanion:UpdateProfitDisplay()
    local casinoList = DiceCompanion.db.casinoList or {}

    -- 🔄 Force rebuild of UI sections each update to prevent stale data
    if profitTracker.sections then
        for _, sec in ipairs(profitTracker.sections) do
            if sec then
                sec:Hide()
                sec:SetParent(nil)
            end
        end
    end
    profitTracker.sections = {}

    local lastSection
    for i, name in ipairs(casinoList) do
        if i > MAX_SECTIONS then break end
        local data = EnsureCasino(name)
        local net = (data.totalWon or 0) - (data.totalLost or 0)

        -- Create a new fresh section for each casino
        local sec = CreateSection(profitTracker.content)
        profitTracker.sections[i] = sec

        sec:ClearAllPoints()
        if not lastSection then
            sec:SetPoint("TOPLEFT", profitTracker.content, "TOPLEFT", 0, -10)
            sec:SetPoint("TOPRIGHT", profitTracker.content, "TOPRIGHT", 0, -10)
        else
            sec:SetPoint("TOPLEFT", lastSection, "BOTTOMLEFT", 0, -10)
            sec:SetPoint("TOPRIGHT", lastSection, "BOTTOMRIGHT", 0, -10)
        end

        local profitColor = net >= 0 and "|cff00ff00" or "|cffff0000"
        sec.title:SetText(name)
        sec.line1:SetText(string.format("Bets: %.2fg    Profit: %s%.2fg|r", data.totalBets or 0, profitColor, net))
        sec.line2:SetText(string.format("Lost: %.2fg    Won: %.2fg", data.totalLost or 0, data.totalWon or 0))

        local best, worst = DetermineBestWorstGuess(data)
        sec.line3:SetText(string.format("Most Profitable Guess: %s    Least Profitable Guess: %s", best or "—", worst or "—"))

        sec:Show()
        lastSection = sec
    end

    -- Dynamically size scrollable area
    local totalHeight = 0
    for _, sec in ipairs(profitTracker.sections) do
        if sec:IsShown() then
            totalHeight = totalHeight + sec:GetHeight() + 10
        end
    end
    profitTracker.content:SetHeight(totalHeight + 20)

    -- Update scroll area
    if scrollFrame.UpdateScrollChildRect then
        scrollFrame:UpdateScrollChildRect()
    end
end

    ------------------------------------------------
    -- Store for later
    ------------------------------------------------
    DiceCompanion.ProfitTracker = profitTracker

    ------------------------------------------------
    -- Register content frame with Main
    ------------------------------------------------
    if DiceCompanion_MainFrame and DiceCompanion_MainFrame.contentFrames then
        DiceCompanion_MainFrame.contentFrames["ProfitTracker"] = profitTracker
    end
end

----------------------------------------------------
-- CHAT LISTENERS
-- Bets & Wins come from SAY/EMOTE (casinos).
-- Your guess comes from your SAY or your WHISPER (to the casino).
----------------------------------------------------
local function OnChatEvent(self, event, msg, author, _, _, _, _, _, _, _, _, targetOrChannel)
    ------------------------------------------------
    -- 🟩 Casino announces a bet or win (SAY / EMOTE / TEXT_EMOTE)
    ------------------------------------------------
    if event == "CHAT_MSG_SAY" or event == "CHAT_MSG_EMOTE" or event == "CHAT_MSG_TEXT_EMOTE" then
        local amount = ParseGold(msg)
        local hasGold = amount and amount > 0

        if hasGold and MessageMentionsPlayer(msg) then
            local casinoRaw = author

            if (not casinoRaw or casinoRaw == "") and (event == "CHAT_MSG_EMOTE" or event == "CHAT_MSG_TEXT_EMOTE") then
                casinoRaw = msg:match("^([%w%p%-]+)%s") or ""
                if casinoRaw == "" then
                    casinoRaw = msg:match("^%p*([%w%-%_]+)%s") or ""
                end
            end

            if not casinoRaw or casinoRaw == "" then
                casinoRaw = FindCasinoInMessage(msg)
            end
            if not casinoRaw or casinoRaw == "" then return end

            casinoRaw = casinoRaw:gsub("[%[%]]", "")
            local casino = casinoRaw
            local normRaw = NormalizeName(casinoRaw)
            local list = DiceCompanion.db.casinoList or {}
            for _, listed in ipairs(list) do
                if NormalizeName(listed) == normRaw then
                    casino = listed
                    break
                end
            end

            local isBet = MessageHasAnyKeyword(msg, BET_KEYWORDS)
            local isWin = MessageHasAnyKeyword(msg, WIN_KEYWORDS)

            if isBet or isWin then
                local data = EnsureCasino(casino)
                data.lastSeenTS = time()

                if isBet then
                    ResolvePendingAsLoss(data)
                    data.totalBets = (data.totalBets or 0) + amount
                    data.pendingBet = { amount = amount, guess = data.pendingBet and data.pendingBet.guess or nil, ts = time() }
                    data._lastBetAmount = amount  -- ✅ Remember last bet for fallback loss handling

                elseif isWin then
                    data.totalWon = (data.totalWon or 0) + amount
                    if data.pendingBet then
                        if data.pendingBet.guess then
                            AddGuessWin(data, data.pendingBet.guess, amount)
                        end
                        data.pendingBet = nil
                    end
                    data._lastBetAmount = nil
                end

                if DiceCompanion and DiceCompanion.UpdateProfitDisplay then
                    DiceCompanion:UpdateProfitDisplay()
                end
                return
            end
        end
        -- ⚠️ Do NOT return here; loss detector below needs to run on same events!
    end

    ------------------------------------------------
    -- 💬 Detect SAY/EMOTE LOSS ("sorry", "apologizes", "bust", "busted", "wrong this time")
    ------------------------------------------------
    if event == "CHAT_MSG_SAY" or event == "CHAT_MSG_EMOTE" or event == "CHAT_MSG_TEXT_EMOTE" then
        local lower = msg:lower()
        local base, full = PlayerNames()

        local cleanMsg = lower:gsub("[%p]", " "):gsub("[–—−]", "-")
        local cleanBase = base:lower():gsub("[–—−]", "-")
        local cleanFull = full:lower():gsub("[–—−]", "-")

        if (cleanMsg:find("apologiz") or cleanMsg:find("sorry") or cleanMsg:find("bust") or cleanMsg:find("busted") or cleanMsg:find("wrong this time")) and
           (cleanMsg:find(cleanBase) or cleanMsg:find(cleanFull)) then

            local casinoRaw = author or FindCasinoInMessage(msg)
            if not casinoRaw or casinoRaw == "" then return end
            casinoRaw = casinoRaw:gsub("[%[%]]", "")

            local normCasino = NormalizeName(casinoRaw)
            local list = DiceCompanion.db.casinoList or {}
            local casino = casinoRaw
            for _, listed in ipairs(list) do
                if NormalizeName(listed) == normCasino then
                    casino = listed
                    break
                end
            end

            local data = EnsureCasino(casino)
            local amt = (data.pendingBet and data.pendingBet.amount) or data._lastBetAmount or 0
            if amt > 0 then
                AddGuessLoss(data, data.pendingBet and data.pendingBet.guess, amt)
                data.totalLost = (data.totalLost or 0) + amt
            end
            data.pendingBet = nil
            data._lastBetAmount = nil
            data.lastSeenTS = time()

            if DiceCompanion and DiceCompanion.UpdateProfitDisplay then
                DiceCompanion:UpdateProfitDisplay()
                C_Timer.After(0.05, function()
                    if DiceCompanion.UpdateProfitDisplay then
                        DiceCompanion:UpdateProfitDisplay()
                    end
                end)
            end
            return
        end
    end

    ------------------------------------------------
    -- 💌 Detect WHISPER LOSS ("Better luck next time" / "wrong this time")
    ------------------------------------------------
    if event == "CHAT_MSG_WHISPER" then
        local lower = msg:lower()
        if lower:find("better luck next time") or lower:find("wrong this time") then
            local list = DiceCompanion.db.casinoList or {}
            local isCasino = false
            for _, name in ipairs(list) do
                if NormalizeName(name) == NormalizeName(author) then
                    isCasino = true
                    break
                end
            end
            if not isCasino then return end

            local casino = author or FindCasinoInMessage(author)
            if casino then
                local normCasino = NormalizeName(casino)
                local list = DiceCompanion.db.casinoList or {}
                for _, listed in ipairs(list) do
                    if NormalizeName(listed) == normCasino then
                        casino = listed
                        break
                    end
                end

                local data = EnsureCasino(casino)
                local amt = (data.pendingBet and data.pendingBet.amount) or data._lastBetAmount or 0
                if amt > 0 then
                    AddGuessLoss(data, data.pendingBet and data.pendingBet.guess, amt)
                    data.totalLost = (data.totalLost or 0) + amt
                end
                data.pendingBet = nil
                data._lastBetAmount = nil
                data.lastSeenTS = time()

                if DiceCompanion and DiceCompanion.UpdateProfitDisplay then
                    DiceCompanion:UpdateProfitDisplay()
                    C_Timer.After(0.05, function()
                        if DiceCompanion.UpdateProfitDisplay then
                            DiceCompanion:UpdateProfitDisplay()
                        end
                    end)
                end
            end
            return
        end
    end

    ------------------------------------------------
    -- 🕒 5-MINUTE AUTO-RESOLVE FALLBACK
    ------------------------------------------------
    if not DiceCompanion._LossTimerActive then
        DiceCompanion._LossTimerActive = true
        C_Timer.NewTicker(60, function()
            if not DiceCompanion or not DiceCompanion.db or not DiceCompanion.db.profitHistory then return end
            for casino, data in pairs(DiceCompanion.db.profitHistory) do
                if data.pendingBet and time() - (data.pendingBet.ts or 0) >= 300 then
                    ResolvePendingAsLoss(data)
                    if DiceCompanion.UpdateProfitDisplay then
                        DiceCompanion:UpdateProfitDisplay()
                    end
                end
            end
        end)
    end

    ------------------------------------------------
    -- 🟦 Your own whispers to casinos (guess messages)
    ------------------------------------------------
    if event == "CHAT_MSG_WHISPER_INFORM" then
        local guess = DetectGuess(msg)
        if not guess then return end
        local receiver = targetOrChannel
        if not receiver then return end

        local casino = FindCasinoInMessage(receiver) or receiver
        if not FindCasinoInMessage(receiver) then
            local list = DiceCompanion.db.casinoList or {}
            local rFull = NormalizeName(receiver)
            for _, name in ipairs(list) do
                if NormalizeName(name) == rFull then casino = name break end
            end
        end
        if not casino then return end

        local data = EnsureCasino(casino)
        if data.pendingBet then
            data.pendingBet.guess = guess
        else
            data.pendingBet = { amount = 0, guess = guess, ts = time() }
        end
        return
    end

    ------------------------------------------------
    -- 🟨 Your public SAY guesses (for the most recent active casino)
    ------------------------------------------------
    if event == "CHAT_MSG_SAY" then
        local base, full = PlayerNames()
        if author ~= base and author ~= full then return end

        local guess = DetectGuess(msg)
        if not guess then return end

        local bestCasino, bestTS = nil, nil
        for name, data in pairs(DiceCompanion.db.profitHistory or {}) do
            if data.pendingBet and (not bestTS or (data.pendingBet.ts or 0) > bestTS) then
                bestTS = data.pendingBet.ts or 0
                bestCasino = name
            end
        end
        if bestCasino then
            local d = EnsureCasino(bestCasino)
            if d.pendingBet then
                d.pendingBet.guess = guess
            end
        end
        return
    end
end

----------------------------------------------------
-- INIT / EVENTS
----------------------------------------------------
local chatInit = CreateFrame("Frame")
chatInit:RegisterEvent("PLAYER_LOGIN")
chatInit:SetScript("OnEvent", function()
    -- Ensure saved vars initialized by core
    DiceCompanionDB = DiceCompanionDB or {}
    DiceCompanion.db = DiceCompanionDB
    DiceCompanion.db.profitHistory = DiceCompanion.db.profitHistory or {}
    DiceCompanion.db.casinoList    = DiceCompanion.db.casinoList or {}

    -- Build UI
    if DiceCompanion and DiceCompanion_MainFrame then
        DiceCompanion:InitializeProfitTracker()
    end

    -- Event listener
    if not DiceCompanion.ProfitListener then
        DiceCompanion.ProfitListener = CreateFrame("Frame")
    end
    local l = DiceCompanion.ProfitListener
    l:UnregisterAllEvents()
    l:RegisterEvent("CHAT_MSG_SAY")
    l:RegisterEvent("CHAT_MSG_EMOTE")
    l:RegisterEvent("CHAT_MSG_TEXT_EMOTE")   -- ✅ add this
    l:RegisterEvent("CHAT_MSG_WHISPER")
    l:RegisterEvent("CHAT_MSG_WHISPER_INFORM") -- your whispers (to casino)

    l:SetScript("OnEvent", OnChatEvent)

    -- First paint
    if DiceCompanion and DiceCompanion.UpdateProfitDisplay then
        DiceCompanion:UpdateProfitDisplay()
    end
end)
