----------------------------------------------------
-- Dice Companion - Main Menu
-- Globally Linked Module
----------------------------------------------------

-- Ensure global table exists
local DC = _G.DiceCompanion or {}
_G.DiceCompanion = DC
local DiceCompanion = DC

----------------------------------------------------
-- MAIN MENU UI SETUP
----------------------------------------------------
function DiceCompanion:InitializeMainMenu()
    local mainMenu = _G["DiceCompanion_MainMenuFrame"] or CreateFrame("Frame", "DiceCompanion_MainMenuFrame", DiceCompanion_MainFrame)
    mainMenu:SetAllPoints(DiceCompanion_MainFrame)
    mainMenu:Hide()

    ----------------------------------------------------
    -- INSTRUCTION TEXT
    ----------------------------------------------------
    local instruction = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    instruction:SetPoint("TOP", mainMenu, "TOP", 0, -35)
    instruction:SetText("Select Say or Whisper, then click a button to announce your guess.")
    instruction:SetJustifyH("CENTER")
    instruction:SetWidth(360)

    ----------------------------------------------------
    -- TOGGLES ROW (Whisper / Say)
    ----------------------------------------------------
    local whisperToggle = CreateFrame("CheckButton", "DiceCompanion_WhisperToggle", mainMenu, "UICheckButtonTemplate")
    whisperToggle:SetPoint("TOPLEFT", mainMenu, "TOPLEFT", 100, -45)
    whisperToggle.text:SetText("Whisper")

    local sayToggle = CreateFrame("CheckButton", "DiceCompanion_SayToggle", mainMenu, "UICheckButtonTemplate")
    sayToggle:SetPoint("LEFT", whisperToggle, "RIGHT", 100, 0)
    sayToggle.text:SetText("Say")

    -- Only one can be active at a time
    local function UpdateToggles(selected)
        if selected == "WHISPER" then
            whisperToggle:SetChecked(true)
            sayToggle:SetChecked(false)
            DiceCompanion.activeChannel = "WHISPER"
        elseif selected == "SAY" then
            sayToggle:SetChecked(true)
            whisperToggle:SetChecked(false)
            DiceCompanion.activeChannel = "SAY"
        end
    end

    local function EnforceSingleToggle(selected)
        C_Timer.After(0, function()
            UpdateToggles(selected)
            whisperToggle:SetChecked(selected == "WHISPER")
            sayToggle:SetChecked(selected == "SAY")
        end)
    end

    whisperToggle:SetScript("OnClick", function() EnforceSingleToggle("WHISPER") end)
    sayToggle:SetScript("OnClick", function() EnforceSingleToggle("SAY") end)
    EnforceSingleToggle("SAY")

    ----------------------------------------------------
    -- BUTTON CREATION FUNCTION
    ----------------------------------------------------
    local function CreateMainButton(name, label, parentFrame, x, y)
        local btn = CreateFrame("Button", name, parentFrame, "UIPanelButtonTemplate")
        btn:SetSize(90, 26)
        btn:SetPoint("TOPLEFT", parentFrame, "TOPLEFT", x, y)
        btn:SetText(label)
        btn:SetScript("OnClick", function()
            local msg = label
            if DiceCompanion.activeChannel == "SAY" then
                SendChatMessage(msg, "SAY")
            elseif DiceCompanion.activeChannel == "WHISPER" then
                local target = UnitName("target")
                if target then
                    SendChatMessage(msg, "WHISPER", nil, target)
                else
                    print("|cff00ff00[Dice Companion]|r You must target a player to whisper them.")
                end
            end
        end)
        return btn
    end

    ----------------------------------------------------
    -- BUTTON ROWS
    ----------------------------------------------------
    local btnOver   = CreateMainButton("DiceCompanion_OverButton", "Over", mainMenu, 15,  -80)
    local btn7      = CreateMainButton("DiceCompanion_7Button", "7", mainMenu, 155, -80)
    local btnUnder  = CreateMainButton("DiceCompanion_UnderButton", "Under", mainMenu, 295, -80)
    local btnSnake  = CreateMainButton("DiceCompanion_SnakeEyesButton", "Snake Eyes", mainMenu, 15, -105)
    local btnCraps  = CreateMainButton("DiceCompanion_CrapsButton", "Craps", mainMenu, 155, -105)
    local btnBoxcar = CreateMainButton("DiceCompanion_BoxcarsButton", "Boxcars", mainMenu, 295, -105)
    local btnHit    = CreateMainButton("DiceCompanion_HitButton", "Hit", mainMenu, 15, -131)
    local btnDouble = CreateMainButton("DiceCompanion_DoublesButton", "Doubles", mainMenu, 155, -131)
    local btnStand  = CreateMainButton("DiceCompanion_StandButton", "Stand", mainMenu, 295, -131)

    ----------------------------------------------------
    -- DIVIDER LINE BELOW DOUBLES BUTTON
    ----------------------------------------------------
    local dividerTop = mainMenu:CreateTexture(nil, "ARTWORK")
    dividerTop:SetColorTexture(0.9, 0.9, 0.9, 0.8)
    dividerTop:SetPoint("TOPLEFT", mainMenu, "TOPLEFT", 25, -164)
    dividerTop:SetPoint("TOPRIGHT", mainMenu, "TOPRIGHT", -25, -164)
    dividerTop:SetHeight(1)

    local dividerBottom = mainMenu:CreateTexture(nil, "ARTWORK")
    dividerBottom:SetColorTexture(0.2, 0.2, 0.2, 0.8)
    dividerBottom:SetPoint("TOPLEFT", mainMenu, "TOPLEFT", 25, -165)
    dividerBottom:SetPoint("TOPRIGHT", mainMenu, "TOPRIGHT", -25, -165)
    dividerBottom:SetHeight(1)

    ----------------------------------------------------
    -- ADD CASINO INSTRUCTION + BUTTON
    ----------------------------------------------------
    local casinoText = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    casinoText:SetPoint("TOPLEFT", mainMenu, "TOPLEFT", 35, -175)
    casinoText:SetWidth(300)
    casinoText:SetJustifyH("LEFT")
    casinoText:SetText("Target casino and click Add Casino to\nadd that casino to your casino list!")

    local addCasinoButton = CreateFrame("Button", "DiceCompanion_AddCasinoButton", mainMenu, "UIPanelButtonTemplate")
    addCasinoButton:SetSize(100, 24)
    addCasinoButton:SetPoint("LEFT", casinoText, "RIGHT", -50, 0)
    addCasinoButton:SetText("Add Casino")

    addCasinoButton:SetScript("OnClick", function()
        local targetName = UnitName("target")
        if not targetName then
            print("|cff00ff00[Dice Companion]|r You must target a player to add them as a casino.")
            return
        end
        DiceCompanion.db = DiceCompanion.db or {}
        DiceCompanion.db.casinoList = DiceCompanion.db.casinoList or {}
        for _, n in ipairs(DiceCompanion.db.casinoList) do
            if n == targetName then
                print("|cff00ff00[Dice Companion]|r " .. targetName .. " is already in your casino list.")
                return
            end
        end
        table.insert(DiceCompanion.db.casinoList, targetName)
        print("|cff00ff00[Dice Companion]|r Added " .. targetName .. " to your casino list.")
        if DiceCompanion.RefreshCasinoDropdowns then DiceCompanion:RefreshCasinoDropdowns() end
    end)

    ----------------------------------------------------
    -- DIVIDER LINE BELOW ADD CASINO SECTION
    ----------------------------------------------------
    local dividerTop2 = mainMenu:CreateTexture(nil, "ARTWORK")
    dividerTop2:SetColorTexture(0.9, 0.9, 0.9, 0.8)
    dividerTop2:SetPoint("TOPLEFT", mainMenu, "TOPLEFT", 25, -210)
    dividerTop2:SetPoint("TOPRIGHT", mainMenu, "TOPRIGHT", -25, -210)
    dividerTop2:SetHeight(1)

    local dividerBottom2 = mainMenu:CreateTexture(nil, "ARTWORK")
    dividerBottom2:SetColorTexture(0.2, 0.2, 0.2, 0.8)
    dividerBottom2:SetPoint("TOPLEFT", mainMenu, "TOPLEFT", 25, -211)
    dividerBottom2:SetPoint("TOPRIGHT", mainMenu, "TOPRIGHT", -25, -211)
    dividerBottom2:SetHeight(1)

    ----------------------------------------------------
    -- GRAND TOTAL SECTION
    ----------------------------------------------------
    local totalTitle = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlight")
    totalTitle:SetPoint("TOPLEFT", 25, -225)
    totalTitle:SetText("Grand Total (All Casinos Combined)")

    local totalLine1 = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    totalLine1:SetPoint("TOPLEFT", totalTitle, "BOTTOMLEFT", 10, -10)
    local totalLine2 = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    totalLine2:SetPoint("TOPLEFT", totalLine1, "BOTTOMLEFT", 0, -6)
    local totalLine3 = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    totalLine3:SetPoint("TOPLEFT", totalLine2, "BOTTOMLEFT", 0, -6)
    local totalLine4 = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    totalLine4:SetPoint("TOPLEFT", totalLine3, "BOTTOMLEFT", 0, -6)
    local totalLine5 = mainMenu:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    totalLine5:SetPoint("TOPLEFT", totalLine4, "BOTTOMLEFT", 0, -6)

    function DiceCompanion:UpdateGrandTotalsDisplay()
        local ph = DiceCompanion.db.profitHistory or {}
        local totalBets, totalWon, totalLost = 0, 0, 0
        for _, data in pairs(ph) do
            totalBets = totalBets + (data.totalBets or 0)
            totalWon  = totalWon  + (data.totalWon or 0)
            totalLost = totalLost + (data.totalLost or 0)
        end
        local net = totalWon - totalLost
        local winRate = totalBets > 0 and (totalWon / totalBets) * 100 or 0
        local profitColor = net >= 0 and "|cff00ff00" or "|cffff0000"
        totalLine1:SetText(string.format("Bets: %.2fg", totalBets))
        totalLine2:SetText(string.format("Won: %.2fg", totalWon))
        totalLine3:SetText(string.format("Lost: %.2fg", totalLost))
        totalLine4:SetText(string.format("Profit: %s%.2fg|r", profitColor, net))
        totalLine5:SetText(string.format("Win Rate: %.1f%%", winRate))
    end

    C_Timer.NewTicker(3, function()
        if mainMenu:IsShown() then
            DiceCompanion:UpdateGrandTotalsDisplay()
        end
    end)

    ----------------------------------------------------
    -- LINK TO TAB CONTENT FRAME
    ----------------------------------------------------
    if DiceCompanion_MainFrame and DiceCompanion_MainFrame.contentFrames then
        DiceCompanion_MainFrame.contentFrames["MainMenu"] = mainMenu
    end

    DiceCompanion.MainMenu = mainMenu
end

----------------------------------------------------
-- INITIALIZATION ON LOGIN
----------------------------------------------------
local initMainMenuFrame = CreateFrame("Frame")
initMainMenuFrame:RegisterEvent("PLAYER_LOGIN")
initMainMenuFrame:SetScript("OnEvent", function()
    if DiceCompanion and DiceCompanion_MainFrame then
        DiceCompanion:InitializeMainMenu()
    end
end)
