Tab = {}
function Tab.size(tab)
    local count = 0
    for key, value in pairs(tab) do
        count = count + 1
    end
    return count
end

function Tab.isSameTableType(tab1, tab2)
    return Tab.isTable(tab1) and Tab.isTable(tab2)
end

function Tab.shallowTableEquals(tab1, tab2)
    local equals = true
    local isTab = Tab.isTable(tab1) and Tab.isTable(tab2)
    if  isTab and Tab.size(tab1) == Tab.size(tab2) then
        for key, value in pairs(tab1) do
            if Tab.isSameTableType(tab2[key], value) ~= true and tab2[key] ~= value then
                equals = false
                break
            end
        end
    else
        equals = false
    end
    return equals
end

function Tab.tableToStr(tab)
    local str = "{"
    for key, value in pairs(tab) do
     str = str .. "[" .. tostring(key) .. "]=" .. tostring(value) .. ", "    
    end
    str = str .. "}"
    return str
end

function Tab.isTable(tab)
    return type(tab) == 'table'
end
messageTemplate = "" ..
"\n*******************************************************************************\n" ..
"%s %s\n" ..
"%s" ..
"%s" ..
"\n*******************************************************************************"

UnitTest = {}
UnitTest.__index = UnitTest

setmetatable(UnitTest, {
  __call = function (cls)
    return cls.new()
  end,
})

function UnitTest.new()
    local self = setmetatable({}, UnitTest)
    self:_init()
    return self
end

function UnitTest._init(self)
end

function UnitTest.setUp(self)

end

function UnitTest.tearDown(self)
    
end

function UnitTest.isFunTest(self, funName)
    return  string.sub(funName, 1, string.len("test_")) == "test_"
end

function UnitTest.isFunction(self, fun)
    return type(fun) == 'function' 
end

function UnitTest.isTestFunction(self, funName, fun)
    return self:isFunTest(funName) and self:isFunction(fun)
end

function UnitTest.createTestResult(self, funName, err)
    local result = {}
    if err ~= nil then
        if err["msg"] ~= nil then
            result["BODY"] = self:createResult(funName, err["msg"], "", 
                                               "Assertion Fail:")
            result["RESULT"] = "FAIL"
        else
            result["BODY"] = self:createResult(funName, err, debugstack(),
                                               "Assertion Error:")
            result["RESULT"] = "ERROR"
        end
else
        result["BODY"] = ""
        result["RESULT"] = "OK"
    end
    return result
end

function UnitTest.createResult(self, funName, err, traceBack, errType)
    return messageTemplate:format(errType, (funName .. "()"), err, traceBack)
end

function UnitTest.getTestsCount(self)
    local number = 0
    for funName,fun in pairs(self.__index) do
        if self:isTestFunction(funName, fun) then
            number = number + 1
        end
    end
    return number
end

function UnitTest.runOneTest(self, funName, fun)
    local result = nil
    self:setUp()
    if self:isTestFunction(funName, fun) then
        callResult, err = pcall(fun, self)
        result = self:createTestResult(funName, err)
    end
    self:tearDown()
    return result
end

function UnitTest.index(self)
   return self.__index 
end

function UnitTest.raiseAssert(self, message, extraMessage)
    if extraMessage == nil then
        extraMessage = ""
    end
    message = message .. "\n" .. extraMessage
    error({msg=message})
end

function UnitTest.assertTrue(self, expected, message)
    if expected ~= true then
        self:raiseAssert(string.format("%s is not true", expected),
                         message)
    end
end

function UnitTest.assertFalse(self, expected, message)
    if expected ~= false then
        self:raiseAssert(string.format("%s is not false", expected),
                         message)
    end
end

function UnitTest.assertEquals(self, actual, expected, message)
    if expected ~= actual then
        self:raiseAssert(
            string.format("Not Equals: actual %s ~= expected %s",
                          tostring(actual), tostring(expected)),
                          message)
    end
end

function UnitTest.assertShallowTableEquals(self, expected, actual, message)
    if Tab.shallowTableEquals(expected, actual) ~= true then
        self:raiseAssert(
            string.format("Not Equals: expected %s ~= actual %s",
                          Tab.tableToStr(expected), Tab.tableToStr(actual)),
                          message)
    end
end


UnitTestRunner = {}
UnitTestRunner.__index = UnitTestRunner

setmetatable(UnitTestRunner, {
  __call = function (cls)
    return cls.new()
  end,
})

function UnitTestRunner.new(self)
    local self = setmetatable({}, UnitTestRunner)
    self:_init()
    return self
end

function UnitTestRunner._init(self)
    self.unitTestObjects = {}
    self.results = {}
end

function UnitTestRunner.with(self, _unitTestObject)
    table.insert(self.unitTestObjects, _unitTestObject)
    return self
end

function UnitTestRunner.run(self)
    self.results = {}
    for _, _unitTest in pairs(self.unitTestObjects) do
        for funName,fun in pairs(_unitTest:index()) do
            local runResult = _unitTest:runOneTest(funName, fun)
            if runResult ~= nil then  -- Function is not for test
                table.insert(self.results, runResult)
                self:printProgress()
            end
        end
    end     
    self:printResults()
end

function UnitTestRunner.printProgress(self)
    local printOutDot = ""
    for key, value in pairs(self.results) do
       printOutDot = self:connectPointsProgress(printOutDot, value)
       --print("\r", printOutDot)
    end
end

function UnitTestRunner.printResults(self)
    self:printCollectedResults(self:collectResults())
    self:printCountedTests(self:countTests())
end

function UnitTestRunner.connectPointsProgress(self, progress, value)
    local testsCharacter = {}
    testsCharacter["OK"] = "."
    testsCharacter["FAIL"] = "F"
    testsCharacter["ERROR"] = "E"
    return progress .. testsCharacter[value["RESULT"]]
end

function UnitTestRunner.collectResults(self)
    local collectedResult = ""
    for _, value in pairs(self.results) do
        collectedResult = collectedResult .. value["BODY"] 
    end
    return collectedResult
end

function UnitTestRunner.printCollectedResults(self, collectedResults)
    print(collectedResults)
end

function UnitTestRunner.countTests(self)
    local testsCharacter = {}
    testsCharacter["OK"] = 0
    testsCharacter["FAIL"] = 0
    testsCharacter["ERROR"] = 0
    for _, value in pairs(self.results) do
        testsCharacter[value["RESULT"]] = testsCharacter[value["RESULT"]] + 1
    end
    return testsCharacter
end

function UnitTestRunner.printCountedTests(self, countedTests)
    print(string.format("Results: OK => %s  FAIL => %s  ERROR => %s", 
        countedTests["OK"], countedTests["FAIL"], countedTests["ERROR"]))
end