local HerbsAndOreTest = {}
HerbsAndOreTest.__index = HerbsAndOreTest

setmetatable(HerbsAndOreTest, {
  __index = UnitTest, -- this is what makes the inheritance work
  __call = function (cls)
    local self = setmetatable({}, cls)
    UnitTest:_init(self)
    return self
  end
})

local ItemsTest = {}
ItemsTest.__index = ItemsTest

setmetatable(ItemsTest, {
  __index = UnitTest, -- this is what makes the inheritance work
  __call = function (cls)
    local self = setmetatable({}, cls)
    UnitTest:_init(self)
    return self
  end
})

local DestroyusProMainTest = {}
DestroyusProMainTest.__index = DestroyusProMainTest

setmetatable(DestroyusProMainTest, {
  __index = UnitTest, -- this is what makes the inheritance work
  __call = function (cls)
    local self = setmetatable({}, cls)
    UnitTest:_init(self)
    return self
  end
})

function HerbsAndOreTest.setUp(self)
	self.bagFunction =  GetContainerItemInfo
	self.functionProfInfo = GetProfessionInfo
	self.functionGetProf = GetProfessions
	self.GetContainerNumSlots = GetContainerNumSlots
	---- 
	_profApi:setFirstProf(INSCRIPTION_LVL367_INDEX1)
	_profApi:setSecondProf(JEWELCRAFTING_LVL125_INDEX2)
	GetProfessions = function() return  _profApi:getProfessions() end
	GetProfessionInfo = function(index) return _profApi:getProfessionInfo(index) end
	GetContainerItemInfo = function(i, j) return bagClass:returnBagSlot(i, j) end
	GetContainerNumSlots = function(i) return bagClass:returnContainerNumSlot(i) end
    self.itemApi = _DPContainerItemApi()
end

function HerbsAndOreTest.tearDown(self)
	GetProfessionInfo = self.functionProfInfo
	GetProfessions = self.functionGetProf
	GetContainerItemInfo = self.bagFunction
	GetContainerNumSlots = self.GetContainerNumSlots
end

function HerbsAndOreTest.test_requiredStackAmountForMillingProspecting(self)					  
	bagClass:setBagSlot(MANA_THISTL_COUNT5_2_9)
	self.itemApi:setItemPos(MANA_THISTL_COUNT5_2_9["i"], MANA_THISTL_COUNT5_2_9["j"])
    self:assertEquals(DestroyusPro:isStackAmountEnough(self.itemApi:getStackSize()), 
	                  true, "Stack 5 should be enough")

	bagClass:setBagSlot(TIN_ORE_COUNT4_2_5)
	self.itemApi:setItemPos(TIN_ORE_COUNT4_2_5["i"], TIN_ORE_COUNT4_2_5["j"])					  
    self:assertEquals(DestroyusPro:isStackAmountEnough(self.itemApi:getStackSize()),
	                  false, "Stack 4 should not be enough")
					  
    bagClass:setBagSlot(FEDELEAF_COUNT6_2_9)
	self.itemApi:setItemPos(FEDELEAF_COUNT6_2_9["i"], FEDELEAF_COUNT6_2_9["j"])
    self:assertEquals(DestroyusPro:isStackAmountEnough(self.itemApi:getStackSize()),
	                  true, "Stack 6 should be enough")
end

function HerbsAndOreTest.test_isHerb(self)
    bagClass:setBagSlot(FEDELEAF_COUNT6_2_9)
	self.itemApi:setItemPos(FEDELEAF_COUNT6_2_9["i"], FEDELEAF_COUNT6_2_9["j"])
    self:assertEquals(DestroyusPro:isHerb(self.itemApi:getItemClassId(),
           	          self.itemApi:getItemSubClassId()), true, "Case 1")
	
	bagClass:setBagSlot(COPPER_ORE_COUNT5_1_1)
	self.itemApi:setItemPos(COPPER_ORE_COUNT5_1_1["i"], COPPER_ORE_COUNT5_1_1["j"])
    self:assertEquals(DestroyusPro:isHerb(self.itemApi:getItemClassId(),
	                  self.itemApi:getItemSubClassId()), false, "Case 2")
	
	bagClass:setBagSlot(FILTHY_PAW_FIST_WEAPON_1_1)
    self.itemApi:setItemPos(FILTHY_PAW_FIST_WEAPON_1_1["i"], FILTHY_PAW_FIST_WEAPON_1_1["j"])
    self:assertEquals(DestroyusPro:isHerb(self.itemApi:getItemClassId(),
                      self.itemApi:getItemSubClassId()), false, "Case 3")
end

function HerbsAndOreTest.test_isOre(self)
	bagClass:setBagSlot(THORIUM_ORE_COUNT20_1_10)
    self.itemApi:setItemPos(THORIUM_ORE_COUNT20_1_10["i"], THORIUM_ORE_COUNT20_1_10["j"])
    self:assertEquals(DestroyusPro:isOre(self.itemApi:getItemClassId(),
	                  self.itemApi:getItemSubClassId(),
                      self.itemApi:getItemId()), true)
	
	-- Is Metal and Trade good. Not ore
	bagClass:setBagSlot(COPPER_BAR_COUNT50_1_1)
    self.itemApi:setItemPos(COPPER_BAR_COUNT50_1_1["i"], COPPER_BAR_COUNT50_1_1["j"])	
    self:assertEquals(DestroyusPro:isOre(self.itemApi:getItemClassId(),
	                  self.itemApi:getItemSubClassId(),
                      self.itemApi:getItemId()), false)
										 
	bagClass:setBagSlot(GHOST_IRON_ORE_COUNT5_1_1)
    self.itemApi:setItemPos(GHOST_IRON_ORE_COUNT5_1_1["i"], GHOST_IRON_ORE_COUNT5_1_1["j"])
    self:assertEquals(DestroyusPro:isOre(self.itemApi:getItemClassId(),
	                  self.itemApi:getItemSubClassId(),
                      self.itemApi:getItemId()), true)
		
	bagClass:setBagSlot(RAZORWIND_MACE_ONE_HAND_MACE_1_1)
    self.itemApi:setItemPos(RAZORWIND_MACE_ONE_HAND_MACE_1_1["i"], RAZORWIND_MACE_ONE_HAND_MACE_1_1["j"])										 
    self:assertEquals(DestroyusPro:isOre(self.itemApi:getItemClassId(),
	                  self.itemApi:getItemSubClassId(),
                      self.itemApi:getItemId()), false)
end

function HerbsAndOreTest.test_herbQualityAllowed(self)
    self:assertEquals(DestroyusPro:isQualityHerbAllowed(1), 
	                  true, "Quality 1 should be allowed")
    self:assertEquals(DestroyusPro:isQualityHerbAllowed(2),
	                  true, "Quality 2 should be allowed")
    self:assertEquals(DestroyusPro:isQualityHerbAllowed(0),
	                  false, "Quality 0 should not be allowed")
    self:assertEquals(DestroyusPro:isQualityHerbAllowed(3),
	                  false, "Quality 3 should be allowed")
end

function HerbsAndOreTest.test_isItemNotConsumable(self)
    -- Item is Herb. Consumable
	bagClass:setBagSlot(BROKEN_FROSTWEED_STEAM_COUNT50_2_3)
    self.itemApi:setItemPos(BROKEN_FROSTWEED_STEAM_COUNT50_2_3["i"],
	                        BROKEN_FROSTWEED_STEAM_COUNT50_2_3["j"])
    self:assertEquals(DestroyusPro:isItemNotConsumable(self.itemApi:getItemId()), 
	                  false, "Items should not be consumable case 1") 
					  
    -- Item is Herb. Not Consumable
	bagClass:setBagSlot(FIREWEED_COUNT4_1_1)
    self.itemApi:setItemPos(FIREWEED_COUNT4_1_1["i"],
	                        FIREWEED_COUNT4_1_1["j"])
    self:assertEquals(DestroyusPro:isItemNotConsumable(self.itemApi:getItemId()), 
	                  true, "Items should be consumable case 2")
					  
    -- Item is Herb. Consumable
	bagClass:setBagSlot(ADDER_TONGUE_STEAM_COUNT50_3_1)
    self.itemApi:setItemPos(ADDER_TONGUE_STEAM_COUNT50_3_1["i"],
	                        ADDER_TONGUE_STEAM_COUNT50_3_1["j"])
    self:assertEquals(DestroyusPro:isItemNotConsumable(self.itemApi:getItemId()), 
	                  false, "Items should not be consumable case 3")
					  
    -- Item is Herb. Not Consumable
	bagClass:setBagSlot(SNOW_LILY_COUNT5_1_1)
    self.itemApi:setItemPos(SNOW_LILY_COUNT5_1_1["i"],
	                        SNOW_LILY_COUNT5_1_1["j"])
    self:assertEquals(DestroyusPro:isItemNotConsumable(self.itemApi:getItemId()), 
	                  true, "Items should be consumable case 4")
end

function HerbsAndOreTest.test_haveItemRequiredProfLvl(self)
	-- Gromsblood
	bagClass:setBagSlot(GROMSBLOOD_COUNT10_1_1)
    self.itemApi:setItemPos(GROMSBLOOD_COUNT10_1_1["i"],
	                        GROMSBLOOD_COUNT10_1_1["j"])
    self:assertEquals(DestroyusPro:haveItemRequiredProfLvl(
	    self.itemApi:getItemId(),
        _DPProfessionApi.skilInscription, DestroyusPro.herbsIlvl),
		true, "Gromsblood should have required prof lvl")
		
	-- Snow Lily
	bagClass:setBagSlot(SNOW_LILY_COUNT5_1_1)
    self.itemApi:setItemPos(SNOW_LILY_COUNT5_1_1["i"],
	                        SNOW_LILY_COUNT5_1_1["j"])
    self:assertEquals(DestroyusPro:haveItemRequiredProfLvl(
		self.itemApi:getItemId(),
        _DPProfessionApi.skilInscription, DestroyusPro.herbsIlvl),
		false, "Snow Lily should not have required prof lvl") 
		
	-- Fireweed
	bagClass:setBagSlot(FIREWEED_COUNT10_1_1)
    self.itemApi:setItemPos(FIREWEED_COUNT10_1_1["i"],
	                        FIREWEED_COUNT10_1_1["j"])
    self:assertEquals(DestroyusPro:haveItemRequiredProfLvl(
		self.itemApi:getItemId(),
        _DPProfessionApi.skilInscription, DestroyusPro.herbsIlvl),
		true, "Fireweed should have required prof lvl")
		
	-- Starlight Rose
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT20_3_12)
    self.itemApi:setItemPos(STARLIGHT_ROSE_COUNT20_3_12["i"],
	                        STARLIGHT_ROSE_COUNT20_3_12["j"])
    self:assertEquals(DestroyusPro:haveItemRequiredProfLvl(
        self.itemApi:getItemId(),
        _DPProfessionApi.skilInscription, DestroyusPro.herbsIlvl),
		true, "Starlight should have required prof lvl")
	
	-- Razorwind mace
	bagClass:setBagSlot(RAZORWIND_MACE_ONE_HAND_MACE_1_1)
    self.itemApi:setItemPos(RAZORWIND_MACE_ONE_HAND_MACE_1_1["i"],
	                        RAZORWIND_MACE_ONE_HAND_MACE_1_1["j"])
    self:assertEquals(DestroyusPro:haveItemRequiredProfLvl(
	    self.itemApi:getItemId(),
        _DPProfessionApi.skilInscription, DestroyusPro.herbsIlvl),
		false, "Razorwind mace should not have required prof lvl")
end

function HerbsAndOreTest.test_isNotUniqueItem(self)
    -- Herb is in unique list
	bagClass:setBagSlot(WILDVINE_COUNT20_2_2)
    self.itemApi:setItemPos(WILDVINE_COUNT20_2_2["i"],
	                        WILDVINE_COUNT20_2_2["j"])
	self:assertEquals(DestroyusPro:isNotUniqueItem(self.itemApi:getItemId(),
	                  DestroyusPro.itemsIdNotForMilling), 
	                  false, "Items should be unique!")
			
    -- Herb is in not on the unique list			
	bagClass:setBagSlot(SNOW_LILY_COUNT5_1_1)
    self.itemApi:setItemPos(SNOW_LILY_COUNT5_1_1["i"],
	                        SNOW_LILY_COUNT5_1_1["j"])
	self:assertEquals(DestroyusPro:isNotUniqueItem(self.itemApi:getItemId(),
	                  DestroyusPro.itemsIdNotForMilling),
	                  true, "Items should not be unique!")
end

function HerbsAndOreTest.test_isItemNotSellable(self)
	-- Item is not for sell 
	self:assertEquals(DestroyusPro:isItemNotSellable(true),
	                  false, "Item should be unique and not for sell")
	-- Item is for sell
	self:assertEquals(DestroyusPro:isItemNotSellable(false),
	                  true, "Item should be unique and can be sold")
end

function HerbsAndOreTest.test_isItemMillingable(self)
	-- Herb which can not be millingable
	bagClass:setBagSlot(SNOW_LILY_COUNT5_1_1) -- itemId, count, sellable
    self.itemApi:setItemPos(SNOW_LILY_COUNT5_1_1["i"],
	                        SNOW_LILY_COUNT5_1_1["j"])
	self:assertEquals(DestroyusPro:isItemMillingable(SNOW_LILY_COUNT5_1_1["i"],
	                  SNOW_LILY_COUNT5_1_1["j"]),
	                  false, "Herb can not be millingable")
					  
	-- Stack size good and herb is millingable
	bagClass:setBagSlot(FIREWEED_COUNT10_1_1)
	self:assertEquals(DestroyusPro:isItemMillingable(FIREWEED_COUNT10_1_1["i"],
	                  FIREWEED_COUNT10_1_1["j"]),
	                  true, "Stack size bad or herb is not millingable")
					  
	-- Stack size small when herb is millingable
	bagClass:setBagSlot(FIREWEED_COUNT4_1_1)
	self:assertEquals(DestroyusPro:isItemMillingable(FIREWEED_COUNT4_1_1["i"],
	                  FIREWEED_COUNT4_1_1["j"]),
	                  false, "Stack size should be 4 < 5!")  
					  
	-- Lvl of prof skill is better than herb
	bagClass:setBagSlot(GROMSBLOOD_COUNT10_1_1)
	self:assertEquals(DestroyusPro:isItemMillingable(GROMSBLOOD_COUNT10_1_1["i"],
	                  GROMSBLOOD_COUNT10_1_1["j"]),
	                  true, "Skill lvl is lover than herb lvl")
					  
	-- Low lvl skill 475 > proff lvl
	bagClass:setBagSlot(CINDERBLOOM_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemMillingable(CINDERBLOOM_COUNT5_1_1["i"],
	                  CINDERBLOOM_COUNT5_1_1["j"]),
	                  false, "Herb can not be milled in current prof lvl")
					  
	-- Item is not herb
	bagClass:setBagSlot(COPPER_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemMillingable(COPPER_ORE_COUNT5_1_1["i"],
	                  COPPER_ORE_COUNT5_1_1["j"]),
	                  false, "Item is not a herb!")
					  
	-- Stack size good and herb is millingable. uncommon quality felwort
	bagClass:setBagSlot(FELWORT_COUNT10_1_1)
	self:assertEquals(DestroyusPro:isItemMillingable(FELWORT_COUNT10_1_1["i"],
	                  FELWORT_COUNT10_1_1["j"]), true,
					  "Felwort uncommon quality should be selected as ok")
end

function HerbsAndOreTest.test_isItemProspectingable(self)
	-- Copper Ore
	bagClass:setBagSlot(COPPER_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(COPPER_ORE_COUNT5_1_1["i"],
	                  COPPER_ORE_COUNT5_1_1["j"]),
	                  true, "Copper ore should be Prospectable")
					  
	-- Iron Ore and prof lvl are equals
	bagClass:setBagSlot(IRON_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(IRON_ORE_COUNT5_1_1["i"],
	                  IRON_ORE_COUNT5_1_1["j"]),
	                  true, "Iron Ore and prof lvl should be equals")
					  
	-- Iron ore and prof lvl are equals, stack small
	bagClass:setBagSlot(IRON_ORE_COUNT4_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(IRON_ORE_COUNT4_1_1["i"],
	                  IRON_ORE_COUNT4_1_1["j"]),
	                  false, "ron ore and prof lvl are equals, stack 4 < 5!")
					  
	-- Fel Iron Ore prof lvl to small
	bagClass:setBagSlot(FEL_IRON_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(FEL_IRON_ORE_COUNT5_1_1["i"],
	                  FEL_IRON_ORE_COUNT5_1_1["j"]),
	                 false, "Fel Iron Ore should not have been noticed")
					 
	-- Ghost Iron Ore lvl to small
	bagClass:setBagSlot(GHOST_IRON_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(GHOST_IRON_ORE_COUNT5_1_1["i"],
	                  GHOST_IRON_ORE_COUNT5_1_1["j"]),
	                 false, "Ghost Iron Ore should not have been noticed")
					 
	-- Copper Bar
	bagClass:setBagSlot(COPPER_BAR_COUNT50_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(COPPER_BAR_COUNT50_1_1["i"],
	                  COPPER_BAR_COUNT50_1_1["j"]),
	                  false, "Copper ore is not Prospectingable")
					  
	-- Leystone Ore legion Ore
	bagClass:setBagSlot(LEYSTONE_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(LEYSTONE_ORE_COUNT5_1_1["i"],
	                  LEYSTONE_ORE_COUNT5_1_1["j"]),
	                  true, "Leystone Ore should be Prospectable")
					  
	-- Felslate Legion 
	bagClass:setBagSlot(FELSLATE_ORE_COUNT5_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(FELSLATE_ORE_COUNT5_1_1["i"],
	                  FELSLATE_ORE_COUNT5_1_1["j"]),
	                  true, "Felslate ore should be Prospectable")
					  
	-- Felslate Stack to small
	bagClass:setBagSlot(FELSLATE_ORE_COUNT4_1_1)
	self:assertEquals(DestroyusPro:isItemProspectingable(FELSLATE_ORE_COUNT4_1_1["i"],
	                  FELSLATE_ORE_COUNT4_1_1["j"]),
	                  false, "Stack size to low. ")
end

-- DISENCHANTING TESTS

function ItemsTest.setUp(self)
	self.functionProfInfo = GetProfessionInfo
	self.functionGetProf = GetProfessions
	self.bagFunction =  GetContainerItemInfo
	self.GetContainerNumSlots = GetContainerNumSlots
	---
	GetProfessions = function() return  _profApi:getProfessions() end
	GetProfessionInfo = function(index) return _profApi:getProfessionInfo(index) end
	GetContainerItemInfo = function(i, j) return bagClass:returnBagSlot(i, j) end
	GetContainerNumSlots = function(i) return bagClass:returnContainerNumSlot(i) end
	self.itemApi = _DPContainerItemApi()
	self.itemApi:setItemPos(1,1)
	
	-- Set Disenchant fields set for tests
	DestroyusProMain.changeQualityDisenchanting("uncommon", 2)
	DestroyusProMain.changeQualityDisenchanting("rare", 3)
	DestroyusProMain.changeQualityDisenchanting("epic", 4)
end

function ItemsTest.tearDown(self)
	GetProfessionInfo = self.functionProfInfo
	GetProfessions = self.functionGetProf
	GetContainerItemInfo = self.bagFunction 
	GetContainerNumSlots = self.GetContainerNumSlots
	
	-- Set Disenchant fields set for tests
	DestroyusProMain.changeQualityDisenchanting("uncommon", 2)
	DestroyusProMain.changeQualityDisenchanting("rare", -1)
	DestroyusProMain.changeQualityDisenchanting("epic", -1)
end

function ItemsTest.test_isItem(self)
	-- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:isItem(self.itemApi:getItemClassId(),
                 	  self.itemApi:getItemSubClassId(), self.itemApi:getEquipSlot()),
	                  true, "Sunspring Vest should be seen as item!")
					  
	-- Crystallizing Mana 141292
	bagClass:setBagSlot(CRYSTALLIZING_MANA_1_1)
	self:assertEquals(DestroyusPro:isItem(self.itemApi:getItemClassId(),
                 	  self.itemApi:getItemSubClassId(), self.itemApi:getEquipSlot()),
	                  true, "Crystallizing Mana should be seen as Artifact Weapon upgrade")
	
	-- Rigid Lapis Lazuli 76502
	bagClass:setBagSlot(RIGID_LAPIS_LAZULI_1_1)
	self:assertEquals(DestroyusPro:isItem(self.itemApi:getItemClassId(),
                 	  self.itemApi:getItemSubClassId(), self.itemApi:getEquipSlot()),
	                  false, "Rigid Lapis Lazuli should not be seen as Artifact Weapon upgrade")
end

function ItemsTest.test_isWeapon(self)
	-- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:isWeapon(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  false, "Sunspring Vest is a Armor!")
	-- Tracker's Musket of the Bandit 36643
	bagClass:setBagSlot(TRACKERS_MUSKET_OF_THE_BANDIT_1_1)
	self:assertEquals(DestroyusPro:isWeapon(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  true, "Tracker's Musket of the Bandit should be Weapon!")
	-- Oval Ring of Stamina 36417 
	bagClass:setBagSlot(OVAL_RING_OF_STAMINA_1_1)
	self:assertEquals(DestroyusPro:isWeapon(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  false, "Oval Ring of Stamina is a Miscellaneous!")

end

function ItemsTest.test_isArtifactWeapon(self)
	-- GRISLY_SOUVENIR_3_2 140413
	bagClass:setBagSlot(GRISLY_SOUVENIR_3_2)
	self:assertEquals(DestroyusPro:isArtifactRelic(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  true, "Grisly Souvenir is a Artifact Weapon upgrade!")
					  
	-- Nightmare Tear 140413
	bagClass:setBagSlot(NIGHTMARE_TEAR_1_1)
	self:assertEquals(DestroyusPro:isArtifactRelic(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  false, "Nightmare Tear is not a Artifact Weapon upgrade!")
end

function ItemsTest.test_isArmor(self)
	-- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:isArmor(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  true, "Sunspring Vest should be armor!")
	-- Tracker's Musket of the Bandit 36643 
	bagClass:setBagSlot(TRACKERS_MUSKET_OF_THE_BANDIT_1_1)
	self:assertEquals(DestroyusPro:isArmor(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  false, "Tracker's Musket of the Bandit is Weapon!") 
	-- Oval Ring of Stamina 36417 
	bagClass:setBagSlot(OVAL_RING_OF_STAMINA_1_1)
	self:assertEquals(DestroyusPro:isArmor(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId()),
	                  false, "Oval Ring of Stamina is a Miscellaneous!") 
	
end 

function ItemsTest.test_isMiscellaneous(self)
	-- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:isMiscellaneous(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId(),
	                  self.itemApi:getEquipSlot()),
	                  false, "Sunspring Vest is armor!")
	-- Tracker's Musket of the Bandit 36643 
	bagClass:setBagSlot(TRACKERS_MUSKET_OF_THE_BANDIT_1_1)
	self:assertEquals(DestroyusPro:isMiscellaneous(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId(),
	                 self.itemApi:getEquipSlot()),
	                 false, "Tracker's Musket of the Bandit is Weapon!")
	-- Oval Ring of Stamina 36417 
	bagClass:setBagSlot(OVAL_RING_OF_STAMINA_1_1)
	self:assertEquals(DestroyusPro:isMiscellaneous(self.itemApi:getItemClassId(), self.itemApi:getItemSubClassId(),
                      self.itemApi:getEquipSlot()),
	                  true, "Oval Ring of Stamina should be a Miscellaneous!")
end 

function ItemsTest.test_getItemProffLvlToDisArmor(self)
	-- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(self.itemApi:getILvl(), self.itemApi:getQuality()),
	                 1, "Prof lvl is different than 1!")
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(nil, 3),
	                  DestroyusPro.numberNotExist, "Value does not exist! ")
	-- Holy Shroud 2721 rare
	bagClass:setBagSlot(HOLY_SHROUD_RARE_1_1)
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(self.itemApi:getILvl(), self.itemApi:getQuality()),
	                 1, "Prof lvl is different than 75!")
	-- Belt of the Blood Nova 50015 epic
	bagClass:setBagSlot(BELT_OF_THE_BLOOD_NOVA_EPIC_1_1)
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(self.itemApi:getILvl(), self.itemApi:getQuality()),
	                  1, "Prof lvl is different than 375!") 	
end

function ItemsTest.test_getItemProffLvlToDisWeapon(self)
	-- Tracker's Musket of the Bandit 36643 
	bagClass:setBagSlot(TRACKERS_MUSKET_OF_THE_BANDIT_1_1)
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(self.itemApi:getILvl(), self.itemApi:getQuality()),
	                  1, "Value of weapon is different tan 325!")
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(-1, 3),
	                  DestroyusPro.numberNotExist, "Number should not exist. Case 1")
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(-1, 4),
	                  DestroyusPro.numberNotExist, "Number should not exist. Case 2")
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(-1, 2),
	                  DestroyusPro.numberNotExist, "Number should not exist. Case 3")
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(1, -1), 
	                  DestroyusPro.numberNotExist, "Number should not exist. Case 4")
end

function ItemsTest.test_getItemProffLvlToDisRing(self)
	-- Oval Ring of Stamina 36417 
	bagClass:setBagSlot(OVAL_RING_OF_STAMINA_1_1)
	self:assertEquals(DestroyusPro:getItemProffLvlToDis(
                      self.itemApi:getILvl(), self.itemApi:getQuality()),
	                  1, "Value is different than 325!")
end

function ItemsTest.test_isOnlyOneMaxStack(self)
	-- Tracker's Musket of the Bandit 36643 
	bagClass:setBagSlot(TRACKERS_MUSKET_OF_THE_BANDIT_1_1)
	self:assertEquals(DestroyusPro:isOnlyOneMaxStack(
	                  self.itemApi:getMaxStackSize()), true,
					  "Item do not have one stack")
					  
	-- Gromsblood 8846
	bagClass:setBagSlot(GROMSBLOOD_COUNT10_1_1)
	self:assertEquals(DestroyusPro:isOnlyOneMaxStack(
	                 self.itemApi:getMaxStackSize()), false,
					 "Item should not be only one stack!")
end

function ItemsTest.test_isQualityAllowed(self)
	-- Tracker's Musket of the Bandit 36643 
	bagClass:setBagSlot(TRACKERS_MUSKET_OF_THE_BANDIT_1_1)
	self:assertEquals(DestroyusPro:isQualityAllowed(
	                  self.itemApi:getQuality()), true,
					  "Quality should be allowed!")
end

function ItemsTest.test_isItemDisenchantable(self) 
	_profApi:setFirstProf(ENCHANTING_LVL421_INDEX3)
	_profApi:setSecondProf(JEWELCRAFTING_LVL125_INDEX2)
	
    -- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  SUNSPRING_VEST_1_1["i"],
	                  SUNSPRING_VEST_1_1["j"]),
	                  true, "Sunspring Vest 106719")
					  
	-- Scorching Cloak 134634
	bagClass:setBagSlot(SCORCHING_CLOAK_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  SCORCHING_CLOAK_1_1["i"],
	                  SCORCHING_CLOAK_1_1["j"]),
	                  true, "Scorching Cloak 134634")
					  
	-- Monastic Ring of Stoicism  88638  UniqueId
	bagClass:setBagSlot(MONASTIC_RING_OF_STOICISM_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  MONASTIC_RING_OF_STOICISM_1_1["i"],
	                  MONASTIC_RING_OF_STOICISM_1_1["j"]),
	                  false, "Monastic Ring of Stoicism  88638  UniqueId")
					  
	-- Necropolis Breastplate 36397
	bagClass:setBagSlot(NECROPOLIS_BREASTPLATE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  NECROPOLIS_BREASTPLATE_1_1["i"],
	                  NECROPOLIS_BREASTPLATE_1_1["j"]),
	                  true, "Necropolis Breastplate 36397")
					  
	-- Belt of the Blood Nova 50015 Quality epic 
	bagClass:setBagSlot(BELT_OF_THE_BLOOD_NOVA_EPIC_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  BELT_OF_THE_BLOOD_NOVA_EPIC_1_1["i"],
	                  BELT_OF_THE_BLOOD_NOVA_EPIC_1_1["j"]),
	                  true, "Belt of the Blood Nova 50015 Quality epic") 
					  
	-- Jacinth Circle 11969
	bagClass:setBagSlot(JACINTH_CIRCLE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  JACINTH_CIRCLE_1_1["i"],
	                  JACINTH_CIRCLE_1_1["j"]),
	                  true, "Jacinth Circle 11969")
					  
	-- Hypnotist's Watch 25786 Trinket 225 ILVL
	bagClass:setBagSlot(HYPNOTISTS_WATCH_TRINKET_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  HYPNOTISTS_WATCH_TRINKET_1_1["i"],
	                  HYPNOTISTS_WATCH_TRINKET_1_1["j"]),
					  true, "Hypnotist's Watch 25786 Trinket 225 ILVL")
end

function ItemsTest.test_isItemDisenchantableLowLvlSkill(self) 
	_profApi:setFirstProf(JEWELCRAFTING_LVL125_INDEX2)
	_profApi:setSecondProf(ENCHANTING_LVL1_INDEX4)
	
	-- Sunspring Vest 106719
	bagClass:setBagSlot(SUNSPRING_VEST_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  SUNSPRING_VEST_1_1["i"],
	                  SUNSPRING_VEST_1_1["j"]),
	                  true, "Sunspring Vest 106719")
					  
	-- Scorching Cloak 134634
	bagClass:setBagSlot(SCORCHING_CLOAK_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  SCORCHING_CLOAK_1_1["i"],
	                  SCORCHING_CLOAK_1_1["j"]),
	                  true, "Scorching Cloak 134634")
					  
	-- Monastic Ring of Stoicism  88638  UniqueId
	bagClass:setBagSlot(MONASTIC_RING_OF_STOICISM_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  MONASTIC_RING_OF_STOICISM_1_1["i"],
	                  MONASTIC_RING_OF_STOICISM_1_1["j"]),
	                  false, "Monastic Ring of Stoicism  88638  UniqueId")

	-- Necropolis Breastplate 36397
	bagClass:setBagSlot(NECROPOLIS_BREASTPLATE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  NECROPOLIS_BREASTPLATE_1_1["i"],
	                  NECROPOLIS_BREASTPLATE_1_1["j"]),
	                  true, "Necropolis Breastplate 36397")
					  
	-- Belt of the Blood Nova 50015 Quality epic
	bagClass:setBagSlot(BELT_OF_THE_BLOOD_NOVA_EPIC_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  BELT_OF_THE_BLOOD_NOVA_EPIC_1_1["i"],
	                  BELT_OF_THE_BLOOD_NOVA_EPIC_1_1["j"]),
	                  true, "Belt of the Blood Nova 50015 Quality epic")
					  
	-- Jacinth Circle 11969
	bagClass:setBagSlot(JACINTH_CIRCLE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
				      JACINTH_CIRCLE_1_1["i"], 
					  JACINTH_CIRCLE_1_1["j"]),
	                  true, "Jacinth Circle 11969")
					  
	-- Vindicator's Iron Legguards 26033 1 lvl disenchanting and armor
	bagClass:setBagSlot(VINDICATORS_IRON_LEGGUARDS_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  VINDICATORS_IRON_LEGGUARDS_1_1["i"],
	                  VINDICATORS_IRON_LEGGUARDS_1_1["j"]),
	                  true, "Vindicator's Iron Legguards 26033 1 lvl disenchanting and armor")

	-- Crystallizing Mana 141292 1 lvl disenchanting
	bagClass:setBagSlot(CRYSTALLIZING_MANA_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  CRYSTALLIZING_MANA_1_1["i"],
	                  CRYSTALLIZING_MANA_1_1["j"]),
	                  true, "Crystallizing Mana 141292 1 lvl disenchanting")
					  
	-- Grisly Souvenir 140413 1 lvl disenchanting
	bagClass:setBagSlot(GRISLY_SOUVENIR_3_2)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  GRISLY_SOUVENIR_3_2["i"],
	                  GRISLY_SOUVENIR_3_2["j"]),
	                  true, "Grisly Souvenir 140413 1 lvl disenchanting rare")

	-- Felsoul Vortex 147112 1 lvl disenchanting
	bagClass:setBagSlot(FELSOUL_VORTEX_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  FELSOUL_VORTEX_1_1["i"],
	                  FELSOUL_VORTEX_1_1["j"]),
	                  true, "Felsoul Vortex 147112 1 lvl disenchanting epic")
end

function ItemsTest.test_isItemDisenchantableTranslateItems(self)
	_profApi:setFirstProf(JEWELCRAFTING_LVL125_INDEX2)
	_profApi:setSecondProf(ENCHANTING_LVL1_INDEX4)
	
	-- Emergency Hatchet 69223 One Hand Axe
	bagClass:setBagSlot(EMERGENCY_HATCHET_ONE_HAND_AXE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  EMERGENCY_HATCHET_ONE_HAND_AXE_1_1["i"],
	                  EMERGENCY_HATCHET_ONE_HAND_AXE_1_1["j"]),
	                  true, "Emergency Hatchet 69223 One Hand Axe")
					  
	-- Lupine Axe 1220 Two Hand Axe
	bagClass:setBagSlot(LUPIN_AXE_TWO_HAND_AXE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  LUPIN_AXE_TWO_HAND_AXE_1_1["i"],
	                  LUPIN_AXE_TWO_HAND_AXE_1_1["j"]),
	                  true, "Lupine Axe 1220 Two Hand Axe")
					  
	-- Shipwreck Bow 52646 Bow
	bagClass:setBagSlot(SHIPWRECK_BOW_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  SHIPWRECK_BOW_1_1["i"],
	                  SHIPWRECK_BOW_1_1["j"]),
	                  true, "Shipwreck Bow 52646 Bow")
					  
	-- Skorn's Rifle 3079 Gun
	bagClass:setBagSlot(SKORNS_RIFLE_GUN_1_1)	
	self:assertEquals(DestroyusPro:isItemDisenchantable(
				      SKORNS_RIFLE_GUN_1_1["i"],
	                  SKORNS_RIFLE_GUN_1_1["j"]),
	                  true, "Skorn's Rifle 3079 Gun")
					  
	--Gnoll Punisher 1214 One Hand Mace
	bagClass:setBagSlot(GNOLL_PUNISHER_ONE_HAND_MACE_1_1)	
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  GNOLL_PUNISHER_ONE_HAND_MACE_1_1["i"],
	                  GNOLL_PUNISHER_ONE_HAND_MACE_1_1["j"]),
					  true, "Gnoll Punisher 1214 One Hand Mace")
					  
	-- Codex Breaker 62878 Two Hand Mace
	bagClass:setBagSlot(CODEX_BREAKER_TWO_HAND_MACE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  CODEX_BREAKER_TWO_HAND_MACE_1_1["i"],
	                  CODEX_BREAKER_TWO_HAND_MACE_1_1["j"]),
					  true, "Codex Breaker 62878 Two Hand Mace")
					  
	-- Stunted Tree Root 57214 Polearms
	bagClass:setBagSlot(STUNTED_TREE_ROOT_POLEARMS_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
				      STUNTED_TREE_ROOT_POLEARMS_1_1["i"],
	                  STUNTED_TREE_ROOT_POLEARMS_1_1["j"]),
					  true, "Stunted Tree Root 57214 Polearms")
					  
	-- Sunwell Blade 22783 One Hand Sword
	bagClass:setBagSlot(SUNWELL_BLADE_ONE_HAND_SWORD_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
			          SUNWELL_BLADE_ONE_HAND_SWORD_1_1["i"],
	                  SUNWELL_BLADE_ONE_HAND_SWORD_1_1["j"]),
					  true, "Sunwell Blade 22783 One Hand Sword")
					  
	-- Brashclaw's Skewer 2204 Two hand Sword
	bagClass:setBagSlot(BRASHCLAWS_SKEWER_TWO_HAND_SWORD_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              BRASHCLAWS_SKEWER_TWO_HAND_SWORD_1_1["i"],
	                  BRASHCLAWS_SKEWER_TWO_HAND_SWORD_1_1["j"]),
					  true, "Brashclaw's Skewer 2204 Two hand Sword")
					  
	-- Gnarled Hermit's Staff 1539 Staff
	bagClass:setBagSlot(GNARLED_HERMITS_STAFF_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
	                  GNARLED_HERMITS_STAFF_1_1["i"],
	                  GNARLED_HERMITS_STAFF_1_1["j"]),
			          true, "Gnarled Hermit's Staff 1539 Staff")
					  
	-- Filthy Paw  58978 Fist Weapon
	bagClass:setBagSlot(FILTHY_PAW_FIST_WEAPON_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
			          FILTHY_PAW_FIST_WEAPON_1_1["i"],
	                  FILTHY_PAW_FIST_WEAPON_1_1["j"]),
	                  true, "Filthy Paw  58978 Fist Weapon")
					  
	-- Venom Web Fang  899 Dagger
	bagClass:setBagSlot(VENOM_WEB_FANG_DAGGER_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              VENOM_WEB_FANG_DAGGER_1_1["i"],
	                  VENOM_WEB_FANG_DAGGER_1_1["j"]),
	                  true, "Venom Web Fang  899 Dagger")
					  
	-- Shrewd Crossbow  59582 Xbow
	bagClass:setBagSlot(SHREWD_CROSSBOW_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
	                  SHREWD_CROSSBOW_1_1["i"],
	                  SHREWD_CROSSBOW_1_1["j"]),
	                  true, "Shrewd Crossbow  59582 Xbow")
					  
	-- Brazie's Plant Light 65689 Wand
	bagClass:setBagSlot(BRAZIES_PLANT_LIGHT_WAND_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
			          BRAZIES_PLANT_LIGHT_WAND_1_1["i"],
					  BRAZIES_PLANT_LIGHT_WAND_1_1["j"]),	
	                  true, "Brazie's Plant Light 65689 Wand")

	-- Glaive of the Fallen 141604 Warglaives
	bagClass:setBagSlot(GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1["i"],
					  GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1["j"]),	
	                  true, "Glaive of the Fallen 141604 Warglaives")
end

function ItemsTest.test_isItemDisenchantableTranslateArmor(self)
	_profApi:setFirstProf(JEWELCRAFTING_LVL125_INDEX2)
	_profApi:setSecondProf(ENCHANTING_LVL1_INDEX4)
	
	-- Blue Overalls 6263 Cloth Armor
	bagClass:setBagSlot(BLUE_OVERALLS_CLOTH_ARMOR_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              BLUE_OVERALLS_CLOTH_ARMOR_1_1["i"],
					  BLUE_OVERALLS_CLOTH_ARMOR_1_1["j"]),
					  true, "Blue Overalls 6263 Cloth Armor")
					  
	-- Dark Leather Tunic 2317 Leather Armor
	bagClass:setBagSlot(DARK_LEARHER_TUNIC_LEATHER_ARMOR_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
	                  DARK_LEARHER_TUNIC_LEATHER_ARMOR_1_1["i"],
					  DARK_LEARHER_TUNIC_LEATHER_ARMOR_1_1["j"]),
					  true, "Dark Leather Tunic 2317 Leather Armor")
					  
	-- Ironforge Breastplate 6731 Mail Armor
	bagClass:setBagSlot(IRONFORGE_BREASTPLATE_MAIL_ARMOR)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  IRONFORGE_BREASTPLATE_MAIL_ARMOR["i"],
					  IRONFORGE_BREASTPLATE_MAIL_ARMOR["j"]),
					  true, "Ironforge Breastplate 6731 Mail Armor")
					  
	-- Stabilized Incendicite Legguards 59066 Plate
	bagClass:setBagSlot(STABILIZED_INCENDICITE_LEGGUARDS_PLATE_ARMOR_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  STABILIZED_INCENDICITE_LEGGUARDS_PLATE_ARMOR_1_1["i"],
					  STABILIZED_INCENDICITE_LEGGUARDS_PLATE_ARMOR_1_1["j"]),
	                  true, "Stabilized Incendicite Legguards 59066 Plate")
					  
	-- Wheel of the Lost Hope 24334 Shield
	bagClass:setBagSlot(WHEEL_OF_THE_LOST_HOPE_SHIELD_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  WHEEL_OF_THE_LOST_HOPE_SHIELD_1_1["i"],
					  WHEEL_OF_THE_LOST_HOPE_SHIELD_1_1["j"]),
	                  true, "Wheel of the Lost Hope 24334 Shield")
					  
	-- Greenweave Cloak 9770 back
	bagClass:setBagSlot(GREENWAVE_CLOAK_BACK_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  GREENWAVE_CLOAK_BACK_1_1["i"],
					  GREENWAVE_CLOAK_BACK_1_1["j"]),
	                  true, "Greenweave Cloak 9770 back")
end

function ItemsTest.test_isItemDisenchantableTranslateMiscellaneous(self)
	_profApi:setFirstProf(JEWELCRAFTING_LVL125_INDEX2)
	_profApi:setSecondProf(ENCHANTING_LVL1_INDEX4)
	
	-- Malachite Pendant 25438 Neck
	bagClass:setBagSlot(MALACHITE_PENDANT_NECK_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
				      MALACHITE_PENDANT_NECK_1_1["i"],
					  MALACHITE_PENDANT_NECK_1_1["j"]),
	                  true, "Malachite Pendant 25438 Neck")
					  
	-- Windrunner Band 62892 Ring
	bagClass:setBagSlot(WINDRUNNER_BAND_RING_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
				      WINDRUNNER_BAND_RING_1_1["i"],
					  WINDRUNNER_BAND_RING_1_1["j"]),
	                  true, "Windrunner Band 62892 Ring")
					  
	-- Hypnotist's Watch 25786 Trinket 225 ILVL
	bagClass:setBagSlot(HYPNOTISTS_WATCH_TRINKET_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  HYPNOTISTS_WATCH_TRINKET_1_1["i"],
					  HYPNOTISTS_WATCH_TRINKET_1_1["j"]),
					  true, "Hypnotist's Watch 25786 Trinket 225 ILVL")
					  
	-- Glowing Murloc Eye 52652 Off Hand
	bagClass:setBagSlot(GLOWING_MURLOCK_EYE_OFF_HAND_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
			          GLOWING_MURLOCK_EYE_OFF_HAND_1_1["i"],
					  GLOWING_MURLOCK_EYE_OFF_HAND_1_1["j"]),
					  true, "Glowing Murloc Eye 52652 Off Hand")
end

function ItemsTest.test_isFoundBug(self)
	-- Fixed ugly bug which exist from the beginning of the addon. 
	-- Current fault appeared when Enchanting profession was learned 
	-- as primary and secondary profession had lover/higher skill level
    -- than primary or does not exist.
	_profApi:setFirstProf(ENCHANTING_LVL367_INDEX10)
	_profApi:setSecondProf(INSCRIPTION_LVL1_INDEX9)
	-- Hypnotist's Watch 25786 
	bagClass:setBagSlot(HYPNOTISTS_WATCH_TRINKET_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              HYPNOTISTS_WATCH_TRINKET_1_1["i"],
					  HYPNOTISTS_WATCH_TRINKET_1_1["j"]),
	                  true, "Hypnotist's Watch 25786 Bug Test")
end

function ItemsTest.test_isItemIlvlForUpgradedItems(self)
	bagClass:setBagSlot(KALDELAR_SABATONS_3_2)
	self.itemApi:setItemPos(KALDELAR_SABATONS_3_2["i"], KALDELAR_SABATONS_3_2["j"])
	self:assertEquals(self.itemApi:getILvl(), 805, "Ilvl of the item is different")
	self:assertEquals(self.itemApi:getQuality(), 3, "Quality of the item is different")
end

function ItemsTest.test_professionNotExist(self)
	_profApi:setFirstProf(UNKNOW_PROFESSIONS)
	_profApi:setSecondProf(UNKNOW_PROFESSIONS)
	-- Hypnotist's Watch 25786  
	bagClass:setBagSlot(HYPNOTISTS_WATCH_TRINKET_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
	                  HYPNOTISTS_WATCH_TRINKET_1_1["i"],
					  HYPNOTISTS_WATCH_TRINKET_1_1["j"]),
	                  false, "Hypnotist's Watch 25786 Bug Test")
end

function ItemsTest.test_isNotUniqueItem(self) 
	-- Monastic Ring of Stoicism  88638  UniqueId
	self:assertEquals(DestroyusPro:isNotUniqueItem(MONASTIC_RING_OF_STOICISM_1_1["itemId"],
                      DestroyusPro.itemsIdNotForDisenchant),
	                  false, "Monastic Ring of Stoicism  88638  UniqueId")
	self:assertEquals(DestroyusPro:isNotUniqueItem(SEASONED_HACKER_ONE_HAND_AXE_1_1["itemId"],
	                  DestroyusPro.itemsIdNotForDisenchant),
	                  false, "Item should be unqiue!") 
	self:assertEquals(DestroyusPro:isNotUniqueItem(SUNSPRING_VEST_1_1["itemId"],
	                  DestroyusPro.itemsIdNotForDisenchant),
	                  true, "Item should not be unique!")
end

function ItemsTest.test_isItemDisenchantableDifferentSettings(self)
	_profApi:setFirstProf(JEWELCRAFTING_LVL125_INDEX2)
	_profApi:setSecondProf(ENCHANTING_LVL421_INDEX3)
	
	-- Emergency Hatchet 69223 One Hand Axe
	local previousValue = _G["DestroyusProItemQualityValue"]["uncommon"]
	_G["DestroyusProItemQualityValue"]["uncommon"] = -1
	bagClass:setBagSlot(EMERGENCY_HATCHET_ONE_HAND_AXE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  EMERGENCY_HATCHET_ONE_HAND_AXE_1_1["i"],
	                  EMERGENCY_HATCHET_ONE_HAND_AXE_1_1["j"]),
	                  false, "Uncommon field is disabled. Should not disenchant")
	_G["DestroyusProItemQualityValue"]["uncommon"] = previousValue
	
	-- Shoulderplates of the Eternal 45320 Shoulder
	previousValue = _G["DestroyusProItemQualityValue"]["epic"]
	_G["DestroyusProItemQualityValue"]["epic"] = -1
	bagClass:setBagSlot(SHOULDERPLATES_OF_THE_ETERNAL_3_2)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              SHOULDERPLATES_OF_THE_ETERNAL_3_2["i"],
					  SHOULDERPLATES_OF_THE_ETERNAL_3_2["j"]),	
	                  false, "Epic field is disabled. Should not disenchant")
					  
	-- Felsoul Vortex 147112 Artifact Weapon
	bagClass:setBagSlot(FELSOUL_VORTEX_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              FELSOUL_VORTEX_1_1["i"],
					  FELSOUL_VORTEX_1_1["j"]),	
	                  false, "Epic field is disabled. Felsoul Vortex 147112 Artifact Weapon")
					  
	_G["DestroyusProItemQualityValue"]["epic"] = previousValue
	
	-- Glaive of the Fallen 141604 Warglaives
	previousValue = _G["DestroyusProItemQualityValue"]["rare"]
	_G["DestroyusProItemQualityValue"]["rare"] = -1
	bagClass:setBagSlot(GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1["i"],
					  GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1["j"]),	
	                  false, "Rare field is disabled. Should not disenchant")
	_G["DestroyusProItemQualityValue"]["rare"] = previousValue

	-- Emergency Hatchet 69223 One Hand Axe
	bagClass:setBagSlot(EMERGENCY_HATCHET_ONE_HAND_AXE_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
					  EMERGENCY_HATCHET_ONE_HAND_AXE_1_1["i"],
	                  EMERGENCY_HATCHET_ONE_HAND_AXE_1_1["j"]),
	                  true, "Uncommon field is enabled. Should  disenchant")
	
	-- Shoulderplates of the Eternal 45320 Shoulder
	bagClass:setBagSlot(SHOULDERPLATES_OF_THE_ETERNAL_3_2)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              SHOULDERPLATES_OF_THE_ETERNAL_3_2["i"],
					  SHOULDERPLATES_OF_THE_ETERNAL_3_2["j"]),	
	                  true, "Epic field is enabled. Should  disenchant")
	
	-- Glaive of the Fallen 141604 Warglaives
	bagClass:setBagSlot(GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1["i"],
					  GLAIVE_OF_THE_FALLEN_WARGLAIVES_1_1["j"]),	
	                  true, "Rare field is enabled. Should  disenchant")

	-- Felsoul Vortex 147112 Artifact Weapon
	bagClass:setBagSlot(FELSOUL_VORTEX_1_1)
	self:assertEquals(DestroyusPro:isItemDisenchantable(
		              FELSOUL_VORTEX_1_1["i"],
					  FELSOUL_VORTEX_1_1["j"]),	
	                  true, "Rare field is enabled. Felsoul Vortex 147112 Artifact Weapon")
end

-- DESTROYUSPRO MAIN TESTS
function createMacroTest(id, i, j)
	return string.format("/cast %s\n/use %d %d", _DPSpellApi(id):getSpellName(), i, j)
end

function itemFoundT(found, _i, _j, _link)
	return {itemFound=found, i=_i, j=_j, link=_link}
end

function DestroyusProMainTest.setUp(self)
	self.functionProfInfo = GetProfessionInfo
	self.functionGetProf = GetProfessions
	self.bagFunction =  GetContainerItemInfo
	self.GetContainerNumSlots = GetContainerNumSlots
	self.spellBookItemInfo = GetSpellBookItemInfo
	---
    GetContainerItemInfo = function(i, j) return bagClass:returnBagSlot(i, j) end
	GetContainerNumSlots = function(i) return bagClass:returnContainerNumSlot(i) end
	_profApi:setFirstProf(INSCRIPTION_LVL367_INDEX1)
	_profApi:setSecondProf(JEWELCRAFTING_LVL125_INDEX2)
	GetProfessions = function() return  _profApi:getProfessions() end
	GetProfessionInfo = function(index) return _profApi:getProfessionInfo(index) end
	GetSpellBookItemInfo = function(name) return _spellBook:haveSpellInBook(name) end
	_spellBook:setSpellBook(SPELLS_ALL_MILLING_PROSPECTING)
	
	-- Set Disenchant fields set for tests
	DestroyusProMain.changeQualityDisenchanting("uncommon", 2)
	DestroyusProMain.changeQualityDisenchanting("rare", 3)
	DestroyusProMain.changeQualityDisenchanting("epic", 4)
	
	
	-- Enable back all macro buttons
	ClickMeForWorkMilling:Enable()
	ClickMeForWorkDisenchanting:Enable()
	ClickMeForWorkProspecting:Enable()
	ClickMeForWorkMassMillingLegion:Enable()
	ClickMeForWorkMassProspectingLegion:Enable()
end

function DestroyusProMainTest.tearDown(self)
	GetProfessionInfo = self.functionProfInfo
	GetProfessions = self.functionGetProf
	GetContainerItemInfo = self.bagFunction
	GetContainerNumSlots = self.GetContainerNumSlots
	
	-- Set Disenchant fields set for tests
	DestroyusProMain.changeQualityDisenchanting("uncommon", 2)
	DestroyusProMain.changeQualityDisenchanting("rare", -1)
	DestroyusProMain.changeQualityDisenchanting("epic", -1)
	
	-- Enable back all macro buttons
	ClickMeForWorkMilling:Enable()
	ClickMeForWorkDisenchanting:Enable()
	ClickMeForWorkProspecting:Enable()
	ClickMeForWorkMassMillingLegion:Enable()
	ClickMeForWorkMassProspectingLegion:Enable()
end

function DestroyusProMainTest.test_createMacroInscription(self)
	-- Talandra's Rose found. profession lvl is ok. 
	bagClass:setBagSlot(TALANDRAS_ROSE_COUNT10_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMilling,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMilling)),
					  createMacroTest(_DPSpellApi.spellMilling,
	                  TALANDRAS_ROSE_COUNT10_3_12["i"],
					  TALANDRAS_ROSE_COUNT10_3_12["j"]),
					  "Macro have not proper string value")
					  
	-- Herb found Azshara's Veil 475. Low lvl profession
	bagClass:setBagSlot(AZSHARAS_VEIL_COUNT10_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMilling,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMilling)),
					  "", "Macro have not proper string value")
end

function DestroyusProMainTest.test_createMacroProspecting(self)
	-- Iron Ore found. profession lvl is ok. 
	bagClass:setBagSlot(IRON_ORE_COUNT10_0_20)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellProspecting,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellProspecting)),
					  createMacroTest(_DPSpellApi.spellProspecting,
		              IRON_ORE_COUNT10_0_20["i"], IRON_ORE_COUNT10_0_20["j"]),
					  "Macro have not proper string value")
					  
	--Obsidium Ore found. Low lvl profession.
	bagClass:setBagSlot(OBSIDIUM_ORE_COUNT10_0_20)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellProspecting,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellProspecting)),
					  "", "Macro have not proper string value")
end

function DestroyusProMainTest.test_createMacroEnchanting(self)
	_profApi:setFirstProf(ENCHANTING_LVL367_INDEX10)
	_profApi:setSecondProf(UNKNOW_PROFESSIONS)
	
	-- Item found. profession lvl is ok. frigid-war-mace id 36499
	bagClass:setBagSlot(FRIGID_WAR_MACE_ONE_HAND_MACE_2_14)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellDisenchant,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellDisenchant)),
					  createMacroTest(_DPSpellApi.spellDisenchant,
	                  FRIGID_WAR_MACE_ONE_HAND_MACE_2_14["i"],
				      FRIGID_WAR_MACE_ONE_HAND_MACE_2_14["j"]),
					  "Macro have not proper string value")
					  
	-- Item found. Low lvl profession. razorwind-mace id 55380
	bagClass:setBagSlot(RAZORWIND_MACE_ONE_HAND_MACE_1_1)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellDisenchant,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellDisenchant)),
					  createMacroTest(_DPSpellApi.spellDisenchant,
	                  RAZORWIND_MACE_ONE_HAND_MACE_1_1["i"],
				      RAZORWIND_MACE_ONE_HAND_MACE_1_1["j"]), "Macro have not proper string value")
end

function DestroyusProMainTest.test_createMacroInscriptionMassMillingLegion(self)
	--Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(AETHRIL_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillAethril,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillAethril)),
					  createMacroTest(_DPSpellApi.spellMassMillAethril,
	                  AETHRIL_COUNT20_3_12["i"], AETHRIL_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Aethril Mass Milling")
					  
	-- Herb found. Low stack size
	bagClass:setBagSlot(AETHRIL_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillAethril,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillAethril)),
					  "", "Macro should be empty. Aethril Mass Millinge")
					  
	--Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(DREAMLEAF_COUNT20_1_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillDreamleaf,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillDreamleaf)),
					  createMacroTest(_DPSpellApi.spellMassMillDreamleaf,
	                  DREAMLEAF_COUNT20_1_12["i"], DREAMLEAF_COUNT20_1_12["j"]),
					  "Macro have not proper string value. Dreamleaf Mass Milling")
					  
	-- Herb found. Low stack size
	bagClass:setBagSlot(DREAMLEAF_COUNT19_1_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillDreamleaf,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillDreamleaf)),
					  "", "Macro should be empty. Dreamleaf Mass Millinge")
					  
	--Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(FELWORT_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillFelwort,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillFelwort)),
					  createMacroTest(_DPSpellApi.spellMassMillFelwort,
	                  FELWORT_COUNT20_3_12["i"], FELWORT_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Felwort Mass Milling")
					  
	-- Herb found. Low stack size
	bagClass:setBagSlot(FELWORT_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillFelwort,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillFelwort)),
					  "", "Macro should be empty. Felwort Mass Millinge")
					  
	-- Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(FAJARNSKAGGL_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillFjarnskaggl,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillFjarnskaggl)),
					  createMacroTest(_DPSpellApi.spellMassMillFjarnskaggl,
	                  FAJARNSKAGGL_COUNT20_3_12["i"], FAJARNSKAGGL_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Fjarnskaggl Mass Milling")
					  
	-- Herb found. Low stack size
	bagClass:setBagSlot(FAJARNSKAGGL_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillFjarnskaggl,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillFjarnskaggl)),
					  "", "Macro should be empty. Fjarnskaggl Mass Millinge")
					  
	--Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(FOXFLOWER_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillFoxflower,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillFoxflower)),
					  createMacroTest(_DPSpellApi.spellMassMillFoxflower,
	                  FOXFLOWER_COUNT20_3_12["i"], FOXFLOWER_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Foxflower Mass Milling")
					  
	-- Herb found. Low stack size
	bagClass:setBagSlot(FOXFLOWER_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillFoxflower,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillFoxflower)),
					  "", "Macro should be empty. Foxflower Mass Millinge")
					  
	--Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillStarlightRose,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillStarlightRose)),
					  createMacroTest(_DPSpellApi.spellMassMillStarlightRose,
	                  STARLIGHT_ROSE_COUNT20_3_12["i"], STARLIGHT_ROSE_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Starlight Rose Mass Milling")
					  
	-- Herb found. Low stack size
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillStarlightRose,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillStarlightRose)),
					  "", "Macro should be empty. Starlight Rose Mass Millinge")
					  
	-- Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(YSERALLINE_SEED_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillYserallineSeeds,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillYserallineSeeds)),
					  createMacroTest(_DPSpellApi.spellMassMillYserallineSeeds,
	                  YSERALLINE_SEED_COUNT20_3_12["i"], YSERALLINE_SEED_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Yseralline Seed Mass Milling")
	
	-- Herb found. Low stack size
	bagClass:setBagSlot(YSERALLINE_SEED_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(
	                  _DPSpellApi.spellMassMillYserallineSeeds,
					  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillYserallineSeeds)),
					  "", "Macro should be empty. Yseralline Seed Mass Millinge")
					  
	-- Herb found. Stack size is enough. Speel exist in book
	bagClass:setBagSlot(ASTRAL_GLORY_COUNT20_3_10)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillAstralGlory,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillAstralGlory)),
					  createMacroTest(_DPSpellApi.spellMassMillAstralGlory,
	                  ASTRAL_GLORY_COUNT20_3_10["i"], ASTRAL_GLORY_COUNT20_3_10["j"]),
					  "Macro have not proper string value. Astral Glory Mass Milling")
	
	-- Herb found. Low stack size
	bagClass:setBagSlot(ASTRAL_GLORY_COUNT19_3_10)
	self:assertEquals(DestroyusProMain.createMacroUsage(
	                  _DPSpellApi.spellMassMillAstralGlory,
					  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillAstralGlory)),
					  "", "Macro should be empty. Astral Glory Mass Millinge")
end

function DestroyusProMainTest.test_createMacroMassMillSpellUnknow(self)
	_spellBook:setSpellBook(SPELLS_AETHRIL_YSERALL_FJARNSKAGGL_LEYSTONE_EMPYRIUM)
	
	-- Herb stack is enoug. Spell is unknown. Empty Macro
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT20_3_12)
	local result, spellId = DestroyusProMain.findItemToWorkMassProf(_DPProfessionApi.skilInscription)
	self:assertEquals(DestroyusProMain.createMacroUsage(spellId, result),
					  "", "Macro should be empty. Starlight Rose Mass Miling")
					  
	-- Yseralline Seed should be noticed. Stack Size ok. Full string Macro
	bagClass:setBagSlot(YSERALLINE_SEED_COUNT20_3_12)
	local result, spellId = DestroyusProMain.findItemToWorkMassProf(_DPProfessionApi.skilInscription)
	self:assertEquals(DestroyusProMain.createMacroUsage(spellId, result),
					  createMacroTest(_DPSpellApi.spellMassMillYserallineSeeds,
					  YSERALLINE_SEED_COUNT20_3_12["i"], YSERALLINE_SEED_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Yseralline Seed Mass Milling")
					  
end

function DestroyusProMainTest.test_createMacroInscriptionMassMillingLegionAnotherItem(self)
	-- Obsidium Ore should not be noticed. Stack Size ok. Empty Macro
	bagClass:setBagSlot(OBSIDIUM_ORE_COUNT20_0_20) 
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillAethril,
                      DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillAethril)),
					  "", "Macro should be empty. Obsidium Ore is not Millinge")
					  
	-- Obsidium Ore should not be noticed. Stack Size not ok. Empty Macro
	bagClass:setBagSlot(OBSIDIUM_ORE_COUNT19_0_20)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassMillAethril,
                      DestroyusProMain.findItemToWork(_DPSpellApi.spellMassMillAethril)),
					  "", "Macro should be empty. Obsidium Ore is not Millinge")						  
end

function DestroyusProMainTest.test_collectMassMillingSpell(self)
	-- Collecting learned Spells from spell books
	_spellBook:setSpellBook(SPELLS_AETHRIL_YSERALL_FJARNSKAGGL_LEYSTONE_EMPYRIUM)
	local expectedSpell = {_DPSpellApi.spellMassMillAethril, _DPSpellApi.spellMassMillFjarnskaggl,
						   _DPSpellApi.spellMassMillYserallineSeeds}
	self:assertShallowTableEquals(DestroyusProMain.collectMassWorkSpells(
	                              DestroyusProMain.spellsToFindMassMillLegion),
	                              expectedSpell, "Speels id's are different")
end

function DestroyusProMainTest.test_collectAllMassMillingSpell(self)
	-- Collecting learned Spells from spell books
	local expectedSpell =  {_DPSpellApi.spellMassMillAethril,
	_DPSpellApi.spellMassMillDreamleaf,
	_DPSpellApi.spellMassMillFelwort,
	_DPSpellApi.spellMassMillFjarnskaggl,
	_DPSpellApi.spellMassMillFoxflower,
	_DPSpellApi.spellMassMillStarlightRose,
	_DPSpellApi.spellMassMillYserallineSeeds,
	_DPSpellApi.spellMassMillAstralGlory}
	self:assertShallowTableEquals(DestroyusProMain.collectMassWorkSpells(
	                              DestroyusProMain.spellsToFindMassMillLegion),
	                              expectedSpell, "Speels id's are different")
end

function DestroyusProMainTest.test_findItemToWorkMassMillingLegionNotAllSpellExist(self)
	_spellBook:setSpellBook(SPELLS_AETHRIL_YSERALL_FJARNSKAGGL_LEYSTONE_EMPYRIUM)
	
	-- Aethril. Stack size ok. Spell exist in book		
	bagClass:setBagSlot(AETHRIL_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   AETHRIL_COUNT20_3_12["i"], AETHRIL_COUNT20_3_12["j"], AETHRIL_COUNT20_3_12["link"]),
			   "Item found should be true Aethril Mass Milling")
					 
	-- Starlight Rose. Stack size ok. Spell not exist in book						 
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Starlight Rose Mass Milling")					 
end

function DestroyusProMainTest.test_findItemToWorkMassMillingLegionAllSpellExist(self)
	-- Aethril. Stack size ok. Spell exist in book		
	bagClass:setBagSlot(AETHRIL_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   AETHRIL_COUNT20_3_12["i"], AETHRIL_COUNT20_3_12["j"], AETHRIL_COUNT20_3_12["link"]),
			   "Item found should be true Aethril Mass Milling")

	-- Starlight Rose. Stack size ok. Spell exist in book						 
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   STARLIGHT_ROSE_COUNT20_3_12["i"], STARLIGHT_ROSE_COUNT20_3_12["j"], STARLIGHT_ROSE_COUNT20_3_12["link"]),
			   "Item found should be true Starlight Rose Mass Milling")

	-- Starlight Rose. Stack size not ok. Spell exist in book						 
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT19_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Starlight Rose Mass Milling")

	-- Dreamleaf. Stack size ok. Spell exist in book					 
	bagClass:setBagSlot(DREAMLEAF_COUNT20_1_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   DREAMLEAF_COUNT20_1_12["i"], DREAMLEAF_COUNT20_1_12["j"], DREAMLEAF_COUNT20_1_12["link"]),
			   "Item found should be true Dreamleaf Mass Milling")

	-- Felwort. Stack size ok. Spell exist in book						 
	bagClass:setBagSlot(FELWORT_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   FELWORT_COUNT20_3_12["i"], FELWORT_COUNT20_3_12["j"], FELWORT_COUNT20_3_12["link"]),
			   "Item found should be true Felwort Mass Milling")
					 
	-- Fjarnskaggl. Stack size ok. Spell exist in book		
	bagClass:setBagSlot(FAJARNSKAGGL_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   FAJARNSKAGGL_COUNT20_3_12["i"], FAJARNSKAGGL_COUNT20_3_12["j"], FAJARNSKAGGL_COUNT20_3_12["link"]),
			   "Item found should be true Fjarnskaggl Mass Milling")
					 
	-- Foxflower. Stack size ok. Spell exist in book	
	bagClass:setBagSlot(FOXFLOWER_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   FOXFLOWER_COUNT20_3_12["i"], FOXFLOWER_COUNT20_3_12["j"], FOXFLOWER_COUNT20_3_12["link"]),
			   "Item found should be true Foxflower Mass Milling")
					 
	-- Yseralline Seed. Stack size ok. Spell exist in book						 
	bagClass:setBagSlot(YSERALLINE_SEED_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   YSERALLINE_SEED_COUNT20_3_12["i"], YSERALLINE_SEED_COUNT20_3_12["j"], YSERALLINE_SEED_COUNT20_3_12["link"]),
			   "Item found should be true Yseralline Seed Mass Milling")
			   
	-- Astral Glory. Stack size ok. Spell exist in book
	bagClass:setBagSlot(ASTRAL_GLORY_COUNT20_3_10)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(true,
			   ASTRAL_GLORY_COUNT20_3_10["i"], ASTRAL_GLORY_COUNT20_3_10["j"], ASTRAL_GLORY_COUNT20_3_10["link"]),
			   "Item found should be true Astral Glory Mass Milling")
end

function DestroyusProMainTest.test_findItemToWorkMassMillingLegionSpellsNotExist(self)
	_spellBook:setSpellBook(EMPTY_SPELL_BOOK)

	-- Aethril. Stack size ok. Spell not exist in book
	bagClass:setBagSlot(AETHRIL_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Aethril Mass Milling")					 
					 
	-- Starlight Rose. Stack size ok. Spell not exist in book						 
	bagClass:setBagSlot(STARLIGHT_ROSE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Starlight Rose Mass Milling")	

	-- Dreamleaf. Stack size ok. Spell not exist in book		
	bagClass:setBagSlot(DREAMLEAF_COUNT20_1_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Dreamleaf Mass Milling")
					 
	-- Felwort. Stack size ok. Spell not exist in book		
	bagClass:setBagSlot(FELWORT_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Felwort Mass Milling")
					 
	-- Fjarnskaggl. Stack size ok. Spell not exist in book	
	bagClass:setBagSlot(FAJARNSKAGGL_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Fjarnskaggl Mass Milling")	
				 
	-- Foxflower. Stack size ok. Spell not exist in book			 
	bagClass:setBagSlot(FOXFLOWER_COUNT20_3_12) 
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Foxflower Mass Milling")
			   
	-- Yseralline Seed. Stack size ok. Spell not exist in book			 
	bagClass:setBagSlot(YSERALLINE_SEED_COUNT20_3_12) 
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Yseralline Seed Mass Milling")
			   
	-- Astral Glory. Stack size ok. Spell not exist in book			 
	bagClass:setBagSlot(ASTRAL_GLORY_COUNT20_3_10) 
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skilInscription)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be false Astral Glory Mass Milling")
end

function DestroyusProMainTest.test_findItemToWorkMassProspectingLegionAllSpellExist(self)
	-- Leystone Ore. Spell for mass milling exist in spell book
	bagClass:setBagSlot(LEYSTONE_ORE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skillJewelcrafting)), itemFoundT(true, LEYSTONE_ORE_COUNT20_3_12["i"],
	           LEYSTONE_ORE_COUNT20_3_12["j"], LEYSTONE_ORE_COUNT20_3_12["link"]),
			   "Item found should be true Leystone Ore Mass Prospecting")	
			   
	-- Felslate Ore. Spell for mass milling exist in spell book
	bagClass:setBagSlot(FELSLATE_ORE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skillJewelcrafting)), itemFoundT(true, FELSLATE_ORE_COUNT20_3_12["i"],
	           FELSLATE_ORE_COUNT20_3_12["j"], FELSLATE_ORE_COUNT20_3_12["link"]),
			   "Item found should be true Felslate Ore Mass Prospecting")
			   
    -- Empyrium Ore. Spell for mass milling exist in spell book
	bagClass:setBagSlot(EPYRIUM_ORE_COUNT20_3_11)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skillJewelcrafting)), itemFoundT(true, EPYRIUM_ORE_COUNT20_3_11["i"],
	           EPYRIUM_ORE_COUNT20_3_11["j"], EPYRIUM_ORE_COUNT20_3_11["link"]),
			   "Item found should be true Empyrium Ore Mass Prospecting")	
end

function DestroyusProMainTest.test_findItemToWorkMassProspectingLegionNotAllSpellExist(self)
	_spellBook:setSpellBook(SPELLS_AETHRIL_YSERALL_FJARNSKAGGL_LEYSTONE_EMPYRIUM)
	-- Felslate Ore. Spell for mass milling not exist in spell book
	bagClass:setBagSlot(FELSLATE_ORE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skillJewelcrafting)), itemFoundT(false, 0, 0, _DPContainerItemApi.unknowLink),
			   "Item not found should be fasle Felslate Ore Mass Prospecting")
			   
	-- Leystone Ore. Spell for mass milling exist in spell book
	bagClass:setBagSlot(LEYSTONE_ORE_COUNT20_3_12)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skillJewelcrafting)), itemFoundT(true, LEYSTONE_ORE_COUNT20_3_12["i"],
	           LEYSTONE_ORE_COUNT20_3_12["j"], LEYSTONE_ORE_COUNT20_3_12["link"]),
			   "Item found should be true Leystone Ore Mass Prospecting")

	-- Empyrium Ore. Spell for mass milling exist in spell book
	bagClass:setBagSlot(EPYRIUM_ORE_COUNT20_3_11)
	self:assertShallowTableEquals(select(1,DestroyusProMain.findItemToWorkMassProf(
	           _DPProfessionApi.skillJewelcrafting)), itemFoundT(true, EPYRIUM_ORE_COUNT20_3_11["i"],
	           EPYRIUM_ORE_COUNT20_3_11["j"], EPYRIUM_ORE_COUNT20_3_11["link"]),
			   "Item found should be true Empyrium Ore Mass Prospecting")			   
end

function DestroyusProMainTest.test_collectMassProspectingSpell(self)
	-- Collect learned skills from inscription 
	_spellBook:setSpellBook(SPELLS_AETHRIL_YSERALL_FJARNSKAGGL_LEYSTONE_EMPYRIUM)
	self:assertShallowTableEquals(DestroyusProMain.collectMassWorkSpells(
	             DestroyusProMain.spellsToFindMassProspectingLegion),
				 {_DPSpellApi.spellMassProspecLeystone, _DPSpellApi.spellMassProspectEmpyrium}, "Speels id's are different")
end

function DestroyusProMainTest.test_createMacroJawelcraftingMassProspectingLegion(self)
	-- Leystone Ore found. Stack Size ok
	bagClass:setBagSlot(LEYSTONE_ORE_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspecLeystone,
                      DestroyusProMain.findItemToWork(_DPSpellApi.spellMassProspecLeystone)),
                      createMacroTest(_DPSpellApi.spellMassProspecLeystone,
					  LEYSTONE_ORE_COUNT20_3_12["i"], LEYSTONE_ORE_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Leystone Ore Prospecting")
					  
	-- Leystone Ore found. Low stack size
	bagClass:setBagSlot(LEYSTONE_ORE_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspecLeystone,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassProspecLeystone)),
					  "", "Macro should be empty. Leystone Ore Prospecting")
					  
	-- Felslate Ore found. Stack size ok
	bagClass:setBagSlot(FELSLATE_ORE_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspectFelslate,
					  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassProspectFelslate)),
					  createMacroTest(_DPSpellApi.spellMassProspectFelslate,
					  FELSLATE_ORE_COUNT20_3_12["i"], FELSLATE_ORE_COUNT20_3_12["j"]),
					  "Macro have not proper string value. Felslate Ore Prospecting")
					  
	-- Felslate Ore found. Low stack size
	bagClass:setBagSlot(FELSLATE_ORE_COUNT19_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspectFelslate,
                      DestroyusProMain.findItemToWork(_DPSpellApi.spellMassProspectFelslate)),
	                  "", "Macro should be empty. Felslate Ore Prospecting")
	
	-- Empyrium Ore found. Stack size ok
	bagClass:setBagSlot(EPYRIUM_ORE_COUNT20_3_11)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspectEmpyrium,
					  DestroyusProMain.findItemToWork(_DPSpellApi.spellMassProspectEmpyrium)),
					  createMacroTest(_DPSpellApi.spellMassProspectEmpyrium,
					  EPYRIUM_ORE_COUNT20_3_11["i"], EPYRIUM_ORE_COUNT20_3_11["j"]),
					  "Macro have not proper string value. Empyrium Ore Prospecting")
					  
	-- Empyrium Ore found. Low stack size
	bagClass:setBagSlot(EPYRIUM_ORE_COUNT19_3_11)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspectEmpyrium,
                      DestroyusProMain.findItemToWork(_DPSpellApi.spellMassProspectEmpyrium)),
	                  "", "Macro should be empty. Empyrium Ore Prospecting")
end

function DestroyusProMainTest.test_createMacroSpellUnknownMassProspecting(self)
	-- Felslate Ore stack is ok. Spell is unknown 
	_spellBook:setSpellBook(EMPTY_SPELL_BOOK)
	bagClass:setBagSlot(FELSLATE_ORE_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspectFelslate,
	                  select(1,DestroyusProMain.findItemToWorkMassProf(_DPProfessionApi.skillJewelcrafting))),
					  "", "Macro should be empty. Felslate Prospecting")
					  
	-- Leystone Ore stack is ok. Spell is unknown
	bagClass:setBagSlot(LEYSTONE_ORE_COUNT20_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspecLeystone,
	                  select(1,DestroyusProMain.findItemToWorkMassProf(_DPProfessionApi.skillJewelcrafting))),
					  "", "Macro should be empty. Leystone Prospecting")

	-- Empyrium Ore stack is ok. Spell is unknown
	bagClass:setBagSlot(EPYRIUM_ORE_COUNT20_3_11)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMassProspectEmpyrium,
	                  select(1,DestroyusProMain.findItemToWorkMassProf(_DPProfessionApi.skillJewelcrafting))),
					  "", "Macro should be empty. Empyrium Prospecting")
end

-- Looter tests
function DestroyusProMainTest.test_LootItemsAfterMacroRun(self)
	-- Talandra's Rose found. profession lvl is ok. 
	bagClass:setBagSlot(TALANDRAS_ROSE_COUNT10_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMilling,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMilling)),
					  createMacroTest(_DPSpellApi.spellMilling,
	                  TALANDRAS_ROSE_COUNT10_3_12["i"],
					  TALANDRAS_ROSE_COUNT10_3_12["j"]),
					  "Macro have not proper string value")
					  
	-- Spell has been casted. Disable all buttons
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, true,
					  "Item has been found, so work should be able start")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Until this time looting is not allowed")
	DestroyusProMain.DestroyusProLootAfterWork("Self", "UNIT_SPELLCAST_START", "player", _, _, _, _DPSpellApi.spellMilling)
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, false,
					  "All buttons should be disabled")
	
	-- Spellcast sucess. Proced work
	DestroyusProMain.DestroyusProLootAfterWork("Self", "UNIT_SPELLCAST_SUCCEEDED", "player", _, _, _, _DPSpellApi.spellMilling)
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, false,
					  "Item has been found, reset value")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, true,
					  "Item has been found, can loot from LootBag")
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, false,
					  "All buttons should be still disabled")
					  
	-- Open loot. Proced work
	DestroyusProMain.DestroyusProLootAfterWork("Self", "LOOT_OPENED", "player", _, _, _, _)
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, false,
					  "Item has been found, work is done")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Item has been found, work is done")
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, false,
					  "All buttons should be still disabled")
					  

	-- Close loot and rest data. 
	DestroyusProMain.DestroyusProLootAfterWork("Self", "LOOT_CLOSED", _, _, _, _, _)
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, false,
					  "Item has been found, work is done")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Item has been found, work is done")
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, true,
					  "All buttons should be still enabled")
end

function DestroyusProMainTest.test_LootItemsAfterMacroRunInterrupted(self)
	-- Talandra's Rose found. profession lvl is ok. 
	bagClass:setBagSlot(TALANDRAS_ROSE_COUNT10_3_12)
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMilling,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMilling)),
					  createMacroTest(_DPSpellApi.spellMilling,
	                  TALANDRAS_ROSE_COUNT10_3_12["i"],
					  TALANDRAS_ROSE_COUNT10_3_12["j"]),
					  "Macro have not proper string value")
					  
	-- Spell has been casted. Disable all buttons
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, true,
					  "Item has been found, so work should be able start")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Until this time looting is not allowed")
	DestroyusProMain.DestroyusProLootAfterWork("Self", "UNIT_SPELLCAST_START", "player", _, _, _, _DPSpellApi.spellMilling)
	
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, false,
					  "All buttons should be disabled")
	
	-- Spellcast sucess. Proced work
	DestroyusProMain.DestroyusProLootAfterWork("Self", "UNIT_SPELLCAST_FAILED", "player", _, _, _, _DPSpellApi.spellMilling)
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, false,
					  "Spell has been interrupted, work is done")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Spell has been interrupted, work is done")
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, true,
					  "All buttons should be still enabled")
	
    --	Another case for the event UNIT_SPELLCAST_INTERRUPTED
	self:assertEquals(DestroyusProMain.createMacroUsage(_DPSpellApi.spellMilling,
	                  DestroyusProMain.findItemToWork(_DPSpellApi.spellMilling)),
					  createMacroTest(_DPSpellApi.spellMilling,
	                  TALANDRAS_ROSE_COUNT10_3_12["i"],
					  TALANDRAS_ROSE_COUNT10_3_12["j"]),
					  "Macro have not proper string value")
					  
	-- Spell has been casted. Disable all buttons
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, true,
					  "Item has been found, so work should be able start")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Until this time looting is not allowed")
	DestroyusProMain.DestroyusProLootAfterWork("Self", "UNIT_SPELLCAST_START", "player", _, _, _, _DPSpellApi.spellMilling)
	
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, false,
					  "All buttons should be disabled")
	
	-- Spellcast sucess. Proced work
	DestroyusProMain.DestroyusProLootAfterWork("Self", "UNIT_SPELLCAST_INTERRUPTED", "player", _, _, _, _DPSpellApi.spellMilling)
	self:assertEquals(DestroyusProMain.itemFoundAndWorkStarted, false,
					  "Spell has been interrupted, work is done")
	self:assertEquals(DestroyusProMain.canLootItemsAfterWork, false,
					  "Spell has been interrupted, work is done")
	self:assertEquals(DestroyusProMain.buttonCanWorkEnable, true,
					  "All buttons should be still enabled")
end

function DestroyusProMainTest.test_DestroyusProLootAfterIsDPSkill(self)
      -- Check if spells are for the DP during spell events 
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMilling), true, "spellMilling for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellDisenchant), true, "spellDisenchant for Loot") 
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellProspecting), true, "spellProspecting for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillAethril), true, "spellMassMillAethril for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillDreamleaf), true, "spellMassMillDreamleaf for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillFelwort), true, "spellMassMillFelwort for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillFjarnskaggl), true, "spellMassMillFjarnskaggl for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillFoxflower), true, "spellMassMillFoxflower for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillStarlightRose), true, "spellMassMillStarlightRose for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillYserallineSeeds), true, "spellMassMillYserallineSeeds for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassMillAstralGlory), true, "spellMassMillAstralGlory for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassProspectFelslate), true, "spellMassProspectFelslate for Loot")
      self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassProspecLeystone), true, "spellMassProspecLeystone for Loot")
	  self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.spellMassProspectEmpyrium), true, "spellMassProspectEmpyrium for Loot")
	  self:assertEquals(DestroyusProMain.DestroyusProLootAfterIsDPSkill(_DPSpellApi.unknowSpellName), false, "unknowSpellName for Loot")
end


local DestroyusProSettingsTest = {}
DestroyusProSettingsTest.__index = DestroyusProSettingsTest

setmetatable(DestroyusProSettingsTest, {
  __index = UnitTest, -- this is what makes the inheritance work
  __call = function (cls)
    local self = setmetatable({}, cls)
    UnitTest:_init(self)
    return self
  end
})

function DestroyusProSettingsTest.setUp(self)
	-- Remove all Macros from game
	-- For the test purpose
	DestroyusProMain.macroMgr:macroCD(_DPMacroManager.DELETED,
	_DPMacroManager.DISENCHANTUS, _DPMacroManager.BLACK_MAGIC_HAND,
	DestroyusProMain.buttonNameDisenchanting)
	
	DestroyusProMain.macroMgr:macroCD(_DPMacroManager.DELETED,
	_DPMacroManager.MILLINGUS, _DPMacroManager.RED_BOWL,
	DestroyusProMain.buttonNameMilling)
	
    DestroyusProMain.macroMgr:macroCD(_DPMacroManager.DELETED,
	_DPMacroManager.MASSPROSPECTINGUS, _DPMacroManager.GROUP_OF_SMALL_ROCKS,
	DestroyusProMain.buttonNameMassProspectingLegion)
	
	DestroyusProMain.macroMgr:macroCD(_DPMacroManager.DELETED,
	_DPMacroManager.MASSMILLINGUS, _DPMacroManager.BLUE_BOWL,
	DestroyusProMain.buttonNameMillingLegion)

	DestroyusProMain.macroMgr:macroCD(_DPMacroManager.DELETED,
	_DPMacroManager.PROSPECTINGUS, _DPMacroManager.SMALL_ROCK,
	DestroyusProMain.buttonNameProspecting)
	
	-- Hide frame if is visible
	_DPMainSettingsFrame:Hide()
end

function DestroyusProSettingsTest.tearDown(self)
	_DPMainSettingsFrame:Hide()
	_DPMainSettingsFrame:Show()
	_DPMainSettingsFrame:Hide()
end

function DestroyusProSettingsTest.test_checkboxDisenchantusValue(self)
	DestroyusProMain.changeQualityDisenchanting("uncommon", DestroyusProApi.strToNumQuality["uncommon"])
	DestroyusProMain.changeQualityDisenchanting("rare", DestroyusProApi.strToNumQuality["rare"])
	DestroyusProMain.changeQualityDisenchanting("epic", DestroyusProApi.strToNumQuality["epic"])
	_DPMainSettingsFrame:Show()
	-- Check if Uncommon quality for disenchantus was set
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["UncommonQualityS"]:GetChecked(),
	                  true, "Uncommon quality has not been checked")
					  
	-- Check if Rare quality for disenchantus was set
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["RareQualityS"]:GetChecked(),
	                  true, "Rare quality has not been checked")
					  
	-- Check if Epic quality for disenchantus was set
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["EpicQualityS"]:GetChecked(),
	                  true, "Epic quality has not been checked")
	
    -- Check if disabled is working 	
	_DPMainSettingsFrame:Hide()
	DestroyusProMain.changeQualityDisenchanting("uncommon", DestroyusProApi.unknowNumber)
	DestroyusProMain.changeQualityDisenchanting("rare", DestroyusProApi.unknowNumber)
	DestroyusProMain.changeQualityDisenchanting("epic", DestroyusProApi.unknowNumber)
	_DPMainSettingsFrame:Show()	
	-- Check if Uncommon quality for disenchantus was set
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["UncommonQualityS"]:GetChecked(),
	                  false, "Uncommon quality has been checked")
					  
	-- Check if Rare quality for disenchantus was set
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["RareQualityS"]:GetChecked(),
	                  false, "Rare quality has been checked")
					  
	-- Check if Epic quality for disenchantus was set
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["EpicQualityS"]:GetChecked(),
	                  false, "Epic quality has been checked")
end

function DestroyusProSettingsTest.test_checkboxDisenchantusOnClick(self)
	DestroyusProMain.changeQualityDisenchanting("uncommon", DestroyusProApi.unknowNumber)
	DestroyusProMain.changeQualityDisenchanting("rare", DestroyusProApi.unknowNumber)
	DestroyusProMain.changeQualityDisenchanting("epic", DestroyusProApi.unknowNumber)
	_DPMainSettingsFrame:Show()
	
	-- Trigger CheckButton Uncommon
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["UncommonQualityS"]:SetChecked(true)
	_DPSettingsManagerUI.Uncommon_OnClick(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["UncommonQualityS"], "OnClick")
	self:assertEquals(_G["DestroyusProItemQualityValue"]["uncommon"],
	                  DestroyusProApi.strToNumQuality["uncommon"], "Uncommon quality has not change value. Should be 2")

	-- Trigger CheckButton Uncommon
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["UncommonQualityS"]:SetChecked(false)
	_DPSettingsManagerUI.Uncommon_OnClick(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["UncommonQualityS"], "OnClick")
	self:assertEquals(_G["DestroyusProItemQualityValue"]["uncommon"],
	                  DestroyusProApi.unknowNumber, "Uncommon quality has not change value. Should be -1")

	-- Trigger CheckButton Rare
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["RareQualityS"]:SetChecked(true)
	_DPSettingsManagerUI.Rare_OnClick(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["RareQualityS"], "OnClick")
	self:assertEquals(_G["DestroyusProItemQualityValue"]["rare"],
	                  DestroyusProApi.strToNumQuality["rare"], "Rare quality has not change value. Should be 3")

	-- Trigger CheckButton Rare
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["RareQualityS"]:SetChecked(false)
	_DPSettingsManagerUI.Rare_OnClick(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["RareQualityS"], "OnClick")
	self:assertEquals(_G["DestroyusProItemQualityValue"]["rare"],
	                  DestroyusProApi.unknowNumber, "Rare quality has not change value. Should be -1")
					  
	-- Trigger CheckButton Epic
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["EpicQualityS"]:SetChecked(true)
	_DPSettingsManagerUI.Epic_OnClick(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["EpicQualityS"], "OnClick")
	self:assertEquals(_G["DestroyusProItemQualityValue"]["epic"],
	                  DestroyusProApi.strToNumQuality["epic"], "Rare quality has not change value. Should be 4")

	-- Trigger CheckButton Epic
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["EpicQualityS"]:SetChecked(false)
	_DPSettingsManagerUI.Epic_OnClick(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["EpicQualityS"], "OnClick")
	self:assertEquals(_G["DestroyusProItemQualityValue"]["epic"],
	                  DestroyusProApi.unknowNumber, "Rare quality has not change value. Should be -1")
end

function DestroyusProSettingsTest.test_checkboxPrintoutOnClick(self)
	DestroyusProMain.changeStateForPrintout(false)
	_DPMainSettingsFrame:Show()
	
	-- Check if disabling printout is positive
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"]:SetChecked(false)
	_DPSettingsManagerUI.changeStatePrintout_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"], "OnClick")
	self:assertEquals(_G["DestroyusProPrintOutState"], true, 
	                  "Printout should be avaiable - true")
					  
	-- Check if disabling printout is negative
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"]:SetChecked(true)
	_DPSettingsManagerUI.changeStatePrintout_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"], "OnClick")
	self:assertEquals(_G["DestroyusProPrintOutState"], false, 
	                  "Printout should be disabled - false")
					  
	-- Check if disabling printout is negative. Back to previous version
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"]:SetChecked(false)
	_DPSettingsManagerUI.changeStatePrintout_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"], "OnClick")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"]:GetChecked(),
	                  false, "Printout should be true. Checked is negative")
	self:assertEquals(_G["DestroyusProPrintOutState"], true, 
	                  "Printout should be avaiable - true")
end

function DestroyusProSettingsTest.test_checkboxPrintoutValue(self)
	DestroyusProMain.changeStateForPrintout(false)
	_DPMainSettingsFrame:Show()
	-- Check if disabling printout is positive
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"]:GetChecked(),
	                  true, "Printout should be false. Checked is positive")
					  
	
    -- Check if disabled is working 	
	_DPMainSettingsFrame:Hide()
	DestroyusProMain.changeStateForPrintout(true)
	_DPMainSettingsFrame:Show()	
	
	-- Check if disabling printout is negative
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["DisablePrintout"]:GetChecked(),
	                  false, "Printout should be true. Checked is negative")
end

function DestroyusProSettingsTest.test_checkboxMacros(self)
    -- Triger Macro Event
	-- Check if all macro buttons are disabled/not created
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Millingus"]:GetChecked(),
	                  false, "Millingus macro should not exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassMillingus"]:GetChecked(),
	                  false, "MassMillingus macro should not exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Disenchantus"]:GetChecked(),
	                  false, "Disenchantus macro should not exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"]:GetChecked(),
	                  false, "Prospectingus macro should not exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassProspectingus"]:GetChecked(),
	                  false, "MassProspectingus macro should not exist")					  
	
	-- Create all Macros for test
	DestroyusProMain.macroMgr:macroCD(_DPMacroManager.CREATED, 
    _DPMacroManager.MASSPROSPECTINGUS, _DPMacroManager.GROUP_OF_SMALL_ROCKS,
    DestroyusProMain.buttonNameMassProspectingLegion)

    DestroyusProMain.macroMgr:macroCD(_DPMacroManager.CREATED, 
    _DPMacroManager.MASSMILLINGUS, _DPMacroManager.BLUE_BOWL,
    DestroyusProMain.buttonNameMillingLegion)

    DestroyusProMain.macroMgr:macroCD(_DPMacroManager.CREATED, 
    _DPMacroManager.PROSPECTINGUS, _DPMacroManager.SMALL_ROCK,
    DestroyusProMain.buttonNameProspecting)

    DestroyusProMain.macroMgr:macroCD(_DPMacroManager.CREATED, 
    _DPMacroManager.MILLINGUS, _DPMacroManager.RED_BOWL,
    DestroyusProMain.buttonNameMilling)

    DestroyusProMain.macroMgr:macroCD(_DPMacroManager.CREATED, 
    _DPMacroManager.DISENCHANTUS, _DPMacroManager.BLACK_MAGIC_HAND,
    DestroyusProMain.buttonNameDisenchanting)
	
	-- Triger Macro Event
	-- Check if all macro buttons are disabled/not created
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Millingus"]:GetChecked(),
	                  true, "Millingus macro should exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassMillingus"]:GetChecked(),
	                  true, "MassMillingus macro should exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Disenchantus"]:GetChecked(),
	                  true, "Disenchantus macro should exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"]:GetChecked(),
	                  true, "Prospectingus macro should exist")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassProspectingus"]:GetChecked(),
	                  true, "MassProspectingus macro should exist")
				  
end

function DestroyusProSettingsTest.test_checkboxMillingusOnClick(self)
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Millingus"]:SetChecked(true)
	_DPSettingsManagerUI.Miling_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Millingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.MILLINGUS), true, 
	                  "Millingus Macro should be created")
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Millingus"]:SetChecked(false)
	_DPSettingsManagerUI.Miling_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Millingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.MILLINGUS), false, 
	                  "Millingus Macro should be deleted")					  
end

function DestroyusProSettingsTest.test_checkboxMassMillingusOnClick(self)
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassMillingus"]:SetChecked(true)
	_DPSettingsManagerUI.MassMilingLegion_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassMillingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.MASSMILLINGUS), true, 
	                  "MassMillingus Macro should be created")
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassMillingus"]:SetChecked(false)
	_DPSettingsManagerUI.MassMilingLegion_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassMillingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.MASSMILLINGUS), false, 
	                  "MassMillingus Macro should be deleted")					  
end

function DestroyusProSettingsTest.test_checkboxMassProspectingusOnClick(self)
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassProspectingus"]:SetChecked(true)
	_DPSettingsManagerUI.MassProspectingLegion_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassProspectingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.MASSPROSPECTINGUS), true, 
	                  "MassProspectingus Macro should be created")
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassProspectingus"]:SetChecked(false)
	_DPSettingsManagerUI.MassProspectingLegion_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["MassProspectingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.MASSPROSPECTINGUS), false, 
	                  "MassProspectingus Macro should be deleted")					  
end

function DestroyusProSettingsTest.test_checkboxDisenchantusOnClick(self)
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Disenchantus"]:SetChecked(true)
	_DPSettingsManagerUI.Disenchanting_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Disenchantus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.DISENCHANTUS), true, 
	                  "Disenchantus Macro should be created")
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Disenchantus"]:SetChecked(false)
	_DPSettingsManagerUI.Disenchanting_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Disenchantus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.DISENCHANTUS), false, 
	                  "Disenchantus Macro should be deleted")					  
end

function DestroyusProSettingsTest.test_checkboxProspectingusOnClick(self)
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"]:SetChecked(true)
	_DPSettingsManagerUI.Prospecting_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.PROSPECTINGUS), true, 
	                  "Prospectingus Macro should be created")
	-- Trigger button press
	_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"]:SetChecked(false)
	_DPSettingsManagerUI.Prospecting_OnClick(
	                    _DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"], "OnClick")
	self:assertEquals(DestroyusProMain.macroMgr:macroExist(_DPMacroManager.PROSPECTINGUS), false, 
	                  "Prospectingus Macro should be deleted")
	self:assertEquals(_DPMainSettingsFrame._DPOptionFrame.panel.checkButtons["Prospectingus"]:GetChecked(),
	                  false, "Prospectingus macro should exist")					  
end

function RunTest()
	utr = UnitTestRunner()
	utr:with(HerbsAndOreTest())
	utr:with(ItemsTest())
	utr:with(DestroyusProMainTest())
	utr:with(DestroyusProSettingsTest())
	utr:run()
end