local addonName, addonData = ...;
local events = {};
local shortageList = {};

-- Variables
local EMISSARY_COUNT = 3;
local WIDTH = 420;
local HEIGHT = 360;
local PADDING = 10;
local PADDING_TOP = 22;
local WIDTH_THIRD = WIDTH / 3 - PADDING;
local HEIGHT_NAV = 25;

local MAX_QUESTS = 25;

local FONT_SIZES = {
	small = 10,
	medium = 12
};

local LEVELING_DEFAULT = format(addonData.L.Vanilla, GetMapNameByID(13), GetMapNameByID(14));
local prestigeIconPath = "Interface\\PVPFrame\\Icons\\prestige-icon-";

-- Settings
local settingsFrame = CreateFrame("Frame", "DLUSettingsFrame");
local eventFrame = CreateFrame("Frame");

-- Notification System
CreateFrame("Frame", "DLUNotificationFrame");
DLUNotificationFrame:SetAlpha(0);
DLUNotificationFrame:SetWidth(320);
DLUNotificationFrame:SetHeight(80);
DLUNotificationFrame:SetPoint("TOP", UIParent, "TOP", 0, 0);
DLUNotificationFrame:CreateFontString("DLUNotification", "ARTWORK", "GameFontNormalLarge");
DLUNotification:SetPoint("CENTER", 0, 0);

-- Change Character UI
hooksecurefunc("PaperDollFrame_UpdateStats", function()
	DLU.showStats("player");
end)

-- Global UI functions
function DLU.getEmissaryValue(id)
	local currentEmissary = DLU.emissaryTable[id];
	local value;
	
	if (DLU.player.level < 110) then
		value = "Not max level yet...";
	elseif (currentEmissary == nil) then
		value = "Already completed";
	elseif (currentEmissary.completed == currentEmissary.required) then
		value = "Ready to turn-in";
	else
		value = format("%i/%i", currentEmissary.completed, currentEmissary.required);
	end
	
	return value;
end

function DLU.updateEmissaries()
	DLU.emissaryTable = DLU.getEmissaries();
	local amount = #DLU.emissaryTable;
	
	DLU.resetEmissaryUi();
	
	for i = 1, amount do
		local value = DLU.getEmissaryValue(i);
		
		if (DLUHQ.Panels.WQ.Emissaries[i].Label:GetText() == nil) then
			if (DLU.emissaryTable[i] ~= nil) then
				local currentEmissary = DLU.emissaryTable[i];
				
				if (DLUHQ.Panels.WQ.Emissaries[i].questId == 0) then
					DLUHQ.Panels.WQ.Emissaries[i].questId = currentEmissary.questId;
				end
				
				DLUHQ.Panels.WQ.Emissaries[i].Label:SetText(DLU.iconToText(currentEmissary.icon));
				DLUHQ.Panels.WQ.Emissaries[i].Tooltip = format("%s", currentEmissary.factionName);
			else
				DLUHQ.Panels.WQ.Emissaries[i].Label:SetText("");
				DLUHQ.Panels.WQ.Emissaries[i].Tooltip = format("Already completed");
			end
		end
		
		DLUHQ.Panels.WQ.Emissaries[i].Value:SetText(value);
	end
end

function DLU.resetEmissaryUi()
	for i = 1, EMISSARY_COUNT do
		local currentEmissary = DLUHQ.Panels.WQ.Emissaries[i];
		currentEmissary.Label:SetText("");
		currentEmissary.Value:SetText("Already completed");
		currentEmissary.Tooltip = "Already completed";
		currentEmissary.questId = 0;
	end
end

function DLU.getQuestTimeLeft(questId)
	local timeLeftMinutes = C_TaskQuest.GetQuestTimeLeftMinutes(questId);
	local timeString = format("Timeleft: %i", timeLeftMinutes);
	
	if (timeLeftMinutes ~= nil) then
		if (timeLeftMinutes < 60) then
			timeString = format("Minutes left: %i", SecondsToTime(timeLeftMinutes * 60));
		elseif (timeLeftMinutes < 24 * 60) then
			timeString = format("Hours left: %i", floor(timeLeftMinutes / 60));
		else
			timeString = format("Days left: %i", floor(timeLeftMinutes) / 1440);
		end
	end
	
	return timeString;
end

-- UI related functions
local function achievementCompleted(achievementId)
	local _, _, _, _, _, _, _, _, _, _, _, _, wasEarnedByMe = GetAchievementInfo(achievementId);
	return wasEarnedByMe;
end

local function addToTracker(achievementId, button)
	if (achievementCompleted(achievementId)) then
		RemoveTrackedAchievement(achievementId);
		if (button ~= nil) then
			disableButton(button, "Already completed!");
		end
	else
		AddTrackedAchievement(achievementId);
	end
end

local function removeFromTracker(achievementId)
	local isTracked = IsTrackedAchievement(achievementId);
	if (isTracked) then
		RemoveTrackedAchievement(achievementId);
	end
end

local function OnEnter(self)
	GameTooltip:SetOwner(self, "ANCHOR_CURSOR");
	GameTooltip:SetText(self.Tooltip);
	
	if (self.questId > 0) then
		local timeString = DLU.getQuestTimeLeft(self.questId);
		GameTooltip:AddLine(format("%s", timeString));
	end
	
	local selfValue = self.Value:GetText();
	local completed, required = selfValue:match("(%d+)/(%d+)");
	
	if (completed and required) then
		GameTooltip:AddLine(format("Completed: %i/%i", completed, required), 1, 1, 1);
	else
		GameTooltip:AddLine(self.Value:GetText(), 1, 1, 1);
	end
	
	GameTooltip:Show();
end

local function OnLeave(self)
	GameTooltip:Hide()
end

-- Shortage
local function checkShortage(dID, classRolesAvailable)
	for i = 1, LFG_ROLE_NUM_SHORTAGE_TYPES do
		local eligible, tank, healer, damage, itemCount = GetLFGRoleShortageRewards(dID, i);
		local tankLocked, healerLocked, damageLocked = GetLFDRoleRestrictions(dID);
		tank = tank and classRolesAvailable.tank and not tankLocked;
		healer = healer and classRolesAvailable.heal and not healerLocked;
		damage = damage and classRolesAvailable.dps and not damageLocked;
		if(eligible and itemCount > 0 and (tank or healer or damage)) then
			local data = {dID = dID, tank = tank, heal = healer, dps = damage};
			return data;
		end
	end
	
	return nil;
end

local function createShortageList()
	shortageList = {};
	
	local canTank, canHealer, canDamage = C_LFGList.GetAvailableRoles()
	local classRolesAvailable = {tank = canTank, heal = canHealer, dps = canDamage};
	
	for i = 1, GetNumRFDungeons() do
		local dID = GetRFDungeonInfo(i);
		local shortage = checkShortage(dID, classRolesAvailable);
		table.insert(shortageList, shortage);
	end
	
	for i = 1, GetNumRandomDungeons() do
        local dID = GetLFGRandomDungeonInfo(i)
		local shortage = checkShortage(dID, classRolesAvailable);
		table.insert(shortageList, shortage);
    end
	
	for i = 1, #DLUHQ.Panels.LFR.ShortageContainer.Content.lines do
		DLUHQ.Panels.LFR.ShortageContainer.Content.lines[i].text:SetText("");
	end
end

-- UI functions
local uniquealyzer = 1;
local checkButtons = {};

local function createCheckbutton(parent, x_loc, y_loc, displayname, option)
	uniquealyzer = uniquealyzer + 1;
	
	local checkbutton = CreateFrame("CheckButton", "checkbutton_0" .. uniquealyzer, parent, "ChatConfigCheckButtonTemplate");
	checkbutton:ClearAllPoints()
	checkbutton:SetPoint("TOPLEFT", x_loc, y_loc);
	getglobal(checkbutton:GetName() .. 'Text'):SetText(displayname);
	
	-- Option checkbox
	if (option) then
		checkButtons[option] = checkbutton;
	end

	return checkbutton;
end

function DLU.createTextRoot(text, alignment, parent, x_loc, y_loc, fontType)
	local title = parent:CreateFontString(nil, "ARTWORK", fontType);
	title:SetPoint(alignment, parent, alignment, x_loc, y_loc);
	title:SetText(text);
	return title;
end

function DLU.createTextChild(text, parent, below, x_loc, y_loc, fontType)
	local textChildNode = parent:CreateFontString(nil, "ARTWORK", fontType);
	textChildNode:SetPoint("TOPLEFT", below, "BOTTOMLEFT", x_loc, y_loc);
	textChildNode:SetText(text);
	return textChildNode;
end

local function createButton(parent, text, width, height, tooltip, x_loc, y_loc)
	local button = CreateFrame("Button", nil, parent, "UIPanelButtonTemplate");
	button:SetPoint("TOPLEFT", parent, "BOTTOMLEFT", x_loc, y_loc);
    button:SetText(text);
    button.tooltipText = tooltip;
    button:SetWidth(width);
    button:SetHeight(height);
	return button;
end

local function disableButton(button, text)
	button:SetText(text);
	button:Disable();
end

local function updateHonorValue()
	local honorLeft, honorLevel, honorMax = DLU.honorLeftValues();
	
	local valueText = honorLeft;
	
	if (honorLevel == honorMax) then
		if (CanPrestige()) then
			valueText = PVP_HONOR_PRESTIGE_AVAILABLE;
		else
			valueText = MAX_HONOR_LEVEL;
		end
	end
	
	DLUHQ.Panels.Character.PvPBar:SetValue(UnitHonor("player"));
	DLUHQ.Panels.Character.PvPBar.Label:SetText(valueText);
	--DLUHQ.Panels.Character.PvP.Value:SetText(valueText);
end

local function updateHonorLabel()
	local _, honorLevel, honorMax, prestigeLevel = DLU.honorLeftValues();
	local honorLevelNext = honorLevel+1;
	local text = "";
	
	if (honorLevel ~= honorMax) then
		text = "Honor needed for honor level %i";
		if (prestigeLevel > 0) then
			text = text .. " (Prestige %i%s)";
		end
		
		text = text .. ":";
	end
	
	print(prestigeIconPath .. prestigeLevel);
	
	DLUHQ.Panels.Character.PvP.Label:SetText(format(text, honorLevelNext, prestigeLevel, DLU.iconToText(prestigeIconPath .. prestigeLevel)));
end

local function updateHonorUI()
	updateHonorValue();
	--updateHonorLabel();
end

local function updateQuestUI()
	local _, numQuests = GetNumQuestLogEntries();
	
	local text = format("%i/%i", numQuests, MAX_QUESTS);
	
	if (numQuests >= MAX_QUESTS) then
		text = DLU.colorizeString(text, "red");
	end
	
	DLUHQ.Panels.Character.Quests.Value:SetText(text);
end

local function checkStoryProgress(questId)
	local storyID = GetZoneStoryID();
	
	if (storyID ~= nil and not QuestUtils_IsQuestWorldQuest(questId)) then
		local numCriteria = GetAchievementNumCriteria(storyID);
		
		local completedCriteria = 0;
		for i = 1, numCriteria do
			local _, _, completed = GetAchievementCriteriaInfo(storyID, i);
			if ( completed ) then
				completedCriteria = completedCriteria + 1;
			end
		end
		
		if (completedCriteria == numCriteria) then
			DLU.createNotification(addonData.L.CompletedZoneMainStoryLine, "leveling");
		end
	end
end

local function updateXpValue()
	local nextLevel, maxLevel = DLU.XpLabelInfo();
	if (DLU.player.level < maxLevel) then	
		local xpNeeded, xpRested, xpPercentageNeeded, xpRestedPercentage, xpDisabled, maxLevel = DLU.XpLeft();
		
		if (xpDisabled) then
			DLUHQ.Panels.Character.XpBar.Label:SetText("Xp disabled");
			DLUHQ.Panels.Character.XpBar:SetValue(0);
		else
			DLUHQ.Panels.Character.XpBar:SetValue(100 - xpPercentageNeeded);
			DLUHQ.Panels.Character.XpBar.Label:SetText(format("%s (%i%%)", xpNeeded, xpPercentageNeeded));
			if (xpRested ~= nil) then
				addonData:ChangeStatusBarColor(DLUHQ.Panels.Character.XpBar, {
					r = .25,
					g = .5,
					b = 1
				});
			else
				addonData:ChangeStatusBarColor(DLUHQ.Panels.Character.XpBar, {
					r = .75,
					g = 0,
					b = .75
				});
			end
		end
	else
		local apNeeded, apNeededPercent = DLU.ArtifactXpLeftValue();
		local apValue;
		
		if (apNeeded ~= nil) then
			if (apNeededPercent ~= nil) then
				apValue = "%s (%i%%)";
				DLUHQ.Panels.Character.XpBar:SetValue(100 - apNeededPercent);
			else
				apValue = "%s";
				DLUHQ.Panels.Character.XpBar:SetValue(100);
			end
		else
			apValue = "No artifact weapon detected...";
			DLUHQ.Panels.Character.XpBar:SetValue(0);
		end
		
		addonData:ChangeStatusBarColor(DLUHQ.Panels.Character.XpBar, {
			r = .901,
			g = .8,
			b =.601
		});
		
		DLUHQ.Panels.Character.XpBar.Label:SetText(format(apValue, apNeeded, apNeededPercent));
	end
end

local function updateXpLabel()
	local nextLevel, maxLevel = DLU.XpLabelInfo();
	
	local text = "";
	
	if (DLU.player.level < maxLevel) then
		text = addonData.L.XpNeededForLevelX .. ":";
		DLUHQ.Panels.Character.Xp.Label:SetText(format(text, nextLevel));
	else
		local currentRank, traitsWaitingForSpending, nextRank = DLU.ArtifactTraits();
		
		if (currentRank ~= nil) then
			text = addonData.L.ApNeededForLevelX;
			if (traitsWaitingForSpending ~= nil and traitsWaitingForSpending > 0) then
				if (traitsWaitingForSpending > 1) then
					text = text .. " (" .. addonData.L.ApUpgrades .. ")";
				else
					text = text .. " (" .. addonData.L.ApUpgrade ..")";
				end
			end
		end
		
		if (text ~= "") then
			text = text .. ":";
		end
		
		DLUHQ.Panels.Character.Xp.Label:SetText(format(text, nextRank, traitsWaitingForSpending));
	end
end

local function updateXpUI()
	updateXpValue();
	--updateXpLabel();
end

local function checkItemLevelUIDisplay()
	if (MIN_PLAYER_LEVEL_FOR_ITEM_LEVEL_DISPLAY < DLU.player.level) then
		DLUHQ.Panels.Character.ItemLevel:Hide();
	end
end

local function updateItemLevelUI()
	local ilvl = DLU.getItemLevel();
	DLUHQ.Panels.Character.ItemLevel.Value:SetText(ilvl);
end

local function changeXpFormat(self, arg1, arg2, checked)
	DLUSettings["numberFormat"] = arg1;
	-- Update the text; if we merely wanted it to display newValue, we would not need to do this
	if (not checked) then
		UIDropDownMenu_SetText(DLU_NumberFormatSetting, DLUSettings["numberFormat"] .. " format");
		updateXpUI();
	end
	-- Because this is called from a sub-menu, only that menu level is closed by default.
	CloseDropDownMenus();
end

local function SetUpAddonOptions()
	local panel = CreateFrame("Frame", "DLUOptionsPanel", InterfaceOptionsFramePanelContainer);
	-- Option panel name
	panel.name = DLU.addonName;
	-- Add info to panel
	local title = DLU.createTextRoot(DLU.addonName, "TOPLEFT", panel, 16, -16, "GameFontNormalLarge");
	local authorText = DLU.createTextChild(DLU.colorizeString(addonData.L.MadeBy, "red") .. ": " .. GetAddOnMetadata(DLU.addonName, "Author"), panel, title, 0, -8, "GameFontHighlightSmall");
	local versionText = DLU.createTextChild(DLU.colorizeString(addonData.L.Version, "red") .. ": ".. DLU.addonVersion, panel, authorText, 0, -8, "GameFontHighlightSmall");
	local descriptionText = DLU.createTextChild(DLU.colorizeString(addonData.L.Description, "red") .. ": " .. GetAddOnMetadata(DLU.addonName, "X-Notes"), panel, versionText, 0, -8, "GameFontHighlightSmall");
	local websiteText = DLU.createTextChild(DLU.colorizeString(addonData.L.Website, "red") .. ": " .. GetAddOnMetadata(DLU.addonName, "X-Website") .. " (it's on Danish)", panel, descriptionText, 0, -8, "GameFontHighlightSmall");
	-- Add help button
	local helpButton = createButton(panel, "Help", 80, 22, "Shows a list of commands, that you can use.", 10, 30);
	helpButton:SetScript("OnClick", function()
		SlashCmdList.DLU("help");
	end);
	-- Add WQ Companion button'
	local wqButton = createButton(panel, "Open DarkruneDK HQ", 160, 22, "Opens up the WQ Companion UI.", 100, 30);
	wqButton:SetScript("OnClick", function()
		SlashCmdList.DLU("hq");
	end);
	-- Add panel to addon options
	InterfaceOptions_AddCategory(panel);
	
	-- SettingsPanel
	local settingsPanel = CreateFrame("Frame", "DLUSettings", UIParent);
	settingsPanel.name = "Settings";
	settingsPanel.parent = panel.name;
	InterfaceOptions_AddCategory(settingsPanel);
	-- SettingsPanel text
	local settingsPanelTitle = DLU.createTextRoot(settingsPanel.name, "TOPLEFT", settingsPanel, 16, -16, "GameFontNormalLarge");
	local settingsPanelDescription = DLU.createTextChild(format("Here you can change %s settings.", DLU.addonNameColored), settingsPanel, settingsPanelTitle, 0, -8, "GameFontHighlightSmall");
	-- Setting buttons
	local checkX = -60;
	for t, s in pairs(DLUSettings) do
		local settingType = type(s);
		if (settingType == "table") then
			local checkBox = createCheckbutton(settingsPanel, 10, checkX, s.text, t);
			checkBox:SetChecked(s.value);
			checkBox:SetScript("OnClick", function()
				local settingText;
				
				if (checkBox:GetChecked()) then
					DLUSettings[t].value = true;
					settingText = DLU.colorizeString("enabled", "green");
				else
					DLUSettings[t].value = false;
					settingText = DLU.colorizeString("disabled", "red");
				end
				
				DLU.createMessage(format("%s option %s!", DLUSettings[t].text, settingText));
			end);
			
			checkX = checkX - 20;
		end
	end
	
	local items = {
		"Blizzard",
		"DarkruneDK"
	};
	
	local dropDown = CreateFrame("Frame", "DLU_NumberFormatSetting", settingsPanel, "UIDropDownMenuTemplate")
	dropDown:SetPoint("TOPLEFT", 0, checkX - 20);
	UIDropDownMenu_SetWidth(dropDown, 200);
	UIDropDownMenu_JustifyText(dropDown, "LEFT");
	UIDropDownMenu_SetText(dropDown, DLUSettings["numberFormat"] .. " format");

	-- Create and bind the initialization function to the dropdown menu
	UIDropDownMenu_Initialize(dropDown, function(self, level, menuList)
		local info = UIDropDownMenu_CreateInfo()
		for _, text in pairs(items) do
			info.text, info.arg1 = text, text;
			if (DLUSettings["numberFormat"] == text) then
				info.checked = true;
			else
				info.checked = false;
			end
		
			info.menuList = 0;
			info.func = changeXpFormat;
			UIDropDownMenu_AddButton(info);
		end
	end);
	
	-- Add reset button
	local resetButton = createButton(settingsPanel, "Reset", 100, 22, "Resets the options for " .. DLU.addonNameColored .. "." , 10, 30);
	resetButton:SetScript("OnClick", function()
		-- Reset Options
		DLU.resetAddonOptions();
		
		-- Reload wow client to ensure settings are saved
		ReloadUI();
		
		print(DLUSettings.numberFormat);
	end);
	
	-- Artifact panel
	local artifactPanel = CreateFrame("Frame", "dluArtifactOptionPanel", UIParent);
	artifactPanel.name = "Artifact";
	artifactPanel.parent = panel.name;
	InterfaceOptions_AddCategory(artifactPanel);
	-- Artifact panel text
	local artiTitle = DLU.createTextRoot(artifactPanel.name, "TOPLEFT", artifactPanel, 16, -16, "GameFontNormalLarge");
	local artiDescription = DLU.createTextChild("This page will have some buttons to help you track artifact achievements.", artifactPanel, artiTitle, 0, -8, "GameFontHighlightSmall");
	-- Hidden artifact	
	local hiddenArtifactAchievementTable = {
		["Dungeons completed"] = 11152,
		["World Quests completed"] = 11153,
		["Honorable kills"] = 11154
	};
	
	local buttonX = 10;
	for t, q in pairs(hiddenArtifactAchievementTable) do
		local trackText = "Track " .. t;
		local untrackText = "Untrack " .. t;
		local button = createButton(artifactPanel, trackText, 200, 22, "Tracks " .. t .. " needed for hidden artifact weapon appearance.", buttonX, 30);
		if (achievementCompleted(q)) then
			disableButton(button, "Already completed!");
		elseif (IsTrackedAchievement(q)) then
			button:SetText(untrackText);
		end
		button:SetScript("OnClick", function()
			local isTracked = IsTrackedAchievement(q);
			if (isTracked) then
				button:SetText(trackText);
				removeFromTracker(q);
			else
				button:SetText(untrackText);
				addToTracker(q);
			end
		end);
		buttonX = buttonX + 200;
	end
	
	-- Class specific trackers --
	local palaPanel = CreateFrame("Frame", "DLUPalaPanel", UIParent);
	palaPanel.name = DLU.getClassColored("Paladin", "PALADIN");
	palaPanel.parent = artifactPanel.name;
	InterfaceOptions_AddCategory(palaPanel);
	-- Text
	local palaTitle = DLU.createTextRoot(palaPanel.name, "TOPLEFT", palaPanel, 16, -16, "GameFontNormalLarge");
	local retArtiName = DLU.itemColorString(6, "Heart of Corruption");
	local palaText = DLU.createTextChild(format("%s requires Artifact Knowledge level 6 (Retribution).", retArtiName), palaPanel, palaTitle, 0, -8, "GameFontHighlightSmall");
	-- Button
	local retButton = createButton(palaPanel, format("Check %s status", retArtiName), 220, 22, "Check how far you are to unlock the Corrupted Ashbringer.", 390, 30);
	retButton:SetScript("OnClick", function()
		DLU.createMessage("Not the exact order!", "error"); -- other color (not really an error)
		
		local steps = {
			["Has talked to Prince Tortheldrin"] = 43682;
		};
		
		if (DLU.player.faction == "Horde") then
			steps["Has talked to Bardu"] = 43683;
		else
			steps["Has talked to Alexia"] = 43683;
		end
		
		steps["Can slime drop Timolain"] = 43684;
		steps["Can Shard be fished up"] = 43685;
		
		DLU.tableStatus(steps);
	end);
	
	-- Mage --
	local magePanel = CreateFrame("Frame", "DLUMagePanel", UIParent);
	magePanel.name = DLU.getClassColored("Mage", "MAGE");
	magePanel.parent = artifactPanel.name;
	InterfaceOptions_AddCategory(magePanel);
	-- Text
	local mageTitle = DLU.createTextRoot(magePanel.name, "TOPLEFT", magePanel, 16, -16, "GameFontNormalLarge");
	local arcArtiName = DLU.itemColorString(6, "The Woolomancer's Charge");
	local mageText = DLU.createTextChild(format("%s required level 6 Artifact Knowledge (Arcane).", arcArtiName), magePanel, mageTitle, 0, -8, "GameFontHighlightSmall");
	-- Button
	local arcButton = createButton(magePanel, format("Check %s status", arcArtiName), 260, 22, "Check how far you are to unlock the Woolomancer's Charge.", 350, 30);
	arcButton:SetScript("OnClick", function()
		local steps = {
			["Aszuna"] = 43787,
			["Stormheim"] = 43789,
			["Val'Sharah"] = 43790,
			["Suramar"] = 43791,
			["High Mountain"] = 43788
		}
		
		DLU.tableStatus(steps);
	end);
	
	-- Priest --
	local priestPanel = CreateFrame("Frame", "DLUPriestPanel", UIParent);
	priestPanel.name = DLU.getClassColored("Priest", "PRIEST");
	priestPanel.parent = artifactPanel.name;
	InterfaceOptions_AddCategory(priestPanel);
	-- Text
	local priestTitle = DLU.createTextRoot(priestPanel.name, "TOPLEFT", priestPanel, 16, -16, "GameFontNormalLarge");
	local discArtiName = DLU.itemColorString(6, "Writings of the End");
	local priestText = DLU.createTextChild(format("%s required Artifact Knowledge 4 (Discipline).", discArtiName), priestPanel, priestTitle, 0, -8, "GameFontHighlightSmall");
	-- Button
	local discButton = createButton(priestPanel, format("Check %s Status", discArtiName), 220, 22, "Check how far you are to unlock the Writtings of the End.", 390, 30);
	discButton:SetScript("OnClick", function()
		DLU.createMessage("Not the exact order!", "error"); -- other color (not an error)
		
		local steps = {
			["Dalaran - The Violet Citadel"] = 44339,
			["Class Order Hall - Juvess the Duskwhisperer"] = 44340,
			["Northrend - New Hearthglen"] = 44341,
			["Class Order Hall - Archivist Inkforge "] = 44342,
			["Scholomance - Chillheart's Room"] = 44343,
			["Class Order Hall - Meridelle Lightspark"] = 44344,
			["Scarlet Halls - The Flameweaver's library"] = 44345,
			["Azsuna - Chief Bitterbrine Azsuna"] = 44346,
			["Suramar - Artificer Lothaire"] = 44347,
			["Black Rook Hold - Library after First Boss"] = 44348,
			["Karazhan - Guardian's Library"] = 44349,
			["Stormheim - Inquisitor Ernstonbok"] = 44350
		}
		
		DLU.tableStatus(steps);
	end);
	
	-- Warrior --
	local warPanel = CreateFrame("Frame", "DLUWarriorPanel", UIParent);
	warPanel.name = DLU.getClassColored("Warrior", "WARRIOR");
	warPanel.parent = artifactPanel.name;
	InterfaceOptions_AddCategory(warPanel);
	-- Text
	local warTitle = DLU.createTextRoot(warPanel.name, "TOPLEFT", warPanel, 16, -16, "GameFontNormalLarge");	
	local protArtiName = DLU.itemColorString(6, "Burning Plate of the Worldbreaker");
	local warText = DLU.createTextChild(format("%s required Artifact Knowledge 5 (Protection).", protArtiName), warPanel, warTitle, 0, -8, "GameFontHighlightSmall");
	-- Button
	local protButton = createButton(warPanel, format("Check %s Status", protArtiName), 300, 22, "Check how far you are to unlock the Burning Plate of the Worldbreaker.", 310, 30);
	protButton:SetScript("OnClick", function()
		DLU.createMessage("Has to be available and not denied!", "error"); -- other color (not an error)
		
		local steps = {
			["Available"] = 44311,
			["Denied"] = 44312
		}
		
		DLU.tableStatus(steps);
	end);
end

-- DLUHQ
local BD_DATA = {
	bg = [[Interface\Tooltips\UI-Tooltip-Background]],
	edge = [[Interface\Buttons\WHITE8X8]]
};

local DLU_BD = {
	primary = { bgFile = BD_DATA.bg, tile = true, tileSize = 16, edgeFile = BD_DATA.edge, edgeSize = 1 },
	secondary = { bgFile = BD_DATA.bg, tile = false, tileSize = 4, edgeFile = BD_DATA.edge, edgeSize = .75 }
};

local function getBackdropAlpha(frame)
	local _, _, _, a = frame:GetBackdropColor();
	return tonumber(format("%.1f", a));
end

local function hightlight(button)
	local alpha = getBackdropAlpha(button);
	if (alpha ~= .5) then
		button:SetBackdropColor(1,1,1,.4);
	end
end
local function unHightlight(button)
	local alpha = getBackdropAlpha(button);
	if (alpha ~= .5) then
		button:SetBackdropColor(1,1,1,.2);
	end
end

local function ShowXpTooltip()
	GameTooltip:ClearLines();
	GameTooltip:SetOwner(DLUHQ.Panels.Character.XpBar, 'ANCHOR_CURSOR', 0, -4);
	
	local nextLevel, maxLevel = DLU.XpLabelInfo();
	if (DLU.player.level < maxLevel) then
		GameTooltip:AddLine(addonData.L.XP);
		GameTooltip:AddLine(" ");
		GameTooltip:AddDoubleLine(addonData.L.NextLevel, nextLevel,1,1,1);
		local _, xpRested, _, xpRestedPercentage = DLU.XpLeft();
		if (xpRested ~= nil) then
			GameTooltip:AddDoubleLine(addonData.L.RestedXp, format("%s", DLU.itemColorString(3, xpRested)), 1,1,1);
		end
	else
		local artifactName, spendPower, currentRank = DLU.ArtifactInfo();
		GameTooltip:AddLine(artifactName, .9, .8, .5);
		GameTooltip:AddLine(" ");
		GameTooltip:AddDoubleLine(addonData.L.CurrentRank, currentRank, 1,1,1);
	end
	
	GameTooltip:Show();
end

local function ShowHonorTooltip()
	GameTooltip:ClearLines();
	GameTooltip:SetOwner(DLUHQ.Panels.Character.PvPBar, 'ANCHOR_CURSOR', 0, -4);
	
	local _, honorLevel, honorMax, prestigeLevel = DLU.honorLeftValues();
	local restedHonor = GetHonorExhaustion();
	
	GameTooltip:AddLine(addonData.L.PvP);
	GameTooltip:AddLine(" ");
	
	GameTooltip:AddDoubleLine(addonData.L.HonorRank, honorLevel, 1,1,1);
	GameTooltip:AddDoubleLine(addonData.L.PrestigeRank, prestigeLevel, 1,1,1);
	
	if (restedHonor ~= nil) then
		GameTooltip:AddDoubleLine(addonData.L.RestedXp, restedHonor, 1,1,1);
	end
	
	GameTooltip:Show();
end

local function HideTooltip()
	GameTooltip:Hide();
end

function addonData:CreateLabel(text, fontSize, alignment, parentFrame)
	if (parentFrame == nil) then
		return
	end

	local label = parentFrame:CreateFontString();
	label:SetFont("Fonts\\FRIZQT__.TTF", fontSize);
	label:SetPoint(alignment, parentFrame, alignment, 0, 0);
	label:SetJustifyH("LEFT");
    label:SetJustifyV("TOP");
    label:SetText(text);
    label:SetTextColor(1, 1, 1, 1);
    return label;
end

function addonData:CreateButton(width, height, alignment, parentFrame, xOffset, yOffset)
	if (parentFrame == nil) then
		return
	end

	xOffset = xOffset or 0;
	yOffset = yOffset or 0;

	local button = CreateFrame("Button", nil, parentFrame);
	button:SetPoint(alignment, parentFrame, alignment, xOffset, yOffset);
	button:SetSize(width, height);
	button:SetBackdrop(DLU_BD.secondary);
	button:SetBackdropColor(1,1,1,.2);
	button:SetBackdropBorderColor(1,1,1,0);
	button:SetScript("OnEnter", hightlight);
	button:SetScript("OnLeave", unHightlight);
	return button;
end

function addonData:CreatePanel(width, height, parentFrame, yOffset)
	local frame = CreateFrame("Frame", nil, parentFrame);
	frame:SetSize(width, height);
	frame:SetPoint("TOPLEFT", parentFrame, "TOPLEFT", 0, yOffset);
	frame:SetBackdrop(DLU_BD.secondary);
	frame:SetBackdropColor(0,0,0,1);
	frame:SetBackdropBorderColor(1,1,1,0);
	return frame;
end

function addonData:CreateStatusBar(parent, width, height, position, xOffset, yOffset, color, hover)
	local frame = CreateFrame("StatusBar", nil, parent);
	frame:SetSize(width, height);
	frame:SetPoint(position, parent, position, xOffset, yOffset);
	frame:SetStatusBarTexture([[Interface\BUTTONS\WHITE8X8.blp]]);
	frame:SetStatusBarColor(color.r, color.g, color.b, color.a);
	frame:SetMinMaxValues(0,100);
	frame:SetScript("OnEnter", hover);
	frame:SetScript("OnLeave", HideTooltip);
	return frame;
end

function addonData:ChangeStatusBarColor(statusbar, newColor)
	statusbar:SetStatusBarColor(newColor.r, newColor.g, newColor.b, .6);
end

function addonData:CreateBorder(parent)
	local frame = CreateFrame("Frame", nil, parent);
	frame:SetPoint("TOPLEFT", parent, "TOPLEFT", -.5, .5);
	frame:SetPoint("BOTTOMRIGHT", parent, "BOTTOMRIGHT", .5, -.5);
	frame:SetBackdrop({bgFile = "Interface/Tooltips/UI-Tooltip-Background", 
                                            edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
                                            tile = false, tileSize = 0, edgeSize = 1, 
											insets = { left = -1, right = -1, top = -1, bottom = -1}
                                            });
	frame:SetBackdropColor(0,0,0,.6);
	frame:SetFrameLevel(parent:GetFrameLevel());
	return frame;
end

local function toggleContainer()
	local isVisible = DLUHQ.Container:IsVisible();
	local name = DLUHQ:GetName();
	
	if (not DLU_UI[name]) then
		DLU_UI[name] = {};
	end
	
	if (isVisible) then
		DLUHQ.Container:Hide();
		DLUHQ.minimize.Label:SetText("+");
	else
		DLUHQ.Container:Show();
		DLUHQ.minimize.Label:SetText("-");
	end
	
	DLU_UI[name].minimized = isVisible;
end

local function scroll(self, sv)
	sv = sv * 16;
	local maximum = max(0, DLUHQ.Panels.LFR.ShortageContainer.Content:GetHeight() - self:GetHeight());
	local value = self:GetVerticalScroll();
	if ((value - sv) < 0) then
		self:SetVerticalScroll(0);
	elseif (value - sv) > maximum then
		self:SetVerticalScroll(maximum)
	else
		self:SetVerticalScroll(value - sv)
	end
end

local function stopUIMoving(self)
	self:StopMovingOrSizing();
	local name = self:GetName();
	
	if (DLU_UI[name]) then
		DLU_UI[name]["xPos"] = self:GetLeft();
		DLU_UI[name]["yPos"] = self:GetBottom();
	else
		DLU_UI[name] = {};
		DLU_UI[name].xPos = self:GetLeft();
		DLU_UI[name].yPos = self:GetBottom();
	end
end

-- Navigation
local function togglePanel(shownPanel)
	for k, v in pairs(DLUHQ.Panels) do
		if (k == shownPanel) then
			DLUHQ[shownPanel]:SetBackdropColor(1,1,1,.5);
			DLUHQ.Panels[shownPanel]:Show();
			DLU_UI["shownPanel"] = shownPanel;
		else
			DLUHQ[k]:SetBackdropColor(1,1,1,.2);
			DLUHQ.Panels[k]:Hide();
		end
	end
end

-- Traveling
local function Traveling_OnClick(self)
	local index = self:GetID();
	UIDropDownMenu_SetSelectedID(TravelingDropdown, index);
	
	local text = LEVELING_DEFAULT;
	local requiredLevel = 0;
	if (index == 2) then
		local _, _, _, _, _, _, _, minLevel = GetAreaMapInfo(465);
		
		requiredLevel = format(addonData.L.RequiredLevel, minLevel);
		text = format("%s|n|n%s %s", requiredLevel, format(addonData.L.StartText, self.value), addonData.L.UsingPortal);
		
		local factionText = "";
		if (DLU.player.faction == "Horde") then
			factionText = format(addonData.L.OutlandHorde, GetMapNameByID(321));
		else
			factionText = format(addonData.L.OutlandAlliance, GetMapNameByID(301));
		end
		
		text = format("%s %s %s.", text, factionText, GetMapNameByID(465));
	elseif (index == 3) then
		local _, _, _, _, _, _, _, minLevel = GetAreaMapInfo(486);
		
		requiredLevel = format(addonData.L.RequiredLevel, minLevel);
		text = format("%s|n|n%s", requiredLevel, format(addonData.L.StartText, self.value));
		local factionText = "";
		if (DLU.player.faction == "Horde") then
			factionText = format(addonData.L.NorthrendHordeTravel, GetMapNameByID(321), GetMapNameByID(382), self.value);
		else
			factionText = format(addonData.L.NorthrendAllianceTravel, GetMapNameByID(301));
		end
		
		text = format("%s %s", text, factionText);
	elseif (index == 4) then
		local _, _, _, _, _, _, _, minLevel = GetAreaMapInfo(606);
		
		requiredLevel = format(addonData.L.RequiredLevel, minLevel);
		if (requiredLevel ~= 0) then
			text = format("%s|n|n%s", requiredLevel, format(addonData.L.CataclysmZones, GetMapNameByID(606), GetMapNameByID(613), GetMapNameByID(640), GetMapNameByID(700), GetMapNameByID(708), GetMapNameByID(720)));
		else
			text = format(addonData.L.CataclysmZones, GetMapNameByID(606), GetMapNameByID(613), GetMapNameByID(640), GetMapNameByID(700), GetMapNameByID(708), GetMapNameByID(720));
		end
		
		if (DLU.player.faction == "Horde") then
			factionText = format("|n|n" .. addonData.L.CataclysmVashjHorde, GetMapNameByID(613), GetMapNameByID(321));
		else
			factionText = format("|n|n" .. addonData.L.CataclysmVashjAlliance, GetMapNameByID(613), GetMapNameByID(301));
		end
		
		text = format("%s%s", text, factionText);
	elseif (index == 5) then
		local pandariaIntroCompleted = IsQuestFlaggedCompleted(31853);
		
		local _, _, _, _, _, _, _, minLevel = GetAreaMapInfo(806);
		
		requiredLevel = format(addonData.L.RequiredLevel, minLevel);
		text = format("%s|n|n", requiredLevel);
		
		if (DLU.player.faction == "Horde") then
			if (pandariaIntroCompleted or DLU.player.level >= 85) then
				text = format("%s%s", text, format(addonData.L.PandariaHordeQuestCompleted, GetMapNameByID(862), GetMapNameByID(321)));
			else
				text = format("%s%s", text, format(addonData.L.PandariaHorde, GetMapNameByID(862), GetMapNameByID(321), 85));
			end
		else
			if (pandariaIntroCompleted or DLU.player.level >= 85) then
				text = format("%s%s", text, format(addonData.L.PandariaAllianceQuestCompleted, GetMapNameByID(862), GetMapNameByID(301)));
			else
				text = format("%s%s", text, format(addonData.L.PandariaAlliance, GetMapNameByID(301), GetMapNameByID(862), 85));
			end
		end
	elseif (index == 6) then
		local _, _, _, _, _, _, _, minLevel = GetAreaMapInfo(941);
		
		requiredLevel = format(addonData.L.RequiredLevel, minLevel);
		text = format("%s|n|n%s %s", requiredLevel, format(addonData.L.StartText, GetMapNameByID(962)), addonData.L.UsingPortal);
		
		local factionText = "";
		if (DLU.player.faction == "Horde") then
			factionText = format(addonData.L.Draenor, GetMapNameByID(321), GetMapNameByID(19));
		else
			factionText = format(addonData.L.Draenor, GetMapNameByID(301), GetMapNameByID(19));
		end
		
		text = format("%s %s", text, factionText);
	elseif (index == 7) then
		
		requiredLevel = format(addonData.L.RequiredLevel, 98);
		text = format("%s|n|n", requiredLevel);
		
		local questCompleted = IsQuestFlaggedCompleted(44663);
		if (questCompleted) then
			text = format("%s%s %s", text, format(addonData.L.StartText, GetMapNameByID(1007)), addonData.L.UsingPortal);
		end
		
		if (DLU.player.faction == "Horde") then
			if (questCompleted) then
				text = format("%s %s", text, format(addonData.L.BrokenIslesCompletedHorde, GetMapNameByID(321)));
			else
				text = format("%s%s", text, format(addonData.L.BrokenIsles, GetMapNameByID(1007), GetMapNameByID(321)));
			end
		else
			if (questCompleted) then
				text = format("%s %s", text, format(addonData.L.BrokenIslesCompletedAlliance, GetMapNameByID(301)));
			else
				text = format("%s%s", text, format(addonData.L.BrokenIsles, GetMapNameByID(1007), GetMapNameByID(321)));
			end
		end
	end
	
	DLUHQ.Panels.Traveling.Info.Text:SetText(text);
end

local function createDLUHQUI()
	DLUHQ = CreateFrame("Frame", "DLUHQ_UI", UIParent);
	DLUHQ:SetSize(WIDTH, 20);
	DLUHQ:SetBackdrop(DLU_BD.primary);
	
	local name = DLUHQ:GetName();
	if (DLU_UI[name] and DLU_UI[name].xPos) then
		DLUHQ:SetPoint("BOTTOMLEFT", DLU_UI[name].xPos, DLU_UI[name].yPos);
	else
		DLUHQ:SetPoint("CENTER", UIParent, "CENTER", 0, (HEIGHT + 20 + 25) / 2);
	end
	
	--DLUHQ:SetFrameStrata("DIALOG");
	DLUHQ:SetBackdropColor(0,0,0,1);
	DLUHQ:SetBackdropBorderColor(.7,.7,.7,1);
	DLUHQ:SetClampedToScreen(true);
	DLUHQ:SetMovable(true);
    DLUHQ:EnableMouse(true);
	DLUHQ:RegisterForDrag("LeftButton");

	DLUHQ:SetScript("OnDragStart", DLUHQ.StartMoving);
    DLUHQ:SetScript("OnDragStop", stopUIMoving);

	-- Title
	DLUHQ.title = DLUHQ:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge");
	DLUHQ.title:SetPoint("TOP", 0, -2);
	DLUHQ.title:SetText(format("%s HQ", "DarkruneDK"));

	-- Settings button
	DLUHQ.settings = addonData:CreateButton(16, 16, "LEFT", DLUHQ);
	DLUHQ.settings.Label = addonData:CreateLabel("?", 10, "CENTER", DLUHQ.settings);
	DLUHQ.settings:SetScript("OnClick", function()
		InterfaceOptionsFrame_OpenToCategory(addonName);
	end);

	-- Minimize button
	DLUHQ.minimize = addonData:CreateButton(16, 16, "RIGHT", DLUHQ, -20);
	DLUHQ.minimize.Label = addonData:CreateLabel("-", 10, "CENTER", DLUHQ.minimize);
	DLUHQ.minimize:SetScript("OnClick", toggleContainer);

	-- Close button
	DLUHQ.close = addonData:CreateButton(16, 16, "RIGHT", DLUHQ);
	DLUHQ.close.Label = addonData:CreateLabel("X", 10, "CENTER", DLUHQ.close);
	DLUHQ.close:SetScript("OnClick", function()
		DLUHQ:Hide();
	end);

	-- Panel Container
	DLUHQ.Container = CreateFrame("Frame", nil, DLUHQ);
	DLUHQ.Container:SetSize(WIDTH, HEIGHT);
	DLUHQ.Container:SetPoint("TOPLEFT", DLUHQ, "TOPLEFT", 0, -20);
	if (DLU_UI[name] and DLU_UI[name].minimized) then
		toggleContainer();
	end

	-- Panel Navigation
	DLUHQ.navigation = CreateFrame("Frame", nil, DLUHQ.Container);
	DLUHQ.navigation:SetSize(WIDTH, HEIGHT_NAV);
	DLUHQ.navigation:SetPoint("TOPLEFT", DLUHQ.Container, "TOPLEFT", 0, 0);
	DLUHQ.navigation:SetBackdrop(DLU_BD.secondary);
	DLUHQ.navigation:SetBackdropColor(.7,0,0, 1);
	DLUHQ.navigation:SetBackdropBorderColor(1,1,1,0);

	-- Panel Buttons
	DLUHQ.WQ = addonData:CreateButton(75, HEIGHT_NAV, "TOPLEFT", DLUHQ.navigation);
	DLUHQ.WQ.Label = addonData:CreateLabel(addonData.L.WorldQuest, FONT_SIZES.small, "CENTER", DLUHQ.WQ);

	DLUHQ.LFR = addonData:CreateButton(75, HEIGHT_NAV, "TOPLEFT", DLUHQ.navigation, 75 + 3);
	DLUHQ.LFR.Label = addonData:CreateLabel(addonData.L.LFR, FONT_SIZES.small, "CENTER", DLUHQ.LFR);
	
	DLUHQ.Character = addonData:CreateButton(75, HEIGHT_NAV, "TOPLEFT", DLUHQ.navigation, (75 + 3) * 2);
	DLUHQ.Character.Label = addonData:CreateLabel(addonData.L.Character, FONT_SIZES.small, "CENTER", DLUHQ.Character);
	
	DLUHQ.Traveling = addonData:CreateButton(75, HEIGHT_NAV, "TOPLEFT", DLUHQ.navigation, (75 + 3) * 3);
	DLUHQ.Traveling.Label = addonData:CreateLabel(addonData.L.Traveling, FONT_SIZES.small, "CENTER", DLUHQ.Traveling);

	------------
	-- Panels --
	------------
	--- World Quests
	DLUHQ.Panels = {};
	DLUHQ.Panels.WQ = addonData:CreatePanel(WIDTH, HEIGHT - HEIGHT_NAV, DLUHQ.Container, -HEIGHT_NAV);
	DLU.createTextRoot(addonData.L.WorldQuest, "TOP", DLUHQ.Panels.WQ, 8, -6, "GameFontNormalLarge");
	DLUHQ.Panels.WQ.Emissaries = {};
	
	for i = 1, 3 do
		local currentEmissary = DLU.emissaryTable[i];
		local questId = 0;
		local labelText = "";
		local valueText = addonData.L.Loading;
		local tooltip = addonData.L.Loading;
		
		if (currentEmissary ~= nil) then
			labelText = format("%s", DLU.iconToText(currentEmissary.icon));
			questId = currentEmissary.questId;
			tooltip = format("%s", currentEmissary.factionName);
			valueText = getEmissariesValue(i);
		end
		
		DLUHQ.Panels.WQ.Emissaries[i] = CreateFrame("Frame", nil, DLUHQ.Panels.WQ);
		DLUHQ.Panels.WQ.Emissaries[i]:SetSize(WIDTH_THIRD, 22);
		
		local xOffset = (WIDTH_THIRD + PADDING)*(i-1);
		
		if (i == 1) then
			xOffset = PADDING/2;
		end
		
		DLUHQ.Panels.WQ.Emissaries[i]:SetPoint("TOPLEFT", DLUHQ.Panels.WQ, "TOPLEFT", xOffset, -PADDING_TOP);
		
		DLUHQ.Panels.WQ.Emissaries[i].questId = questId;
		DLUHQ.Panels.WQ.Emissaries[i].Tooltip = tooltip;
		
		DLUHQ.Panels.WQ.Emissaries[i].Label = addonData:CreateLabel(labelText, FONT_SIZES.medium, "LEFT", DLUHQ.Panels.WQ.Emissaries[i]);
		DLUHQ.Panels.WQ.Emissaries[i].Value = addonData:CreateLabel(valueText, FONT_SIZES.medium, "RIGHT", DLUHQ.Panels.WQ.Emissaries[i]);
		
		DLUHQ.Panels.WQ.Emissaries[i]:SetScript("OnEnter", OnEnter);
		DLUHQ.Panels.WQ.Emissaries[i]:SetScript("OnLeave", OnLeave);
	end

	--- LFR
	DLUHQ.Panels.LFR = addonData:CreatePanel(WIDTH, HEIGHT - HEIGHT_NAV, DLUHQ.Container, -HEIGHT_NAV);
	DLU.createTextRoot(addonData.L.LFR, "TOP", DLUHQ.Panels.LFR, 8, -6, "GameFontNormalLarge");
	DLUHQ.Panels.LFR.ShortageContainer = CreateFrame("ScrollFrame", nil, DLUHQ.Panels.LFR);
	DLUHQ.Panels.LFR.ShortageContainer.Content = CreateFrame("Frame", nil, DLUHQ.Panels.LFR.ShortageContainer);
	DLUHQ.Panels.LFR.ShortageContainer:SetScrollChild(DLUHQ.Panels.LFR.ShortageContainer.Content);
	DLUHQ.Panels.LFR.ShortageContainer:SetScript("OnMouseWheel", scroll);
	DLUHQ.Panels.LFR.ShortageContainer:SetSize(WIDTH, 300);
	DLUHQ.Panels.LFR.ShortageContainer:SetPoint("TOPLEFT", DLUHQ.Panels.LFR, "TOPLEFT", 0, -PADDING_TOP);

	DLUHQ.Panels.LFR.ShortageContainer.Content:SetSize(WIDTH, 300);
	DLUHQ.Panels.LFR.ShortageContainer.Content.lines = {};
	for i = 1, GetNumRFDungeons() do
		local line = CreateFrame("Frame", nil, DLUHQ.Panels.LFR.ShortageContainer.Content);
		line:SetPoint("TOPLEFT", 0, -(i-1)*16);
		line:SetSize(1, 16);

		line.text = line:CreateFontString(nil, "ARTWORK", "GameFontWhite");
		line.text:SetPoint("LEFT", 5, -5);

		DLUHQ.Panels.LFR.ShortageContainer.Content.lines[i] = line;
	end
	
	--- Character
	DLUHQ.Panels.Character = addonData:CreatePanel(WIDTH, HEIGHT - HEIGHT_NAV, DLUHQ.Container, -HEIGHT_NAV);
	DLU.createTextRoot(addonData.L.Character, "TOP", DLUHQ.Panels.Character, 8, -6, "GameFontNormalLarge");
	
	-- New XP/AP (status bar)
	DLUHQ.Panels.Character.XpBar = addonData:CreateStatusBar(DLUHQ.Panels.Character, WIDTH - PADDING, FONT_SIZES.medium, "TOPLEFT", PADDING/2, -(5 + PADDING_TOP), {
		r = 1,
		g = .2,
		b = .8
	}, ShowXpTooltip);
	
	DLUHQ.Panels.Character.XpBar.border = addonData:CreateBorder(DLUHQ.Panels.Character.XpBar);
	DLUHQ.Panels.Character.XpBar.Label = addonData:CreateLabel("", FONT_SIZES.medium-2, "CENTER", DLUHQ.Panels.Character.XpBar);
	updateXpUI();
	
	-- PvP/Honor
	DLUHQ.Panels.Character.PvPBar = addonData:CreateStatusBar(DLUHQ.Panels.Character, WIDTH - PADDING, FONT_SIZES.medium, "TOPLEFT", PADDING/2, -(5 + PADDING_TOP + FONT_SIZES.medium + 6), {
		r = .941,
		g = .552,
		b = .254
	}, ShowHonorTooltip);
	DLUHQ.Panels.Character.PvPBar.border = addonData:CreateBorder(DLUHQ.Panels.Character.PvPBar);
	DLUHQ.Panels.Character.PvPBar.Label = addonData:CreateLabel("", FONT_SIZES.medium-2, "CENTER", DLUHQ.Panels.Character.PvPBar);
	DLUHQ.Panels.Character.PvPBar:SetMinMaxValues(0, UnitHonorMax("player"));
	updateHonorUI();
	
	-- Quests
	DLUHQ.Panels.Character.Quests = CreateFrame("Frame", nil, DLUHQ.Panels.Character);
	DLUHQ.Panels.Character.Quests:SetSize(WIDTH - PADDING, FONT_SIZES.medium);
	DLUHQ.Panels.Character.Quests:SetPoint("TOPLEFT", DLUHQ.Panels.Character, "TOPLEFT", PADDING/2, -(5 + PADDING_TOP + FONT_SIZES.medium + 12 * 2))
	
	DLUHQ.Panels.Character.Quests.Label = addonData:CreateLabel("Amount of Quests:", FONT_SIZES.medium, "LEFT", DLUHQ.Panels.Character.Quests);
	DLUHQ.Panels.Character.Quests.Value = addonData:CreateLabel("", FONT_SIZES.medium, "RIGHT", DLUHQ.Panels.Character.Quests);
	updateQuestUI();
	
	-- Item level
	DLUHQ.Panels.Character.ItemLevel = CreateFrame("Frame", nil, DLUHQ.Panels.Character);
	DLUHQ.Panels.Character.ItemLevel:SetSize(WIDTH - PADDING, FONT_SIZES.medium);
	DLUHQ.Panels.Character.ItemLevel:SetPoint("TOPLEFT", DLUHQ.Panels.Character, "TOPLEFT", PADDING/2, -(5 + PADDING_TOP + FONT_SIZES.medium + 12 * 3));
	
	DLUHQ.Panels.Character.ItemLevel.Label = addonData:CreateLabel("Item level:", FONT_SIZES.medium, "LEFT", DLUHQ.Panels.Character.ItemLevel);
	DLUHQ.Panels.Character.ItemLevel.Value = addonData:CreateLabel("0", FONT_SIZES.medium, "RIGHT", DLUHQ.Panels.Character.ItemLevel);
	checkItemLevelUIDisplay();
	if (DLUHQ.Panels.Character.ItemLevel:IsVisible()) then
		updateItemLevelUI();
	end
	
	--- Traveling
	DLUHQ.Panels.Traveling = addonData:CreatePanel(WIDTH, HEIGHT - HEIGHT_NAV, DLUHQ.Container, -HEIGHT_NAV);
	DLU.createTextRoot(addonData.L.Traveling, "TOP", DLUHQ.Panels.Traveling, 8, -6, "GameFontNormalLarge");
	
	CreateFrame("Button", "TravelingDropdown", DLUHQ.Panels.Traveling, "UIDropDownMenuTemplate");
	TravelingDropdown:ClearAllPoints();
	TravelingDropdown:SetPoint("TOPLEFT", DLUHQ.Panels.Traveling, "TOPLEFT", 0, -30);
	TravelingDropdown:Show();
	
	local items = {
		"Vanilla",
		"Outland",
		"Northrend",
		"Cataclysm",
		"Pandaria",
		"Draenor",
		"Broken Isles"
	};
	
	UIDropDownMenu_Initialize(TravelingDropdown, function(selv, level)
		local info = UIDropDownMenu_CreateInfo()
		for k,v in pairs(items) do
			info = UIDropDownMenu_CreateInfo()
			info.text = v
			info.value = v
			info.func = Traveling_OnClick
			UIDropDownMenu_AddButton(info, level)
		end
	end);
	UIDropDownMenu_SetWidth(TravelingDropdown, 100);
	UIDropDownMenu_SetButtonWidth(TravelingDropdown, 124);
	UIDropDownMenu_SetSelectedID(TravelingDropdown, 1);
	UIDropDownMenu_JustifyText(TravelingDropdown, "LEFT");
	
	DLUHQ.Panels.Traveling.Info = CreateFrame("Frame", nil, DLUHQ.Panels.Traveling);
	DLUHQ.Panels.Traveling.Info:SetSize(WIDTH - PADDING, HEIGHT - HEIGHT_NAV - 65);
	DLUHQ.Panels.Traveling.Info:SetPoint("TOPLEFT", DLUHQ.Panels.Traveling, "TOPLEFT", PADDING/2, -65);
	
	DLUHQ.Panels.Traveling.Info.Text = addonData:CreateLabel(LEVELING_DEFAULT, FONT_SIZES.medium, "TOPLEFT", DLUHQ.Panels.Traveling.Info);

	-- Hook navigation
	DLUHQ.WQ:SetScript("OnClick", function()
		togglePanel("WQ");
	end);
	DLUHQ.LFR:SetScript("OnClick", function()
		togglePanel("LFR");
	end);
	DLUHQ.Character:SetScript("OnClick", function()
		togglePanel("Character");
	end);
	DLUHQ.Traveling:SetScript("OnClick", function()
		togglePanel("Traveling");
	end);
	
	-- Navigate to front page by default
	if (DLU_UI["shownPanel"]) then
		togglePanel(DLU_UI["shownPanel"]);
	else
		togglePanel("WQ");
	end
end

local function checkHQVisibility()
	if (DLU.player.level < 10 and not DLUSettings.showDLUHQOnLogon.value or not DLUSettings.showDLUHQOnLogon.value) then
		DLUHQ:Hide();
	end
end

-- Events
function events:LFG_UPDATE_RANDOM_INFO(...)
	local icons = {
		tank = DLU.iconToText("Interface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES:16:16:0:0:64:64:0:19:22:41"),
		heal = DLU.iconToText("Interface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES:16:16:0:0:64:64:20:39:1:20"),
		dps = DLU.iconToText("Interface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES:16:16:0:0:64:64:20:39:22:41")
	}

	createShortageList();

	local shortageAmount = #shortageList;
	if (shortageAmount > 0) then
		for i = 1, shortageAmount do
			local line = DLUHQ.Panels.LFR.ShortageContainer.Content.lines[i];

			local shortage = shortageList[i];
			local dunName = GetLFGDungeonInfo(shortage.dID);
			local text = dunName;

			if (shortage.tank) then
				text = format("%s %s", icons.tank, text);
			end
			if (shortage.heal) then
				text = format("%s %s", icons.heal, text);
			end
			if (shortage.dps) then
				text = format("%s %s", icons.dps, text);
			end

			line.text:SetText(text);
		end
	else
		local line = DLUHQ.Panels.LFR.ShortageContainer.Content.lines[1];
		line.text:SetText("No bonus rewards available at this moment...");
	end
end

function events:PLAYER_LEVEL_UP(self, event, ...)
	if (DLU.player.level == 110) then
		DLU.updateEmissaries();
	end
end

function events:ARTIFACT_XP_UPDATE()
	if (DLU.player.level >= 100) then
		updateXpUI();
	end
end
function events:UNIT_INVENTORY_CHANGED()
	if (DLU.player.level >= 100) then
		updateXpUI();
	end
	if (DLUHQ.Panels.Character.ItemLevel:IsVisible()) then
		updateItemLevelUI();
	end
end
function events:HONOR_XP_UPDATE()
	updateHonorValue();
end
function events:HONOR_LEVEL_UPDATE()
	DLUHQ.Panels.Character.PvPBar:SetMinMaxValues(0, UnitHonorMax("player"));
	--updateHonorLabel();
end
function events:HONOR_PRESTIGE_UPDATE()
	--updateHonorLabel();
end
function events:PLAYER_XP_UPDATE()
	updateXpValue();
end
function events:PLAYER_LEVEL_UP()
	--updateXpLabel();
	checkItemLevelUIDisplay();
end
function events:QUEST_REMOVED(questId)
	updateQuestUI();
	checkStoryProgress(questId);
end
function events:QUEST_ACCEPTED()
	updateQuestUI();
end
function events:PLAYER_UPDATE_RESTING()
	updateXpValue();
end

-- Initialization
settingsFrame:RegisterEvent("PLAYER_LOGIN");
settingsFrame:SetScript("OnEvent", function()
	DLU_UI = DLU_UI or {};
	
	SetUpAddonOptions();
	createDLUHQUI();

	checkHQVisibility()
	DLU.updateEmissaries();

	eventFrame:SetScript("OnEvent", function(self, event, ...)
		events[event](self, ...);
	end);
	for k, v in pairs(events) do
		eventFrame:RegisterEvent(k);
	end
	
	settingsFrame:UnregisterEvent("PLAYER_LOGIN")
end);