-- Import
local addonName, addonData = ...;

-- Globals --
DLU = {
	prefix = "[DLU]",
	addonName = addonName,
	addonNameColored = addonName,
	addonVersion = "0",
	mountTable = {
		waterMounts = {},
		pvpMounts = {},
		lowLevelMounts = {}
	},
	genderTable = {
		"Unknown",
		"Male",
		"Female"
	},
	player = {
		faction = nil,
		level = UnitLevel("player")
	},
	emissaryTable = {},
	worldQuestTable = {}
}
DLU.addonNameColored = GetAddOnMetadata(DLU.addonName, "Title");
DLU.addonVersion = GetAddOnMetadata(DLU.addonName, "Version");

-- Frames
CreateFrame("Frame", "DLUCoreFrame");

-- Settings
function DLU.getDefaultOptions()
	return {
		party = {
			text = "Party option",
			value = false
		},
		pvp = {
			text = "PvP option",
			value = false
		},
		WQTracking = {
			text = "World Quest tracking (including invasion notification)",
			value = false
		},
		WQPartyNotification = {
			text = "World Quest Party notification",
			value = false
		},
		showDLUHQOnLogon = {
			text = "Show DarkruneDK HQ on start up",
			value = false
		},
		numberFormat = "Blizzard"
	};
end

function DLU.getSettings()
	DLUSettings = DLUSettings or DLU.getDefaultOptions();
end

function DLU.resetAddonOptions()
	DLUSettings = DLU.getDefaultOptions();
end

-- DLU functions
function DLU.tableContains(tableToCheck, element)
	if (tableToCheck ~= nil) then
		for i = 0, #tableToCheck do
			if (element == tableToCheck[i]) then
				return tableToCheck[i];
			end
		end
	end

	return false;
end

function DLU.tableLength(tableToCheck)
	local count = 0;
	for _ in pairs(tableToCheck) do count = count + 1 end
	return count;
end

function DLU.tableMerge(t1, t2)
	for k,v in ipairs(t2) do
		table.insert(t1, v);
	end
end

function DLU.colorizeString(text, colorType)
	if (colorType == "red") then
		return "|cffCC0000" .. text .. "|r";
	elseif (colorType == "green") then
		return "|cff00CC00" .. text .. "|r";
	elseif (colorType == "orange") then
		return "|cffffa500" .. text .. "|r";
	else
		return text;
	end
end

function DLU.createMessage(message, messageType)
	local prefixText = DLU.colorizeString(DLU.prefix, "orange");
	local text = format("%s %s", prefixText, message);

	if (messageType == "error") then
		text = format("%s %s", prefixText, DLU.colorizeString(message, "red"));
	end

	print(text);
end

function DLU.numberFormat(number)
	if (number == nil or number <= 0) then
		return 0;
	elseif (number > 0 and number < 1000) then
		return number;
	elseif (number > 1000 and  number < 1000000) then
		return format("%.1f%s", number * 0.001, "K");
	elseif (number >= 1000000 and number < 1000000000) then
        return format("%.1f%s", number * 0.000001, "M");
    elseif (number >= 1000000000) then
        return format("%.1f%s", number * 0.000000001, "B");
	elseif (number >= 1000000000000) then
		return format("%.1f%s", number * 0.000000000001, "T");
	end

	return number;
end

function DLU.getClassColored(itemToColor, classFileName)
	local result = "|c" .. RAID_CLASS_COLORS[classFileName].colorStr .. itemToColor .. "|r";
	return result;
end

function DLU.RGBPercToHex(r, g, b)
	r = r <= 1 and r >= 0 and r or 0
	g = g <= 1 and g >= 0 and g or 0
	b = b <= 1 and b >= 0 and b or 0
	return string.format("%02x%02x%02x", r*255, g*255, b*255)
end

function DLU.itemColorString(num, text)
	if (text ~= nil) then
		local color = ITEM_QUALITY_COLORS[num];
		local result = color.hex .. text .. "|r";
		return result;
	end
	
	return;
end

function DLU.iconToText(iconPath)
	local result = format("|T%s:0|t", iconPath);
	return result;
end

function DLU.createNotification(message, notificationType)
	local sounds = {
		["Invasion"] = 14772,
		["Default"] = 4574
	};
	
	local sound = sounds["Default"];
	
	if (sounds[notificationType]) then
		sound = sounds[notificationType];
	end
	
	PlaySound(sound);
	UIFrameFlash(DLUNotificationFrame, 3, 3, 16, false, 10, 0);
	DLUNotification:SetText(message);
end

function DLU.boolToText(boolValue)
	if (boolValue) then
		return DLU.colorizeString("Yes", "green");
	end

	return DLU.colorizeString("No", "red");
end

function DLU.tableStatus(tableToCheck)
	local completionAmount = 0;
	for t, q in pairs(tableToCheck) do
		local completed = IsQuestFlaggedCompleted(q);
		if (completed) then
			completionAmount = completionAmount + 1;
		end

		local result = DLU.boolToText(completed);
		print(format("%s: %s", t, result));
	end

	print(format("You have completed %i/%i", completionAmount, DLU.tableLength(tableToCheck)));
end

function DLU.timeleft(minutes)

	local timeleftString = "Unknown...";

	if (minutes > 0) then
		local t = time() + minutes * 60;
		t = floor(t / 60) * 60;
		timeleftString = date("%c", t);
		if (date("%x", t) == date("%x")) then
			timeleftString = date("%X", t);
		end
	end

	return timeleftString;
end

function DLU.ArtifactXpLeftValue()
	local _, _, _, _, spendPower, currentTraits, _, _, _, _, _, _, artifactTier = C_ArtifactUI.GetEquippedArtifactInfo();
	
	if (spendPower ~= nil) then
		local result;
		local traitsWaitingForSpending, currentPower, powerForNextTrait = MainMenuBar_GetNumArtifactTraitsPurchasableFromXP(currentTraits, spendPower, artifactTier);
		
		if (powerForNextTrait > 0) then
			local apNeeded = powerForNextTrait - currentPower;
			local apPercentGained = math.floor((currentPower / powerForNextTrait) * 100);
			local apPercentNeeded = 100 - apPercentGained;
			
			local apFormat = BreakUpLargeNumbers(apNeeded);
			if (DLUSettings.numberFormat == "DarkruneDK") then
				apFormat = DLU.numberFormat(apNeeded);
			end
			
			result = format("%s", apFormat);
		else
			result = "Max level!";
		end
		
		return result, apPercentNeeded;
	end
	
	return nil;
end

function DLU.ArtifactTraits()
	local _, _, _, _, spendPower, currentTraits, _, _, _, _, _, _, artifactTier = C_ArtifactUI.GetEquippedArtifactInfo();
	
	if (spendPower ~= nil) then
		local traitsWaitingForSpending, currentPower, powerForNextTrait = MainMenuBar_GetNumArtifactTraitsPurchasableFromXP(currentTraits, spendPower, artifactTier);
		local currentRank = currentTraits + traitsWaitingForSpending;
		
		return currentRank, traitsWaitingForSpending, currentRank + 1;
	end
	
	return nil;
end

function DLU.ArtifactInfo()
	local _,_,name,_, spendPower, currentRank,_,_,_,_,_,_, artifactTier = C_ArtifactUI.GetEquippedArtifactInfo();
	if (name ~= nil and spendPower ~= nil) then
		return name, spendPower, currentRank;
	end
	
	return nil;
end

function DLU.XpLeft()
	local xpToNextLevel = (UnitXPMax("player") - UnitXP("player"));
	local xpRested = GetXPExhaustion();
	local xpRestedPercentage;
	local xpRestedFormat;
	
	if (xpRested ~= nil) then
		xpRestedPercentage = floor(xpRested / UnitXPMax("player")) * 100;
	end

	local xpPercentageGained = floor((UnitXP("player") / UnitXPMax("player")) * 100);
	local xpPercentageNeeded = 100 - xpPercentageGained;

	local xpDisabled = IsXPUserDisabled();
	local maxLevel = GetMaxPlayerLevel();
	
	local xpFormat = BreakUpLargeNumbers(xpToNextLevel);
	if (DLUSettings.numberFormat == "DarkruneDK") then
		xpFormat = DLU.numberFormat(xpToNextLevel);
	end
	
	if (xpRested) then
		xpRestedFormat = BreakUpLargeNumbers(xpRested);
		if (DLUSettings.numberFormat == "DarkruneDK") then
			xpRestedFormat = DLU.numberFormat(xpRested);
		end
	end
	
	return xpFormat, xpRestedFormat, xpPercentageNeeded, xpRestedPercentage, xpDisabled, maxLevel;
end

function DLU.XpLabelInfo()
	local nextLevel = DLU.player.level + 1;
	local maxLevel = GetMaxPlayerLevel();
	
	return nextLevel, maxLevel;
end

function DLU.honorLeftValues()
	local honorLeft = UnitHonorMax("player") - UnitHonor("player");
	local honorLevel = UnitHonorLevel("player");
	local honorMax = GetMaxPlayerHonorLevel();
	local prestigeLevel = UnitPrestige("player");
	
	local xpFormat = BreakUpLargeNumbers(honorLeft);
	if (DLUSettings.numberFormat == "DarkruneDK") then
		xpFormat = DLU.numberFormat(honorLeft);
	end
	
	return xpFormat, honorLevel, honorMax, prestigeLevel;
end

function DLU.getMovementSpeedInfo()
	local _, groundSpeed, flightSpeed, swimSpeed = GetUnitSpeed("player");

	local playerGroundSpeed = floor((groundSpeed / BASE_MOVEMENT_SPEED) * 100);
	local playerFlySpeed = floor((flightSpeed / BASE_MOVEMENT_SPEED) * 100);
	local playerSwimSpeed = floor((swimSpeed / BASE_MOVEMENT_SPEED) * 100);

	local data = {
		ground = playerGroundSpeed,
		flying = playerFlySpeed,
		swimming = playerSwimSpeed
	};

	return data;
end

local function getInventoryItemLevels()
	local averageIlvl, equippedIlvl, pvpIlvl = GetAverageItemLevel();
	local multiplyer = 100;
	averageIlvl = floor(multiplyer * averageIlvl)/multiplyer;
	equippedIlvl = floor(multiplyer * equippedIlvl)/multiplyer;
	pvpIlvl = floor(multiplyer * pvpIlvl) / multiplyer;
	local iLvls = {total = averageIlvl, equipped = equippedIlvl, pvp = pvpIlvl};
	return iLvls;
end

function DLU.getItemLevel()
	local iLvls = getInventoryItemLevels();
	local isInstance, instanceType = IsInInstance()
	if (UnitInBattleground("player")) then
		return iLvls.pvp;
	else
		return iLvls.equipped;
	end
end

-- local variables
local helpList;

local function ArtifactXpLeft()
	
	local apNeeded, apPercentNeeded = DLU.ArtifactXpLeftValue();
	local currentRank, traitsWaitingForSpending, nextRank = DLU.ArtifactTraits();
	
	if (apNeeded ~= nil and currentRank ~= nil) then
		
		local specId, _, _, _, role = GetSpecializationInfo(GetSpecialization());
		local artifactLink = GetInventoryItemLink("player", 16);
		if (specId == 73 or specId == 66) then
			artifactLink = GetInventoryItemLink("player", 17);
		end
		
		local upgradeText = "upgrade";
		local text = "You need %s AP (%i%%) to get %s to rank: %i";
		if (traitsWaitingForSpending > 0) then
			text = text .. " (%i trait %s)";
			if (traitsWaitingForSpending > 1) then
				upgradeText = upgradeText .. "s";
			end
		end
		
		return format(text, apNeeded, apPercentNeeded, artifactLink, nextRank, traitsWaitingForSpending, upgradeText);
	end
	
	return false;
end

local function xpLeft()
	local text = nil;
	local expansionLevel = GetExpansionLevel();
	local xpFormat, xpRestedFormat, xpPercentageNeeded, xpRestedPercentage, xpDisabled, maxLevel = DLU.XpLeft();

	if (DLU.player.level == maxLevel) then
		text = "You are currently the highest level you can be, congratz :)";
	else
		local nextLevel = DLU.player.level + 1;
		if (xpDisabled) then
			text = "You can't get more xp, because your xp has been disabled.";
		else
			text = "You need %s xp (%i%%) to get to level %i.";
			if (xpRestedFormat ~= nil) then
				text = text .. " (currently have %s rested xp)";
			end
		end

		text = format(text, xpFormat, xpPercentageNeeded, nextLevel, DLU.itemColorString(3, xpRestedFormat));
	end

	if (expansionLevel >= 6) then
		local apXpLeft = ArtifactXpLeft();
		if (apXpLeft ~= false and DLU.player.level == maxLevel) then
			text = apXpLeft;
		elseif (apXpLeft ~= false) then
			text = text .. "\n" .. apXpLeft;
		end
	end

	print(text);
end

local function mountUp()
	local isOutdoors = IsOutdoors();
	if (isOutdoors) then
		local sumMountId = 0;
		local swimming = IsSwimming();
		local buffs = {240980, 3714};
		local hasBuff = false;

		for buff in pairs(buffs) do
			if (UnitBuff("player", buff)) then
				hasBuff = true;
			else
				hasBuff = false;
			end
		end

		-- Get players riding skill to check if it should use leveling mount
		local riding_skill = DLU.player.getRidingSkill();

		if (riding_skill == 0 and #DLU.mountTable.lowLevelMounts > 0) then
			index = random(#DLU.mountTable.lowLevelMounts);
			sumMountId = DLU.mountTable.lowLevelMounts[index];
			C_MountJournal.SummonByID(sumMountId);
		elseif (swimming and #DLU.mountTable.waterMounts > 0 and not hasBuff) then
			index = random(#DLU.mountTable.waterMounts);
			sumMountId = DLU.mountTable.waterMounts[index];
			C_MountJournal.SummonByID(sumMountId);
		elseif (UnitInBattleground("player") and #DLU.mountTable.pvpMounts > 0) then
			index = random(#DLU.mountTable.pvpMounts);
			sumMountId = DLU.mountTable.pvpMounts[index];
			C_MountJournal.SummonByID(sumMountId);
		else
			C_MountJournal.SummonByID(sumMountId);
		end
	else
		DLU.createMessage("You need to be outdoors to use mounts.", "error");
	end
end

-- legion specific functions
local function checkSuramarManaStatus()
	local manaIncreaseQuests = {
		["Feeding Shal'Aran (quest)"] = 41138,
		["The Valewalker's Burden (quest)"] = 42230,
		["Thalyssra's Abode (quest)"] = 42488,
		["How It's Made: Arcwine (quest)"] = 42833,
		["Make Your Mark (quest)"] = 42792,
		["Kel'danath's Manaflask (item)"] = 42842,
		["Volatile Leyline Crystal (item)"] = 43988,
		["Infinite Stone (item)"] = 43989,
		["Enchanted Burial Urn (item)"] = 43986,
		["Kyrtos's Research Notes (item)"] = 43987
	};

	DLU.tableStatus(manaIncreaseQuests);
end

local function checkLeylineStatus()
	local leylineQuests = {
		["Leyline Feed: Elor'shan"] = 43587,
		["Leyline Feed: Falanaar Arcway"] = 43592,
		["Leyline Feed: Falanaar Depths"] = 43593,
		["Leyline Feed: Halls of the Eclipse"] = 43594,
		["Leyline Feed: Kel'balor"] = 43588,
		["Leyline Feed: Ley Station Aethenar"] = 43591,
		["Leyline Feed: Ley Station Moonfall"] = 43590,
		["Tapping the Leylines (main quest)"] = 40010
	};

	DLU.tableStatus(leylineQuests);
end

-- Character Stats
DLU.StatFrames = {
	["ItemLevelFrame"] = {
		frame = CharacterStatsPane.ItemLevelFrame,
		updateFunc = function(statFrame, unit)
			ilvl = DLU.getItemLevel();
			PaperDollFrame_SetLabelAndText(statFrame, STAT_AVERAGE_ITEM_LEVEL, ilvl, false, ilvl);
			statFrame.tooltip = HIGHLIGHT_FONT_COLOR_CODE .. format("Item Level: %s", ilvl) .. FONT_COLOR_CODE_CLOSE;
		end
	}
}

function DLU.checkStats()
	local addonList = {"DejaCharacterStats"}; -- List of addons, that already changes the char stats window
	local disableCharStats = false;
	
	for _, v in pairs(addonList) do
		local _, _, _, enabled = GetAddOnInfo(v);
		if (enabled) then
			disableCharStats = true;
			break;
		end
	end
	
	return disableCharStats;
end

local hideStats = DLU.checkStats();
if (not hideStats) then
	for k, v in pairs(DLU.StatFrames) do
		local index = 0;
		if (not v.frame) then
			local category = PAPERDOLL_STATCATEGORIES[1];
			local categoryFrame = category.categoryFrame;

			for _, v in pairs(CharacterStatsPane[categoryFrame]) do
				if (v) then
					index = index+1;
				end
			end

			v.frame = CreateFrame("Frame", nil, CharacterStatsPane[categoryFrame], "CharacterStatFrameTemplate");
			local height = v.frame:GetHeight();
			local xCord = -(index*height);
			v.frame:SetPoint("BOTTOMLEFT", 4, xCord);
			index = index+1;
		end
	end
end

function DLU.showStats(unit)
	if (not hideStats) then
		for _, v in pairs(DLU.StatFrames) do
			if (v.updateFunc) then
				if (v.frame) then
					v.updateFunc(v.frame, unit);
				end
			end
		end
	end
end

local function testIt()

end

local function createHelpList()
	helpList = {
		[1] = {
			["Command"] = SLASH_RELOADUI1,
			["Explanation"] = "Short version of the in-build /reload"
		},
		[2] = {
			["Command"] = SLASH_DLU1 .. " xpleft",
			["Explanation"] = "Tells how much xp left to next level"
		},
		[3] = {
			["Command"] = SLASH_DLU1 .. " ms",
			["Explanation"] = "Tells the characters different types of movement speed"
		},
		[4] = {
			["Command"] = SLASH_DLU1 .. " il or " .. SLASH_DLU1 .. " ilvl",
			["Explanation"] = "Tells you the characters item level"
		},
		[5] = {
			["Command"] = SLASH_DLU1 .. " mount",
			["Explanation"] = "Uses a mount appropriate for current situation"
		},
		-- Legion Specific
		[6] = {
			["Command"] = SLASH_DLU1 .. " sms",
			["Explanation"] = "Tells you how many ancient mana upgrades the characters has and which it needs"
		},
		[7] = {
			["Command"] = SLASH_DLU1 .. " sls",
			["Explanation"] = "Tells you how leyline upgrades the characters has and which it needs"
		},
		-- World Quests
		[8] = {
			["Command"] = SLASH_DLU1 .. " wql",
			["Explanation"] = "Prints all the loaded World Quest titles into the chat window"
		},
		[9] = {
			["Command"] = SLASH_DLU1 .. " wqr",
			["Explanation"] = "Refreshes the world quest list used by " .. SLASH_DLU1 .. " wql"
		},
		-- UI
		[10] = {
			["Command"] = SLASH_DLU1 .. " hq",
			["Explanation"] = "Opens the DarkruneDK HQ"
		}
	};
end

local function addonHelp()
	print("Following commands can be used with " .. DLU.addonNameColored .. ":");
	
	if (helpList == nil) then
		createHelpList();
	end
	
	for k,v in ipairs(helpList) do
		local command = DLU.colorizeString(v["Command"], "green");
		local text = v["Explanation"];
		print(format("%s (%s)", command, text));
	end
	
	print("You can use the commands with macros if you want easier/faster execution.");
end

-- Addon commands --
SLASH_RELOADUI1 = "/rl";
SlashCmdList.RELOADUI = ReloadUI;

SLASH_XP_HELP1 = "/dluhelp";
SlashCmdList.XP_HELP = function()
	print(format(addonData.L.HelpErrorText, addonName));
end

SLASH_DLU1 = "/dlu";
SlashCmdList.DLU = function(commandName)
	local command, arg2 = strsplit(" ", commandName);
	local arg = command:lower();

	if (arg == "") then
		InterfaceOptionsFrame_OpenToCategory(DLU.addonName);
	elseif (arg == "help") then
		addonHelp();
	elseif (arg == "test") then
		testIt();
	elseif (arg == "il" or commandName:lower() == "ilvl") then
		local iLvl = DLU.getItemLevel();
		local _, _, _, _, _, _, _, _, _, _, _, _, superiorCompleted = GetAchievementInfo(10764);
		local _, _, _, _, _, _, _, _, _, _, _, _, epicCompleted = GetAchievementInfo(10765);
		if (epicCompleted and iLvl > 840) then
			iLvl = DLU.itemColorString(4, iLvl);
		elseif (superiorCompleted and iLvl > 820) then
			iLvl = DLU.itemColorString(3, iLvl);
		end

		print(string.format("Your item level is: %s", iLvl));

	elseif (arg == "mount") then
		mountUp();
	elseif (arg == "xpleft") then
		xpLeft();
	elseif (arg == "honorleft") then
		DLU.honorLeft()
	elseif (arg == "xp") then
		if (UnitInBattleground("player")) then
			DLU.honorLeft()
		else
			xpLeft();
		end
	elseif (arg == "ms") then
		local data = DLU.getMovementSpeedInfo();
		print(format("Ground speed: %i%% | Flying speed: %i%% | Swimming speed: %i%%", data.ground, data.flying, data.swimming));
	elseif (arg == "hq") then
		DLUHQ:Show();
	-- Legion specific commands
	elseif (arg == "sms") then
		checkSuramarManaStatus();
	elseif (arg == "sls") then
		checkLeylineStatus();
	elseif (arg == "wql") then
		DLU.WQDetails();
	elseif (arg == "wqr") then
		DLU.WorldQuestRefreshTable();
	elseif (arg == "es") then
		DLU.EmissaryStatus();
	elseif (arg == "rms") then
		local pages = { 45470, 47207, 47208, 47209, 47210, 47211, 47212, 47213 };
		local pageNumbers = { 9, 78, 161, 665, 845, 1127, 2351, 5555 }
		local completionAmount = 0;

		for i, q in ipairs(pages) do
			local pageStatus = IsQuestFlaggedCompleted(q);
			local status = DLU.colorizeString("Not done", "red");
			if (pageStatus) then
				completionAmount = completionAmount + 1;
				status = DLU.colorizeString("Done", "green");
			end
			print(format("Page %i: %s", pageNumbers[i], status));
		end

		local message = format("Completed %i/%i", completionAmount, #pages);
		if (completionAmount == #pages) then
			message = "Now you only need to go to westfall and find the chest at a boat :-)";
		end

		print(message);
	elseif (arg == "pet" and arg2) then
		local petID = C_PetJournal.GetPetLoadOutInfo(arg2);
		DLU.checkBattlePetBreed(petID);
	else
		DLU.createMessage("Unknown command! Check '/dlu help' if you are uncertain.", "error");
	end
end

-- Functions --
local function loadPlayerMounts()
	local mountCount = C_MountJournal.GetMountIDs();

	for i = 1, #mountCount do
		local _, _, _, _, isUsable, _, _, _, _, _, _, mountID = C_MountJournal.GetMountInfoByID(mountCount[i]);
		if (isUsable) then
			-- Water mounts
			if (mountID == 449 or mountID == 488) then
				table.insert(DLU.mountTable.waterMounts, mountID);
			end
			-- PvP mounts
			if (mountID > 74 and mountID < 83 or mountID == 272 or mountID == 108 or mountID == 162 or mountID == 220 or mountID == 338 or mountID == 305 or mountID > 293 and mountID < 304 or mountID == 330 or mountID == 332 or mountID == 423 or mountID == 555 or mountID == 641 or mountID == 756 or mountID == 784 or mountID > 841 and mountID < 844 and mountID == 874 and mountID == 882 and mountID == 901 and mountID == 946) then
				table.insert(DLU.mountTable.pvpMounts, mountID);
			end
			-- Low level mounts
			if (mountID == 678 or mountID == 679) then
				table.insert(DLU.mountTable.lowLevelMounts, mountID);
			end
		end
	end
end

-- Excecute --
if (IsAddOnLoaded(DLU.addonName)) then
	DLU.createMessage(format(addonData.L.Initialized .. " " .. addonData.L.HelpText, DLU.addonNameColored, SLASH_DLU1 .. " help"));
end

-- Registering Event --
DLUCoreFrame:RegisterEvent("PLAYER_LOGIN");
DLUCoreFrame:SetScript("OnEvent", function()

DLU.getSettings();
loadPlayerMounts();

DLUCoreFrame:UnregisterEvent("PLAYER_LOGIN")
end);
