local questCompanionFrame, events = CreateFrame("Frame", "DLUQuestCompanionFrame"), {};
local settingsFrame = CreateFrame("Frame", "DLUSettingsFrame");

local mapIds = {};
local brokenIslesMapIds = {1014, 1015, 1017, 1018, 1021, 1024, 1033, 1096};
local argusMapIds = {1135, 1170, 1171};
DLU.tableMerge(mapIds, brokenIslesMapIds);
DLU.tableMerge(mapIds, argusMapIds);

for k,v in pairs(mapIds) do
	DLU.worldQuestTable[v] = {};
end

local showMessage = true;
local lastRefreshHour, lastRefreshMinute = nil;
local invasionWarning = false;

local WQIcons = {
	[1] = "Interface\\ICONS\\Trade_Engineering",
	[3] = "Interface\\PVPFrame\\Icons\\prestige-icon-3",
	[4] = "Interface\\ICONS\\Pet_Type_Beast",
	[6] = "Interface\\MINIMAP\\Dungeon",
	[7] = "Interface\\MINIMAP\\TempleofKotmogu_ball_green",
	[8] = "Interface\\MINIMAP\\Raid",
	["Elite"] = "Interface\\Tooltips\\EliteNameplateIcon"
};

-- Event functions
function events:WORLD_MAP_UPDATE(...)
	if (DLUSettings.WQTracking.value) then
		DLU.emissariesTable = DLU.getEmissaries();
		
		local currentGameTimeHour, currentGameTimeMinute = GetGameTime();
		if (lastRefreshHour ~= currentGameTimeHour and lastRefreshMinute == currentGameTimeMinute) then
			DLU.WorldQuestRefreshTable();
		else
			DLU.loadWQuests();
		end
	end
end

function events:QUEST_REMOVED(...)
	for i = 1, #mapIds do
		local zoneId = mapIds[i];
		local questInfo = DLU.worldQuestTable[zoneId];
		for k, v in pairs(questInfo) do
			local isCompleted = IsQuestFlaggedCompleted(v.id);
			if (isCompleted) then
				local message = format("%s completed!", v.title);
				DLU.createMessage(message);
				if (DLUSettings.WQPartyNotification.value) then
					DLU.notifyParty(message);
				end
				table.remove(questInfo, k);
			end
		end
	end
	
	DLU.updateEmissaries();
end

-- Globals
function DLU.loadWQuests()
	if (DLUSettings.WQTracking.value) then
		local currentMapId = GetCurrentMapAreaID();
		
		if (currentMapId == 1007 or currentMapId == 1184) then
			if (showMessage) then
				DLU.createMessage("Loading World Quests...");
			end
			
			for i = 1, #mapIds do
				if (mapIds[i] ~= nil) then
					local zoneId = mapIds[i];
					local currentTaskInfo = C_TaskQuest.GetQuestsForPlayerByMapID(zoneId);
					if (currentTaskInfo ~= nil) then
						for j=1, #currentTaskInfo do
							if (QuestUtils_IsQuestWorldQuest(currentTaskInfo[j].questId)) then
								DLU.WQInsert(currentTaskInfo[j], zoneId);
							end
						end
					end
				end
			end
		end
		
		--DLU.updateWQL();
		
		if (showMessage) then
			for i = 1, #mapIds do
				local zoneId = mapIds[i];
				local questInfo = DLU.worldQuestTable[zoneId];
				local questAmount = #questInfo;
				local zoneName = GetMapNameByID(mapIds[i]);
				if (questAmount > 0) then
					DLU.createMessage(format("%i %s World Quests loaded...", questAmount, zoneName));
				end
			end
		end
		
		showMessage = false;
	end
end

function DLU.WorldQuestRefreshTable()
	DLU.createMessage("Refreshing World Quests (could take a few seconds)...");
	
	lastRefreshHour, lastRefreshMinute = GetGameTime();
	invasionWarning = false;
	
	for i = 1, #mapIds do
		local zoneId = mapIds[i];
		local questInfo = DLU.worldQuestTable[zoneId];
		for k in pairs(questInfo) do
			DLU.worldQuestTable[zoneId][k] = nil;
		end
	end
	
	for j = 1, #mapIds do
		if (mapIds[j] ~= nil) then
			local zoneId = mapIds[j];
			local currentTaskInfo = C_TaskQuest.GetQuestsForPlayerByMapID(zoneId);
			if (currentTaskInfo ~= nil) then
				for f = 1, #currentTaskInfo do
					if (QuestUtils_IsQuestWorldQuest(currentTaskInfo[f].questId)) then
						DLU.WQInsert(currentTaskInfo[f], zoneId);
					end
				end
			end
		end
	end
end

--[[
	World Quest types:
	1 = prof
	2 = normal
	3 = PvP
	4 = Pet Battle
	6 = Dungeon (Interface\MINIMAP\Dungeon)
	7 = Legion Invasion
	8 = Raid (Interface\MINIMAP\Raid)
	
	Maybe = (Interface\QuestTypeIcons)
	Elite (Interface\Tooltips\EliteNameplateIcon)
	Rare Elite (Interface\Tooltips\RareEliteNameplateIcon)
	
	Tradeskills:
	6 = Cooking
	7 = Fishing
	8 = Primary profession (Mining, Leatherworking etc)
]]
function DLU.WQInsert(currentTask, zoneId)
	
	local questId = currentTask.questId;
	local title, factionId = C_TaskQuest.GetQuestInfoByQuestID(questId);
	local _, _, WQType, rarity, isElite, tradeskillLineIndex = GetQuestTagInfo(questId);
	
	local WQ = {
		id = currentTask.questId,
		title = title,
		factionId = factionId,
		type = WQType,
		rarity = rarity,
		isElite = isElite,
		profSkillIndex = tradeskillLineIndex,
		timeLeft = DLU.timeleft(C_TaskQuest.GetQuestTimeLeftMinutes(questId))
	};
	
	if (WQ.profSkillIndex == nil and WQ.factionId ~= nil) then
		local factionName = GetFactionInfoByID(factionId);
		WQ["factionName"] = factionName;
	elseif (WQ.profSkillIndex ~= nil) then
		local _, icon = GetProfessionInfo(WQ.profSkillIndex);
		WQ["Icon"] = icon;
	end
	
	local isEmissary, emissaryIcon = DLU.WQCheckEmissary(WQ);
	WQ["isEmissary"] = isEmissary;
	WQ["emissaryIcon"] = emissaryIcon;
	
	if (GetNumQuestLogRewards(WQ.id) > 0) then
		local name, icon, rewardAmount, quality = GetQuestLogRewardInfo(1, WQ.id);
		
		--[[
		if (rewardAmount > 1) then
			WQ.reward = format("%s%i %s", DLU.iconToText(icon), rewardAmount, name);
		else
			WQ.reward = format("%s %s", DLU.iconToText(icon), name);
		end
		]]
		if (quality and quality >= LE_ITEM_QUALITY_COMMON and ITEM_QUALITY_COLORS[quality]) then
			--WQ.reward = DLU.itemColorString(quality, WQ.reward);
			WQ.rewardIcon = icon;
		end
	end
	
	if (WQ.type == LE_QUEST_TAG_TYPE_RAID or WQ.type == LE_QUEST_TAG_TYPE_DUNGEON or WQ.type == LE_QUEST_TAG_TYPE_PVP or WQ.type == LE_QUEST_TAG_TYPE_PET_BATTLE or WQ.type == LE_QUEST_TAG_TYPE_INVASION) then
		WQ.title = format("%s%s", DLU.iconToText(WQIcons[WQ.type]), WQ.title);
	elseif (WQ.type == LE_QUEST_TAG_TYPE_PROFESSION) then
		if (WQ.rewardIcon ~= nil) then
			WQ.title = format("%s%s", DLU.iconToText(WQ.rewardIcon), WQ.title);
		else
			WQ.title = format("%s%s", DLU.iconToText(WQ.Icon), WQ.title);
		end
	elseif (WQ.isElite) then
		WQ.title = format("%s%s", DLU.iconToText(WQIcons["Elite"]), WQ.title);
	end
	
	if (WQ.rarity ~= nil) then
		local result = rarity;
		if (WQ.type == 8) then
			result = 4;
		elseif (WQ.rarity == 2) then
			result = 3;
		end
		
		WQ.title = DLU.itemColorString(result, WQ.title);
	end
	
	local exists = false;
	local questInfo = DLU.worldQuestTable[zoneId];
	for _, info in pairs(DLU.worldQuestTable[zoneId]) do
		if (info.id == WQ.id) then
			exists = true;
		end
	end
	
	if (exists ~= true) then
		DLU.worldQuestTable[zoneId][#DLU.worldQuestTable[zoneId] + 1] = WQ;
		if (WQ.type == LE_QUEST_TAG_TYPE_INVASION and invasionWarning == false) then
			local zoneName = GetMapNameByID(zoneId);
			DLU.createNotification(format("Legion invasion detected in %s (Disappears at: %s)", zoneName, WQ.timeLeft), "Invasion");
			invasionWarning = true;
		end
	end
end

function DLU.WQDetails()
	for i = 1, #mapIds do
		local zoneId = mapIds[i];
		local zoneName = GetMapNameByID(zoneId);
		local zonesShown = 0;
		
		if (DLU.worldQuestTable[zoneId] ~= nil) then
			local questAmount = #DLU.worldQuestTable[zoneId];
			if (questAmount > 0) then
				DLU.createMessage(format("%i World Quests in %s:", questAmount, zoneName));
				for j = 1, #DLU.worldQuestTable[zoneId] do
					local WQ = DLU.worldQuestTable[zoneId][j];
					local title = WQ.title;
					if (WQ.isEmissary) then
						title = format("%s %s", DLU.iconToText(WQ.emissaryIcon), DLU.itemColorString(5, title));
					end
					
					if (WQ.reward) then
						print(format("%s gives %s (disappears at: %s)", title, WQ.reward, WQ.timeLeft));
					else
						print(format("%s (disappears at: %s)", title, WQ.timeLeft));
					end
					
				end
				zonesShown = zonesShown +1;
			end
		end
	end
	
	if (zonesShown == 0) then
		DLU.createMessage("No World Quests found...")
	end
end

-- Emissaries
function DLU.EmissaryCheck(bountyData)
	local numCompleted = 0;
	local numTotal = 0;
	
	for i = 1, bountyData.numObjectives do
		local objectiveText, _, _, numFulfilled, numRequired = GetQuestObjectiveInfo(bountyData.questID, i, false);
		if objectiveText and #objectiveText > 0 and numRequired > 0 then
			for objectiveSubIndex = 1, numRequired do
				if objectiveSubIndex <= numFulfilled then
					numCompleted = numCompleted + 1;
				end
				numTotal = numTotal + 1;

				if numTotal >= MAX_BOUNTY_OBJECTIVES then
					return numCompleted, numTotal;
				end
			end
		end
	end
	
	return numCompleted, numTotal;
end

function DLU.getEmissaries()
	local bounty = GetQuestBountyInfoForMapID(1007);
	local data = {};
	
	if (bounty and type(bounty) == "table") then	
		local index = 1;
		
		for _, bountyData in pairs(bounty) do
			local completed, required = DLU.EmissaryCheck(bountyData);
			local questIndex = GetQuestLogIndexByID(bountyData.questID);
			local factionName = GetQuestLogTitle(questIndex);
			
			data[index] = {
				questId = bountyData.questID,
				completed = completed,
				required = required,
				questIndex = questIndex,
				factionName = factionName,
				icon = bountyData.icon
			}
			
			index = index+1;
		end
	end
	
	return data;
end

function DLU.EmissaryStatus()
	local notification = "Your emissaries: ";
	for _, bountyData in pairs(DLU.emissariesTable) do
		local questIndex = GetQuestLogIndexByID(bountyData.questId);
		local factionName = GetQuestLogTitle(questIndex);
		if (bountyData.completed == bountyData.required) then
			notification = format("%s %s %s (ready for turn-in)", notification, DLU.iconToText(bountyData.icon), factionName);
		else
			notification = format("%s %s %s (%i/%i)", notification, DLU.iconToText(bountyData.icon), factionName, bountyData.completed, bountyData.required);
		end
	end
	
	DLU.createNotification(notification);
end

function DLU.WQCheckEmissary(WQ)
	local isEmissary = false;
	for _, bountyData in pairs(DLU.emissariesTable) do
		isEmissary = IsQuestCriteriaForBounty(WQ.id, bountyData.questId);
		if (isEmissary) then
			return isEmissary, bountyData.icon;
		end
	end
	
	return isEmissary, nil;
end

function DLU.getRefreshTime()
	return format("%i:%i", lastRefreshHour, lastRefreshMinute);
end

function DLU.CheckEmissaries()
	local emissariesCount = #emissariesTable;
	local WQEmissariesCount = #WQCompanion.emissaries;
	
	for i = 1, WQEmissariesCount do
		local questId = WQCompanion.emissaries[i].questID;
		local eQuestId = emissariesTable[i].questID;
		if (questId ~= eQuestId) then
			WQCompanion.emissaries[i] = nil;
			if (WQCompanion.emissaries.ui and WQCompanion.emissaries.ui[i]) then
				WQCompanion.emissaries.ui[i] = nil;
			end
		end
	end
end

--[[
function DLU.updateWQL()
	local line = 1;
	
	for i = 1, #mapIds do
		local zoneId = mapIds[i];
		local zoneName = GetMapNameByID(zoneId);
		
		if (DLU.worldQuestTable[zoneId] ~= nil) then
			local questAmount = #DLU.worldQuestTable[zoneId];
			if (questAmount > 0) then
				for j = 1, questAmount do
					local WQ = DLU.worldQuestTable[zoneId][j];
					local title = WQ.title;
					if (WQ.isEmissary) then
						title = format("%s %s", DLU.iconToText(WQ.emissaryIcon), DLU.itemColorString(5, title));
					end
					
					local text = format("%s (disappears at: %s)", title, WQ.timeLeft);
					if (WQ.reward) then
						text = format("%s gives %s (disappears at: %s)", title, WQ.reward, WQ.timeLeft);
					end
					
					WQCompanion.WQL.Content.lines[line].text:SetText(text);
					line = line+1;
				end
			end
		end
	end
end
]]

-- Initialize
settingsFrame:RegisterEvent("PLAYER_LOGIN");
settingsFrame:SetScript("OnEvent", function()
	DLU.getSettings();
	
	if (DLUSettings.WQTracking.value) then
		lastRefreshHour, lastRefreshMinute = GetGameTime();
		
		questCompanionFrame:SetScript("OnEvent", function(self, event, ...)
			events[event](self, ...);
		end);
		for k, v in pairs(events) do
			questCompanionFrame:RegisterEvent(k);
		end
		
		DLU.emissariesTable = DLU.getEmissaries();
	end
end);