local addonName = ...;

local contributionEtaData = {
	[1] = {}, -- Mage Tower
	[3] = {}, -- Command Center
	[4] = {} -- Nether Disruptor
};

local contribFrame = CreateFrame("Frame");
contribFrame.debug = false;
contribFrame:RegisterEvent("ADDON_LOADED");
contribFrame:RegisterEvent("CONTRIBUTION_COLLECTOR_UPDATE");
contribFrame:RegisterEvent("CONTRIBUTION_COLLECTOR_UPDATE_SINGLE");

function contribFrame:UpdateSingle(contributionID)
	local etaData = contributionEtaData[contributionID];
	if not contributionEtaData[contributionID] then return end
	local state, pct = C_ContributionCollector.GetState(contributionID);
	if self.debug then
		local contributionName = C_ContributionCollector.GetName(contributionID);
		print(("%s: id %u, state %u, pct %f"):format(contributionName, contributionID, state, pct));
	end
	if state == 1 then
		if not etaData.hasData then
			etaData.hasData = true;
			etaData.startTime = time();
			etaData.startPct = pct;
			etaData.eta = 0;
		end
		local pctDiff = pct - etaData.startPct;
		local timeDiff = time() - etaData.startTime;
		if timeDiff > 1800 then -- reset every 30 min to show more accurate data for long sessions
			etaData.startTime = time();
			etaData.startPct = pct;
			etaData.eta = 0;
			timeDiff = 0;
			pctDiff = 0;
		end
		if timeDiff < 1 or pctDiff < 0.00001 then return end
		local buildSpeed = pctDiff * 100 / timeDiff; -- pct/sec
		local pctLeft = 100 - pct * 100; -- pct
		etaData.eta = pctLeft / buildSpeed; -- pct / pct/sec
		if self.debug then
			print(("pctDiff %f, timeDiff %f, buildSpeed %f, eta %f sec, eta %s"):format(pctDiff, timeDiff, buildSpeed, etaData.eta, SecondsToTime(etaData.eta)));
		end
	else
		etaData.hasData = false;
		etaData.startTime = 0;
		etaData.startPct = 0;
		etaData.eta = 0;
	end
end

function contribFrame:Update()
	for k, _ in pairs(contributionEtaData) do
		self:UpdateSingle(k);
	end
end

function contribFrame:OnEvent(event, ...)
	if event == "CONTRIBUTION_COLLECTOR_UPDATE" then
		self:Update();
	elseif event == "CONTRIBUTION_COLLECTOR_UPDATE_SINGLE" then
		local contributionID = ...;
		self:UpdateSingle(contributionID);
	elseif event == "ADDON_LOADED" then
		local addon = ...;
		if addon ~= addonName then return end
		--print(addon .. " loaded!");
		C_Timer.After(6, function() C_ContributionCollector.GetManagedContributionsForCreatureID(119305) end);
	end
end

contribFrame:SetScript("OnEvent", contribFrame.OnEvent);
contribFrame:Update();

local etaTooltipTextFmt = " (" .. CONTRIBUTION_POI_TOOLTIP_REMAINING_TIME .. ")";

local function MyWorldMapPOI_AddContributionsToTooltip(tooltip, ...)
	for i = 1, select("#", ...) do
		local contributionID = select(i, ...);
		local contributionName = C_ContributionCollector.GetName(contributionID);
		local state, stateAmount, timeOfNextStateChange = C_ContributionCollector.GetState(contributionID);
		local appearanceData = CONTRIBUTION_APPEARANCE_DATA[state];

		if i ~= 1 then
			tooltip:AddLine(" ");
		end

		tooltip:AddLine(contributionName, HIGHLIGHT_FONT_COLOR:GetRGB());

		local tooltipLine = appearanceData.tooltipLine;
		if tooltipLine then
			if timeOfNextStateChange and appearanceData.tooltipUseTimeRemaining then
				local time = timeOfNextStateChange - GetServerTime()
				tooltipLine = tooltipLine:format(SecondsToTime(time, false, true, 2));
			elseif timeOfNextStateChange and (state == 2 or state == 4) then
				local time = timeOfNextStateChange - GetServerTime();
				tooltipLine = tooltipLine .. etaTooltipTextFmt:format(SecondsToTime(time, false, true, 2));
			elseif contributionEtaData[contributionID] and contributionEtaData[contributionID].hasData and contributionEtaData[contributionID].eta > 0 then
				local etaData = contributionEtaData[contributionID];
				tooltipLine = tooltipLine:format(FormatPercentage(stateAmount)) .. etaTooltipTextFmt:format("~" .. SecondsToTime(etaData.eta, false, true, 2));
			else
				tooltipLine = tooltipLine:format(FormatPercentage(stateAmount));
			end

			tooltip:AddLine(tooltipLine, appearanceData.stateColor:GetRGB());
		end
	end
end

local function MyWorldMapPOI_OnEnter(self)
	if ( self.specialPOIInfo and self.specialPOIInfo.onEnter ) then
		return
	else
		if self.landmarkType == LE_MAP_LANDMARK_TYPE_CONTRIBUTION then
			WorldMapTooltip:SetOwner(self, "ANCHOR_RIGHT");
			WorldMapTooltip:SetText(self.name, HIGHLIGHT_FONT_COLOR:GetRGB());
			WorldMapTooltip:AddLine(" ");

			MyWorldMapPOI_AddContributionsToTooltip(WorldMapTooltip, C_ContributionCollector.GetManagedContributionsForCreatureID(self.mapLinkID));

			WorldMapTooltip:Show();
		end
	end
end

hooksecurefunc("WorldMapPOI_OnEnter", MyWorldMapPOI_OnEnter);
