-- BuffGuard - Enhanced Customizable buff watcher addon
-- Author: Justinction
-- Version: 2.2 - Major UI overhaul with tabbed interface and enhanced customization

local addonName, addonTable = ...

local addonName = "BuffGuard"

-- Constants
local CONSTANTS = {
    WARNING_THRESHOLD = 60,
    MAX_BUTTONS = 10,
    UPDATE_INTERVAL = 1.0,
    ALERT_COOLDOWN = 3,
    CACHE_DURATION = 0.5,
    FLOATING_TEXT_DURATION = 5.0
}

-- Default configuration
local defaults = {
    enabled = true,
    numButtons = 5,
    buttonWidth = 200,
    buttonHeight = 30,
    fontSize = 12,
    fontStyle = "OUTLINE", -- OUTLINE, THICK, MONOCHROME, or ""
    scale = 1.0, -- Overall scale multiplier
    position = { x = 0, y = 0 },
    hideBlizzardBuffs = false,
    minimapButtonAngle = 45,
    audioAlerts = true,
    warningSound = "igQuestLogAbandonQuest",
    expiredSound = "igQuestFailed",
    buffs = {},
    framePositions = {},
    customAlerts = {},
    -- Additional useful settings
    showTimer = true, -- Show/hide timer text
    showIcon = true, -- Show/hide spell icons
    backgroundColor = { r = 0.1, g = 0.1, b = 0.1, a = 0.8 }, -- Button background color
    borderColor = { r = 0.5, g = 0.5, b = 0.5, a = 1.0 }, -- Button border color
    lockFrames = false, -- Lock frames to prevent accidental movement
    showTooltips = true -- Enable/disable tooltips
}

-- Main addon object - make sure it's a singleton
if not _G.BuffGuard then
    _G.BuffGuard = {
        buttons = {},
        buffStates = {},
        buffCache = {},
        cacheTime = 0,
        floatingTexts = {},
        eventFrame = nil,
        mainFrame = nil,
        configFrame = nil,
        minimapButton = nil,
        buffEditBoxes = {},
        initialized = false
    }
end

local BuffGuard = _G.BuffGuard

-- Performance monitoring
local PerformanceMonitor = {
    timers = {}
}

function PerformanceMonitor:StartTimer(name)
    self.timers[name] = debugprofilestop()
end

function PerformanceMonitor:EndTimer(name)
    if not self.timers[name] then return end
    local elapsed = debugprofilestop() - self.timers[name]
    if elapsed > 5 then
        -- Performance monitoring can be silent now
    end
    self.timers[name] = nil
end

-- Utility functions
local function ValidateSpellName(spellName)
    if not spellName or spellName == "" then
        return false, "Empty spell name"
    end
    
    local spellID = select(7, GetSpellInfo(spellName))
    if not spellID then
        return false, "Spell not found"
    end
    
    return true, spellID
end

local function CapitalizeSpellName(spellName)
    if not spellName or spellName == "" then
        return spellName
    end
    
    local words = {}
    for word in string.gmatch(spellName, "%S+") do
        local lowerWord = string.lower(word)
        if lowerWord == "of" or lowerWord == "the" then
            table.insert(words, lowerWord)
        else
            local firstLetter = string.upper(string.sub(word, 1, 1))
            local restOfWord = string.lower(string.sub(word, 2))
            table.insert(words, firstLetter .. restOfWord)
        end
    end
    
    return table.concat(words, " ")
end

-- Enhanced buff detection with caching
function BuffGuard:GetPlayerBuffsWithCache()
    local currentTime = GetTime()
    if currentTime - self.cacheTime < CONSTANTS.CACHE_DURATION then
        return self.buffCache
    end
    
    self.buffCache = {}
    for i = 1, 40 do
        local name, _, _, _, _, duration, expirationTime, _, _, spellId = UnitBuff("player", i)
        if not name then break end
        
        self.buffCache[string.lower(name)] = {
            duration = duration,
            expirationTime = expirationTime,
            spellId = spellId,
            isPermanent = not expirationTime or expirationTime == 0
        }
    end
    
    self.cacheTime = currentTime
    return self.buffCache
end

-- Enhanced buff detection - force fresh check for expiration
function BuffGuard:FindPlayerBuff(searchName)
    if not searchName or searchName == "" then
        return false, 0, 0, false
    end
    
    searchName = string.lower(string.trim(searchName))
    
    -- ALWAYS get fresh buff data for accurate expiration detection
    local currentTime = GetTime()
    local freshBuffs = {}
    
    for i = 1, 40 do
        local name, _, _, _, _, duration, expirationTime, _, _, spellId = UnitBuff("player", i)
        if not name then break end
        
        freshBuffs[string.lower(name)] = {
            duration = duration,
            expirationTime = expirationTime,
            spellId = spellId,
            isPermanent = not expirationTime or expirationTime == 0
        }
    end
    
    -- Try exact match first
    local buff = freshBuffs[searchName]
    if buff then
        local timeLeft = 0
        if buff.isPermanent then
            timeLeft = 999999
        elseif buff.expirationTime and buff.expirationTime > 0 then
            timeLeft = math.max(0, buff.expirationTime - currentTime)
        end
        return true, timeLeft, buff.duration or 300, buff.isPermanent
    end
    
    -- Try partial matches
    for buffName, buffData in pairs(freshBuffs) do
        if string.find(buffName, searchName, 1, true) or 
           string.find(searchName, buffName, 1, true) then
            local timeLeft = 0
            if buffData.isPermanent then
                timeLeft = 999999
            elseif buffData.expirationTime and buffData.expirationTime > 0 then
                timeLeft = math.max(0, buffData.expirationTime - currentTime)
            end
            return true, timeLeft, buffData.duration or 300, buffData.isPermanent
        end
    end
    
    return false, 0, 0, false
end

-- Initialize saved variables with per-character support
function BuffGuard:InitializeDB()
    -- Initialize global settings (addon-wide preferences)
    if not BuffGuardDB then
        BuffGuardDB = {
            version = "2.2"
        }
    else
        -- PRESERVE existing data, just update version if needed
        if not BuffGuardDB.version then
            BuffGuardDB.version = "2.2"
        end
    end
    
    -- Initialize per-character settings
    if not BuffGuardCharDB then
        BuffGuardCharDB = {}
    end
    
    -- Get unique character identifier
    local playerName = UnitName("player")
    local realmName = GetRealmName()
    local characterKey = playerName .. "-" .. realmName
    
    -- Check if this character already has data - PRESERVE IT
    if not BuffGuardCharDB[characterKey] then
        BuffGuardCharDB[characterKey] = {}
    end
    
    -- ONLY set defaults for missing values - don't overwrite existing data
    for key, value in pairs(defaults) do
        if BuffGuardCharDB[characterKey][key] == nil then
            if type(value) == "table" then
                BuffGuardCharDB[characterKey][key] = {}
                if key == "buffs" then
                    -- Initialize empty buff array
                    for i = 1, CONSTANTS.MAX_BUTTONS do
                        BuffGuardCharDB[characterKey][key][i] = ""
                    end
                elseif key == "framePositions" then
                    -- Empty table for frame positions
                else
                    -- Copy other table defaults
                    for k, v in pairs(value) do
                        BuffGuardCharDB[characterKey][key][k] = v
                    end
                end
            else
                BuffGuardCharDB[characterKey][key] = value
            end
        end
    end
    
    -- CRITICAL FIX: Ensure buffs table exists and has correct size
    if not BuffGuardCharDB[characterKey].buffs then
        BuffGuardCharDB[characterKey].buffs = {}
        for i = 1, CONSTANTS.MAX_BUTTONS do
            BuffGuardCharDB[characterKey].buffs[i] = ""
        end
    else
        -- Only extend the array if needed, don't overwrite existing entries
        for i = 1, CONSTANTS.MAX_BUTTONS do
            if BuffGuardCharDB[characterKey].buffs[i] == nil then
                BuffGuardCharDB[characterKey].buffs[i] = ""
            end
        end
    end
    
    -- Store current character reference for easy access
    self.currentCharacterKey = characterKey
    -- CRITICAL: Create a direct reference to the saved data, not a copy
    self.settings = BuffGuardCharDB[characterKey]
    
    -- VERIFICATION: Ensure the reference is correct
    if self.settings ~= BuffGuardCharDB[characterKey] then
        return false
    end
    
    -- VERIFICATION: Ensure buffs table is accessible
    if not self.settings.buffs then
        return false
    end
    
    -- Initialize buff states for audio alerts
    self:InitializeBuffStates()
    
    -- Hide/show Blizzard buffs based on setting
    if self.settings.hideBlizzardBuffs then
        BuffFrame:Hide()
    else
        BuffFrame:Show()
    end
    
    -- Mark as initialized
    self.dbInitialized = true
    
    return true
end

-- Initialize buff states with proper structure
function BuffGuard:InitializeBuffStates()
    if not self.buffStates then
        self.buffStates = {}
    end
    
    for i = 1, CONSTANTS.MAX_BUTTONS do
        if not self.buffStates[i] then
            self.buffStates[i] = { 
                lastTimeLeft = 0, 
                hasAlerted60 = false, 
                hasAlertedExpired = false,
                lastAlertTime = 0,
                lastBuffState = false -- Track if buff was present last update
            }
        end
    end
end

-- Floating text without debug spam
function BuffGuard:CreateFloatingText(text, color)
    if UnitIsDead("player") or not UnitIsVisible("player") then
        return
    end
    
    -- Clean up expired texts first
    self:CleanupFloatingTexts()
    
    local floatingText = UIParent:CreateFontString(nil, "OVERLAY", "GameFontNormalHuge")
    if not floatingText then
        return
    end
    
    floatingText:SetPoint("CENTER", UIParent, "CENTER", 0, 100 + (#self.floatingTexts * 35))
    floatingText:SetText(text)
    floatingText:SetTextColor(color.r, color.g, color.b, 1)
    floatingText:SetFont("Fonts\\FRIZQT__.TTF", 20, "OUTLINE")
    floatingText:SetJustifyH("CENTER")
    
    local textData = {
        frame = floatingText,
        startTime = GetTime(),
        duration = CONSTANTS.FLOATING_TEXT_DURATION,
        startY = 100 + (#self.floatingTexts * 35),
        color = color,
        active = true,
        animationFrame = nil
    }
    
    table.insert(self.floatingTexts, textData)
    
    -- Use a simple timer approach instead of complex animation frames
    local updateTimer = 0
    local function updateText(frame, elapsed)
        if not textData.active or not textData.frame then
            frame:SetScript("OnUpdate", nil)
            return
        end
        
        updateTimer = updateTimer + elapsed
        if updateTimer < 0.1 then return end -- Update 10 times per second
        
        local currentTime = GetTime()
        local elapsedTime = currentTime - textData.startTime
        local progress = elapsedTime / textData.duration
        
        if progress >= 1 then
            BuffGuard:RemoveFloatingText(textData)
            return
        end
        
        -- Move up and fade out
        local yOffset = textData.startY + (progress * 100)
        local alpha = math.max(0, 1 - progress)
        
        if textData.frame then
            pcall(function()
                textData.frame:ClearAllPoints()
                textData.frame:SetPoint("CENTER", UIParent, "CENTER", 0, yOffset)
                textData.frame:SetTextColor(textData.color.r, textData.color.g, textData.color.b, alpha)
            end)
        end
        
        updateTimer = 0
    end
    
    -- Use the main update frame instead of creating new ones
    if not self.textUpdateFrame then
        self.textUpdateFrame = CreateFrame("Frame")
    end
    
    self.textUpdateFrame:SetScript("OnUpdate", updateText)
end

function BuffGuard:RemoveFloatingText(textData)
    if not textData then return end
    
    textData.active = false
    
    -- Safely cleanup animation frame
    if textData.animationFrame then
        pcall(function() textData.animationFrame:SetScript("OnUpdate", nil) end)
        pcall(function() textData.animationFrame:Hide() end)
        textData.animationFrame = nil
    end
    
    -- Safely cleanup text frame
    if textData.frame then
        pcall(function() textData.frame:SetScript("OnUpdate", nil) end)
        pcall(function() textData.frame:Hide() end)
        textData.frame = nil
    end
    
    -- Remove from list
    for i, data in ipairs(self.floatingTexts) do
        if data == textData then
            table.remove(self.floatingTexts, i)
            break
        end
    end
end

function BuffGuard:CleanupFloatingTexts()
    local currentTime = GetTime()
    for i = #self.floatingTexts, 1, -1 do
        local textData = self.floatingTexts[i]
        if not textData or not textData.active or not textData.frame or 
           (currentTime - textData.startTime) >= textData.duration then
            self:RemoveFloatingText(textData)
        end
    end
    
    -- Stop the update frame if no texts remain
    if #self.floatingTexts == 0 and self.textUpdateFrame then
        self.textUpdateFrame:SetScript("OnUpdate", nil)
    end
end

-- Frame position management with per-character settings
function BuffGuard:SaveFramePosition(frame, settingKey)
    if not self.settings or not self.settings.framePositions then
        return
    end
    
    local point, relativeTo, relativePoint, xOfs, yOfs = frame:GetPoint()
    self.settings.framePositions[settingKey] = {
        point = point,
        relativePoint = relativePoint,
        xOffset = xOfs,
        yOffset = yOfs
    }
end

function BuffGuard:RestoreFramePosition(frame, settingKey)
    if not self.settings or not self.settings.framePositions then
        return false
    end
    
    local positions = self.settings.framePositions
    if not positions or not positions[settingKey] then
        return false
    end
    
    local pos = positions[settingKey]
    frame:ClearAllPoints()
    frame:SetPoint(pos.point, UIParent, pos.relativePoint, pos.xOffset, pos.yOffset)
    return true
end

-- Create main frame with proper cleanup
function BuffGuard:CreateMainFrame()
    -- Destroy any existing main frame first
    if self.mainFrame then
        self.mainFrame:Hide()
        self.mainFrame:SetParent(nil)
        self.mainFrame = nil
    end
    
    self.mainFrame = CreateFrame("Frame", "BuffGuardMainFrame", UIParent)
    self.mainFrame:SetWidth(220)
    self.mainFrame:SetHeight(170)
    
    if not self:RestoreFramePosition(self.mainFrame, "mainFrame") then
        self.mainFrame:SetPoint("CENTER", UIParent, "CENTER", 0, 0)
    end
    
    self.mainFrame:SetBackdrop({
        bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
        edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border",
        tile = true,
        tileSize = 32,
        edgeSize = 32,
        insets = { left = 8, right = 8, top = 8, bottom = 8 }
    })
    self.mainFrame:SetBackdropColor(0, 0, 0, 0.8)
    self.mainFrame:EnableMouse(true)
    self.mainFrame:SetMovable(true)
    self.mainFrame:RegisterForDrag("LeftButton")
    
    self.mainFrame:SetScript("OnDragStart", function(frame) 
        frame:StartMoving() 
    end)
    
    self.mainFrame:SetScript("OnDragStop", function(frame) 
        frame:StopMovingOrSizing()
        BuffGuard:SaveFramePosition(frame, "mainFrame")
    end)
end

-- Create buff buttons using per-character settings with scale support
function BuffGuard:CreateBuffButton(index)
    if not self.settings then
        return nil
    end
    
    local scale = self.settings.scale or 1.0
    local scaledWidth = (self.settings.buttonWidth or 200) * scale
    local scaledHeight = (self.settings.buttonHeight or 30) * scale
    
    local button = CreateFrame("Button", "BuffGuardButton" .. index, self.mainFrame, "SecureActionButtonTemplate")
    button:SetWidth(scaledWidth)
    button:SetHeight(scaledHeight)
    button:SetPoint("TOPLEFT", self.mainFrame, "TOPLEFT", 10 * scale, -10 * scale - (index - 1) * (scaledHeight + 2 * scale))
    
    -- Button background with customizable colors
    button:SetBackdrop({
        bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
        edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
        tile = true,
        tileSize = 16,
        edgeSize = 16,
        insets = { left = 2, right = 2, top = 2, bottom = 2 }
    })
    
    local bgColor = self.settings.backgroundColor or { r = 0.1, g = 0.1, b = 0.1, a = 0.8 }
    local borderColor = self.settings.borderColor or { r = 0.5, g = 0.5, b = 0.5, a = 1.0 }
    button:SetBackdropColor(bgColor.r, bgColor.g, bgColor.b, bgColor.a)
    button:SetBackdropBorderColor(borderColor.r, borderColor.g, borderColor.b, borderColor.a)
    
    -- Icon (optional)
    local icon = button:CreateTexture(nil, "ARTWORK")
    local iconSize = (scaledHeight - 4 * scale)
    icon:SetWidth(iconSize)
    icon:SetHeight(iconSize)
    icon:SetPoint("LEFT", button, "LEFT", 2 * scale, 0)
    icon:SetTexture("Interface\\Icons\\INV_Misc_QuestionMark")
    button.icon = icon
    
    -- Hide icon if setting is disabled
    if not (self.settings.showIcon == nil and true or self.settings.showIcon) then
        icon:Hide()
    end
    
    -- Timer bar
    local timerBar = CreateFrame("StatusBar", nil, button)
    if self.settings.showIcon == nil or self.settings.showIcon then
        timerBar:SetPoint("LEFT", icon, "RIGHT", 2 * scale, 0)
    else
        timerBar:SetPoint("LEFT", button, "LEFT", 2 * scale, 0)
    end
    timerBar:SetPoint("RIGHT", button, "RIGHT", -2 * scale, 0)
    timerBar:SetHeight(scaledHeight - 8 * scale)
    timerBar:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar")
    timerBar:SetStatusBarColor(1, 0, 0, 0.8)
    timerBar:SetMinMaxValues(0, 100)
    timerBar:SetValue(0)
    button.timerBar = timerBar
    
    -- Timer text overlay (optional)
    local timerText = timerBar:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    timerText:SetPoint("CENTER", timerBar, "CENTER", 0, 0)
    
    -- Apply font settings with scale
    local fontSize = (self.settings.fontSize or 12) * scale
    local fontStyle = self.settings.fontStyle or "OUTLINE"
    timerText:SetFont("Fonts\\FRIZQT__.TTF", fontSize, fontStyle)
    timerText:SetText("No Buff")
    timerText:SetTextColor(1, 1, 1, 1)
    button.timerText = timerText
    
    -- Hide timer text if setting is disabled
    if not (self.settings.showTimer == nil and true or self.settings.showTimer) then
        timerText:Hide()
    end
    
    -- Configure secure button for spell casting
    button:SetAttribute("type", "spell")
    button:SetAttribute("unit", "player")
    button:RegisterForClicks("LeftButtonUp")
    
    button.index = index
    
    -- Enhanced tooltip (optional) - MOVED TO THE END AFTER BUTTON IS CREATED
    if self.settings.showTooltips then
        self:CreateEnhancedTooltip(button)
    end
    
    return button
end

function BuffGuard:CreateEnhancedTooltip(button)
    button:SetScript("OnEnter", function(self)
        -- ALWAYS check setting first - if disabled, show nothing
        if not BuffGuard.settings.showTooltips then
            return
        end
        
        -- Use BuffGuard.settings.buffs instead of BuffGuardDB.buffs
        if not BuffGuard.settings or not BuffGuard.settings.buffs then
            GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
            GameTooltip:SetText("Settings not loaded")
            GameTooltip:Show()
            return
        end
        
        local spellName = BuffGuard.settings.buffs[self.index]
        if not spellName or spellName == "" then
            GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
            GameTooltip:SetText("Empty Slot")
            GameTooltip:AddLine("Click to configure this slot", 0.7, 0.7, 0.7)
            GameTooltip:Show()
            return
        end
        
        GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
        
        -- Use capitalized spell name for display
        local capitalizedName = CapitalizeSpellName(spellName)
        
        local valid, spellID = ValidateSpellName(spellName)
        if valid then
            -- Use the game's spell tooltip as the base
            GameTooltip:SetSpellByID(spellID)
        else
            -- For invalid spells, show our custom tooltip
            GameTooltip:SetText(capitalizedName, 1, 1, 0) -- Yellow spell name
            GameTooltip:AddLine("Invalid spell name", 1, 0, 0)
        end
        
        -- Add current status information (for both valid and invalid spells)
        local found, timeLeft, duration, isPermanent = BuffGuard:FindPlayerBuff(spellName)
        if found then
            if isPermanent then
                GameTooltip:AddLine("BuffGuard Status: Active (Permanent)", 0, 1, 0)
            else
                local timeText
                if timeLeft > 60 then
                    timeText = string.format("%.1f minutes", timeLeft / 60)
                else
                    timeText = string.format("%.0f seconds", timeLeft)
                end
                GameTooltip:AddLine("BuffGuard Status: Active", 0, 1, 0)
                GameTooltip:AddLine("Time Remaining: " .. timeText, 1, 1, 0)
                
                -- Show max duration for context
                if duration and duration > 0 then
                    local maxText
                    if duration > 60 then
                        maxText = string.format("%.1f minutes", duration / 60)
                    else
                        maxText = string.format("%.0f seconds", duration)
                    end
                    GameTooltip:AddLine("Max Duration: " .. maxText, 0.7, 0.7, 0.7)
                end
            end
        else
            GameTooltip:AddLine("BuffGuard Status: Not Active", 1, 0, 0)
        end
        
        -- Add usage instructions
        GameTooltip:AddLine(" ") -- Blank line
        GameTooltip:AddLine("Left Click: Cast " .. capitalizedName, 0.7, 0.7, 0.7)
        
        GameTooltip:Show()
    end)
    
    button:SetScript("OnLeave", function(self)
        GameTooltip:Hide()
    end)
end

-- Alert system using per-character settings
function BuffGuard:CheckAudioAlerts(index, spellName, buffFound, timeLeft, isPermanent)
    if not self.settings or not self.settings.audioAlerts or UnitIsDead("player") or not self.buffStates[index] then
        return
    end
    
    local state = self.buffStates[index]
    local currentTime = GetTime()
    
    -- Skip permanent buffs
    if isPermanent then
        state.lastTimeLeft = 999999
        state.lastBuffState = true
        return
    end
    
    -- Simple logic: buff is active if found and timeLeft > 0
    local buffActive = buffFound and timeLeft > 0
    
    if buffActive then
        -- Buff is active
        
        -- Reset expiration alert when buff becomes active again
        if not state.lastBuffState then
            state.hasAlertedExpired = false
        end
        
        -- Detect refresh (timeLeft increased significantly)
        if timeLeft > state.lastTimeLeft + 30 then
            state.hasAlerted60 = false
            state.hasAlertedExpired = false
        end
        
        -- 60 second warning
        if timeLeft <= CONSTANTS.WARNING_THRESHOLD and 
           state.lastTimeLeft > CONSTANTS.WARNING_THRESHOLD and 
           not state.hasAlerted60 then
            
            local capitalizedName = CapitalizeSpellName(spellName)
            PlaySound(self.settings.warningSound or "igQuestLogAbandonQuest")
            self:CreateFloatingText("60 Second Warning: Rebuff " .. capitalizedName .. " Now!", {r=1, g=0.5, b=0})
            
            state.hasAlerted60 = true
            state.lastAlertTime = currentTime
        end
        
        state.lastTimeLeft = timeLeft
        state.lastBuffState = true
        
    else
        -- Buff is not active (expired or missing)
        
        -- Expiration alert: only if buff was active before and we haven't already alerted
        if state.lastBuffState and not state.hasAlertedExpired then
            
            local capitalizedName = CapitalizeSpellName(spellName)
            PlaySound(self.settings.expiredSound or "igQuestFailed")
            self:CreateFloatingText(capitalizedName .. " EXPIRED! Rebuff Now!", {r=1, g=0, b=0})
            
            state.hasAlertedExpired = true
            state.lastAlertTime = currentTime
        end
        
        state.lastTimeLeft = 0
        state.lastBuffState = false
    end
end

-- Update button using per-character settings
function BuffGuard:UpdateButton(button)
    if not button or not self.settings or not self.settings.buffs then 
        return 
    end
    
    local index = button.index
    local spellName = self.settings.buffs[index] -- Use self.settings.buffs instead of BuffGuardDB.buffs
    
    if not spellName or spellName == "" then
        button.icon:SetTexture("Interface\\Icons\\INV_Misc_QuestionMark")
        button.timerBar:SetStatusBarColor(0.3, 0.3, 0.3, 0.8)
        button.timerBar:SetValue(0)
        button.timerText:SetText("Empty")
        button:SetAttribute("spell", nil)
        return
    end
    
    -- Set the spell for secure casting
    button:SetAttribute("spell", spellName)
    
    -- Get spell info
    local spellTexture = GetSpellTexture(spellName)
    if spellTexture then
        button.icon:SetTexture(spellTexture)
    else
        button.icon:SetTexture("Interface\\Icons\\INV_Misc_QuestionMark")
    end
    
    -- Check for buff using improved detection
    local buffFound, timeLeft, maxDuration, isPermanent = self:FindPlayerBuff(spellName)
    
    -- Check audio alerts BEFORE updating display
    self:CheckAudioAlerts(index, spellName, buffFound, timeLeft, isPermanent)
    
    -- Consider buff active only if found AND timeLeft > 0
    local buffActive = buffFound and timeLeft > 0
    
    if buffActive then
        if isPermanent then
            button.timerBar:SetStatusBarColor(0, 0.8, 1, 0.8)
            button.timerBar:SetValue(100)
            button.timerText:SetText("Active")
            button.timerText:SetTextColor(1, 1, 1, 1)
        else
            local percentage = math.min(100, (timeLeft / maxDuration) * 100)
            button.timerBar:SetValue(percentage)
            
            -- Proper time display logic - minutes rounded UP until 60s
            if timeLeft > CONSTANTS.WARNING_THRESHOLD then
                -- More than 60 seconds - show minutes rounded UP
                button.timerBar:SetStatusBarColor(0, 1, 0, 0.8)
                local minutes = math.ceil(timeLeft / 60) -- Round UP to next minute
                button.timerText:SetText(string.format("%dm", minutes))
            else
                -- 60 seconds or less - show countdown in seconds with orange/amber
                button.timerBar:SetStatusBarColor(1, 0.65, 0, 0.8)
                button.timerText:SetText(string.format("%ds", math.floor(timeLeft)))
            end
            button.timerText:SetTextColor(1, 1, 1, 1)
        end
    else
        -- Buff not active (missing, expired, or timeLeft <= 0) - show red "No Buff"
        button.timerBar:SetStatusBarColor(1, 0, 0, 0.9)
        button.timerBar:SetValue(100)
        button.timerText:SetText("No Buff")
        button.timerText:SetTextColor(1, 1, 1, 1)
    end
end

-- Create buttons with proper validation and scale support
function BuffGuard:CreateButtons()
    PerformanceMonitor:StartTimer("CreateButtons")
    
    -- Clear existing buttons properly
    for i = 1, #self.buttons do
        if self.buttons[i] then
            self.buttons[i]:Hide()
            self.buttons[i]:SetParent(nil)
            self.buttons[i] = nil
        end
    end
    wipe(self.buttons) -- Clear the table completely
    
    if not self.settings then 
        return 
    end
    
    -- Create new buttons
    local numButtons = self.settings.numButtons or 5
    
    for i = 1, numButtons do
        local button = self:CreateBuffButton(i)
        if button then
            self.buttons[i] = button
        end
    end
    
    -- Resize main frame to fit buttons properly with scale
    local scale = self.settings.scale or 1.0
    local scaledButtonHeight = (self.settings.buttonHeight or 30) * scale
    local totalHeight = numButtons * (scaledButtonHeight + 2 * scale) + 16 * scale
    local totalWidth = (self.settings.buttonWidth or 200) * scale + 20 * scale
    
    self.mainFrame:SetWidth(totalWidth)
    self.mainFrame:SetHeight(totalHeight)
    
    -- Apply frame lock if enabled
    if self.settings.lockFrames then
        self.mainFrame:SetMovable(false)
        self.mainFrame:EnableMouse(false)
    else
        self.mainFrame:SetMovable(true)
        self.mainFrame:EnableMouse(true)
    end
    
    PerformanceMonitor:EndTimer("CreateButtons")
end

-- Update all buttons with validation
function BuffGuard:UpdateAllButtons()
    if not self.settings or not self.settings.buffs then
        return
    end
    
    PerformanceMonitor:StartTimer("UpdateAllButtons")
    
    for i = 1, #self.buttons do
        if self.buttons[i] then
            self:UpdateButton(self.buttons[i])
        end
    end
    
    PerformanceMonitor:EndTimer("UpdateAllButtons")
end

-- Create minimap button
function BuffGuard:CreateMinimapButton()
    if self.minimapButton then
        self.minimapButton:Hide()
        self.minimapButton:SetParent(nil)
        self.minimapButton = nil
    end
    
    self.minimapButton = CreateFrame("Button", "BuffGuardMinimapButton", Minimap)
    self.minimapButton:SetFrameLevel(8)
    self.minimapButton:SetWidth(31)
    self.minimapButton:SetHeight(31)
    self.minimapButton:SetFrameStrata("MEDIUM")
    
    local function UpdateMinimapButtonPosition()
        local angle = self.settings.minimapButtonAngle or 45
        local cos = math.cos(math.rad(angle))
        local sin = math.sin(math.rad(angle))
        local x = cos * 80
        local y = sin * 80
        self.minimapButton:ClearAllPoints()
        self.minimapButton:SetPoint("CENTER", Minimap, "CENTER", x, y)
    end
    
    UpdateMinimapButtonPosition()
    
    local background = self.minimapButton:CreateTexture(nil, "BACKGROUND")
    background:SetWidth(20)
    background:SetHeight(20)
    background:SetPoint("CENTER", 0, 1)
    background:SetTexture(0, 0, 0, 1)
    
    local texture = self.minimapButton:CreateTexture(nil, "ARTWORK")
    texture:SetWidth(20)
    texture:SetHeight(20)
    texture:SetPoint("CENTER", 0, 1)
    texture:SetTexture("Interface\\Icons\\Spell_Holy_SealOfWisdom")
    self.minimapButton.texture = texture
    
    local overlay = self.minimapButton:CreateTexture(nil, "OVERLAY")
    overlay:SetWidth(53)
    overlay:SetHeight(53)
    overlay:SetPoint("TOPLEFT", 0, 0)
    overlay:SetTexture("Interface\\Minimap\\MiniMap-TrackingBorder")
    
    local isDragging = false
    
    self.minimapButton:EnableMouse(true)
    self.minimapButton:RegisterForDrag("LeftButton")
    self.minimapButton:RegisterForClicks("AnyUp")
    
    self.minimapButton:SetScript("OnDragStart", function(frame)
        isDragging = true
    end)
    
    self.minimapButton:SetScript("OnDragStop", function(frame)
        isDragging = false
    end)
    
    self.minimapButton:SetScript("OnUpdate", function(frame)
        if isDragging then
            local mx, my = Minimap:GetCenter()
            local px, py = GetCursorPosition()
            local scale = Minimap:GetEffectiveScale()
            px, py = px / scale, py / scale
            
            local angle = math.deg(math.atan2(py - my, px - mx))
            self.settings.minimapButtonAngle = angle
            UpdateMinimapButtonPosition()
        end
    end)
    
    self.minimapButton:SetScript("OnClick", function(frame, button)
        if button == "LeftButton" and not isDragging then
            BuffGuard:ToggleConfigFrame()
        elseif button == "RightButton" then
            BuffGuard:ToggleMainFrame()
        end
    end)
    
    self.minimapButton:SetScript("OnEnter", function(frame)
        GameTooltip:SetOwner(frame, "ANCHOR_LEFT")
        GameTooltip:SetText("BuffGuard")
        GameTooltip:AddLine("Left click: Open configuration", 1, 1, 1)
        GameTooltip:AddLine("Right click: Toggle main panel", 1, 1, 1)
        GameTooltip:AddLine("Drag: Move around minimap", 1, 1, 1)
        GameTooltip:Show()
    end)
    
    self.minimapButton:SetScript("OnLeave", function(frame)
        GameTooltip:Hide()
    end)
    
    self.minimapButton:Show()
end

-- Create config frame with tabbed interface
function BuffGuard:CreateConfigFrame()
    -- Only create if it doesn't exist
    if self.configFrame and self.configFrame:IsShown() then
        return
    end
    
    -- Clean up old frame completely
    if self.configFrame then
        self.configFrame:Hide()
        self.configFrame:SetParent(nil)
        self.configFrame = nil
    end
    
    -- Clear edit boxes
    wipe(self.buffEditBoxes)
    
    -- Create new config frame with compact size
    self.configFrame = CreateFrame("Frame", "BuffGuardConfigFrame" .. GetTime(), UIParent)
    self.configFrame:SetFrameLevel(100)
    self.configFrame:SetFrameStrata("DIALOG")
    
    -- Much more compact size
    self.configFrame:SetWidth(450)
    self.configFrame:SetHeight(500)
    
    if not self:RestoreFramePosition(self.configFrame, "configFrame") then
        self.configFrame:SetPoint("CENTER", UIParent, "CENTER", 0, 0)
    end
    
    self.configFrame:SetBackdrop({
        bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
        edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border",
        tile = true,
        tileSize = 32,
        edgeSize = 32,
        insets = { left = 8, right = 8, top = 8, bottom = 8 }
    })
    self.configFrame:SetBackdropColor(0, 0, 0, 0.9)
    
    self.configFrame:EnableMouse(true)
    self.configFrame:SetMovable(true)
    self.configFrame:RegisterForDrag("LeftButton")
    
    self.configFrame:SetScript("OnDragStart", function(frame) 
        frame:StartMoving() 
    end)
    
    self.configFrame:SetScript("OnDragStop", function(frame) 
        frame:StopMovingOrSizing()
        BuffGuard:SaveFramePosition(frame, "configFrame")
    end)
    
    -- Title
    local configTitle = self.configFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    configTitle:SetPoint("TOP", self.configFrame, "TOP", 0, -20)
    configTitle:SetText("BuffGuard Configuration")
    configTitle:SetTextColor(1, 1, 0, 1)

    -- Create tab system
    self:CreateConfigTabs()

    -- Close button
 local closeButton = CreateFrame("Button", nil, self.configFrame, "UIPanelButtonTemplate")
closeButton:SetPoint("BOTTOM", self.configFrame, "BOTTOM", 0, 20)
closeButton:SetSize(80, 22)
closeButton:SetText("Close")
closeButton:SetScript("OnClick", function()
    BuffGuard:HideConfigFrame()
end)
    
local versionText = self.configFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
versionText:SetPoint("BOTTOMRIGHT", self.configFrame, "BOTTOMRIGHT", -15, 10)
versionText:SetText("BuffGuard v2.2 by Justinction")
versionText:SetTextColor(0.7, 0.7, 0.7, 0.8) -- Subtle gray color
versionText:SetJustifyH("RIGHT")

self.configFrame:Show()
end

-- Create tabbed interface with improved spacing
function BuffGuard:CreateConfigTabs()
    -- Tab container with proper spacing from tab buttons
    local tabContainer = CreateFrame("Frame", nil, self.configFrame)
    tabContainer:SetPoint("TOPLEFT", self.configFrame, "TOPLEFT", 15, -85) -- Moved down from -50 to -85
    tabContainer:SetPoint("BOTTOMRIGHT", self.configFrame, "BOTTOMRIGHT", -15, 60)
    
    -- Tab buttons
    local tabs = {
        {name = "Buffs", key = "buffs"},
        {name = "Appearance", key = "appearance"}, 
        {name = "Alerts", key = "alerts"}
    }
    
    self.configTabs = {}
    self.configTabButtons = {}
    
    -- Create tab buttons
    for i, tabInfo in ipairs(tabs) do
        local tabButton = CreateFrame("Button", nil, self.configFrame, "UIPanelButtonTemplate")
        tabButton:SetSize(100, 25)
        tabButton:SetPoint("TOPLEFT", self.configFrame, "TOPLEFT", 15 + (i-1) * 105, -50)
        tabButton:SetText(tabInfo.name)
        tabButton.tabKey = tabInfo.key
        
        tabButton:SetScript("OnClick", function(button)
            BuffGuard:ShowConfigTab(button.tabKey)
        end)
        
        self.configTabButtons[tabInfo.key] = tabButton
        
        -- Create tab content frame
        local tabFrame = CreateFrame("Frame", nil, tabContainer)
        tabFrame:SetAllPoints(tabContainer)
        tabFrame:Hide()
        
        self.configTabs[tabInfo.key] = tabFrame
    end
    
    -- Create content for each tab
    self:CreateBuffsTab()
    self:CreateAppearanceTab()
    self:CreateAlertsTab()
    
    -- Show first tab by default
    self:ShowConfigTab("buffs")
end

-- Show specific tab
function BuffGuard:ShowConfigTab(tabKey)
    -- Hide all tabs and reset button states
    for key, frame in pairs(self.configTabs) do
        frame:Hide()
        if self.configTabButtons[key] then
            self.configTabButtons[key]:SetText(self.configTabButtons[key]:GetText())
            -- Reset button appearance - make it look unpressed
            self.configTabButtons[key]:GetNormalTexture():SetVertexColor(1, 1, 1)
        end
    end
    
    -- Show selected tab and highlight button
    if self.configTabs[tabKey] then
        self.configTabs[tabKey]:Show()
        if self.configTabButtons[tabKey] then
            -- Highlight selected button
            self.configTabButtons[tabKey]:GetNormalTexture():SetVertexColor(1, 1, 0.7)
        end
    end
end

-- Create Buffs tab content with improved text visibility
function BuffGuard:CreateBuffsTab()
    local frame = self.configTabs["buffs"]
    local yPos = -10 -- Start closer to top since we have more space
    
    -- Number of buttons control
    local numButtonsLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    numButtonsLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", 10, yPos)
    numButtonsLabel:SetText("Number of Buttons: " .. self.settings.numButtons)
    numButtonsLabel:SetTextColor(1, 1, 1, 1)
    
    local numButtonsMinusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    numButtonsMinusBtn:SetPoint("TOPLEFT", numButtonsLabel, "BOTTOMLEFT", 0, -8)
    numButtonsMinusBtn:SetSize(30, 20)
    numButtonsMinusBtn:SetText("-")
    numButtonsMinusBtn:SetScript("OnClick", function()
        if self.settings.numButtons > 1 then
            self.settings.numButtons = self.settings.numButtons - 1
            numButtonsLabel:SetText("Number of Buttons: " .. self.settings.numButtons)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
            BuffGuard:UpdateBuffsTabEditBoxes()
        end
    end)
    
    local numButtonsPlusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    numButtonsPlusBtn:SetPoint("LEFT", numButtonsMinusBtn, "RIGHT", 5, 0)
    numButtonsPlusBtn:SetSize(30, 20)
    numButtonsPlusBtn:SetText("+")
    numButtonsPlusBtn:SetScript("OnClick", function()
        if self.settings.numButtons < CONSTANTS.MAX_BUTTONS then
            self.settings.numButtons = self.settings.numButtons + 1
            numButtonsLabel:SetText("Number of Buttons: " .. self.settings.numButtons)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
            BuffGuard:UpdateBuffsTabEditBoxes()
        end
    end)
    
    yPos = yPos - 60
    
    -- Tracked Buffs label
    local buffsLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    buffsLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", 10, yPos)
    buffsLabel:SetText("Tracked Buffs:")
    buffsLabel:SetTextColor(1, 1, 0, 1)
    
    yPos = yPos - 30
    
    -- Create scroll frame for buff edit boxes
    local scrollFrame = CreateFrame("ScrollFrame", "BuffGuardScrollFrame" .. GetTime(), frame, "UIPanelScrollFrameTemplate")
    scrollFrame:SetPoint("TOPLEFT", frame, "TOPLEFT", 10, yPos)
    scrollFrame:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -30, 10)
    
    local scrollChild = CreateFrame("Frame", nil, scrollFrame)
    scrollFrame:SetScrollChild(scrollChild)
    scrollChild:SetWidth(scrollFrame:GetWidth() - 20)
    
    self.buffsScrollFrame = scrollFrame
    self.buffsScrollChild = scrollChild
    
    -- Create buff edit boxes
    self:CreateBuffsTabEditBoxes()
end

-- Create buff edit boxes with improved text visibility and styling
function BuffGuard:CreateBuffsTabEditBoxes()
    -- Clear existing edit boxes
    if self.buffEditBoxes then
        for i, boxData in pairs(self.buffEditBoxes) do
            if boxData.label then boxData.label:Hide() end
            if boxData.editbox then boxData.editbox:Hide() end
        end
    end
    wipe(self.buffEditBoxes)
    
    local scrollChild = self.buffsScrollChild
    local yPos = -15
    
    for i = 1, self.settings.numButtons do
        local buffLabel = scrollChild:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
        buffLabel:SetPoint("TOPLEFT", scrollChild, "TOPLEFT", 5, yPos)
        buffLabel:SetText("Buff " .. i .. ":")
        buffLabel:SetTextColor(1, 1, 0, 1) -- Yellow labels
        
        -- Create SIMPLE edit box with NO template
        local editbox = CreateFrame("EditBox", nil, scrollChild)
        editbox:SetPoint("TOPLEFT", buffLabel, "BOTTOMLEFT", 0, -5)
        editbox:SetSize(350, 24)
        editbox:SetAutoFocus(false)
        editbox:SetText(self.settings.buffs[i] or "")
        
        -- SIMPLE STYLING
        editbox:SetFont("Fonts\\FRIZQT__.TTF", 12, "")
        editbox:SetTextColor(1, 1, 1, 1) -- White text
        editbox:SetTextInsets(6, 6, 0, 0)
        editbox:EnableMouse(true)
        editbox:EnableKeyboard(true)
        editbox:SetMaxLetters(50)
        
        -- LIGHT GREY BORDER FIRST (behind the background)
        local border = editbox:CreateTexture(nil, "BACKGROUND")
        border:SetPoint("TOPLEFT", editbox, "TOPLEFT", -2, 2)
        border:SetPoint("BOTTOMRIGHT", editbox, "BOTTOMRIGHT", 2, -2)
        border:SetTexture(0.7, 0.7, 0.7, 1) -- Light grey border
        
        -- DARK GREY BACKGROUND ON TOP
        local bg = editbox:CreateTexture(nil, "BORDER")
        bg:SetAllPoints(editbox)
        bg:SetTexture(0.15, 0.15, 0.15, 0.9) -- Dark grey background
        
        editbox.buffIndex = i
        editbox.background = bg
        
        -- FOCUS HIGHLIGHTING
        editbox:SetScript("OnEditFocusGained", function(self)
            self:HighlightText()
            self.background:SetTexture(0.25, 0.25, 0.25, 0.9) -- Slightly lighter dark grey when focused
        end)
        
        editbox:SetScript("OnEditFocusLost", function(self)
            local text = self:GetText()
            text = CapitalizeSpellName(text)
            self:SetText(text)
            BuffGuard.settings.buffs[self.buffIndex] = text
            BuffGuard:UpdateAllButtons()
            self.background:SetTexture(0.15, 0.15, 0.15, 0.9) -- Reset to dark grey
        end)
        
        editbox:SetScript("OnEnterPressed", function(self)
            local text = self:GetText()
            text = CapitalizeSpellName(text)
            self:SetText(text)
            BuffGuard.settings.buffs[self.buffIndex] = text
            self:ClearFocus()
            BuffGuard:UpdateAllButtons()
        end)
        
        editbox:SetScript("OnEscapePressed", function(self)
            self:ClearFocus()
        end)
        
        -- Tab navigation
        editbox:SetScript("OnTabPressed", function(self)
            local nextIndex = self.buffIndex + 1
            if nextIndex > BuffGuard.settings.numButtons then
                nextIndex = 1
            end
            
            if BuffGuard.buffEditBoxes[nextIndex] and BuffGuard.buffEditBoxes[nextIndex].editbox then
                BuffGuard.buffEditBoxes[nextIndex].editbox:SetFocus()
            end
        end)
        
        self.buffEditBoxes[i] = {
            label = buffLabel,
            editbox = editbox
        }
        
        yPos = yPos - 55
    end
    
    -- Set scroll child height
    local totalHeight = math.abs(yPos) + 30
    scrollChild:SetHeight(math.max(totalHeight, self.buffsScrollFrame:GetHeight()))
end

-- Update buff edit boxes when number changes
function BuffGuard:UpdateBuffsTabEditBoxes()
    if self.configTabs and self.configTabs["buffs"]:IsShown() then
        self:CreateBuffsTabEditBoxes()
    end
end

-- Create Appearance tab content with improved spacing
function BuffGuard:CreateAppearanceTab()
    local frame = self.configTabs["appearance"]
    local yPos = -10 -- Better starting position
    local leftCol = 15  -- Slightly more padding
    local rightCol = 225
    
    -- Scale control
    local scaleLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    scaleLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", leftCol, yPos)
    scaleLabel:SetText(string.format("Scale: %.1f", self.settings.scale or 1.0))
    scaleLabel:SetTextColor(1, 1, 1, 1)
    
    local scaleMinusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    scaleMinusBtn:SetPoint("TOPLEFT", scaleLabel, "BOTTOMLEFT", 0, -8)
    scaleMinusBtn:SetSize(30, 22) -- Slightly larger buttons
    scaleMinusBtn:SetText("-")
    scaleMinusBtn:SetScript("OnClick", function()
        local currentScale = self.settings.scale or 1.0
        if currentScale > 0.5 then
            self.settings.scale = math.max(0.5, currentScale - 0.1)
            scaleLabel:SetText(string.format("Scale: %.1f", self.settings.scale))
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    local scalePlusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    scalePlusBtn:SetPoint("LEFT", scaleMinusBtn, "RIGHT", 5, 0)
    scalePlusBtn:SetSize(30, 22)
    scalePlusBtn:SetText("+")
    scalePlusBtn:SetScript("OnClick", function()
        local currentScale = self.settings.scale or 1.0
        if currentScale < 2.0 then
            self.settings.scale = math.min(2.0, currentScale + 0.1)
            scaleLabel:SetText(string.format("Scale: %.1f", self.settings.scale))
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    -- Button width control (right column)
    local widthLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    widthLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", rightCol, yPos)
    widthLabel:SetText("Width: " .. self.settings.buttonWidth)
    widthLabel:SetTextColor(1, 1, 1, 1)
    
    local widthMinusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    widthMinusBtn:SetPoint("TOPLEFT", widthLabel, "BOTTOMLEFT", 0, -8)
    widthMinusBtn:SetSize(30, 22)
    widthMinusBtn:SetText("-")
    widthMinusBtn:SetScript("OnClick", function()
        if self.settings.buttonWidth > 100 then
            self.settings.buttonWidth = self.settings.buttonWidth - 10
            widthLabel:SetText("Width: " .. self.settings.buttonWidth)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    local widthPlusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    widthPlusBtn:SetPoint("LEFT", widthMinusBtn, "RIGHT", 5, 0)
    widthPlusBtn:SetSize(30, 22)
    widthPlusBtn:SetText("+")
    widthPlusBtn:SetScript("OnClick", function()
        if self.settings.buttonWidth < 400 then
            self.settings.buttonWidth = self.settings.buttonWidth + 10
            widthLabel:SetText("Width: " .. self.settings.buttonWidth)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    yPos = yPos - 75 -- Better spacing between sections
    
    -- Font size control (left column)
    local fontSizeLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    fontSizeLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", leftCol, yPos)
    fontSizeLabel:SetText("Font Size: " .. (self.settings.fontSize or 12))
    fontSizeLabel:SetTextColor(1, 1, 1, 1)
    
    local fontSizeMinusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    fontSizeMinusBtn:SetPoint("TOPLEFT", fontSizeLabel, "BOTTOMLEFT", 0, -8)
    fontSizeMinusBtn:SetSize(30, 22)
    fontSizeMinusBtn:SetText("-")
    fontSizeMinusBtn:SetScript("OnClick", function()
        local currentSize = self.settings.fontSize or 12
        if currentSize > 8 then
            self.settings.fontSize = currentSize - 1
            fontSizeLabel:SetText("Font Size: " .. self.settings.fontSize)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    local fontSizePlusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    fontSizePlusBtn:SetPoint("LEFT", fontSizeMinusBtn, "RIGHT", 5, 0)
    fontSizePlusBtn:SetSize(30, 22)
    fontSizePlusBtn:SetText("+")
    fontSizePlusBtn:SetScript("OnClick", function()
        local currentSize = self.settings.fontSize or 12
        if currentSize < 24 then
            self.settings.fontSize = currentSize + 1
            fontSizeLabel:SetText("Font Size: " .. self.settings.fontSize)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    -- Button height control (right column)
    local heightLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    heightLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", rightCol, yPos)
    heightLabel:SetText("Height: " .. self.settings.buttonHeight)
    heightLabel:SetTextColor(1, 1, 1, 1)
    
    local heightMinusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    heightMinusBtn:SetPoint("TOPLEFT", heightLabel, "BOTTOMLEFT", 0, -8)
    heightMinusBtn:SetSize(30, 22)
    heightMinusBtn:SetText("-")
    heightMinusBtn:SetScript("OnClick", function()
        if self.settings.buttonHeight > 20 then
            self.settings.buttonHeight = self.settings.buttonHeight - 2
            heightLabel:SetText("Height: " .. self.settings.buttonHeight)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    local heightPlusBtn = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    heightPlusBtn:SetPoint("LEFT", heightMinusBtn, "RIGHT", 5, 0)
    heightPlusBtn:SetSize(30, 22)
    heightPlusBtn:SetText("+")
    heightPlusBtn:SetScript("OnClick", function()
        if self.settings.buttonHeight < 60 then
            self.settings.buttonHeight = self.settings.buttonHeight + 2
            heightLabel:SetText("Height: " .. self.settings.buttonHeight)
            BuffGuard:CreateButtons()
            BuffGuard:UpdateAllButtons()
        end
    end)
    
    yPos = yPos - 75
    
    -- Font style control (spans both columns)
    local fontStyleLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    fontStyleLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", leftCol, yPos)
    fontStyleLabel:SetText("Font Style:")
    fontStyleLabel:SetTextColor(1, 1, 1, 1)
    
    local fontStyleButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    fontStyleButton:SetPoint("TOPLEFT", fontStyleLabel, "BOTTOMLEFT", 0, -8)
    fontStyleButton:SetSize(100, 22)
    
    local fontStyles = {
        {name = "Outline", style = "OUTLINE"},
        {name = "Thick", style = "THICK"},
        {name = "Monochrome", style = "MONOCHROME"},
        {name = "None", style = ""}
    }
    
    local function getFontStyleName()
        local currentStyle = self.settings.fontStyle or "OUTLINE"
        for _, styleData in ipairs(fontStyles) do
            if styleData.style == currentStyle then
                return styleData.name
            end
        end
        return "Outline"
    end
    
    fontStyleButton:SetText(getFontStyleName())
    fontStyleButton:SetScript("OnClick", function()
        local currentIndex = 1
        local currentStyle = self.settings.fontStyle or "OUTLINE"
        for i, styleData in ipairs(fontStyles) do
            if styleData.style == currentStyle then
                currentIndex = i
                break
            end
        end
        
        currentIndex = currentIndex + 1
        if currentIndex > #fontStyles then
            currentIndex = 1
        end
        
        self.settings.fontStyle = fontStyles[currentIndex].style
        fontStyleButton:SetText(fontStyles[currentIndex].name)
        BuffGuard:CreateButtons()
        BuffGuard:UpdateAllButtons()
    end)
    
    yPos = yPos - 65
    
    -- Display options checkboxes with better spacing
    local showTimerCheckbox = CreateFrame("CheckButton", nil, frame, "UICheckButtonTemplate")
    showTimerCheckbox:SetPoint("TOPLEFT", frame, "TOPLEFT", leftCol, yPos)
    showTimerCheckbox:SetSize(20, 20)
    showTimerCheckbox:SetChecked(self.settings.showTimer == nil and true or self.settings.showTimer)
    
    local showTimerLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    showTimerLabel:SetPoint("LEFT", showTimerCheckbox, "RIGHT", 8, 0) -- Better spacing
    showTimerLabel:SetText("Show Timer Text")
    showTimerLabel:SetTextColor(1, 1, 1, 1)
    
  showTimerCheckbox:SetScript("OnClick", function(checkbox)
    self.settings.showTimer = checkbox:GetChecked()
    -- Update all existing buttons to show/hide timer text
    for i = 1, #self.buttons do
        if self.buttons[i] and self.buttons[i].timerText then
            if self.settings.showTimer then
                self.buttons[i].timerText:Show()
            else
                self.buttons[i].timerText:Hide()
            end
        end
    end
end)
    
    yPos = yPos - 35 -- Better spacing between checkboxes
    
    local showIconCheckbox = CreateFrame("CheckButton", nil, frame, "UICheckButtonTemplate")
    showIconCheckbox:SetPoint("TOPLEFT", frame, "TOPLEFT", leftCol, yPos)
    showIconCheckbox:SetSize(20, 20)
    showIconCheckbox:SetChecked(self.settings.showIcon == nil and true or self.settings.showIcon)
    
    local showIconLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    showIconLabel:SetPoint("LEFT", showIconCheckbox, "RIGHT", 8, 0)
    showIconLabel:SetText("Show Spell Icons")
    showIconLabel:SetTextColor(1, 1, 1, 1)
    
 showIconCheckbox:SetScript("OnClick", function(checkbox)
    self.settings.showIcon = checkbox:GetChecked()
    -- Update all existing buttons to show/hide icons AND reposition timer bars
    for i = 1, #self.buttons do
        if self.buttons[i] and self.buttons[i].icon and self.buttons[i].timerBar then
            if self.settings.showIcon then
                self.buttons[i].icon:Show()
                -- Reposition timer bar to be after the icon
                self.buttons[i].timerBar:ClearAllPoints()
                local scale = self.settings.scale or 1.0
                self.buttons[i].timerBar:SetPoint("LEFT", self.buttons[i].icon, "RIGHT", 2 * scale, 0)
                self.buttons[i].timerBar:SetPoint("RIGHT", self.buttons[i], "RIGHT", -2 * scale, 0)
            else
                self.buttons[i].icon:Hide()
                -- Reposition timer bar to fill the whole button
                self.buttons[i].timerBar:ClearAllPoints()
                local scale = self.settings.scale or 1.0
                self.buttons[i].timerBar:SetPoint("LEFT", self.buttons[i], "LEFT", 2 * scale, 0)
                self.buttons[i].timerBar:SetPoint("RIGHT", self.buttons[i], "RIGHT", -2 * scale, 0)
            end
        end
    end
end)
    
    yPos = yPos - 35
    
    local lockFramesCheckbox = CreateFrame("CheckButton", nil, frame, "UICheckButtonTemplate")
    lockFramesCheckbox:SetPoint("TOPLEFT", frame, "TOPLEFT", leftCol, yPos)
    lockFramesCheckbox:SetSize(20, 20)
    lockFramesCheckbox:SetChecked(self.settings.lockFrames or false)
    
    local lockFramesLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    lockFramesLabel:SetPoint("LEFT", lockFramesCheckbox, "RIGHT", 8, 0)
    lockFramesLabel:SetText("Lock Frames (prevent movement)")
    lockFramesLabel:SetTextColor(1, 1, 1, 1)
    
    lockFramesCheckbox:SetScript("OnClick", function(checkbox)
        self.settings.lockFrames = checkbox:GetChecked()
        BuffGuard:CreateButtons()
        BuffGuard:UpdateAllButtons()
    end)
end

-- Create Alerts tab content with improved spacing and styling
function BuffGuard:CreateAlertsTab()
    local frame = self.configTabs["alerts"]
    local yPos = -10 -- Better starting position
    local leftPadding = 15 -- Consistent with other tabs
    
    -- Hide Blizzard Buffs checkbox
    local hideBuffsCheckbox = CreateFrame("CheckButton", nil, frame, "UICheckButtonTemplate")
    hideBuffsCheckbox:SetPoint("TOPLEFT", frame, "TOPLEFT", leftPadding, yPos)
    hideBuffsCheckbox:SetSize(20, 20)
    hideBuffsCheckbox:SetChecked(self.settings.hideBlizzardBuffs)
    
    local hideBuffsLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    hideBuffsLabel:SetPoint("LEFT", hideBuffsCheckbox, "RIGHT", 8, 0)
    hideBuffsLabel:SetText("Hide Blizzard Buff Frame")
    hideBuffsLabel:SetTextColor(1, 1, 1, 1)
    
    hideBuffsCheckbox:SetScript("OnClick", function(checkbox)
        self.settings.hideBlizzardBuffs = checkbox:GetChecked()
        if self.settings.hideBlizzardBuffs then
            BuffFrame:Hide()
        else
            BuffFrame:Show()
        end
    end)

    yPos = yPos - 35

    -- Audio Alerts checkbox
    local audioAlertsCheckbox = CreateFrame("CheckButton", nil, frame, "UICheckButtonTemplate")
    audioAlertsCheckbox:SetPoint("TOPLEFT", frame, "TOPLEFT", leftPadding, yPos)
    audioAlertsCheckbox:SetSize(20, 20)
    audioAlertsCheckbox:SetChecked(self.settings.audioAlerts)
    
    local audioAlertsLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    audioAlertsLabel:SetPoint("LEFT", audioAlertsCheckbox, "RIGHT", 8, 0)
    audioAlertsLabel:SetText("Audio Alerts (60s warning + expiration)")
    audioAlertsLabel:SetTextColor(1, 1, 1, 1)
    
    audioAlertsCheckbox:SetScript("OnClick", function(checkbox)
        self.settings.audioAlerts = checkbox:GetChecked()
    end)

    yPos = yPos - 35

    -- Show Tooltips checkbox
    local showTooltipsCheckbox = CreateFrame("CheckButton", nil, frame, "UICheckButtonTemplate")
    showTooltipsCheckbox:SetPoint("TOPLEFT", frame, "TOPLEFT", leftPadding, yPos)
    showTooltipsCheckbox:SetSize(20, 20)
    showTooltipsCheckbox:SetChecked(self.settings.showTooltips == nil and true or self.settings.showTooltips)
    
    local showTooltipsLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    showTooltipsLabel:SetPoint("LEFT", showTooltipsCheckbox, "RIGHT", 8, 0)
    showTooltipsLabel:SetText("Show Tooltips")
    showTooltipsLabel:SetTextColor(1, 1, 1, 1)
    
showTooltipsCheckbox:SetScript("OnClick", function(checkbox)
    BuffGuard.settings.showTooltips = checkbox:GetChecked()
    -- Force all buttons to be recreated with new tooltip settings
    BuffGuard:CreateButtons()
    BuffGuard:UpdateAllButtons()
end)

    yPos = yPos - 55 -- Extra space before sound controls

    -- Sound controls with improved layout
    self:CreateSoundControlsInTab(frame, yPos)
end

-- Create sound controls within the alerts tab with better spacing
function BuffGuard:CreateSoundControlsInTab(frame, yPos)
    local leftPadding = 15
    
    -- Warning Sound control
    local warningSoundLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    warningSoundLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", leftPadding, yPos)
    warningSoundLabel:SetText("60s Warning Sound:")
    warningSoundLabel:SetTextColor(1, 1, 0, 1) -- Yellow for section headers
    
    local warningSoundButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    warningSoundButton:SetPoint("TOPLEFT", warningSoundLabel, "BOTTOMLEFT", 0, -8)
    warningSoundButton:SetSize(130, 22) -- Slightly larger for better readability
    
    local warningSounds = {
        {name = "Quest Log", sound = "igQuestLogAbandonQuest"},
        {name = "Whisper", sound = "TellMessage"},
        {name = "Item Pickup", sound = "igBackPackOpen"},
        {name = "Error Beep", sound = "igPlayerInviteDecline"},
        {name = "Level Up", sound = "LevelUp"},
        {name = "Auction House", sound = "AuctionWindowOpen"},
        {name = "Raid Warning", sound = "RaidWarning"},
        {name = "PvP Enter", sound = "PVPEnterQueue"},
        {name = "Map Ping", sound = "MapPing"},
        {name = "Ready Check", sound = "ReadyCheck"},
        {name = "Chat Open", sound = "igChatEmoteButton"},
        {name = "Interface Click", sound = "igMainMenuOption"}
    }
    
    local function getWarningSoundName()
        for _, soundData in ipairs(warningSounds) do
            if soundData.sound == (self.settings.warningSound or "igQuestLogAbandonQuest") then
                return soundData.name
            end
        end
        return "Quest Log"
    end
    
    warningSoundButton:SetText(getWarningSoundName())
    warningSoundButton:SetScript("OnClick", function()
        local currentIndex = 1
        for i, soundData in ipairs(warningSounds) do
            if soundData.sound == (self.settings.warningSound or "igQuestLogAbandonQuest") then
                currentIndex = i
                break
            end
        end
        
        currentIndex = currentIndex + 1
        if currentIndex > #warningSounds then
            currentIndex = 1
        end
        
        self.settings.warningSound = warningSounds[currentIndex].sound
        warningSoundButton:SetText(warningSounds[currentIndex].name)
        PlaySound(self.settings.warningSound)
    end)

    yPos = yPos - 55 -- Better spacing between sound controls

    -- Expired Sound control
    local expiredSoundLabel = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    expiredSoundLabel:SetPoint("TOPLEFT", frame, "TOPLEFT", leftPadding, yPos)
    expiredSoundLabel:SetText("Expiration Sound:")
    expiredSoundLabel:SetTextColor(1, 1, 0, 1)
    
    local expiredSoundButton = CreateFrame("Button", nil, frame, "UIPanelButtonTemplate")
    expiredSoundButton:SetPoint("TOPLEFT", expiredSoundLabel, "BOTTOMLEFT", 0, -8)
    expiredSoundButton:SetSize(130, 22)
    
    local expiredSounds = {
        {name = "Quest Failed", sound = "igQuestFailed"},
        {name = "Error Beep", sound = "igPlayerInviteDecline"},
        {name = "Raid Warning", sound = "RaidWarning"},
        {name = "Death Sound", sound = "igCreatureAggroSelect"},
        {name = "PvP Flag", sound = "igPVPUpdate"},
        {name = "Chat Scroll", sound = "igChatScrollButton"},
        {name = "Ability Failed", sound = "igAbilityIconDrop"},
        {name = "Aggro Warning", sound = "igCreatureAggroSelect"},
        {name = "Quest List", sound = "igQuestListSelect"},
        {name = "Low Health", sound = "igPlayerLowHealthWarning"},
        {name = "Bag Open", sound = "igBagOpen"},
        {name = "Inventory Full", sound = "igBackpackCantPlaceItems"}
    }
    
    local function getExpiredSoundName()
        for _, soundData in ipairs(expiredSounds) do
            if soundData.sound == (self.settings.expiredSound or "igQuestFailed") then
                return soundData.name
            end
        end
        return "Quest Failed"
    end
    
    expiredSoundButton:SetText(getExpiredSoundName())
    expiredSoundButton:SetScript("OnClick", function()
        local currentIndex = 1
        for i, soundData in ipairs(expiredSounds) do
            if soundData.sound == (self.settings.expiredSound or "igQuestFailed") then
                currentIndex = i
                break
            end
        end
        
        currentIndex = currentIndex + 1
        if currentIndex > #expiredSounds then
            currentIndex = 1
        end
        
        self.settings.expiredSound = expiredSounds[currentIndex].sound
        expiredSoundButton:SetText(expiredSounds[currentIndex].name)
        PlaySound(self.settings.expiredSound)
    end)
end

-- Legacy function - now redirects to tab-specific update
function BuffGuard:UpdateBuffEditBoxes()
    self:UpdateBuffsTabEditBoxes()
end

-- Auto-capitalize all buff names in per-character database
function BuffGuard:CapitalizeAllBuffNames()
    if not self.settings or not self.settings.buffs then return end
    
    for i = 1, CONSTANTS.MAX_BUTTONS do
        if self.settings.buffs[i] and self.settings.buffs[i] ~= "" then
            self.settings.buffs[i] = CapitalizeSpellName(self.settings.buffs[i])
        end
    end
end

-- Refresh text box contents with capitalized names from per-character settings
function BuffGuard:RefreshBuffEditBoxes()
    if not self.configFrame or not self.buffEditBoxes then return end
    
    for i = 1, self.settings.numButtons do
        if self.buffEditBoxes[i] and self.buffEditBoxes[i].editbox then
            local currentText = self.settings.buffs[i] or ""
            self.buffEditBoxes[i].editbox:SetText(currentText)
        end
    end
end

function BuffGuard:ShowConfigFrame()
    if not self.configFrame then
        self:CreateConfigFrame()
    else
        self.configFrame:Show()
        self.configFrame:Raise()
    end
    
    -- Auto-capitalize and refresh text boxes when opening
    self:CapitalizeAllBuffNames()
    self:RefreshBuffEditBoxes()
end

function BuffGuard:HideConfigFrame()
    if self.configFrame then
        -- Auto-capitalize all buff names before closing
        self:CapitalizeAllBuffNames()
        self.configFrame:Hide()
    end
end

function BuffGuard:ToggleConfigFrame()
    if self.configFrame and self.configFrame:IsShown() then
        self:HideConfigFrame()
    else
        self:ShowConfigFrame()
    end
end

function BuffGuard:ToggleMainFrame()
    if self.mainFrame:IsShown() then
        self.mainFrame:Hide()
        self.settings.enabled = false
    else
        self.mainFrame:Show()
        self.settings.enabled = true
    end
end

-- Enhanced slash command for debugging and functionality
function BuffGuard:SlashCommandHandler(msg)
    msg = msg or ""
    
    if msg == "config" or msg == "" then
        self:ToggleConfigFrame()
    elseif msg == "toggle" then
        self:ToggleMainFrame()
    elseif msg == "reset" then
        -- Reset current character's settings
        if self.currentCharacterKey and BuffGuardCharDB then
            BuffGuardCharDB[self.currentCharacterKey] = nil
            self:InitializeDB()
            self:CreateButtons()
            self:UpdateAllButtons()
            print("BuffGuard: Configuration reset to defaults for this character")
        end
    elseif msg == "test" then
        print("BuffGuard: === DIAGNOSTIC TEST ===")
        print("Current character:", self.currentCharacterKey or "nil")
        print("Settings exist:", self.settings ~= nil)
        print("Settings.buffs exist:", self.settings and self.settings.buffs ~= nil)
        print("Global BuffGuardDB exists:", BuffGuardDB ~= nil)
        print("Global BuffGuardCharDB exists:", BuffGuardCharDB ~= nil)
        
        if self.currentCharacterKey and BuffGuardCharDB then
            print("Character data exists in global:", BuffGuardCharDB[self.currentCharacterKey] ~= nil)
            if BuffGuardCharDB[self.currentCharacterKey] then
                local charData = BuffGuardCharDB[self.currentCharacterKey]
                print("Number of buttons:", charData.numButtons)
                print("Enabled:", charData.enabled)
                print("Audio alerts:", charData.audioAlerts)
                print("Button width:", charData.buttonWidth)
                print("Button height:", charData.buttonHeight)
                print("Buffs table exists:", charData.buffs ~= nil)
                if charData.buffs then
                    print("Buffs table size:", #charData.buffs)
                    print("First buff:", charData.buffs[1] or "empty")
                end
                print("Settings reference matches global:", self.settings == charData)
            end
        end
        
        if self.settings and self.settings.buffs then
            print("Can access settings.buffs[1]:", self.settings.buffs[1] or "empty")
        end
        
        print("Main frame exists:", self.mainFrame ~= nil)
        print("Config frame exists:", self.configFrame ~= nil)
        print("Minimap button exists:", self.minimapButton ~= nil)
        print("Button count:", #self.buttons)
        print("=== END DIAGNOSTIC ===")
    elseif msg == "save" then
        -- Force save test
        if self.settings then
            print("BuffGuard: Forcing save test...")
            self.settings.testValue = GetTime()
            print("BuffGuard: Set test value:", self.settings.testValue)
            print("BuffGuard: Global contains test value:", BuffGuardCharDB[self.currentCharacterKey].testValue)
        end
    else
        print("BuffGuard Commands:")
        print("/buffguard or /buffguard config - Open/close configuration")
        print("/buffguard toggle - Toggle main frame")
        print("/buffguard reset - Reset to defaults")
        print("/buffguard test - Test command and show detailed settings")
        print("/buffguard save - Test save functionality")
    end
end

-- Event handling with proper saved variables loading
function BuffGuard:CreateEventFrame()
    if self.eventFrame then
        return -- Don't create multiple event frames
    end
    
    self.eventFrame = CreateFrame("Frame")
    self.eventFrame:RegisterEvent("ADDON_LOADED")
    self.eventFrame:RegisterEvent("PLAYER_LOGIN") 
    self.eventFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
    
    -- Update timer for buff checking
    local updateTimer = 0
    self.eventFrame:SetScript("OnUpdate", function(frame, elapsed)
        updateTimer = updateTimer + elapsed
        if updateTimer >= CONSTANTS.UPDATE_INTERVAL then
            BuffGuard:UpdateAllButtons()
            BuffGuard:CleanupFloatingTexts()
            updateTimer = 0
        end
    end)
    
    self.eventFrame:SetScript("OnEvent", function(frame, event, ...)
        if event == "ADDON_LOADED" then
            local loadedAddon = ...
            if loadedAddon == "BuffGuard" then
                -- Don't initialize yet - wait for PLAYER_LOGIN when all data is ready
            end
        elseif event == "PLAYER_LOGIN" then
            -- This is when saved variables are fully loaded and character data is available
            local success = BuffGuard:InitializeDB()
            if success then
                BuffGuard:Initialize()
            end
        elseif event == "PLAYER_ENTERING_WORLD" then
            -- Ensure we're initialized by now
            if not BuffGuard.initialized then
                local success = BuffGuard:InitializeDB()
                if success then
                    BuffGuard:Initialize()
                end
            end
            BuffGuard:UpdateAllButtons()
        end
    end)
end

-- Main initialization with validation
function BuffGuard:Initialize()
    if self.initialized then
        return
    end
    
    -- Ensure database is initialized
    if not self.dbInitialized then
        local success = self:InitializeDB()
        if not success then
            return
        end
    end
    
    -- Verify settings are available
    if not self.settings then
        return
    end
    
    self:CreateMainFrame()
    self:CreateButtons()
    self:CreateMinimapButton()
    
    if self.settings and self.settings.enabled then
        self.mainFrame:Show()
    else
        self.mainFrame:Hide()
    end
    
    self.initialized = true
end

-- Register slash commands
SLASH_BUFFGUARD1 = "/buffguard"
SLASH_BUFFGUARD2 = "/bg"
SlashCmdList["BUFFGUARD"] = function(msg)
    BuffGuard:SlashCommandHandler(msg)
end

-- Start the addon
BuffGuard:CreateEventFrame()